! Copyright 2019
!
! Alex G. Harvey with contributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!-----------------------------------------------------------------------
!> @brief Orthogonal polynomials, rotation matrices, angular momentum algebra, etc.
!> @author Alex Harvey
!> @date  2019
!>
!> Contains: 
!> - Spherical harmonics, Legendre polynomials and associated functions.
!> - Wigner 3j and Clebsch-Gordan coefficients
!> - Wigner D matrices (real and complex forms)
!> - Coulomb phase function.
!> - Various indexing functions and other misc. functions. 
!----------------------------------------------------------------------- 
module dipelm_special_functions
use dipelm_defs, only: idp, pi, eye
!implicit none

!   integer, parameter :: idp = kind( 1.d0 )
   integer, parameter :: rk = idp
!   real(kind=idp), parameter :: pi=3.1415926535897932_idp
!   complex(kind=rk), parameter :: eye=(0._rk,1.0_rk) !sqrt of minus 1

    type, private :: time
       
       real :: start
       real :: finish
       character(len=:), allocatable :: message
       
    contains
    
       procedure, pass :: tic
       procedure, pass :: toc
    
    end type
    
   
contains

   subroutine tic(this)
      implicit none
      
      class(time) :: this
      
      call cpu_time(this % start)   

   end subroutine
   
   subroutine toc(this)
      implicit none
      
      class(time) :: this

      
      call cpu_time(this % finish)
      
      if (allocated( this % message))  print *,  this % message

      write(6, '("  Time = ",f6.3," seconds.")') this % finish - this % start
      print *, ""  
      deallocate( this % message)    

   end subroutine   
   

!***********************************************************************
!
!  Spherical Harmonics and Wigner Rotation matrices 
!
!***********************************************************************
   integer function binomial(n,k)
   implicit none
   
      integer, intent(in) :: n,k
      integer :: i

      binomial = 1
      if(k .gt. n-k) then
         do i=1,n-k
            binomial=binomial*(i+k)/i
         end do
      else
         do i=1,k
            binomial=binomial*(i+n-k)/i
         end do
   end if

   end function binomial
!
!>  @brief Calculate the Jacobi polynomials using recurrence relations.
!>
!> \f{align}
!>  P^{(a,b)}_0(x) &= 1 \\
!>  P^{(a,b)}_1(x) &=  \frac{(2+a+b)x+(a-b)}{2}  \\
!>  P^{(a,b)}_n(x) &=  (2n+a+b-1)((a^2-b^2) + (2n+a+b)(2n+a+b-2)x)P^{(a,b)}_{n-1}(x) -
!>                     \frac{2(n-1+a)(n-1+b)(2n+a+b))}{(2n(n+a+b)(2n-2+a+b))}P^{(a,b)}_{n-2}(x) 
!> \f}
!>
!>  
!>
   recursive function jacobi(n,a,b,x) result(jp)
   implicit none

      real(kind=idp) :: jp
      integer, intent(in) :: n,a,b
      real(kind=idp), intent(in) :: x  
      real(kind=idp) :: c,d,e
     
      if (n .eq. 0) then
         jp = 1._idp
      else if (n .eq. 1) then
         jp = ((2+a+b)*x+(a-b))/2._idp
      else
    
         c=(2._idp*n+a+b-1) * ((a**2-b**2)+(2*n+a+b)*(2*n+a+b-2)*x)
         d=2._idp*(n-1+a)*(n-1+b)*(2*n+a+b)
         e=(2._idp*n*(n+a+b)*(2*n-2+a+b))
       
         jp= (c*jacobi(n-1,a,b,x) - d*jacobi(n-2,a,b,x))/e
       
      end  if
   end function jacobi

!  flm   
!
!  Calculate (l-m)!/(l+m)! recursively to avoid overflow problems.
!  Neede for calculation of associated Legendre polynomials   
   recursive function flm(l,m) result(fact)
      implicit none
      
      real(idp) :: fact
      integer :: l, m
      
      if ((l .eq. 0) .and. (m .eq. 0)) then
        fact = 1
      else if  (m .eq. 0) then
        fact = 1
      else        
        fact = flm(l,m-1)/sqrt( real( (l-m+1) * (l+m), idp) )    
      end if
    
    end function
!
!   a_legendre_p
!
!  Calculate all assocaited Legendre polynomials up to lmax.
!   
!  They include the Condon-Shortley phase and are multiplied by
!  a normalization factor given by (l-m)!/(l+m)!
! 
   subroutine  a_legendre_p(lmax,beta, Plm)
      implicit none
!     Arguments      
      integer                :: lmax
      real(idp)              :: beta
      real(idp), allocatable :: Plm(:)

!     Local
      real(idp) :: cb, sb
      integer   :: l, m, i, j, im1, im2, ip1, ip2
            
      allocate( Plm( (lmax + 1)**2 ) ) 
      Plm = 0._idp
 
      cb = cos(beta)
      sb = sin(beta)

!     Starting values
!     ---------------          

      Plm(1) = 1
      Plm(3) = cb
      
!     Use recurrence relations
!     ------------------------
     
!     Populate m = l

      do l = 1, lmax
        
         i   = lm2i(l,l)
         im1 = lm2i(l-1,l-1)
        
         Plm(i) = -(2*(l-1) + 1) * sb * Plm(im1)
      
      end do
    
    
!     Populate m = l-1

      do l = 0, lmax-1
         i   = lm2i(l,l)
         ip1 = lm2i(l+1,l)
        
         Plm(ip1) = cb * (2*l + 1) * Plm(i) 
   
      end DO    
    
!     Populate the rest

      do l=2, lmax
         do m = -l+2, l-2
         
            im2 =  lm2i(l-2,m)
            im1 =  lm2i(l-1,m)
            i   =  lm2i(l,m)
            
            Plm(i) = ( (2*l-1) * cb * Plm(im1) - (l+m-1) * Plm(im2) )/(l-m)    
                 
         end do 
      
      end do

!     Apply normalization and use symmetry to populate
!     those elements not already populated.
  
      i=0
      do l = 0, lmax         
         do m = -l, l
       
            i=i+1
          
            if (m .ge. 0) then 
             
               Plm(i) = Plm(i) * flm(l,m)
             
               j      = lm2i(l,-m)
              
               Plm(j) = Plm(i) * (-1)**m
            
            end if
          
         end do       
      end do
   
   end subroutine
   
   subroutine a_sp_harm(lmax, theta, phi, Ylm)

      implicit none

!     Arguments      
      integer                    :: lmax
      real(idp)                  :: theta, phi
      complex(idp), allocatable  :: Ylm(:)

!     Local      
      integer                :: i, l, m
      real(idp), allocatable :: Plm(:)
      real(idp)              :: C
      
      call a_legendre_p(lmax, theta, Plm)
      allocate( Ylm( (lmax+1)**2 ) )
    
      i = 0
      do l = 0,lmax
      
         C = sqrt( (2*l+1)/(4*pi) )
      
         do m = -l, l
            
             i = i+1
            
             Ylm(i) = C * Plm(i) * exp(eye*m*phi) 
            
         end do
         
      end do


   end subroutine
   
   subroutine a_re_sp_harm(lmax, theta, phi, Ylm)

      implicit none

!     Arguments      
      integer                    :: lmax
      real(idp)                  :: theta, phi
      complex(idp), allocatable  :: Ylm(:)

!     Local      
      integer                :: i, l, m
      real(idp), allocatable :: Plm(:)
      real(idp)              :: C, C0
      
      call a_legendre_p(lmax, theta, Plm)
      allocate( Ylm( (lmax+1)**2 ) )
    
      i = 1
      do l = 0,lmax
      
         C0 = sqrt( (2*l+1)/(4*pi) )
         C  = sqrt(2._idp) * C0
      
         do m = -l, -1
            
             Ylm(i) = (-1)**m * C * Plm(l*l+l-m+1) * sin( abs(m)*phi )
             i = i+1 
            
         end do
         
         Ylm(i) = C0 * Plm(l*l+l+1)
         i = i + 1 
         
         do m = 1, l
            
             Ylm(i) = (-1)**m * C * Plm(l*l+l+m+1) * cos( m*phi )
             i = i+1
            
         end do         
         
      end do


   end subroutine
   
   subroutine grid_theta_phi(steps, arange, theta_grid, phi_grid)
      implicit none 
      
!     Arguments      
      integer                   :: steps(:)
      real(idp)                 :: arange(:)
      real(idp), allocatable    :: theta_grid(:), phi_grid(:)

!     Local      
      integer                   :: it, ip, nt, np
      real(idp)                 :: delta_t, delta_p
         
      nt = steps(1)
      np = steps(2)
      
      if (nt .gt. 1) then
         delta_t = ( arange(3) - arange(1) ) / (nt-1)  ! includes end of range for theta
      else
         delta_t = 0._idp
      end if
      
      delta_p = ( arange(4) - arange(2) ) / np         ! does not include end of range for phi
      
      allocate( theta_grid(nt), phi_grid(np) )
      
      do ip = 1, np
         
         phi_grid(ip) = arange(2) + (ip-1)*delta_p
         
      end do
         
      do it = 1, nt
         
            theta_grid(it) = arange(1) + (it-1)*delta_t
            
      end do
       
   end subroutine
   
   subroutine grid_sp_harm(lmax, steps, arange, Ylm_grid, basis_type )
      implicit none 
      
!     Arguments      
      integer                    :: lmax, steps(:)
      real(idp)                  :: arange(:)
      complex(idp), allocatable  :: Ylm_grid(:,:)
      character(len=3)           :: basis_type

!     Local      
      integer                   :: it, ip, nt, np, i 
      complex(idp), allocatable :: Ylm(:)
      real(idp)                 :: theta, phi, delta_t, delta_p
      character(len=3)          :: bt
      
      bt = basis_type  ! 'Ylm' = complex form,  'Slm' = real form   
      
      nt = steps(1)
      np = steps(2)
      
      if (nt .gt. 1) then
         delta_t = ( arange(3) - arange(1) ) / (nt-1)  ! includes end of range for theta
      else
         delta_t = 0._idp
      end if
      
      delta_p = ( arange(4) - arange(2) ) / np         ! does not include end of range for phi
      
      allocate( Ylm_grid( nt*np, (lmax+1)**2  ) )

      if (bt .eq. 'Ylm') then
         i = 1
         do ip = 1, np
         
            phi = arange(2) + (ip-1)*delta_p
         
            do it = 1, nt
         
               theta = arange(1) + (it-1)*delta_t

               call a_sp_harm(lmax, theta, phi, Ylm)
            
               Ylm_grid(i,:) = Ylm
            
               i = i + 1
            
               deallocate(Ylm)
            
            end do
         end do
      
      elseif(bt .eq. 'Slm') then
      
         i = 1
         do ip = 1, np
         
            phi = arange(2) + (ip-1)*delta_p
         
            do it = 1, nt
         
               theta = arange(1) + (it-1)*delta_t

               call a_re_sp_harm(lmax, theta, phi, Ylm)
            
               Ylm_grid(i,:) = Ylm
            
               i = i + 1
            
               deallocate(Ylm)
            
            end do
         end do      
      
      end if

   end subroutine
   
   subroutine grid_wigner_D(lmax, steps, arange, D_grid, basis_type )
      implicit none 
      
!     Arguments      
      integer                    :: lmax, steps(:)
      real(idp)                  :: arange(:)
      complex(idp), allocatable  :: D_grid(:,:,:)
      character(len=3)           :: basis_type
            
!     Local      
      integer                   :: ia, ib, ig, na, nb, ng, i 
      complex(idp), allocatable :: D(:,:)
      real(idp)                 :: alpha, beta, gamma, delta_a, delta_b, delta_g
      character(len=3)          :: bt
      
      bt = basis_type  ! 'Ylm' = complex form,  'Slm' = real form
            
      na = steps(1)
      nb = steps(2)
      ng = steps(3)
      
      if (nb .gt. 1) then
         delta_b = ( arange(5) - arange(2) ) / (nb-1)  ! includes end of range for beta
      else
         delta_b = 0._idp
      end if
      
      delta_a = ( arange(4) - arange(1) ) / na         ! does not include end of range for alpha
      delta_a = ( arange(5) - arange(2) ) / nb   
      delta_g = ( arange(6) - arange(3) ) / ng         ! does not include end of range for gamma
           
      allocate( D_grid( (lmax+1)**2, (lmax+1)**2, na*nb*ng ) )
 
      if (bt .eq. 'Ylm' ) then    
         i = 1
         do ig = 1, ng
         
            gamma = arange(3) + (ig-1)*delta_g
         
            do ib = 1, nb
         
               beta = arange(2) + (ib-1)*delta_b
         
               do ia = 1, na
         
                  alpha = arange(1) + (ia-1)*delta_a

                  call a_wigner_d(lmax, alpha, beta, gamma, D)
            
                  D_grid(:,:,i) = D
            
                  i = i + 1
            
                  deallocate(D)
            
               end do
            end do
         end do
         
      elseif (bt .eq. 'Slm') then
      
         i = 1
         do ig = 1, ng
         
            gamma = arange(3) + (ig-1)*delta_g
         
            do ib = 1, nb
         
               beta = arange(2) + (ib-1)*delta_b
         
               do ia = 1, na
         
                  alpha = arange(1) + (ia-1)*delta_a

                  call a_re_wigner_d(lmax, alpha, beta, gamma, D)
            
                  D_grid(:,:,i) = D
            
                  i = i + 1
            
                  deallocate(D)
            
               end do
            end do
         end do      
    
      end if
      
      
   end subroutine
!
!> @brief Compute the value of the Wigner d matrix.
!
!> The explicit expression involving Jacobi Polynomials is used:
!>
!> \f[
!> d^{j}_{mn}(\beta)=(-1)^{\lambda} \binom{2j-k}{k+a}^{\frac{1}{2}} \binom{k+b}{b}^{-\frac{1}{2}} \left(\sin\frac{\beta}{2}\right)^a \left(\cos\frac{\beta}{2}\right)^b P^{(a,b)}_k(\cos\beta)
!> \f]
!>
!> where \f$ k = \min(j+n, j-n, j+m, j-m). \f$
!>
!> \f$ a \f$ and \f$ \lambda \f$ depend on the form of \f$ k \f$ and are given by 
!>
!> \f[ 
!>  k = \begin{cases}
!>        j+n:  & a=m-n;\quad \lambda=m-n\\
!>        j-n:  & a=n-m;\quad \lambda= 0 \\
!>        j+m: & a=n-m;\quad \lambda= 0 \\
!>        j-m: & a=m-n;\quad \lambda=m-n \\
!>  \end{cases}
!> \f]
!>
!> and \f$ b=2j-2k-a \f$
!>
!> @see wigner_d
!> @todo Implement recurrence relation approach to calculating wigner d
   real(kind=kind(1.d0)) function wigner_small_d(beta,j,m,n)
   implicit none
   
      integer, parameter :: idp = kind( 1.d0 )
      real(kind=idp), intent(in) :: beta 
      integer, intent(in) :: j,m,n
      integer, dimension(4) :: arr
      integer :: k, i, a, b, lambda
      real(kind=idp) :: cosbeta   

      arr = (/ j+n,j-n,j+m,j-m /)
      k = minval(arr, dim=1)
      i = minloc(arr, dim=1)
    
      if ((i .eq. 1) .or. (i .eq. 4)) then
         a = m-n
         lambda = m-n
      else if ((i .eq. 2) .or. (i .eq. 3)) then
         a = n-m
         lambda = 0        
      end if
    
      b=2*j-2*k-a
    
      cosbeta=cos(beta)
      wigner_small_d = (-1.0)**lambda * sqrt(binom(2*j-k,k+a))&
&              * 1/sqrt(binom(k+b,b)) &
&              * sin(beta/2)**a * cos(beta/2)**b * jacobi(k,a,b,cosbeta)
    
   end function wigner_small_d
   
!> @brief Calculate the wigner D function
!>
!> Rotation matrix elements for the complex spherical harmonics.
!> They have the form:
!>
!>   
!>
   complex(kind=idp) function wigner_d(alpha, beta, gamma,j,m,n)
   implicit none
  
      integer, intent(in) :: j,m,n
      real(kind=idp), intent(in) :: beta, alpha, gamma
      real(kind=idp) :: re_rm,im_rm
    
      re_rm=cos(m*alpha+n*gamma)*wigner_small_d(beta,j,m,n) 
      im_rm=-sin(m*alpha+n*gamma)*wigner_small_d(beta,j,m,n)
    
      wigner_d=cmplx(re_rm,im_rm,idp)


   
   end function wigner_d
   
   real(idp) function re_wigner_d(alpha, beta, gamma,j,m,n)

      implicit none

      integer, intent(in)   :: j,m,n
      real(idp), intent(in) :: beta, alpha, gamma

      real(idp) :: phi_pm, phi_pn,phi_mm, phi_mn 
      real(idp) :: d1, d2
      integer   :: ma, na

      phi_pm = azim_fn(alpha, m)
      phi_mm = azim_fn(alpha, -m)
      phi_pn = azim_fn(gamma, n)
      phi_mn = azim_fn(gamma, -n)

      ma     = abs(m)
      na     = abs(n)
      
      d1     = wigner_small_d(beta,j,na,ma)
      d2     = wigner_small_d(beta,j,ma,-na)

      re_wigner_d  = sign(1,n)*phi_pm*phi_pn * ( d1 + (-1)**m * d2 ) / 2 &
     &              -sign(1,m)*phi_mm*phi_mn * ( d1 - (-1)**m * d2 ) / 2

    end function
      
    real(idp) function azim_fn(theta, m)
      implicit none
      integer, intent(in)        :: m
      real(kind=idp), intent(in) :: theta
      real(kind=idp)             :: temp

      select case(m)
      case(1:)
         temp = sqrt(2._idp)*cos(m*theta)
      case(0)
         temp = 1._idp
      case(:-1)
         temp = sqrt(2._idp)*sin(abs(m)*theta)
      end select

      azim_fn=temp

   end function azim_fn
             
   subroutine a_wigner_d(jmax, alpha, beta, gamma, Djmn)
      implicit none
      
      integer, intent(in) :: jmax
      real(kind=idp), intent(in) :: beta, alpha, gamma
      complex(kind=idp), allocatable, intent(inout) :: Djmn(:,:)
      
      integer :: nc, j, m, n, ijm, ijn
      
      nc = (jmax+1)**2
      allocate( Djmn(nc,nc) )
      Djmn = 0._idp
      
      do j = 0, jmax
         do n = -j, j
            do m = -j, j 
                
                ijm = lm2i(j,m)
                ijn = lm2i(j,n)
                
                Djmn(ijm,ijn) = wigner_d(alpha, beta, gamma,j,m,n)
    
            end do
         end do
      end do

   end subroutine
   
    subroutine a_re_wigner_d(jmax, alpha, beta, gamma, Djmn)
      implicit none
      
      integer, intent(in)                      :: jmax
      real(idp), intent(in)                    :: beta, alpha, gamma
      complex(idp), allocatable, intent(inout) :: Djmn(:,:)
      
      integer :: nc, j, m, n, ijm, ijn
      
      nc = (jmax+1)**2
      allocate( Djmn(nc,nc) )
      Djmn = 0._idp
      
      do j = 0, jmax
         do n = -j, j
            do m = -j, j 
                
                ijm = lm2i(j,m)
                ijn = lm2i(j,n)
                
                Djmn(ijm,ijn) = re_wigner_d(alpha, beta, gamma,j,m,n)
    
            end do
         end do
      end do

   end subroutine  
!
!> @brief Calculate spherical harmonic using wigner small d
!>
!> The relation used is:
!>
!> \f{align}
!>  Y_{jm}(\theta, \phi) &= \sqrt\frac{2j+1}{4\pi}d^j_{|m|0}(\theta) e^{im\phi} \qquad m \geq 0 \\
!>                       &= (-1)^m \sqrt\frac{2j+1}{4\pi}d^j_{|m|0}(\theta) e^{im\phi} \qquad m < 0 \\
!> \f}  
   complex(kind=idp) function sp_harm(j,m,theta,phi)
   implicit none

      integer, intent(in) :: j,m
      real(kind=idp), intent(in) :: theta, phi
      real(kind=idp) :: re_sh,im_sh

      if (m .ge. 0) then
         re_sh = sqrt((2*j+1)/(4*pi))*cos(abs(m)*phi)&
         &     * wigner_small_d(theta,j,abs(m),0) 
         im_sh = sqrt((2*j+1)/(4*pi))*sin(abs(m)*phi)&
         &     * wigner_small_d(theta,j,abs(m),0)
      else
         re_sh = sqrt((2*j+1)/(4*pi))*cos(abs(m)*phi)&
         &     * wigner_small_d(theta,j,abs(m),0)*(-1.0)**abs(m) 
         im_sh =-sqrt((2*j+1)/(4*pi))*sin(abs(m)*phi)&
         &     * wigner_small_d(theta,j,abs(m),0)*(-1.0)**abs(m) 
      end if

      sp_harm = cmplx(re_sh,im_sh,kind=idp)


   end function sp_harm   
   
   real(kind=idp) function re_sp_harm(j,m,theta,phi)
   implicit none
   
      integer, intent(in) :: j,m
      real(kind=idp), intent(in) :: theta, phi
      real(kind=idp) :: temp,norm

      norm=((-1)**m)/sqrt(2d0)

      select case(m)
      case(1:)
         temp = norm*2*sqrt((2*j+1)/(4*pi))*cos(m*phi)&
         &    * wigner_small_d(theta,j,m,0) 
      case(0)
         temp = sqrt((2*j+1)/(4*pi))*wigner_small_d(theta,j,0,0)
      case(:-1)
         temp = norm*2*sqrt((2*j+1)/(4*pi))*sin(abs(m)*phi)&
         &    * wigner_small_d(theta,j,abs(m),0)
      end select

      re_sp_harm = temp

   end function re_sp_harm
 
   function sph_basis_transform_elm(l,m,mp, basis_type)
   implicit none
!     Arguments   
      integer, intent(in)    :: l, m, mp
      character(3)           :: basis_type    
      complex(rk)            :: sph_basis_transform_elm
!     Local      
      character(3) :: bt
      integer      :: ma, mpa
      
!     basis_type = "Ylm" transform from complex to real spherical harmonics 
!     basis_type = "Slm" transform from real to complex spherical harmonics
      
      bt = basis_type  
      
      ma  = abs(m)
      mpa = abs(mp)
  
      if (bt .eq. "Slm") then
      
         if ( ma .eq. mpa) then
            
            if (m  .eq. 0)                    sph_basis_transform_elm = sqrt(2.0_rk) 
            if ((m .gt. 0) .and. (mp .gt. 0)) sph_basis_transform_elm = (-1.0_rk)**ma 
            if ((m .gt. 0) .and. (mp .lt. 0)) sph_basis_transform_elm = eye*(-1.0_rk)**ma
            if ((m .lt. 0) .and. (mp .gt. 0)) sph_basis_transform_elm = 1._rk
            if ((m .lt. 0) .and. (mp .lt. 0)) sph_basis_transform_elm = -eye
            
         else
                  
            sph_basis_transform_elm = 0._rk       
         
         end if
         
      elseif (bt .eq. "Ylm") then 
      
         if ( ma .eq. mpa) then
            
            if (m  .eq. 0)                    sph_basis_transform_elm = sqrt(2.0_rk) 
            if ((m .gt. 0) .and. (mp .gt. 0)) sph_basis_transform_elm = (-1.0_rk)**ma 
            if ((m .gt. 0) .and. (mp .lt. 0)) sph_basis_transform_elm = 1._rk
            if ((m .lt. 0) .and. (mp .gt. 0)) sph_basis_transform_elm = -eye*(-1.0_rk)**ma
            if ((m .lt. 0) .and. (mp .lt. 0)) sph_basis_transform_elm = eye           
            
         else
                  
            sph_basis_transform_elm = 0._rk       
         
         end if      
      
         
      end if

      sph_basis_transform_elm = sph_basis_transform_elm/sqrt(2.0_rk)

   end function sph_basis_transform_elm

!  Transform, U, is defined to be such that X' = UX   
   subroutine sph_basis_transform_matrix( U, lmax, basis_type )
   implicit none
   
!     Arguments     
      complex(rk), allocatable, intent(inout) :: U(:,:)
      integer, intent(in)                     :: lmax
      character(3)                            :: basis_type    
     
!     Local     
      character(3) :: bt
      integer      :: i,j, li, mi, lj, mj, nlm
      
!     basis_type = "Ylm" transform from complex to real spherical harmonics 
!     basis_type = "Slm" transform from real to complex spherical harmonics
   
      bt = basis_type
   
      nlm = (lmax +1)**2
      
      allocate( U(nlm,nlm) )
      U= 0._rk
      
      do j = 1, nlm
      
         call i2lm(j, lj, mj)
         
         do i = 1, nlm
         
            call i2lm(i, li, mi)
            
            if (li .eq. lj ) then 
            
               U(i, j) = sph_basis_transform_elm(li,mi,mj, bt)
            
            end  if
         
         end do
         
      end do
      
      
   
   end subroutine

!  -------------------
!  Indexing procedures      
!  ------------------- 

!  Maps l and m quantum numbers to single index i
!  order is (0,0), (1,-1), (1,0) (1,1) etc   
   integer function lm2i(l,m)
      implicit none
      integer :: l,m

      lm2i=l*(l+1)+(m+1)

      return
      
   end function lm2i
   
!  Maps  single index i to l and m quantum numbers 
!  order is (0,0), (1,-1), (1,0) (1,1) etc 
   subroutine i2lm(i,l,m)
      implicit none      
      ! Arguments
      integer :: i,l,m
      
      l=0
      
      do
         if ((l+1)**2 .ge. i) exit
         l=l+1
      end do
      
      m=i-(l+1)**2+l

      return
   end subroutine i2lm
        
!*********************************************************************** 
! 
!  Angular momentum algebra functions
!
! 
!  Note: binom() and cleb() taken from library of angular 
!  momentum coupling  coefficient routines in fortran 90
!  Paul Stevenson, Oxford University/Oak Ridge National Laboratory.
!  spaul@mail.phy.ornl.gov
! 
!***********************************************************************
   
   recursive function binom(n,r) result(res)
   implicit none
      
      integer :: n,r
      real(rk) :: res
      real(rk) :: tmp

      if(n==r .or. r==0) then
         res = 1.0
      else if (r==1) then
         res = real(n,rk)
      else
         res = real(n,rk)/real(n-r,rk)*binom(n-1,r)
      end if
      
   end function binom

   function cleb(j1,m1,j2,m2,j,m)
   implicit none  
   
    ! calculate a clebsch-gordan coefficient < j1/2 m1/2 j2/2 m2/2 | j/2 m/2 >
    ! arguments are integer and twice the true value. 

      real(rk)    :: cleb,factor,sum
      integer :: j1,m1,j2,m2,j,m,par,z,zmin,zmax

    ! some checks for validity (let's just return zero for bogus arguments)

       if (2*(j1/2)-int(2*(j1/2.0_rk)) /= 2*(abs(m1)/2)-int(2*(abs(m1)/2.0_rk)) .or. &
          2*(j2/2)-int(2*(j2/2.0_rk)) /= 2*(abs(m2)/2)-int(2*(abs(m2)/2.0_rk)) .or. &
          2*(j/2)-int(2*(j/2.0_rk)) /= 2*(abs(m)/2)-int(2*(abs(m)/2.0_rk)) .or. &
          j1<0 .or. j2<0 .or. j<0 .or. abs(m1)>j1 .or. abs(m2)>j2 .or.&
          abs(m)>j .or. j1+j2<j .or. abs(j1-j2)>j .or. m1+m2/=m) then
          cleb= 0.0_rk
       else
    
          factor = 0.0_rk
          factor = binom(j1,(j1+j2-j)/2) / binom((j1+j2+j+2)/2,(j1+j2-j)/2)
          factor = factor * binom(j2,(j1+j2-j)/2) / binom(j1,(j1-m1)/2)
          factor = factor / binom(j2,(j2-m2)/2) / binom(j,(j-m)/2)
          factor = sqrt(factor)
       
          zmin = max(0,j2+(j1-m1)/2-(j1+j2+j)/2,j1+(j2+m2)/2-(j1+j2+j)/2)
          zmax = min((j1+j2-j)/2,(j1-m1)/2,(j2+m2)/2)
       
          sum=0.0_rk
          do z = zmin,zmax
             par=1
             if(2*(z/2)-int(2*(z/2.0_rk)) /= 0) par=-1
                sum=sum+par*binom((j1+j2-j)/2,z)*binom((j1-j2+j)/2,(j1-m1)/2-z)*&
                    binom((-j1+j2+j)/2,(j2+m2)/2-z)
          end do
       
          cleb = factor*sum
       end if
   end function cleb

   function threej(j1,m1,j2,m2,j,m)
   implicit none
   
      integer :: j1,m1,j2,m2,j,m
      real(rk) :: threej,cleb_temp

      cleb_temp=cleb(j1,m1,j2,m2,j,-m)

      threej = (-1)**((j1 - j2 - m) / 2) / sqrt(real(j + 1, rk))&
      &      * cleb_temp


   end function threej

!*********************************************************************** 
! 
!  Misc.  functions
!
!***********************************************************************
!
!  Calculates the Coulomb phase
!
   function cphaz(l,eta,iwrite)

      implicit double precision (a-h,o-z)
!
      data half/0.5d0/,zero/0.0d0/,two/2.0d0/,ip1/9/
      data c1/0.83333333333333333d-01/,c2/0.27777777777777778d-02/, &
           c3/0.79365079365079365d-03/,c4/0.59523809523809523d-03/, &
           c5/0.84175084175084175d-03/
!
 1000 format(' Coulomb phase, sigma(',i3,';',d16.8,' ) =',d30.17)
!
!     arg ( gamma (l+1+i*eta) ) = imag ( ln gamma (p+i*eta) )
!                                 - sum( s=l+1 to s=p-1 ) atan ( eta/s )
!     for some integer p
!
!     here take p=10
!
      d1=zero
      b=eta
      k1=l+1
      if(k1>ip1) go to 30
!
!     sum arc tangent terms
!
   10 do 20 j=k1,ip1,1
      a=dble(j)
      d=datan2(b,a)
      d1=d+d1
   20 continue
      a=dble(ip1+1)
      go to 40
!
   30 a=dble(l+1)
      d1=zero
!
!     evaluate ln ( gamma (p-i*eta) ) using an asymptotic formula
!     reference: n.b.s. tables eqns 6.1.40 and 6.1.41
!
   40 c=half*dlog(a*a+b*b)
      d=datan2(b,a)
      d=d*(a-half)+c*b
      d=d-b
!
!     evaluate  1 / (p+i*eta) and ( 1 / (p+i*eta) ) **2
!
      e=a*a+b*b
      a=a/e
      e=-b/e
      f=a*a-e*e
      g=two*e*a
!
!     -r1 = h  = 1/1188 f - 1/1680
!     -i1 = h1 = 1/1188 g
!
      h=c5*f-c4
      h1=c5*g
      h2=h
!
!     r2 = h  = 1/1260 + f * ( -r1 ) - g * ( -i1 )
!     i2 = h1 =          f * ( -i1 ) + g * ( -r1 )
!
      h=h*f-h1*g
      h1=h1*f+h2*g
      h=h+c3
      h2=h
!
!     -r3 = h  = -1/360 + f * ( r2 ) - g * ( i2 )
!     -i3 = h1 =          f * ( i2 ) + g * ( r2 )
!
      h=h*f-h1*g
      h1=h1*f+h2*g
      h=h-c2
      h2=h
!
!     r4 = h  = 1/12 + f * ( -r3 ) - g * ( -i3 )
!     i4 = h1 =        f * ( -i3 ) + g * ( -r3 )
!
      h=h*f-h1*g
      h1=h1*f+h2*g
      h=h+c1
      h2=h
!
!     evaluate imag ( ( r4 + i*i4 ) / ( p + i*eta ) )
!
      h1=h1*a+h2*e
!
!     calculate final value for phase shift
!
      cphaz=h1+d-d1
!
!      write(iwrite,1000)l,eta,cphaz
!
      return
   end function cphaz
   
   subroutine linspace(from, to, no_steps, array, include_end_point)
      implicit none
!     Arguments      
      real(idp), intent(in)  :: from, to
      integer                :: no_steps
      real(idp), allocatable :: array(:)
      logical, optional      :: include_end_point

!     Local      
      real(idp) :: range
      logical   :: end_point
      integer :: i
          
      end_point = .true.
      if ( present(include_end_point) ) end_point = include_end_point
    

      range = to - from

      if (no_steps .eq. 0) return
    
      allocate( array(no_steps) )

      if (no_steps .eq. 1) then
    
         array(1) = from
         return
    
      end if

      if (end_point) then
    
         do i = 1, no_steps
    
            array(i) = from + range * (i - 1) / (no_steps - 1)
    
         end do
       
      else
    
         do i = 1, no_steps
    
            array(i) = from + range * (i - 1) / no_steps
    
         end do    
    
      end if
    
   end subroutine
         
end module dipelm_special_functions
