! Copyright 2019
!
! Alex G. Harvey with ontributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      module dipelmprocs
      use dipelmdefs
      use angmom_procs
      use dipelmio
      use blas95_compak
      implicit none

      contains
      
!     **********************************************************************************************
!
!     dipelm_drv: Constructs partial wave dipoles out of wavefunction coefficients and inner
!                 region dipoles. This routine is likely to be combined with COMPAK in the 
!                 future so that wavefunction coefficients dont need to be stored (they can get 
!                 very large).
!
!     **********************************************************************************************
      subroutine  dipelm_drv( dip_elm, ichl, evchl, lvchl, mvchl,no_channels, starg, mtarg, gtarg, escat, no_dipole_components, &
                              dipole_component_order,bound_state_energies,target_energy, lu_pw_dipoles, nset_pw_dipoles, &
                              format_pw_dipoles, ifail )
      use photo_outerio, only: read_pw_dipoles, write_pw_dipoles
      use ieee_arithmetic, only: ieee_is_nan
      implicit none

 !    Arguments
      integer :: ifail 
      complex(kind=icp), allocatable :: dip_elm(:,:,:,:)          
      character(len=1) :: dipole_component_order(maxprop_par)                
      real(kind=idp), allocatable :: escat(:), evchl(:,:)
      integer, allocatable :: ichl(:,:), lvchl(:,:), mvchl(:,:), starg(:), mtarg(:), gtarg(:)

!     Local variables
      integer, dimension(:) :: lu_pw_dipoles(maxprop_par), no_channels(maxprop_par), &
                               mgvn(maxprop_par), stot(maxprop_par), gutot(maxprop_par)
      integer :: point_group_index, ncomponents_assigned
      character(len=11) ::     format_pw_dipoles, test_formatted
      integer, dimension(:) :: nset_pw_dipoles(maxprop_par)
      character(len=80) ::     title
      character(len=3) ::      point_group
      character(len=1) ::      letter_box(3)=(/'y','z','x'/)
      
      real(kind=idp) :: target_energy
      real(kind=idp), allocatable :: escat_temp(:), evchl_temp(:), re_pw_dipoles(:,:,:,:), im_pw_dipoles(:,:,:,:),&
                                     bound_state_energies(:)
      integer, allocatable :: ichl_temp(:), lvchl_temp(:), mvchl_temp(:), dip_comp_present(:)

      integer :: lmax_property, maxchs, no_bound_states, no_scattering_energies, &
                 no_dipole_components, i, j, k, idcmp, ienergy, icomponent, iprint, iwrite

      integer :: ierr
      logical :: mismatch

      namelist /DIPELMINP/ lu_pw_dipoles, &
                           format_pw_dipoles, &
                           nset_pw_dipoles, &
                           iprint,          &
                           iwrite,          &
                           lmax_property, &
                           dipole_component_order, & !Deprecated
                           no_dipole_components, &   !Deprecated
                           point_group ! possible values are 'C1','Cs','C2','Ci','C2v','C2h','D2','D2h' ! Deprecated

      lu_pw_dipoles=0; no_channels=0; mgvn=0; stot=0; gutot=0; nset_pw_dipoles=0


      ifail=0

!     Default namelist DIPELMINP values.
      iprint=1
      iwrite=6
      lmax_property=1                ! Maximum property number 1=dipoles 2= dipoles + quadrapoles      
      format_pw_dipoles ='UNFORMATTED'
      test_formatted ='FORMATTED'
      no_dipole_components=lmax_property**2+2*lmax_property
      point_group='' ! Point group is now deprecated

!     Read namelist
!     -------------
      read(5,DIPELMINP,ERR=200, iostat=ierr)
      dipole_component_order="" ! No longer needs to be entered in the namelist
      no_dipole_components=lmax_property**2+2*lmax_property !Note. This means no_dipole_components is now redundant.
                                                            !      Just kept for backwards compatability.
  200 if (ierr .gt. 0) then
         write (iwrite,2000)
         ifail=2000
         return

      end if

!     Interrogate headers of partial wave dipoles to get information 
!     about the number of channels, check that all the dipole components are there, etc.

      call get_channel_info( lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, mgvn, no_bound_states, &
     &                       no_dipole_components,dipole_component_order,letter_box, no_channels, &
     &                       no_scattering_energies, iwrite )
      maxchs=maxval(no_channels)
      allocate(escat(no_scattering_energies), evchl(no_dipole_components,maxchs), ichl(no_dipole_components,maxchs),&
     &         lvchl(no_dipole_components,maxchs),mvchl(no_dipole_components,maxchs))
      allocate( dip_elm(maxchs, no_bound_states, no_dipole_components, no_scattering_energies) )
      escat=0_idp;evchl=0._idp;ichl=0;lvchl=0;mvchl=0;dip_elm=0._idp

      do i=1,no_dipole_components
         call read_pw_dipoles( lu_pw_dipoles(i), nset_pw_dipoles(i), format_pw_dipoles, title, &
     &                         mgvn(i), stot(i), gutot(i),starg, mtarg, gtarg, ichl_temp, lvchl_temp, mvchl_temp,  &
     &                         evchl_temp, escat_temp, lmax_property, dip_comp_present,        &
     &                         bound_state_energies, target_energy, re_pw_dipoles, im_pw_dipoles,  &
     &                         iprint, iwrite, ifail )

         if (i .eq. 1) escat = escat_temp
         mismatch = .false.

         do ienergy=1, no_scattering_energies
            if (i > 1) then
               if (escat(i) .ne. escat_temp(i)) then
                  write(*,'("Mismatch in energy values: ",i10,2e25.15)') ienergy,escat(i),escat_temp(i)
                  mismatch = .true.
               endif
               if (mismatch) then
                  write(*,'("The energy grids for component 1 and ",i2," do not match!")') i
                  stop "error"
               endif
            endif            
            do icomponent=1,no_dipole_components
               if(dip_comp_present(icomponent).eq. 1) then
                  !todo ZM is the -im_pw_dipoles there because we need to
                  !complex conjugate the radial integrals since the
                  !photoelectron wavefunction is the bra in the matrix element?
                  dip_elm(:,:,icomponent,ienergy)=transpose( cmplx(re_pw_dipoles(:,:,icomponent,ienergy),&
                                                                  -im_pw_dipoles(:,:,icomponent,ienergy), kind=icp) )
                  ichl(  icomponent,1:no_channels(i) )=ichl_temp
                  lvchl( icomponent,1:no_channels(i) )=lvchl_temp
                  mvchl( icomponent,1:no_channels(i) )=mvchl_temp
                  evchl( icomponent,1:no_channels(i) )=evchl_temp
                  !escat=escat_temp !ZM this should not be in the inner-most loop
                  
               end if
               
            end do
            if(any(ieee_is_nan(real(dip_elm(:,:,:,ienergy))))) then
               dip_elm(:,:,:,ienergy)= dip_elm(:,:,:,ienergy-1)! remove NANs from dipoles (Danilo)
            end if
         end do
         deallocate(ichl_temp,lvchl_temp,mvchl_temp,evchl_temp, escat_temp, re_pw_dipoles,im_pw_dipoles)
         
      end do
      
      return

!     format statements
 2000 format(/,5X,'Problems reading the namelist DIPELMINP',/)

      end subroutine dipelm_drv
      
!     **********************************************************************************************
!
!     get_channel_info: Interogate partial wave dipole files and fetch info about channels, states,
!                       scattering energies etc
!                       Also do some consistency checking. Number of bound states, and energy grid
!                       should be the same for each file.
!                       Future implementation should check for correct dipole components.
!
!     **********************************************************************************************  
    
      subroutine get_channel_info( lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, mgvn, nbound,&
      &                            no_dipole_components, dipole_component_order, letter_box,       &
      &                            no_channels, no_scat_energies, iwrite ) 
      
      implicit none
      
!     Arguments
      integer :: lu_pw_dipoles(:), nset_pw_dipoles(:), no_channels(:), no_scat_energies, &
     &           no_dipole_components, mgvn(:), iwrite
      character(len=11) :: format_pw_dipoles
      character(len=1) ::      letter_box(:),dipole_component_order(:)

!     Local variables
      integer :: icomponent, keydip, nset_pw_dipoles_temp, mgvn_temp, stot, gutot, nchan, nbound, &
    &            ierr, jcomponent, ncomponents_assigned,lmax_property, no_target_states
      integer, allocatable :: dip_comp_present_temp(:)
      character(len=80) ::  title
      character(len=8) :: str_temp1
      character(len=1) :: letter_box_tmp(size(letter_box))
      
      intent(in) :: lu_pw_dipoles, format_pw_dipoles, nset_pw_dipoles
      intent(inout) :: no_dipole_components, no_channels
 
      no_channels=0; ncomponents_assigned=0
      letter_box_tmp = letter_box !ZM store the initial array so we can reinitialize letter_box at the end to the same contents it had at the beginning

      allocate(dip_comp_present_temp(no_dipole_components))
      dip_comp_present_temp=0 
      
    
      do icomponent=1, no_dipole_components
         dip_comp_present_temp=0 
         if(format_pw_dipoles .eq. 'UNFORMATTED') then 
            open (unit=lu_pw_dipoles(icomponent), form=format_pw_dipoles, iostat=ierr, err=100)   
       
            read(lu_pw_dipoles(icomponent)) keydip, nset_pw_dipoles_temp, lmax_property
            read(lu_pw_dipoles(icomponent)) title
            read(lu_pw_dipoles(icomponent)) mgvn_temp, stot, gutot, nchan, nbound, &
     &                                   no_target_states, no_scat_energies, dip_comp_present_temp
         else if (format_pw_dipoles .eq. 'FORMATTED') then
         
            open (unit=lu_pw_dipoles(icomponent), form=format_pw_dipoles, iostat=ierr, err=100)   
            
            read(lu_pw_dipoles(icomponent), *) keydip,nset_pw_dipoles_temp,lmax_property
            read(lu_pw_dipoles(icomponent),'(a8, a80)') str_temp1, title
            read(lu_pw_dipoles(icomponent), *) mgvn_temp, stot, gutot, nchan, nbound, no_target_states, no_scat_energies    
            read(lu_pw_dipoles(icomponent), *) dip_comp_present_temp         
         
         end if
      
         no_channels(icomponent)=nchan
         mgvn(icomponent)=mgvn_temp

         rewind(lu_pw_dipoles(icomponent))
         close(unit=lu_pw_dipoles(icomponent))
         
         do jcomponent=1, no_dipole_components
            if ( (dip_comp_present_temp(jcomponent) .eq. 1) .and. (letter_box(jcomponent) .ne. '') ) then
               dipole_component_order(icomponent)=letter_box(jcomponent)
               letter_box(jcomponent) = ''
               ncomponents_assigned=ncomponents_assigned+1            
               exit ! Enforces the one file passed per component requirement (means same file may have to be passed multiple times)
                    ! Would be nice to remove the above requirement at some point. 
            end if
            
         end do
         
      end do

!     Consistency checking
!     --------------------
      if (ncomponents_assigned .eq. no_dipole_components) then
         write(iwrite,'(" The dipole component order is ",20(a1,1x))') &
     &         (dipole_component_order(icomponent), icomponent=1, no_dipole_components)
      else
         write(iwrite,'(" ERROR: Could not find partial wave dipole file for components ",20(a1,1x))')&
     &         (letter_box(icomponent), icomponent=1, no_dipole_components )
         stop " ERROR: Reading partial wave dipoles"
      end if

      letter_box = letter_box_tmp
       
      return   
  100 stop " ERROR: Reading partial wave dipoles"          
      end subroutine get_channel_info
      
      subroutine first_last(isym_ini_state, isym_fin_states, &
     &                      no_inner_states_fin_syms, &
     &                      no_inner_states_ini_sym, nfinstart, nfinend, & 
     &                      ninistart, niniend)
      implicit none

      integer,intent(in)  :: isym_ini_state, isym_fin_states, &
     &                       no_inner_states_fin_syms, &
     &                       no_inner_states_ini_sym 
      integer,intent(out) :: nfinstart, nfinend, ninistart, niniend

!     Require symmetry order of elements from denprop
!     may need to be enforced in denprop run
!     Determine index of first element in each symmetry
!     in the moments array.

      if (isym_ini_state .gt. isym_fin_states) then
         ninistart=1+no_inner_states_fin_syms
         niniend=no_inner_states_fin_syms+no_inner_states_ini_sym
         nfinstart=1
         nfinend=no_inner_states_fin_syms
      else if (isym_ini_state .lt. isym_fin_states) then
         ninistart=1
         niniend=no_inner_states_ini_sym
         nfinstart=1+no_inner_states_ini_sym
         nfinend=no_inner_states_fin_syms+no_inner_states_ini_sym
!      else
!          ninistart=1
!          niniend=no_inner_states_ini_sym
!          nfinstart=1
!          nfinend=no_inner_states_fin_syms
      end if

      print *, ninistart, niniend
      print *, nfinstart, nfinend

      end subroutine first_last      

      integer function  dcomp2i(dipchar)
      implicit none
      character(len=1) :: dipchar
      integer :: ntmp      

      if ((dipchar .eq. 'x') .or. (dipchar .eq. 'X')) then
         ntmp=3
      else if((dipchar .eq. 'y') .or. (dipchar .eq. 'Y')) then
         ntmp=1
      else if((dipchar .eq. 'z') .or. (dipchar .eq. 'Z')) then
         ntmp=2
      end if

      dcomp2i=ntmp

      end function dcomp2i
      
!     **********************************************************************************************
!
!     diptrans_drv: Takes MOL frame partial wave dipoles and transforms to LAB frame momentum
!                   basis.
!
!     **********************************************************************************************
      subroutine diptrans_drv(dip_elm, ichl, evchl, lvchl, mvchl, no_channels_scat_states, &
     &                        escat, no_scattering_energies, no_bound_states, no_dipole_components,&
     &                        dipole_component_order, bound_state_energies,target_energy, ifail)
      use dipelmdefs
      use blas95_compak
      use interpolate, only: dspline
      implicit none
      integer :: ifail, no_scattering_energies, no_bound_states
      integer,dimension(:) :: no_channels_scat_states
      integer, dimension(:,:) ::ichl,lvchl,mvchl
      real(kind=idp),dimension(:,:) :: evchl
      real(kind=idp), dimension(:) :: escat,bound_state_energies
      real(kind=idp) :: target_energy
      complex(kind=icp), dimension(:,:,:,:)  :: dip_elm  
      character(len=1), dimension(:) :: dipole_component_order(3)

      integer :: idcmp,no_dipole_components, no_scattering_angles,itarget, iparent_state
      integer :: i,j,k,l,m,n,en,nstat,ii,ie,it,ip, ia,ib,ig, nt,np,na,nb,ng
          
      real(kind=idp), allocatable :: exp_alignment_distribution(:,:), &
     &                               int_exp_alignment_distribution(:)
      complex(kind=icp), allocatable, dimension(:,:,:,:,:)  :: dip_full,dipoles,dipoles_lab
      type(dspline) :: dinterpolated_alignment_distribution
      type(dspline), allocatable :: averaged_dcs_splines(:,:,:)

!     Coulomb phase
      complex(kind=icp), allocatable :: coulomb_phase(:,:,:)
      real(kind=idp) :: k_final, charge, eta

      complex(kind=icp), allocatable :: temp_matrix1(:,:),temp_matrix2(:,:),temp_matrix3(:,:), & 
     &                                  z_reRotl1(:,:,:)

      !----namelist 
      integer :: max_lcontinuum, no_targ_states, iplane, lebedev, luexpoints,lu_photo_xsec, lu_photo_dcs
      integer :: ibound_state, iorient_averaging, iuse_calculated_IP!boundstate from which to calculate dipoles
      integer, dimension(:) ::ngrdproj(2), ngrdalign(3)
      integer :: ipunits  !0=atomic units 1=megabarns
      real(kind=idp), dimension(:) :: degeneracy(max_states_for_dipoles), & 
     &                                experimental_IP(max_states_for_dipoles)
      real(kind=idp) :: degeneracy_threshold
      character(len=80) :: molecule_name
      !--------
      
      real(kind=idp) :: thmax, phmax, almax, bemax, gamax
      real(kind=idp), allocatable, dimension(:) :: th_grid, ph_grid, a_grid, b_grid, g_grid, &
     &                                             ph_grid_plus_1
!     matrices
      real(kind=idp),allocatable, dimension(:,:) :: reSpH, grid_proj, lebedev_grid, &
     &                                              extremal_points_cart, extremal_points_spher, &
     &                                              theta_phi_av_dcs
      real(kind=idp),allocatable, dimension(:,:,:) :: reRot, reRotl1, grid_align, &
     &                                                rotated_real_spherical_harmonics
      real(kind=idp), allocatable :: dcs_averaged(:,:,:,:), dcs_averaged_2(:,:,:,:), &
     &                               dcs_averaged_interpolated(:,:,:,:)
      complex(kind=idp), dimension(:) :: temp_lab_dipole(3) 

      complex(kind=icp),allocatable :: photo_dipoles(:,:,:,:),temp_photo_dipoles(:,:,:) !NEW
      real(kind=idp), allocatable :: photo_dcs(:,:,:,:),temp_photo_dcs(:,:,:),photo_dcs_averaged(:,:,:,:),&
                                     photo_dcs_averaged_test(:,:,:,:)!NEW

      real(kind=idp), allocatable, dimension(:) :: etarg
      real(kind=idp) :: first_IP,pconst,xx,yy,theta0, temp, re_dip_tmp,im_dip_tmp
      character(len=120) :: name       
    
      real(kind=idp) :: euler_angle_limits(6),scat_angle_limits(4),theta,phi

      real(kind=idp) :: CROSS_SECTION_TEMP,D_THETA,d_phi,PHOTON_ENERGY,IONIZATION_POT
      real(kind=idp),allocatable :: CROSS_SECTION(:,:),total_xsec(:), partial_xsec(:,:), &
                                    asymmetry_param(:,:)

      real(kind=idp) :: x_xsec,y_xsec, z_xsec


      integer :: ierr, itarg, iangle,ipw, ieuler, icomponent, no_euler, &
     &           ntheta_octant, nphi_octant,iangle_octant, ichild,ienergy, &
     &           iphi, itheta,no_energies,itheta_new,iphi_new,idegeneracy, icount

      integer :: no_alpha_points, no_beta_points, no_gamma_points, no_theta_points, no_phi_points, &
     &           ialpha, ibeta, igamma
      real(kind=idp) :: d_alpha, d_beta, d_gamma, beta, a_norm, b_norm, g_norm, Aconst
      integer :: no_partial_waves, no_partial_waves_2, idistribution,lab_component,ienergy_step,  &
     &           ipara_or_perp_light,icalc_asymmetry_param
      real(kind=idp) :: phase_factor
!     Testing and timeing
      integer(kind=8) :: itime1, itime2, iclock_rate, ipartial
      real(kind=idp) :: time1, time2, test1
  

      
      namelist /DIPTRANS/ name,&
     &                    max_lcontinuum,&
     &                    no_targ_states,&
     &                    ngrdproj,&
     &                    ngrdalign,&
     &                    ibound_state,& 
     &                    first_IP,&
     &                    ipunits,&
     &                    iplane,&
     &                    degeneracy,&
     &                    molecule_name,&
     &                    euler_angle_limits,&
     &                    scat_angle_limits,& 
     &                    lebedev,&
     &                    luexpoints,&
     &                    idistribution,& 
     &                    lab_component,&
     &                    ienergy_step,&
     &                    experimental_IP, &
     &                    iuse_calculated_IP, &
     &                    degeneracy_threshold, &
     &                    lu_photo_xsec, &
     &                    lu_photo_dcs,  &
     &                    iorient_averaging,&
     &                    ipara_or_perp_light, &  !Set to 1 for parallel light, 2 for perpindicular light
     &                    phase_factor, &
     &                    icalc_asymmetry_param                        

!     Default namelist DIPTRANS values
      lu_photo_xsec=2220
      lu_photo_dcs=100
      max_lcontinuum=4 
      no_targ_states=4  !Including degenerate states.
      ibound_state=1
      ngrdproj=(/ 60,4 /)
      ngrdalign=(/ 1,1,1 /)
      thmax=pi;phmax=2*pi;almax=2*pi;bemax=pi;gamax=2*pi
      charge=1_idp
      ipunits=1
      iplane=1 !1: z-x plane, 2: z-y plane photodipoles. Need to request 4 phi points, e.g. ngrdproj=60,4
      iorient_averaging=0
      ipara_or_perp_light=0
      iuse_calculated_IP=0
      phase_factor=1._idp
      icalc_asymmetry_param=1
      first_IP=0.0_idp
      
      do itarget=1,max_states_for_dipoles
         !Defaults to no degeneracy
         degeneracy(itarget)=itarget
      
      end do

      molecule_name='mol'
      euler_angle_limits=(/ 0._idp,0._idp,0._idp,360._idp,180._idp,360._idp /)
      scat_angle_limits=(/ 0._idp,0._idp,180._idp,360._idp /)
      lebedev=0
      luexpoints=100
      idistribution=1
      lab_component=1 !  x=1, y=2, z=3
      ienergy_step=1
      experimental_IP=-1.0_idp
      degeneracy_threshold=1.d-5

      read(5,DIPTRANS,ERR=300, iostat=ierr)
  300 if (ierr .gt. 0) then
         write (6,3000)
      end if

      if(first_IP .eq. 0.0_idp) iuse_calculated_IP = 1

      if (iuse_calculated_IP .eq. 1) then
         first_IP= -(bound_state_energies(ibound_state)-target_energy)
      end if 

      euler_angle_limits=euler_angle_limits*pi/180._idp
      scat_angle_limits=scat_angle_limits*pi/180._idp


!     Determine target energies from channel energies
!     ------------------------------------------------------------------

      allocate(etarg(no_targ_states))
      etarg=0.0_idp
      itarget=1

      do i=1,size(evchl,2)
         itarget=ichl(dcomp2i(dipole_component_order(1)),i)
         if(itarget .ne. 0) etarg(itarget)=evchl(dcomp2i(dipole_component_order(1)),i)/2.0_idp
      end do

!     Check for target state degeneracy - Not fully implemented
      icount = 0
      do itarget=2,no_targ_states
         if( abs(etarg(itarget)-etarg(itarget-1)) .lt. degeneracy_threshold )  then
            icount=icount+1

         end if     

      end do

      write(6,*) ' Target state energies'
      do itarget=1,no_targ_states
         write(6,'(i5,D20.5)') itarget,etarg(itarget)
      end do

!     Construct angular grids these are the scattering angle grid,
!     the euler angle grid if quadrature methods are not being used,
!     the spherical harmonics grid, the rotation matrix grid and
!     the coulomb phase grid.
!     ------------------------------------------------------------------
      nt=ngrdproj(1); np=ngrdproj(2)
      na= ngrdalign(1); nb=ngrdalign(2); ng=ngrdalign(3)

      i=(max_lcontinuum+1)**2

      no_partial_waves=(max_lcontinuum+1)**2
      no_partial_waves_2=(2*max_lcontinuum+1)**2

      no_scattering_angles=nt*np

      if (lebedev .eq. 0) no_euler=na*nb*ng

      no_theta_points = nt
      no_phi_points   = np
      no_energies     = no_scattering_energies
      
      no_alpha_points = na
      no_beta_points  = nb
      no_gamma_points = ng

      if (lebedev .ne. 0) no_euler=size(lebedev_grid,2)

      write(6,*) 'NO. OF ORIENTATIONS =',no_euler

!     Construct angular grids
!     -----------------------

      allocate(th_grid(nt),ph_grid(np),a_grid(na),b_grid(nb),g_grid(ng))
      th_grid=0._idp;ph_grid=0._idp;a_grid=0._idp;b_grid=0._idp;g_grid=0._idp

      call grid_theta_phi(th_grid,ph_grid,scat_angle_limits)

      if(lebedev .eq. 0) then
         call grid_euler(a_grid,b_grid,g_grid,euler_angle_limits)
      end if

!     Construct (real) spherical harmonics     
!     ------------------------------------

      allocate(reSpH(nt*np,no_partial_waves),reRot(no_partial_waves,no_partial_waves,no_euler))
      reSpH=0._idp; reRot=0._idp

      call re_sph_grid(reSpH,th_grid,ph_grid,max_lcontinuum)

      if (lebedev .eq. 0) then
         call re_rot_mat_grid(reRot,a_grid,b_grid,g_grid,max_lcontinuum)
      end if

!     Construct $i^{-l}e^(i\sigma_l)$
!     ------------------------------- 

      allocate(coulomb_phase(no_partial_waves,no_scattering_energies,no_targ_states))
      coulomb_phase=0._idp

      do itarg=1,no_targ_states
         do ienergy=1,no_scattering_energies
            do l=0,max_lcontinuum
               do m=-l,l
                  if((escat(ienergy)-etarg(itarg)) .gt. 0) then

                     k_final=sqrt(2*(escat(ienergy)-etarg(itarg)))
                     eta=-charge/k_final
                     ipw=lm2i(l,m)
                     coulomb_phase(ipw,ienergy,itarg)=(-eye)**(l)* exp(eye*CPHAZ(l,eta,6))

                  end if
               end do     
            end do
         end do
      end do

!     Re-jig the moments matrix so that the partial wave index 
!     corresponds to that used by the spherical harmonics, coulomb phase
!     and rotation matrices.
!     ------------------------------------------------------------------

      allocate( dip_full(no_partial_waves,3,no_scattering_energies,no_targ_states,no_bound_states))
      dip_full=0_idp
      

      do en=1,no_scattering_energies
         do j=1,no_dipole_components
            idcmp=dcomp2i(dipole_component_order(j))

            do k=1,no_channels_scat_states(j)
               l=lvchl(idcmp,k)
               m=mvchl(idcmp,k)
              
!~  !              m=-mvchl(idcmp,k)  !TEST 

               ii=lm2i(l,m)

               do i=1,no_bound_states 
                  
                  dip_full(ii,idcmp,en,ichl(idcmp,k),i)= dip_elm(k,i,idcmp,en) !*(-1)**(l)

!~                   if (((l .eq. m)) .and. (m .ge.0) .and. ((-1)**m .eq. 1) ) dip_full(ii,idcmp,en,ichl(idcmp,k),i)= -dip_full(ii,idcmp,en,ichl(idcmp,k),i)

               end do
            end do

         end do
      end do
      

!~ dip_full(:,1,:,:,:)=0._idp
!~ dip_full(:,2,:,:,:)=0._idp
!~ dip_full(:,3,:,:,:)=0._idp

dip_full(:,1,:,:,:)=dip_full(:,1,:,:,:)!*(-eye)
dip_full(:,2,:,:,:)=dip_full(:,2,:,:,:)
dip_full(:,3,:,:,:)=dip_full(:,3,:,:,:)

!     -----------------------------------------------------------
!
!     Calculate the partial integral cross section using analytic
!     orientational averaging.
!
!     -----------------------------------------------------------
! TESTING -KILL the L=0 partial wave
!~ dip_full(1,:,:,:,:)=0._idp!-eye*dip_full(1,:,:,:,:)
!END TESTING

      allocate( total_xsec(no_scattering_energies), &
     &          partial_xsec(no_targ_states,no_scattering_energies))
      total_xsec=0._idp;partial_xsec=0

      do i=1, no_targ_states

         if (experimental_IP(1) .lt. 0) then
            ionization_pot=first_IP
            write(6,*) ' Calculated Ionization Potential is' ,ionization_pot+etarg(i), 'a.u.',  &
     &                 (ionization_pot+etarg(i))*27.211, 'eV'
         else
            ionization_pot=first_IP+(experimental_IP(i)-(first_IP+etarg(i)))
            write(6,*) ' Experimental Ionization Potential is' ,ionization_pot+etarg(i), 'a.u.',&
     &                  (ionization_pot+etarg(i))*27.211, 'eV'
         end if

         call calculate_orientation_averaged_total_cross_section(ibound_state, i, escat, ionization_pot,   &
     &                                                           dip_full, partial_xsec )

         call calculate_orientation_averaged_asymmetry_parameter(max_lcontinuum, &
     &                          ibound_state, i, etarg, degeneracy, escat, ionization_pot, dip_full)
   
         call test_print_partial_wave_cross_sections( 1000, max_lcontinuum, ibound_state, i, escat, &
     &                          etarg, ionization_pot, lvchl, mvchl, ichl, no_channels_scat_states, &
     &                          dipole_component_order, dip_full )
   
      end do

      allocate(cross_section(no_targ_states,no_scattering_energies))
      cross_section=0   
        
      do i=1, no_targ_states

!        Add cross sections from degenerate states.

         do itarg=1, no_targ_states
            if( degeneracy(i) .eq. degeneracy(itarg) ) then
               cross_section(i,:)=cross_section(i,:)+partial_xsec(itarg,:)
            end if
         end do

!        Write cross sections (should go in its own routine)

         do ienergy=1,no_energies
            photon_energy=escat(ienergy)+ionization_pot !-etarg(itarget)
            write(lu_photo_xsec +i,1010)  photon_energy*27.211,cross_section(i,ienergy)

         end do
         total_xsec=total_xsec+partial_xsec(i,:)
      end do

      do ienergy=1,no_energies
         photon_energy=escat(ienergy)+ionization_pot !-etarg(itarget)
         write(lu_photo_xsec,1010)  photon_energy*27.211,total_xsec(ienergy)

      end do      

!     ----------------------------------------------------------------
!
!     Calculate photo-(ionization/recombination) angular distributions
!
!     ----------------------------------------------------------------


     !no_components=size(dip_full,2)
      allocate( photo_dipoles( size(dip_full,2), no_scattering_angles, no_energies, no_targ_states) )
      allocate( temp_photo_dipoles( size(dip_full,2),no_scattering_angles, no_energies) )
      temp_photo_dipoles=0;photo_dipoles=0
      allocate(photo_dcs( size(dip_full,2), no_scattering_angles, no_energies, no_targ_states) )
      allocate( temp_photo_dcs( size(dip_full,2), no_scattering_angles, no_energies) )
      photo_dcs=0;temp_photo_dcs=0

      if (iorient_averaging .eq. 1) then
         allocate( photo_dcs_averaged( size(dip_full,2), no_scattering_angles, no_energies, no_targ_states) )
         photo_dcs_averaged=0

!        Figure out 

         d_alpha=1; d_beta=1; d_gamma=1
         a_norm=1; b_norm=1; g_norm=1

         if(no_alpha_points .gt. 1) then
            d_alpha=a_grid(2)-a_grid(1)
            a_norm=(2._idp*pi)
         end if
         if(no_beta_points .gt. 1) then
            d_beta= b_grid(2)-b_grid(1)
            b_norm=2._idp
         end if
         if(no_gamma_points .gt. 1) then 
            d_gamma=g_grid(2)-g_grid(1)
            g_norm=(2._idp*pi)
         end if

      end if

!    Calculate dipoles first then photoelectron angular distributions. 
      do ialpha=1,no_alpha_points
      do ibeta=1,no_beta_points
      do igamma=1,no_gamma_points

         ieuler=abg2i(ialpha, ibeta, igamma, no_beta_points, no_gamma_points)

         if(no_beta_points .gt. 1) then
!             beta=b_grid(ibeta)
            beta=sin(b_grid(ibeta))
         else
            beta=1 
         end if



         do iparent_state=1,no_bound_states
            do itarget=1, no_targ_states

!~                call calculate_photodipoles( reSpH, coulomb_phase, max_lcontinuum, iparent_state, &
!~      &                                      itarget, ieuler,dip_full, reRot, photo_dipoles )

               call calculate_photodipoles_efficiently( reSpH, coulomb_phase, max_lcontinuum, iparent_state, &
     &                                                  itarget, ieuler,dip_full, reRot, photo_dipoles )
     
!~                call calculate_photodipoles_more_efficiently( reSpH, coulomb_phase(:,:,itarget), max_lcontinuum, iparent_state, &
!~      &                                                       itarget, ieuler,dip_full(:,:,:,itarget,iparent_state), reRot, photo_dipoles(:,:,:,itarget) )
     
               call calculate_photo_dcs( etarg, itarget, escat, first_IP, experimental_IP,  &
     &                                   photo_dipoles, photo_dcs )

            end do

         end do

!     Sum angular distributions over degenerate states
!     ------------------------------------------------

!        Add cross sections from degenerate states.
         do itarget=1, no_targ_states
            temp_photo_dcs=0

            do itarg=1, no_targ_states
               if( degeneracy(itarget) .eq. degeneracy(itarg) ) then
                  temp_photo_dcs=temp_photo_dcs+photo_dcs(:,:,:,itarg)
               end if
            end do

            if (iorient_averaging .eq. 0) then

!              Write out the MFPAD for a single orientation
               if (ipara_or_perp_light .eq. 0) then
                  call write_oriented_photo_dcs(lu_photo_dcs,escat,ionization_pot,itarget,lab_component,th_grid,ph_grid,&
                                                temp_photo_dcs)
               else
                  call write_para_or_perp_dcs(lu_photo_dcs,escat,ionization_pot,itarget,lab_component,b_grid(ibeta),temp_photo_dcs)
               end if 

            else

!             Numerical orientational averaging.
!             ----------------------------------

               photo_dcs_averaged(:,:,:,itarget) = photo_dcs_averaged(:,:,:,itarget) &
     &                                           + temp_photo_dcs(:,:,:)             & 
     &                                           * beta*d_alpha*d_beta*d_gamma
!      &                                           *exp_alignment_distribution(ibeta)  

            end if

         end do
         
         
      end do
      end do
      end do

      allocate(asymmetry_param(no_targ_states,no_scattering_energies))
      asymmetry_param=0._idp
      
      allocate( photo_dcs_averaged_test( size(dip_full,2), no_scattering_angles, no_energies, no_targ_states) )
      photo_dcs_averaged_test=0      
      
      if (iorient_averaging .eq. 1) then
         photo_dcs_averaged=photo_dcs_averaged*(1._idp/(a_norm*b_norm*g_norm))

!TEST
         if (icalc_asymmetry_param .eq. 1) then
            !Assumes that the first theta, phi grid point in (0,0)
            do itarget=1, no_targ_states
               do ienergy=1, no_scattering_energies
                   photon_energy=escat(ienergy)+ionization_pot
                   if(partial_xsec(itarget,ienergy) .ne. 0.0_idp) then
                       asymmetry_param(itarget,ienergy)=&
                        4._idp*pi*photo_dcs_averaged(2,1,ienergy,itarget)/cross_section(itarget,ienergy)-1
                       write(lu_photo_dcs+itarget+100,*) photon_energy*27.211, asymmetry_param(itarget,ienergy)
                       
                   end if 
               end do
               
            end do
         
         end if
!END TEST
         do itarget=1, no_targ_states
            call write_oriented_photo_dcs(lu_photo_dcs,escat,ionization_pot,itarget,lab_component,th_grid,ph_grid,&
                                          photo_dcs_averaged(:,:,:,itarget))

         end do
      end if

!~ !     <TEST> 
!~ !     ----------------------------------------------------------------------------------------------
!~       d_theta=1;d_phi=1
!~       if(no_theta_points .gt. 1) d_theta=th_grid(2)-th_grid(1)
!~       if(no_phi_points .gt. 1)   d_phi=  ph_grid(2)-ph_grid(1)
      
!     Numerical orientationally averaged cross sections
!~       do itarget=1, no_targ_states
!~          do ienergy=1, no_energies
!~             cross_section_temp=0
!~             do itheta=1, no_theta_points
!~                do iphi=1,no_phi_points
!~                   iangle=tp2i(itheta,iphi,no_phi_points)
!~                   theta=th_grid(itheta)
!~                   cross_section_temp = cross_section_temp &
!~      &                               + photo_dcs_averaged(lab_component,iangle,ienergy,itarget) &
!~      &                               * sin(theta)*d_theta*d_phi
!~ 
!~ 			   end do
!~             end do
!~          
!~          photon_energy=escat(ienergy)+ionization_pot
!~          write(lu_photo_xsec+itarget+100,1010)  photon_energy*27.211,cross_section_temp           
!~          end do
!~          
!~       end do
!~ !     </TEST> 
!     ----------------------------------------------------------------------------------------------
!       call system_clock(itime1)
!       call system_clock(itime2, iclock_rate)
!       write(6,*) 'full run: time taken', real(itime2-itime1)/real(iclock_rate)

      return
      !---Format statements
 1000 format(6(E20.12,10x))
 1010 format(3(E20.5,1x))
 1020 format(8(E20.12,10x))
 3000 format(/,5X,'Problems reading the namelist DIPTRANS',/)
      end subroutine diptrans_drv
     
      integer function lspan(escat,gamma)
!     find  4*gamma (2*FWHM for lorentzian)
      implicit none
      real(kind=idp) :: gamma, deltaE
      real(kind=idp), dimension(:) :: escat
      
      deltaE=escat(2)-escat(1)
      lspan=(4*gamma)/deltaE

      end function lspan

      subroutine create_col_format(fstring,ncols,colfmt,ncolspc)
      implicit none
!     Creates a format statement, fstring,  with a definable number of
!     columns, ncols, column format, colfmt, and spacing, ncolspc. 
      integer, intent(in) :: ncols,ncolspc
      character(len=20), intent(inout) :: fstring
      character(len=6), intent(in) :: colfmt
!     local variables
      character(len=3) :: str_ncols,str_ncolspc

      write( str_ncols, '(i3)' ) ncols
      write( str_ncolspc, '(i3)' ) ncolspc
      fstring=str_ncols//'('//colfmt//','//str_ncolspc//'x)'

      write(6,*) fstring
      
      end subroutine create_col_format

      subroutine calculate_orientation_averaged_total_cross_section( iparent_state,itarget, &
     &                                                       escat,first_IP,dip_full,total_xsec )
      implicit none

!     Arguments
      integer :: iparent_state,itarget
      real(kind=idp), dimension(:) :: escat
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      real(kind=idp), dimension(:,:) ::total_xsec

!     Local variables
      integer :: ienergy, ipartial, no_energies, no_partials
      real(kind=idp) :: Aconst,pconst, cross_section_temp, echarge, photon_energy, ionization_pot, &
     &                  first_IP
      real(kind=idp), allocatable, dimension(:) :: cross_section 


      intent(in) :: iparent_state,itarget,escat,dip_full
      intent(inout) :: total_xsec

      echarge=1.0_idp

      no_energies=size(dip_full,3)
      no_partials=size(dip_full,1)

      allocate(cross_section(no_energies))
      cross_section=0.0_idp

!     The orientationally averaged partial cross section is given by the sum of the squares of the 
!     partial wave cross sections (summed over each photon polaristion)
!     --------------------------------------------------------------------------------------------

      do ienergy=1, no_energies
         cross_section_temp=0.0_idp


         do ipartial=1,no_partials

            cross_section_temp = cross_section_temp & 
     &                         + abs(dip_full(ipartial,1,ienergy,itarget,iparent_state))**2 &
     &                         + abs(dip_full(ipartial,2,ienergy,itarget,iparent_state))**2 &
     &                         + abs(dip_full(ipartial,3,ienergy,itarget,iparent_state))**2


         end do

         photon_energy=escat(ienergy)+first_IP
         Aconst=(4._idp/3._idp)*(pi**2)*alpha*photon_energy

         cross_section(ienergy)=cross_section_temp *Aconst*convert_au_to_megabarns


      end do


      total_xsec(itarget,:)=cross_section

      end subroutine calculate_orientation_averaged_total_cross_section

      subroutine calculate_photodipoles( reSpH, coulomb_phase, max_lcontinuum, iparent_state, &
     &                                  itarget, ieuler, dip_full, reRot, photo_dipoles)
      USE omp_lib
      implicit none

!     Arguments
      real(kind=idp), dimension(:,:) ::reSpH
      integer :: max_lcontinuum,iparent_state,itarget
      complex(kind=icp) :: coulomb_phase(:,:,:)

      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      real(kind=idp) :: reRot(:,:,:)

      complex(kind=icp) :: photo_dipoles(:,:,:,:)

!     Local variables
      integer ::  no_energies,no_scattering_angles, icount,ienergy, no_euler_angles
      integer :: iangle,no_components,no_total_channels, ifail


      real(kind=idp), allocatable :: rotated_real_spherical_harmonics(:,:)
      complex(kind=icp),allocatable :: zrotated_real_spherical_harmonics(:,:)
      complex(kind=icp),allocatable ::temp_photo_dipoles(:), CPreSpH_temp(:,:), CPreSpH_temp2(:)


      integer ::  ipartial,no_partial_waves,no_targ_states, itarg, ien, ieuler, ienergy_step, ipw

      real(kind=idp) :: reRotl1(3,3)
real(kind=idp) ::  time1,time2
!~ real(kind=idp), allocatable :: testmat(:,:,:)


      intent(in) :: reSpH, coulomb_phase, max_lcontinuum, iparent_state,itarget, dip_full,reRot
      intent(inout) :: photo_dipoles


!     figure out partial wave channels involved
      no_components=size(dip_full,2)
      no_targ_states=size(dip_full,4)
      no_energies=size(dip_full,3)

      no_partial_waves=size(reRot,2)
      no_scattering_angles=size(reSpH,1)
      no_euler_angles=size(reRot,3)
  
!~ allocate(testmat(no_components,no_scattering_angles,no_energies))

!     Transform from partial wave to momentum basi
!     ------------------------------------------------------------------
!       print*, ' Transform from partial wave to momentum basis'

      allocate( CPreSpH_temp(1,no_partial_waves),  STAT = ifail )
      allocate(CPreSpH_temp2(no_partial_waves))

!       reRotl1=transpose(reRot(2:4,2:4,ieuler)) !For rotating the vector dipole operator
      reRotl1=reRot(2:4,2:4,ieuler) !For rotating the vector dipole operator

      allocate( rotated_real_spherical_harmonics(no_scattering_angles,no_partial_waves) )
!~       rotated_real_spherical_harmonics=0._idp

!     New code version gives correct Euler angle order.
      call gemm(reSpH,reRot(:,:,ieuler),rotated_real_spherical_harmonics(:,:))

!     Old code version... (for testing -)      
!~       do iangle=1,no_scattering_angles
!~          rotated_real_spherical_harmonics(iangle,:)=matmul(reRot(:,:,ieuler), reSpH(iangle,:))
!~       end do
!~       

      allocate(temp_photo_dipoles(3))

! <TEST> TIMING      
call cpu_time(time1)
time1= omp_get_wtime()
! </TEST>
      
      do ienergy=1,no_energies!,ienergy_step
         do iangle=1,no_scattering_angles

!           Construct angular factor containing sperical harmonic, coulomb phase and normalisation

            do ipw = 1, no_partial_waves
               CPreSpH_temp(1,ipw) = rotated_real_spherical_harmonics(iangle,ipw) &
     &                             * coulomb_phase(ipw,ienergy,itarget)
!~ CPreSpH_temp2(ipw)=CPreSpH_temp(1,ipw)
            end do



            call gemv(dip_full(:,:,ienergy,itarget,iparent_state),CPreSpH_temp(1,:),&
     &                photo_dipoles(:,iangle,ienergy,itarget),trans='T')

            
!~             photo_dipoles(:,iangle,ienergy,itarget)=matmul(CPreSpH_temp2,dip_full(:,:,ienergy,itarget,iparent_state))

            temp_photo_dipoles(:)=matmul( reRotl1(:,:) ,photo_dipoles(:,iangle,ienergy,itarget))
!             call gemv( reRotl1(:,:),photo_dipoles(:,iangle,ienergy,itarget), temp_photo_dipoles)
            photo_dipoles(:,iangle,ienergy,itarget)=temp_photo_dipoles


         end do
      end do
      
! <TEST> TIMING 
call cpu_time(time2)
time2= omp_get_wtime() 
write(654,*) 'Energy and angle loop', time2-time1 
 ! </TEST>
 
      end subroutine calculate_photodipoles
      
      subroutine calculate_photodipoles_efficiently( reSpH, coulomb_phase, max_lcontinuum, & 
     &                              iparent_state, itarget, ieuler, dip_full, reRot, photo_dipoles )
      USE omp_lib
      implicit none

!     Arguments
      real(kind=idp), dimension(:,:) ::reSpH
      integer :: max_lcontinuum,iparent_state,itarget
      complex(kind=icp) :: coulomb_phase(:,:,:)

      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      real(kind=idp) :: reRot(:,:,:)

      complex(kind=icp) :: photo_dipoles(:,:,:,:)

!     Local variables
      integer ::  no_energies,no_scattering_angles, icount,ienergy, no_euler_angles
      integer :: iangle,no_components,no_total_channels, ifail


      real(kind=idp), allocatable :: rotated_real_spherical_harmonics(:,:)
      complex(kind=icp),allocatable :: zrotated_real_spherical_harmonics(:,:)
      complex(kind=icp),allocatable ::temp_photo_dipoles(:,:), CPreSpH_temp(:,:), CPreSpH_temp2(:)

      integer ::  ipartial,no_partial_waves,no_targ_states, itarg, ien, ieuler, ienergy_step, ipw

      real(kind=idp) :: reRotl1(3,3)
!~ complex(kind=icp) :: reRotl1(3,3)
!~ real(kind=idp) ::  time1,time2
!~ real(kind=idp), allocatable :: testmat(:,:,:)

      intent(in) :: reSpH, coulomb_phase, max_lcontinuum, iparent_state,itarget, dip_full,reRot
      intent(inout) :: photo_dipoles

      no_components=size(dip_full,2)
      no_targ_states=size(dip_full,4)
      no_energies=size(dip_full,3)

      no_partial_waves=size(reRot,2)
      no_scattering_angles=size(reSpH,1)
      no_euler_angles=size(reRot,3)
  

!     Transform from MOL frame partial wave to LAB frame momentum basis
!     -----------------------------------------------------------------
      allocate( CPreSpH_temp(no_scattering_angles,no_partial_waves),  STAT = ifail )

!       reRotl1=transpose(reRot(2:4,2:4,ieuler)) !For rotating the vector dipole operator
      reRotl1=reRot(2:4,2:4,ieuler) !For rotating the vector dipole operator

      allocate( rotated_real_spherical_harmonics(no_scattering_angles,no_partial_waves) )

!     Rotate spherical harmonics from LAB to MOL frame
      call gemm(reSpH,reRot(:,:,ieuler),rotated_real_spherical_harmonics(:,:))
  
!~       allocate(temp_photo_dipoles(no_components,no_scattering_angles))
!~  call cpu_time(time1)
!~  time1= omp_get_wtime()    
      do ienergy=1,no_energies!,ienergy_step

!           Construct angular factor containing sperical harmonic and coulomb phase
            do ipw = 1, no_partial_waves
               CPreSpH_temp(:,ipw) = rotated_real_spherical_harmonics(:,ipw) & 
     &                             * coulomb_phase(ipw,ienergy,itarget)
               
            end do

!           Multiply partial wave dipoles by spherical harmonics and coulomb phase            
!~             call gemm(dip_full(:,:,ienergy,itarget,iparent_state),CPreSpH_temp(:,:),&
!~      &                temp_photo_dipoles(:,:),transa='T', transb='T')
     
            call gemm(dip_full(:,:,ienergy,itarget,iparent_state),CPreSpH_temp(:,:),&
     &                photo_dipoles(:,:,ienergy,itarget),transa='T', transb='T')   !original version 
     

!           Rotate dipole components (photon polarisation) from MOL to LAB frame          
!~             call gemm(reRotl1(:,:) ,temp_photo_dipoles(:,:),photo_dipoles(:,:,ienergy,itarget))
            photo_dipoles(:,:,ienergy,itarget)=matmul( reRotl1(:,:) ,photo_dipoles(:,:,ienergy,itarget))
     
      end do
      
!~  call cpu_time(time2)
!~  time2= omp_get_wtime() 
!~  write(654,*) 'Energy loop only', time2-time1  

      end subroutine calculate_photodipoles_efficiently

      subroutine calculate_photodipoles_more_efficiently( reSpH, coulomb_phase, max_lcontinuum, iparent_state, &
     &                                               itarget, ieuler, dip_full, reRot, photo_dipoles)
      USE omp_lib
      implicit none

!     Arguments
      real(kind=idp), dimension(:,:) ::reSpH
      integer :: max_lcontinuum,iparent_state,itarget
      complex(kind=icp) :: coulomb_phase(:,:)

      complex(kind=icp), dimension(:,:,:)  :: dip_full
      real(kind=idp) :: reRot(:,:,:)

      complex(kind=icp) :: photo_dipoles(:,:,:)

!     Local variables
      integer ::  no_energies,no_scattering_angles, icount,ienergy, no_euler_angles
      integer :: iangle,no_components,no_total_channels, ifail


      real(kind=idp), allocatable :: rotated_real_spherical_harmonics(:,:)
      complex(kind=icp),allocatable :: zrotated_real_spherical_harmonics(:,:)
      complex(kind=icp),allocatable ::temp_photo_dipoles(:), CPreSpH_temp(:,:), CPreSpH_temp2(:)


      integer ::  ipartial,no_partial_waves,no_targ_states, itarg, ien, ieuler, ienergy_step, ipw

      real(kind=idp) :: reRotl1(3,3)
real(kind=idp) ::  time1,time2
!~ real(kind=idp), allocatable :: testmat(:,:,:)


      intent(in) :: reSpH, coulomb_phase, max_lcontinuum, iparent_state,itarget, dip_full,reRot
      intent(inout) :: photo_dipoles


!     figure out partial wave channels involved
      no_components=size(dip_full,2)
      no_energies=size(dip_full,3)

      no_partial_waves=size(reRot,2)
      no_scattering_angles=size(reSpH,1)
      no_euler_angles=size(reRot,3)
  
!~ allocate(testmat(no_components,no_scattering_angles,no_energies))

!     Transform from partial wave to momentum basi
!     ------------------------------------------------------------------
!       print*, ' Transform from partial wave to momentum basis'

      allocate( CPreSpH_temp(no_scattering_angles,no_partial_waves),  STAT = ifail )


!       reRotl1=transpose(reRot(2:4,2:4,ieuler)) !For rotating the vector dipole operator
      reRotl1=reRot(2:4,2:4,ieuler) !For rotating the vector dipole operator

      allocate( rotated_real_spherical_harmonics(no_scattering_angles,no_partial_waves) )
!~       rotated_real_spherical_harmonics=0._idp

!     New code version gives correct Euler angle order.
      call gemm(reSpH,reRot(:,:,ieuler),rotated_real_spherical_harmonics(:,:))

!     Old code version... (for testing -)      
!~       do iangle=1,no_scattering_angles
!~          rotated_real_spherical_harmonics(iangle,:)=matmul(reRot(:,:,ieuler), reSpH(iangle,:))
!~       end do
!~       

      allocate(temp_photo_dipoles(3))
 call cpu_time(time1) 
   time1= omp_get_wtime()     
      do ienergy=1,no_energies!,ienergy_step

!           Construct angular factor containing sperical harmonic, coulomb phase and normalisation

            do ipw = 1, no_partial_waves
               CPreSpH_temp(:,ipw)=rotated_real_spherical_harmonics(:,ipw)*coulomb_phase(ipw,ienergy)
               
            end do
            
            call gemm(dip_full(:,:,ienergy),CPreSpH_temp(:,:),&
     &                photo_dipoles(:,:,ienergy),transa='T', transb='T')
     
            photo_dipoles(:,:,ienergy)=matmul( reRotl1(:,:) ,photo_dipoles(:,:,ienergy))
     
      end do
      
 call cpu_time(time2)
   time2= omp_get_wtime() 
 write(654,*) 'Energy loop only 2', time2-time1  

      end subroutine calculate_photodipoles_more_efficiently
            
      subroutine calculate_photo_dcs(etarg, itarget, escat, first_IP, experimental_IP,&
     &                               photo_dipoles, photo_dcs)
      implicit none
 
!     Argument variables
      integer :: itarget
      real(kind=idp) :: first_IP, escat(:),etarg(:),experimental_IP(:),photo_dcs(:,:,:,:)
      complex(kind=icp) :: photo_dipoles(:,:,:,:)

!     Local variables
      integer ::  no_target_states, no_components,no_energies, no_scattering_angles, &
     &            ienergy,  iangle, icomponent
      real(kind=idp) :: Aconst,pconst, echarge, photon_energy, ionization_potential

      intent(in) :: etarg, itarget,escat,photo_dipoles,first_IP, experimental_IP
      intent(inout) :: photo_dcs

      if (experimental_IP(1) .lt. 0) then
         ionization_potential=first_IP
      else
         ionization_potential=first_IP+(experimental_IP(itarget)-(first_IP+etarg(itarget)))
      end if

      echarge=1.0_idp
      
      no_components=size(photo_dipoles,1)
      no_scattering_angles=size(photo_dipoles,2)
      no_energies=size(photo_dipoles,3)
      no_target_states=size(photo_dipoles,4)
     
      do  ienergy=1, no_energies

         photon_energy=escat(ienergy)+ionization_potential
         Aconst=4*(pi**2)*alpha*photon_energy*convert_au_to_megabarns

         do icomponent=1,no_components
            do iangle=1,no_scattering_angles
                  photo_dcs(icomponent,iangle,ienergy,itarget)=abs(photo_dipoles(icomponent,iangle,ienergy,itarget))**2 * Aconst
 

            end do

         end do

      end do
   
      return
 1010 format(3(E20.5,1x))
      end subroutine calculate_photo_dcs
    
      subroutine gaussian_smooth( dip_elm, ichl, evchl, lvchl, mvchl, nchans, escat, nesc, nbound, &
     &                            ndcomp, dcomp, lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, ifail)
      use interpolate
      use dipelmdefs, only: maxprop_par
      use photo_outerio, only: read_pw_dipoles, write_pw_dipoles
      implicit none
!      type(zbspline) :: interp_dip

      integer :: lu_pw_dipoles(maxprop_par), nset_pw_dipoles(maxprop_par)
      character(len=11) :: format_pw_dipoles
      integer :: ifail, nesc, nbound,ndcomp
      integer,dimension(:) :: nchans(3)
      integer, dimension(:,:) ::ichl,lvchl,mvchl
      real(kind=idp), dimension(:,:) :: evchl
      real(kind=idp), dimension(:) :: escat
      real(kind=idp), allocatable, dimension(:) :: escat2, dip_re(:), dip_im(:)
      complex(kind=icp), dimension(:,:,:,:)  :: dip_elm
      complex(kind=icp), allocatable, dimension(:,:,:,:)  :: dip_elm_tmp
      character(len=1), dimension(:) :: dcomp(3)
      integer,allocatable, dimension(:,:) :: limits
      real(kind=idp) :: E,deltax,start_integration, finish_integration, energy
      complex(kind=icp) :: dipole
      integer :: ieleft,i,j,k,m,n,ien, ierr,idcmp,ie_chan_left, iwrite, iprint
      type(zspline), allocatable, dimension(:,:,:) :: zinterpolated_dipole

!test
real(kind=idp) :: pconst

!     namelist variables
      integer :: ismooth,pol_order
      real(kind=idp) ::Eleft,Ewidth,deltaE
      character(len=11) :: format_smoothed_pw_dipoles
      integer :: lu_smoothed_pw_dipoles(maxprop_par), nset_smoothed_pw_dipoles(maxprop_par)
      namelist /SMOOTH/ Eleft, Ewidth, ismooth, deltaE, pol_order,&
     &                  lu_smoothed_pw_dipoles, nset_smoothed_pw_dipoles, format_smoothed_pw_dipoles
!     Default namelist values
      Eleft=0.1        !Start point for smoothing
      Ewidth= 0.07_idp !Width of smoothing filter
      deltaE=0.001     !Integration step size
      ismooth=1        !1=Smooth;0=Don't smooth 
      pol_order=5
      iwrite=6
      iprint =1
      lu_smoothed_pw_dipoles=0       
      nset_smoothed_pw_dipoles=nset_pw_dipoles
      format_smoothed_pw_dipoles=format_pw_dipoles
      
      read(5,SMOOTH,ERR=200, iostat=ierr)

  200 if (ierr .gt. 0) then
         write (iwrite,4000)
         ifail=4000
         return
      end if
 

      select case(ismooth)
      case(0)
      case(1) !Gaussian smoothing - No interpolation
      
!        Method without interpolation, requires a fine energy grid to be
!        good
!        ---------------------------------------------------------------
          write(iwrite,'(" Gaussian smoothing of partial wave dipoles ")')
 !        Find integration limits in terms of energy index
          call  sigma2i(E,Eleft,escat,escat2,ieleft,limits)
 !          do i=1,nesc
 !             print*, limits(1,i),limits(2,i)
 !          end do
 !          print *,dip_elm      
          i=size(dip_elm,1);j=size(dip_elm,2);k=size(dip_elm,3);m=size(dip_elm,4)

          allocate(dip_elm_tmp(i,j,k,m))
 
          dip_elm_tmp=dip_elm
          dip_elm_tmp(:,:,:,ieleft:nesc)=0_idp
          deltax=escat(2)-escat(1)
 
 !        Perfrom integration
          do i=1,nbound
             do j=1,ndcomp
                idcmp=dcomp2i(dcomp(j))
  
                do k=1,nchans(j)
                   call binary_search(escat,evchl(idcmp,k)/2._idp,0,ie_chan_left)
                   
                   do ien=ieleft,nesc
                      !do n=limits(1,ien),limits(2,ien)
                      Ewidth=sqrt((escat(ien)-escat(ieleft))/(escat(nesc)-escat(ieleft))) !Energy dependent width
                      Ewidth=ewidth/10.0_idp +0.0001                                      !Energy dependent width
                      dipole=0
                      do n=1,nesc
                         
                         if(n .lt. ieleft) then !Less than lowest smoothing point
                            dipole = dipole + gaussian(escat2(ien),escat2(n), Ewidth) * dip_elm(k,i,idcmp,ieleft) * deltax
                            !dip_elm_tmp(k,i,idcmp,ien)=dip_elm_tmp(k,i,idcmp,ien)+lorentzian(escat(ien),escat(n),Ewidth)*dip_elm(k,i,idcmp,n)*deltax
                            
                         else if (escat2(n) .lt. evchl(idcmp,k)/2._idp) then !less than channel threshold  
                            dipole = dipole + gaussian(escat2(ien),escat2(n), Ewidth) * dip_elm(k,i,idcmp,ie_chan_left) * deltax

                         else
                            dipole = dipole + gaussian(escat2(ien),escat2(n), Ewidth) * dip_elm(k,i,idcmp,n) * deltax
                            
                         end if
                      end do
                      
                      if (escat2(ien) .ge. evchl(idcmp,k)/2._idp) then
                         dip_elm(k,i,idcmp,ien)=dipole
                      
                      else
                         dip_elm(k,i,idcmp,ien)=0._idp
                      
                      end if
                      
                   end do
                   
                end do
                 
             end do
                
          end do
!~           dip_elm=dip_elm_tmp
          
      case(2) !Gaussian smoothing - With interpolation
      
!       Method using interpolated dipoles
!       ------------------------------------------------------------------

        write(iwrite,'(" Gaussian smoothing of partial wave dipoles - using interpolated dipoles")')     
        allocate(zinterpolated_dipole(size(dip_elm,1),size(dip_elm,2),size(dip_elm,3)))
        allocate(dip_elm_tmp(size(dip_elm,1),size(dip_elm,2),size(dip_elm,3),size(dip_elm,4)))
        dip_elm_tmp=0.0_idp

        do i=1,nbound
           do j=1,ndcomp
              idcmp=dcomp2i(dcomp(j))!old ordering
              do k=1,nchans(j)
                 call bspline_init(escat,dip_elm(k,i,idcmp,:),zinterpolated_dipole(k,i,idcmp),8) !old ordering
              end do
           end do
        end do


!     Method using interpolated dipoles
!     ------------------------------------------------------------------
      call  sigma2i(E,Eleft,escat,escat2,ieleft,limits)
      print *, "IELEFT=",ieleft

         do i=1,nbound
            do j=1,ndcomp
               idcmp=dcomp2i(dcomp(j)) !old ordering
!~                print *, "idcmp",j, idcmp,nchans(idcmp)
               write(iwrite, '("Applying gaussian smoothing to component ", a1)')  dcomp(j)
               call binary_search(escat,Eleft,0,ieleft)
               do k=1,nchans(j)
                  
!~                   if (evchl(idcmp,k) .ge. Eleft) then
!~                     call binary_search(escat,evchl(idcmp,k)/2._idp,0,ieleft)
!~                     write(6,'("IELEFT CHAN= ", 3i6, 2d20.5)') idcmp, k, ieleft,evchl(idcmp,k), escat(ieleft)
!~                   else
!~                     call binary_search(escat,Eleft,0,ieleft)
!~                   
!~                   end if

                  !$OMP PARALLEL DEFAULT(NONE) PRIVATE(ien,start_integration,finish_integration,energy,ie_chan_left,dipole) &
                  !$OMP SHARED(ieleft,nesc,ewidth,escat,evchl,idcmp,i,j,k,zinterpolated_dipole,deltaE,dip_elm_tmp)
                  !$OMP DO
                  do ien=ieleft,nesc
                     
                     start_integration=escat(ien)-3*Ewidth
                     finish_integration=escat(ien)+3*Ewidth
                     energy=start_integration
                     call binary_search(escat,evchl(idcmp,k)/2._idp,0,ie_chan_left)
                     dipole=0.0_idp

                     do while(energy .le. finish_integration)

                        if(energy .lt. escat(ieleft)) then !Less than lowest smoothing point
!                            dipole=dipole +lorentzian(energy,escat(ien),eEwidth)*bspline_int(escat(ieleft),zinterpolated_dipole(k,i,idcmp))*deltaE
                           dipole=dipole +gaussian(energy,escat(ien),Ewidth)&
                                         *bspline_int(escat(ieleft),zinterpolated_dipole(k,i,idcmp))*deltaE
                           
                        else if (energy .lt. evchl(idcmp,k)/2._idp) then !less than channel threshold
                        
                           dipole=dipole +gaussian(energy,escat(ien),Ewidth)&
                                         *bspline_int(escat(ie_chan_left),zinterpolated_dipole(k,i,idcmp))*deltaE
!~                            dipole=dipole +gaussian(energy,escat(ien),Ewidth)*bspline_int(evchl(idcmp,k)/2._idp,zinterpolated_dipole(k,i,idcmp))*deltaE
                           
                        else if (energy .gt. escat(nesc)) then !greater than highest energy
!                            dipole=dipole +lorentzian(energy,escat(ien),Ewidth)*bspline_int(escat(nesc),zinterpolated_dipole(k,i,idcmp))*deltaE
                           dipole=dipole +gaussian(energy,escat(ien),Ewidth)&
                                         *bspline_int(escat(nesc),zinterpolated_dipole(k,i,idcmp))*deltaE
                           
                        else
!                            dipole=dipole +lorentzian(energy,escat(ien),Ewidth)*bspline_int(energy,zinterpolated_dipole(k,i,idcmp))*deltaE
                           dipole=dipole +gaussian(energy,escat(ien),Ewidth)&
                                         *bspline_int(energy,zinterpolated_dipole(k,i,idcmp))*deltaE
                           
                        end if
                        energy=energy+deltaE

                     end do
                     
                     if (escat(ien) .ge. evchl(idcmp,k)/2._idp) then
                        dip_elm_tmp(k,i,idcmp,ien)=dipole
                        
                     else
                        dip_elm_tmp(k,i,idcmp,ien)=0._idp
                        
                     end if
                     
                  end do
                  !$OMP END DO
                  !$OMP END PARALLEL

               end do

            end do

         end do
         dip_elm=dip_elm_tmp

      case(3) ! Least squares fit
        write(iwrite,'(" Least squares fit to partial wave dipoles, polynomial order = ", i3)') pol_order

         if (Eleft .lt. 0.0_idp) then
! !        default case         
!          do ien=1,ienergy_last
!             if(test_matrix(ien) .gt. 0._idp) then
!                ieleft=ien
!                energy_first=escat(ieleft)
!                exit
!             end if
!          end do
         else
         !user set initial energy
            do ien=1,nesc
               if(escat(ien) .ge. Eleft ) then
                  ieleft=ien
                  Eleft=escat(ieleft)
                  exit
               end if
            end do
         end if

         allocate(dip_re(nesc), dip_im(nesc))
         dip_re=0._idp;dip_im=0._idp
         do i=1,nbound
            do j=1,ndcomp
               idcmp=dcomp2i(dcomp(j))

               do k=1,nchans(j)
                  call binary_search(escat,evchl(idcmp,k)/2._idp,0,ie_chan_left)
                  dip_re=real(dip_elm(k,i,idcmp,:))
                  dip_im=aimag(dip_elm(k,i,idcmp,:))
                  call least_squares(pol_order,ieleft,nesc,escat,dip_re)
                  call least_squares(pol_order,ieleft,nesc,escat,dip_im)
                  dip_elm(k,i,idcmp,ie_chan_left:) = cmplx(dip_re(ie_chan_left:),dip_im(ie_chan_left:),kind=icp)   
                  
               end do
            end do
         end do


      end select

      if (lu_smoothed_pw_dipoles(1) .ne. 0) then
         call write_smoothed_pwd_dipoles( lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, &
     &                                    lu_smoothed_pw_dipoles, nset_smoothed_pw_dipoles,  &
     &                                    format_smoothed_pw_dipoles, dip_elm )

      end if
      
      
      return
        !---Format statements
 4000 format(/,5X,'Problems reading the namelist SMOOTH',/)
      end subroutine gaussian_smooth
      
      subroutine least_squares(pol_order,istart_point,iend_point,x_points,y_points)
      use dipelmdefs
      implicit none

!     Arguments
      integer ::  pol_order,istart_point,iend_point
      real(kind=idp) :: x_points(:),y_points(:)

!     Local variables
      integer :: i, info , j, number_of_points,lwork
      real(kind=idp), allocatable :: YY(:), XX(:,:), WORK(:),X0(:,:)
      real(kind=idp) :: valx,valy,aux,coeff

      intent(in) :: istart_point,iend_point, x_points
      intent(inout) :: y_points


!       EXTERNAL         DGELS

      ! Pol order + 1 (coefficients)
!       pol_order = 3

      number_of_points=iend_point-istart_point+1

      allocate(XX(number_of_points,pol_order),YY(number_of_points),X0(number_of_points,pol_order))

      YY=y_points(istart_point:iend_point)

!     weight the first and the last point 
      do i=1,number_of_points

         if ((i==1).or.(i==number_of_points)) then
            coeff = 100._idp
         else
            coeff = 1._idp
         endif
      
         YY(i) = coeff*YY(i)
         do j=1,pol_order
            X0(i,j) = coeff*x_points(istart_point+i-1)**(j-1)
         enddo

      enddo

      lwork = 2*number_of_points*pol_order
      allocate(WORK(lwork))

!
!     Solve the least squares problem min( norm2(b - Ax) ) for x
!
      XX= X0
      CALL DGELS('n',number_of_points,pol_order,1,XX,number_of_points,YY,number_of_points,WORK,LWORK,INFO)

!        WRITE (6,*) 'Least squares solution'
!        WRITE (6,*) (YY(I),I=1,pol_order)

      do i=1,number_of_points
         aux=0.d0
         do j=1,pol_order
            aux = aux + (YY(j))*x_points(istart_point+i-1)**(j-1)
         end do
         y_points(istart_point+i-1)=aux
      end do


      return

      END subroutine least_squares

      !Subroutine to create gaussian smoothing function
      !*************************************************************
      real(kind=idp)function gaussian(E,Ep,sigma_E)
      use dipelmdefs
      real(kind=idp) :: sigma_E,E,Ep, peak_amp

!       sigma_E=sigma(E)
      peak_amp=(1.0_idp/(sqrt(2.0_idp*pi)*sigma_E))
      gaussian= peak_amp*exp(-(E-Ep)**2/(2.0_idp*sigma_E**2))

      end function gaussian
 
      real(kind=idp) function sigma(E)
      use dipelmdefs
      implicit none
      real(kind=idp) :: E, sigma_tmp

      !sigma=(1.0_idp/3.0_idp)*sqrt(1.0_idp/(2.0_idp*sqrt(E)))
      sigma= 0.3_idp

      end function sigma 


      real(kind=idp)function lorentzian(E,Ep,gamma)
      use dipelmdefs
      real(kind=idp) :: gamma,E,Ep, peak_amp

      peak_amp=gamma/pi
      lorentzian= peak_amp/((E-Ep)**2+gamma**2)

      end function lorentzian

      subroutine sigma2i(E,Estart,escat,escat2,is,limits)
      implicit none
      real(kind=idp) ::E,Estart,three_sigma, low_lim,upp_lim
      integer :: en,i,j, nesc, is
      integer, allocatable,dimension(:,:) :: limits
      real(kind=idp), dimension(:) :: escat
      real(kind=idp), allocatable, dimension(:) :: escat2

!     written for elastic scattering first

!     1. Find the first energy point above the starting energy
!        This is now the zero energy point.
!     2. Create new escat array with new energy zero.
!     3. For each point above zero point 
!        a. Find 3sigma
!        b. Find nearest energy point to E- 3*sigma
!        c. Find nearest energy point to e+ 3*sigma
!        Note for zero point 
!     4. Save to array integ_lims

!     Find the first element by binary search
      nesc=size(escat)
      allocate(escat2(nesc), limits(2,nesc))
      escat2=0.0_idp;limits=0

      call binary_search(escat,Estart,0,is)
      print*, "Left matching point for smoothing is", escat(is)
      escat2=escat-escat(is)

      limits(1,is)=is
      limits(2,is)=is
      do en=is+1,nesc
         three_sigma=3*sigma(escat2(en))
         !print*, escat2(en),three_sigma, escat(2)-escat(1)
         low_lim=escat2(en)-three_sigma
         upp_lim=escat2(en)+three_sigma
         !print*,"lu", low_lim, upp_lim
         call binary_search(escat2,low_lim,0,limits(1,en))
         call binary_search(escat2,upp_lim,1,limits(2,en))
      end do

      end subroutine sigma2i

      subroutine binary_search(arr,val,lr,i)
      implicit none
!     returns right most value
      real(kind=idp), dimension(:) :: arr
      real(kind=idp) ::val
      integer :: i,lr
      integer :: lp,rp, mp

      lp=1;rp=size(arr)
      
      do while((rp-lp) .gt. 1)
         mp=(lp+rp)/2
         if (val .gt. arr(mp)) then
            lp=mp
         else
            rp=mp
         end if
      end do
      select case (lr)
      case(0) !First value greater or equal to val
         if (abs(val-arr(lp)) .lt. small_int) then 
            i=lp
         else
            i=rp
         end if
      case(1) !First val less than or equal to val 
         if (abs(val-arr(rp)) .lt. small_int) then 
            i=rp
         else
            i=lp
         end if
      end select

      end subroutine binary_search

      subroutine calculate_orientation_averaged_asymmetry_parameter(maxpw,iparent_state,ichild_state,etarg,degeneracy,escat,potionz&
                                                                    ,dip_full)
      implicit none
!     Modified by ZM to include calculation of the beta_1 parameter.
!     Argument variables
      integer :: iparent_state,ichild_state, maxpw
      real(kind=idp) :: potionz
      real(kind=idp), dimension(:) :: escat
      real(kind=idp), dimension(:) :: etarg
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      real(kind=idp), dimension(:) :: degeneracy

!     Local variables
      integer :: no_energies, no_partials, Lbig, ienergy, lu_beta1, lu_beta2, lu_beta0
      real(kind=idp) :: photon_energy, Aconst,echarge,ionization_pot, val
      complex(kind=idp), allocatable, dimension(:) :: AL_zero, AL_two, AL_one
      complex(kind=idp),allocatable,  dimension(:) :: beta_parameter
      character(len=132) :: path
      character(len=5) :: cationic_state, parent_state

!     Testing
      integer :: iequation

      iequation=1

      ionization_pot=potionz
      echarge=1.0_idp

      no_energies=size(dip_full,3)
      no_partials=size(dip_full,1)

      allocate (AL_zero(no_energies), AL_two(no_energies),beta_parameter(no_energies), AL_one(no_energies))
      AL_zero=0._idp
      AL_two=0._idp
      AL_one=0._idp
      beta_parameter=0._idp

      if (iequation.eq.0) then
!~          call dcs_legendre_pol_expansion_coefficient(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 2, AL_two)
!~          call dcs_legendre_pol_expansion_coefficient(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 0, AL_zero)
      elseif (iequation.eq.1) then
!         call make_burke_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 0, AL_zero) 
!         call make_ritchie_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 1, 1, AL_one)
!         call make_burke_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 2, AL_two)

!For Al=0,2 the Ritchie routines are equivalent to the Burke routines.
         call make_ritchie_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 0, 0, AL_zero)
         call make_ritchie_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 1, 1, AL_one)
         call make_ritchie_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 2, 0, AL_two)
      end if

      write(cationic_state,'(i5)') ichild_state
      write(parent_state,'(i5)') iparent_state

      path = 'beta_0_cationic_state_'//trim(adjustl(cationic_state))//'_parent_state_'//trim(adjustl(parent_state))
      open(newunit=lu_beta0,file=path,status='replace',form='FORMATTED')
      path = 'beta_1_cationic_state_'//trim(adjustl(cationic_state))//'_parent_state_'//trim(adjustl(parent_state))
      open(newunit=lu_beta1,file=path,status='replace',form='FORMATTED')
      path = 'beta_2_cationic_state_'//trim(adjustl(cationic_state))//'_parent_state_'//trim(adjustl(parent_state))
      open(newunit=lu_beta2,file=path,status='replace',form='FORMATTED')

      do ienergy=1,no_energies

         photon_energy=escat(ienergy)+potionz
         Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
         
         if (abs(AL_zero(ienergy)) .ne. 0) then
            beta_parameter(ienergy)= AL_two(ienergy)/AL_zero(ienergy)
            write(lu_beta2,'(10e20.5)') photon_energy*27.211, real(beta_parameter(ienergy),kind=idp)!, real(AL_two(ienergy)),Aconst*real(AL_zero(ienergy))

            val = real(AL_one(ienergy)/AL_zero(ienergy),kind=idp)
            write(lu_beta1,'(10e20.5)') photon_energy*27.211, val

            val = real(AL_zero(ienergy),kind=idp)*Aconst !Normalize to produce the integral cross section
            write(lu_beta0,'(10e20.5)') photon_energy*27.211, val
         end if
      end do

      close(lu_beta0)
      close(lu_beta1)
      close(lu_beta2)

!<TESTING>
! Test cross section
!~ do ienergy=1,no_energies
!~    photon_energy=escat(ienergy)+potionz
!~    aconst=4*pi*(alpha**3)*(photon_energy)**3/(3*echarge**4)
!~    if (iequation.eq.1) aconst=4*pi**2*alpha*photon_energy
!~    write(5220+ichild_state,1010)  photon_energy*27.211,degeneracy(ichild_state)*aconst*real(AL_zero(ienergy))*28.0028518_idp*4*pi
!~ end do
!</TESTING>
      return
 1010 format(3(E20.5,1x))
      end subroutine calculate_orientation_averaged_asymmetry_parameter

!~      subroutine calculate_orientation_averaged_circular_dichroism(lucircd,maxpw,iparent_state,ichild_state,etarg,degeneracy,escat,potionz,dip_full)
!~       implicit none
!~ 
!~ !     Argument variables
!~       integer :: lucircd,iparent_state,ichild_state, maxpw
!~       real(kind=idp) :: potionz
!~       real(kind=idp), dimension(:) :: escat
!~       real(kind=idp), dimension(:) :: etarg
!~       complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
!~       real(kind=idp), dimension(:) :: degeneracy
!~ 
!~ !     Local variables
!~       integer :: no_energies, no_partials, Lbig, ienergy
!~       real(kind=idp) :: photon_energy, Aconst,echarge,ionization_pot
!~       complex(kind=idp), allocatable, dimension(:) :: AL_zero, AL_one 
!~       complex(kind=idp),allocatable,  dimension(:) :: beta_parameter
!~ 
!~ !     Testing
!~       integer :: iequation
!~ 
!~       iequation=1
!~ 
!~       ionization_pot=potionz
!~       echarge=1.0_idp
!~ 
!~       no_energies=size(dip_full,3)
!~       no_partials=size(dip_full,1)
!~ 
!~       allocate (AL_zero(no_energies), AL_one(no_energies),beta_parameter(no_energies))
!~       AL_zero=0._idp
!~       AL_one=0._idp
!~       beta_parameter=0._idp
!~ 
!~       if (iequation.eq.0) then
!~ !!~          call dcs_legendre_pol_expansion_coefficient(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 2, AL_one)
!~ !!~          call dcs_legendre_pol_expansion_coefficient(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 0, AL_zero)
!~       elseif (iequation.eq.1) then
!~          call make_burke_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 1, AL_one)
!~          call make_burke_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, 0, AL_zero) 
!~       end if
!~ 
!~       do ienergy=1,no_energies
!~ 
!~          photon_energy=escat(ienergy)+potionz
!~          Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
!~          
!~          if (abs(AL_zero(ienergy)) .ne. 0) then
!~             beta_parameter(ienergy)= AL_one(ienergy)/AL_zero(ienergy)
!~             write(lucircd+ichild_state,'(10d20.5)') photon_energy*27.211, real(beta_parameter(ienergy),kind=idp)!, real(AL_one(ienergy)),Aconst*real(AL_zero(ienergy))
!~          end if
!~       end do
!~ 
!~ !!<TESTING>
!~ !! Test cross section
!~ !!~ do ienergy=1,no_energies
!~ !!~    photon_energy=escat(ienergy)+potionz
!~ !!~    aconst=4*pi*(alpha**3)*(photon_energy)**3/(3*echarge**4)
!~ !!~    if (iequation.eq.1) aconst=4*pi**2*alpha*photon_energy
!~ !!~    write(5220+ichild_state,1010)  photon_energy*27.211,degeneracy(ichild_state)*aconst*real(AL_zero(ienergy))*28.0028518_idp*4*pi
!~ !!~ end do
!~ !!</TESTING>
!~       return
!~  1010 format(3(E20.5,x))
!~       end subroutine calculate_orientation_averaged_circular_dichroism

      subroutine dcs_legendre_pol_expansion_coefficient(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, Lbig,&
                                                        AL_coefficient)
      implicit none

!     Argument variables
      integer :: iparent_state,ichild_state, Lbig, maxpw
      real(kind=idp), dimension(:) :: escat
      real(kind=idp), dimension(:) :: etarg
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      complex(kind=idp), dimension(:) :: AL_coefficient

!     Local variables
      integer :: ienergy, ipartial, no_energies, no_partials,no_components,no_child_states,no_parent_states
      real(kind=idp) :: Aconst,pconst, asymmetry_parm_temp, echarge, photon_energy, ionization_pot, potionz


      real(kind=idp) :: coeff_one,coeff_two, k_final, charge, eta
      integer :: l, lp, m, mp, lambda, lambdap, ep
      integer :: ilamb, ilambp, ilm, ilmp
      complex(kind=idp) :: coulomb_phase,dipole_product, sph_dip_comp, sph_dip_comp_p
      complex(kind=icp), allocatable, dimension(:,:,:,:,:)  :: dip_full_csph

      real(kind=idp), allocatable :: mytest(:)

      intent(in) :: iparent_state,ichild_state,escat!,dip_full

      charge=1._idp

      no_partials=size(dip_full,1)
      no_components=size(dip_full,2)
      no_child_states=size(dip_full,4)
      no_parent_states=size(dip_full,5)
      no_energies=size(dip_full,3)


!       allocate(AL_coefficient(no_energies))
       AL_coefficient=0.0_idp
      
      allocate(dip_full_csph(no_partials,no_components,no_energies,no_child_states,no_parent_states)) 
      dip_full_csph=0._idp
      call transform_dipoles_to_complex_spherical_harmonic_basis(maxpw,iparent_state,ichild_state,dip_full,dip_full_csph)

!~       dip_full_csph=dip_full !TEST 
!       dip_full=dip_full_csph

      coeff_one=sqrt(2.0*Lbig+1.0)*threej(2,0,2,0,2*Lbig,0)
    !  some loops
allocate(mytest(no_energies))    
mytest=0    
      do ilm=1,no_partials
         do ilmp=1,no_partials
            call i2lm(ilm,l,m)
            call i2lm(ilmp,lp,mp)
!  write(999,'(6i5)') ilm,l,m,ilmp,lp,mp
            do ilamb=1,3
               do ilambp=1,3
                  lambda=(ilamb-2)
                  lambdap=(ilambp-2)
                  ep=m-mp
                  
                  coeff_two=(-1.0)**(m+lambda)*sqrt(2.0*l+1.0)*sqrt(2.0*lp+1.0) &
     &                     *threej(2*l,0,2*lp,0,2*Lbig,0) * threej(2*l,-2*m,2*lp,2*mp,2*Lbig,2*ep) &
     &                     *threej(2,2*lambda,2,-2*lambdap,2*Lbig,-2*ep)
     
                  if (coeff_two .ne. 0.0_idp) then
!                      if ((Lbig.eq.0) .and. (ichild_state .eq. 2)) print *, "l,m",l,m,"lp,mp",lp,mp
                     !Energy loop here
                     do ienergy=1,no_energies 
!                       
                        sph_dip_comp=dip_full_csph(ilm,ilamb,ienergy,ichild_state,iparent_state)                        
                        sph_dip_comp_p=dip_full_csph(ilmp,ilambp,ienergy,ichild_state,iparent_state) 
                        dipole_product=sph_dip_comp*conjg(sph_dip_comp_p)

!                          if (dipole_product .ne. 0.0_idp) then
!~   if ((Lbig.eq.0) .and. (ichild_state .eq. 1) .and. (ilm .eq. ilmp)) then
!~      mytest(ienergy)=mytest(ienergy)+dipole_product
!~      write(8889, *) coeff_one*coeff_two
!~   end if
                           if (escat(ienergy) .ge. etarg(ichild_state)) then
                               k_final=sqrt(2.0*(escat(ienergy)-etarg(ichild_state)))
                               eta=-charge/k_final
                               coulomb_phase=(-eye)**(l-lp)*exp(eye*(CPHAZ(l,ETA,6) - CPHAZ(lp,ETA,6) ) )
                               AL_coefficient(ienergy)=AL_coefficient(ienergy)+coulomb_phase*coeff_two*dipole_product
                           end if
!                          end if

  if ((Lbig.eq.0) .and. (ichild_state .eq. 1) .and. (ilm .eq. ilmp)) then
     mytest(ienergy)=mytest(ienergy)+dipole_product
     write(8889, *) coeff_one*coeff_two
  end if


                     end do

                  end if

               end do

            end do

         end do

      end do
      AL_coefficient=coeff_one*AL_coefficient

if ((Lbig.eq.0) .and. (ichild_state .eq. 1)) then
 do ienergy=1,no_energies 
  write(8888,'(2d20.5)') (escat(ienergy)+potionz)*27.211, (4._idp/3._idp)*pi**2*alpha*(escat(ienergy)+potionz)*mytest(ienergy)&
                                                          *convert_au_to_megabarns
  write(8887,'(2d20.5)') (escat(ienergy)+potionz)*27.211, (4._idp)*pi**2*alpha*(escat(ienergy)+potionz)&
                                                          *real(AL_coefficient(ienergy))*convert_au_to_megabarns
 end do
end if

      end subroutine dcs_legendre_pol_expansion_coefficient                  

      subroutine transform_dipoles_to_complex_spherical_harmonic_basis(maxpw,iparent_state,ichild_state,dip_full,dip_full_csph)
      implicit none
!     NOTE: In principle one should probably perform the same transformation on target states
!           to take them from D2h to Dinfh.

!     Arguments
      integer :: maxpw,ichild_state,iparent_state
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full,dip_full_csph

!     Local variables
      integer :: no_energies,no_partials,no_components,no_child_states,no_parent_states
      complex(kind=icp), dimension(:) :: sph_dip_comp(3)
      complex(kind=icp), allocatable, dimension(:,:) :: basis_transform,basis_transform_dag
      integer :: l,m,mp,ilm,ilmp, ienergy, icomp,m_gamma
      
!     First we set up the trasnformation matrix between real and complex form of the
!     spherical harmonics. This is the conjugate transpose of the matrix given in
!     FORM 1:  H.H.H. Homeier, E.O. Steinborn/Journal of Molecular Structure (Theochem) 368 (1996) 32
!      or
!     FORM 2:  M.A. Blanco et al./Journal of Molecular Structure (Theochem) 419 (1997) 1927

      no_partials=size(dip_full,1)
      no_components=size(dip_full,2)
      no_child_states=size(dip_full,4)
      no_parent_states=size(dip_full,5)
      no_energies=size(dip_full,3)      

      allocate(basis_transform(no_partials,no_partials), basis_transform_dag(no_partials,no_partials))
      basis_transform=0._idp;basis_transform_dag=0._idp

      do l=0,maxpw
         do m=-l,l
            do mp=-l,l
                if (abs(m).eq.abs(mp)) then 

                   ilm=lm2i(l,m)
                   ilmp=lm2i(l,mp)

 !                  FORM 1: 
!~                     if (m.eq.0) then
!~                        basis_transform(ilm,ilmp)=sqrt(2.0_idp)
!~                     elseif ((m.gt.0) .and. (mp.gt.0)) then
!~                        basis_transform(ilm,ilmp)=1
!~                     elseif ((m.gt.0) .and. (mp.lt.0)) then
!~                        basis_transform(ilm,ilmp)=eye
!~                     elseif ((m.lt.0) .and. (mp.gt.0)) then
!~                        basis_transform(ilm,ilmp)=(-1.0)**abs(m)
!~                     elseif ((m.lt.0) .and. (mp.lt.0)) then
!~                        basis_transform(ilm,ilmp)=(-1.0)**abs(m) * (-eye)
!~                     end if
!~                     
!~                     if (m.eq.0) then
!~                        basis_transform(ilm,ilmp)=sqrt(2.0_idp)
!~                     elseif ((m.gt.0) .and. (mp.gt.0)) then
!~                        basis_transform(ilm,ilmp)=1
!~                     elseif ((m.gt.0) .and. (mp.lt.0)) then
!~                        basis_transform(ilm,ilmp)=(-1.0)**abs(m)
!~                     elseif ((m.lt.0) .and. (mp.gt.0)) then
!~                        basis_transform(ilm,ilmp)=-eye
!~                     elseif ((m.lt.0) .and. (mp.lt.0)) then
!~                        basis_transform(ilm,ilmp)=(-1.0)**abs(m) * (eye)
!~                     end if                    
                    

!~ !                  FORM 2.1: U for transforming the basis of real spherical harmonics
                   if (m.eq.0) then
                      basis_transform(ilm,ilmp)=sqrt(2.0_idp)      
                   elseif ((m.gt.0) .and. (mp.gt.0)) then
                      basis_transform(ilm,ilmp)=(-1.0)**abs(m)     
                   elseif ((m.gt.0) .and. (mp.lt.0)) then
                      basis_transform(ilm,ilmp)=eye*(-1.0)**abs(m) !*(-eye)   
                   elseif ((m.lt.0) .and. (mp.gt.0)) then
                      basis_transform(ilm,ilmp)=1                 
                   elseif ((m.lt.0) .and. (mp.lt.0)) then
                      basis_transform(ilm,ilmp)=-eye               !*(-eye) 
                   end if
                   

 !                 FORM 2.2 - U^{\dagger} for transforming the coefficients of real spherical harmonics    *****               
!~                    if (m.eq.0) then
!~                       basis_transform(ilm,ilmp)=sqrt(2._idp)      
!~                    elseif ((m.gt.0) .and. (mp.gt.0)) then
!~                       basis_transform(ilm,ilmp)=(-1._idp)**abs(m)       
!~                    elseif ((m.gt.0) .and. (mp.lt.0)) then             
!~                       basis_transform(ilm,ilmp)=1._idp                 
!~                    elseif ((m.lt.0) .and. (mp.gt.0)) then
!~                       basis_transform(ilm,ilmp)=-eye*(-1._idp)**abs(m)              
!~                    elseif ((m.lt.0) .and. (mp.lt.0)) then
!~                       basis_transform(ilm,ilmp)=eye                    
!~                    end if

                end if
            end do
         end do
      end do
!<TESTING>      
!~       write(999,'("------ THE TRANSFORMATION MATRIX ------")')
!~       do ilm=1, no_partials
!~          write(999,'(100("(",D9.2,",",D9.2,")"))')  (basis_transform(ilm,ilmp), ilmp=1,no_partials)
!~       end do
!</TESTING>
      basis_transform=(1.0_idp/sqrt(2.0_idp))* basis_transform
      basis_transform_dag=transpose(conjg(basis_transform))
      

      do ienergy=1, no_energies

!        Apply the basis set transformation
         do icomp=1,no_components
!~             dip_full_csph(:,icomp,ienergy,ichild_state,iparent_state)=matmul( basis_transform,dip_full(:,icomp,ienergy,ichild_state,iparent_state))
            dip_full_csph(:,icomp,ienergy,ichild_state,iparent_state)=matmul( dip_full(:,icomp,ienergy,ichild_state,iparent_state),&
                          basis_transform_dag)
         end do

!        Now transform from cartesian (y,z,x) to spherical (Y_{1-1}, Y_{10},(Y_{11}) components.
         do ilm=1, no_partials
!~ !           Y_{1-1}
!~             sph_dip_comp(1)=sqrt(0.5_idp)*(dip_full_csph(ilm,3,ienergy,ichild_state,iparent_state)-eye*dip_full_csph(ilm,1,ienergy,ichild_state,iparent_state))
!~ !           Y_{10}
!~             sph_dip_comp(2)=dip_full_csph(ilm,2,ienergy,ichild_state,iparent_state)
!~ !           Y_{11}
!~             sph_dip_comp(3)=-sqrt(0.5_idp)*(dip_full_csph(ilm,3,ienergy,ichild_state,iparent_state)+eye*dip_full_csph(ilm,1,ienergy,ichild_state,iparent_state))
            
            sph_dip_comp=matmul(basis_transform(2:4,2:4), dip_full_csph(ilm,:,ienergy,ichild_state,iparent_state))
            
            dip_full_csph(ilm,:,ienergy,ichild_state,iparent_state)=sph_dip_comp
         end do

      end do
      
      end subroutine transform_dipoles_to_complex_spherical_harmonic_basis

      subroutine make_burke_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, Lbig, AL_coefficient)
      implicit none
!
!      Equation for the asymmetry parameter can be found in Atomic and Molecular Collision Theory
!      Springer (1982) eds
!
 
!     Argument variables
      integer :: iparent_state,ichild_state, Lbig, maxpw
      real(kind=idp) :: potionz
      real(kind=idp), dimension(:) :: escat
      real(kind=idp), dimension(:) :: etarg
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      complex(kind=idp), dimension(:) :: AL_coefficient
      complex(kind=idp),allocatable,  dimension(:,:,:,:,:)  :: dip_mom_trans

!     Local variables
      integer :: no_energies,no_partials_mom_trans
      integer :: ijm_alpha,j,m_alpha, l, lp, ienergy
      real(kind=idp) :: coeff_two, k_final, charge, eta, photon_energy,aconst,pconst
      complex(kind=idp) ::coeff_one,dipole_product

      charge=1.0
      AL_coefficient=0.0_idp

!     Transform dipoles to momentum transfer form.
      call make_momentum_transfer_pw_dipoles(maxpw,dip_full,iparent_state,ichild_state, dip_mom_trans)

      no_partials_mom_trans=size(dip_mom_trans,1)
      no_energies=size(dip_full,3)

      do ienergy=1,no_energies
         do l=0,maxpw
            do lp=0,maxpw
               do ijm_alpha=1,no_partials_mom_trans
                  call i2lm(ijm_alpha,j,m_alpha)
                  if (escat(ienergy) .ge. etarg(ichild_state)) then
                     k_final=sqrt(2.0*(escat(ienergy)-etarg(ichild_state)))
                     eta=-charge/k_final
                     !coeff_one=(2.0*j+1.0)**(-1) * exp(eye*(CPHAZ(l,ETA,6) - CPHAZ(lp,ETA,6)))
                     coeff_one= exp(eye*(CPHAZ(l,ETA,6) - CPHAZ(lp,ETA,6))) *eye**(lp-l)

                     dipole_product=dip_mom_trans(ijm_alpha,l+1, ienergy,ichild_state,iparent_state)&
                                    *conjg(dip_mom_trans(ijm_alpha,lp+1, ienergy,ichild_state,iparent_state))                     
!~                      coeff_two=(2.0*j+1.0)/(4.0*pi) * sqrt((2.0*l+1.0)*(2.0*lp+1.0)) * (-1.0)**j * sixj(2,2,2*Lbig,2*l,2*lp,2*j)*cleb(2,0,2,0,2*Lbig,0)*cleb(2*l,0,2*lp,0,2*Lbig,0)
                     coeff_two= sqrt((2.0*l+1.0)*(2.0*lp+1.0)) * (-1.0)**j * &
     &                          sixj(2,2,2*Lbig, 2*l,2*lp,2*j) * cleb(2,0,2,0,2*Lbig,0) * cleb(2*l,0,2*lp,0,2*Lbig,0)

                     AL_coefficient(ienergy)=AL_coefficient(ienergy) + coeff_one * coeff_two * dipole_product
!<TESTING>                     
!~ if ((ienergy .eq. 1) .and. (Lbig .eq. 2) .and. (ichild_state .eq. 1) .and. (l.eq.lp)) then
!~  write(8889, '(5i3, 10D20.5)') ijm_alpha, j, m_alpha, l, lp, real(coeff_one*coeff_two), dipole_product
!~  
!~ end if
!</TESTING> 
                  end if
               end do
            end do
         end do
      end do
!<TESTING>
!~ if (Lbig.eq.0) call test_calc_xsec_from_mom_transfer_dipoles(9000+ichild_state,maxpw,iparent_state,ichild_state,escat,etarg,potionz,dip_mom_trans)
! if (Lbig.eq.0) then
! do ienergy=1,no_energies
!    photon_energy=escat(ienergy)+potionz !-etarg(ichild_state)
! !    Aconst=4*pi*(alpha**3)*(photon_energy)**3/(3*charge**4)
! !    pconst=pi*(charge**2/((photon_energy)*alpha))**2
!    aconst=4*pi**2*alpha*photon_energy
!    write(5220+ichild_state,1010)  photon_energy*27.211,aconst*real(AL_coefficient(ienergy))*28.0028518_idp*4*pi
! end do
! end if
!</TESTING>
      return
 1010 format(3(E20.5,1x))
      end subroutine  make_burke_AL_coeff

      subroutine make_ritchie_AL_coeff(maxpw,iparent_state,ichild_state,etarg,escat,potionz,dip_full, Lbig,p,AL_coefficient)
      implicit none
!
!     B. Ritchie, PRA 13, 1411, (1976): formulae 11a,11b
!     On input p stands for the desired photon polarization p = -1,0,1
!     The calculated AL_coefficient does not include the normalization factor Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns.
!     This can be applied outside this routine to ensure e.g. that the L=0 coefficient gives the correct integral cross section (see calculate_orientation_averaged_asymmetry_parameter).
 
!     Argument variables
      integer :: iparent_state,ichild_state, Lbig, maxpw,p
      real(kind=idp) :: potionz
      real(kind=idp), dimension(:) :: escat
      real(kind=idp), dimension(:) :: etarg
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      complex(kind=idp), dimension(:) :: AL_coefficient

!     Local variables
      integer :: no_energies
      integer :: l, lp, ienergy, m, mp, ilpmp, ilm, q, qp, no_partials, no_components, no_child_states, no_parent_states
      real(kind=idp) :: k_final, charge, eta, pconst,cf_polarization,cf_angmom,cf_space,cf_photon, coupling, Aconst, photon_energy
      complex(kind=idp) :: coeff_one, dipole_product
      complex(kind=icp), allocatable :: dip_full_csph(:,:,:,:,:), energy_cf(:)

      charge=1.0_idp
      AL_coefficient=0.0_idp

      no_partials = (maxpw+1)**2
      no_components = 3
      no_energies=size(dip_full,3)
      no_child_states=size(dip_full,4)
      no_parent_states=size(dip_full,5)
      allocate(dip_full_csph(no_partials,no_components,no_energies,no_child_states,no_parent_states),energy_cf(no_energies)) 
      dip_full_csph=0.0_idp

      call transform_dipoles_to_complex_spherical_harmonic_basis(maxpw,iparent_state,ichild_state,dip_full,dip_full_csph)

      cf_polarization = threej(2*1,2*p,2*1,-2*p,2*Lbig,0) !set p=+-1 on input for circular polarization
      do l=0,maxpw
         do lp=0,maxpw

            cf_angmom = threej(2*l,0,2*lp,0,2*Lbig,0)*(2*Lbig+1.0_idp)*sqrt(2*l+1.0_idp)*sqrt(2*lp+1.0_idp)
            if (cf_angmom .eq. 0.0_idp) cycle

            energy_cf = 0.0_idp
            do ienergy=1,no_energies
               if (escat(ienergy) .ge. etarg(ichild_state)) then
                  k_final=sqrt(2.0_idp*(escat(ienergy)-etarg(ichild_state)))
                  if (k_final .le. 0.0_idp) exit
                  eta=-charge/k_final
                  coeff_one= exp(eye*(CPHAZ(l,ETA,6) - CPHAZ(lp,ETA,6))) *eye**(lp-l)
                  photon_energy=escat(ienergy)+potionz
                  Aconst=1.0_idp !(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
                  energy_cf(ienergy) = coeff_one*Aconst
               end if
            enddo

            do m=-l,l

               ilm=lm2i(l,m)

               do mp=-lp,lp

                  cf_space = threej(2*l,2*m,2*lp,-2*mp,2*Lbig,-2*(m-mp))
                  if (cf_space .eq. 0.0_idp) cycle

                  ilpmp=lm2i(lp,mp)

                  do q=-1,1
                     do qp=-1,1

                        cf_photon = threej(2*1,2*q,2*1,-2*qp,2*Lbig,-2*(m-mp))
                        if (cf_photon .eq. 0.0_idp) cycle

                        coupling = (-1)**(p+q+m)*cf_angmom*cf_polarization*cf_space*cf_photon
                        do ienergy=1,no_energies
                           dipole_product=dip_full_csph(ilm,2+q,ienergy,ichild_state,iparent_state)&
                                          *conjg(dip_full_csph(ilpmp,2+qp,ienergy,ichild_state,iparent_state))
                           AL_coefficient(ienergy)=AL_coefficient(ienergy) + energy_cf(ienergy) * coupling * dipole_product
                        enddo

                     enddo
                  enddo 
               enddo
            enddo
         enddo
      enddo

      end subroutine  make_ritchie_AL_coeff

      subroutine make_momentum_transfer_pw_dipoles(maxpw,dip_full,iparent_state,ichild_state, dip_mom_trans)
      implicit none

!     Arguments
      integer :: maxpw
      complex(kind=icp), dimension(:,:,:,:,:)  :: dip_full
      complex(kind=idp),allocatable,  dimension(:,:,:,:,:)  :: dip_mom_trans

!     Local variables
      integer :: no_partials,no_components,no_child_states,no_parent_states,no_energies,no_partials_mom_trans
      integer :: ilm,ijm_alpha,j,m_alpha, l,m,ichild_state,iparent_state, ienergy,icomp, m_gamma
      complex(kind=idp) :: dip_times_cg_coeff,sph_dip_comp
      complex(kind=icp), allocatable, dimension(:,:,:,:,:)  :: dip_full_csph
!     Testing
      integer :: icount

!     First we need to figure out the momentum transfer dipole matrix dimensions
      
      no_partials=size(dip_full,1)
      no_components=size(dip_full,2)
      no_child_states=size(dip_full,4)
      no_parent_states=size(dip_full,5)
      no_energies=size(dip_full,3)

      allocate(dip_full_csph(no_partials,no_components,no_energies,no_child_states,no_parent_states)) 
      dip_full_csph=0._idp
      call transform_dipoles_to_complex_spherical_harmonic_basis(maxpw,iparent_state,ichild_state,dip_full,dip_full_csph)

      no_partials_mom_trans= (maxpw+2)**2

      allocate(dip_mom_trans(no_partials_mom_trans,maxpw+1,no_energies,no_child_states, no_parent_states))
      dip_mom_trans=0

      do ienergy=1,no_energies
         do l=0, maxpw
            do ijm_alpha=1,no_partials_mom_trans
               call i2lm(ijm_alpha,j,m_alpha)
               if (lm2i(j,m_alpha) .ne. ijm_alpha) stop 'ERROR IN i2lm'

               ! The summation
               do icomp=1,3
                  m_gamma=icomp-2
                  do m=-l,l
                     if ( ((m-m_gamma).eq. m_alpha) .and. ( (j .le. (l+1)) .or. (j .ge. abs(l-1))) ) then

                        ilm=lm2i(l,m)
                        sph_dip_comp=dip_full_csph(ilm,icomp,ienergy,ichild_state,iparent_state)
                        dip_times_cg_coeff=sph_dip_comp * cleb(2,-2*m_gamma,2*l,2*m,2*j,2*m_alpha) * (-1.0)**m_gamma
                        dip_mom_trans(ijm_alpha,l+1,ienergy,ichild_state,iparent_state) = &
                                      dip_mom_trans(ijm_alpha,l+1,ienergy,ichild_state,iparent_state)+dip_times_cg_coeff

! <TESTING>
!~ if ((ienergy .eq. 1)  .and. (ichild_state .eq. 1) .and. (l.eq.3)) then
!~  write(8888, '(5i3, 100D20.5)') ijm_alpha, j, m_alpha ,m, m_gamma, cleb(2,-2*m_gamma,2*l,2*m,2*j,2*m_alpha) * (-1.0)**m_gamma, sph_dip_comp
!~  
!~ end if
!</TESTING>

                     end if
                  end do
               end do

            end do
         end do
      end do
      end subroutine make_momentum_transfer_pw_dipoles


      subroutine test_calc_xsec_from_mom_transfer_dipoles(luxsec,maxpw,iparent_state,ichild_state,escat,etarg,potionz,dip_mom_trans)
      implicit none

!     Arguments
      integer ::  luxsec,maxpw, iparent_state,ichild_state
      complex(kind=idp),dimension(:,:,:,:,:)  :: dip_mom_trans
      real(kind=idp), dimension(:) :: escat,etarg
 
!     Local variables
      integer :: ienergy, ipartial, no_energies, no_partials, l
      real(kind=idp) :: Aconst,pconst, cross_section_temp, echarge, photon_energy, ionization_pot, potionz
      real(kind=idp), allocatable, dimension(:) :: cross_section 
      real(kind=idp), dimension(:) :: degeneracy(6)


      !pconst=4*(pi**2)*(1/137.0_idp)*28.0028518_idp
      ionization_pot=potionz
      echarge=1.0_idp

      no_energies=size(dip_mom_trans,3)
      no_partials=size(dip_mom_trans,1)

      allocate(cross_section(no_energies))
      cross_section=0.0_idp

      do ienergy=1, no_energies
         cross_section_temp=0.0_idp

         do ipartial=1,no_partials
            do l=1,maxpw+1
                cross_section_temp=cross_section_temp+abs(dip_mom_trans(ipartial,l,ienergy,ichild_state,iparent_state))**2 
            end do
         end do

         photon_energy=escat(ienergy)+ionization_pot 

         Aconst=4*pi*(alpha**3)*(photon_energy)**3/(3*echarge**4)
         pconst=pi*(echarge**2/((photon_energy)*alpha))**2

         cross_section(ienergy)=pconst*Aconst*cross_section_temp*28.0028518_idp

      end do

      do ienergy=1,no_energies
         photon_energy=escat(ienergy)+ionization_pot
         write(luxsec,1010)  photon_energy*27.211,cross_section(ienergy)
      end do

 1010 format(3(E20.5,1x))

      end subroutine test_calc_xsec_from_mom_transfer_dipoles


      subroutine test_print_partial_wave_cross_sections(luxsec,maxpw,iparent_state,ichild_state,escat,etarg,potionz,lvchl, &
                                                        mvchl, ichl, no_channels_scat_states,dipole_component_order, dip_full)
      implicit none

!     Arguments
      integer ::  luxsec, maxpw, iparent_state, ichild_state, lvchl(:,:), mvchl(:,:), ichl(:,:),no_channels_scat_states(:)
      complex(kind=idp),dimension(:,:,:,:,:)  :: dip_full
      real(kind=idp), dimension(:) :: escat,etarg
      character(len=1), dimension(:) :: dipole_component_order(3)  
          
!     Local variables
      integer :: ienergy, ipartial, no_energies, no_partials, icomponent, l, m, i,j,k, ii, idcmp
      real(kind=idp) :: Aconst,pconst, cross_section_temp, echarge, photon_energy, ionization_pot, potionz
      real(kind=idp), allocatable :: cross_section(:) 
      real(kind=idp), dimension(:) :: degeneracy(6)


      !pconst=4*(pi**2)*(1/137.0_idp)*28.0028518_idp
      ionization_pot=potionz
      echarge=1.0_idp

      no_energies=size(dip_full,3)
      no_partials=size(dip_full,1)

      allocate(cross_section(no_energies))
      cross_section=0.0_idp


      do j=1,3
         idcmp=dcomp2i(dipole_component_order(j))

         do k=1,no_channels_scat_states(j)
            l=lvchl(idcmp,k)
            m=mvchl(idcmp,k)

            ipartial=lm2i(l,m)

            do ienergy=1,no_energies
               cross_section_temp=0.0_idp
        
               photon_energy=escat(ienergy)+ionization_pot 
               Aconst=4*pi*(alpha**3)*(photon_energy)**3/(3*echarge**4)
               pconst=pi*(echarge**2/((photon_energy)*alpha))**2
                         
               cross_section_temp = abs(dip_full(ipartial,idcmp,ienergy,ichild_state,iparent_state))**2 &
                                    *pconst*Aconst*28.0028518_idp

               write(luxsec+10*idcmp+ichild_state,'(2e20.5, 2i2)')  photon_energy*27.211,cross_section_temp, l, m
            end do

            write(luxsec+10*idcmp+ichild_state,*) ""
         end do
         
         
      end do

 1010 format(3(E20.5,1x))

      end subroutine test_print_partial_wave_cross_sections



      end module dipelmprocs

