! Copyright 2019
!
! Alex G. Harvey with ontributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      module angmom_procs
      use dipelmdefs
      use anglib
  

      contains

!     Whole bunch of functions taken from align

      real(kind=idp) function red_rot_mat(beta,J,M,N)
      implicit none
      integer,intent(in) :: J,M,N
      real(kind=idp), intent(in) :: beta
      integer :: t,tmax,tmin, t1, t2, t3
      real(kind=idp) :: tmp,S,C,tmp1, tmp2, tmp3 
      tmp=0d0

!     Determine allowed values of t to ensure factorials are 
!     non-negative

      t1=J-N

      t2=J+M
      t3=m-n
    
      if (t3 .gt. 0) then 
         tmin=t3
      else
         tmin=0
      end if
      if (t1 .lt. t2) then
         tmax=t1
      else
         tmax=t2
      end if
      
      if ((abs(M).gt.J) .or. (abs(N).gt.J)) then
         red_rot_mat=0
         return
      end if
    
!     explicit expression for the reduced rotation matrix elements 
!     (Brink & Satchler 1968 "Angular Momentum")
    
      S=sin(beta/2d0)
      C=cos(beta/2d0)
      tmp1=sqrt(factorial(J+M)*factorial(J-M)*factorial(J+N)*factorial(J-N))
    
      do t=tmin,tmax
         tmp = tmp + ((-1)**t)*(tmp1/(factorial(J+M-t)* &
     &               factorial(J-N-t)*factorial(t)*factorial(t+N-M)))* &
     &               C**(2*J+M-N-2*t)*S**(2*t+N-M)
      end do

      red_rot_mat=tmp
      end function red_rot_mat

      complex(kind=idp) function rot_mat(alpha, beta, gamma,j,m,n)
      implicit none
!     Version of rot_mat using complex number type    
      integer, intent(in) :: j,m,n
      real(kind=idp), intent(in) :: beta, alpha, gamma
      real(kind=idp) :: re_rm,im_rm
    
      re_rm=cos(m*alpha+n*gamma)*red_rot_mat(beta,j,m,n) 
      im_rm=-sin(m*alpha+n*gamma)*red_rot_mat(beta,j,m,n)
    
      rot_mat=cmplx(re_rm,im_rm,idp)

      end function rot_mat

      real(kind=idp) function azim_fn(theta, m)
      implicit none
      integer, intent(in) :: m
      real(kind=idp), intent(in) :: theta
      real(kind=idp) ::  temp

      select case(m)
      case(1:)
         temp=sqrt(2d0)*cos(m*theta)
      case(0)
         temp=1d0
      case(:-1)
         temp=sqrt(2d0)*sin(abs(m)*theta)
      end select

      azim_fn=temp

      end function azim_fn

      real(kind=idp) function re_rot_mat(alpha, beta, gamma,j,m,n)
      implicit none
      integer, intent(in) :: j,m,n
      real(kind=idp), intent(in) :: beta, alpha, gamma
      real(kind=idp) :: phi_pm, phi_pn,phi_mm, phi_mn 
      real(kind=idp) ::  rm

      phi_pm=azim_fn(alpha, m)
      phi_mm=azim_fn(alpha, -m)
      phi_pn=azim_fn(gamma, n)
      phi_mn=azim_fn(gamma, -n)

      rm= sign(1,n)*phi_pm*phi_pn*(red_rot_mat(beta,j,abs(n),abs(m))+((-1)**m)*red_rot_mat(beta,j,abs(m),-abs(n)))/2d0 &
     &  -sign(1,m)*phi_mm*phi_mn*(red_rot_mat(beta,j,abs(n),abs(m))-((-1)**m)*red_rot_mat(beta,j,abs(m),-abs(n)))/2d0

      re_rot_mat=rm

      end function re_rot_mat


!     Olga's rotation matrix Z-Y-Z convention active rotation of vector

      subroutine cartesian_rot_mat(alpha, beta, gamma, rot_mat)
      implicit none
      real(kind=idp), intent(in) :: beta, alpha, gamma
      real(kind=idp) , intent(out) :: rot_mat(3,3)  

      real(kind=idp) :: c1,c2,c3, s1,s2,s3 
 

      c1=cos(alpha)
      c2=cos(beta)
      c3=cos(gamma)
      s1=sin(alpha)
      s2=sin(beta)
      s3=sin(gamma)     

!     For use with vectors organised as (x,y,z)
!       rot_mat(1,1)=c1*c2*c3-s1*s3;  rot_mat(1,2)=-c3*s1-c1*c2*s3; rot_mat(1,3)=c1*s2
!       rot_mat(2,1)=c1*s3+c2*c3*s1;  rot_mat(2,2)=c1*c3-c2*s1*s3;  rot_mat(2,3)=s1*s2  
!       rot_mat(3,1)=-c3*s2;          rot_mat(3,2)=s2*s3;           rot_mat(3,3)=c2

!     For use with vectors organised as (y,z,x)
      rot_mat(1,1)=c1*c3-c2*s1*s3;  rot_mat(1,2)=s1*s2;           rot_mat(1,3)=c1*s3+c2*c3*s1
      rot_mat(2,1)=s2*s3;           rot_mat(2,2)=c2;              rot_mat(2,3)=-c3*s2
      rot_mat(3,1)=-c3*s1-c1*c2*s3; rot_mat(3,2)=c1*s2;           rot_mat(3,3)=c1*c2*c3-s1*s3

      end subroutine cartesian_rot_mat

      complex(kind=idp) function sp_harm(j,m,theta,phi)
      implicit none
      integer, intent(in) :: j,m
      real(kind=idp), intent(in) :: theta, phi
      real(kind=idp) :: re_sh,im_sh

!     Old (and maybe wrong)
!       re_sh=sqrt((2*j+1)/(4*pi))*cos(real(m*phi))*red_rot_mat(theta,j,m,0) 
!       im_sh=sqrt(real(2*j+1)/(4*pi))*sin(real(m*phi))*red_rot_mat(theta,j,m,0)
! 
!       sp_harm=cmplx(re_sh,im_sh,idp)
      if (m .ge. 0) then
         re_sh= sqrt((2*j+1)/(4*pi))*cos(real(abs(m)*phi))*red_rot_mat(theta,j,abs(m),0) 
         im_sh= sqrt((2*j+1)/(4*pi))*sin(real(abs(m)*phi))*red_rot_mat(theta,j,abs(m),0)
      else
         re_sh= sqrt((2*j+1)/(4*pi))*cos(real(abs(m)*phi))*red_rot_mat(theta,j,abs(m),0)*(-1.0)**abs(m) 
         im_sh=-sqrt((2*j+1)/(4*pi))*sin(real(abs(m)*phi))*red_rot_mat(theta,j,abs(m),0)*(-1.0)**abs(m) 
      end if

      sp_harm=cmplx(re_sh,im_sh,idp)


      end function sp_harm


      real(kind=idp) function re_sp_harm(j,m,theta,phi)
      implicit none
      integer, intent(in) :: j,m
      real(kind=idp), intent(in) :: theta, phi
      real(kind=idp) :: temp,norm

      norm=((-1)**m)/sqrt(2d0)
!       norm=1._idp/sqrt(2._idp)

      select case(m)
      case(1:)
         temp=norm*2*sqrt((2*j+1)/(4*pi))*cos(real(m*phi))*red_rot_mat(theta,j,m,0) 
      case(0)
         temp=sqrt((2*j+1)/(4*pi))*red_rot_mat(theta,j,0,0)
      case(:-1)
         temp=norm*2*sqrt(real(2*j+1)/(4*pi))*sin(real(abs(m)*phi))*red_rot_mat(theta,j,abs(m),0)
!          temp=norm*2*sqrt(real(2*j+1)/(4*pi))*sin(real(m*phi))*red_rot_mat(theta,j,abs(m),0)
      end select

      re_sp_harm=temp

      end function re_sp_harm

      real(kind=idp) function re_sp_harm_gaunt(j,m,theta,phi)
      implicit none
      integer, intent(in) :: j,m
      real(kind=idp), intent(in) :: theta, phi
      real(kind=idp) :: temp,norm
      complex(kind=idp) :: y_plus, y_minus

      norm=1._idp/sqrt(2._idp)
!       norm=(-1._idp)**m/sqrt(2._idp)

!       select case(m)
!       case(1:)
!          y_plus= eye**(abs(m)+abs(m))*sp_harm(j,abs(m),theta,phi)
!          y_minus= eye**(-abs(m)+abs(m))*sp_harm(j,-abs(m),theta,phi)
!          temp=real(norm*(y_plus+((-1)**m)*y_minus))
!       case(0)
!          temp=real(sp_harm(j,0,theta,phi))
!       case(:-1)
!          y_plus= eye**(abs(m)+abs(m))*sp_harm(j,abs(m),theta,phi)
!          y_minus= eye**(-abs(m)+abs(m))*sp_harm(j,-abs(m),theta,phi)
!          temp=real(norm*eye*(((-1)**m)*y_plus-y_minus))
!       end select

      select case(m)
      case(1:)
         temp=sqrt(2.0)*real(eye**(abs(m)+abs(m))*sp_harm(j,abs(m),theta,phi))
      case(0)
         temp=real(sp_harm(j,0,theta,phi))
      case(:-1)
         temp=sqrt(2.0)*aimag(eye**(abs(m)+abs(m))*sp_harm(j,abs(m),theta,phi))
      end select


      re_sp_harm_gaunt=temp



      end function re_sp_harm_gaunt

      real(kind=idp) function re_sp_harm_old(j,m,theta,phi)
      implicit none
      integer, intent(in) :: j,m
      real(kind=idp), intent(in) :: theta, phi
      real(kind=idp) :: temp,norm
      complex(kind=idp) :: y_plus, y_minus

      norm=1._idp/sqrt(2._idp)
!       norm=(-1._idp)**m/sqrt(2._idp)

      select case(m)
      case(1:)
         y_plus= sp_harm(j,abs(m),theta,phi)
         y_minus= sp_harm(j,-abs(m),theta,phi)
         temp=real(norm*(y_plus+((-1)**m)*y_minus))
      case(0)
         temp=real(sp_harm(j,0,theta,phi))
      case(:-1)
         y_plus= sp_harm(j,abs(m),theta,phi)
         y_minus= sp_harm(j,-abs(m),theta,phi)
         temp=real(norm*eye*(((-1)**m)*y_plus-y_minus))
      end select

      re_sp_harm_old=temp

      end function re_sp_harm_old




      function cphaz(l,eta,iwrite)
!
!     ******************************************************************
!
!     cphaz evaluates the Coulomb phase, sigma(l)
!           sigma(l) = arg (  gamma(l+1+i*eta) )
!     normally, for electron scattering ...
!
!           eta = -z / k       where z is the residual nuclear charge
!                                    k is the electronic momentum
!                              ( all quantities in atomic units )
!
!     ******************************************************************
!
      implicit double precision (a-h,o-z)
!
      data half/0.5d0/,zero/0.0d0/,two/2.0d0/,ip1/9/
      data c1/0.83333333333333333d-01/,c2/0.27777777777777778d-02/, &
           c3/0.79365079365079365d-03/,c4/0.59523809523809523d-03/, &
           c5/0.84175084175084175d-03/
!
 1000 format(' Coulomb phase, sigma(',i3,';',d16.8,' ) =',d30.17)
!
!     arg ( gamma (l+1+i*eta) ) = imag ( ln gamma (p+i*eta) )
!                                 - sum( s=l+1 to s=p-1 ) atan ( eta/s )
!     for some integer p
!
!     here take p=10
!
      d1=zero
      b=eta
      k1=l+1
      if (k1 <= ip1) then
        go to 10
      else
        go to 30
      end if
!
!     sum arc tangent terms
!
   10 do 20 j=k1,ip1,1
      a=dble(j)
      d=datan2(b,a)
      d1=d+d1
   20 continue
      a=dble(ip1+1)
      go to 40
!
   30 a=dble(l+1)
      d1=zero
!
!     evaluate ln ( gamma (p-i*eta) ) using an asymptotic formula
!     reference: n.b.s. tables eqns 6.1.40 and 6.1.41
!
   40 c=half*dlog(a*a+b*b)
      d=datan2(b,a)
      d=d*(a-half)+c*b
      d=d-b
!
!     evaluate  1 / (p+i*eta) and ( 1 / (p+i*eta) ) **2
!
      e=a*a+b*b
      a=a/e
      e=-b/e
      f=a*a-e*e
      g=two*e*a
!
!     -r1 = h  = 1/1188 f - 1/1680
!     -i1 = h1 = 1/1188 g
!
      h=c5*f-c4
      h1=c5*g
      h2=h
!
!     r2 = h  = 1/1260 + f * ( -r1 ) - g * ( -i1 )
!     i2 = h1 =          f * ( -i1 ) + g * ( -r1 )
!
      h=h*f-h1*g
      h1=h1*f+h2*g
      h=h+c3
      h2=h
!
!     -r3 = h  = -1/360 + f * ( r2 ) - g * ( i2 )
!     -i3 = h1 =          f * ( i2 ) + g * ( r2 )
!
      h=h*f-h1*g
      h1=h1*f+h2*g
      h=h-c2
      h2=h
!
!     r4 = h  = 1/12 + f * ( -r3 ) - g * ( -i3 )
!     i4 = h1 =        f * ( -i3 ) + g * ( -r3 )
!
      h=h*f-h1*g
      h1=h1*f+h2*g
      h=h+c1
      h2=h
!
!     evaluate imag ( ( r4 + i*i4 ) / ( p + i*eta ) )
!
      h1=h1*a+h2*e
!
!     calculate final value for phase shift
!
      cphaz=h1+d-d1
!
!      write(iwrite,1000)l,eta,cphaz
!
      return
      end function cphaz

!     The following create angular grids of (theta, phi) and the Euler
!     angles (alpha,beta,gamma)

!       subroutine grid_theta_phi(th_grid,ph_grid)
!       use dipelmdefs
!       implicit none
! !     ******************************************************************
! !     creates angular grids for theta, phi apropriate for
! !     spherical harmonics 
! !     0<=theta<=pi
! !     0<=phi<2pi
! !     ******************************************************************
!       real(kind=idp), dimension(:) :: th_grid, ph_grid
! 
!       real(kind=idp) :: theta_max,phi_max
!       integer :: i,m,n
! 
!       theta_max=pi
!       phi_max=2*pi
!       m=size(th_grid)
!       n=size(ph_grid)
! 
! !     theta grid
!       do i=1,m
!          if (m .ne. 1) then
!             th_grid(i)=(i-1)*theta_max/real((m-1),kind=idp)
!          else
!             th_grid(i)=0_idp
!          end if
!       end do
! !     phi grid
!       do i=1,n
!          ph_grid(i)=(i-1)*phi_max/real(n,kind=idp)
!       end do
! 
!       end subroutine grid_theta_phi

      subroutine grid_theta_phi(th_grid,ph_grid,ang_range)
      implicit none
!     ******************************************************************
!     creates angular grids for theta, phi apropriate for
!     spherical harmonics 
!     0<=theta<=pi
!     0<=phi<2pi
!     ******************************************************************
      real(kind=idp), dimension(:) :: th_grid, ph_grid
      real(kind=idp), dimension(4) :: ang_range

      real(kind=idp) :: theta_range,phi_range
      integer :: i,m,n

!       theta_range=pi
!       phi_range=2*pi

      theta_range=ang_range(3)-ang_range(1)
      phi_range=ang_range(4)-ang_range(2)

      m=size(th_grid)
      n=size(ph_grid)

!     theta grid
      do i=1,m
         if (m .ne. 1) then
            th_grid(i)=ang_range(1)+(i-1)*theta_range/real((m-1),kind=idp)
         else
            th_grid(i)=ang_range(1)
         end if
      end do
!     phi grid
      do i=1,n
         ph_grid(i)=ang_range(2)+(i-1)*phi_range/real(n,kind=idp)
      end do

      end subroutine grid_theta_phi


      subroutine grid_euler(a_grid,b_grid,g_grid,ang_range)
      implicit none
!     ******************************************************************
!     creates angular grids for the Euler angles alpha, beta, gamma 
!     apropriate for rotation matrices
!     0<=alpha<2pi
!     0<=beta<=pi
!     0<=gamma<2pi
!     ******************************************************************

      real(kind=idp), dimension(:) :: a_grid,b_grid,g_grid
      real(kind=idp), dimension(6) :: ang_range

      real(kind=idp) :: alpha_range,beta_range,gamma_range
      integer :: i,l,m,n

!       alpha_range=2*pi
!       beta_range=pi
!       gamma_range=2*pi

      alpha_range=ang_range(4)-ang_range(1)
      beta_range=ang_range(5)-ang_range(2)
      gamma_range=ang_range(6)-ang_range(3)

      l=size(a_grid)
      m=size(b_grid)
      n=size(g_grid)

!     alpha grid
      do i=1,l
         if (l .ne. 1) then
            a_grid(i)=ang_range(1)+(i-1)*alpha_range/real(l,kind=idp)
         else
            a_grid(i)=ang_range(1)
         end if
      end do
!     beta grid
      do i=1,m
         if (m .ne. 1) then
            b_grid(i)=ang_range(2)+(i-1)*beta_range/real((m-1),kind=idp)
         else
            b_grid(i)=ang_range(2)
         end if
      end do
!     gamma grid
      do i=1,n
         if (n .ne. 1) then
            g_grid(i)=ang_range(3)+(i-1)*gamma_range/real(n,kind=idp)
         else
            g_grid(i)=ang_range(3)
         end if
      end do

      end subroutine grid_euler


!     l2mi,abg2i and tp2i are indexing functions for the l and m quantum
!     quantum numbers, the euler angles and the angular arguments
!     of the spherical harmonic, theta and phi.

      integer function lm2i(l,m)
      implicit none
      !Maps l and m quantum numbers to single index i
      !order is (0,0), (1,-1), (1,0) (1,1) etc 
      integer :: l,m

      lm2i=l*(l+1)+(m+1)

      return
      end function lm2i

      subroutine i2lm(i,l,m)
      implicit none
      !Maps  single index i to l and m quantum numbers 
      !order is (0,0), (1,-1), (1,0) (1,1) etc 
      
      ! Arguments
      integer :: i,l,m
      
      l=0
      do
          if ((l+1)**2 .ge. i) exit
          l=l+1
      end do
      m=i-(l+1)**2+l

      return
      end subroutine i2lm



      integer function abg2i(ialpha, ibeta, igamma, ibmax,igmax)
      implicit none
!     Indexing function for euler angle index
!     order is  (1,1,1), (1,1,2),..., (1,2,1),..., 
      integer :: ialpha, ibeta, igamma, ibmax,igmax

      abg2i=(ialpha-1)*ibmax*igmax+(ibeta-1)*igmax+igamma

      return
      end function abg2i

      integer function tp2i(itheta, iphi, ipmax)
      implicit none
!     Indexing function for theta,phi angle index
!     order is  (1,1), (1,2),..., (2,1),..., 
      integer :: itheta, iphi, ipmax

      tp2i=(itheta-1)*ipmax+iphi

      return
      end function tp2i

!     The following construct grids of spherical harmonics, rotation
!     matrices and coulomb phases. These are indexed by the previously
!     defined indexing functions lm2i, abg2i and tp2i

      subroutine re_sph_grid(sph_grid,th_grid,ph_grid,maxpw)
      implicit none
!     Construct (real) spherical harmonics matrix
      integer :: maxpw
      real(kind=idp), dimension(:) :: th_grid,ph_grid
      real(kind=idp), dimension(:,:) :: sph_grid 
      intent(in) :: th_grid,ph_grid, maxpw
       
      integer :: nth,nph,ipw,iangle,j,k,l,m
      
      nth=size(th_grid)
      nph=size(ph_grid)
      
      do l=0,maxpw
         do m=-l,l
!            i=l*(l+1)+m+1
            ipw=lm2i(l,m)
            do j=1,nth
               do k=1,nph
                  iangle=tp2i(j,k,nph)
                  sph_grid(iangle,ipw)= re_sp_harm(l,m,th_grid(j),ph_grid(k))
!~                   sph_grid(iangle,ipw)= re_sp_harm_gaunt(l,m,th_grid(j),ph_grid(k))
!                 
               end do
            end do 
         end do     
      end do

      end subroutine re_sph_grid


      subroutine re_sph_grid2(sph_grid,scat_grid,maxpw)
      implicit none
!     Construct (real) spherical harmonics matrix
      integer :: maxpw
      real(kind=idp), dimension(:,:) :: scat_grid
      real(kind=idp), dimension(:,:) :: sph_grid 
      intent(in) :: scat_grid, maxpw
      
      integer :: nangles,iangle,ipw,l,m
      
      nangles=size(scat_grid,2)

      do l=0,maxpw
         do m=-l,l
            ipw=lm2i(l,m)
            do iangle=1,nangles
                  sph_grid(ipw,iangle)= re_sp_harm(l,m,scat_grid(1,iangle),scat_grid(2,iangle))
            end do 
         end do     
      end do

      end subroutine re_sph_grid2


      subroutine re_rot_mat_grid(rot_grid,a_grid,b_grid,g_grid,maxpw)
!     Construct (real) Wigner D matrix
      implicit none
      integer :: maxpw
      real(kind=idp), dimension(:) :: a_grid,b_grid, g_grid
      real(kind=idp), dimension(:,:,:) :: rot_grid 
      intent(in) :: a_grid,b_grid,g_grid,maxpw

      integer :: na,nb,ng,ia,ib,ig, i,ii,j,l,m,n

      na=size(a_grid)
      nb=size(b_grid)
      ng=size(g_grid)
rot_grid=0._idp
      do l=0,maxpw
         do m=-l,l
            do n=-l,l
               i=lm2i(l,m)
               j=lm2i(l,n)
               do ia=1,na
                  do ib=1,nb
                     do ig=1,ng
                        ii=abg2i(ia,ib,ig,nb,ng)
                        !print *, l,re_rot_mat(a_grid(ia), b_grid(ib), g_grid(ig),l,m,n)
                        rot_grid(i,j,ii)=re_rot_mat(a_grid(ia), b_grid(ib), g_grid(ig),l,m,n)
                     end do
                  end do
               end do
            end do     
         end do 
      end do

      end subroutine re_rot_mat_grid

      subroutine lebedev_re_rot_mat_grid(rot_grid,lebedev_grid,maxpw)
!     Construct (real) Wigner D matrix
      implicit none
      integer :: maxpw
      real(kind=idp), dimension(:,:) :: lebedev_grid
      real(kind=idp), dimension(:,:,:) :: rot_grid 
      intent(in) :: lebedev_grid,maxpw

      integer :: neuler,nb,ng,ieuler, i,ii,j,l,m,n

      neuler=size(lebedev_grid,2)

      rot_grid=0._idp
      do l=0,maxpw
         do m=-l,l
            do n=-l,l
               i=lm2i(l,m)
               j=lm2i(l,n)
               do ieuler=1,neuler
!                   rot_grid(i,j,ieuler)=re_rot_mat(lebedev_grid(1,ieuler), lebedev_grid(2,ieuler), 0.0_idp,l,m,n)
                  rot_grid(i,j,ieuler)=re_rot_mat(lebedev_grid(1,ieuler), lebedev_grid(2,ieuler), pi/2._idp,l,m,n) 
               end do
            end do     
         end do 
      end do

      end subroutine lebedev_re_rot_mat_grid

      subroutine lebedev_gamma_re_rot_mat_grid(rot_grid,lebedev_gamma_grid,maxpw)
!     Construct (real) Wigner D matrix
      implicit none
      integer :: maxpw
      real(kind=idp), dimension(:,:) :: lebedev_gamma_grid
      real(kind=idp), dimension(:,:,:) :: rot_grid 
      intent(in) :: lebedev_gamma_grid,maxpw

      integer :: neuler,nb,ng,ieuler, i,ii,j,l,m,n

      neuler=size(lebedev_gamma_grid,2)

      rot_grid=0._idp
      do l=0,maxpw
         do m=-l,l
            do n=-l,l
               i=lm2i(l,m)
               j=lm2i(l,n)
               do ieuler=1,neuler
!                   rot_grid(i,j,ieuler)=re_rot_mat(lebedev_grid(1,ieuler), lebedev_grid(2,ieuler), 0.0_idp,l,m,n)
                  rot_grid(i,j,ieuler)=re_rot_mat(lebedev_gamma_grid(1,ieuler), lebedev_gamma_grid(2,ieuler),&
                                                  lebedev_gamma_grid(3,ieuler) ,l,m,n) 
               end do
            end do     
         end do 
      end do

      end subroutine lebedev_gamma_re_rot_mat_grid


      subroutine cartesian_rot_mat_grid(cart_rot_grid,a_grid,b_grid,g_grid)
!     Construct (real) Wigner D matrix
      implicit none
      real(kind=idp), dimension(:) :: a_grid,b_grid, g_grid
      real(kind=idp), dimension(:,:,:) :: cart_rot_grid 
      intent(in) :: a_grid,b_grid,g_grid

      integer :: na,nb,ng,ia,ib,ig,ieuler

      na=size(a_grid)
      nb=size(b_grid)
      ng=size(g_grid)
      cart_rot_grid=0._idp

               do ia=1,na
                  do ib=1,nb
                     do ig=1,ng
                        ieuler=abg2i(ia,ib,ig,nb,ng)
                        call cartesian_rot_mat(a_grid(ia), b_grid(ib), g_grid(ig),cart_rot_grid(:,:,ieuler))
                     end do
                  end do
               end do
    
      end subroutine cartesian_rot_mat_grid


      end module angmom_procs
