! Copyright 2019
!
! Alex G. Harvey with ontributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
module anglib
! Library of angular momentum coupling coefficient routines in fortran 90
! Paul Stevenson, Oxford University/Oak Ridge National Laboratory.
! spaul@mail.phy.ornl.gov

  integer, parameter :: rk = selected_real_kind(p=15)
  
contains

  function cleb(j1,m1,j2,m2,j,m)
    implicit none
    ! calculate a clebsch-gordan coefficient < j1/2 m1/2 j2/2 m2/2 | j/2 m/2 >
    ! arguments are integer and twice the true value. 

    real(rk)    :: cleb,factor,sum
    integer :: j1,m1,j2,m2,j,m,par,z,zmin,zmax

    ! some checks for validity (let's just return zero for bogus arguments)

    if (2*(j1/2)-int(2*(j1/2.0_rk)) /= 2*(abs(m1)/2)-int(2*(abs(m1)/2.0_rk)) .or. &
         2*(j2/2)-int(2*(j2/2.0_rk)) /= 2*(abs(m2)/2)-int(2*(abs(m2)/2.0_rk)) .or. &
         2*(j/2)-int(2*(j/2.0_rk)) /= 2*(abs(m)/2)-int(2*(abs(m)/2.0_rk)) .or. &
         j1<0 .or. j2<0 .or. j<0 .or. abs(m1)>j1 .or. abs(m2)>j2 .or.&
         abs(m)>j .or. j1+j2<j .or. abs(j1-j2)>j .or. m1+m2/=m) then
       cleb= 0.0_rk
    else
    
       factor = 0.0_rk
       factor = binom(j1,(j1+j2-j)/2) / binom((j1+j2+j+2)/2,(j1+j2-j)/2)
       factor = factor * binom(j2,(j1+j2-j)/2) / binom(j1,(j1-m1)/2)
       factor = factor / binom(j2,(j2-m2)/2) / binom(j,(j-m)/2)
       factor = sqrt(factor)
       
       zmin = max(0,j2+(j1-m1)/2-(j1+j2+j)/2,j1+(j2+m2)/2-(j1+j2+j)/2)
       zmax = min((j1+j2-j)/2,(j1-m1)/2,(j2+m2)/2)
       
       sum=0.0_rk
       do z = zmin,zmax
          par=1
          if(2*(z/2)-int(2*(z/2.0_rk)) /= 0) par=-1
          sum=sum+par*binom((j1+j2-j)/2,z)*binom((j1-j2+j)/2,(j1-m1)/2-z)*&
               binom((-j1+j2+j)/2,(j2+m2)/2-z)
       end do
       
       cleb = factor*sum
    end if
  end function cleb

  function threej(j1,m1,j2,m2,j,m)
  implicit none
  integer :: j1,m1,j2,m2,j,m
  real(rk) :: threej,cleb_temp

  cleb_temp=cleb(j1,m1,j2,m2,j,-m)

!   if(abs(cleb_temp) .gt. 0._rk ) then
     threej = (-1)**((j1 - j2 - m) / 2) / sqrt(real(j + 1, rk)) * cleb_temp
!   else
!     threej=0._rk
!   end if

  end function threej

!   function basis_transform(j,m1,m2)
!   integer :: j,m1,m2,jq,mq,mqp
!   complex(rk) :: basis_transform
!   complex(kind=rk), parameter :: eye=(0,1.0) !sqrt of minus 1
!    
!   jq=j/2
!   mq=m1/2
!   mqp=m2/2
! 
! !   if ((abs(m1)+abs(m2)) .le. abs(j)) then
!   if ((abs(mq)+abs(mqp)) .le. abs(j)) then
!      if (abs(mq).eq.abs(mqp)) then
! 
!         if (mq.eq.0) then
!            basis_transform=sqrt(2.0_rk)
!         elseif ((mq.gt.0) .and. (mqp.gt.0)) then
!            basis_transform=(-1.0)**abs(mq)
!         elseif ((mq.gt.0) .and. (mqp.lt.0)) then
!            basis_transform=eye*(-1.0)**abs(mq)
!         elseif ((mq.lt.0) .and. (mqp.gt.0)) then
!            basis_transform=1
!         elseif ((mq.lt.0) .and. (mqp.lt.0)) then
!            basis_transform=-eye
!         end if
! 
!      else
!         basis_transform=0
!      end if
!   else
!      basis_transform=0
!   end if
! 
!   basis_transform= basis_transform/sqrt(2.0_rk)
! 
!   end function basis_transform

  function basis_transform(j,m1,m2)
  integer :: j,m1,m2,jq,mq,mqp
  complex(rk) :: basis_transform
  complex(kind=rk), parameter :: eye=(0,1.0) !sqrt of minus 1
! From the gaunt coefficient paper  
  jq=int(j/2.0)
  mq=int(m1/2.0)
  mqp=int(m2/2.0)

!   if ((abs(m1)+abs(m2)) .le. abs(j)) then
  if ((abs(mq)+abs(mqp)) .le. abs(j)) then
     if (abs(mq).eq.abs(mqp)) then

        if (mq.eq.0) then
           basis_transform=sqrt(2.0_rk)
        elseif ((mq.gt.0) .and. (mqp.gt.0)) then
           basis_transform=1.0
        elseif ((mq.gt.0) .and. (mqp.lt.0)) then
           basis_transform=(-1.0)**abs(mq)
        elseif ((mq.lt.0) .and. (mqp.gt.0)) then
           basis_transform=-eye
        elseif ((mq.lt.0) .and. (mqp.lt.0)) then
           basis_transform=eye*(-1.0)**abs(mq)
        end if

     else
        basis_transform=0
     end if
  else
     basis_transform=0
  end if

  basis_transform= basis_transform/sqrt(2.0_rk)

  end function basis_transform





  function gaunt(j1,m1,j2,m2,j,m)
  integer :: j1,m1,j2,m2,j,m
  real(rk) :: gaunt
  real(kind=rk), parameter :: pi=3.1415926535897932_rk

  gaunt=sqrt((j1+1)*(j2+1)*(j+1)/(4.0*pi)) * threej(j1,m1,j2,m2,j,-m) * threej(j1,0,j2,0,j,0)*(-1.0)**abs(m/2) !int Yjm^{*}Yj1m1Yj2m2
!   gaunt=sqrt((j1+1)*(j2+1)*(j+1)/(4.0*pi)) * threej(j1,m1,j2,m2,j,m) * threej(j1,0,j2,0,j,0) !int Yjm Yj1m1Yj2m2

  end function gaunt

  function gaunt_resp2(j2,m2,j3,m3,j1,m1)
  integer :: j1,m1,j2,m2,j3,m3
  real(rk) :: gaunt_resp2,cleb_temp
  integer :: perm_j1, perm_j2, perm_j3, perm_m1, perm_m2, perm_m3, num_zero_m 
  
  num_zero_m=0
  if (m1.eq.0)  num_zero_m= num_zero_m+1
  if (m2.eq.0)  num_zero_m= num_zero_m+1
  if (m3.eq.0)   num_zero_m= num_zero_m+1

  if ( num_zero_m.eq.0 ) then
!  print *, 'no m is zero', m1,m2,m

     gaunt_resp2=2*gaunt(j2,m2,j3,m3,j1,m2+m3)*real(conjg(basis_transform(j1,m1,(m2+m3)))*basis_transform(j2,m2,m2)&
          *basis_transform(j3,m3,m3)) &
          +2*gaunt(j2,m2,j3,-m3,j1,m2-m3)*real(conjg(basis_transform(j1,m1,(m2-m3)))*basis_transform(j2,m2,m2)&
          *basis_transform(j2,m3,-m3))
    
!   write(6,'(2(3i3,2x))') j1/2,m1/2,(m2+m3)/2, j1/2,m1/2,(m2-m3)/2
!   write(6,'(10e15.5)') gaunt_resp2, 2*gaunt(j2,m2,j3,m3,j1,m2+m3), conjg(basis_transform(j1,m1,(m2+m3))),basis_transform(j2,m2,m2),basis_transform(j3,m3,m3)
!   write(6,'(10e15.5)') gaunt_resp2, 2*gaunt(j2,m2,j3,-m3,j1,m2-m3),conjg(basis_transform(j1,m1,(m2-m3))),basis_transform(j2,m2,m2),basis_transform(j2,m3,-m3)

  elseif( num_zero_m.eq.1 ) then
!  print *, 'one m is zero', m1,m2,m
     !May need to permutate
     if (m1.eq.0) then
        perm_j1=j2
        perm_m1=m2
        perm_j2=j3
        perm_m2=m3
        perm_j3 =j1
        perm_m3=m1
     elseif (m2.eq.0) then
        perm_j1=j3
        perm_m1=m3
        perm_j2=j1
        perm_m2=m1
        perm_j3 =j2
        perm_m3=m2
     else
        perm_j1=j1
        perm_m1=m1
        perm_j2=j2
        perm_m2=m2
        perm_j3 =j3
        perm_m3=m3       
     end if

!      gaunt=2*cleb(perm_j1,perm_m1,perm_j2,perm_m2,perm_j,0)*real(conjg(basis_transform(perm_j1,perm_m1,perm_m2))*basis_transform(perm_j2,perm_m2,perm_m2))
     gaunt_resp2=2*gaunt(perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1)*real(conjg(basis_transform(perm_j1,perm_m1,perm_m2))&
                  *basis_transform(perm_j2,perm_m2,perm_m2))
!      write(6,*) "hello"
!      write(6,'(3(2i3,2x))') perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1
!      write(6,'(10e15.5)') 2*gaunt(perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1),conjg(basis_transform(perm_j1,perm_m1,perm_m2)),basis_transform(perm_j2,perm_m2,perm_m2)
  else
     if( num_zero_m.eq.3 ) then
! print *, 'all m are zero', m1,m2,m
        gaunt_resp2=gaunt(j2,0,j3,0,j1,0)
     else
        gaunt_resp2=0
     end if
  end if

  end function gaunt_resp2

  function gaunt_resp4(j2,m2,j3,m3,j1,m1)
  integer :: j1,m1,j2,m2,j3,m3
  real(rk) :: gaunt_resp4,cleb_temp
  integer :: perm_j1, perm_j2, perm_j3, perm_m1, perm_m2, perm_m3, num_zero_m 
  
  num_zero_m=0
  if (m1.eq.0)  num_zero_m= num_zero_m+1
  if (m2.eq.0)  num_zero_m= num_zero_m+1
  if (m3.eq.0)   num_zero_m= num_zero_m+1

  if ( num_zero_m.eq.0 ) then
!  print *, 'no m is zero', m1,m2,m

     gaunt_resp4=gaunt(j2,m2,j3,m3,j1,m2+m3) *( conjg(basis_transform(j1,m1,(m2+m3)))*basis_transform(j2,m2,m2)&
                                                 *basis_transform(j3,m3,m3)+conjg(basis_transform(j1,m1,-(m2+m3)))&
                                                 *basis_transform(j2,m2,-m2)*basis_transform(j3,m3,-m3) ) &
                +gaunt(j2,m2,j3,-m3,j1,m2-m3)*( conjg(basis_transform(j1,m1,(m2-m3)))*basis_transform(j2,m2,m2)&
                *basis_transform(j2,m3,-m3)   &
                +conjg(basis_transform(j1,m1,-(m2-m3)))*basis_transform(j2,m2,-m2)*basis_transform(j2,m3,m3)  )
    
!   write(6,'(2(3i3,2x))') j1/2,m1/2,(m2+m3)/2, j1/2,m1/2,(m2-m3)/2
!   write(6,'(10e15.5)') gaunt_resp4, 2*gaunt(j2,m2,j3,m3,j1,m2+m3), conjg(basis_transform(j1,m1,(m2+m3))),basis_transform(j2,m2,m2),basis_transform(j3,m3,m3)
!   write(6,'(10e15.5)') gaunt_resp4, 2*gaunt(j2,m2,j3,-m3,j1,m2-m3),conjg(basis_transform(j1,m1,(m2-m3))),basis_transform(j2,m2,m2),basis_transform(j2,m3,-m3)

  elseif( num_zero_m.eq.1 ) then
!  print *, 'one m is zero', m1,m2,m
     !May need to permutate
     if (m1.eq.0) then
        perm_j1=j2
        perm_m1=m2
        perm_j2=j3
        perm_m2=m3
        perm_j3 =j1
        perm_m3=m1
     elseif (m2.eq.0) then
        perm_j1=j3
        perm_m1=m3
        perm_j2=j1
        perm_m2=m1
        perm_j3 =j2
        perm_m3=m2
     else
        perm_j1=j1
        perm_m1=m1
        perm_j2=j2
        perm_m2=m2
        perm_j3 =j3
        perm_m3=m3       
     end if

!      gaunt=2*cleb(perm_j1,perm_m1,perm_j2,perm_m2,perm_j,0)*real(conjg(basis_transform(perm_j1,perm_m1,perm_m2))*basis_transform(perm_j2,perm_m2,perm_m2))
     gaunt_resp4=gaunt(perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1)*( conjg(basis_transform(perm_j1,perm_m1,perm_m2))&
                                                                         *basis_transform(perm_j2,perm_m2,perm_m2) &
                                                                         +conjg(basis_transform(perm_j1,perm_m1,-perm_m2))&
                                                                         *basis_transform(perm_j2,perm_m2,-perm_m2) )
!      write(6,*) "hello"
!      write(6,'(3(2i3,2x))') perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1
!      write(6,'(10e15.5)') 2*gaunt(perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1),conjg(basis_transform(perm_j1,perm_m1,perm_m2)),basis_transform(perm_j2,perm_m2,perm_m2)
  else
     if( num_zero_m.eq.3 ) then
! print *, 'all m are zero', m1,m2,m
        gaunt_resp4=gaunt(j2,0,j3,0,j1,0)
     else
        gaunt_resp4=0
     end if
  end if

  end function gaunt_resp4

  function gaunt_resp3(j2,m2,j3,m3,j1,m1)
  integer :: j1,m1,j2,m2,j3,m3
  real(rk) :: gaunt_resp3,cleb_temp
  integer :: temp_m,temp_j, num_zero_m, num_neg_m 
  
  num_zero_m=0
  if (m1.eq.0)  num_zero_m= num_zero_m+1
  if (m2.eq.0)  num_zero_m= num_zero_m+1
  if (m3.eq.0)   num_zero_m= num_zero_m+1
  num_neg_m=0
  if (m1.lt.0)  num_neg_m= num_neg_m+1
  if (m2.lt.0)  num_neg_m= num_neg_m+1
  if (m3.lt.0)  num_neg_m= num_neg_m+1

  if ( num_zero_m.eq.0 ) then
     if ( num_neg_m.eq.0 ) then
        if     ( m1.eq.(m2+m3)) then
           gaunt_resp3=-1.0**(m1/2)*0.5*sqrt(2.0)*gaunt(j2,m2,j3,m3,j1,-m1)
        elseif ( m2.eq.(m1+m3)) then
           gaunt_resp3=-1.0**(m2/2)*0.5*sqrt(2.0)*gaunt(j2,-m2,j3,m3,j1,m1)
        elseif ( m3.eq.(m1+m2)) then
           gaunt_resp3=-1.0**(m3/2)*0.5*sqrt(2.0)*gaunt(j2,m2,j3,-m3,j1,m1)
        else
           gaunt_resp3=0
        end if
     elseif ( num_neg_m.eq.2 ) then
        if (m2.gt.0) then
           temp_m=m1
           temp_j=j1
           m1=m2
           j1=j2    
           m2=temp_m
           j2=temp_j
        elseif (m3.gt.0) then
           temp_m=m1
           temp_j=j1
           m1=m3
           j1=j3    
           m3=temp_m
           j3=temp_j
        end if
        if     ( m2.eq.(m1+m3)) then
           gaunt_resp3=-1.0**(m3/2)*0.5*sqrt(2.0)*gaunt(j2,-m2,j3,m3,j1,m1)
        elseif ( m3.eq.(m1+m2)) then
           gaunt_resp3=-1.0**(m2/2)*0.5*sqrt(2.0)*gaunt(j2,m2,j3,-m3,j1,m1)
        elseif ( (m1+m2+m3) .eq. 0 ) then
           gaunt_resp3=-(-1.0)**(m1/2)*0.5*sqrt(2.0)*gaunt(j2,m2,j3,m3,j1,m1)
        else
           gaunt_resp3=0
        end if
     else
        gaunt_resp3=0
     end if

  elseif( num_zero_m.eq.1 ) then
        if     ( (m1.eq.0) .and. (m2.eq.m3)) then
           gaunt_resp3=-1.0**(m2/2)*gaunt(j2,m2,j3,-m2,j1,0)
        elseif ( (m2.eq.0) .and. (m1.eq.m3)) then
           gaunt_resp3=-1.0**(m3/2)*gaunt(j2,0,j3,m3,j1,-m3)
        elseif ( (m3.eq.0) .and. (m1.eq.m2)) then
           gaunt_resp3=-1.0**(m1/2)*gaunt(j2,-m1,j3,0,j1,m1)
        else
           gaunt_resp3=0
        end if
  elseif( num_zero_m.ge.2 ) then
     gaunt_resp3=gaunt(j2,0,j3,0,j1,0)
  else
     gaunt_resp3=gaunt(j2,0,j3,0,j1,0)
  end if

  end function gaunt_resp3


  function clebsch_resp4(j2,m2,j3,m3,j1,m1)
  integer :: j1,m1,j2,m2,j3,m3
  real(rk) :: clebsch_resp4,cleb_temp
  integer :: perm_j1, perm_j2, perm_j3, perm_m1, perm_m2, perm_m3, num_zero_m 
  
  num_zero_m=0
  if (m1.eq.0)  num_zero_m= num_zero_m+1
  if (m2.eq.0)  num_zero_m= num_zero_m+1
  if (m3.eq.0)   num_zero_m= num_zero_m+1

  if ( num_zero_m.eq.0 ) then
!  print *, 'no m is zero', m1,m2,m

     clebsch_resp4=cleb(j2,m2,j3,m3,j1,m2+m3) *( conjg(basis_transform(j1,m1,(m2+m3)))*basis_transform(j2,m2,m2)&
                                                 *basis_transform(j3,m3,m3)   &
                                                 +conjg(basis_transform(j1,m1,-(m2+m3)))*basis_transform(j2,m2,-m2)&
                                                 *basis_transform(j3,m3,-m3) ) &
                +cleb(j2,m2,j3,-m3,j1,m2-m3)*( conjg(basis_transform(j1,m1,(m2-m3)))*basis_transform(j2,m2,m2)&
                                                 *basis_transform(j2,m3,-m3)   &
                                                 +conjg(basis_transform(j1,m1,-(m2-m3)))*basis_transform(j2,m2,-m2)&
                                                 *basis_transform(j2,m3,m3)  )
    
!   write(6,'(2(3i3,2x))') j1/2,m1/2,(m2+m3)/2, j1/2,m1/2,(m2-m3)/2
!   write(6,'(10e15.5)') gaunt_resp4, 2*gaunt(j2,m2,j3,m3,j1,m2+m3), conjg(basis_transform(j1,m1,(m2+m3))),basis_transform(j2,m2,m2),basis_transform(j3,m3,m3)
!   write(6,'(10e15.5)') gaunt_resp4, 2*gaunt(j2,m2,j3,-m3,j1,m2-m3),conjg(basis_transform(j1,m1,(m2-m3))),basis_transform(j2,m2,m2),basis_transform(j2,m3,-m3)

  elseif( num_zero_m.eq.1 ) then
!  print *, 'one m is zero', m1,m2,m
     !May need to permutate
     if (m1.eq.0) then
        perm_j1=j2
        perm_m1=m2
        perm_j2=j3
        perm_m2=m3
        perm_j3 =j1
        perm_m3=m1
     elseif (m2.eq.0) then
        perm_j1=j3
        perm_m1=m3
        perm_j2=j1
        perm_m2=m1
        perm_j3 =j2
        perm_m3=m2
     else
        perm_j1=j1
        perm_m1=m1
        perm_j2=j2
        perm_m2=m2
        perm_j3 =j3
        perm_m3=m3       
     end if

!      gaunt=2*cleb(perm_j1,perm_m1,perm_j2,perm_m2,perm_j,0)*real(conjg(basis_transform(perm_j1,perm_m1,perm_m2))*basis_transform(perm_j2,perm_m2,perm_m2))
     clebsch_resp4=cleb(perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1)*( conjg(basis_transform(perm_j1,perm_m1,perm_m2))&
                                                                           *basis_transform(perm_j2,perm_m2,perm_m2) &
                                                                           +conjg(basis_transform(perm_j1,perm_m1,-perm_m2))&
                                                                           *basis_transform(perm_j2,perm_m2,-perm_m2) )
!      write(6,*) "hello"
!      write(6,'(3(2i3,2x))') perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1
!      write(6,'(10e15.5)') 2*gaunt(perm_j2,perm_m2,perm_j3,perm_m3,perm_j1,perm_m1),conjg(basis_transform(perm_j1,perm_m1,perm_m2)),basis_transform(perm_j2,perm_m2,perm_m2)
  else
     if( num_zero_m.eq.3 ) then
! print *, 'all m are zero', m1,m2,m
        clebsch_resp4=cleb(j2,0,j3,0,j1,0)
     else
        clebsch_resp4=0
     end if
  end if

  end function clebsch_resp4

  function real_threej(j1,m1,j2,m2,J,M)
  integer :: j1,m1,j2,m2,J,M
  real(rk) :: real_threej

  real_threej=0.0
!  write(999999, '(3(2i3,x),10d20.5,"3j1")'), j1,m1,j2,m2,J,M
  real_threej = threej(j1,m1,j2,m2,J,M)   *conjg(basis_transform(J,-M,-M)) *basis_transform(j1,m1, m1) *basis_transform(j2,m2, m2)&
             + threej(j1,m1,j2,-m2,J,M)   *conjg(basis_transform(J,-M,-M)) *basis_transform(j1,m1, m1) *basis_transform(j2,m2,-m2)&
             + threej(j1,-m1,j2,m2,J,M)   *conjg(basis_transform(J,-M,-M)) *basis_transform(j1,m1,-m1) *basis_transform(j2,m2, m2)&
             + threej(j1,-m1,j2,-m2,J,M)  *conjg(basis_transform(J,-M,-M)) *basis_transform(j1,m1,-m1) *basis_transform(j2,m2,-m2)&
             + threej(j1,m1,j2,m2,J,-M)   *conjg(basis_transform(J,-M, M)) *basis_transform(j1,m1, m1) *basis_transform(j2,m2, m2)&
             + threej(j1,m1,j2,-m2,J,-M)  *conjg(basis_transform(J,-M, M)) *basis_transform(j1,m1, m1) *basis_transform(j2,m2,-m2)&
             + threej(j1,-m1,j2,m2,J,-M)  *conjg(basis_transform(J,-M, M)) *basis_transform(j1,m1,-m1) *basis_transform(j2,m2, m2)&
             + threej(j1,-m1,j2,-m2,J,-M) *conjg(basis_transform(J,-M, M)) *basis_transform(j1,m1,-m1) *basis_transform(j2,m2,-m2)


  end function real_threej

  function real_threej2(j1,m1,j2,m2,J,M)
  integer :: j1,m1,j2,m2,J,M
  real(rk) :: real_threej2
  real_threej2=0.0
!   write(999999, '(3(2i3,x),10d20.5,"3j2")'), j1,m1,j2,m2,J,M
  do m1p=-j1,j1,2
     do m2p=-j2,j2,2
        do Mp=-J,J,2

           if ((abs(m1) .eq. abs(m1p)) .and. (abs(m2).eq.abs(m2p)) .and. (abs(M).eq.abs(Mp)) ) then
              real_threej2=real_threej2+threej(j1,m1p,j2,m2p,J,Mp)*conjg(basis_transform(J,-M,-Mp))*basis_transform(j1,m1,m1p)&
                           *basis_transform(j2,m2,m2p) 
!               write(999999, '(3(2i3,x),10d20.5)'), j1,m1p,j2,m2p,J,Mp, threej(j1,m1p,j2,m2p,J,Mp)*conjg(basis_transform(J,-M,-Mp))*basis_transform(j1,m1,m1p)*basis_transform(j2,m2,m2p)
           end if

        end do
     end do
  end do

  end function real_threej2


  function clebsch_resp_slow(j1,m1,j2,m2,J,M)
  integer :: j1,m1,j2,m2,J,M
  real(rk) :: clebsch_resp_slow,cleb_temp

  
  clebsch_resp_slow=(-1.0)**(-j1/2+j2/2-M/2)*sqrt(real(J+1.0))*real_threej2(j1,m1,j2,m2,J,-M)

  end function clebsch_resp_slow

  function clebsch_resp_slowest(j1,m1,j2,m2,J,M)
  integer :: j1,m1,j2,m2,J,M, m1p,m2p, Mp
  real(rk) :: clebsch_resp_slowest


  clebsch_resp_slowest=0.0
  do m1p=-j1,j1,2
     do m2p=-j2,j2,2
        do Mp=-J,J,2
           if ((abs(m1) .eq. abs(m1p)) .and. (abs(m2).eq.abs(m2p)) .and. (abs(M).eq.abs(Mp)) ) then
           clebsch_resp_slowest=clebsch_resp_slowest+cleb(j1,m1p,j2,m2p,J,Mp)*conjg(basis_transform(J,M,Mp))&
                                *basis_transform(j1,m1,m1p)*basis_transform(j2,m2,m2p) 
           end if
        end do
     end do
  end do

  end function clebsch_resp_slowest

  function sixj(a,b,c,d,e,f)
    implicit none
    integer, intent(in) :: a,b,c,d,e,f
    real(rk) :: sixj
    integer :: phase, nlo, nhi, n
    real(rk) :: outfactors, sum, sumterm
    ! calculates a Wigner 6-j symbol. Argument a-f are integer and are
    ! twice the true value of the 6-j's arguments, in the form
    ! { a b c }
    ! { d e f }
    ! Calculated using binomial coefficients to allow for (reasonably) high
    ! arguments.

    ! First check for consistency of arguments:
    sixj=0.0_rk
    if(mod(a+b,2)/=mod(c,2)) return
    if(mod(c+d,2)/=mod(e,2)) return
    if(mod(a+e,2)/=mod(f,2)) return
    if(mod(b+d,2)/=mod(f,2)) return
    if(abs(a-b)>c .or. a+b<c) return
    if(abs(c-d)>e .or. c+d<e) return
    if(abs(a-e)>f .or. a+e<f) return
    if(abs(b-d)>f .or. b+d<f) return

    outfactors = angdelta(a,e,f)/angdelta(a,b,c)
    outfactors = outfactors * angdelta(b,d,f)*angdelta(c,d,e)

    nlo = max( (a+b+c)/2, (c+d+e)/2, (b+d+f)/2, (a+e+f)/2 )
    nhi = min( (a+b+d+e)/2, (b+c+e+f)/2, (a+c+d+f)/2)

    sum=0.0_rk
    do n=nlo,nhi
       sumterm = (-1)**n
       sumterm = sumterm * binom(n+1,n-(a+b+c)/2)
       sumterm = sumterm * binom((a+b-c)/2,n-(c+d+e)/2)
       sumterm = sumterm * binom((a-b+c)/2,n-(b+d+f)/2)
       sumterm = sumterm * binom((b-a+c)/2,n-(a+e+f)/2)
       sum=sum+sumterm
    end do

    sixj = sum * outfactors

  end function sixj

  function angdelta(a,b,c)
    implicit none
    integer :: a,b,c
    real(rk)    :: angdelta, scr1
    ! calculate the function delta as defined in varshalovich et al. for
    ! use in 6-j symbol:
    scr1= factorial((a+b-c)/2)
    scr1=scr1/factorial((a+b+c)/2+1)
    scr1=scr1*factorial((a-b+c)/2)
    scr1=scr1*factorial((-a+b+c)/2)
    angdelta=sqrt(scr1)
  end function angdelta

  function ninej(a,b,c,d,e,f,g,h,i)
    implicit none
    integer :: a,b,c,d,e,f,g,h,i
    real(rk)    :: ninej, sum
    integer :: xlo, xhi
    integer :: x
    ! calculate a 9-j symbol. The arguments are given as integers twice the
    ! value of the true arguments in the form
    ! { a b c }
    ! { d e f }
    ! { g h i }

    ninej=0.0
    ! first check for bogus arguments (and return zero if so)
    if(abs(a-b)>c .or. a+b<c) return
    if(abs(d-e)>f .or. d+e<f) return
    if(abs(g-h)>i .or. g+h<i) return
    if(abs(a-d)>g .or. a+d<g) return
    if(abs(b-e)>h .or. b+e<h) return
    if(abs(c-f)>i .or. c+f<i) return
    
    xlo = max(abs(b-f),abs(a-i),abs(h-d))
    xhi = min(b+f,a+i,h+d)
    
    sum=0.0
    do x=xlo,xhi,2
       sum=sum+(-1)**x*(x+1)*sixj(a,b,c,f,i,x)*sixj(d,e,f,b,x,h)*&
            sixj(g,h,i,x,a,d)
    end do
    ninej=sum

  end function ninej

  recursive function factorial(n) result(res)
    implicit none
    integer :: n
    real(rk) :: res

    if (n==0 .or. n==1) then
       res=1.0
    else
       res=n*factorial(n-1)
    end if
  end function factorial


  recursive function binom(n,r) result(res)
    implicit none
    integer :: n,r
    real(rk) :: res
    real(rk) :: tmp

    if(n==r .or. r==0) then
       res = 1.0
    else if (r==1) then
       res = real(n,rk)
    else
       res = real(n,rk)/real(n-r,rk)*binom(n-1,r)
    end if
  end function binom
end module anglib
