classdef rmatrix_data
    %RMATRIX_DATA Interface to the R-matrix data needed for HHG. Version 1.0, 15/10/2016, ZM.
    %
    %   Initialization via method rmatrix_data (input_files,every_nth_energy)
    %   --------------------------------------
    %
    %       input_files: a cell vector of paths to the files of the type dipoles_data_matlab containing the data generated by the R-matrix program dipole_tools.
    %       every_nth_energy: integer specifying that data for only every n-th energy should be considered.
    %
    %       Following a call to rmatrix_data the methods show_states, Dyson_signs, momentum_space_photodipoles, initial_state_dipoles, final_state_dipoles,
    %       initial_state_energy, final_state_energy are available.
    %
    %       Units: length in Bohr, dipole moments in atomic units.
    %
    %   Methods
    %   -------
    %
    %       show_states
    %       ===========
    %
    %       Input: none
    %       ------
    %
    %       Output:
    %       -------
    %       List of states and their indices for which dipole and energy data is available.
    %
    %       Dyson_signs
    %       ===========
    %
    %       Input: (final_state,initial_state,grid_vectors)
    %       ------
    %       final_state: index of the final state
    %       initial_state: index of the initial state
    %       grid_vectors: optional, cell array of size 2.
    %                     grid_vectors{1}: grid for internal coordinate 1 (bending angle gamma in degrees in case of NO2)
    %                     grid_vectors{2}: grid for internal coordinate 2 (asymmetric stretch ra in Bohr in case of NO2)
    %                     If grid_vectors are not given then the Dyson orbital signs are evaluated for geometry with index 1.
    %
    %       Output: integer(n,n_1,n_2)
    %       -------
    %       Array of Dyson orbital signs evaluated for all angular points (n) and the gamma, ra nuclear grid. In case the nuclear grid grid_vectors was not given on input the
    %       Dyson orbital signs are evaluated for geometry with index 1 and the output array was dimensions (n,1).
    %       IMPORTANT: the Dyson orbital signs are not interpolated accross the nuclear grid. Instead, for each set of nuclear coordinates from grid_vectors the Dyson orbital 
    %                  signs are taken from the nearest sampling point.
    %       n_1: size of the grid for internal coordinate 1 (gamma in case of NO2).
    %       n_2: size of the grid for internal coordinate 2 (ra in case of NO2).
    %
    %       momentum_space_photodipoles
    %       ===========================
    %
    %       Input: (theta,phi,initial_state,final_state,grid_vectors)
    %       ------
    %       theta, phi: spherical coordinates in radians of the direction of the emitted electron.
    %       initial_state: index of the initial state
    %       final_state: index of the final state
    %       grid_vectors: cell array of size at least 1.
    %                     grid_vectors{1}: grid of SCATTERING ELECTRON ENERGIES in Hartree measured wrt ground final state.
    %                                      I.e. these are electron energies only if the dipoles come from a single-state (HF) calculation.
    %                     grid_vectors{2}: optional, grid for internal coordinate 1 (bending angle in degrees in case of NO2)
    %                     grid_vectors{3}: optional, grid for internal coordinate 2 (asymmetric stretch in Bohr in case of NO2)
    %                     If grid_vectors{2} is given then grid_vectors{3} must be present too.
    %                     If only grid_vectors{1} is given then the photodipoles are evaluated for geometry with index 1.
    %
    %       Output: complex(3,n_en,n_1,n_2)
    %       -------
    %       Photoelectron dipoles (x,y,z) evaluated for the given combination of initial and final states and photoelectron emission direction.
    %       n_en: size of the energy grid.
    %       n_1: size of the grid for internal coordinate 1 (gamma in case of NO2).
    %       n_2: size of the grid for internal coordinate 2 (ra in case of NO2).
    %       The photoelectron dipoles are evaluated interpolating the partial wave dipole matrix elements on the grid of energies and nuclear coordinates.
    %
    %       initial_state_dipoles
    %       =====================
    %
    %       Input: (initial_stateI,initial_stateF,grid_vectors)
    %       ------
    %       initial_stateI,initial_stateF: indices of the pair of initial states for which the dipole matrix elements are required.
    %       grid_vectors: optional, cell array of size corresponding to the number of internal nuclear coordinates.
    %                     If the grid_vectors are not given then the dipoles are evaluated for geometry with index 1.
    %
    %       Output: double(3,n_1,n_2)
    %       -------
    %       Dipole matrix elements (x,y,z) between the given initial states on the specified grid of nuclear coordinates.
    %       n_1: size of the grid for internal coordinate 1 (gamma in case of NO2).
    %       n_2: size of the grid for internal coordinate 2 (ra in case of NO2).
    %
    %       final_state_dipoles
    %       =====================
    %
    %       Input: (final_stateI,final_stateF,grid_vectors)
    %       ------
    %       final_stateI,final_stateF: indices of the pair of final states for which the dipole matrix elements are required.
    %       grid_vectors: optional, cell array of size corresponding to the number of internal nuclear coordinates.
    %                     If the grid_vectors are not given then the dipoles are evaluated for geometry with index 1.
    %
    %       Output: double(3,n_1,n_2)
    %       -------
    %       Dipole matrix elements (x,y,z) between the given final states on the specified grid of nuclear coordinates.
    %       n_1: size of the grid for internal coordinate 1 (gamma in case of NO2).
    %       n_2: size of the grid for internal coordinate 2 (ra in case of NO2).
    %
    %       initial_state_energy
    %       =====================
    %
    %       Input: (initial_state,grid_vectors)
    %       ------
    %       initial_state: index of the initial state.
    %       grid_vectors: optional, cell array of size corresponding to the number of internal nuclear coordinates.
    %                     If the grid_vectors are not given then the energy is evaluated for geometry with index 1.
    %
    %       Output: double(n_1,n_2)
    %       -------
    %       Energies of the given initial state on the specified grid of nuclear coordinates.
    %       n_1: size of the grid for internal coordinate 1 (gamma in case of NO2).
    %       n_2: size of the grid for internal coordinate 2 (ra in case of NO2).
    %
    %       final_state_energy
    %       =====================
    %
    %       Input: (final_state,grid_vectors)
    %       ------
    %       final_state: index of the final state.
    %       grid_vectors: optional, cell array of size corresponding to the number of internal nuclear coordinates.
    %                     If the grid_vectors are not given then the energy are evaluated for geometry with index 1.
    %
    %       Output: double(n_1,n_2)
    %       -------
    %       Energies of the given final state on the specified grid of nuclear coordinates.    
    %       n_1: size of the grid for internal coordinate 1 (gamma in case of NO2).
    %       n_2: size of the grid for internal coordinate 2 (ra in case of NO2).
    %
    %   Data structures
    %   ---------------
    %
    %Source_files: names of the files that were used to construct rmatrix_data.
    %Molecule: name of the molecule.
    %Max_L: The largest continuum angular momentum.
    %N_energies: number of continuum electron energies.
    %N_final_states: number of cationic final states.
    %N_initial_states: number of bound initial states.
    %N_angular_points: number of angular grid points that were used to evaluate the Dyson orbital signs.
    %N_atoms: number of atoms in the molecule including the scattering centre.
    %Final_states: cell array of dimensions (3,N_final_states).
    %              Final_states{1,i}: string containing the label for the i-th state (number, symmetry, spin).
    %              Final_states{2,i}: energy (in H) of the final state.
    %              Final_states{3,i}: array of dimensions (3,N_final_states) containing the three components of the dipole couplings between the i-th state and all other final states.
    %Initial_states: cell array of dimensions (3,N_initial_states). The data in the cells is of the same type but this time for the bound initial states.
    %Dyson_orbitals: cell array of dimensions (2,N_final_states,N_initial_states)
    %               Dyson_orbitals(1,f,i): string containing the label for the Dyson orbital corresponding to the initial state i and the final state f.
    %               Dyson_orbitals(2,f,i): array of dimension (N_angular_points,1) containing the signs of the Dyson orbitals evaluated on the angular grid.
    %Atoms: cell array of dimensions (2,N_atoms).
    %       Atoms{1,i}: vector (3,1) containing the coordinates (in Bohr) of the i-th atom.
    %       Atoms{2,i}: string containing the name of the i-th atom.
    %Internal_coordinates: array of dimensions(ninternal,N_geometries) containing, for each geometry, the internal nuclear coordinates. In case of NO2 ninternal = 2 corresponding to the gamma,ra coordinates.
    %                      This array is only constructed if N_geometries > 1.
    %Internal_coordinates_limits: array of dimensions (2,n), where n is the number of internal nuclear coordinates. At the moment n can be either 0 or 2. This array is only constructed if N_geometries > 1.
    %Angular_grid_cartesian: array with dimensions (N_angular_points,1) containing the grid points (x,y,z,w) defining the angular grid.
    %Angular_grid_spherical: the same as above but in spherical coordinates (theta,phi,w).
    %Electron_energies: array with dimensions (N_energies,1) containing the energies of the photoelectron.
    %Max_energy: The largest energy in the array Electron_energies.
    %Min_energy: The smallest energy in the array Electron_energies.
    properties (SetAccess = protected)
        Source_files
        Molecule
        Max_L
        N_energies
        N_final_states
        N_initial_states
        N_angular_points
        N_atoms
        N_geometries
        Final_states
        Initial_states
        Dyson_orbitals
        Atoms
        Internal_coordinates
        Internal_coordinates_limits
        Angular_grid_cartesian
        Angular_grid_spherical
        Electron_energies
        Max_energy
        Min_energy
    end
    
    %Xlm_precalculated: array of dimensions (N_lm,N_angular_points) containing the real spherical harmonics for each L,M up to L=Max_L evaluated on the angular grid Angular_grid_cartesian.
    %N_lm: number of L,M partial waves up to L=Max_L.
    %Have_interpolant: logical telling me if interpolation accross geometries is possible or not.
    %D_lm_xyz: cell array of dimensions (N_lm,3,N_final_states,N_initial_states) containing the griddedInterpolants which are used by 
    %          momentum_space_photodipoles when interpolating accross the internal coordinates (and electron energy).
    %Dipoles_initial: cell array of dimensions (3,N_initial_states,N_initial_states) containing the griddedInterpolants used by initial_state_dipoles when interpolating accross the internal coordinates.
    %Dipoles_final: cell array (3,N_final_states,N_final_states) containing the griddedInterpolants used by final_state_dipoles when interpolating accross the internal coordinates.
    %Energies_initial: cell array of dimensions (N_initial_states) containing the griddedInterpolants used by initial_state_energy when interpolating accross the internal coordinates.
    %Energies_final: cell array of dimensions (N_final_states) containing the griddedInterpolants used by final_state_energy when interpolating accross the internal coordinates.
    %Dipoles: array of dimensions (N_lm,3,N_energies,N_final_states,N_initial_states)
    %         Dipoles(lm,e,f,i): photoionization dipole matrix element for partial wave with index lm, energy index e, final state f and initial state i.    
    %Dyson_interpolant: cell array (N_angular_points,N_final_states,N_initial_states) containing the griddedInterpolants used by Dyson_signs.
    properties (Access = private)
        Xlm_precalculated
        N_lm
        Have_interpolant
        D_lm_xyz
        Dipoles_initial
        Dipoles_final
        Energies_initial
        Energies_final
        Dipoles        
        Dyson_interpolant
    end
    
    methods
         %Constructor: read the data file containing all R-matrix data
         function obj = rmatrix_data(input_files,every_nth_energy)
             
            if nargin == 1
                en_step = 1;
            elseif nargin == 2
                en_step = round(every_nth_energy);
                msg = strcat('Accepting energies in steps of:',num2str(en_step));
                disp(msg)
            else
                error('Expecting either one or two arguments on input.')
            end
            
            if not(iscell(input_files))
                error('The argument input_files must be a cell vector')
            end
                
            obj.N_geometries = size(input_files,1);
            obj.Source_files = input_files;
            obj.Have_interpolant = false;
            Dipoles_read = cell(obj.N_geometries,1);
            Energy_grid = cell(obj.N_geometries,1);
            N_energies_geom = zeros(obj.N_geometries,1);

            if obj.N_geometries > 1
                msg = strcat('Number of geometries on input:',num2str(obj.N_geometries));
                disp(msg)
            else
                disp('Data set for a single geometry will be generated.')
            end

            %Read-in data for all geometries
            for geom = 1:obj.N_geometries

                %The first geometry will be used as a reference one
                %for the energy, angular grid, etc. All other
                %geometries must contain the same data for the
                %reference values
                if geom == 1
                    is_ref_geom = 1;
                else
                    is_ref_geom = 0;
                end

                if not(ischar(input_files{geom,1}))
                    error('Path must be character')
                end
                msg = strcat('Reading geometry: ',input_files{geom,1});
                disp(msg)
                fileID = fopen([input_files{geom,1}],'r');

                C = fgetl(fileID);
                if is_ref_geom
                    obj.Molecule = C;
                elseif not(isequal(obj.Molecule,C))
                    disp(obj.Molecule)
                    disp(C)
                    error('Flag Molecule is not the same for all geometries.')
                end

                %Read the dimensions
                C = fscanf(fileID,'%d',7);
                N_energies_geom(geom) = C(4); %Number of electron energies for this geometry.

                if is_ref_geom
                    obj.Max_L = C(1);
                    obj.N_lm = C(2);
                    obj.N_final_states = C(3);
                    obj.N_angular_points = C(5);
                    obj.N_atoms = C(6);
                    obj.N_initial_states = C(7);

                    %Declarations
                    obj.Initial_states = cell(3,obj.N_initial_states,obj.N_geometries);
                    obj.Final_states = cell(6,obj.N_final_states,obj.N_geometries);
                    obj.Atoms = cell(2,obj.N_atoms,obj.N_geometries);
                    obj.Angular_grid_cartesian = zeros(4,obj.N_angular_points); %x,y,z,w
                    obj.Angular_grid_spherical = zeros(3,obj.N_angular_points); %theta,phi,w
                    obj.Xlm_precalculated = zeros(obj.N_lm,obj.N_angular_points);
                    obj.Dyson_orbitals = cell(2,obj.N_final_states,obj.N_initial_states,obj.N_geometries);
                else
                    N_energies_geom(geom) = C(4);
                    C(4) = 0; %we omit the energy grid from the check since we determine the common set of energy points at the end
                    ref = [obj.Max_L;obj.N_lm;obj.N_final_states;0;obj.N_angular_points;obj.N_atoms;obj.N_initial_states];
                    if not(isequal(ref,C))
                        disp(ref)
                        disp(C)
                        error('Dimension parameters are not the same for all geometries.')
                    end
                end

                for i=1:obj.N_initial_states
                    C = fscanf(fileID,'%d',3);
                    D = fscanf(fileID,'%e',1);
                    %row 1 = initial state indices:
                    Snum = strcat(' Number = ',num2str(C(1)));   %relative state number within its spin/space symmetry
                    Ssym = strcat(' Symmetry = ',num2str(C(2))); %state symmetry
                    Sspin = strcat(' Spin = ',num2str(C(3)));    %state spin
                    obj.Initial_states{1,i,geom} = strcat(Snum,Ssym);
                    obj.Initial_states{1,i,geom} = strcat(obj.Initial_states{1,i,geom},Sspin);     
                    obj.Initial_states{2,i,geom} = D(1); %row 2 = initial state energy

                    if not(is_ref_geom)
                        if not(isequal(obj.Initial_states{1,i,geom},obj.Initial_states{1,i,1}))
                            error('Initial state symmetry and/or number are not the same for all geometries.')
                        end
                    end
                end

                %Dipole couplings between the initial states.
                for i=1:obj.N_initial_states
                    D = zeros(3,obj.N_initial_states);
                    for j=1:obj.N_initial_states
                        C = fscanf(fileID,'%e',3);
                        D(1:3,j) = C;
                    end
                    obj.Initial_states{3,i,geom} = D;
                end

                for i = 1:obj.N_final_states
                    C = fscanf(fileID,'%d',3);
                    D = fscanf(fileID,'%e',1);
                    %row 1 = final state indices:
                    Snum = strcat(' Number = ',num2str(C(1)));   %relative state number within its spin/space symmetry
                    Ssym = strcat(' Symmetry = ',num2str(C(2))); %state symmetry
                    Sspin = strcat(' Spin = ',num2str(C(3)));    %state spin
                    obj.Final_states{1,i,geom} = strcat(Snum,Ssym);
                    obj.Final_states{1,i,geom} = strcat(obj.Final_states{1,i,geom},Sspin);
                    %row 2 = final state energy
                    obj.Final_states{2,i,geom} = D(1);

                    if not(is_ref_geom)
                        if not(isequal(obj.Final_states{1,i,geom},obj.Final_states{1,i,1}))
                            error('Final state symmetry and/or number are not the same for all geometries.')
                        end
                    end                         
                end

                %Dipole couplings between the final states.
                for i=1:obj.N_final_states
                    D = zeros(3,obj.N_final_states);
                    for j=1:obj.N_final_states
                        C = fscanf(fileID,'%e',3);
                        D(1:3,j) = C;
                    end
                    obj.Final_states{3,i,geom} = D;
                end

                %Read the atom coordinates (Bohr)
                for i=1:obj.N_atoms
                    C = fscanf(fileID,'%e',3);
                    obj.Atoms{1,i,geom} = C;
                end

                %Read the atom names
                for i=1:obj.N_atoms
                    C = fscanf(fileID,'%s',1);
                    obj.Atoms{2,i,geom} = C;

                    if not(is_ref_geom)
                        if not(isequal(obj.Atoms{2,i,geom},obj.Atoms{2,i,1}))
                            error('Atom names for different geometries are either different or permuted.')
                        end
                    end                            
                end

                %Read the list of x,y,z points defining the angular
                %grid and precalculate the real spherical harmonics on this grid.
                for i = 1:obj.N_angular_points
                    C = fscanf(fileID,'%e',4);
                    if is_ref_geom
                        obj.Angular_grid_cartesian(1,i) = C(1); %x
                        obj.Angular_grid_cartesian(2,i) = C(2); %y
                        obj.Angular_grid_cartesian(3,i) = C(3); %z
                        obj.Angular_grid_cartesian(4,i) = C(4); %quadrature weight
                        %note that theta and phi are swapped since xyz_to_tp uses the Mathematical convention for spherical coordinates!
                        [obj.Angular_grid_spherical(2,i) obj.Angular_grid_spherical(1,i)]= obj.xyz_to_tp(obj.Angular_grid_cartesian(1,i),obj.Angular_grid_cartesian(2,i),obj.Angular_grid_cartesian(3,i));

                        %Precalculate Xlm:
                        obj.Angular_grid_spherical(3,i) = C(4);
                        theta = obj.Angular_grid_spherical(1,i);
                        phi = obj.Angular_grid_spherical(2,i);
                        lm = 0;
                        for l=0:obj.Max_L
                            for m=-l:l
                                lm = lm + 1;
                                obj.Xlm_precalculated(lm,i) = obj.re_sp_harm(l,m,theta,phi);
                            end
                        end
                    elseif not(isequal(obj.Angular_grid_cartesian(:,i),C))
                        error('The angular grid is not the same for all geometries.')
                    end
                end

                %Read the energy grid
                Energy_grid{geom,1} = zeros(N_energies_geom(geom,1),1);
                Energy_grid{geom,1} = fscanf(fileID,'%e',N_energies_geom(geom,1));

                %Read the Dyson orbital signs evaluated on the angular grid.
                for k = 1:obj.N_initial_states
                    for i = 1:obj.N_final_states
                        C = fscanf(fileID,'%d',2);
                        Dnum = strcat(' Dyson orbital number = ',num2str(C(1)));
                        Dsym = strcat(' Dyson orbital symmetry = ',num2str(C(2)));
                        %Number and symmetry of the Dyson orbital (only needed for book keeping so I know which Dysons were taken from the R-matrix files).
                        flag = strcat(Dnum,Dsym);
                        if is_ref_geom
                            obj.Dyson_orbitals{1,i,k} = flag;
                        %elseif not(flag == obj.Dyson_orbitals{1,i,k})
                        %    error('The Dyson orbital flags are not the same for all geometries.')
                        end
                        Dyson_signs = zeros(obj.N_angular_points,1);                            
                        for j = 1:obj.N_angular_points
                            Dyson_signs(j) = fscanf(fileID,'%d',1);
                        end
                        obj.Dyson_orbitals{2,i,k,geom} = Dyson_signs;
                    end
                end

                %Read the partial wave photoionization dipoles.
                Re = fscanf(fileID,'%e',obj.N_lm*3*N_energies_geom(geom,1)*obj.N_final_states*obj.N_initial_states);
                Im = fscanf(fileID,'%e',obj.N_lm*3*N_energies_geom(geom,1)*obj.N_final_states*obj.N_initial_states);
                Dipoles_read{geom,1} = reshape(Re,[obj.N_lm,3,N_energies_geom(geom,1),obj.N_final_states,obj.N_initial_states]) + 1j*reshape(Im,[obj.N_lm,3,N_energies_geom(geom,1),obj.N_final_states,obj.N_initial_states]);

                fclose(fileID);
            end
            
            %Use only every en_step-th energy as requested on input.
            n = N_energies_geom(1,1);
            all_energies = [Energy_grid{1,1}];
            energies = zeros(n,1);
            j = 0;
            for i=1:n
                if mod(i,en_step) == 0
                    j = j + 1;
                    energies(j,1) = all_energies(i,1);
                end
            end
            n = j;            

            if obj.N_geometries > 1
                
                %Find the common set of energy points, use the grid for geometry 1 as the set of starting common points.                
                energy_map = zeros(n,obj.N_geometries,'int32');
                obj.N_energies = 0;
                for i=1:n
                    map = zeros(obj.N_geometries,'int32');
                    for geom = 1:obj.N_geometries
                        map(geom) = 0;
                        test = [Energy_grid{geom,1}];
                        geom_has_energy = false;
                        for j = 1:N_energies_geom(geom,1)
                            if energies(i,1) == test(j,1)
                                geom_has_energy = true;
                                map(geom) = j;
                                break;
                            end
                        end
                        if (geom_has_energy)
                            have_energy = true;
                        else
                            have_energy = false;
                            break;
                        end
                    end
                    if (have_energy == true)
                        obj.N_energies = obj.N_energies + 1;
                        energies(obj.N_energies,1) = energies(i);
                        %str = num2str(energies(i,1));
                        %disp(strcat('Accepted energy:',str))
                        for geom = 1:obj.N_geometries
                            energy_map(obj.N_energies,geom) = map(geom);
                        end
                    else
                        str = num2str(energies(i,1));
                        disp(strcat('Dropping energy:',str))
                    end
                end
                obj.Electron_energies = zeros(obj.N_energies,1);
                obj.Electron_energies(1:obj.N_energies,1) = energies(1:obj.N_energies,1);

                %Put the dipoles from the Dipoles_read structure to the obj.Dipoles structure.
                disp('Processing photodipoles arrays...')
                obj.Dipoles = zeros(obj.N_lm,3,obj.N_energies,obj.N_final_states,obj.N_initial_states,obj.N_geometries);
                for geom = 1:obj.N_geometries %parfor geom = 1:obj.N_geometries
                    dips = [Dipoles_read{geom,1}];
                    for in = 1:obj.N_initial_states
                        for f = 1:obj.N_final_states
                            for e = 1:obj.N_energies
                                i = energy_map(e,geom);
                                obj.Dipoles(1:obj.N_lm,1:3,e,f,in,geom) = dips(1:obj.N_lm,1:3,i,f,in);
                            end
                        end
                    end
                end
                disp('done')

                [obj.Internal_coordinates obj.Internal_coordinates_limits]= obj.generate_internal_coordinates;

                %Generate the griddedInterpolant for all dipole matrix elements, energies and Dyson orbital signs.
                [obj.D_lm_xyz obj.Dipoles_initial obj.Dipoles_final obj.Energies_initial obj.Energies_final obj.Dyson_interpolant obj.Have_interpolant] = obj.construct_dipole_interpolant;
            else
                geom = 1;
                obj.N_energies = n;
                obj.Electron_energies = energies(1:n,1);
                obj.Dipoles = [Dipoles_read{geom,1}];
                obj.D_lm_xyz = cell(obj.N_lm,3,obj.N_final_states,obj.N_initial_states);
                X = zeros(obj.N_energies,1);
                V = zeros(obj.N_energies,1);
                ref_grid = [Energy_grid{geom,1}]; %grid of energies as read-in from the file
                map = zeros(N_energies_geom(geom,1)); %mapping between the new energy grid and the full grid.
                for i = 1:obj.N_energies
                    X(i) = obj.Electron_energies(i,1);
                    for j = 1:N_energies_geom(geom,1)
                        if (ref_grid(j,1) == obj.Electron_energies(i,1))
                            map(i,1) = j;
                        end
                    end
                    if map(i,1) == 0
                        error('Error in energy grid compression.')
                    end
                end
                for in = 1:obj.N_initial_states
                    for f = 1:obj.N_final_states
                        for xyz = 1:3
                            for lm = 1:obj.N_lm
                                if en_step == 1
                                    V(1:obj.N_energies) = obj.Dipoles(lm,xyz,1:obj.N_energies,f,in,geom);
                                else
                                    for i = 1:obj.N_energies
                                        V(i) = obj.Dipoles(lm,xyz,map(i,1),f,in,geom);
                                    end
                                end
                                obj.D_lm_xyz{lm,xyz,f,in} = griddedInterpolant(X,V,'spline');
                            end
                        end
                    end
                end               
            end  

            obj.Max_energy = max(obj.Electron_energies(:));
            obj.Min_energy = min(obj.Electron_energies(:));
         end
         
         function show_states(obj)
             
             disp(strcat('Molecule:',obj.Molecule))
             
             disp('Initial states:')
             for i=1:obj.N_initial_states
                 disp(obj.Initial_states{1,i,1})
             end
             
             disp('Final states:')
             for i=1:obj.N_final_states
                 disp(obj.Final_states{1,i,1})
             end             
             
         end

         function s = Dyson_signs(obj,final_state,initial_state,grid_vectors)

            if nargin > 4 || nargin < 3
               error('Expecting 2 or 3 input parameters')
            end

            if nargin == 4
                if not(iscell(grid_vectors))
                    error('On input grid_vectors must a cell array.')
                end
                n_coords = size(grid_vectors,2);            
            else
                n_coords = 0;
            end

            if n_coords == 0
                geom = 1; %no interpolation; use the geometry with index 1
            elseif n_coords == 2
                geom = 0; %interpolate for the given internal_coordinates
                if not(obj.Have_interpolant)
                    error('Interpolation requested but not possible since the interpolant could not be constructed.')
                end
                n_gamma = size([grid_vectors{1}],2); %number of gamma values
                n_ra = size([grid_vectors{2}],2); %number of gamma values
            else
                error('On input the size of the cell array grid_vectors must be either 1 or 3.')
            end

            if final_state > obj.N_final_states || final_state <= 0
                error('On input final_state was out of range.')
            end            
            
            if initial_state > obj.N_initial_states || initial_state <= 0
                error('On input initial_state was out of range.')
            end            

            if geom == 1
               s = [obj.Dyson_orbitals{2,final_state,initial_state,geom}];
            else
               s = zeros(obj.N_angular_points,n_gamma,n_ra);
               for i=1:obj.N_angular_points
                   s(i,1:n_gamma,1:n_ra) = obj.Dyson_interpolant{i,final_state,initial_state}(grid_vectors);
               end
            end

         end
        
         function e = final_state_energy(obj,final_state,grid_vectors)
             
            if nargin == 3
                if not(iscell(grid_vectors))
                    error('On input grid_vectors must a cell array.')
                end
                n_coords = size(grid_vectors,2);            
            else
                n_coords = 0;
            end
            
            if nargin > 3 || nargin < 2
                error('Expecting either 1 or 2 parameters on input.')
            end
            
            if final_state > obj.N_final_states || final_state <= 0
                error('On input final_state was out of range.')
            end            

            if n_coords == 0
                geom = 1; %no interpolation; use the geometry with index 1
            elseif n_coords == 2
                geom = 0; %interpolate for the given internal_coordinates
                if not(obj.Have_interpolant)
                    error('Interpolation requested but not possible since the interpolant could not be constructed.')
                end
                n_gamma = size([grid_vectors{1}],2); %number of gamma values
                n_ra = size([grid_vectors{2}],2); %number of gamma values
            else
                error('On input the size of the cell array grid_vectors must be either 1 or 3.')
            end
            
            %Check the range of the internal_coordinates
            if geom == 0
                if not(n_coords == size(obj.Internal_coordinates_limits,2))
                    error('The number of supplied grid vectors for the nuclear coordinates does not equal the actual number of nuclear coordinates.')
                end
                for i = 1:n_coords
                    vectors = [grid_vectors{i}];
                    n = size([grid_vectors{i}],2);
                    for j = 1:n
                        if vectors(j) < obj.Internal_coordinates_limits(1,i)
                            disp(num2str(vectors(j)))
                            disp(num2str(obj.Internal_coordinates_limits(1,i)))
                            error('On input internal coordinate was too small.')
                        end
                        if vectors(j) > obj.Internal_coordinates_limits(2,i)
                            disp(num2str(vectors(j)))
                            disp(num2str(obj.Internal_coordinates_limits(1,i)))                            
                            error('On input internal coordinate was too large.')
                        end
                    end
                end
            end             
       
            if geom == 1
               e = obj.Final_states{2,final_state,geom};
            else
               e = zeros(n_gamma,n_ra);
               e(1:n_gamma,1:n_ra) = obj.Energies_final{final_state,1}(grid_vectors);                
            end
            
         end        
        
         function e = initial_state_energy(obj,initial_state,grid_vectors)
             
            if nargin == 3
                if not(iscell(grid_vectors))
                    error('On input grid_vectors must a cell array.')
                end
                n_coords = size(grid_vectors,2);            
            else
                n_coords = 0;
            end
            
            if nargin > 3 || nargin < 2
                error('Expecting either 1 or 2 parameters on input.')
            end
            
            if initial_state > obj.N_initial_states || initial_state <= 0
                error('On input initial_state was out of range.')
            end            

            if n_coords == 0
                geom = 1; %no interpolation; use the geometry with index 1
            elseif n_coords == 2
                geom = 0; %interpolate for the given internal_coordinates
                if not(obj.Have_interpolant)
                    error('Interpolation requested but not possible since the interpolant could not be constructed.')
                end
                n_gamma = size([grid_vectors{1}],2); %number of gamma values
                n_ra = size([grid_vectors{2}],2); %number of gamma values
            else
                error('On input the size of the cell array grid_vectors must be either 1 or 3.')
            end
            
            %Check the range of the internal_coordinates
            if geom == 0
                if not(n_coords == size(obj.Internal_coordinates_limits,2))
                    error('The number of supplied grid vectors for the nuclear coordinates does not equal the actual number of nuclear coordinates.')
                end
                for i = 1:n_coords
                    vectors = [grid_vectors{i}];
                    n = size([grid_vectors{i}],2);
                    for j = 1:n
                        if vectors(j) < obj.Internal_coordinates_limits(1,i)
                            disp(num2str(vectors(j)))
                            disp(num2str(obj.Internal_coordinates_limits(1,i)))                            
                            error('On input internal coordinate was too small.')
                        end
                        if vectors(j) > obj.Internal_coordinates_limits(2,i)
                            disp(num2str(vectors(j)))
                            disp(num2str(obj.Internal_coordinates_limits(1,i)))                            
                            error('On input internal coordinate was too large.')
                        end
                    end
                end
            end             
       
            if geom == 1
               e = obj.Initial_states{2,initial_state,geom};
            else
               e = zeros(n_gamma,n_ra);
               e(1:n_gamma,1:n_ra) = obj.Energies_initial{initial_state,1}(grid_vectors);                
            end
            
        end       

        function d = final_state_dipoles(obj,final_stateI,final_stateF,grid_vectors)
            
            if nargin == 4
                if not(iscell(grid_vectors))
                    error('On input grid_vectors must a cell array.')
                end
                n_coords = size(grid_vectors,2);            
            else
                n_coords = 0;
            end
            
            if nargin > 4 || nargin < 3
                error('Expecting either 2 or 3 parameters on input.')
            end

            if final_stateI > obj.N_final_states || final_stateI <= 0
                error('On input final_stateI was out of range.')
            end
            
            if final_stateF > obj.N_final_states || final_stateF <= 0
                error('On input final_stateF was out of range.')
            end            

            if n_coords == 0
                geom = 1; %no interpolation; use the geometry with index 1
            elseif n_coords == 2
                geom = 0; %interpolate for the given internal_coordinates
                if not(obj.Have_interpolant)
                    error('Interpolation requested but not possible since the interpolant could not be constructed.')
                end
                n_gamma = size([grid_vectors{1}],2); %number of gamma values
                n_ra = size([grid_vectors{2}],2); %number of gamma values
            else
                error('On input the size of the cell array grid_vectors must be either 1 or 3.')
            end
            
            %Check the range of the internal_coordinates
            if geom == 0
                if not(n_coords == size(obj.Internal_coordinates_limits,2))
                    error('The number of supplied grid vectors for the nuclear coordinates does not equal the actual number of nuclear coordinates.')
                end
                for i = 1:n_coords
                    vectors = [grid_vectors{i}];
                    n = size([grid_vectors{i}],2);
                    for j = 1:n
                        if vectors(j) < obj.Internal_coordinates_limits(1,i)
                            error('On input internal coordinate was too small.')
                        end
                        if vectors(j) > obj.Internal_coordinates_limits(2,i)
                            error('On input internal coordinate was too large.')
                        end
                    end
                end
            end
            
            if geom == 1
                d = zeros(3,1);
            else
                d = zeros(3,n_gamma,n_ra);
            end            
            
            if geom == 1
               FD = [obj.Final_states{3,final_stateI,geom}];
               for xyz = 1:3
                   d(xyz,1) = FD(xyz,final_stateF);
               end
            else
                for xyz = 1:3
                    d(xyz,1:n_gamma,1:n_ra) = obj.Dipoles_final{xyz,final_stateI,final_stateF}(grid_vectors);
                end
            end
            
        end
        
        function d = initial_state_dipoles(obj,initial_stateI,initial_stateF,grid_vectors)
            
            if nargin == 4
                if not(iscell(grid_vectors))
                    error('On input grid_vectors must a cell array.')
                end
                n_coords = size(grid_vectors,2);            
            else
                n_coords = 0;
            end
            
            if nargin > 4 || nargin < 3
                error('Expecting either 2 or 3 parameters on input.')
            end

            if initial_stateI > obj.N_initial_states || initial_stateI <= 0
                error('On input initial_stateI was out of range.')
            end
            
            if initial_stateF > obj.N_initial_states || initial_stateF <= 0
                error('On input initial_stateF was out of range.')
            end            

            if n_coords == 0
                geom = 1; %no interpolation; use the geometry with index 1
            elseif n_coords == 2
                geom = 0; %interpolate for the given internal_coordinates
                if not(obj.Have_interpolant)
                    error('Interpolation requested but not possible since the interpolant could not be constructed.')
                end
                n_gamma = size([grid_vectors{1}],2); %number of gamma values
                n_ra = size([grid_vectors{2}],2); %number of gamma values
            else
                error('On input the size of the cell array grid_vectors must be either 1 or 3.')
            end
            
            %Check the range of the internal_coordinates
            if geom == 0
                if not(n_coords == size(obj.Internal_coordinates_limits,2))
                    error('The number of supplied grid vectors for the nuclear coordinates does not equal the actual number of nuclear coordinates.')
                end
                for i = 1:n_coords
                    vectors = [grid_vectors{i}];
                    n = size([grid_vectors{i}],2);
                    for j = 1:n
                        if vectors(j) < obj.Internal_coordinates_limits(1,i)
                            error('On input internal coordinate was too small.')
                        end
                        if vectors(j) > obj.Internal_coordinates_limits(2,i)
                            error('On input internal coordinate was too large.')
                        end
                    end
                end
            end
            
            if geom == 1
                d = zeros(3,1);
            else
                d = zeros(3,n_gamma,n_ra);
            end            
            
            if geom == 1
               FD = [obj.Initial_states{3,initial_stateI,geom}];
               for xyz = 1:3
                   d(xyz,1) = FD(xyz,initial_stateF);
               end
            else
                for xyz = 1:3
                    d(xyz,1:n_gamma,1:n_ra) = obj.Dipoles_initial{xyz,initial_stateI,initial_stateF}(grid_vectors);
                end
            end
            
        end        
            
        function d = momentum_space_photodipoles(obj,theta,phi,initial_state,final_state,grid_vectors)
            
            if not(iscell(grid_vectors))
                error('On input grid_vectors must a cell array.')
            end
                       
            n_coords = size(grid_vectors,2);
            
            if nargin < 6 || nargin > 7
                disp(nargin)
                error('The number of input parameters (excluding the object itself) must be either 5 or 6.')
            end
            
            if n_coords == 1
                geom = 1; %no interpolation; use the geometry with index 1
            elseif n_coords == 3
                geom = 0; %interpolate for the given internal_coordinates
                if not(obj.Have_interpolant)
                    error('Interpolation requested but not possible since the interpolant could not be constructed.')
                end
                n_gamma = size([grid_vectors{2}],2); %number of gamma values
                n_ra = size([grid_vectors{3}],2); %number of gamma values
            else
                error('On input the size of the cell array grid_vectors must be either 1 or 3.')
            end
            
            energies = [grid_vectors{1}]; %list of energies
            n_en = size(energies,2);
            
            for i=1:n_en
                energy = energies(i);
                if energy > obj.Max_energy || energy < obj.Min_energy
                    disp(num2str(energy))
                    error('On input an energy was outside of the energy grid limits.')
                end
            end
            
            if final_state > obj.N_final_states || final_state <= 0
                error('On input final_state was out of range.')
            end
            
            if initial_state > obj.N_initial_states || initial_state <= 0
                error('On input initial_state was out of range.')
            end
            
            %Check the range of the internal_coordinates
            if geom == 0
                if not(n_coords-1 == size(obj.Internal_coordinates_limits,2))
                    error('The number of supplied grid vectors for the nuclear coordinates does not equal the actual number of nuclear coordinates.')
                end
                for i = 1:n_coords-1
                    vectors = [grid_vectors{i+1}];
                    n = size([grid_vectors{i+1}],2);
                    for j = 1:n
                        if vectors(j) < obj.Internal_coordinates_limits(1,i)
                            error('On input internal coordinate was too small.')
                        end
                        if vectors(j) > obj.Internal_coordinates_limits(2,i)
                            error('On input internal coordinate was too large.')
                        end
                    end
                end
            end
            
            if geom == 1
                dinterp = zeros(obj.N_lm,3,n_en);
                d = zeros(3,n_en);
            else
                dinterp = zeros(obj.N_lm,3,n_en,n_en,n_en);
                d = zeros(3,n_en,n_gamma,n_ra);
            end
            
            found = 0;
            for i=1:obj.N_angular_points
                if obj.Angular_grid_spherical(1,i) == theta && obj.Angular_grid_spherical(2,i) == phi
                    found = i;
                end
            end
            
            Xlm = zeros(1,obj.N_lm);
            lm = 0;
            for l=0:obj.Max_L
              for m=-l:l
                 lm = lm + 1;
                 %Take the real spherical harmonic from the precalculated
                 %buffer where possible otherwise calculate it for the
                 %point given.
                 if found > 1
                     Xlm(1,lm) = obj.Xlm_precalculated(lm,found);
                 else
                     Xlm(1,lm) = obj.re_sp_harm(l,m,theta,phi);
                 end
                 
                 if geom == 1 %Single geometry calculation
                    %1D interpolation in energy for each of the x,y,z components of the partial wave dipole.
                    for i=1:3
                         dinterp(lm,i,1:n_en) = obj.D_lm_xyz{lm,i,final_state,initial_state}(grid_vectors);
                    end
                 else %Include interpolation accross geometries
                    %interpolation in energy (1D) and geometry (2D) for each of the x,y,z components of the partial wave dipole.
                    for i=1:3
                         dinterp(lm,i,1:n_en,1:n_gamma,1:n_ra) = obj.D_lm_xyz{lm,i,final_state,initial_state}(grid_vectors);
                    end                     
                 end
              end
            end
            
            if geom == 1
                %Output is a matrix(3,n_en) corresponding to the x,y,z dipole components evaluated for the n_en energy points
                for i=1:n_en
                    d(1:3,i) = Xlm(1,:)*dinterp(:,1:3,i);
                end
            else
                %Output is a matrix(3,n_en,n_en,n_en) corresponding to the x,y,z dipole components evaluated for the n_en * n_en * n_en points in the energy,gamma,ra space.
                for i=1:n_ra
                    for j=1:n_gamma
                        for k=1:n_en
                            d(1:3,k,j,i) = Xlm(1,:)*dinterp(:,1:3,k,j,i);
                        end
                    end
                end               
            end
            
        end
        
    end
   
    %Auxiliary functions needed to evaluate the dipoles in the momentum basis.
    methods(Static, Hidden)
        
        function r = red_rot_mat( beta,J,M,N )
        %r Reimplementation of the angmom_proc/red_rot_mat routine.
        %   Reduced rotational matrix elements in the basis of real spherical
        %   harmonics (Brink & Satchler).

            tmp = 0;

            t1 = J-N;
            t2 = J+M;
            t3 = M-N;

            if t3 > 0
                tmin = t3;
            else
                tmin = 0;
            end

            if t1 < t2
                tmax = t1;
            else
                tmax = t2;
            end

            if abs(M) > J || abs(N) > J
                r = 0;
                return
            end

            S=sin(beta/2);
            C=cos(beta/2);
            tmp1=sqrt(factorial(J+M)*factorial(J-M)*factorial(J+N)*factorial(J-N));

            for t=tmin:tmax
                tmp = tmp + ((-1)^t)*(tmp1/(factorial(J+M-t)*factorial(J-N-t)*factorial(t)*factorial(t+N-M)))*C^(2*J+M-N-2*t)*S^(2*t+N-M);
            end

            r = tmp;
    
        end      
       
        %note that the meaning of theta and phi is swapped since xyz_to_tp uses the Mathematical convention for spherical coordinates (theta is azimuthal angle, phi is the polar angle)!
        function [t,p] = xyz_to_tp( x,y,z )
            p = acos(z);
            fact = sqrt(x*x + y*y);
            
            if 0 < fact
                t = acos(x/fact);
            else
                t = acos(x);
            end
            
            if y < 0
                t = -t;
            end
            
        end
        
    end
    
    methods(Hidden)
        
        %Generates molecule-specific internal coordinates
        %   This routine should contain branches into the molecule-specific
        %   way of generating the internal coordinates.
        function [internal_coordinates internal_coordinates_limits]= generate_internal_coordinates(obj)
            
            disp(strcat('Determining internal coordinates for molecule: ',obj.Molecule))
            
            if strcmp(strtrim(obj.Molecule),'NO2')
                
                internal_coordinates = zeros(2,obj.N_geometries);                
                for geom=1:obj.N_geometries
                    geometry = [obj.Atoms{1,:,geom}];
                    N = geometry(:,1); %N
                    O1 = geometry(:,2); %O1
                    O2 = geometry(:,3); %O2
            
                    no1 = sqrt(dot(N-O1,N-O1));
                    no2 = sqrt(dot(N-O2,N-O2));
                    ra = abs(no1-no2);
            
                    cgamma = dot(O1-N,O2-N)/(no1*no2);
                    gamma = acos(cgamma);
                    gamma = gamma/pi*180;                    

                    internal_coordinates(1,geom) = round(gamma*100000)/100000;
                    internal_coordinates(2,geom) = round(ra*100000)/100000;
                end
                
                %figure out the limits on the values of the internal coordinates:
                internal_coordinates_limits = zeros(2,2);
                
                internal_coordinates_limits(1,1) = min(internal_coordinates(1,:));
                internal_coordinates_limits(2,1) = max(internal_coordinates(1,:));
                
                internal_coordinates_limits(1,2) = min(internal_coordinates(2,:));
                internal_coordinates_limits(2,2) = max(internal_coordinates(2,:));
            else
                error('Determination of the internal coordinates for this molecule has not been implemented.')
            end
            
        end 
        
        %Generates the griddedInterpolant for the partial wave dipoles in the space of electron energies and nuclear internal coordinates.
        %The result is stored in the cell array obj.D_lm_xyz of dimensions (obj.N_lm,3).
        %   The interpolant is constructed for each L,M partial wave and each x,y,z dipole component.
        function [D_lm_xyz Dipoles_initial Dipoles_final Energies_initial Energies_final Dyson_interpolant Have_interpolant] = construct_dipole_interpolant(obj)
            
            D_lm_xyz = cell(obj.N_lm,3,obj.N_final_states,obj.N_initial_states);
            Dipoles_initial = cell(3,obj.N_initial_states,obj.N_initial_states);
            Dipoles_final = cell(3,obj.N_final_states,obj.N_final_states);
            Energies_final = cell(obj.N_final_states,1);
            Energies_initial = cell(obj.N_initial_states,1);
            Dyson_interpolant = cell(obj.N_angular_points,obj.N_final_states,obj.N_initial_states);
            
            n_internal = size(obj.Internal_coordinates,1);
            %disp(obj.Internal_coordinates)
            if not(n_internal == 2)
                disp('Dipole interpolant will not be constructed: this has been implemented only for the number of internal coordinates equal to 2.')
                Have_interpolant = false;
                return
            else
                Have_interpolant = true;
            end
            
            %Determine all different values of the internal coordinates
            unique_internal(1:n_internal,1:obj.N_geometries) = -1000;
            n_unique = zeros(n_internal,1);
            for geom=1:obj.N_geometries
                for i=1:n_internal
                    unique = true;
                    for j=1:n_unique
                        if unique_internal(i,j) == obj.Internal_coordinates(i,geom)
                            unique = false;
                            break;
                        end
                    end
                    if unique
                        n_unique(i) = n_unique(i)+1;
                        unique_internal(i,n_unique(i)) = obj.Internal_coordinates(i,geom);
                    end
                end
            end
            
            %The mesh points constructed below must be in ascending order
            method = 'spline'; %set the interpolation method: prefer splines where possible.
            disp('Grid of internal coordinates')
            for i=1:n_internal
                if n_unique(i) == 1
                    error('The interpolation requires at least two values of the coordinate in each dimension. ')
                end   
                unique_internal(i,1:n_unique(i)) = sort(unique_internal(i,1:n_unique(i)));
                if n_unique(i) < 4 || obj.N_energies < 4 %spline interpolation requires at least 4 points in each dimension
                    method = 'linear';
                end
                msg = strcat('Internal coordinate:',num2str(i));
                disp(msg)
                disp(unique_internal(i,1:n_unique(i)))
            end
            
            msg = strcat('Interpolation method:',method);
            disp(msg)

            %Mapping from the set of unique internal coordinates to the geom index
            unique_to_geom = zeros(n_unique(1),n_unique(2));
            for geom=1:obj.N_geometries
                for j=1:n_unique(2)
                    for i=1:n_unique(1)
                        if obj.Internal_coordinates(1,geom) == unique_internal(1,i) && obj.Internal_coordinates(2,geom) == unique_internal(2,j)
                           unique_to_geom(i,j) = geom;
                        end
                    end
                end
            end
            %disp(unique_to_geom)
            
            %Check that the data set is complete, i.e. that we have data for every combination of the nuclear coordinates.
            for j=1:n_unique(2)
                for i=1:n_unique(1)
                    if unique_to_geom(i,j) == 0
                        disp(num2str([i j]))
                        disp(num2str([unique_internal(1,i) unique_internal(1,j)]))
                        error('Missing data for a combination of unique nuclear coordinates.')
                    end
                end
            end
            
            if n_internal == 2

                %Construct the mesh grid: values of the x,y,z coordinates at each mesh point.
                X = zeros(obj.N_energies,n_unique(1),n_unique(2));
                Y = zeros(obj.N_energies,n_unique(1),n_unique(2));
                Z = zeros(obj.N_energies,n_unique(1),n_unique(2));     
                for j=1:n_unique(2)
                    Z(:,:,j) = unique_internal(2,j);
                    for i=1:n_unique(1)
                        Y(:,i,j) = unique_internal(1,i);                
                        for energy=1:obj.N_energies
                            X(energy,i,j) = obj.Electron_energies(energy);
                        end
                    end
                end
                
                for in = 1:obj.N_initial_states
                    for f = 1:obj.N_final_states
                        for xyz = 1:3
                            
                            %parpool(24);
                            for lm = 1:obj.N_lm %parfor lm = 1:obj.N_lm
                                
                                str = int2str([lm xyz f in]);
                                disp(strcat('Generating interpolant for lm,xyz,final,initial: ',str));
                                V = zeros(obj.N_energies,n_unique(1),n_unique(2));
                                
                                for j=1:n_unique(2)
                                    for i=1:n_unique(1)
                                        for energy=1:obj.N_energies
                                            % If I don't have the dipoles for the
                                            % desired X(i,j) then I can perform
                                            % a 1D interpolation here and put the
                                            % result into V(energy,i,j).
                                            geom = unique_to_geom(i,j);
                                            V(energy,i,j) = obj.Dipoles(lm,xyz,energy,f,in,geom);
                                        end
                                    end
                                end
                                
                                %Construct 3D interpolant for the photodipoles
                                D_lm_xyz{lm,xyz,f,in} = griddedInterpolant(X,Y,Z,V,method);
                            end
                        end
                    end
                end
                
                X = zeros(n_unique(1),n_unique(2));
                Y = zeros(n_unique(1),n_unique(2));     
                VI = zeros(n_unique(1),n_unique(2),3,obj.N_initial_states,obj.N_initial_states);
                VF = zeros(n_unique(1),n_unique(2),3,obj.N_final_states,obj.N_final_states);
                EI = zeros(n_unique(1),n_unique(2),obj.N_initial_states);
                EF = zeros(n_unique(1),n_unique(2),obj.N_final_states);
                DS = zeros(n_unique(1),n_unique(2),obj.N_angular_points,obj.N_final_states,obj.N_initial_states)
                
                for j=1:n_unique(2)
                    Y(:,j) = unique_internal(2,j);
                    for i=1:n_unique(1)
                        X(i,j) = unique_internal(1,i);
                    end
                end
                
                for j=1:n_unique(2)
                    for i=1:n_unique(1)
                        geom = unique_to_geom(i,j);
                        
                        for f=1:obj.N_final_states
                            EF(i,j,f) = [obj.Final_states{2,f,geom}];
                            dipsF = [obj.Final_states{3,f,geom}];
                            for fp = 1:obj.N_final_states
                                for xyz = 1:3
                                    VF(i,j,xyz,fp,f) = dipsF(xyz,fp,1);
                                end
                            end
                        end
                        for in=1:obj.N_initial_states
                            EI(i,j,in) = [obj.Initial_states{2,in,geom}];
                            dipsI = [obj.Initial_states{3,in,geom}];
                            for ip = 1:obj.N_initial_states
                                for xyz = 1:3
                                    VI(i,j,xyz,ip,in) = dipsI(xyz,ip,1);
                                end
                            end
                        end

                        for in=1:obj.N_initial_states
                            for f=1:obj.N_final_states
                                s = [obj.Dyson_orbitals{2,f,in,geom}];
                                for p=1:obj.N_angular_points
                                   DS(i,j,p,f,in) = s(p,1);
                                end
                            end
                        end
                       
                   end
                end

                %Construct 'interpolant' for the Dyson orbital signs.
                %This is just a fake interpolant since the method is nearest-neighbor so no actual interpolation accross geometries is being done.
                for in=1:obj.N_initial_states
                    for f=1:obj.N_final_states
                        for p=1:obj.N_angular_points
                            Dyson_interpolant{p,f,in} = griddedInterpolant(X,Y,DS(:,:,p,f,in),'nearest');
                        end
                    end
                end
                        
                %Construct 2D interpolant for the final and initial state dipoles.
                for in=1:obj.N_initial_states
                    Energies_initial{in,1} = griddedInterpolant(X,Y,EI(:,:,in),method); %interpolant for initial state energies
                    for ip = 1:obj.N_initial_states                       
                        for xyz=1:3
                            str = int2str([xyz in ip]);
                            disp(strcat('Generating interpolant for xyz,initial,initial: ',str));
                            Dipoles_initial{xyz,ip,in} = griddedInterpolant(X,Y,VI(:,:,xyz,ip,in),method);
                        end
                    end
                end
                for f=1:obj.N_final_states
                    Energies_final{f,1} = griddedInterpolant(X,Y,EF(:,:,f),method); %interpolant for final state energies
                    for fp = 1:obj.N_final_states                       
                        for xyz=1:3
                            str = int2str([xyz f fp]);
                            disp(strcat('Generating interpolant for xyz,final,final: ',str));                            
                            Dipoles_final{xyz,fp,f} = griddedInterpolant(X,Y,VF(:,:,xyz,fp,f),method);
                        end
                    end
                end                        
            else
                error('Not implemented for n_internal != 2')
            end
            
        end        
        
        function r = re_sp_harm( obj,j,m,theta,phi )
            %r Reimplementation of the angmom_proc/re_sp_harm routine.
            %   Real spherical harmonic
        
            norm=((-1)^m)/sqrt(2);

            if m >= 1
                temp=norm*2*sqrt((2*j+1)/(4*pi))*cos(m*phi)*obj.red_rot_mat(theta,j,m,0);
            elseif m == 0
                temp=sqrt((2*j+1)/(4*pi))*obj.red_rot_mat(theta,j,0,0);
            elseif m <= -1
                temp=norm*2*sqrt((2*j+1)/(4*pi))*sin(abs(m)*phi)*obj.red_rot_mat(theta,j,abs(m),0);
            end
           
            r=temp;
     
        end
        
    end
    
end
