! Copyright 2019
!
! Zdenek Masin
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
module properties_file_mod
   use precisn_gbl, only: wp, cfp
   use utils_gbl, only: xermsg
   implicit none

   !> Parameters used by properties_file_obj.
   integer, parameter :: n_keys = 8, maxl = 8

   !> Maximum number of N+1 symmetries.
   integer, parameter :: max_symmetries = 8, max_spin = 4

   type :: properties_file_obj
      integer, private :: n_lines = 0, nstat = 0, lutarg = 0, n_initial_states = 0
      character(len=132), allocatable, private :: file_lines(:)
      character(len=1), private :: str_key(n_keys)
      integer, allocatable, private :: key_start_end(:,:), absolute_to_relative(:,:)
      real(kind=wp), allocatable, private :: state_energies(:)
      integer :: sym_index(max_symmetries,(2*max_spin+1))
      logical, private :: prop_avail((maxl+1)**2)
      logical, private :: initialized = .false.
   contains
      procedure :: load_file
      procedure :: get_properties
      procedure :: get_state_dipoles
      procedure :: get_single_property
      procedure :: final
   end type properties_file_obj

contains

   subroutine load_file(this,path)
      implicit none
      class(properties_file_obj) :: this
      character(len=*), intent(in) :: path

      integer, parameter :: lutarg = 100
      integer :: err, i, j, irr, state_no, inx(8), rel_state_no, bra_or_ket
      real(kind=wp) :: dnx
      character(len=26) :: head
      character(len=132) :: one_line
      character(len=*), parameter :: str_state_no = 'State No.'

        write(*,'(/,10X,"load_file: ",a)') trim(path)
        write(*,'(  10X,"==========")')

        this%lutarg = lutarg

        open(unit=this%lutarg,file=path,form='FORMATTED',status='old',iostat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','load_file','Error opening the properties file.',2,1)

        !Count the number of lines of the properties file:
        rewind this%lutarg
        this%n_lines = 0
        do
           read(this%lutarg,'(a)',end=100) one_line
           this%n_lines = this%n_lines + 1
        enddo
    100 rewind this%lutarg

        write(*,'(/,5X,"Number of lines on the input file: ",i15)') this%n_lines

        if (allocated(this%file_lines)) deallocate(this%file_lines)
        if (allocated(this%key_start_end)) deallocate(this%key_start_end)
        allocate(this%file_lines(this%n_lines),this%key_start_end(2,n_keys),stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','load_file','Memory allocation 1 failed.',err,1)

        do j=1,n_keys
           write(this%str_key(j),'(i1)') j
        enddo

        !Read the whole file into the array setting the ranges of the key positions
        this%key_start_end = 0
        do i=1,this%n_lines
           read(this%lutarg,'(a)') one_line
           this%file_lines(i) = one_line

           do j=1,n_keys
              if (one_line(1:1) .eq. this%str_key(j)) then
                 if (this%key_start_end(1,j) .eq. 0) this%key_start_end(1,j) = i
                 this%key_start_end(2,j) = i
              endif
           enddo
        enddo !i

        close(this%lutarg)

        write(*,'(/,5X,"Keys found on the input unit and their line number ranges:")')
        do j=1,n_keys
           if (this%key_start_end(1,j) > 0) write(*,'(5X,"Key ",i2," start and end: ",2i15)') j, this%key_start_end(1:2,j)
        enddo

        if (this%key_start_end(1,5) .eq. 0) call xermsg('properties_file_mod','load_file',&
                                                        'Energy data not present on the properties file.',1,1)
        if (this%key_start_end(1,1) .eq. 0) call xermsg('properties_file_mod','load_file',&
                                                        'Property data not present on the properties file.',2,1)

        this%nstat = this%key_start_end(2,5)-this%key_start_end(1,5)+1
        write(*,'(/,5X,"Total number of states: ",i15)') this%nstat

        !Read the energy data (key = 5) for the states and map them to their order within each IRR
        if (allocated(this%absolute_to_relative)) deallocate(this%absolute_to_relative)
        if (allocated(this%state_energies)) deallocate(this%state_energies)
        allocate(this%absolute_to_relative(5,this%nstat),this%state_energies(this%nstat),stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','load_file','Memory allocation 2 failed.',err,1)

        this%absolute_to_relative = 0
        this%sym_index = 0
        this%n_initial_states = 0 !this is effective in case of CDENPROP dipoles which contain dipoles for a small number of bra states (n_initial_states) and a large number of ket states.
        bra_or_ket = 1
        do i=this%key_start_end(1,5),this%key_start_end(2,5)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx,head
           if (inx(1) .ne. 5) call xermsg('properties_file_mod','load_file',&
                                          'Unexcpected key number: error parsing the properties file.',3,1)
           irr = inx(5)+1
           state_no = inx(2)
           this%sym_index(irr,inx(6)) = this%sym_index(irr,inx(6)) + 1
           this%absolute_to_relative(1,state_no) = this%sym_index(irr,inx(6))
           this%absolute_to_relative(2,state_no) = irr
           this%absolute_to_relative(3,state_no) = inx(6)
           this%state_energies(state_no) = dnx
           one_line = this%file_lines(i)
           j = index(one_line,str_state_no)
           read(one_line(j+len(str_state_no):),*) rel_state_no
           this%absolute_to_relative(4,state_no) = rel_state_no
           this%absolute_to_relative(5,state_no) = bra_or_ket !1 = bra or ket state, 2 = ket state
           if (state_no > 1) then
              if (this%absolute_to_relative(4,state_no) .le. this%absolute_to_relative(4,state_no-1)) then
                 this%n_initial_states = this%absolute_to_relative(4,state_no-1)
                 bra_or_ket = 2
                 this%absolute_to_relative(5,state_no) = bra_or_ket
              endif
           endif
        enddo !i

        if (this%n_initial_states > 0) then
           write(*,'(/,5X,"This is a CDENPROP file for ",i10," initial states.")') this%n_initial_states
        endif

        write(*,'(/,5X,"Number of states per spin/space symmetry")')
        do i=1,2*max_spin+1
           do irr=1,max_symmetries
              if (this%sym_index(irr,i) .ne. 0) then
                 write(*,'(/,5X,"Spin: ",i2,", IRR: ",i1)') i,irr
                 write(*,'(  5X,"Number of states: ",i10)') this%sym_index(irr,i)
              endif
           enddo
        enddo

        this%initialized = .true.

        write(*,'(/,10X,"load_file: finished")')

   end subroutine load_file

   subroutine final(this)
      implicit none
      class(properties_file_obj) :: this

        write(*,'(/,10X,"final")')
        write(*,'(  10X,"====================")')

        if (allocated(this%file_lines)) deallocate(this%file_lines)
        if (allocated(this%key_start_end)) deallocate(this%key_start_end)
        this%initialized = .false. 

        write(*,'(/,10X,"final: finished")')

   end subroutine final

   !> All properties with L <= lmaxprop will be read-in.
   subroutine get_properties(this,lmaxprop,prop,state_energies,absolute_to_relative)
      implicit none
      class(properties_file_obj) :: this
      integer, intent(in) :: lmaxprop
      real(kind=wp), allocatable :: prop(:,:,:), state_energies(:)
      integer, allocatable :: absolute_to_relative(:,:)

      character(len=132) :: one_line
      character(len=1) :: str_X
      integer :: err, i, j, inx(8), it1, it2, lop, mop, iq, block, state_no, irr, n_prop, m, l, bra_or_ket, rel_state_no
      real(kind=wp) :: dnx
      logical :: prop_avail((maxl+1)**2)

        write(*,'(/,10X,"get_properties")')
        write(*,'(  10X,"====================")')

        write(*,'(  5X,"Properties to read: all properties with L <= ",i5)') lmaxprop

        if (lmaxprop < 0) call xermsg('properties_file_mod','get_properties','On input lmaxprop < 0.',1,1)

        if (.not.(this%initialized)) call xermsg('properties_file_mod','get_properties',&
                                                 'The properties file has not been loaded into memory: call this%load_file first.',&
                                                 2,1)

        if (allocated(absolute_to_relative)) deallocate(absolute_to_relative)
        if (allocated(state_energies)) deallocate(state_energies)
        if (allocated(prop)) deallocate(prop)

        allocate(absolute_to_relative,source=this%absolute_to_relative,stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','get_properties','Memory allocation 2 failed.',err,1)
        allocate(state_energies(this%nstat),source=this%state_energies,stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','get_properties','Memory allocation 3a failed.',err,1)

        !Read the property data (key = 1)
        iq = (lmaxprop+1)**2
        if (this%n_initial_states > 0) then
           i = this%n_initial_states
           j = this%nstat-i
        else
           i = this%nstat
           j = this%nstat
        endif
        allocate(prop(i,j,iq),stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','get_properties','Memory allocation 4 failed.',err,1)

        n_prop = 0
        prop = 0.0_wp
        prop_avail = .false.
        do i=this%key_start_end(1,1),this%key_start_end(2,1)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx
           it1=inx(2)
           it2=inx(4)
           lop=inx(7)
           mop=inx(8)
           if (lop .le. maxl) then
              iq = lop*lop+lop+mop+1
              prop_avail(iq) = .true.
           else
              write(*,'("Maximum L of available property exceeds the size of the prop_avail array.")')
           endif
           if (lop < 1 .or. lop > lmaxprop) cycle
           iq = lop*lop+lop+mop+1
           !Save the dipoles to the prop array using their relative index within the initial and final symmetries.
           if (this%n_initial_states > 0) then
              rel_state_no = this%absolute_to_relative(4,it1)
              bra_or_ket = this%absolute_to_relative(5,it1)
              if (bra_or_ket .eq. 1) then !it1 is the initial state
                 it1 = rel_state_no
                 it2 = this%absolute_to_relative(4,it2)
              elseif (bra_or_ket .eq. 2) then !it2 is the initial state and it1 is the final state
                 it1 = this%absolute_to_relative(4,it2)
                 it2 = rel_state_no
              else
                 stop "error in bra_or_ket value"
              endif
              prop(it1,it2,iq) = dnx
           else
              prop(it1,it2,iq) = dnx
              prop(it2,it1,iq) = prop(it1,it2,iq)
           endif
           n_prop = n_prop+1
        enddo !i

        write(*,'(/,5X,"Properties (m,l) available on the file: R marks those that have been read-in")')
        i = 0
        do l=0,maxl
           str_X = ''
           if (l .le. lmaxprop) str_X = 'R'
           do m=-l,l
              i = i + 1
              if (prop_avail(i)) write(*,'(5X,i2,1X,i2,1X,a2)') m,l,str_X
           enddo
        enddo 

        write(*,'(/,5X,"Total number of properties that have been read-in: ",i15)') n_prop
        write(*,'(/,10X,"get_properties: finished")')

   end subroutine get_properties

   subroutine get_state_dipoles(this,state_num,state_sym,state_spin,n_states,dipole_table,state_energy,state_index_map)
     implicit none
     class(properties_file_obj) :: this
     integer, intent(in) :: state_num(:),state_sym(:),state_spin(:),n_states
     integer, intent(out) :: state_index_map(:)
     real(kind=wp), allocatable :: dipole_table(:,:,:), state_energy(:)

     integer, parameter :: lmaxprop = 1
     real(kind=wp), allocatable :: prop(:,:,:), state_energies(:)
     integer, allocatable :: absolute_to_relative(:,:)
     integer :: i, j, m, iq, xyz, ii, jj
     logical :: found_state(n_states)
     character(len=1) :: str
     
        if (allocated(dipole_table)) deallocate(dipole_table)
        if (allocated(state_energy)) deallocate(state_energy)
        allocate(dipole_table(3,n_states,n_states),state_energy(n_states))
        dipole_table = 0.0_wp; state_energy = 0.0_wp
 
        call this%get_properties(lmaxprop,prop,state_energies,absolute_to_relative)

        found_state = .false.
        do j=1,n_states
           do i=1,this%nstat
              if (this%absolute_to_relative(1,i) .eq. state_num(j) .and. this%absolute_to_relative(2,i) .eq. state_sym(j)&
                  .and. this%absolute_to_relative(3,i) .eq. state_spin(j)) then
                 state_index_map(j) = i
                 found_state(j) = .true.
                 state_energy(j) = this%state_energies(i)
                 write(*,'(5X,"Requested state number ",i4," maps to state number ",i4," from the properties file.")') j, i
                 exit
              endif
           enddo
           if (count(found_state) .eq. n_states) exit
        enddo

        if (count(found_state) .ne. n_states) then
           write(*,'("Some requested states are missing from the properties file.")')
           print *,found_state
           stop "error"
        endif

        write(*,'(5X,"Dipole table")')
        do j=1,n_states
           !check that each state and the one differing by offset are the same:
           !this must be true for the CDENPROP file containing dipoles between
           !the initial and final state symmetry states.
           ii = state_index_map(j)
           jj = state_index_map(j)+this%n_initial_states
           if (this%n_initial_states > 0) then
              if (this%absolute_to_relative(1,ii)+this%n_initial_states .ne. this%absolute_to_relative(1,jj) .or. &
                  this%absolute_to_relative(2,ii)        .ne. this%absolute_to_relative(2,jj) .or. &
                  this%absolute_to_relative(3,ii)        .ne. this%absolute_to_relative(3,jj)) then
                 print *,ii,jj
                 print *,this%absolute_to_relative(1:3,ii)
                 print *,this%absolute_to_relative(1:3,jj)
                 stop "error: wrong cdenprop file on input?"
              endif
           endif
           ii = this%absolute_to_relative(4,state_index_map(j)) !relative index of the j-state
           do i=1,n_states
              jj = this%absolute_to_relative(4,state_index_map(i)) !relative index of the i-state
              do m=-1,1
                 iq = lmaxprop*lmaxprop+lmaxprop+m+1
                 !Map spherical index m to cartesians
                 select case(m)
                 case (-1)
                    xyz = 2 !y
                    str = 'y'
                 case (0)
                    xyz = 3 !z
                    str = 'z'
                 case ( 1)
                    xyz = 1 !x
                    str = 'x'
                 end select
                 dipole_table(xyz,i,j) = prop(ii,jj,iq)
                 write(*,'(5X,a1,1X,2i10,e25.15)') str,i,j,dipole_table(xyz,i,j)
              enddo
           enddo
        enddo

   end subroutine get_state_dipoles

   !> A single property with the given L,M will be read-in. The output data have the structure required by RMT.
   subroutine get_single_property(this,lp,mp,dipsto,iidip,ifdip,intr,state_energies,absolute_to_relative)
      implicit none
      class(properties_file_obj) :: this
      integer, intent(in) :: lp, mp
      integer, intent(out) :: intr
      real(kind=wp), allocatable :: dipsto(:,:,:), state_energies(:)
      integer, allocatable :: absolute_to_relative(:,:)
      integer, allocatable :: iidip(:), ifdip(:)

      character(len=132) :: one_line
      character(len=1) :: str_X
      integer :: err, i, j, inx(8), it1, it2, lop, mop, irr1, irr2, block, state_no, irr, n_prop, m, l, mxstat
      real(kind=wp) :: dnx
      integer :: irr_map(max_symmetries,max_symmetries)

        write(*,'(/,10X,"get_single_property")')
        write(*,'(  10X,"====================")')

        write(*,'(  5X,"Properties to read: a single property with L,M: ",2i5)') lp,mp

        if (lp < 0 .or. abs(mp) > lp) call xermsg('properties_file_mod','get_single_property','On input lp,mp were out of range.',&
                                                  1,1)

        if (.not.(this%initialized)) call xermsg('properties_file_mod','get_single_property',&
                                                 'The properties file has not been loaded into memory: call this%load_file first.',&
                                                 2,1)

        if (allocated(absolute_to_relative)) deallocate(absolute_to_relative)
        if (allocated(state_energies)) deallocate(state_energies)
        allocate(absolute_to_relative,source=this%absolute_to_relative,stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','get_single_property','Memory allocation 2 failed.',err,1)
        allocate(state_energies,source=this%state_energies,stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','get_single_property','Memory allocation 3a failed.',err,1)

        mxstat = maxval(this%sym_index)
        if (mxstat .le. 0) call xermsg('properties_file_mod','get_single_property','There are no states on the file!',4,1)

        !Read the property data (key = 1): determine the number of different symmetry blocks
        irr_map = 0
        block = 0
        do i=this%key_start_end(1,1),this%key_start_end(2,1)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx
           it1=inx(2)
           it2=inx(4)
           lop=inx(7)
           mop=inx(8)
           if (lop .ne. lp .or. mop .ne. mp) cycle
           irr1 = inx(3)+1
           irr2 = inx(5)+1

           if (absolute_to_relative(2,it1) .ne. irr1 .or. absolute_to_relative(2,it2) .ne. irr2) then
              print *,it1,irr1,absolute_to_relative(2,it1)
              print *,it2,irr2,absolute_to_relative(2,it2)
              call xermsg('properties_file_mod','get_single_property',&
                          'State data and property data are not compatible...corrupted property file?',5,1)
           endif

           if (irr_map(irr1,irr2) .ne. 0) cycle

           block = block + 1
           irr_map(irr1,irr2) = block
        enddo !i

        intr = block

        if (block .eq. 0) write(*,'(5X,"The selected property is zero for all pairs of states.")')
        i = max(block,1)

        if (allocated(dipsto)) deallocate(dipsto)
        if (allocated(iidip)) deallocate(iidip)
        if (allocated(ifdip)) deallocate(ifdip)
        allocate(dipsto(mxstat,mxstat,i),iidip(i),ifdip(i),stat=err)
        if (err .ne. 0) call xermsg('properties_file_mod','get_single_property','Memory allocation 3 failed.',err,1)

        !Read the property data (key = 1): save the dipoles into the symmetry blocks
        n_prop = 0
        dipsto = 0.0_wp
        iidip = -1
        ifdip = -1
        do i=this%key_start_end(1,1),this%key_start_end(2,1)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx
           it1=inx(2)
           it2=inx(4)
           lop=inx(7)
           mop=inx(8)
           if (lop .ne. lp .or. mop .ne. mp) cycle
           irr1 = inx(3)+1
           irr2 = inx(5)+1

           block = irr_map(irr1,irr2)
           iidip(block) = irr1
           ifdip(block) = irr2

           it1 = absolute_to_relative(1,it1)
           it2 = absolute_to_relative(1,it2)

           dipsto(it1,it2,block) = dnx
           if (irr1 .eq. irr2) dipsto(it2,it1,block) = dipsto(it1,it2,block)
           !write(*,'(2i7,i2,e25.15)') it1,it2,block,dnx

           n_prop = n_prop+1
        enddo !i

        write(*,'(/,5X,"Number of different symmetry blocks:",i2)') intr
        write(*,'(/,5X,"IRRs of the wavefunctions in the symmetry blocks:")')
        do block=1,intr
           write(*,'(5X,i2,": ",i1,1X,i1)') block,iidip(block),ifdip(block)
        enddo

        write(*,'(/,5X,"Total number of properties that have been read-in: ",i15)') n_prop
        write(*,'(/,10X,"get_single_property: finished")')

   end subroutine get_single_property

end module properties_file_mod

!ZM: reads-in the dipole files produced by compak and the dyson orbital file produced by CDENPROP/integral code. The dipoles and Dyson orbitals are evaluated on a grid of chosen order and the
!results are output to files. Correctness of the code has been checked against dipelm for CO2. The Dyson orbitals are always evaluated in the three planes: xy,yz,xz.
program dipoles_hhg
   use dipelmdefs
   use utils_gbl, only: xermsg
   use atomic_basis_gbl
   use molecular_basis_gbl
   use mpi_gbl
   use precisn_gbl
   use const_gbl, only: nuc_nam_len
   use properties_file_mod
   use phys_const_gbl, only: to_ev
   implicit none

   integer, parameter :: max_selected_states = 10, max_grid_points = 2*360

   !Order of x,y,z components from dipelmprocs: dcomp2i
   integer, parameter :: component_map(1:3) = (/3,1,2/)

   integer :: n_points, n, max_l, n_energies, n_final_states, n_initial_states, bound_state_index, i, grid_type,&
              phase_bound(max_selected_states)
   integer :: final_state_num(max_selected_states),final_state_sym(max_selected_states),final_state_spin(max_selected_states),&
              final_state_index_map(max_selected_states)
   integer :: initial_state_num(max_selected_states),initial_state_sym(max_selected_states),&
              initial_state_spin(max_selected_states),initial_state_index_map(max_selected_states), max_initial_index
   real(kind=idp), allocatable :: escat(:), initial_state_energy(:), final_state_energy(:)
   real(kind=idp) :: radius, delta_r, max_r, first_energy_ev, angular_grid(4,max_grid_points), ip, accurate_ip
   complex(kind=idp), allocatable :: Xlm_on_grid(:,:), dipoles(:,:,:,:,:) !mol frame photodipoles: lm,q,E,final_state,initial_state
   complex(kind=idp), allocatable :: dipoles_on_grid(:,:,:,:,:) !component(x,y,z),point,E,final_state,initial_state
   complex(kind=idp), allocatable :: dipoles_on_grid_parallel_light(:,:,:,:,:) !component(x,y,z),point,E,final_state,initial_state
   real(kind=idp), allocatable :: dipoles_phase(:,:,:,:) !point,E,final_state,initial_state
   integer, allocatable :: sign_at_r(:,:,:), orbital_index(:)
   real(kind=idp), allocatable :: atom_xyz(:,:), final_state_dipole_table(:,:,:), initial_state_dipole_table(:,:,:)
   real(kind=idp), allocatable :: photo_dcs(:,:,:,:)
   character(len=nuc_nam_len), allocatable :: atom_name(:)
   integer :: n_atoms
   character(len=80) :: molecule_name, dyson_ukrmolp_file, dyson_cdenprop_file, final_state_property_file,&
                        initial_state_property_file
   character(len=1) :: dcomp(maxprop_par)
   character(len=2) :: plane
   integer :: orbital_num(max_selected_states,max_selected_states), orbital_sym(max_selected_states,max_selected_states),&
              format_flag
   logical :: generate_matlab_data, eval_dysons_in_planes, generate_dipoles_on_grid

   type(atomic_orbital_basis_obj), target :: atomic_orbital_basis
   type(molecular_orbital_basis_obj) :: dyson_orbitals
   type(properties_file_obj) :: target_properties, np1_properties

   NAMELIST/INPUT/n_final_states,molecule_name,bound_state_index,radius,dyson_ukrmolp_file,delta_r,max_r,first_energy_ev,&
                  generate_matlab_data,generate_dipoles_on_grid,eval_dysons_in_planes,dyson_cdenprop_file,final_state_num,&
                  final_state_sym,final_state_spin,phase_bound,final_state_property_file,initial_state_property_file,&
                  n_initial_states,initial_state_num,initial_state_sym,initial_state_spin,format_flag,accurate_ip
   NAMELIST/ANGGRID/grid_type,n,angular_grid

      call print_ukrmol_header(6)
      call mpi_mod_start

      write(*,'(/,10X,"Dipoles: generating dipole and Dyson orbital data for HHG calculations")')
      write(*,'(  10X,"======================================================================",/)')

      molecule_name = 'no name'

      dyson_ukrmolp_file = 'dyson_orbitals.ukrmolp0' !File containing the Dyson orbitals in the UKRMol+ format
      dyson_cdenprop_file = 'fort.123'               !File containing the Dyson orbitals in the CDENPROP format
      final_state_property_file = 'fort.24'          !File containing the target properties
      initial_state_property_file = 'fort.667'       !File containing the dipoles between the bound states

      bound_state_index = 1                          !Index of the initial bound state on the dipoles file
      phase_bound = 1                                !Phases to be applied to the dipoles and Dysons for the corresponding bound states.
      !FOR EACH INITIAL STATE SPECIFY ITS SYMMETRY AND INDEX WITHIN SYMMETRY:
      n_initial_states = -1                          !Number of initial states for which dipoles and Dysons are requested
      initial_state_num = 1  !relative index of the state within its own spin-space symmetry
      initial_state_sym = 1  !1:8   
      initial_state_spin = 1 !2*S+1
      !FOR EACH CATIONIC STATE SPECIFY ITS SYMMETRY AND INDEX WITHIN SYMMETRY:
      n_final_states = -1                !Number of final states for which dipoles and Dysons are requested
      final_state_num = 1  !relative index of the state within its own spin-space symmetry
      final_state_sym = 1  !1:8   
      final_state_spin = 1 !2*S+1

      first_energy_ev = 0.0_cfp !starting energy for the output of dipoles: the values below this energy will be set to 0.
      accurate_ip = -1.0_cfp
      generate_dipoles_on_grid = .false. !If set to true then the momentum dipoles will be evaluated on the chosen angular grid, parallel light cross sections computed and everything saved to disk
      generate_matlab_data = .false.  !If set to true then the final set of data will be put on one file suitable for use with the HHG scripts. 
                                      !If set to false the data will be put into separate files as used in the CO2 HHG work.
      eval_dysons_in_planes = .false. !If set to true then the Dyson orbitals will be evaluated in the three coordinate planes xy,yz,xz, too.
      radius = -1.0_idp                              !Radius in a.u. at which the Dyson orbitals will be evaluated to get their signs
      delta_r = 0.1_cfp !spacing for evaluation of the Dyson orbitals in the plane
      max_r = 10.0_cfp  !maximum distance for which the Dyson orbital will be evaluated in the plane
      format_flag = 1   !used in routines write_dipoles_on_grid* to select format of the output surface data: python (1), gnuplot (2), binary for phase-matching algorithm (3)

      READ(5,nml=INPUT)

      if (n_final_states .le. 0) stop "On input: n_final_states .le. 0"
      if (n_initial_states .le. 0) stop "On input: n_initial_states .le. 0"
      if (radius .le. 0.0_idp) stop "On input: radius .le. 0.0_idp"
      if (accurate_ip .le. 0.0_idp) stop "On input: accurate_ip was .le. 0"
      max_initial_index = maxval(initial_state_num(1:n_initial_states))

      n = 0                                         !Either order of the Lebedev quadrature (NOT THE NUMBER OF QUADRATURE POINTS) or the number of points in a different angular grid (see below)
      ! Choose the angular grid:
      !1 = Lebedev grid of order n, 2:4 = (zy,zx,xy) planes regular theta grid with n points, 5 = custom grid of n x,y,z,w points to be taken from the user-supplied array angular_grid
      !If grid_type .le. 0 then the dipoles will not be processed, i.e. not even required on input. This functionality can be used to work only with the Dysons.
      grid_type = 0
      angular_grid = 0.0_idp !if grid_type = 5 specify x,y,z,w for each grid point.

      READ(5,nml=ANGGRID)

      if (grid_type > 0 .and. n .le. 0) stop "On input: n .le. 0 while grid_type > 0"
      if (n > max_grid_points) stop "On input n exceeds the limiting value max_grid_points; change the code and recompile"
      initial_state_index_map = 1

!SET-UP THE ANGULAR GRID

      write(*,'(/,10X,"ANGULAR GRID")')

      select case (grid_type)
      case (:0)
         write(*,'("No angular grid has been selected: the dipoles will not be processed.")')
      case (1)
         call eval_lebedev_grid(n,angular_grid,n_points)
      case (2:4)
         if (grid_type .eq. 2) plane = 'zy'
         if (grid_type .eq. 3) plane = 'zx'
         if (grid_type .eq. 4) plane = 'xy'
         call eval_in_plane_grid(plane,n,angular_grid,n_points)
      case (5)
         n_points = n
         write(*,'("Using custom angular grid:")')
         do i=1,n_points
            write(*,'(i10,4e25.15)') i,angular_grid(1:4,i)
         enddo
      case (6:)
         stop "wrong value of grid_type"
      end select

!PROCESS THE DYSON ORBITALS

      write(*,'(/,10X,"DYSON ORBITALS")')

      dyson_orbitals%ao_basis => atomic_orbital_basis
      call atomic_orbital_basis%read(dyson_ukrmolp_file)

      call dyson_orbitals%read(dyson_ukrmolp_file)

      !Work out the mapping between the requested cationic states, the Dyson orbital indices on the ukrmolp file. Output orbital_num, orbital_sym for all bound states on the file.
      call read_cdenprop_dysons(dyson_cdenprop_file,n_final_states,final_state_num,final_state_sym,final_state_spin,orbital_num,&
                                orbital_sym)

      !Evaluate the amplitudes and signs of the Dyson orbitals for all requested
      !final states and all bound states with index .le. max_initial_index.
      if (grid_type > 0) call eval_Dysons_on_grid(dyson_orbitals,n_final_states,orbital_num,orbital_sym,radius,angular_grid,&
                                                  n_points,sign_at_r,max_initial_index)

      if (eval_dysons_in_planes) then
         do i=1,3
            if (i .eq. 1) plane = 'xy'
            if (i .eq. 2) plane = 'zy'
            if (i .eq. 3) plane = 'zx'
            call eval_Dysons_in_plane(dyson_orbitals,max_initial_index,n_final_states,orbital_num,orbital_sym,plane,delta_r,max_r,&
                                      delta_r,max_r)
         enddo
      endif

!PROCESS THE DIPOLES BUT ONLY IF A GRID HAS BEEN SELECTED
!APPLY THE PHASE OF THE BOUND STATE TO THE DIPOLES AND DYSONS

      write(*,'(/,10X,"PHOTODIPOLES")')

      if (grid_type > 0) then
         !Get the partial wave dipoles for the requested cationic states and all
         !bound states up the state with index max_initial_index.
         call get_dipoles(max_initial_index,initial_state_num,final_state_num,final_state_sym,final_state_spin,ip,dipoles,dcomp,&
                          max_l,n_energies,n_final_states,escat)
   
         !Extract dipole couplings and energies for the requested cationic states
         call target_properties%load_file(final_state_property_file)
         call target_properties%get_state_dipoles(final_state_num,final_state_sym,final_state_spin,n_final_states,&
                                                  final_state_dipole_table,final_state_energy,final_state_index_map)
         !Extract dipole couplings and energies for the requested bound states and find the mapping between the bound states requested and their indices on the dipoles files.
         !Note that this is not a universal solution: if bound states are requested which do not all appear in a single CDENPROP file then this method would not work.
         call np1_properties%load_file(initial_state_property_file)
         call np1_properties%get_state_dipoles(initial_state_num,initial_state_sym,initial_state_spin,n_initial_states,&
                                               initial_state_dipole_table,initial_state_energy,initial_state_index_map)

         if (generate_matlab_data)  then
            !Molecular geometry
            call get_geometry(dyson_orbitals,atom_xyz,atom_name,n_atoms)
            !Use the format for use in the Matlab scripts: put all data into one file
            call write_data_matlab(phase_bound,dipoles,max_l,n_final_states,escat,size(escat),angular_grid,n_points,sign_at_r,&
                                   atom_xyz,atom_name,n_atoms,initial_state_energy,final_state_energy,final_state_num,&
                                   final_state_sym,final_state_spin,orbital_num,orbital_sym,initial_state_index_map,&
                                   n_initial_states,initial_state_num,initial_state_sym,initial_state_spin,max_initial_index,&
                                   molecule_name,final_state_dipole_table,initial_state_dipole_table)
         endif

         if (generate_dipoles_on_grid) then
            !Use standard format for the dipoles as used in the CO2 work
            call write_energy_grid(escat,size(escat),accurate_ip)
            call write_angular_grid(angular_grid,n_points)
            call write_angular_grid_spherical(angular_grid,n_points)
            call write_dyson_signs(phase_bound,sign_at_r,n_points,n_final_states,max_initial_index,initial_state_index_map)

            write(6,'("Starting energy (eV) for output of dipoles: ",e25.15)') first_energy_ev

            call eval_Xlm_on_grid(max_l,angular_grid,n_points,Xlm_on_grid)

            call eval_dipoles_on_grid(dipoles,max_l,n_final_states,n_points,Xlm_on_grid,dipoles_on_grid,max_initial_index)
            call write_dipoles_on_grid(phase_bound,dipoles_on_grid,escat,first_energy_ev,n_final_states,n_initial_states,&
                                       initial_state_index_map,molecule_name,angular_grid,n_points,format_flag,'REI')

            call transform_xyz_dipoles_to_parallel_light_dipoles(dipoles_on_grid,phase_bound,angular_grid,&
                                                                 dipoles_on_grid_parallel_light,dipoles_phase)
            call write_dipoles_on_grid(phase_bound,dipoles_on_grid_parallel_light,escat,first_energy_ev,n_final_states,&
                                       n_initial_states,initial_state_index_map,molecule_name,angular_grid,n_points,format_flag,&
                                       'PAR')
            call compute_dcs(dipoles_on_grid_parallel_light,escat,accurate_ip,photo_dcs)
            call write_dcs(photo_dcs,escat,accurate_ip,initial_state_index_map,molecule_name,format_flag)
            call write_dipoles_phase(dipoles_phase,escat,accurate_ip,initial_state_index_map,molecule_name,format_flag)
         endif
      endif

      write(*,'(/,10X,"Dipoles: finished",/)')

      call mpi_mod_finalize

contains

   subroutine get_geometry(dyson_orbitals,atom_xyz,atom_name,n_atoms)
     use const_gbl, only: nuc_nam_len
     implicit none
     type(molecular_orbital_basis_obj), intent(in) :: dyson_orbitals
     real(kind=idp), allocatable :: atom_xyz(:,:)
     character(len=nuc_nam_len), allocatable :: atom_name(:)
     integer, intent(out) :: n_atoms

     integer :: err, i, j
     logical :: have_continuum

        if (allocated(atom_xyz)) deallocate(atom_xyz)
        if (allocated(atom_name)) deallocate(atom_name)

        n_atoms = dyson_orbitals%symmetry_data%no_nuc-1 !exclude the scattering centre

        allocate(atom_xyz(3,n_atoms),atom_name(n_atoms),stat=err)
        if (err .ne. 0) stop "memory allocation failed"

        have_continuum = .false.
        j = 0
        do i=1,size(dyson_orbitals%symmetry_data%nucleus)
           if (dyson_orbitals%symmetry_data%nucleus(i)%is_continuum()) then
              have_continuum = .true.
              cycle
           endif
           j = j + 1
           atom_xyz(1:3,j) = dyson_orbitals%symmetry_data%nucleus(j)%center(1:3)
           atom_name(j) = dyson_orbitals%symmetry_data%nucleus(j)%name
        enddo

        if (j .ne. n_atoms .or. .not.(have_continuum)) stop "error in geometry data: expecting one continuuum centre!"

   end subroutine get_geometry

   subroutine read_cdenprop_dysons(dyson_cdenprop_file,n_final_states,final_state_num,final_state_sym,final_state_spin,orbital_num,&
                                   orbital_sym)
     implicit none
     character(len=80) :: dyson_cdenprop_file
     integer, intent(in) :: final_state_num(max_selected_states),final_state_sym(max_selected_states),&
                            final_state_spin(max_selected_states),n_final_states
     integer, intent(out) :: orbital_num(max_selected_states,max_selected_states), &
                             orbital_sym(max_selected_states,max_selected_states)

     integer :: lu, indices(8), dyson_relative_index(8), dyson_sym, i, j, tmp(8), &
                state_relative_index(n_final_states,max_selected_states)
     real(kind=idp) :: cf
     logical :: found_dyson(n_final_states,max_selected_states)

        write(*,'("Reading CDENPROP Dyson orbitals file...")')

        lu = 100        
        open(file=dyson_cdenprop_file,unit=lu,status='old',form='FORMATTED')

        !Skip header
        read(lu,*)
        read(lu,*)

        dyson_relative_index = 0 !Relative index of each Dyson orbital within its spatial symmetry
        state_relative_index = 0
        found_dyson = .false.
        do
           read(lu,*,end=100) !Skip the line with Dyson orbital norm

           !Read the Dyson orbital coefficients and the corresponding state symmetries, etc.
           do
              read(lu,*,end=110,err=110) tmp,cf
              indices = tmp
           enddo
       110 dyson_sym = indices(5)+1
           dyson_relative_index(dyson_sym) = dyson_relative_index(dyson_sym) + 1

           if (indices(1) > max_selected_states) then
              print *,max_selected_states,indices(1)
              stop "max_selected_states too small: increase and recompile"
           endif

           do j=1,n_final_states
              if (final_state_spin(j) .eq. indices(3) .and. final_state_sym(j) .eq. indices(4)+1) then
                 state_relative_index(j,indices(1)) = state_relative_index(j,indices(1)) + 1
              endif
              if (final_state_num(j) .eq. state_relative_index(j,indices(1)) .and. final_state_spin(j) .eq. indices(3)&
                  .and. final_state_sym(j) .eq. indices(4)+1) then
                 write(*,'("Bound state index: ",i4)') indices(1)
                 orbital_num(j,indices(1)) = dyson_relative_index(dyson_sym)
                 orbital_sym(j,indices(1)) = dyson_sym
                 found_dyson(j,indices(1)) = .true.
                 write(*,'("Cationic state ",i4," has been matched with Dyson orbital (num,sym): ",i4,".",i1)') &
                       j,orbital_num(j,indices(1)),orbital_sym(j,indices(1))
                 exit
              endif
           enddo

           read(lu,*,end=100) !empty line

        enddo

    100 write(*,'("Finished reading CDENPROP Dyson orbitals file.")')

        if (count(found_dyson(1:n_final_states,1:indices(1))) .ne. n_final_states*indices(1)) then
           print *,found_dyson
           stop "Error: some Dyson orbitals corresponding to the given bound state and cationic states have not been found"
        endif

   end subroutine read_cdenprop_dysons

   subroutine write_angular_grid(angular_grid,n_points)
     implicit none
     real(kind=idp), intent(in) :: angular_grid(4,max_grid_points)
     integer, intent(in) :: n_points

     integer :: lu, i, err

        lu = 100
        open(file='angular_grid',unit=lu,status='replace',form='FORMATTED')

        !write(lu,'("#Data in columns: grid point index, grid point x, grid point y, grid point z, quadrature weight")')
        write(lu,'(i10)') n_points
        do i=1,n_points
           write(lu,'(i10,4(e25.15))') i, angular_grid(1:4,i)
        enddo

        close(lu)

        write(6,'("Grid has been written successfully")')

   end subroutine write_angular_grid

   subroutine write_angular_grid_spherical(angular_grid,n_points)
     use lebedev_gbl
     implicit none
     real(kind=idp), intent(in) :: angular_grid(4,max_grid_points)
     integer, intent(in) :: n_points

     integer :: lu, i, err
     real(kind=cfp) :: theta,phi,x,y,z

        lu = 100
        open(file='angular_grid_spherical',unit=lu,status='replace',form='FORMATTED')

        !write(lu,'("#Data in columns: grid point index, grid point x, grid point y, grid point z, quadrature weight")')
        write(lu,'(i10)') n_points
        do i=1,n_points
           x = angular_grid(1,i); y = angular_grid(2,i); z = angular_grid(3,i)
           call xyz_to_tp(x,y,z,phi,theta)
           write(lu,'(i10,3(e25.15))') i, theta,phi,angular_grid(4,i)
        enddo

        close(lu)

        write(6,'("Grid has been written successfully")')

   end subroutine write_angular_grid_spherical

   subroutine eval_Dysons_on_grid(dyson_orbitals,n_final_states,orbital_num,orbital_sym,radius,angular_grid,n_points,sign_at_r,&
                                  n_total_initial_states)
     implicit none
     type(molecular_orbital_basis_obj) :: dyson_orbitals
     integer, intent(in) :: n_final_states, orbital_num(max_selected_states,max_selected_states), &
                            orbital_sym(max_selected_states,max_selected_states), n_points,n_total_initial_states
     real(kind=idp), intent(in) :: radius, angular_grid(4,max_grid_points)
     real(kind=cfp), allocatable :: orbital_at_r(:), r(:,:)
     integer, allocatable :: sign_at_r(:,:,:)

     integer, allocatable :: tmp(:)
     character(len=80) :: str, path

     integer :: i, num, sym, j, cnt, k

        write(6,'("Evaluating the Dyson orbitals on the grid...")')

        write(6,'("Radius = ",e25.15)') radius

        if (allocated(sign_at_r)) deallocate(sign_at_r)
        allocate(r(3,n_points),sign_at_r(n_points,n_final_states,n_total_initial_states))
        sign_at_r = 0

        do i=1,n_points
           r(1,i) = radius*angular_grid(1,i)
           r(2,i) = radius*angular_grid(2,i)
           r(3,i) = radius*angular_grid(3,i)
        enddo !i

        do k=1,n_total_initial_states
           cnt = 0
           do i=1,dyson_orbitals%number_of_functions
              num = dyson_orbitals%get_index_within_symmetry(i)
              sym = dyson_orbitals%get_orbital_symmetry(i)
              do j=1,n_final_states
                 if (orbital_num(j,k) .eq. num .and. orbital_sym(j,k) .eq. sym) then
                    cnt = cnt + 1
                    write(6,'("Evaluating orbital: ",i5,".",i1)') num,sym
                    call dyson_orbitals%eval_orbital(i,r,n_points,orbital_at_r,tmp)
   
                    sign_at_r(1:n_points,cnt,k) = tmp(1:n_points)
       
                    exit
                 endif
              enddo
           enddo !i
           if (cnt .ne. n_final_states) call xermsg('main','eval_Dysons_on_grid',&
                                        'Some of the requested Dyson orbitals have not been found on the Dyson orbital file.',1,1)
        enddo !k

        write(6,'("...done")')

   end subroutine eval_Dysons_on_grid

   subroutine write_dyson_signs(phase_bound,sign_at_r,n_points,n_final_states,n_total_initial_states,initial_state_index_map)
     implicit none
     integer, intent(in) :: n_total_initial_states, phase_bound(max_selected_states),n_points,n_final_states,&
                            sign_at_r(n_points,n_final_states,n_total_initial_states),initial_state_index_map(max_selected_states)

     integer :: lu, i, j, k
     character(len=80) :: str, path

     do k=1,n_total_initial_states

        if (phase_bound(k) .ne. 1) write(*,'("Applying phase-correction ",i2," to the bound state.")') phase_bound(k)
   
        do j=1,n_final_states
   
           write(str,'(i5)') j
           path = 'dyson_orbital_sign'//'.'//trim(adjustl(str))
           write(6,'("Saving to file: ",a)') adjustl(trim(path))
           lu = 100+j
           open(file=trim(path),unit=lu,form='FORMATTED')
   
           write(lu,'("#Data in columns: grid point index, sign at grid point, value at grid point")')
           do i=1,n_points
              write(lu,'(i10,1X,i2)') i, sign_at_r(i,j,initial_state_index_map(k))*phase_bound(k)
           enddo
        enddo

     enddo

   end subroutine write_dyson_signs

   subroutine eval_Dysons_on_grid_for_different_radii(dyson_orbitals,n_final_states,orbital_num,orbital_sym,delta_r,max_r,&
                                                      angular_grid,n_points)
     implicit none
     type(molecular_orbital_basis_obj) :: dyson_orbitals
     integer, intent(in) :: n_final_states, orbital_num(max_selected_states), orbital_sym(max_selected_states), n_points
     real(kind=idp), intent(in) :: angular_grid(4,max_grid_points), delta_r, max_r
     real(kind=cfp), allocatable :: orbital_at_r(:), r(:,:)
     integer, allocatable :: sign_at_r(:)
     character(len=80) :: str, path

     integer :: i, num, sym, j, k, r_it, lu, cnt, n_r, n_total_points, n
     real(kind=cfp) :: radius

        write(6,'("Evaluating the Dyson orbitals on the grid with radial spacing: ",e25.15)') delta_r

        write(6,'("Maximum radius = ",e25.15)') max_r

        if (max_r .le. 0.0_cfp .or. delta_r .le. 0.0_cfp) call xermsg('main','eval_Dysons_on_grid_for_different_radii',&
                                                                      'On input max_r and/or delta_r were incorrect.',1,1)

        n_r = max_r/delta_r
        n_total_points = n_r*n_points

        allocate(r(3,n_total_points))
   
        radius = 0.0_cfp
        k = 0
        do j=1,n_r
           radius = radius + delta_r
           do i=1,n_points
              k = k + 1
              r(1,k) = radius*angular_grid(1,i)
              r(2,k) = radius*angular_grid(2,i)
              r(3,k) = radius*angular_grid(3,i)
           enddo !i
        enddo !j

        cnt = 0
        do i=1,dyson_orbitals%number_of_functions
           num = dyson_orbitals%get_index_within_symmetry(i)
           sym = dyson_orbitals%get_orbital_symmetry(i)
           do j=1,n_final_states
              if (orbital_num(j) .eq. num .and. orbital_sym(j) .eq. sym) then
                 cnt = cnt + 1
                 write(6,'("Evaluating orbital: ",i5,".",i1)') num,sym
                 call dyson_orbitals%eval_orbital(i,r,n_total_points,orbital_at_r,sign_at_r)
    
                 write(str,'(i5)') j
                 path = 'dyson_orbital_for_different_r'//'.'//trim(adjustl(str))
                 write(6,'("Saving to file: ",a)') adjustl(trim(path))
                 lu = 100+j
                 open(file=trim(path),unit=lu,form='FORMATTED')
  
                 write(lu,'("#Orbital: ",i5,".",i1)') num,sym
                 n = 0
                 do r_it=1,n_r
                    write(lu,'(1000e25.15)') orbital_at_r(n+1:n+n_points)
                    n = n + n_points
                 enddo !r
   
                 close(lu)
                 exit
              endif
           enddo
        enddo !i

        if (cnt .ne. n_final_states) call xermsg('main','eval_Dysons_on_grid_for_different_radii',&
                                     'Some of the requested Dyson orbitals have not been found on the Dyson orbital file.',1,1)

        write(6,'("...done")')

   end subroutine eval_Dysons_on_grid_for_different_radii

   subroutine eval_Dysons_in_plane(dyson_orbitals,n_initial_states,n_final_states,orbital_num,orbital_sym,plane,delta_1,max_1,&
                                   delta_2,max_2)
     implicit none
     type(molecular_orbital_basis_obj) :: dyson_orbitals
     integer, intent(in) :: n_final_states, orbital_num(max_selected_states,max_selected_states), &
                            orbital_sym(max_selected_states,max_selected_states), n_initial_states
     real(kind=idp), intent(in) :: delta_1, max_1, delta_2, max_2
     character(len=2), intent(in) :: plane

     real(kind=cfp), allocatable :: orbital_at_r(:), r(:,:)
     integer, allocatable :: sign_at_r(:)
     character(len=80) :: str, path, strI

     integer :: i, num, sym, j, k, r_it, lu, cnt, n_1, n_2, n_total_points, n
     real(kind=cfp) :: x, z

        write(6,'("Evaluating the Dyson orbitals in the plane: ",a2)') plane

        write(6,'("Maximum radius and step: ",4e25.15)') max_1, delta_1, max_2, delta_2

        if (max_1 .le. 0.0_cfp .or. delta_1 .le. 0.0_cfp) call xermsg('main','eval_Dysons_on_grid_for_different_radii',&
                                                                      'On input max_1 and/or delta_1 were incorrect.',1,1)
        if (max_2 .le. 0.0_cfp .or. delta_2 .le. 0.0_cfp) call xermsg('main','eval_Dysons_on_grid_for_different_radii',&
                                                                      'On input max_2 and/or delta_2 were incorrect.',2,1)

        n_1 = 2*max_1/delta_1
        n_2 = 2*max_2/delta_2
        n_total_points = n_1*n_2

        allocate(r(3,n_total_points))
        r = 0.0_cfp
   
        z = -max_2
        k = 0
        do j=1,n_2
           z = z + delta_2
           x = -max_1
           do i=1,n_1
              k = k + 1
              x = x + delta_1
              if (plane .eq. 'zx') then
                 r(1,k) = x
                 r(3,k) = z
              elseif (plane .eq. 'zy') then
                 r(2,k) = x
                 r(3,k) = z
              elseif (plane .eq. 'xy') then
                 r(1,k) = x
                 r(2,k) = z
              else
                 stop "wrong plane string"
              endif
           enddo !i
        enddo !j

        do k=1,n_initial_states
           write(strI,'(i5)') k
           cnt = 0
           do i=1,dyson_orbitals%number_of_functions
              num = dyson_orbitals%get_index_within_symmetry(i)
              sym = dyson_orbitals%get_orbital_symmetry(i)
              do j=1,n_final_states
                 if (orbital_num(j,k) .eq. num .and. orbital_sym(j,k) .eq. sym) then
                    cnt = cnt + 1
                    write(6,'("Evaluating orbital: ",i5,".",i1)') num,sym
                    call dyson_orbitals%eval_orbital(i,r,n_total_points,orbital_at_r,sign_at_r)
       
                    write(str,'(i5)') j
                    path = 'dyson_orbital_in_plane_'//plane//'.'//trim(adjustl(str))//'.'//trim(adjustl(strI))
                    write(6,'("Saving to file: ",a)') adjustl(trim(path))
                    lu = 100+j
                    open(file=trim(path),unit=lu,form='FORMATTED')
     
                    write(lu,'("#Orbital: ",i5,".",i1)') num,sym
                    n = 0
                    do r_it=1,n_2
                       write(lu,'(1000e25.15)') orbital_at_r(n+1:n+n_1)
                       n = n + n_1
                    enddo !r
      
                    close(lu)
                    exit
                 endif
              enddo
           enddo !i
           if (cnt .ne. n_final_states) call xermsg('main','eval_Dysons_on_grid_for_different_radii',&
                                        'Some of the requested Dyson orbitals have not been found on the Dyson orbital file.',1,1)
        enddo !k

        write(6,'("...done")')

  end subroutine eval_Dysons_in_plane

  subroutine write_energy_grid(escat,no_scattering_energies,ip)
     implicit none
     integer, intent(in) :: no_scattering_energies
     real(kind=idp) :: escat(no_scattering_energies), ip

     integer :: i, lu, lu1

        write(6,'("Writing the energy grid data...")')

        lu = 100
        lu1 = 101
        open(file='energy_grid',unit=lu,status='replace',form='FORMATTED')
        open(file='photon_energy_grid_eV',unit=lu1,status='replace',form='FORMATTED')

        write(lu,'(i10)') no_scattering_energies
        write(lu1,'(i10)') no_scattering_energies

        do i=1,no_scattering_energies
           write(lu,'(i10,e25.15)') i,escat(i)
           write(lu1,'(i10,e25.15)') i,(escat(i)+ip)*to_ev
        enddo

        close(lu)
        close(lu1)

        write(6,'("...done")')

  end subroutine write_energy_grid

  subroutine get_dipoles(n_total_initial_states,initial_state_num,final_state_num,final_state_sym,final_state_spin,ip,dipoles,dcomp&
                         ,max_l,n_energies,n_final_states,escat)
     use dipelmprocs, only: dipelm_drv, dcomp2i, cphaz, transform_dipoles_to_complex_spherical_harmonic_basis
     use angmom_procs, only: lm2i
     implicit none
     integer, intent(in) :: n_total_initial_states,initial_state_num(max_selected_states),final_state_num(max_selected_states),&
                            final_state_sym(max_selected_states),final_state_spin(max_selected_states),n_final_states
     integer, intent(out) :: max_l, n_energies
     complex(kind=idp), allocatable :: dipoles(:,:,:,:,:) !mol frame photodipoles: lm,q,E,final_state,initial_state
     real(kind=idp), allocatable :: escat(:)
     real(kind=idp) :: ip

     real(kind=idp) :: target_energy
     integer :: ifail, nesc,nbound,ndcomp
     integer,dimension(:) :: nchans(maxprop_par)
     integer, allocatable,dimension(:,:) :: ichl,lvchl,mvchl
     integer, allocatable,dimension(:) :: starg, mtarg, gtarg 
     real(kind=idp), allocatable,dimension(:,:) :: evchl
     real(kind=idp), allocatable, dimension(:) :: bound_state_energies 
     complex(kind=idp), allocatable, dimension(:,:,:,:)  :: dip_elm   
     character(len=1) :: dcomp(maxprop_par)
     integer :: lu_pw_dipoles(maxprop_par), nset_pw_dipoles(maxprop_par)
     character(len=11) :: format_pw_dipoles

     integer :: ierr, intermediate_state, no_partial_waves, no_scattering_energies, no_bound_states, no_targ_states
     integer :: i,j,k,l,m,en,ii,idcmp,itarget,iii,state_relative_index(n_final_states)
     complex(kind=idp), allocatable :: dip_full(:,:,:,:,:), coulomb_phase(:,:,:)
     real(kind=idp), allocatable :: etarg(:)
     real(kind=idp) :: k_final, eta, charge, d
     logical :: found_state(max_selected_states)

        call dipelm_drv( dip_elm, ichl, evchl, lvchl, mvchl, nchans, starg, mtarg, gtarg, escat, ndcomp, dcomp, &
                         bound_state_energies, target_energy, lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, ifail )
        if (ifail .ne. 0) then
           write(6,1010) ifail
           stop
        end if

        ip = target_energy - bound_state_energies(1)

        charge = 1.0_idp
        max_l = maxval(lvchl)
        no_partial_waves=(max_l+1)**2
        no_scattering_energies = size(dip_elm,4)
        no_bound_states = size(dip_elm,2)
        no_targ_states = maxval(ichl)

        n_energies = no_scattering_energies

        if (maxval(initial_state_num) > no_bound_states) then
           stop "The in input value in index_initial_state exceeds the number of bound states available on the dipoles file"
        endif

        !     Determine target energies from channel energies
        !     ------------------------------------------------------------------
        
        allocate(etarg(no_targ_states))
        etarg=0.0_idp
        itarget=1
        
        do i=1,size(evchl,2)
           itarget=ichl(dcomp2i(dcomp(1)),i)
           if(itarget .ne. 0) etarg(itarget)=evchl(dcomp2i(dcomp(1)),i)/2.0_idp
        end do

        !     Construct $i^{-l}e^(i\sigma_l)$
        !     ------------------------------- 
        
        allocate(coulomb_phase(no_partial_waves,no_scattering_energies,no_targ_states))
        coulomb_phase=0.0_idp
        
        do itarget=1,no_targ_states
           do en=1,no_scattering_energies
              do l=0,max_l
                 do m=-l,l
                    if((escat(en)-etarg(itarget)) .gt. 0) then
        
                       k_final=sqrt(2*(escat(en)-etarg(itarget)))
                       eta=-charge/k_final
                       ii=lm2i(l,m)
                       coulomb_phase(ii,en,itarget)=(-eye)**(l)* exp(eye*CPHAZ(l,eta,6))
        
                    end if
                 end do     
              end do
           end do
        end do

        !     Re-jig the moments matrix so that the partial wave index 
        !     corresponds to that used by the spherical harmonics, coulomb phase
        !     and rotation matrices. Multiply the dipoles by the Coulomb phase.
        !     ------------------------------------------------------------------

        allocate(dip_full(no_partial_waves,3,no_scattering_energies,no_targ_states,n_total_initial_states))
        dip_full=0.0_idp
        
        do en=1,no_scattering_energies
           do j=1,ndcomp
              idcmp=dcomp2i(dcomp(j))
        
              do k=1,nchans(j)
                 l=lvchl(idcmp,k)
                 m=mvchl(idcmp,k)
                 ii=lm2i(l,m)
                 itarget=ichl(idcmp,k)
        
                 do i=1,n_total_initial_states
                    dip_full(ii,idcmp,en,itarget,i)= dip_elm(k,initial_state_num(i),idcmp,en)*coulomb_phase(ii,en,itarget)
                 enddo
              end do
        
           end do
        end do

        if (allocated(dipoles)) deallocate(dipoles)
        allocate(dipoles(no_partial_waves,ndcomp,no_scattering_energies,n_final_states,n_total_initial_states))

        do k=1,n_total_initial_states
           found_state = .false.
           state_relative_index = 0
           do i=1,no_targ_states
              do j=1,n_final_states
                    if (final_state_spin(j) .eq. starg(i) .and. final_state_sym(j) .eq. mtarg(i)+1) then
                       state_relative_index(j) = state_relative_index(j) + 1
                       if (final_state_num(j) .eq. state_relative_index(j)) then
                          write(*,'("Requested cationic state ",i4," has been matched with state ",i4," from the dipoles file.")')&
                                    j,i
                          dipoles(1:no_partial_waves,1:ndcomp,1:no_scattering_energies,j,k) = &
                                 dip_full(1:no_partial_waves,1:ndcomp,1:no_scattering_energies,i,k)
                          found_state(j) = .true.
                          exit
                       endif
                    endif
              enddo
           enddo
   
           if (count(found_state) .ne. n_final_states) then
              print *,found_state
              stop "Error: some requested cationic sates have not been found on the dipoles file"
           endif
        enddo !k

 1010 format(/,5X,'ERROR CREATING DIPOLES, ERROR CODE',1X,I6,/)

  end subroutine get_dipoles

  subroutine eval_in_plane_grid(plane,n,angular_grid,n_points)
     use phys_const_gbl, only: pi
     implicit none
     integer, intent(in) :: n
     integer, intent(out) :: n_points
     real(kind=idp), intent(out) :: angular_grid(4,max_grid_points)
     character(len=2), intent(in) :: plane

     integer :: point
     real(kind=idp) :: theta

        write(6,'("Evaluating grid points in the plane...")')

        if (n .le. 0) stop "error in input n"

        if (plane .eq. 'zy') then
           write(6,'("Theta grid in the zy plane.")')
        elseif (plane .eq. 'zx') then
           write(6,'("Theta grid in the zx plane.")')
        elseif (plane .eq. 'xy') then
           write(6,'("Theta grid in the xy plane.")')
        else
           print *,'unsupported plane',plane
           stop
        endif

        n_points = n
        if (n_points > max_grid_points) stop "n_points > max_grid_points: increase the parameter max_grid_points and recompile"
     
        angular_grid(4,:) = 0.0_idp !todo temporary: replace with a quadrature rule

        do point = 1,n_points
           theta = ((point-1)/real(n_points))
           write(*,'("Point, angle: ",i10,e25.15)') point,theta*360
           theta = theta*2*pi
           if (plane .eq. 'zy') then
              angular_grid(2,point) = sin(theta)
              angular_grid(3,point) = cos(theta)
           elseif (plane .eq. 'zx') then
              angular_grid(1,point) = sin(theta)
              angular_grid(3,point) = cos(theta)
           elseif (plane .eq. 'xy') then
              angular_grid(1,point) = cos(theta)
              angular_grid(2,point) = sin(theta)
           endif
        enddo

  end subroutine eval_in_plane_grid

  subroutine eval_lebedev_grid(n,angular_grid,n_points)
     use phys_const_gbl, only: fourpi
     use lebedev_gbl
     implicit none
     integer, intent(inout) :: n
     integer, intent(out) :: n_points
     real(kind=idp), intent(out) :: angular_grid(4,max_grid_points)

     integer :: point, l, m, i, n_rule, available, lm, err
     real(kind=cfp), allocatable :: tmp1(:), tmp2(:), tmp3(:), tmp4(:)
     real(kind=cfp) :: x,y,z,t,p

        write(6,'("Evaluating Lebedev grid points...")')

        if (n .le. 0) stop "error in input n"

        allocate(tmp1(mmax),tmp2(mmax),tmp3(mmax),tmp4(mmax),stat=err)
        if (err .ne. 0) call xermsg('eval_Ylm_on_lebedev_grid','eval_Ylm_on_lebedev_grid','Memory allocation 1 failed.',err,1)
        tmp1 = 0.0_idp
        tmp2 = 0.0_idp
        tmp3 = 0.0_idp
        tmp4  = 0.0_idp

        !Get the rule for the given n or the nearest higher available n.
        do i=n,rule_max
           available = available_table(i)
           if (available == 1) then
              n_points = order_table(i)
              call ld_by_order(n_points,tmp1,tmp2,tmp3,tmp4)
              exit
           endif
        enddo !i

        print *,'got order vs input',n,i
        n = i
        print *,'lebedev points',n_points
        if (n_points > max_grid_points) stop "n_points > max_grid_points: increase the parameter max_grid_points and recompile"

        !Array of Lebedev points and weights
        angular_grid(1,1:n_points) = tmp1(1:n_points)
        angular_grid(2,1:n_points) = tmp2(1:n_points)
        angular_grid(3,1:n_points) = tmp3(1:n_points)
        angular_grid(4,1:n_points) = tmp4(1:n_points)

  end subroutine eval_lebedev_grid

  subroutine eval_Xlm_on_grid(max_l,angular_grid,n_points,Xlm_on_grid)
     use phys_const_gbl, only: fourpi, pi
     use angmom_procs, only: re_sp_harm
     use lebedev_gbl, only: xyz_to_tp
     implicit none
     integer, intent(in) :: max_l
     integer, intent(in) :: n_points
     real(kind=idp) :: angular_grid(4,max_grid_points)
     complex(kind=idp), allocatable :: Xlm_on_grid(:,:)

     integer :: point, l, m, i, err, n_rule, available, lm
     real(kind=cfp) :: x,y,z,t,p
     real(kind=idp) :: theta, phi
     real(kind=idp), parameter :: deg2rad = pi/180.0_idp

        write(6,'("Evaluating real spherical harmonics on the grid...")')

        if (max_l < 0 .or. n_points .le. 0) stop "error in max_l or n_points"

        if (allocated(Xlm_on_grid)) deallocate(Xlm_on_grid)
        allocate(Xlm_on_grid((max_l+1)**2,n_points),stat=err)
        if (err .ne. 0) call xermsg('eval_Ylm_on_lebedev_grid','eval_Ylm_on_lebedev_grid','Memory allocation 3 failed.',err,1)

        !$OMP PARALLEL DEFAULT(NONE) PRIVATE(point,theta,phi,t,p,x,y,z,lm,l,m) SHARED(n_points,angular_grid,max_l,Xlm_on_grid)
        !$OMP DO
        do point=1,n_points
           x = angular_grid(1,point); y = angular_grid(2,point); z = angular_grid(3,point)
           !WARNING: in xyz_to_tp routine the meaning of theta and phi is swapped (mathematical convention), i.e. t is the azimuth and p is the polar angle, see below:
           call xyz_to_tp(x,y,z,t,p)
           theta = p*deg2rad; phi = t*deg2rad
           lm = 0
           do l=0,max_l
              do m=-l,l
                 lm = lm + 1
                 !todo try using my cfp_resh !!!
                 Xlm_on_grid(lm,point) = re_sp_harm(l,m,theta,phi)
              enddo !m
           enddo !l
        enddo !point
        !$OMP END DO
        !$OMP END PARALLEL

        write(6,'("...done")')

  end subroutine eval_Xlm_on_grid

  !> Constructs the molecular frame dipoles in the momentum basis for the set of directions given by the grid points.
  subroutine eval_dipoles_on_grid(dipoles,max_l,n_final_states,n_points,Xlm_on_grid,dipoles_on_grid,n_total_initial_states)
     implicit none
     integer, intent(in) :: n_points, max_l, n_final_states, n_total_initial_states
     complex(kind=idp), intent(in) :: Xlm_on_grid((max_l+1)**2,n_points), dipoles(:,:,:,:,:) !mol frame photodipoles: lm,q,E,final_state,initial_state
     complex(kind=idp), allocatable :: dipoles_on_grid(:,:,:,:,:) !component,point,E,final_state,initial_state

     integer :: n_lm, final_state, energy, point, lm, component, initial_state

        write(6,'("Evaluating the dipoles on the grid...")')

        if (allocated(dipoles_on_grid)) deallocate(dipoles_on_grid)
        allocate(dipoles_on_grid(3,n_points,size(dipoles,3),n_final_states,n_total_initial_states))

        n_lm = (max_l+1)**2

        !$OMP PARALLEL DEFAULT(NONE) PRIVATE(final_state,energy,point,lm,component,initial_state) &
        !$OMP SHARED(n_final_states,dipoles,n_points,dipoles_on_grid,n_lm,Xlm_on_grid,n_total_initial_states)
        do initial_state=1,n_total_initial_states
           do final_state=1,n_final_states
              !$OMP DO
              do energy=1,size(dipoles,3)
                 do point=1,n_points
                    dipoles_on_grid(1:3,point,energy,final_state,initial_state) = 0.0_idp
                    do component=1,3
                       dipoles_on_grid(component,point,energy,final_state,initial_state) = &
                         sum(dipoles(1:n_lm,component,energy,final_state,initial_state)*Xlm_on_grid(1:n_lm,point))
                    enddo !component
                 enddo !point
              enddo !energy
              !$OMP END DO
           enddo !final_state
        enddo !initial_state
        !$OMP END PARALLEL

        write(6,'("...done")')

  end subroutine eval_dipoles_on_grid

  !> Transforms the molecular frame momentum dipoles for the x,y,z components into dipoles for parallel light given by the directions in angular_points.
  subroutine transform_xyz_dipoles_to_parallel_light_dipoles(dipoles_on_grid,phase_bound,angular_grid,&
                                                             dipoles_on_grid_parallel_light,dipoles_phase)
     implicit none
     integer, intent(in) :: phase_bound(max_selected_states)
     real(kind=idp) :: angular_grid(4,max_grid_points)
     complex(kind=idp), allocatable :: dipoles_on_grid(:,:,:,:,:) !component,point,E,final_state,initial_state
     complex(kind=idp), allocatable :: dipoles_on_grid_parallel_light(:,:,:,:,:) !component,point,E,final_state,initial_state
     real(kind=idp), allocatable :: dipoles_phase(:,:,:,:) !point,E,final_state,initial_state

     integer :: final_state, energy, point, initial_state, xyz
     complex(kind=idp) :: dipole

        write(6,'("Evaluating the parallel-light dipoles on the grid...")')

        if (.not.(allocated(dipoles_on_grid))) stop "dipoles_on_grid must be computed first"

        if (allocated(dipoles_on_grid_parallel_light)) deallocate(dipoles_on_grid_parallel_light)
        allocate(dipoles_on_grid_parallel_light,source=dipoles_on_grid)
        dipoles_on_grid_parallel_light = 0.0_idp

        if (allocated(dipoles_phase)) deallocate(dipoles_phase)
        allocate(dipoles_phase(size(dipoles_on_grid,2),size(dipoles_on_grid,3),size(dipoles_on_grid,4),size(dipoles_on_grid,5)))
        dipoles_phase = 0.0_idp

        !$OMP PARALLEL DEFAULT(NONE) PRIVATE(final_state,energy,point,xyz,initial_state,dipole) &
        !$OMP SHARED(dipoles_on_grid,angular_grid,dipoles_on_grid_parallel_light,dipoles_phase,phase_bound)
        do initial_state=1,size(dipoles_on_grid,5)
           do final_state=1,size(dipoles_on_grid,4)
              !$OMP DO
              do energy=1,size(dipoles_on_grid,3)
                 do point=1,size(dipoles_on_grid,2)
                    dipole = 0.0_idp
                    do xyz=1,3
                       dipole = dipole&
                              + angular_grid(xyz,point)*dipoles_on_grid(component_map(xyz),point,energy,final_state,initial_state)
                    enddo !xyz
                    dipole = dipole*phase_bound(initial_state) !phase correction for the bound state
                    !We save the parallel light dipole into the first component:
                    dipoles_on_grid_parallel_light(1,point,energy,final_state,initial_state) = dipole
                    dipoles_phase(point,energy,final_state,initial_state) = atan2(aimag(dipole),real(dipole))
                 enddo !point
              enddo !energy
              !$OMP END DO
           enddo !final_state
        enddo !initial_state
        !$OMP END PARALLEL

        write(6,'("...done")')

  end subroutine transform_xyz_dipoles_to_parallel_light_dipoles

  subroutine compute_dcs(dipoles_on_grid,escat,ip,photo_dcs)
     use dipelmdefs
     implicit none
     complex(kind=idp), intent(in) :: dipoles_on_grid(:,:,:,:,:)
     real(kind=idp), intent(in) :: escat(:), ip
     real(kind=idp), allocatable :: photo_dcs(:,:,:,:)

     integer :: initial_state,final_state,point,energy,xyz,n_initial_states,n_final_states,n_points,no_scattering_energies
     real(kind=idp) :: photon_energy,Aconst

        write(6,'("Computing DCS from the dipoles evaluated on the grid...")')

        n_initial_states = size(dipoles_on_grid,5)
        n_final_states = size(dipoles_on_grid,4)
        no_scattering_energies = size(dipoles_on_grid,3)
        n_points = size(dipoles_on_grid,2)

        if (allocated(photo_dcs)) deallocate(photo_dcs)
        allocate(photo_dcs(no_scattering_energies,n_points,n_final_states,n_initial_states))
        photo_dcs = 0.0_idp

        !$OMP PARALLEL DEFAULT(NONE) PRIVATE(final_state,energy,point,xyz,initial_state,photon_energy,Aconst) &
        !$OMP SHARED(dipoles_on_grid,escat,photo_dcs,n_initial_states,n_final_states,n_points,no_scattering_energies,ip)
        do initial_state=1,n_initial_states
           do final_state=1,n_final_states
             !$OMP DO
              do point=1,n_points
                 do energy=1,no_scattering_energies

                    photon_energy=escat(energy)+ip !0.50633934805
                    Aconst=4*(pi**2)*alpha*photon_energy*convert_au_to_megabarns

                    !photoionization/recombination cross section:
                    do xyz=1,3 !here we don't need to use component_map(xyz) since we're just summing all components in the same way
                       photo_dcs(energy,point,final_state,initial_state) = photo_dcs(energy,point,final_state,initial_state) &
                        + real(dipoles_on_grid(xyz,point,energy,final_state,initial_state))**2 &
                        + aimag(dipoles_on_grid(xyz,point,energy,final_state,initial_state))**2
                    enddo
                    photo_dcs(energy,point,final_state,initial_state) = Aconst * photo_dcs(energy,point,final_state,initial_state) 

                 enddo
              enddo
              !$OMP END DO
           enddo
        enddo
        !$OMP END PARALLEL

        write(6,'("...done")')

  end subroutine compute_dcs

  subroutine write_data_matlab(phase_bound,dipoles,max_l,n_final_states,escat,n_energies,angular_grid,n_points,sign_at_r,atom_xyz,&
    atom_name,n_atoms,initial_state_energy,final_state_energy,final_state_num,final_state_sym,final_state_spin,orbital_num,&
    orbital_sym,initial_state_index_map,n_initial_states,initial_state_num,initial_state_sym,initial_state_spin,n,molecule_name,&
    final_state_dipole_table,initial_state_dipole_table)
     implicit none
     integer, intent(in) :: n_atoms
     integer, intent(in) :: phase_bound(max_selected_states), max_l, n_final_states, n_energies, n_points, &
                            n_initial_states,n,sign_at_r(n_points,n_final_states,n)
     integer, intent(in) :: final_state_num(max_selected_states),final_state_sym(max_selected_states),&
                            final_state_spin(max_selected_states)
     integer, intent(in) :: initial_state_num(max_selected_states),initial_state_sym(max_selected_states),&
                            initial_state_spin(max_selected_states),initial_state_index_map(max_selected_states)
     integer, intent(in) :: orbital_num(max_selected_states,max_selected_states),&
                            orbital_sym(max_selected_states,max_selected_states)
     complex(kind=idp), intent(in) :: dipoles(:,:,:,:,:) !mol frame photodipoles: lm,q,E,final_state,initial_state
     real(kind=idp), intent(in) :: escat(n_energies), angular_grid(4,max_grid_points),initial_state_energy(n_initial_states),&
                                   final_state_energy(n_final_states)
     real(kind=idp) :: atom_xyz(3,n_atoms), final_state_dipole_table(3,n_final_states,n_final_states),&
                       initial_state_dipole_table(3,n_initial_states,n_initial_states)
     character(len=nuc_nam_len) :: atom_name(n_atoms)
     character(len=80), intent(in) :: molecule_name

     integer :: n_lm, final_state, energy, lm, lu, xyz, i, j, k, initial_state

        !Note that the bound state indices in the arrays dipoles, orbital_*, sign_at_r are mapped via the array initial_state_index_map.

        n_lm = (max_l+1)**2
        !n_energies = size(dipoles,3)

        write(6,'("Writing partial wave dipoles and Dyson orbital signs in the Matlab format...")')

        lu = 100
        open(file='dipoles_data_matlab',unit=lu,form='FORMATTED',status='replace')

        write(lu,'(a80)') molecule_name
        write(lu,'(7i10)') max_l, n_lm, n_final_states, n_energies, n_points, n_atoms, n_initial_states

        do i=1,n_initial_states
           write(lu,'(3i10)') initial_state_num(i), initial_state_sym(i), initial_state_spin(i)
           write(lu,'(e25.15)') initial_state_energy(i)
        enddo

        do i=1,n_initial_states
           if (phase_bound(i) .ne. 1) write(*,'("Applying phase-correction ",i2," to the initial state dipoles.")') phase_bound(i)
           do j=1,n_initial_states
              write(lu,'(3e25.15)') initial_state_dipole_table(1:3,j,i)*phase_bound(i)*phase_bound(j)
           enddo
        enddo 

        do i=1,n_final_states
           write(lu,'(3i10)') final_state_num(i), final_state_sym(i), final_state_spin(i)
           write(lu,'(e25.15)') final_state_energy(i)
        enddo

        do i=1,n_final_states
           do j=1,n_final_states
              write(lu,'(3e25.15)') final_state_dipole_table(1:3,j,i)
           enddo
        enddo 

        do i=1,n_atoms
           write(lu,'(3e25.15)') atom_xyz(1:3,i)
        enddo

        do i=1,n_atoms
           write(lu,'(a2)') atom_name(i)
        enddo

        do i=1,n_points
           write(lu,'(4e25.15)') angular_grid(1:4,i)
        enddo

        do i=1,n_energies
           write(lu,'(e25.15)') escat(i)
        enddo

        do k=1,n_initial_states
           if (phase_bound(k) .ne. 1) write(*,'("Applying phase-correction ",i2," to the Dyson orbitals.")') phase_bound(k)
           do i=1,n_final_states
              write(lu,'(2i10)') orbital_num(i,initial_state_index_map(k)),orbital_sym(i,initial_state_index_map(k))
              do j=1,n_points
                 write(lu,'(i2)') sign_at_r(j,i,initial_state_index_map(k))*phase_bound(k)
              enddo
           enddo
        enddo

        !Real part
        do initial_state=1,n_initial_states
           if (phase_bound(initial_state) .ne. 1) write(*,'("Applying phase-correction ",i2," to the photodipoles.")') &
                                                             phase_bound(initial_state)
           do final_state=1,n_final_states
              do energy=1,n_energies
                 do xyz=1,3
                    !Dipole components saved in the x,y,z order
                    do lm=1,n_lm
                       write(lu,'(e25.15)') real(&
                                         dipoles(lm,component_map(xyz),energy,final_state,initial_state_index_map(initial_state))&
                                         *phase_bound(initial_state))
                     enddo !lm
                 enddo !component
              enddo !energy
           enddo !final_state
        enddo !initial_state

        !Imaginary part
        do initial_state=1,n_initial_states
           if (phase_bound(initial_state) .ne. 1) write(*,'("Applying phase-correction ",i2," to the photodipoles.")') &
                                                             phase_bound(initial_state)
           do final_state=1,n_final_states
              do energy=1,n_energies
                 do xyz=1,3
                    !Dipole components saved in the x,y,z order
                    do lm=1,n_lm
                       write(lu,'(e25.15)') aimag(&
                                         dipoles(lm,component_map(xyz),energy,final_state,initial_state_index_map(initial_state))&
                                         *phase_bound(initial_state))
                     enddo !lm
                 enddo !component
              enddo !energy
           enddo !final_state
        enddo !initial_state

        close(lu)

        write(6,'("...done")')

  end subroutine write_data_matlab

  subroutine write_dipoles_on_grid(phase_bound,dipoles_on_grid,escat,first_energy_ev,n_final_states,n_initial_states,&
                                   initial_state_index_map,molecule_name,angular_grid,n_points,format_flag,prefix)
     implicit none
     integer, intent(in) :: phase_bound(max_selected_states),n_final_states, n_points, n_initial_states, &
                            initial_state_index_map(max_selected_states), format_flag
     complex(kind=idp), intent(in) :: dipoles_on_grid(:,:,:,:,:)
     real(kind=idp), intent(in) :: escat(:), first_energy_ev
     character(len=80), intent(in) :: molecule_name
     character(len=*), intent(in) :: prefix
     real(kind=idp), intent(in) :: angular_grid(4,max_grid_points)

     integer :: no_scattering_energies, final_state, energy, point, component, luR, luI, initial_state
     character(len=5) :: str_ichild, str_no_chans, str_parent
     character(len=1) :: comp_str
     character(len=80) :: pathR, pathI, frm
     real(kind=idp) :: en_ev,real_dip,imag_dip
     real(kind=idp), allocatable :: tmp(:,:), diptmp(:,:)
     logical :: all_zero

        write(6,'("Writing the dipoles evaluated on the grid...")')

        no_scattering_energies = size(escat)
        if (no_scattering_energies .ne. size(dipoles_on_grid,3)) then
           stop "escat and dipoles_on_grid have a different number of energies"
        endif

        frm = 'formatted'
        if (format_flag .eq. 3) frm = 'unformatted'

        write(6,'("Number of points, energies: ",2i10)') n_points,no_scattering_energies

        write(str_no_chans,'(i5)') n_final_states

        !$OMP PARALLEL DEFAULT(NONE) PRIVATE(final_state,str_ichild,energy,point,en_ev,tmp,real_dip,imag_dip,comp_str,component,&
        !$OMP & luR,luI,pathR,pathI,initial_state,str_parent,diptmp,all_zero) &
        !$OMP & SHARED(n_final_states,no_scattering_energies,n_points,dipoles_on_grid,molecule_name,str_no_chans,escat,&
        !$OMP & first_energy_ev,phase_bound,n_initial_states,initial_state_index_map,prefix,format_flag,frm)
        allocate(tmp(no_scattering_energies,2))
        if (format_flag .eq. 3) allocate(diptmp(n_points,no_scattering_energies))

        do initial_state=1,n_initial_states

           if (phase_bound(initial_state) .ne. 1) write(*,'("Applying phase-correction ",i2," to the bound state.")')&
                                                             phase_bound(initial_state)

           write(str_parent,'(i5)') initial_state

           !$OMP DO
           do final_state=1,n_final_states
   
              write(str_ichild,'(i5)') final_state
   
              do component=1,3

                 all_zero = .true.
                 do point=1,n_points
                    do energy=1,no_scattering_energies
                       en_ev = escat(energy)*to_ev
                       if (en_eV .ge. first_energy_ev) then
                          real_dip = real(&
                                     dipoles_on_grid(component,point,energy,final_state,initial_state_index_map(initial_state)))
                          imag_dip = aimag(&
                                     dipoles_on_grid(component,point,energy,final_state,initial_state_index_map(initial_state)))
                          if ((real_dip .ne. 0.0_idp) .or. (imag_dip .ne. 0.0_idp)) all_zero = .false.
                       else
                          real_dip = 0
                          imag_dip = 0
                       endif
                    enddo !energy
                 enddo !point

                 if (all_zero) then
                    cycle !save only the components which are non-zero
                 endif
   
                 !Order taken from dipelmprocs: dcomp2i
                 select case(component)
                    case (1)
                       comp_str = 'y'
                    case (2)
                       comp_str = 'z'
                    case (3)
                       comp_str = 'x'
                 end select
   
                 pathR = trim(adjustl(prefix))//comp_str//'_'//trim(adjustl(str_ichild))//'_'//trim(adjustl(str_parent))//'_R.txt'
                 pathI = trim(adjustl(prefix))//comp_str//'_'//trim(adjustl(str_ichild))//'_'//trim(adjustl(str_parent))//'_I.txt'

                 luR = 100+final_state
                 open(unit=luR, file=trim(adjustl(pathR)),form=frm, access='sequential')
                 luI = 200+final_state
                 open(unit=luI, file=trim(adjustl(pathI)),form=frm, access='sequential')

                 if (format_flag .eq. 3) then

                    diptmp(:,:) = real(dipoles_on_grid(component,:,:,final_state,initial_state_index_map(initial_state)))   
                    write(luR) diptmp
                    diptmp(:,:) = aimag(dipoles_on_grid(component,:,:,final_state,initial_state_index_map(initial_state)))   
                    write(luI) diptmp
                
                 else
                 
                    do point=1,n_points
                       do energy=1,no_scattering_energies
                          en_ev = escat(energy)*to_ev
                          if (en_eV .ge. first_energy_ev) then
                             real_dip = &
                              real(dipoles_on_grid(component,point,energy,final_state,initial_state_index_map(initial_state)))
                             imag_dip = &
                              aimag(dipoles_on_grid(component,point,energy,final_state,initial_state_index_map(initial_state)))
                          else
                             real_dip = 0
                             imag_dip = 0
                          endif
                          tmp(energy,1) = real_dip
                          tmp(energy,2) = imag_dip
                       enddo !energy
                       !write to disk:
                       select case (format_flag)
                       case (1) !python
                          write(luR,'(10000e25.15)') tmp(1:no_scattering_energies,1)*phase_bound(initial_state)
                          write(luI,'(10000e25.15)') tmp(1:no_scattering_energies,2)*phase_bound(initial_state)
                       case (2) !gnuplot: for use with 'splot'
                          do energy=1,no_scattering_energies
                             en_ev = escat(energy)*to_ev
                             write(luR,'(e25.15,i10,e25.15)') en_ev,point,tmp(energy,1)*phase_bound(initial_state)
                             write(luI,'(e25.15,i10,e25.15)') en_ev,point,tmp(energy,2)*phase_bound(initial_state)
                          enddo !energy
                          write(luR,'("")') 
                          write(luI,'("")') 
                       case default
                          stop "on input format_flag was not one of: 1, 2"
                       end select
                    enddo !point

                 endif
   
              enddo !component
   
              close(luR)
              close(luI)
           enddo
           !$OMP END DO
        enddo
        !$OMP END PARALLEL

        write(6,'("...done")')

  end subroutine write_dipoles_on_grid

  subroutine write_dcs(photo_dcs,escat,ip,initial_state_index_map,molecule_name,format_flag)
     implicit none
     real(kind=idp), intent(in) :: escat(:), ip
     real(kind=idp), allocatable :: photo_dcs(:,:,:,:)
     integer, intent(in) :: initial_state_index_map(max_selected_states), format_flag
     character(len=80), intent(in) :: molecule_name

     integer :: initial_state,n_initial_states,final_state,n_final_states,point,n_points,energy,no_scattering_energies,lu
     real(kind=idp) :: photon_energy
     character(len=5) :: str_ichild, str_parent
     character(len=80) :: path, frm

        write(6,'("Writing the DCS evaluated on the grid...")')

        n_initial_states = size(photo_dcs,4)
        n_final_states = size(photo_dcs,3)
        n_points = size(photo_dcs,2)
        no_scattering_energies = size(photo_dcs,1)

        frm = 'formatted'
        if (format_flag .eq. 3) frm = 'unformatted'

        write(6,'("Number of energies, points: ",2i10)') no_scattering_energies,n_points

        do initial_state=1,n_initial_states

           write(str_parent,'(i5)') initial_state

           do final_state=1,n_final_states

              write(str_ichild,'(i5)') final_state
   
              path = 'photo_dcs'//'_'//trim(adjustl(str_ichild))//'_'//trim(adjustl(str_parent))

              lu = 100+final_state
              open(unit=lu, file=trim(adjustl(path)),form=frm, access='sequential')

              if (format_flag .eq. 3) then

                 write(lu) photo_dcs(:,:,final_state,initial_state_index_map(initial_state))

              else

                 do point=1,n_points
   
                    select case (format_flag)
                    case (1) !python
                       write(lu,'(10000e25.15)') &
                        photo_dcs(1:no_scattering_energies,point,final_state,initial_state_index_map(initial_state))
                    case (2) !gnuplot: for use with 'splot'
                       do energy=1,no_scattering_energies
                          photon_energy=27.211*(escat(energy)+ip) !0.50633934805)
                          write(lu,'(e25.15,i10,e25.15)') &
                           photon_energy,point,photo_dcs(energy,point,final_state,initial_state_index_map(initial_state))
                       enddo
                       write(lu,'("")') 
                    case default
                       stop "on input format_flag was not one of: 1, 2"
                    end select
   
                 enddo !point

              endif

              close(lu)

           enddo !final_state
        enddo !initial_state

        write(6,'("...done")')

  end subroutine write_dcs

  subroutine write_dipoles_phase(dipoles_phase,escat,ip,initial_state_index_map,molecule_name,format_flag)
     implicit none
     real(kind=idp), intent(in) :: escat(:), ip
     real(kind=idp), allocatable :: dipoles_phase(:,:,:,:) !point,E,final_state,initial_state
     integer, intent(in) :: initial_state_index_map(max_selected_states), format_flag
     character(len=80), intent(in) :: molecule_name

     integer :: initial_state,n_initial_states,final_state,n_final_states,point,n_points,energy,no_scattering_energies,lu
     real(kind=idp) :: photon_energy
     character(len=5) :: str_ichild, str_parent
     character(len=80) :: path, frm

        write(6,'("Writing the dipole phases evaluated on the grid...")')

        n_initial_states = size(dipoles_phase,4)
        n_final_states = size(dipoles_phase,3)
        no_scattering_energies = size(dipoles_phase,2)
        n_points = size(dipoles_phase,1)

        frm = 'formatted'
        if (format_flag .eq. 3) frm = 'unformatted'

        write(6,'("Number of points, energies: ",2i10)') n_points,no_scattering_energies

        do initial_state=1,n_initial_states

           write(str_parent,'(i5)') initial_state

           do final_state=1,n_final_states

              write(str_ichild,'(i5)') final_state
   
              path = 'dipoles_phase'//'_'//trim(adjustl(str_ichild))//'_'//trim(adjustl(str_parent))

              lu = 100+final_state
              open(unit=lu, file=trim(adjustl(path)),form=frm, access='sequential')

              if (format_flag .eq. 3) then

                 write(lu) dipoles_phase(:,:,final_state,initial_state_index_map(initial_state))

              else

                 do point=1,n_points
   
                    select case (format_flag)
                    case (1) !python
                       write(lu,'(10000e25.15)') &
                        dipoles_phase(point,1:no_scattering_energies,final_state,initial_state_index_map(initial_state))
                    case (2) !gnuplot: for use with 'splot'
                       do energy=1,no_scattering_energies
                          photon_energy=27.211*(escat(energy)+ip) !0.50633934805)
                          write(lu,'(e25.15,i10,e25.15)') &
                           photon_energy,point,dipoles_phase(point,energy,final_state,initial_state_index_map(initial_state))
                       enddo
                       write(lu,'("")') 
                    case default
                       stop "on input format_flag was not one of: 1, 2"
                    end select
   
                 enddo !point

              endif

              close(lu)

           enddo !final_state
        enddo !initial_state

        write(6,'("...done")')

  end subroutine write_dipoles_phase

end program dipoles_hhg
