! Copyright 2019
!
! Alex G. Harvey with contributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!-----------------------------------------------------------------------
!> @brief Transition moment container types
!> @author Alex Harvey
!> @date  2019
!>
!> Types for holding transition moments and associated 
!> information.
!
!----------------------------------------------------------------------- 
module dipelm_types

   use dipelm_defs, only: idp
   use sorting, only: qsort
   use dipelm_special_functions, only: lm2i, i2lm
   
   implicit none


   !integer, parameter :: idp = kind( 1.d0 )
   real(idp), parameter :: e_thresh = 1.d-6 !TODO: Make private or move to dipelm_defs?
!
!>  @brief Molecule details
!
!>  Hold details of the molecule (might be desirable in the future
!>  to put geometry etc in here)
!
   type molecule
   
      character(len=3) :: point_group     !< Molecular point group.
      real(idp)        :: ion_charge      !< Ion charge (atomic units).
      
   end type
!
!> @brief Holds set of bound states.
!
!> Holds arrays describing bound state quantum numbers and energies.
!> Array size is equal to the number of bound states.
!> 
!> Note: naming conventions are a holdover from the diatomic version
!> of the UKRmol codes where the z-component of the total angular 
!> momentum was a good quantum number.
!   
   type bound

      integer,   allocatable :: stot(:)      !< Spin of each state.
      integer,   allocatable :: mtot(:)      !< Symmetry of each state.
      integer,   allocatable :: gutot(:)     !< Gerade/ungerade character of each state.
      real(idp), allocatable :: energies(:)  !< Bound state energy of each state (Hartree).
   
   contains
   
      procedure, pass :: print => print_bound
           
   end type
   
   interface bound
   
       module procedure new_bound
          
   end interface bound
    
!> Check if two members of a type are equal.      
   interface operator(.eq.)
       module procedure eq_bound       
   end interface   
!
!> @brief Holds set of partial wave channels
!
!> Holds arrays describing partial wave scattering/photoionization channels.
!> A channel is specified by final ionic  state and l and m quantum number of the partial wave 
!> (expansion of the continuum electron in an angular momentum basis).
!>
!> Note: Only symmetry allowed channels are stored @ref occupancy. 
!
   type channels

      type(bound)            :: ion_states            !< The ionic state quantum numbers and energies
      integer,   allocatable :: ichl(:)               !< The ionic state index   
      integer,   allocatable :: lchl(:)               !< l character of partial waves
      integer,   allocatable :: mchl(:)               !< m character of partial waves
      real(idp), allocatable :: threshold_energies(:) !< The channel threshold energies relative to the ionic ground state (Hartree).
   
   contains
   
      procedure, pass :: print     => print_channels
      procedure, pass :: occupancy => channels_occupancy   !< Maps each ichl(i), lchl(i), mchl(i) to a single canonical channel index 
                                                           !< and returns an array that indicates which canonical channels are occupied.
   end type channels
   
   interface channels
       module procedure new_channels          
   end interface channels
 
   interface operator(.eq.)
       module procedure eq_channels  
   end interface 
   
!> Check for equality of photoionization model between two members of a type     
   interface operator(.samemodel.)
       module procedure channels_from_same_model       
   end interface

!> Add two members of a type      
   interface operator(+)
       module procedure add_channels       
   end interface
!    
!>  @brief Holds set of continuum states.
!>
!>  A continuum state with a particular energy is decomposed into one or more symmetry components 
!>  labelled by the three arrays stot, mtot and gutot. 
!>  Each components is further decomposed into partial wave channels.
!>  Currently
!
!  ni = no of ions
!  nc = no of partial wave channels
!  ne = no of energies
!  nmc = no moment components   
!  moment_components = the components of the moment operator that give non-zero transition moments
!  for this component of the continuum states.
!  energies = the energy of the continuum electron.
!  
   type continuum

      integer, allocatable   :: stot(:)              !< Spin of each component
      integer, allocatable   :: mtot(:)              !< Symmetry of each component 
      integer, allocatable   :: gutot(:)             !< Gerade/ungerade character of each component.     
      type(channels)         :: channels             !< Specifies the photoionizations channels
      
      integer, allocatable   :: moment_components(:) !< The moment components that couple intial bound state 
                                                     !< of the neutral with each component present in the continuum states 
                                                     !< (size = 3 for dipoles, 5 for quadrupoles, 8 for both).
                                                     !< @todo Perhaps this should be moved into the moments type
                                                     
      real(idp), allocatable :: energies(:)          !< Energy of each continuum state (Hartree)
      
   contains
   
      procedure, pass :: print => print_continuum          
        
   end type
   
   interface continuum
       module procedure new_continuum   
   end interface continuum
      
   interface operator(.samemodel.)
       module procedure continuum_from_same_model       
   end interface
   
   interface operator(+)
       module procedure continuum_add       
   end interface
!
!> @brief Transition moments container type.    
!
!>  Holds a description of the states and operators involved in the
!>  transition moments and the transition moment matrix elements themselves.
!>
!>  Note: we call the pre-ionized molecule the neutral regardless of 
!>  its charge.
! 
!  elements= the transition moment matrix elements where the partial
!  waves are given a canonical index. (see function lm2i(l,m) )
!
!  npw=maximum partial wave index
!  ncc= number of continuum state components required to describe the
!  moments. e.g. for dipoles at most 3 are required to describe the
!  3 components of the dipole operator (see xyz_irrep above)  
!
   type moments
     
      type(molecule)            :: mol                 !< Some details of the molecule.
      type(bound)               :: neutral_states      !< Neutral states present in calculation (it is assumed they are all of the same spin and symmetry)
      type(continuum)           :: scattering_states   !< Continuum states present in the calculation @todo Rename to continuum_states?
      complex(idp), allocatable :: elements(:,:,:,:,:) !< Transition matrix elements between the neutral and continuum states. The partial waves waves are given a canonical index (see @ref channels::occupancy()).
      character(len=3)          :: basis_type_photon   !< Basis type use to describe the transition moment operator ('Ylm' for complex spherical harmonics, 'Slm' for real spherical harmonics (equivalent to cartesian for dipoles))
      character(len=3)          :: basis_type_electron !< Basis type use to describe the continuum electron ('Ylm' for complex spherical harmonics, 'Slm' for real spherical harmonics)
      
   contains
   
      procedure, pass :: print => print_moments
      procedure, pass :: convert_moments
                    
   end type
     
   interface moments
       module procedure new_moments
   end interface moments
      
   interface operator(.samemodel.)
       module procedure moments_from_same_model       
   end interface
  
   interface operator(+)
       module procedure add_moments
   end interface

!> Used to apply tranformations to transition matrix elements (e.g. change of basis).    
   interface operator(*)
   
       module procedure right_matrix_multiply_moments
       module procedure left_matrix_multiply_moments
       
   end interface   
   
contains

!  -------------------------
!  Procedures for type BOUND
!  -------------------------
   subroutine print_bound(this,iwrite)
 
!     Arguments   
      class(bound) :: this
      integer     :: iwrite
      
      integer     :: i, no_bound_states
      
      no_bound_states =  size( this%stot )
      
      write(iwrite, *) ""      
      write(iwrite, '("   State  2S+1  I.Rep. g/u    Energies")')
      write(iwrite, '("  --------------------------------------")')
      
      do i = 1, no_bound_states
         write(iwrite, '(4i6,D20.8)') i, this%stot(i), this%mtot(i), this%gutot(i), this%energies(i)
      end do

   end subroutine
   
   function new_bound(nb)
   
      integer     :: nb
      type(bound) :: new_bound
      
      allocate( new_bound%stot(nb) )
      allocate( new_bound%mtot(nb) )
      allocate( new_bound%gutot(nb) )
      allocate( new_bound%energies(nb) )
      
      new_bound%stot  = 0
      new_bound%mtot  = 0
      new_bound%gutot = 0
      new_bound%energies = 0._idp
      
   end function new_bound
   
   function eq_bound(b1,b2)
   
      type(bound), intent(in) :: b1, b2
      logical     :: eq_bound
   
      integer :: no_b1, no_b2, i
      
      eq_bound = .true.
      
      no_b1 = size( b1%stot )
      no_b2 = size( b2%stot )
    
      if ( no_b1 .eq. no_b2 ) then
      
         do i = 1, no_b1
         
            if ( b1%stot(i)  .ne. b2%stot(i) )  eq_bound = .false.
            if ( b1%mtot(i)  .ne. b2%mtot(i) )  eq_bound = .false.
            if ( b1%gutot(i) .ne. b2%gutot(i) ) eq_bound = .false.
            
            if ( abs(b1%energies(i) - b2%energies(i)) .gt. e_thresh ) eq_bound = .false.
          
         end do
     
      else
         eq_bound = .false.
      end if
   
   end function
   
!  ----------------------------   
!  Procedures for type CHANNELS
!  ----------------------------
   subroutine print_channels(this, iwrite)
 
!     Arguments   
      class(channels) :: this
      integer     :: iwrite
      
      integer     :: i, no_channels, itarg
      
      no_channels =  size( this%ichl )
      
      write(iwrite, *) ""
      write(iwrite, '("  Channel    Targ. 2S+1  I.Rep. g/u    l     m     Channel Thresholds")')
      write(iwrite, '("  -------------------------------------------------------------------")')
      
      do i=1,no_channels
    
            itarg = this%ichl(i)
           
            write(iwrite, '(2i8,5i6,D20.8)') i, itarg, this%ion_states%stot(itarg),  &
            &                                         this%ion_states%mtot(itarg),  & 
            &                                         this%ion_states%gutot(itarg), &
            &                                         this%lchl(i), this%mchl(i),   &
            &                                         this%threshold_energies(i)
      end do 

   end subroutine
   
!> Assigns a single canonical index to a partial wave channel 
!> 
!> \f$ j = (i-1)N_\mathrm{p} + l(l+1)+(m+1) \f$
!>
!> where \f$ N_\mathrm{p} = (l_{\mathrm{max}} + 1)^2 \f$  and \f$ i \f$ is the ionic state index. 
!>
!> @param occ  = returned array where entries are 1 if the canonical channel is occupied and 0 if not.
!> Size = \f$  N_\mathrm{i} \times N_\mathrm{p} \f$, where  \f$  N_\mathrm{i} \f$ is the number of ionic states.
!> @param lmax = the maximum l value for partial wave expansion of the continuum (optional)
!> 
!> @see dipelm_special_functions::lm2i and dipelm_special_functions::i2lm
   subroutine channels_occupancy(this,occ,lmax)
      
!     Arguments   
      class(channels)      :: this   
      integer, allocatable :: occ(:)
      integer, optional    :: lmax
   
      integer :: l_max, N_p, ilm_max, no_states, nchan, i, ilm
      
      if ( present(lmax) ) then
         l_max = lmax
      else    
         l_max = maxval( this%lchl)
      end if
   
      N_p    = (l_max + 1)**2      
      no_states = maxval( this%ichl )
      ilm_max   = no_states * N_p      
      nchan     = size( this%ichl )

      allocate( occ(ilm_max) )
      occ = 0
      
      do i = 1, nchan
      
         ilm = ( this%ichl(i)-1 )*N_p  + lm2i( this%lchl(i), this%mchl(i) )
         occ(ilm) = 1
      
      end do
   
   
   end subroutine
         
   function new_channels(ni,nc)
   
      integer        :: ni, nc
      type(channels) :: new_channels
      
      new_channels%ion_states = bound(ni)
      allocate( new_channels%ichl(nc) )
      allocate( new_channels%lchl(nc) )
      allocate( new_channels%mchl(nc) )
      allocate( new_channels%threshold_energies(nc) )
      
      new_channels%ichl = 0
      new_channels%lchl = 0
      new_channels%mchl = 0
      
      new_channels%threshold_energies = 0._idp

   end function new_channels
   
   function eq_channels(ch1,ch2)
!     Arguments   
      type(channels), intent(in) :: ch1, ch2
      logical                    :: eq_channels
      
      integer :: no_ch1, no_ch2 , no_states, i
      
      eq_channels = .true.
      
      if (ch1%ion_states .eq. ch2%ion_states) then
      
         no_ch1    = size( ch1%ichl )
         no_ch2    = size( ch2%ichl )
         no_states = size( ch1%ion_states%mtot )
      
         if ( no_ch1 .eq. no_ch1 ) then 
           
            do i = 1, no_ch1
                  
                if ( ch1%ichl(i) .ne. ch2%ichl(i)  ) eq_channels = .false.
                if ( ch1%lchl(i) .ne. ch2%lchl(i)  ) eq_channels = .false.
                if ( ch1%mchl(i) .ne. ch2%mchl(i)  ) eq_channels = .false.
                  
                if ( abs(ch1%threshold_energies(i) - ch2%threshold_energies(i)) .gt. e_thresh  ) eq_channels = .false.
                  
            end do 
         else
         
            eq_channels = .false.
         
         end if
         
      else
      
         eq_channels = .false.
         
      end if
      
   end function
   
   function channels_from_same_model(ch1,ch2)
!     Arguments   
      type(channels), intent(in) :: ch1, ch2
      logical                    :: channels_from_same_model
      
      channels_from_same_model = .false.
   
      if (ch1%ion_states .eq. ch2%ion_states) channels_from_same_model = .true.
      
   end function
   
   function add_channels(ch1,ch2)
      
!     Arguments   
      type(channels), intent(in) :: ch1, ch2
      type(channels)             :: add_channels 

!     Local      
      integer :: no_states, nchan, i, lmax, lmax1, lmax2, nocc ,ilm, l, m, istate, ichan
      integer, allocatable :: occ1(:), occ2(:), occ(:)
      real(idp) :: ion_gs_energy
        
      if (ch1 .samemodel. ch2) then
  
         lmax1 = maxval( ch1%lchl )
         lmax2 = maxval( ch2%lchl )
         lmax  = max( lmax1, lmax2 )
         
         call ch1%occupancy(occ1,lmax)
         call ch2%occupancy(occ2,lmax)
          
         nocc= size( occ1 ) 
         allocate( occ(nocc))
         occ=0
         
         occ = occ1 + occ2
      
         no_states = size( ch1%ion_states%mtot )
         
         nchan = count(occ .ne. 0) 
           
         add_channels            = channels( no_states, nchan )         
         add_channels%ion_states = ch1%ion_states
         
         ion_gs_energy = add_channels%ion_states%energies(1)
        
         ichan = 1
         do i = 1, nocc
         
            if ( occ(i) .ne. 0 ) then
            
               istate = (i-1)/(lmax + 1)**2 + 1            
               ilm    = mod(i-1, (lmax + 1)**2) +1
               
               
               call i2lm(ilm, l, m)
       
               add_channels%ichl(ichan) = istate
               add_channels%lchl(ichan) = l
               add_channels%mchl(ichan) = m
               
               add_channels%threshold_energies(ichan) = add_channels%ion_states%energies(istate) - ion_gs_energy
               
               ichan = ichan + 1

            end if
         
         end do
         
      else
      
         ERROR STOP "ERROR: Channel models do not match!" 
                    
      end if
 
   end function
!  -----------------------------   
!  Procedures for type CONTINUUM
!  -----------------------------
      
   function new_continuum(nsym, ni,nc,ne,nmc)   

      integer           :: nsym
      integer, optional :: ni, nc, ne, nmc 
      type(continuum) :: new_continuum
      
      allocate( new_continuum%stot(nsym)  )
      allocate( new_continuum%mtot(nsym)  )
      allocate( new_continuum%gutot(nsym) )
      
      new_continuum%stot  = 0
      new_continuum%mtot  = 0
      new_continuum%gutot = 0
      
      if (present(ni).and. present(nc)) then
         new_continuum%channels = channels(ni,nc)
      end if
      
      if ( present(nmc) ) then
         allocate( new_continuum%moment_components(nmc) )
         new_continuum%moment_components = 0
      end if
      
      if ( present(ne) ) then
         allocate( new_continuum%energies(ne) )
         new_continuum%energies = 0._idp
      end if
   
   end function new_continuum
   
   function continuum_from_same_model(c1,c2)
!     Arguments   
      type(continuum), intent(in) :: c1, c2
      logical                     :: continuum_from_same_model
      
      integer :: i, no_energies_c1, no_energies_c2
      
      continuum_from_same_model = .false.
      
      no_energies_c1 = size( c1%energies )
      no_energies_c2 = size( c2%energies )
      
      if (c1%channels .samemodel. c1%channels) then
      
         continuum_from_same_model = .true.
      
         if ( no_energies_c1 .eq. no_energies_c2 ) then
      
             do i = 1, no_energies_c1
             
                if (abs(c1%energies(i) - c1%energies(i)) .gt. e_thresh ) continuum_from_same_model = .false.
             
             end do
      
         else
       
            continuum_from_same_model = .false.
         
         end if
      end if
      
   end function
   
   subroutine print_continuum(this,iwrite, iprint)
 
!     Arguments   
      class(continuum) :: this
      integer     :: iwrite, iprint
      
      character(len=5), parameter :: multipole_labels(8) = (/&     
      &                            "  Y  ","  Z  ","  X  "," XY  "," YZ  "," ZZ  "," XZ  ","XX-YY"/)      
      integer     :: i, no_mom_comp, no_energies, no_sym
      
      no_sym      =  size( this%stot )
      no_mom_comp =  size( this%moment_components )
      no_energies =  size( this%energies )
      
      write(iwrite, *) ""
      write(iwrite, '("  Scattering symmetry:   ")')      
      write(iwrite, '("   2S+1  I.Rep. g/u   ")')
      do i = 1, no_sym
         write(iwrite, '(3i6)') this%stot(i), this%mtot(i), this%gutot(i)
      end do
      
      write(iwrite, '("  Corresponding moment components:   ")')       
      write(iwrite, '( 3x, 8a5 )') ( multipole_labels(i), i = 1,no_mom_comp )     
      write(iwrite, '( 3x, 8(2x,i1,2x) )') ( this%moment_components(i), i = 1,no_mom_comp )
      
      if (iprint .gt. 0) then
      
         call this%channels%print(iwrite)

         
      end if
      
 
      if (iprint .gt. 1) then
         write(iwrite, *) ""     
         write(iwrite, '("  Scattering energies   ")') 
         write(iwrite, '("  -------------------   ")')
         do i=1, no_energies
            write(iwrite, '(D20.8)') this%energies(i)
         end do
         
      end if
      
   end subroutine
   
   function continuum_add(c1,c2)
      
!     Arguments   
      type(continuum), intent(in) :: c1, c2
      type(continuum)             :: continuum_add
      
      
      integer              :: nsym_c1, nsym_c2, nsym_big, i, j, max_syms, nsym
      integer, allocatable :: syms_in_common(:)
      

      
            
      if (c1 .samemodel. c2) then
    
         ! Now check if we c1 or c2 have continuum symmetries in common
         
         nsym_c1  = size( c1%mtot )
         nsym_c2  = size( c2%mtot )
         max_syms = size( c1%moment_components )
         
         nsym = nsym_c1 + nsym_c2
         
         allocate(syms_in_common( max_syms ) )         
         syms_in_common=0
         
         do i = 1, nsym_c1
            do j = 1, nsym_c2
            
               if ( (c1%stot(i) .eq. c2%stot(j)) .and. (c1%mtot(i) .eq. c2%mtot(j)) .and. (c1%gutot(i) .eq. c2%gutot(j))  ) then
               
                  syms_in_common(i)=j
                  nsym = nsym -1
               
               end if
         
            end do
         end do
         
         continuum_add = continuum( nsym )
         
         continuum_add % stot(1:nsym_c1) = c1 % stot
         continuum_add % mtot(1:nsym_c1) = c1 % mtot
         continuum_add % gutot(1:nsym_c1) = c1 % gutot
         
         i = nsym_c1 + 1
         do j = 1, nsym_c2 
         
            if ( any(syms_in_common .eq. j)  ) then
            else
                           
               continuum_add % stot(i)  = c2 % stot(j)
               continuum_add % mtot(i)  = c2 % mtot(j)
               continuum_add % gutot(i) = c2 % gutot(j)  
               
            end if
         
         end do

         allocate(continuum_add % moment_components( max_syms ) )
         continuum_add % moment_components = c1 % moment_components
         
         do i = 1, max_syms
         
            if ( c2%moment_components(i) .eq. 1 ) then
            
               continuum_add % moment_components(i) = 1
               
            end if
         
         end do
         
         continuum_add % channels = c1%channels + c2%channels
         
         allocate(continuum_add % energies( size(c1 % energies,1) ) )  ! Fix for ifort < 17       
         continuum_add % energies = c1 % energies
      
      end if
      
   end function
!  ---------------------------   
!  Procedures for type MOMENTS
!  ---------------------------     
   function new_moments(npw,nmc,ne,ni,nn, nc, nsym)   

      integer       :: npw, ni, nn, nc, ne, nmc, nsym
      type(moments) :: new_moments
      
      integer :: i
      
      new_moments%neutral_states = bound(nn)
 
      new_moments%scattering_states = continuum(nsym, ni,nc,ne,nmc)
            
      allocate( new_moments%elements(npw,nmc,ne,ni,nn) )
      new_moments%elements=0._idp
         
   end function new_moments
   
   function moments_from_same_model(tm1,tm2)
!     Arguments   
      type(moments), intent(in) :: tm1, tm2
      logical                     :: moments_from_same_model
      
      moments_from_same_model = .false.
   
      if (       ( tm1 % neutral_states    .eq.        tm2 % neutral_states   ) &
      &    .and. ( tm1 % scattering_states .samemodel. tm2 % scattering_states) ) then
           
         moments_from_same_model = .true.
 
      end if
      
   end function
   
    
   function add_moments(tm1,tm2)

!     Arguments   
      type(moments), intent(in) :: tm1, tm2
      type(moments)             :: add_moments

!     Local   
      integer                   :: i, npw_tm1, npw_tm2

      if ( tm1 .samemodel. tm2 ) then
      
!         add_moments % mol                 = tm1 % mol
    
!         add_moments % neutral_states      = tm1 % neutral_states
         
         add_moments % scattering_states   = tm1 % scattering_states + tm2 % scattering_states
         
!         add_moments % basis_type_photon   = tm1 % basis_type_photon 
         
!         add_moments % basis_type_electron = tm1 % basis_type_electron

         npw_tm1 = size( tm1%elements, 1 )
         npw_tm2 = size( tm2%elements, 1 )
         
         if (npw_tm1 .ge. npw_tm2) then
         
!            add_moments % elements = tm1%elements

            add_moments = tm1                                                                     ! Fix for ifort < 17
            add_moments % scattering_states   = tm1 % scattering_states + tm2 % scattering_states ! Fix for ifort < 17
            
            add_moments % elements(1:npw_tm2,:,:,:,:) = add_moments % elements(1:npw_tm2,:,:,:,:) &
            &                                         + tm2%elements
            
         else
         
!            add_moments % elements = tm2%elements
            
            add_moments = tm2                                                                     ! Fix for ifort < 17 
            add_moments % scattering_states   = tm1 % scattering_states + tm2 % scattering_states ! Fix for ifort < 17 
            add_moments % elements(1:npw_tm1,:,:,:,:) = add_moments % elements(1:npw_tm1,:,:,:,:) &
            &                                         + tm1%elements
            
         end if
      
      end if
   
   end function

!  Define * as matrix multiplication for moments
!  ---------------------------------------------
  function left_matrix_multiply_moments(mat, tm)

!     Arguments   
      type(moments), intent(in) :: tm
      complex(idp), intent(in) :: mat(:,:)
      type(moments)             :: left_matrix_multiply_moments

      integer :: ne, ni, nn 
      integer :: i, j, k 

      left_matrix_multiply_moments=tm 
      
      ne  = size( tm%elements, 3 )
      ni  = size( tm%elements, 4 )
      nn  = size( tm%elements, 5 )
          
      do k = 1, nn
         do j = 1, ni   
            do i = 1,ne      
      
               left_matrix_multiply_moments%elements(:,:,i,j,k) = matmul( mat,tm%elements(:,:,i,j,k) )
               
            end do
         end do
      end do
            
   end function left_matrix_multiply_moments
  
   function right_matrix_multiply_moments(tm, mat)

!     Arguments   
      type(moments), intent(in) :: tm
      complex(idp), intent(in)     :: mat(:,:)
      type(moments)             :: right_matrix_multiply_moments
      
      integer :: ne, ni, nn 
      integer :: i, j, k 

      right_matrix_multiply_moments=tm   
      
      ne  = size( tm%elements, 3 )
      ni  = size( tm%elements, 4 )
      nn  = size( tm%elements, 5 )
      
      do k = 1, nn
         do j = 1, ni   
            do i = 1,ne
            
               right_matrix_multiply_moments%elements(:,:,i,j,k) = matmul( tm%elements(:,:,i,j,k), mat )
               
            end do
         end do
      end do
      
   end function right_matrix_multiply_moments  
 
   subroutine print_moments(this, iwrite, iprint)
 
!     Arguments   
      class(moments) :: this
      integer     :: iwrite, iprint
      
      integer     :: i, no_scat_syms

      write( iwrite,  '(/, " -------------------------------")')    
      write( iwrite,  '(   " Partial Wave Transition Moments")') 
      write( iwrite,  '(   " -------------------------------")')      

      call this%neutral_states%print(iwrite)
      
      call this%scattering_states%print(iwrite,iprint)
      
      if (iprint .ge. 2) then
      
!        print moments          
      
      end if 
    
!      no_scat_syms= size( this%scattering_states )
      
!      write(iwrite, *) ""      
!      write(iwrite, '("   Transition Moments")')
!      write(iwrite, '("  -------------------")')
      

   end subroutine

!
!> @brief Converts moments from the format ouput by rsolve
!>
!>  Photoionization wave functions are kets in rsolve
!>  so dipole matrix elements have the form 
!>  \f$ \langle \Phi_i| d |\Psi^{-}_f(E) \rangle \f$. 
!>  Therefore matrix elements need to be conjugated. 
!  ------------------------------------------------   
   subroutine convert_moments(this, re_mom, im_mom)
      use ieee_arithmetic, only: ieee_is_nan
!     Arguments   
      class(moments) :: this
      real(idp)      :: re_mom(:,:,:,:), im_mom(:,:,:,:)

!     Local
!      complex(idp) :: mom_tmp( size(re_mom,2), size(re_mom,1),&
!      &                        size(re_mom,3), size(re_mom,4) )
      complex(idp), allocatable :: mom_tmp(:,:,:,:)      

      integer      :: no_channels, no_neutrals, no_energies, &
      &               no_components, ich, ine, ien, ico, lmax, no_lm, &
      &               no_ions, ion, l, m, i  
      
     
!     Assumes that this%neutral_states and this%scattering_states
!     have already been populated.

      no_neutrals   = size(re_mom,1)
      no_channels   = size(re_mom,2)
      no_components = size(re_mom,3)
      no_energies   = size(re_mom,4)
      
      allocate( mom_tmp( no_channels, no_neutrals, no_components, no_energies ) )
      mom_tmp = 0._idp
      
!     conjugate moments    
      do ien = 1, no_energies
      
         do ico = 1, no_components
      
            if( this % scattering_states % moment_components(ico) .eq. 1) then
         
               mom_tmp(:,:,ico,ien) = transpose( &
            &                            cmplx( re_mom(:,:,ico,ien), &
            &                                  -im_mom(:,:,ico,ien), &
            &                                   kind=idp) )
            
            end if
               
         end do

         ! remove NANs from dipoles (replace with previous energy or with zeros)
         if (any(ieee_is_nan(real(mom_tmp(:,:,:,ien))))) then
            if (ien > 1) then
               mom_tmp(:,:,:,ien) = mom_tmp(:,:,:,ien-1)
            else
               mom_tmp(:,:,:,ien) = 0
            end if
         end if

      end do
      
          

!     Re-jig the moments matrix so that the partial wave index 
!     corresponds to that used by the spherical harmonics, coulomb phase
!     and rotation matrices.
!     ------------------------------------------------------------------

      lmax    = maxval( this % scattering_states % channels % lchl )
      no_ions = maxval( this % scattering_states % channels % ichl )
      
      no_lm   = (lmax + 1)**2
      
      allocate( this % elements( no_lm,no_components,no_energies,no_ions,no_neutrals ) )
      this % elements = 0_idp
      
      do ien = 1, no_energies
      
         do ico = 1, no_components

            do ich = 1, no_channels
            
               ion  = this % scattering_states % channels % ichl(ich)            
               l    = this % scattering_states % channels % lchl(ich)
               m    = this % scattering_states % channels % mchl(ich)

               i    = lm2i(l,m) !Single index for l and m quantum numbers

               do ine = 1, no_neutrals 
                  
                  this % elements(i,ico,ien,ion,ine) = mom_tmp(ich,ine,ico,ien) 

               end do
               
            end do

         end do
         
      end do
   
   end subroutine
   
end module dipelm_types



