! Copyright 2019
!
! Alex G. Harvey with contributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!-----------------------------------------------------------------------
!> @brief Dipole smoothing routines.
!> @author Alex Harvey
!> @date  2019
!>
!> Routines for smoothing partial wave dipoles before calculation of observables. 
!> @param ismooth Test
!> 
!

!----------------------------------------------------------------------- 
module dipelm_smooth
     
   use interpolate, only: zspline, bspline_init, bspline_int
   use dipelm_defs, only: idp, small_int, maxprop_par, pi
   use dipelm_types
   use dipelm_special_functions, only: lm2i
   use photo_outerio, only: read_pw_dipoles, write_pw_dipoles

   implicit none

contains

!> @brief Energy-dependent Gaussian smoothing width
   real(idp) function energy_dependent_width(Ek, Eleft, Eright) result(width)

      real(idp), intent(in) :: Ek, Eleft, Eright

      width = 1e-4_idp + 0.1_idp * sqrt((Ek - Eleft) / (Eright - Eleft))

   end function energy_dependent_width


!> @brief Runs one of three  different smoothing routines
   subroutine gaussian_smooth(dipoles, selected_ion_states, selected_neutral_states, ifail)
         
      implicit none

!     Arguments
      type(moments)     :: dipoles
      integer           :: selected_ion_states(:), selected_neutral_states(:), ifail

      real(idp), allocatable      :: E_elec2(:), dip_re(:), dip_im(:)      
      integer, allocatable        :: limits(:,:)
      real(idp)                   :: E, width, deltax, start_integration, finish_integration, energy
      complex(idp)                :: dipole
      integer                     :: ieleft, i, j, k, m, n, ierr, idcmp, ie_chan_left, iwrite, iprint
      logical                     :: width_auto
      type(zspline), allocatable  :: zinterpolated_dipole(:,:,:)

      integer       :: no_components, no_energies, no_ion_states, no_neutral_states, no_channels,& 
      &                ine, ico, ich, ilm, ien, jen, ion
      real(idp)     :: E_ch_thresh
      
      type(moments)             :: smoothed_dipoles
      complex(idp), allocatable :: SD(:,:,:,:,:)

!     Namelist variables
!     ------------------
!>    @defgroup SMOOTH Namelist: &smooth
!>    @{
      integer           :: ismooth                               = 0           !< Controls smoothing 
                                                                               !< * 0 = No smoothing
                                                                               !< * 1 = Gaussian smoothing (no interpolation)
                                                                               !< * 2 = Gaussian smoothing (with interpolation)
                                                                               !< * 3 = Least squares fit
      real(idp)         :: Eleft                                 = 0.1_idp     !< Start point for smoothing (Hartree)
      real(idp)         :: Ewidth                                = 0           !< Width of smoothing filter (Hartree), <=0 for auto.
      real(idp)         :: deltaE                                = 0.001_idp   !< Integration step size
      integer           :: pol_order                             = 5           !< Polynomial order for least squares fit.
      integer           :: lu_smoothed_pw_dipoles(maxprop_par)   = 0           !< File units for output of smoothed dipoles
      integer           :: nset_smoothed_pw_dipoles(maxprop_par) = 0           !< Set number on lu_smoothed_pw_dipoles
      character(len=11) :: format_smoothed_pw_dipoles            = 'FORMATTED' !< Format of the output smoothed dipoles
!> @} 
          
      namelist /SMOOTH/ Eleft,   & 
      &                 Ewidth,  &
      &                 ismooth, &
      &                 deltaE,  &
      &                 pol_order, &
      &                 lu_smoothed_pw_dipoles, &
      &                 nset_smoothed_pw_dipoles, &
      &                 format_smoothed_pw_dipoles

!     TODO: Re-implement writing of smoothed dipoles

!     Default namelist values
!     -----------------------

!      Eleft     = 0.1_idp        
!      Ewidth    = 0.07_idp      
!      deltaE    = 0.001_idp      
!      ismooth   = 1             
!      pol_order = 5              

!     Other defaults not presentyl in namelist      
      iwrite    = 6              !< Destination for standard out.
      iprint    = 1              !< Level of detail output to standard out.
    
      rewind(5)
      read(5,nml=SMOOTH,ERR=200, iostat=ierr)

  200 if (ierr .gt. 0) then
  
         write (iwrite,4000)
         ifail=4000
         return

      end if
 
      smoothed_dipoles = dipoles
      width_auto = (Ewidth <= 0)
      width = Ewidth
 
      associate ( D      => dipoles % elements, &
      &           E_elec => dipoles % scattering_states % energies, &
      &           echl   => dipoles % scattering_states % channels % threshold_energies, &
      &           ichl   => dipoles % scattering_states % channels % ichl, &
      &           lchl   => dipoles % scattering_states % channels % lchl, &
      &           mchl   => dipoles % scattering_states % channels % mchl  ) 
      
      no_channels        =  size( ichl, 1 )
      no_components      =  size( dipoles % elements, 2 )
      no_energies        =  size( dipoles % elements, 3 )
      no_ion_states      =  size( dipoles % elements, 4 )
      no_neutral_states  =  size( dipoles % elements, 5 )      

      call move_alloc(smoothed_dipoles % elements, SD)

      select case(ismooth)
      case(0)
      case(1) !Gaussian smoothing - No interpolation
      
!        Method without interpolation, requires a fine energy grid to be
!        good
!        ---------------------------------------------------------------

         write(iwrite,'(" Gaussian smoothing of partial wave dipoles ")')
          
 !       Find integration limits in terms of energy index
         call  sigma2i( E, Eleft, E_elec, E_elec2, ieleft, limits)

         E_elec2 = E_elec
 
         SD(:,:,ieleft:no_energies,:,:) = 0._idp
         deltax                         = E_elec(2)-E_elec(1)
 
 !       Perform integration
 
         do ine = 1, no_neutral_states
            
            if ( any(selected_neutral_states .eq. ine) ) then
          
            do ico = 1, no_components
                   
                               
               do ich = 1, no_channels
                  
                  if ( any(selected_ion_states .eq. ichl(ich)) ) then
                  
                  ilm = lm2i( lchl(ich), mchl(ich) )
                  ion = ichl(ich)
                  E_ch_thresh = echl(ich)

                  call binary_search( E_elec, E_ch_thresh, 0, ie_chan_left )

                  !$OMP PARALLEL DO DEFAULT(NONE) PRIVATE(ien, jen, dipole) SHARED(E_elec2, SD, dipoles) &
                  !$OMP FIRSTPRIVATE(ieleft, no_energies, width_auto, deltax, ilm, ion, ico, ine, ich, ie_chan_left) &
                  !$OMP FIRSTPRIVATE(width, E_ch_thresh)
                  do ien = ieleft, no_energies

                     if (width_auto) then
                        width = energy_dependent_width(E_elec2(ien), E_elec2(ieleft), E_elec2(no_energies))
                     end if

                     dipole = 0

                     do jen = 1,no_energies
                         
                        if(jen .lt. ieleft) then !Less than lowest smoothing point
                         
                           dipole = dipole + gaussian(E_elec2(ien), E_elec2(jen), width) &
                           &               * dipoles % elements(ilm, ico, ieleft, ion, ine) * deltax
                            
                        else if ( E_elec2(jen) .lt. E_ch_thresh ) then !less than channel threshold
                          
                           dipole = dipole + gaussian(E_elec2(ien), E_elec2(jen), width) &
                           &               * dipoles % elements(ilm, ico, ie_chan_left, ion, ine) * deltax
                                                                                             
                        else
                         
                           dipole = dipole + gaussian(E_elec2(ien), E_elec2(jen), width) &
                           &               * dipoles % elements(ilm, ico, jen, ion, ine) * deltax
                            
                        end if
                         
                     end do
                      
                     if ( E_elec2(ien) .ge. E_ch_thresh ) then
                         
                        SD(ilm, ico, ien, ion, ine) = dipole
                      
                     else
                      
                        SD(ilm, ico, ien, ion, ine) = 0._idp
                      
                     end if
                      
                  end do
                  
                  end if 
               end do
                 
            end do
            
            end if    
         end do

          
      case(2) !Gaussian smoothing - With interpolation
      
!        Method using interpolated dipoles
!        ------------------------------------------------------------------

         write(iwrite,'(" Gaussian smoothing of partial wave dipoles - using interpolated dipoles")') 
            
         allocate( zinterpolated_dipole(no_channels, no_components, no_neutral_states) )
        

         do ine = 1, no_neutral_states
         
            if ( any(selected_neutral_states .eq. ine) ) then
                          
            do ico = 1, no_components
           
               do ich = 1, no_channels
               
                  if ( any(selected_ion_states .eq. ichl(ich)) ) then
           
                  ilm = lm2i( lchl(ich), mchl(ich) )
                  
           
                  call bspline_init( E_elec, D( ilm, ico, :, ichl(ich), ine), zinterpolated_dipole(ich, ico, ine), 8 )
                  
                  end if
                  
               end do
           
            end do
            
            end if
         end do
         

         call  sigma2i( E, Eleft, E_elec, E_elec2, ieleft, limits)
        
         E_elec2 = E_elec ! Needed because OpenMP has problems with associate variables
        
         do ine = 1, no_neutral_states
         
            if ( any(selected_neutral_states .eq. ine) ) then        
         
            do ico = 1, no_components

!               write(iwrite, '("Applying gaussian smoothing to component ", a1)')  
               
               call binary_search( E_elec, Eleft, 0, ieleft )
               
               do ich = 1, no_channels
               
                  if ( any(selected_ion_states .eq. ichl(ich)) ) then
         
                  ilm         = lm2i( lchl(ich), mchl(ich) )
                  ion         = ichl(ich)
                  E_ch_thresh = echl(ich)

                  !$OMP PARALLEL DEFAULT(NONE) PRIVATE(ien,start_integration,finish_integration,energy,ie_chan_left,dipole) &
                  !$OMP SHARED(ieleft,no_energies,E_elec2,E_ch_thresh,ion,ine,ico,ich,ilm,zinterpolated_dipole,deltaE,SD) &
                  !$OMP FIRSTPRIVATE(width, width_auto)
                  !$OMP DO
                  do ien = ieleft, no_energies

                     if (width_auto) then
                        width = energy_dependent_width(E_elec2(ien), E_elec2(ieleft), E_elec2(no_energies))
                     end if

                     start_integration  = E_elec2(ien) - 3*width
                     finish_integration = E_elec2(ien) + 3*width
                     energy             = start_integration
                     
                     call binary_search( E_elec2, E_ch_thresh, 0, ie_chan_left )
                     
                     dipole = 0.0_idp

                     do while( energy .le. finish_integration )

                        if( energy .lt. E_elec2(ieleft) ) then !Less than lowest smoothing point

                           dipole = dipole + gaussian( energy, E_elec2(ien), width )&
                           &               * bspline_int( E_elec2(ieleft), zinterpolated_dipole(ich, ico, ine) ) * deltaE
                           
                        else if ( energy .lt. E_ch_thresh ) then !less than channel threshold
                        
                           dipole = dipole + gaussian( energy, E_elec2(ien), width )&
                           &               * bspline_int( E_elec2(ie_chan_left), zinterpolated_dipole(ich, ico, ine) ) * deltaE

                           
                        else if ( energy .gt. E_elec2(no_energies) ) then !greater than highest energy

                           dipole = dipole + gaussian( energy, E_elec2(ien), width )&
                                           * bspline_int( E_elec2(no_energies), zinterpolated_dipole(ich, ico, ine) ) * deltaE
                           
                        else

                           dipole = dipole + gaussian( energy, E_elec2(ien), width )&
                                           * bspline_int( energy, zinterpolated_dipole(ich, ico, ine) ) * deltaE
                           
                        end if
                        
                        energy = energy + deltaE

                     end do
                     
                     if (E_elec2(ien) .ge. E_ch_thresh) then
                        
       
                        SD(ilm, ico, ien, ion, ine) = dipole
                        
                     else
                        
                        SD(ilm, ico, ien, ion, ine) = 0._idp
                        
                     end if
                     
                  end do
                  !$OMP END DO
                  !$OMP END PARALLEL
                  
                  end if
                  
               end do

            end do
            
            end if
            
         end do


      case(3) ! Least squares fit
      
        write(iwrite,'(" Least squares fit to partial wave dipoles, polynomial order = ", i3)') pol_order

         if (Eleft .lt. 0.0_idp) then

         else

         !user set initial energy

            do ien = 1, no_energies
            
               if(E_elec(ien) .ge. Eleft ) then
               
                  ieleft = ien
                  Eleft  = E_elec(ieleft)
               
                  exit
               
               end if
               
            end do
            
         end if

         allocate(dip_re(no_energies), dip_im(no_energies))
         dip_re=0._idp;dip_im=0._idp
         
         do ine = 1, no_neutral_states
         
            if ( any(selected_neutral_states .eq. ine) ) then 
                       
            do ico = 1, no_components

               do ich = 1, no_channels
               
                  if ( any(selected_ion_states .eq. ichl(ich)) ) then
               
                  ilm = lm2i( lchl(ich), mchl(ich) )
         
                  call binary_search( E_elec, echl(ich), 0, ie_chan_left)
               
                  dip_re = real( D(ilm,ico,:,ichl(ich), ine) )
                  dip_im = aimag( D(ilm,ico,:,ichl(ich), ine) )
               
                  call least_squares( pol_order, ieleft, no_energies, E_elec, dip_re )
                  call least_squares( pol_order, ieleft, no_energies, E_elec, dip_im )
                  
                  SD(ilm, ico, ie_chan_left:, ichl(ich), ine) =  cmplx(dip_re(ie_chan_left:), dip_im(ie_chan_left:), kind=idp)  
                  
                  end if
                  
               end do
               
            end do
            
            end if
                        
         end do


      end select

!      if (lu_smoothed_pw_dipoles(1) .ne. 0) then
!         call write_smoothed_pwd_dipoles( lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, &
!     &                                    lu_smoothed_pw_dipoles, nset_smoothed_pw_dipoles,  &
!     &                                    format_smoothed_pw_dipoles, dip_elm )

!      end if

      D = SD
      
      end associate
      
      
      
      return
        !---Format statements
 4000 format(/,5X,'Problems reading the namelist SMOOTH',/)
      end subroutine gaussian_smooth
      
   subroutine sigma2i(E,Estart,escat,escat2,is,limits)
      implicit none

!     Arguments      
      real(kind=idp)              :: E, Estart, three_sigma, low_lim, upp_lim
      integer                     :: en, i, j, nesc, is
      integer, allocatable        :: limits(:,:)
      real(kind=idp)              :: escat(:)
      real(kind=idp), allocatable :: escat2(:)

!     written for elastic scattering first

!     1. Find the first energy point above the starting energy
!        This is now the zero energy point.
!     2. Create new escat array with new energy zero.
!     3. For each point above zero point 
!        a. Find 3sigma
!        b. Find nearest energy point to E - 3*sigma
!        c. Find nearest energy point to E + 3*sigma
!        Note for zero point 
!     4. Save to array integ_lims

!     Find the first element by binary search

      nesc = size(escat)
      allocate(escat2(nesc), limits(2,nesc))
      escat2 = 0.0_idp
      limits = 0

      call binary_search( escat, Estart, 0, is )
      
      print*, "Left matching point for smoothing is:", escat(is)
      
      escat2 = escat-escat(is)

      limits(1,is) = is
      limits(2,is) = is
      
      do en = is+1, nesc
      
         three_sigma = 3*sigma(escat2(en))

         low_lim     = escat2(en) - three_sigma
         upp_lim     = escat2(en) + three_sigma

         call binary_search( escat2, low_lim, 0, limits(1,en) )
         call binary_search( escat2, upp_lim, 1, limits(2,en) )
         
      end do

   end subroutine sigma2i
   
   real(kind=idp) function sigma(E)
   
      implicit none
      
      real(kind=idp) :: E, sigma_tmp

      !sigma=(1.0_idp/3.0_idp)*sqrt(1.0_idp/(2.0_idp*sqrt(E)))
      sigma= 0.3_idp

   end function sigma 
      
   subroutine binary_search(arr,val,lr,i)
      implicit none
!     returns right most value

      real(kind=idp) :: arr(:)
      real(kind=idp) :: val
      integer        :: i, lr
      integer        :: lp, rp, mp

      lp = 1 
      rp = size(arr)
      
      do while((rp-lp) .gt. 1)
      
         mp = (lp+rp)/2
      
         if (val .gt. arr(mp)) then
            lp = mp
         else
            rp = mp
         end if
      
      end do

      select case (lr)
      case(0) !First value greater or equal to val
      
         if (abs(val-arr(lp)) .lt. small_int) then 
            i = lp
         else
            i = rp
         end if
      
      case(1) !First val less than or equal to val 
      
         if (abs(val-arr(rp)) .lt. small_int) then 
            i = rp
         else
            i = lp
         end if
      
      end select

      end subroutine binary_search 
                 
      subroutine least_squares( pol_order, istart_point, iend_point, x_points, y_points)
      implicit none

!     Arguments
      integer   ::  pol_order, istart_point, iend_point
      real(idp) :: x_points(:), y_points(:)

!     Local variables
      integer                :: i, info , j, number_of_points,lwork
      real(idp), allocatable :: YY(:), XX(:,:), WORK(:),X0(:,:)
      real(idp)              :: valx, valy, aux,coeff

      intent(in)    :: istart_point,iend_point, x_points
      intent(inout) :: y_points


!       EXTERNAL         DGELS

      ! Pol order + 1 (coefficients)
!       pol_order = 3

      number_of_points = iend_point - istart_point + 1

      allocate(XX(number_of_points,pol_order), YY(number_of_points), X0(number_of_points,pol_order))

      YY = y_points(istart_point:iend_point)

!     weight the first and the last point 
      do i = 1, number_of_points

         if ( (i .eq. 1) .or. (i .eq. number_of_points) ) then
            coeff = 100._idp
         else
            coeff = 1._idp
         endif
      
         YY(i) = coeff*YY(i)
         
         do j = 1, pol_order
         
            X0(i,j) = coeff * x_points(istart_point+i-1)**(j-1)
         
         end do

      enddo

      lwork = 2*number_of_points*pol_order
      allocate(WORK(lwork))

!
!     Solve the least squares problem min( norm2(b - Ax) ) for x
!
      XX = X0
      CALL DGELS( 'n', number_of_points, pol_order, 1, XX, number_of_points, YY, number_of_points, &
      &            WORK, LWORK, INFO )

!        WRITE (6,*) 'Least squares solution'
!        WRITE (6,*) (YY(I),I=1,pol_order)

      do i = 1, number_of_points
      
         aux = 0._idp
      
         do j = 1, pol_order
         
            aux = aux + (YY(j))*x_points(istart_point+i-1)**(j-1)
         
         end do
         
         y_points(istart_point+i-1) = aux
      
      end do


      return

      end subroutine least_squares

!     Gaussian smoothing function
!     ---------------------------
      real(kind=idp)function gaussian(E,Ep,sigma_E)
         implicit none
         
!        Arguments         
         real(idp) :: sigma_E, E, Ep
!        Local
         real(idp) :: peak_amp

         peak_amp =  1.0_idp / ( sqrt(2*pi)*sigma_E ) 

         gaussian = peak_amp * exp( -(E-Ep)**2/(2*sigma_E**2) )

      end function gaussian 
                 
end module
