! Copyright 2019
!
! Alex G. Harvey with contributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
module dipelm_procs

   use dipelm_defs
   use dipelm_special_functions
   use dipelm_types
   use dipelm_smooth
   use dipelm_io
   use blas95_compak, only: gemm
   
   implicit none

contains

   subroutine  dipelm_drv(ifail)    
      implicit none

!     Arguments
      integer :: ifail 
                                                   
!     Local

!     Namelist variables
!     ------------------
!>    @defgroup DIPELMINP Namelist: &dipelminp
!>    @{
      integer              :: lu_pw_dipoles(maxprop_par)   = 0           !< File units holding partial wave dipoles from rsolve.
      integer              :: nset_pw_dipoles(maxprop_par) = 0           !< Set number of dipoles on each unit.
      character(len=11)    :: format_pw_dipoles            = 'FORMATTED' !< Partial wave dipoles file format.
      
      integer              :: output_style                 = 0           !< Choose output file style for randomly oriented molecule observables. 
                                                                         !< 0 =  All states in a single file.
                                                                         !< 1 =  A file for each (ion state, neutral state) pair.

      integer              :: iprint                       = 0           !< The higher the number the more is written to iwrite.
      integer              :: iwrite                       = 6           !< Destination for writing calculation details.   
      real(idp)            :: first_IP                     = 0._idp      !< Allows adjustment of 1st Ip. If zero calculated Ip is used.
      real(idp)            :: ion_charge                   = 1._idp      !< Charge of ionized molecule.
      integer              :: no_ion_states                = 0           !< No of ion states for which to calculate randomly oriented observables.
      integer              :: no_neutral_states            = 0           !< No of neutral states for which to calculate randomly oriented observables.
      integer, allocatable :: select_ion_states(:)                       !< Define subset of ion states for which to  calculate randomly oriented observables.
      integer, allocatable :: select_neutral_states(:)                   !< Define subset of neutrals states for which to  calculate randomly oriented observables.     
      logical              :: calc_oriented_observables    = .false.     !< Controls calculation of oriented obsesrvables. If .true. namelist orient nust be supplied.
      logical              :: calc_aligned_observables     = .false.     !< Controls calculation of aligned PADs. If .true. namelist ALIGNINP nust be supplied.
      logical              :: smooth                       = .false.     !< Controls gaussian smoothing of dipoles. If .true. namelist smooth must be supplied.
      character(len=3)     :: point_group                  = ''          !< Not used currently.
!>    @}

!     For holding the dipoles      
      type(moments)     :: dipoles      
                             
!     Needed for reading DIPTRANS if we are using the old input style      
      integer   :: ngrdproj(2), ngrdalign(3), neutral_state
      real(idp) :: euler_angle_limits(6),scat_angle_limits(4)
      logical   :: legacy_input
      
      integer              :: ierr, i
      character(len=1000)  :: line
      
      integer              :: no_ion_states_all, no_neutral_states_all
      integer, allocatable :: tmp_arr1(:), tmp_arr2(:)
      logical              :: use_exp_IP
      real(idp)            :: calc_1st_IP 
            
!     For basis transformation      
      complex(idp), allocatable :: U(:,:), U1(:,:)
      integer                   :: lmax
      type(moments)             :: cmp_dipoles

!     Cross section and asymmetry parameters   
      real(idp), allocatable  :: xsec(:,:,:), beta_1(:,:,:), beta_2l(:,:,:), beta_2c(:,:,:)

!     Aligned cross section and asymmetry parameters
      complex(idp), allocatable :: cmp_xsec(:,:,:,:), beta_LQl(:,:,:,:,:), beta_LQc(:,:,:,:,:)
      real(idp), allocatable    :: time(:)

!     Aligned Distribution Moments
      integer, allocatable      :: KQ(:,:)
      complex(idp), allocatable :: AKQ(:,:)
      integer                   :: Lbig_max

!     Namelist variables
!     ------------------
!>    @defgroup ALIGNINP Namelist: &aligninp
!>    @{
      character(len=255)   :: ADM_KQ_file           = 'ADM_KQ.inp' !< Formatted file containing the ADM coefficients for molecules with ADM given by expansion into spherical harmonics
      character(len=255)   :: time_file             = 'time.txt'   !< Formatted file containing the time steps (one per line). This is only used at output.
      real(idp), allocatable :: select_photon_en(:)                !< Photon energies in eV for which the aligned PADs are to be saved to disk as individual files. They must must be from the energy interval specified by [en_start, en_end] below.
      integer              :: en_start = 1, en_end = -1            !< Range of energy indices for which the calculation will be run. Value of en_end <= 0 stands for the last energy index.
      integer              :: L_max_write = -1                     !< Maximum L for which the aligned PADs will be written out to disk. Default is L_max_write = L_max
      logical              :: perfect_aligned = .false.            !< If set to .true. then betas for perfectly aligned case are computed for molecular axis in the z-y plane going for 90 points in the angular range [0,180] degrees. The input variables ADM_KQ_file, time_file are ignored
!>    @}

      namelist /DIPELMINP/  lu_pw_dipoles,                        &
      &                     format_pw_dipoles,                    &
      &                     nset_pw_dipoles,                      &
      &                     output_style,                         &
      &                     iprint,                               &
      &                     iwrite,                               &
      &                     first_IP,                             &
      &                     ion_charge,                           &
      &                     no_ion_states,                        &
      &                     no_neutral_states,                    &
      &                     select_ion_states,                  &
      &                     select_neutral_states,              &          
      &                     calc_oriented_observables,            &   
      &                     calc_aligned_observables,             &   
      &                     smooth,                               &
      &                     point_group

      namelist/ALIGNINP/    ADM_KQ_file,                         &
      &                     time_file,                           &
      &                     select_photon_en,                    &
      &                     en_start, en_end, L_max_write,       &
      &                     perfect_aligned

!     Default values for DIPELMINP namelist
!     -------------------------------------

!      lu_pw_dipoles             = 0             
!      nset_pw_dipoles           = 0            
!      format_pw_dipoles         = 'FORMATTED'   ! 'UNFORMATTED' 
!      iprint                    = 0             
!      iwrite                    = 6             
!      first_IP                  = 0.0_idp       
!      ion_charge                = 1.0_idp      
!      no_ion_states             = 0            
!      no_neutral_states         = 0            
!      calc_oriented_observables = .false.                                     
!      smooth                    = .false.         
!      point_group               = ''          

      allocate(select_ion_states(max_states_for_dipoles), select_neutral_states(max_states_for_dipoles) )
      select_ion_states       = 0  
      select_neutral_states   = 0 

      legacy_input              = .false.        
!     Read namelists
!     --------------
      
!     Read dipelminp
            
      ierr = 0
      rewind(5)
      read(5, nml=DIPELMINP, iostat=ierr)
      
      if (ierr .ne. 0) then
         
         backspace(5)
         read(5,fmt='(A)') line
         write(iwrite,'(A)') &
           'Invalid line in namelist DIPELMINP: '//trim(line)
         write (iwrite,2000)
         ifail = 2000 
         
         return
           
      end if
    
!     For backwards compatability with old style of DIPELM input          

!     Check to see if DIPTRANS exists and if so get the needed variables
!     ( for random orientation these are first_IP, no_ion_states and neutral_state)

      call read_old_DIPTRANS( legacy_input, first_IP, no_ion_states, neutral_state, ngrdproj, ngrdalign, &
      &                       euler_angle_limits, scat_angle_limits )
      
      if (legacy_input) then         

!        Whether we smooth or not is controlled by the ismooth namelist 
!        variable in the old style input. The namelist smooth was always read.  
     
         smooth = .true. 
         
         write( iwrite,  '(/, " Old style of dipelm namelist input used. ")')   

      end if  

!     If first_IP is set, use it instead of calculated IP      
      use_exp_IP = .false. 
      if(first_IP .ne. 0.0_idp) use_exp_IP = .true. 
      
      
!     Read transition moments from file      
!     --------------------------------- 

      call read_moments( lu_pw_dipoles, format_pw_dipoles, nset_pw_dipoles, dipoles, iprint, &
      &                  iwrite, ifail )
      
      
      dipoles % mol % ion_charge = ion_charge ! Ion charge is not specified in the pw dipole files.
                                              ! TODO: This should be changed in the future
    
!     Determine how many and which states we want to calculate observables for

      no_ion_states_all      = size(dipoles % scattering_states % channels % ion_states % energies)
      no_neutral_states_all  = size(dipoles % neutral_states % energies)
      
      call move_alloc(select_ion_states, tmp_arr1)
      call move_alloc(select_neutral_states, tmp_arr2)
      
      allocate(select_ion_states(no_ion_states_all), select_neutral_states(no_neutral_states_all) )
      select_ion_states     = 0
      select_neutral_states = 0
      
      select_ion_states(1:no_ion_states)         = tmp_arr1(1:no_ion_states) 
      select_neutral_states(1:no_neutral_states) = tmp_arr2(1:no_neutral_states)
      
      call determine_states_selected( no_ion_states, select_ion_states, no_ion_states_all, &
   &                                  no_neutral_states, select_neutral_states, no_neutral_states_all)              
      
!     If we are using an experimental 1st IP then we adjust the neutral state energies accordingly
!     Energies of excited ion and neutral states relative to their respective ground states are 
!     not affected.      
     
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies )
      
      calc_1st_IP = E_ion(1) - E_neut(1)
      
      if (use_exp_IP) then
         
         write( iwrite,  '(/, " Calculated Ip = ", D20.8)')  calc_1st_IP
         write( iwrite,  '(   " User input Ip = ", D20.8)')  first_IP           
 
         do  i = 1, size( E_neut )
         
            E_neut(i) = E_neut(i) + (calc_1st_IP - first_IP)
         
         end do
    
      else 

         write( iwrite,  '(/, " Calculated Ip = ", D20.8)')  calc_1st_IP 
                
      end if
      
      end associate

!     Smooth dipoles if requested
!     ---------------------------
            
      if (smooth) then
      
         call gaussian_smooth(dipoles, select_ion_states, select_neutral_states,  ifail)
      
      end if

!     -----------------------------------------------------      
!          
!     Calculate observables for randomly oriented molecules
!
!     -----------------------------------------------------      
      
!     Transform from basis of real spherical harmonics 
!     to complex spherical harmomnics      
!     ------------------------------------------------

      lmax = maxval( dipoles % scattering_states % channels % lchl )

      call sph_basis_transform_matrix(U, lmax, 'Slm')
      call sph_basis_transform_matrix(U1, 1, 'Slm')
      
!     Note that * is defined as matrix multiplication for the moments type, and a moment type is returned.
!>    @todo: Currently matmul is used for type(moment) B = type(moment) A * matrix, change to gemm.
      
      cmp_dipoles = conjg(U) * dipoles * transpose( U1(2:4,2:4) )

      cmp_dipoles % basis_type_electron = 'Ylm'
      cmp_dipoles % basis_type_photon   = 'Ylm'

!     Calculate Cross section and beta parameters.
!     --------------------------------------------

!>     @todo: Total xsection (summed over states) for unoriented molecules. 

      call calculate_cross_section_and_beta_parameters( cmp_dipoles, select_ion_states, select_neutral_states, &
      &                                                 xsec, beta_1, beta_2l, beta_2c )
      
      call write_cross_section_and_beta_parameters( dipoles, select_ion_states, select_neutral_states, output_style, &
      &                                             xsec, beta_1, beta_2l, beta_2c )
      
      write( iwrite,  '(/, " Unoriented cross sections and asymmetry parameters calculated. ")')      
      
!     Oriented observables.
!     ---------------------
      !@todo this option is currently never executed since the related namelist ORIENT has disappeared from the code.
      if (calc_oriented_observables) then

!        call oriented_observables(cmp_dipoles, legacy_input, iprint, iwrite, ifail) ! Ylm basis for electron and photon.
         call oriented_observables(dipoles, legacy_input, iprint, iwrite, ifail)     ! Slm basis for electron and photon.
      
      end if

!     Aligned observables.
!     ---------------------
      if (calc_aligned_observables) then

         write( iwrite,  '(/, " Calculation of Aligned PADs has been selected. ")')      

         allocate(select_photon_en(max_energies_for_aligned))
         select_photon_en = 0

         rewind(5)
         read(5, nml=ALIGNINP)

         !Check value of en_start here, further checks will be done in calculate_aligned_cross_section_and_beta_parameters
         if (en_start <= 0) then
            print *,'incorrect value of en_start'
            stop 1
         endif

         if (perfect_aligned) then

            write( iwrite,  '(/, " Perfectly aligned case has been selected. ")')
            !Read/generate the theta grid
            call read_time_file(iwrite, time_file, time)

         else

            write( iwrite,  '(/, " Alignment distribution selected. ")')

            !Read the KQ, AKQ coefficients from ADM_KQ_file
            call read_ADM_KQ(iwrite, ADM_KQ_file, KQ, AKQ)
            call read_time_file(iwrite, time_file, time)

         endif

         call calculate_aligned_cross_section_and_beta_parameters( cmp_dipoles, select_ion_states, select_neutral_states,&
        &                                               en_start, en_end, KQ, AKQ, cmp_xsec, beta_LQl, beta_LQc, Lbig_max,&
        &                                               perfect_aligned)

         if (L_max_write < 0) L_max_write = Lbig_max

         call write_aligned_cross_section_and_beta_parameters( dipoles, select_ion_states, select_neutral_states, time,&
        &                                    en_start, en_end, select_photon_en, L_max_write, cmp_xsec, beta_LQl, beta_LQc )

      endif
      
      return
      
!     format statements
 2000 format(/,5X,'Problems reading the namelist DIPELMINP',/)    
   end subroutine dipelm_drv
   
   subroutine oriented_observables(dipoles, legacy_input, iprint, iwrite, ifail)   
      implicit  none
      
!     Arguments
      type(moments) :: dipoles
      logical       :: legacy_input
      integer       :: iprint, iwrite, ifail
      
!     Local

!     Namelist variables
!>    @defgroup ORIENT Namelist: &orient
!>    @{
      integer              :: ngrdproj(2)                    = [ 31, 4 ]    !< No. of photoelectron direction points (no_theta, no_phi)
      integer              :: ngrdalign(3)                   = [ 1, 1, 1 ]  !< No. of orientation (no_alpha, no_beta, no_gamma) points
      
                                                                            
      real(idp)            :: euler_angle_limits(6)          = [ 0._idp, 0._idp, 0._idp, 360._idp, 180._idp, 360._idp ]!<
                                                                            !< Defines the angular range for the molecular orientation
                                
      real(idp)            :: scat_angle_limits(4)           = [ 0._idp, 0._idp, 180._idp, 360._idp ] !<
                                                                            !< Defines the angular range for the photoelectron
                                                                            
      integer              :: no_ion_states_orient           = 0            !< No of ion states for which to calculate oriented observables
      integer              :: no_neutral_states_orient       = 0            !< No of neutral states for which to calculate oriented observables
      integer, allocatable :: select_ion_states_orient(:)                   !< Define subset of ion states for which to calculate oriented observables
      integer, allocatable :: select_neutral_states_orient(:)               !< Define subset of neutral states for which to calculate oriented observables
      logical              :: calc_mol_frame_observables     = .true.       !< Controls calculation of molecular frame observables (alpha = beta = gamma = 0)
      logical              :: calc_lab_frame_observables     = .false.      !< Controls calculation of lab frame observables
      logical              :: write_dipoles                  = .false.      !< Controls writing of Energy-angle form of the dipoles
!>    @}
      
      integer              :: lmax, no_channels, no_components, no_energies, no_ion_states_all, &
      &                       no_neutral_states_all, ien, ion ,ine, i, j, l, m, no_alpha, no_beta, &
      &                       no_gamma, no_euler, no_theta_phi, iangle, ia, ib, ig

      complex(idp), allocatable :: MFDip(:,:,:), LFDip(:,:,:,:), Ylm(:,:)
      real(idp),    allocatable :: MFPAD(:,:,:), LFPAD(:,:,:,:), grid_alpha(:), grid_beta(:), grid_gamma(:)
      
      integer   :: ierr
      
      real(idp) :: first_IP
      integer   :: neutral_state 
      
      namelist /ORIENT/   ngrdproj, &                    
     &                    ngrdalign,&                    
     &                    euler_angle_limits,&           
     &                    scat_angle_limits, &          
     &                    no_ion_states_orient, &               
     &                    no_neutral_states_orient, &           
     &                    select_ion_states_orient,&          
     &                    select_neutral_states_orient, &     
     &                    calc_mol_frame_observables, &  
     &                    calc_lab_frame_observables, &  
     &                    write_dipoles                  

!     Namelist ORIENT  defaults
!     --------------------------

!      ngrdproj                   = (/ 31, 4 /)
!      ngrdalign                  = (/ 1, 1, 1 /)
!      euler_angle_limits         = (/ 0._idp, 0._idp, 0._idp, 360._idp, 180._idp, 360._idp /)
!      scat_angle_limits          = (/ 0._idp, 0._idp, 180._idp, 360._idp /)
!      no_ion_states_orient              = 0
!      no_neutral_states_orient          = 0
!      calc_mol_frame_observables = .true. 
!      calc_lab_frame_observables = .false.
!      write_dipoles              = .false.
      
      no_ion_states_all      = size(dipoles % scattering_states % channels % ion_states % energies)
      no_neutral_states_all  = size(dipoles % neutral_states % energies)
      
      allocate(select_ion_states_orient(no_ion_states_all), select_neutral_states_orient(no_neutral_states_all) )
      select_ion_states_orient     = 0 
      select_neutral_states_orient = 0

      if(legacy_input) then
      
         call read_old_DIPTRANS( legacy_input, first_IP, no_ion_states_orient, neutral_state, ngrdproj, ngrdalign, &
         &                       euler_angle_limits, scat_angle_limits ) 
              
      else
      
         rewind(5)

         read(5,nml=ORIENT, iostat=ierr)
 
         if (ierr .gt. 0) then
            write (6,2001)
            ifail = 2001
            return
         end if
         
      end if
      
!     Process namelist input
!     ----------------------

!     Figure out how many and which states we should calculate oriented observables for.      

      call determine_states_selected( no_ion_states_orient, select_ion_states_orient, no_ion_states_all, &
   &                                  no_neutral_states_orient, select_neutral_states_orient, no_neutral_states_all)
      
!     Convert from degrees to radians

      euler_angle_limits = euler_angle_limits * 2*pi/360     
      scat_angle_limits  = scat_angle_limits  * 2*pi/360

!     Start calculation of oriented observables
!     -----------------------------------------      
      
!     Construct $i^{-l}e^(i\sigma_l) and multiply into dipoles.

      call multiply_in_coulomb_phase_factor(dipoles)

!     Calculate Molecular Frame Dipoles (MFDip) and Molecular Frame Photoelectron Angular 
!     Distributions (MFPAD)

      lmax  =  maxval( dipoles % scattering_states % channels % lchl )

      call grid_sp_harm(lmax, ngrdproj, scat_angle_limits, Ylm, dipoles % basis_type_electron)
    
      if (calc_mol_frame_observables) then

         do i = 1, no_neutral_states_orient
      
            ine = select_neutral_states_orient(i) 
      
            do j = 1, no_ion_states_orient
            
               ion = select_ion_states_orient(j)

               call calculate_oriented_dipoles( dipoles, ion, ine, 0._idp, 0._idp, 0._idp, Ylm, MFDip )
                
               if (write_dipoles) call write_MFDip( dipoles, scat_angle_limits, ngrdproj, ion, ine, MFDip ) 
      
               call calculate_MFPAD(dipoles, MFDip, ion, ine, MFPAD)
    
               call write_MFPAD( dipoles, scat_angle_limits, ngrdproj, ion, ine, MFPAD)
             
               deallocate(MFDip, MFPAD)
            
            end do
         
         end do
         
         if (write_dipoles) write( iwrite,  '(/, " MFDip calculated. ")') 
         write( iwrite,  '(/, " MFPAD calculated. ")')
                  
      end if
    
!     Calculate Lab Frame Dipoles (LFDip) and Lab Frame Photoelectron Angular 
!     Distributions (LFPAD). Warning output files quickly become large!
!     We loop over selected states, and orientations.      
      
      if (calc_lab_frame_observables) then

         no_theta_phi  = ngrdproj(1) * ngrdproj(2)
         no_alpha      = ngrdalign(1)
         no_beta       = ngrdalign(2)
         no_gamma      = ngrdalign(3)
         no_euler      = no_alpha*no_beta*no_gamma
         
         no_energies   = size( dipoles % scattering_states % energies )
         no_components = size( dipoles % scattering_states % moment_components )
          
!        First construct euler angle grids    
         call linspace( euler_angle_limits(1), euler_angle_limits(4), no_alpha, grid_alpha, include_end_point = .false. )
         call linspace( euler_angle_limits(2), euler_angle_limits(5), no_beta , grid_beta )
         call linspace( euler_angle_limits(3), euler_angle_limits(6), no_gamma, grid_gamma, include_end_point = .false. )            

         do i = 1, no_neutral_states_orient
      
            ine = select_neutral_states_orient(i) 
      
            do j = 1, no_ion_states_orient
            
               ion = select_ion_states_orient(j)
               
               allocate( LFDip(no_euler, no_theta_phi, no_components, no_energies), &
               &         LFPAD(no_euler, no_theta_phi, no_components, no_energies)  )
               iangle = 1
               
               do ig = 1, no_gamma
               do ib = 1, no_beta
               do ia = 1, no_alpha

                  call calculate_oriented_dipoles( dipoles, ion, ine, grid_alpha(ia), grid_beta(ib), grid_gamma(ig), Ylm, MFDip )
      
                  call calculate_MFPAD(dipoles, MFDip, ion, ine, MFPAD)
                  
                  LFDip(iangle,:,:,:) = MFDip
                  LFPAD(iangle,:,:,:) = MFPAD
                  
                  iangle = iangle + 1
            
                  deallocate(MFDip, MFPAD)
               
               end do
               end do
               end do
            
               if (write_dipoles) call write_LFDip( dipoles, scat_angle_limits, ngrdproj, & 
               &                                    grid_alpha, grid_beta, grid_gamma, ion, ine, LFDip )
               
               call write_LFPAD( dipoles, scat_angle_limits, ngrdproj, &
               &                 grid_alpha, grid_beta, grid_gamma, ion, ine, LFPAD )
               
               deallocate(LFDip, LFPAD)
            
            end do
         
         end do
         
         if (write_dipoles) write( iwrite,  '(/, " LFDip calculated. ")') 
         write( iwrite,  '(/, " LFPAD calculated. ")') 
              
      end if     

      return
!     format statements
 2001 format(/,5X,'Problems reading the namelist ORIENT',/)
   end subroutine
      
   subroutine calculate_oriented_dipoles( dipoles, ion_state, neutral_state, alpha, beta, gamma, Ylm, MFDip)
      implicit none
      
!     Arguments
      type(moments)             :: dipoles
      integer                   :: ion_state, neutral_state
      real(idp)                 :: alpha, beta, gamma      
      complex(idp)              :: Ylm(:,:)
      complex(idp), allocatable :: MFDip(:,:,:)      
!     Local

      
      integer                   :: lmax, no_channels, no_components, no_energies, no_ion_states, &
      &                            no_neutral_states, no_theta_phi
      integer                   :: i, it, ip, ien, ion, ine
      complex(idp), allocatable :: rot_Ylm(:,:), dk(:,:), R(:,:),  R1(:,:), U(:,:), Slm(:,:)
   
      associate ( D      => dipoles % elements )
      
      lmax               =  maxval( dipoles % scattering_states % channels % lchl )
      no_channels        =  size( D, 1 )
      no_components      =  size( D, 2 )
      no_energies        =  size( D, 3 )
      no_theta_phi       =  size(Ylm, 1)

      allocate( MFDip( no_theta_phi, no_components, no_energies) )      
      allocate( dk(no_theta_phi, no_components))

!     Create real or complex wigner D matrix elements depending on the basis type

      if (dipoles % basis_type_photon .eq. 'Ylm') then
      
         call a_wigner_d(1, alpha, beta, gamma, R1)
      
      else
      
         call a_re_wigner_d(1, alpha, beta, gamma, R1)
      
      end if
      
      if (dipoles % basis_type_electron .eq. 'Ylm') then
      
         call a_wigner_d(lmax, alpha, beta, gamma, R)
      
      else
      
         call a_re_wigner_d(lmax, alpha, beta, gamma, R)
      
      end if     

!     Rotate the spherical harmonics

!      allocate( rot_Ylm, mold = Ylm ) !F2008
      allocate( rot_Ylm( size(Ylm,1), size(Ylm,2) ) )
            
      call gemm( Ylm, R, rot_Ylm )          

!     Mulitply the rotated spherical harmonics into the dipoles and then 
!     rotate the dipole components 
         
      do ien = 1, no_energies
            
         call  gemm( rot_Ylm(:,:), D(:,:, ien, ion_state, neutral_state), dk )
               
         call  gemm( dk, R1(2:4,2:4), mfdip(:,:,ien), transb='T')

      end do
      
      end associate   
   end subroutine
      
   subroutine calculate_MFPAD(dipoles, MFDip, ion_state, neutral_state, MFPAD)
      implicit none
 
!     Argument variables
      type(moments)          :: dipoles
      complex(kind=idp)      :: MFDip(:,:,:)
      real(idp), allocatable :: MFPAD(:,:,:)
      
!     Local variables
      integer ::  no_ion_states, no_neutral_states, no_components, no_energies, no_angles, &
      &           ine, ion, ien, ico, ian, ion_state, neutral_state
      real(kind=idp) :: Aconst, photon_energy, Ip

      intent(in) :: dipoles
      
      no_angles            = size(MFDip,1)
      no_components        = size(MFDip,2)
      no_energies          = size(MFDip,3)
!      no_ion_states        = size(MFDip,4)
!      no_neutral_states    = size(MFDip,5)
        
      allocate( MFPAD(no_angles, no_components, no_energies) )
      
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )  
     
            
      Ip = E_ion(1) - E_neut( neutral_state ) 
    
      do  ien = 1, no_energies

         photon_energy = E_elec(ien) + Ip  
         Aconst        = 4*(pi**2)*alpha*photon_energy*convert_au_to_megabarns
               
         MFPAD(:,:,ien) = abs( MFDip(:,:,ien) )**2 * Aconst
               
      end do

      
      end associate
   
      return
! 1010 format(3(E20.5,x))
   end subroutine
   
   
   subroutine calculate_LFDip( dipoles, electron_direction, mol_orientation, steps_elec_direction, &
   &                           steps_mol_orientation, ion_state, neutral_state, MFDip)
      implicit none
!     Arguments
      type(moments)             :: dipoles
      integer                   :: steps_elec_direction(2),  steps_mol_orientation(3)
      real(idp)                 :: electron_direction(4),  mol_orientation(6)
      integer                   :: ion_state, neutral_state
      complex(idp), allocatable :: MFDip(:,:,:)      
!     Local

      
      integer                   :: lmax,no_channels, no_components, no_energies, no_ion_states, &
      &                            no_neutral_states, no_theta, no_phi, no_alpha, no_beta, no_gamma
      integer                   :: i, it, ip, ien, ion, ine, ia
      real(idp)                 :: alpha, beta, gamma
      complex(idp), allocatable :: Ylm(:,:), dk(:,:), R(:,:,:), R1(:,:,:), rot_Ylm(:,:)
   
      associate ( D      =>    dipoles % elements)
      
      lmax               =  maxval( dipoles % scattering_states % channels % lchl )
      no_channels        =  size( D, 1 )
      no_components      =  size( D, 2 )
      no_energies        =  size( D, 3 )
      
      no_theta   = steps_elec_direction(1)
      no_phi     = steps_elec_direction(2)
      no_alpha   = steps_mol_orientation(1)
      no_beta    = steps_mol_orientation(2)
      no_gamma   = steps_mol_orientation(3)

      allocate( MFDip( no_theta*no_phi, no_components, no_energies) )      
      allocate( dk(no_theta*no_phi, no_components))

      
      call grid_sp_harm(lmax, steps_elec_direction, electron_direction, Ylm, dipoles % basis_type_electron )
      
      call grid_wigner_D(lmax, steps_mol_orientation, mol_orientation, R, dipoles % basis_type_electron )
      
      call grid_wigner_D(1, steps_mol_orientation, mol_orientation, R1, dipoles % basis_type_photon )
      
!      allocate(rot_Ylm, mold = Ylm ) !F2008 standard
      allocate( rot_Ylm( size(Ylm,1), size(Ylm,1) ) )
            
      do ia = 1, no_alpha*no_beta*no_gamma

         call gemm(R(:,:,ia), Ylm, rot_Ylm )
         
         do ien = 1, no_energies
            
            call  gemm( rot_Ylm(:,:), D(:,:, ien, ion_state, neutral_state), dk )
               
            call  gemm( dk, R1(2:4,2:4, ia), mfdip(:,:,ien), transb='T')

         end do
         
      end do
      
      end associate   
   end subroutine  
       
   subroutine calculate_MFDip( dipoles, electron_direction, mol_orientation, steps_elec_direction, &
   &                           steps_mol_orientation, ion_state, neutral_state, MFDip)
      implicit none
!     Arguments
      type(moments)             :: dipoles
      integer                   :: steps_elec_direction(2),  steps_mol_orientation(3)
      real(idp)                 :: electron_direction(4),  mol_orientation(6)
      integer                   :: ion_state, neutral_state
      complex(idp), allocatable :: MFDip(:,:,:)      
!     Local

      
      integer                   :: lmax,no_channels, no_components, no_energies, no_ion_states, &
      &                            no_neutral_states, no_theta, no_phi, no_alpha, no_beta, no_gamma
      integer                   :: i, it, ip, ien, ion, ine
      real(idp)                 :: alpha, beta, gamma
      complex(idp), allocatable :: Ylm(:,:), dk(:,:),  R1(:,:), U(:,:), Slm(:,:)
   
      associate ( D      => dipoles % elements )
      
      lmax               =  maxval( dipoles % scattering_states % channels % lchl )
      no_channels        =  size( D, 1 )
      no_components      =  size( D, 2 )
      no_energies        =  size( D, 3 )
!      no_ion_states      =  size( D, 4 )
!      no_neutral_states  =  size( D, 5 )
      
      no_theta = steps_elec_direction(1)
      no_phi   = steps_elec_direction(2)

      allocate( MFDip( no_theta*no_phi, no_components, no_energies) )      
      allocate( dk(no_theta*no_phi, no_components))
      
      call grid_sp_harm(lmax, steps_elec_direction, electron_direction, Ylm, 'Ylm')
      
      alpha = mol_orientation(1)
      beta  = mol_orientation(2)
      gamma = mol_orientation(3)
      
      call a_wigner_d(1, alpha, beta, gamma, R1)     
      
      if (dipoles % basis_type_electron == 'Slm') then      
        
!        Would be more efficient to construct the real spherical harmonics directly
         call sph_basis_transform_matrix( U, lmax, 'Ylm' )
         allocate( Slm( no_theta*no_phi, no_channels ) )
         
         call gemm( Ylm, U, Slm, transb='T' )  
         Ylm = Slm   
         deallocate(Slm, U)
         
      end if
            
      if (dipoles % basis_type_photon == 'Slm') then
!        Would be more efficient to construct the real wigner d matrix elements directly      
         call sph_basis_transform_matrix(U, 1, 'Slm')
         
         R1 = matmul(R1, U)
         deallocate(U)
        
         call sph_basis_transform_matrix(U, 1, 'Ylm')
         R1 = matmul(U, R1)
         
      end if

         
      do ien = 1, no_energies
            
         call  gemm( Ylm(:,:), D(:,:, ien, ion_state, neutral_state), dk )
               
         call  gemm( dk, R1(2:4,2:4), mfdip(:,:,ien), transb='T')

      end do
      
      end associate   
   end subroutine
   
   subroutine calculate_cross_section_and_beta_parameters( dipoles, select_ion_states, select_neutral_states,&
   &                                                       xsec, beta_1, beta_2l, beta_2c)   
      implicit none
      
!     Modified by ZM to include calculation of the beta_1 parameter.
!     Arguments
      type(moments)          :: dipoles
      integer                :: select_ion_states(:), select_neutral_states(:)
      real(idp), allocatable :: xsec(:,:,:), beta_1(:,:,:), beta_2l(:,:,:), beta_2c(:,:,:)

!     Local variables 
      integer                   :: no_ion_states, no_neutral_states, no_energies, ion, ine, ien, &
      &                            Lbig,  no_ion_states_all, no_neutral_states_all, isne, ison     
      complex(idp), allocatable :: AL_zero(:), AL_one(:), AL_two(:), AL_two_c(:)
      real(idp)                 :: Ip, photon_energy, Aconst


      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies ) 

      no_ion_states_all     = size( E_ion  )
      no_neutral_states_all = size( E_neut )
      no_energies           = size( E_elec )
      
      no_ion_states         = size( select_ion_states )
      no_neutral_states     = size( select_neutral_states )
      
      allocate( AL_zero(no_energies), AL_one(no_energies), AL_two(no_energies), AL_two_c(no_energies) )
      
      allocate( xsec(no_energies, no_ion_states, no_neutral_states) )
      allocate( beta_1(no_energies, no_ion_states, no_neutral_states) )
      allocate( beta_2l(no_energies, no_ion_states, no_neutral_states) )
      allocate( beta_2c(no_energies, no_ion_states, no_neutral_states) )
      
      xsec   = 0._idp
      beta_1 = 0._idp
      beta_2l = 0._idp
      beta_2c = 0._idp
      
      
      do ine = 1, no_neutral_states_all
      
        if ( any(select_neutral_states .eq. ine) ) then
        
!         isne = findloc(select_neutral_states, ine, 1) !F2008 feature
         isne = maxloc(merge(1,0,select_neutral_states == ine),dim=1) !F2003 equivalent
      
         do ion = 1, no_ion_states_all
         
            if ( any(select_ion_states .eq. ion ) ) then
            
!            ison = findloc(select_ion_states, ion, 1) !F2008 feature
            ison = maxloc(merge(1,0,select_ion_states == ion), dim=1) !F2003 equivalent
         
            AL_zero        = 0._idp
            AL_one         = 0._idp           
            AL_two         = 0._idp
            AL_two_c       = 0._idp
            
            !>  @todo Check whether xsec and beta_2 depend on if the 
            !!       light polarization is circular or linear. (I think they actually might do)
            !!       Looks like beta_2 does. 

            !Linear polarization
            call make_ritchie_AL_coeff( dipoles, ion, ine, 0, 0, AL_zero )
            call make_ritchie_AL_coeff( dipoles, ion, ine, 2, 0, AL_two  )
            
            !Circular polarization            
            call make_ritchie_AL_coeff( dipoles, ion, ine, 1, 1, AL_one    )
            call make_ritchie_AL_coeff( dipoles, ion, ine, 2, 1, AL_two_c  )

            Ip = E_ion(1) - E_neut( ine )  

            do ien = 1, no_energies

               photon_energy = E_elec(ien) + Ip
               Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
         
               if (abs(AL_zero(ien)) .ne. 0) then
               
                  xsec(ien, ison, isne)    = real( AL_zero(ien), kind=idp ) * Aconst !Normalize to produce the integral cross section
                  
                  beta_1(ien, ison, isne)  = real( AL_one(ien)/AL_zero(ien), kind=idp )
               
                  beta_2l(ien, ison, isne) = real( AL_two(ien)/AL_zero(ien), kind=idp )
                  
                  beta_2c(ien, ison, isne) = real( AL_two_c(ien)/AL_zero(ien), kind=idp )
    
               end if
               
            end do
            
            end if
            
         end do
         
         end if
         
      end do

      end associate
   
   end subroutine

!
!  make_ritchie_AL_coeff
! 
!  B. Ritchie, PRA 13, 1411, (1976): formulae 11a,11b
!
!  On input p stands for the desired photon polarization p = -1,0,1 (left circ., linear, right circ.)
!  The calculated AL_coefficient does not include the normalization factor 
!  Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns.
!  This can be applied outside this routine to ensure e.g. that the L=0 coefficient gives the 
!  correct integral cross section (see calculate_orientation_averaged_asymmetry_parameter). 
!     
   subroutine make_ritchie_AL_coeff( dipoles, ion_state, neutral_state, Lbig, p, AL_coefficient )
      implicit none
 
!     Arguments
      integer           :: ion_state, neutral_state, Lbig, p
      type(moments)     :: dipoles  
      complex(kind=idp) :: AL_coefficient(:)

!     Local
      integer                        :: no_energies, no_components, no_ion_states, &
      &                                 no_partials, no_neutral_states
      
      integer                        :: lmax, l, lp, ien, m, mp, ilpmp, ilm, q, qp
      
      real(kind=idp)                 :: k_final, charge, eta, pconst, cf_polarization, cf_angmom, &
      &                                 cf_space,cf_photon, coupling, Aconst, photon_energy, Ip,  &
      &                                 E_chan_thresh
      
      complex(kind=idp)              :: coeff_one, dipole_product
      
      complex(kind=icp), allocatable :: energy_cf(:)

      charge           = dipoles % mol % ion_charge
      AL_coefficient   = 0.0_idp
      
      associate ( D      => dipoles % elements, &
      &           E_elec => dipoles % scattering_states % energies, &
      &           E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &           E_neut => dipoles % neutral_states % energies ) 

      no_components      =  size( dipoles % elements, 2 )
      no_energies        =  size( dipoles % elements, 3 )
      no_ion_states      =  size( dipoles % elements, 4 )
      no_neutral_states  =  size( dipoles % elements, 5 )
      
      allocate( energy_cf(no_energies) )
      
      lmax = maxval( dipoles % scattering_states % channels % lchl)

      cf_polarization = threej( 2*1, 2*p, 2*1, -2*p, 2*Lbig, 0) !set p=+-1 on input for circular polarization
      
      !Ip = E_ion(ion_state) - E_neut(neutral_state)
      
      do l = 0, lmax
      
         do lp = 0, lmax

            cf_angmom = threej( 2*l, 0, 2*lp, 0, 2*Lbig, 0 ) * (2*Lbig + 1.0_idp) &
            &         * sqrt( 2*l + 1.0_idp ) * sqrt( 2*lp + 1.0_idp )
            
            if ( cf_angmom .eq. 0.0_idp ) cycle

            energy_cf = 0.0_idp
            
            do ien = 1, no_energies
            
               E_chan_thresh = E_ion(ion_state) - E_ion(1) ! Electron energies are relative to the
                                                           ! ground state of the ion.
            
               if ( E_elec(ien) .ge.  E_chan_thresh ) then
               
                  k_final = sqrt( 2.0_idp * (E_elec(ien) - E_chan_thresh) )
               
                  if ( k_final .le. 0.0_idp ) exit
               
                  eta            = -charge/k_final                  
                  coeff_one      =  exp( eye*( CPHAZ(l,eta,6) - CPHAZ(lp,eta,6) ) ) * eye**(lp-l)  !< @todo Multiply coulomb phase into dipoles from the beginning                
                  !photon_energy  =  E_elec(ien)+Ip
                  !Aconst         =  1.0_idp !(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
                  energy_cf(ien) =  coeff_one ! * Aconst
               
               end if
               
            end do

            do m = -l, l

               ilm = lm2i(l,m)

               do mp = -lp, lp

                  cf_space = threej( 2*l, 2*m, 2*lp, -2*mp, 2*Lbig, -2*(m-mp) )
                  
                  if (cf_space .eq. 0.0_idp) cycle

                  ilpmp = lm2i(lp,mp)

                  do q = -1, 1
                  
                     do qp = -1, 1

                        cf_photon = threej( 2*1, 2*q, 2*1, -2*qp, 2*Lbig, -2*(m-mp) )
                        
                        if (cf_photon .eq. 0.0_idp) cycle

                        coupling = (-1)**(p+q+m) * cf_angmom * cf_polarization * cf_space * cf_photon
                        
                        do ien = 1, no_energies
                        
                           dipole_product = D(ilm, 2+q, ien, ion_state, neutral_state) &
                                          * conjg( D(ilpmp, 2+qp, ien, ion_state, neutral_state) )
                                          
                           AL_coefficient(ien) = AL_coefficient(ien) + ( energy_cf(ien) & 
                           &                       * coupling * dipole_product )
                           
                        end do

                     end do
                     
                  end do
                   
               end do
               
            end do
            
         end do
         
      end do
      end associate
      
   end subroutine  make_ritchie_AL_coeff

   subroutine calculate_aligned_cross_section_and_beta_parameters( dipoles, select_ion_states, select_neutral_states,&
   &                                             en_start, en_end, KQ, AKQ, xsec, beta_LQl, beta_LQc, Lbig_max, perfect_aligned)
      implicit none
      
!     Arguments
      type(moments)          :: dipoles
      integer                :: select_ion_states(:), select_neutral_states(:)
      !Note that as opposed to the randomly oriented case the INDIVIDIAL beta_LQ coefficients need not be real.
      !Of course the full PAD summed over all beta_LQ contributions should be if the AKQ coefficients have been correctly determined.
      complex(idp), allocatable :: xsec(:,:,:,:), beta_LQl(:,:,:,:,:), beta_LQc(:,:,:,:,:)
      integer                :: KQ(:,:), Lbig_max, en_start, en_end
      complex(kind=idp)      :: AKQ(:,:)
      logical                :: perfect_aligned

!     Local variables 
      integer                   :: no_ion_states, no_neutral_states, no_energies, ion, ine, ien, &
      &                            no_ion_states_all, no_neutral_states_all, isne, ison, p, i
      complex(idp), allocatable :: BLQ_coefficient_l(:,:,:), BLQ_coefficient_c(:,:,:), polarization(:,:)
      real(idp)                 :: Ip, photon_energy, Aconst, theta
      integer                   :: K_max, n_LQ, n_t, LQ_ind, t


      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies ) 

      no_ion_states_all     = size( E_ion  )
      no_neutral_states_all = size( E_neut )
      no_energies           = size( E_elec )
      
      no_ion_states         = size( select_ion_states )
      no_neutral_states     = size( select_neutral_states )

      if (perfect_aligned) then

         n_t = 90
         Lbig_max = 2*maxval( dipoles % scattering_states % channels % lchl)
         n_LQ = (Lbig_max+1)**2 

         allocate(polarization(-1:1,n_t))

         !polarization = cos(theta)*z + sin(theta)*y
         do i = 0, n_t-1
            theta = pi / (n_t-1) * i
            polarization(-1,i+1) = sin(theta) * eye/sqrt(2.0_idp)
            polarization( 0,i+1) = cos(theta)
            polarization( 1,i+1) = sin(theta) * eye/sqrt(2.0_idp)
         enddo

      else

         K_max = maxval(KQ(1,:))
         Lbig_max = K_max + 2
         n_LQ = (Lbig_max+1)**2 
   
         n_t = size(AKQ,1)

      endif

      if (en_start > no_energies) then
         print *,'incorrect value of en_start > no_energies'
         stop 1
      endif

      if (en_end > no_energies) then
         print *,'incorrect value of en_end > no_energies'
         stop 1
      endif

      if (en_end <= 0) en_end = no_energies

      write(*,'(/," Selected range of energy indices: [",i0,",",i0,"]")') en_start, en_end

      !@todo The allocation for no_ion_states is an overkill in case only a subset of states is actually selected
      allocate( xsec(n_t, en_start:en_end, no_ion_states, no_neutral_states) )
      allocate( beta_LQl(n_t, en_start:en_end, n_LQ, no_ion_states, no_neutral_states) )
      allocate( beta_LQc(n_t, en_start:en_end, n_LQ, no_ion_states, no_neutral_states) )

      xsec     = 0._idp
      beta_LQl = 0._idp
      beta_LQc = 0._idp           
      
      do ine = 1, no_neutral_states_all
      
        if ( any(select_neutral_states .eq. ine) ) then
        
!         isne = findloc(select_neutral_states, ine, 1) !F2008 feature
         isne = maxloc(merge(1,0,select_neutral_states == ine),dim=1) !F2003 equivalent

         Ip = E_ion(1) - E_neut( ine )  
      
         do ion = 1, no_ion_states_all
         
            if ( any(select_ion_states .eq. ion ) ) then
            
!            ison = findloc(select_ion_states, ion, 1) !F2008 feature
            ison = maxloc(merge(1,0,select_ion_states == ion), dim=1) !F2003 equivalent

            if (perfect_aligned) then

               if (allocated(BLQ_coefficient_c)) deallocate(BLQ_coefficient_c)
               
               write(*,'(" Computing Beta_LQ coefficients for perfectly aligned case and ion state: ",i0)') ion
               call make_perfect_aligned_BLQ_coeff( dipoles, en_start, en_end, ion, ine, polarization, n_t, BLQ_coefficient_l )

               allocate(BLQ_coefficient_c,source=BLQ_coefficient_l)

            else

               !Linear polarization
               write(*,'(" Computing Beta_LQ coefficients for linearly polarized light and ion state: ",i0)') ion
               p = 0
               call make_aligned_BLQ_coeff( dipoles, en_start, en_end, ion, ine, KQ, AKQ, p, BLQ_coefficient_l )
               
               !Circular polarization            
               write(*,'(" Computing Beta_LQ coefficients for circularly polarized light and ion state: ",i0)') ion
               p = 1
               call make_aligned_BLQ_coeff( dipoles, en_start, en_end, ion, ine, KQ, AKQ, p, BLQ_coefficient_c )

            endif

            if (n_LQ /= size(BLQ_coefficient_l,3)) then
               write(*, '("Unexpected size of the BLQ_coefficient_ arrays")')
               stop 1
            endif

            !Cross section
            !$OMP PARALLEL DEFAULT(SHARED) PRIVATE(ien, t, photon_energy, Aconst)
            !$OMP DO
            do ien = en_start, en_end
   
               photon_energy = E_elec(ien) + Ip
               Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
  
               do t = 1, n_t
                  xsec(t, ien, ison, isne) = BLQ_coefficient_l(t,ien,1) * Aconst !Normalize to produce the integral cross section
               enddo !t
               
            end do !ien
            !$OMP ENDDO
            !$OMP END PARALLEL

            !Beta parameters
            do LQ_ind = 1, n_LQ

               !$OMP PARALLEL DEFAULT(SHARED) PRIVATE(ien, t)
               !$OMP DO
               do ien = en_start, en_end
   
                  do t = 1, n_t
           
                     !beta_{L=0,Q=0} corresponds to LQ_ind == 1
                     if (abs(BLQ_coefficient_l(t,ien,1)) .ne. 0) then
                     
                        beta_LQl(t, ien, LQ_ind, ison, isne)  = BLQ_coefficient_l(t,ien,LQ_ind)/BLQ_coefficient_l(t,ien,1)
                        beta_LQc(t, ien, LQ_ind, ison, isne)  = BLQ_coefficient_c(t,ien,LQ_ind)/BLQ_coefficient_l(t,ien,1)
          
                     end if
   
                  enddo !t
                  
               end do !ien
               !$OMP ENDDO
               !$OMP END PARALLEL

            end do !LQ_ind
            
            end if
            
         end do
         
         end if
         
      end do

      end associate

   end subroutine calculate_aligned_cross_section_and_beta_parameters

!  On input p stands for the desired photon polarization p = -1,0,1 (left circ., linear, right circ.)
!  The calculated BLQ_coefficient does not include the normalization factor 
!  Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns.
!
!  Input of the alignment distribution:
!  ------------------------------------
!  * Array KQ:
!  K = KQ(1,:)
!  Q = KQ(2,:)
!  2nd dimension of KQ runs over all K,Q pairs of indices characterizing the alignment distribution
!  * Array AKQ(t,KQ) contains the actual time-dependent complex AKQ coefficients for the K,Q indices
!    given in KQ.
!    1st dimension of AKQ: time-steps
!    2nd dimension of AKQ: KQ-indices
!
!  Output:
!  -------
!  Array BLQ_coefficient(t,energy,KQ) containing all allowed values of the Beta parameters:
!  1st dimension: time-steps
!  2nd dimension: photon/photoelectron energy
!  3rd dimension: KQ pairs in the order given on input in the KQ array 
   subroutine make_aligned_BLQ_coeff( dipoles, en_start, en_end, ion_state, neutral_state, KQ, AKQ, p, BLQ_coefficient )
      implicit none
 
!     Arguments
      integer           :: ion_state, neutral_state, p
      type(moments)     :: dipoles  
      integer           :: KQ(:,:), en_start, en_end
      complex(kind=idp) :: AKQ(:,:)
      complex(kind=idp), allocatable :: BLQ_coefficient(:,:,:)

!     Local
      integer                        :: no_energies, no_components, no_ion_states, &
      &                                 no_partials, no_neutral_states
      
      integer                        :: lmax, l, lp, ien, m, mp, ilpmp, ilm, q, qp, n_kq, Lbig, Qbig, &
      &                                 LQ_ind, Lbig_max, K_max, K1, KQ_ind, K, n_t, t
      
      real(kind=idp)                 :: k_final, charge, eta, pconst, cf_polarization, cf_angmom, &
      &                                 cf_space,cf_photon, coupling, Aconst, photon_energy, Ip,  &
      &                                 E_chan_thresh, cf_AKQ
      
      complex(kind=idp)              :: coeff_one, dipole_product
      
      complex(kind=icp), allocatable :: energy_cf(:)

      charge           = dipoles % mol % ion_charge
      
      associate ( D      => dipoles % elements, &
      &           E_elec => dipoles % scattering_states % energies, &
      &           E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &           E_neut => dipoles % neutral_states % energies ) 

      no_components      =  size( dipoles % elements, 2 )
      no_energies        =  size( dipoles % elements, 3 )
      no_ion_states      =  size( dipoles % elements, 4 )
      no_neutral_states  =  size( dipoles % elements, 5 )
      
      allocate( energy_cf(en_start:en_end) )
      
      lmax = maxval( dipoles % scattering_states % channels % lchl)
      n_t = size(AKQ,1) !number of time-steps
      n_kq = size(AKQ,2)

      if (n_kq /= size(KQ,2)) then
         print *,'make_aligned_BLQ_coeff: inconsistent dimensions of KQ and AKQ on input.'
         stop 1
      endif

      K_max = maxval(KQ(1,:))
      Lbig_max = K_max + 2

      if (allocated(BLQ_coefficient)) deallocate(BLQ_coefficient)
      allocate(BLQ_coefficient(n_t,en_start:en_end,(Lbig_max+1)**2))

      BLQ_coefficient   = 0.0_idp
      LQ_ind = 0

      do Lbig = 0, Lbig_max

         do K1 = 0, 2

            cf_polarization = (2*K1+1)*threej( 2*1, 2*p, 2*1, -2*p, 2*K1, 0) !set p=+-1 on input for circular polarization

            if (cf_polarization == 0.0_idp) cycle

            do KQ_ind = 1, n_kq

               !Note that here I am using the A_{K,Q} coefficient as opposed to A_{K,-Q} which the formula prescribes. This means
               !that I have swapped Q -> -Q and I am calculating the coefficient for the Y_{L,-Q} spherical harmonic instead
               !as is indeed seen from the calculation of LQ_ind below.
               !The multiplication by the A_{K,Q} coefficient is postponed until the inner-most loop over time 
               K    = KQ(1,KQ_ind)
               Qbig = KQ(2,KQ_ind)

               LQ_ind = Lbig**2 + Lbig - Qbig + 1 !index of the Y_{L,-Q} spherical harmonic of photoelectron momentum

               cf_AKQ = sqrt(2*K + 1.0_idp)*threej( 2*K1, 2*0, 2*Lbig, -2*Qbig, 2*K, 2*Qbig)

               if (cf_AKQ == 0.0_idp) cycle

               do l = 0, lmax
               
                  do lp = 0, lmax
         
                     cf_angmom = threej( 2*l, 0, 2*lp, 0, 2*Lbig, 0 ) * (2*Lbig + 1.0_idp) &
                     &         * sqrt( 2*l + 1.0_idp ) * sqrt( 2*lp + 1.0_idp )
                     
                     if ( cf_angmom == 0.0_idp ) cycle
         
                     energy_cf = 0.0_idp
                     
                     do ien = en_start, en_end
                     
                        E_chan_thresh = E_ion(ion_state) - E_ion(1) ! Electron energies are relative to the
                                                                    ! ground state of the ion.
                     
                        if ( E_elec(ien) .ge.  E_chan_thresh ) then
                        
                           k_final = sqrt( 2.0_idp * (E_elec(ien) - E_chan_thresh) )
                        
                           if ( k_final <= 0.0_idp ) exit
                        
                           eta            = -charge/k_final                  
                           coeff_one      =  exp( eye*( CPHAZ(l,eta,6) - CPHAZ(lp,eta,6) ) ) * eye**(lp-l)  !< @todo Multiply coulomb phase into dipoles from the beginning                
                           !photon_energy  =  E_elec(ien)+Ip
                           !Aconst         =  1.0_idp !(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
                           energy_cf(ien) =  coeff_one ! * Aconst
                        
                        end if
                        
                     end do
         
                     do m = -l, l
         
                        ilm = lm2i(l,m)
         
                        do mp = -lp, lp
        
                           !mu = mp-m 
                           !cf_space = threej( 2*l, 2*m, 2*lp, -2*mp, 2*Lbig, 2*mu )
                           
                           !if (cf_space == 0.0_idp) cycle
         
                           ilpmp = lm2i(lp,mp)
         
                           do q = -1, 1
                           
                              do qp = -1, 1
         
                                 cf_photon = threej( 2*1, 2*q, 2*1, -2*qp, 2*K1, 2*(qp-q) ) &
                                &          * threej( 2*K1, 2*(q-qp), 2*Lbig, 2*(qp-q), 2*K, 2*0 )
                                 if (cf_photon == 0.0_idp) cycle

                                 cf_space = threej( 2*l, 2*m, 2*lp, -2*mp, 2*Lbig, 2*(qp-q) )
                                 if (cf_space == 0.0_idp) cycle
         
                                 coupling = (-1)**(p+q+mp) * cf_angmom * cf_polarization * cf_space * cf_photon * cf_AKQ

                                 !$OMP PARALLEL DEFAULT(SHARED) PRIVATE(ien, dipole_product, t)
                                 !$OMP DO 
                                 do ien = en_start, en_end
                                 
                                    dipole_product = D(ilm, 2+q, ien, ion_state, neutral_state) &
                                                   * conjg( D(ilpmp, 2+qp, ien, ion_state, neutral_state) )
                     
                                    do t = 1, n_t                              

                                       BLQ_coefficient(t, ien,LQ_ind) = BLQ_coefficient(t,ien,LQ_ind) + ( energy_cf(ien) & 
                                       &                       * coupling * dipole_product * AKQ(t, KQ_ind)) /( (2*K+1)/(8*pi**2))

                                    enddo !t
                                    
                                 end do
                                 !$OMP ENDDO
                                 !$OMP END PARALLEL
         
                              end do
                              
                           end do
                            
                        end do
                        
                     end do
                     
                  end do
                  
               end do



            enddo !KQ_ind

         enddo !K1
      enddo !L_big

      end associate
      
   end subroutine  make_aligned_BLQ_coeff

!  On input p stands for the desired photon polarization p = -1,0,1 (left circ., linear, right circ.)
!  The calculated BLQ_coefficient does not include the normalization factor 
!  Aconst=(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns.
!
!  Input:
!  n_t                      number of photon polarizations on input
!  polarization(-1:1,1:n_t) complex polarization vectors of the photon in spherical coordinates for which to perform the calculations
!
!  Output:
!  -------
!  Array BLQ_coefficient(energy,KQ) containing all allowed values of the Beta parameters:
!  1st dimension: polarization vectors
!  2nd dimension: photon/photoelectron energy
!  3rd dimension: KQ pairs with standard orderding for K = 0,...,2*L_max, where L_max is the maximum pw.
   subroutine make_perfect_aligned_BLQ_coeff(dipoles,en_start,en_end,ion_state,neutral_state,polarization,n_t,BLQ_coefficient)
      implicit none
 
!     Arguments
      integer           :: ion_state, neutral_state, n_t
      type(moments)     :: dipoles  
      integer           :: en_start, en_end
      complex(kind=idp) :: polarization(-1:1,1:n_t)
      complex(kind=idp), allocatable :: BLQ_coefficient(:,:,:)

!     Local
      integer                        :: no_energies, no_components, no_ion_states, &
      &                                 no_partials, no_neutral_states
      
      integer                        :: lmax, l, lp, ien, m, mp, ilpmp, ilm, q, qp, Qbig, &
      &                                 KQ_ind, K_max, K, t
      
      real(kind=idp)                 :: k_final, charge, eta, cf_angmom, &
      &                                 cf_space, Aconst, photon_energy, Ip,  &
      &                                 E_chan_thresh
      
      complex(kind=idp)              :: coeff_one, dipole_product, cf_pol, coupling
      
      complex(kind=icp), allocatable :: energy_cf(:)

      charge           = dipoles % mol % ion_charge
      
      associate ( D      => dipoles % elements, &
      &           E_elec => dipoles % scattering_states % energies, &
      &           E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &           E_neut => dipoles % neutral_states % energies ) 

      no_components      =  size( dipoles % elements, 2 )
      no_energies        =  size( dipoles % elements, 3 )
      no_ion_states      =  size( dipoles % elements, 4 )
      no_neutral_states  =  size( dipoles % elements, 5 )
      
      allocate( energy_cf(en_start:en_end) )
      
      lmax = maxval( dipoles % scattering_states % channels % lchl)
      K_max = 2*lmax

      if (allocated(BLQ_coefficient)) deallocate(BLQ_coefficient)
      allocate(BLQ_coefficient(n_t,en_start:en_end,(K_max+1)**2))

      BLQ_coefficient   = 0.0_idp
      KQ_ind = 0

      do K = 0, K_max

         do Qbig = -K, K

            KQ_ind = KQ_ind + 1
            
            do l = 0, lmax
            
               do lp = 0, lmax
         
                  cf_angmom = threej( 2*l, 0, 2*lp, 0, 2*K, 0 ) * &
                              sqrt((2*K + 1.0_idp)*(2*l + 1.0_idp)*(2*lp + 1.0_idp)/(4*pi))
                  
                  if ( cf_angmom == 0.0_idp ) cycle
         
                  energy_cf = 0.0_idp
                  
                  do ien = en_start, en_end
                  
                     E_chan_thresh = E_ion(ion_state) - E_ion(1) ! Electron energies are relative to the
                                                                 ! ground state of the ion.
                  
                     if ( E_elec(ien) .ge.  E_chan_thresh ) then
                     
                        k_final = sqrt( 2.0_idp * (E_elec(ien) - E_chan_thresh) )
                     
                        if ( k_final <= 0.0_idp ) exit
                     
                        eta            = -charge/k_final                  
                        coeff_one      =  exp( eye*( CPHAZ(l,eta,6) - CPHAZ(lp,eta,6) ) ) * eye**(lp-l)
                        !photon_energy  =  E_elec(ien)+Ip
                        !Aconst         =  1.0_idp !(4._idp) * (pi**2) * alpha * photon_energy*convert_au_to_megabarns
                        energy_cf(ien) =  coeff_one ! * Aconst
                     
                     end if
                     
                  end do
         
                  do m = -l, l
         
                     ilm = lm2i(l,m)
         
                     do mp = -lp, lp
        
                        cf_space = (-1)**m * threej( 2*l, -2*m, 2*lp, 2*mp, 2*K, 2*Qbig )
                        
                        if (cf_space == 0.0_idp) cycle
         
                        ilpmp = lm2i(lp,mp)
         
                        !$OMP PARALLEL DEFAULT(SHARED) PRIVATE(ien, dipole_product, t, cf_pol, coupling, q, qp)
                        !$OMP DO 
                        do ien = en_start, en_end

                           do q = -1, 1
                              do qp = -1, 1
                        
                                    dipole_product = D(ilm, 2+q, ien, ion_state, neutral_state) &
                                                   * conjg( D(ilpmp, 2+qp, ien, ion_state, neutral_state) )
                        
                                    do t = 1, n_t
            
                                       cf_pol = polarization(q,t)*conjg(polarization(qp,t))
                                       coupling = cf_pol * cf_space * cf_angmom
            
                                       BLQ_coefficient(t,ien,KQ_ind) = BLQ_coefficient(t,ien,KQ_ind) + &
                                                           ( energy_cf(ien) * coupling * dipole_product )
                                    
                                    enddo !t

                              end do !qp
                           end do !q

                        end do
                        !$OMP ENDDO
                        !$OMP END PARALLEL

                        
                     end do !mp
                     
                  end do !m
                  
               end do !lp

            enddo !l

         enddo !Q_big
      enddo !K

      end associate
      
   end subroutine  make_perfect_aligned_BLQ_coeff
   
   subroutine determine_states_selected( no_ion_states, select_ion_states, no_ion_states_all, &
   &                                     no_neutral_states, select_neutral_states, no_neutral_states_all)
      implicit none

!     Arguments      
      integer              :: no_ion_states, no_ion_states_all, no_neutral_states, no_neutral_states_all
      integer, allocatable :: select_ion_states(:), select_neutral_states(:)
!     Local      
      integer, allocatable :: tmp_arr(:)
      integer              :: i
   
!     Figure out how many and which states we should calculate oriented observables for.      
      if (no_ion_states .eq. 0) then
      
         no_ion_states = no_ion_states_all
        
         select_ion_states     = [ (i,i=1,no_ion_states,1) ]
      
      else if(no_ion_states .gt. 0) then

         if (select_ion_states(1) .eq. 0) then
             
             deallocate(select_ion_states)
             allocate( tmp_arr(no_ion_states) )
             tmp_arr = [ (i,i=1,no_ion_states,1) ]
          
             call move_alloc( tmp_arr, select_ion_states )

         else
         
             allocate( tmp_arr(no_ion_states) )     
             tmp_arr = select_ion_states(1:no_ion_states)             
             deallocate(select_ion_states)
             call move_alloc( tmp_arr, select_ion_states )
         
         end if
         
      end if

      if (no_neutral_states .eq. 0) then
      
         no_neutral_states = no_neutral_states_all
         select_neutral_states     = [ (i,i=1,no_neutral_states,1) ]
      
      else if(no_neutral_states .gt. 0) then
      
         if (select_neutral_states(1) .eq. 0) then
             
             deallocate(select_neutral_states)
             allocate( tmp_arr(no_neutral_states) )
             tmp_arr = [ (i,i=1,no_neutral_states,1) ]
             
             call move_alloc( tmp_arr, select_neutral_states )
             
         else
         
             allocate( tmp_arr(no_neutral_states) )         
             tmp_arr = select_neutral_states(1:no_neutral_states)
             deallocate(select_neutral_states)
             call move_alloc( tmp_arr, select_neutral_states )
         
         end if
         
      end if           
    
   end subroutine
   
!  Construct $i^{-l}e^(i\sigma_l) and multiply into dipoles.
!  ---------------------------------------------------------  
   subroutine multiply_in_coulomb_phase_factor(dipoles)
      implicit none
      
      type(moments) :: dipoles
      
      integer      :: lmax, no_channels, no_components, no_energies, no_ion_states, i, l, m, ien, ion
      real(idp)    :: E_chan_thresh, k_final, eta
      complex(idp) :: coulomb_phase

      associate ( D      => dipoles % elements, &
      &           E_elec => dipoles % scattering_states % energies, &
      &           E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &           E_neut => dipoles % neutral_states % energies, &
      &           Z      => dipoles % mol % ion_charge ) 

      lmax               =  maxval( dipoles % scattering_states % channels % lchl )
      no_channels        =  size( D, 1 )
      no_components      =  size( D, 2 )
      no_energies        =  size( D, 3 )
      no_ion_states      =  size( D, 4 )


      do ion = 1, no_ion_states
         do ien = 1, no_energies
            
            E_chan_thresh = E_ion(ion) - E_ion(1)          ! Electron energies are relative to the
                                                           ! ground state of the ion.
            if ( E_elec(ien) .ge.  E_chan_thresh ) then
               
               k_final = sqrt( 2.0_idp * (E_elec(ien) - E_chan_thresh) )
               
               if ( k_final .le. 0.0_idp ) exit
               
               eta  = -Z/k_final  
               
               i=1
               do l = 0, lmax
               
                  coulomb_phase = (-eye)**l * exp( eye * CPHAZ(l,eta,6) )
                  
                  do m = -l, l
               
                     D(i,:,ien,ion,:) = D(i,:,ien,ion,:) * coulomb_phase
                     i = i + 1
                  
                  end do
               end do
            
            end if
            
         end do      
      end do
      
      end associate   
   
   end subroutine

end module
