! Copyright 2019
!
! Alex G. Harvey with contributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
module dipelm_io

   use dipelm_defs
   use dipelm_types
   use dipelm_special_functions
   use photo_outerio, only: read_pw_dipoles!, write_pw_dipoles
   
   implicit none
   

contains

   subroutine read_moments(lu_pw_dipoles, format_pw_dipoles, nset_pw_dipoles, dipoles, iprint, &
   &                       iwrite, ifail )
      use dipelm_defs, only: idp, maxprop_par
      use photo_outerio, only: read_pw_dipoles!, write_pw_dipoles
      
      implicit none

!     Arguments       
      integer, intent(in)           :: lu_pw_dipoles(maxprop_par), nset_pw_dipoles(maxprop_par) 
      integer                       :: iprint, iwrite, ifail
      character(len=11), intent(in) :: format_pw_dipoles
      type(moments)                 :: dipoles, dipoles_tmp


!     Local      
      type(moments), allocatable     :: dm(:)
      integer           :: no_scat_syms, ir, i, j, no_ion_states, no_neutral_states, no_chan
      
      character(len=80) ::  title
      integer           :: lmax_property
      real(idp), allocatable :: re_pw_dipoles(:,:,:,:), im_pw_dipoles(:,:,:,:)
      real(idp) :: target_energy
      
!     Assumes each scattering syms corresponds to a single file (i.e. stacking different syms
!     in different sets on the same file is not implemented yet)
      
      no_scat_syms = count(lu_pw_dipoles .ne. 0)
      allocate( dm(no_scat_syms) )
      
      
      do ir=1, no_scat_syms
      
         
         dm(ir) % scattering_states = new_continuum(1) ! We will read a single scattering sym at a time
         
         call read_pw_dipoles( lu_pw_dipoles(ir), nset_pw_dipoles(ir), format_pw_dipoles, title, &
         &                     dm(ir) % scattering_states % mtot(1), &
         &                     dm(ir) % scattering_states % stot(1), & 
         &                     dm(ir) % scattering_states % gutot(1), & 
         &                     dm(ir) % scattering_states % channels % ion_states % stot, &
         &                     dm(ir) % scattering_states % channels % ion_states % mtot, &
         &                     dm(ir) % scattering_states % channels % ion_states % gutot, &
         &                     dm(ir) % scattering_states % channels % ichl,  &
         &                     dm(ir) % scattering_states % channels % lchl, &
         &                     dm(ir) % scattering_states % channels % mchl, &
         &                     dm(ir) % scattering_states % channels % threshold_energies, &
         &                     dm(ir) % scattering_states % energies,  &
         &                     lmax_property, &
         &                     dm(ir) % scattering_states % moment_components, &
         &                     dm(ir) % neutral_states % energies, &
         &                     target_energy,&
         &                     re_pw_dipoles, im_pw_dipoles, iprint, iwrite, ifail )

!        Calculate ionic state energies         
         no_ion_states = size( dm(ir)%scattering_states%channels%ion_states%mtot )
         allocate( dm(ir)%scattering_states%channels%ion_states%energies(no_ion_states) )
         no_chan = size( dm(ir)%scattering_states%channels%ichl )
         
         associate ( E_ion => dm(ir) % scattering_states % channels % ion_states % energies, &
         &           E_rel => dm(ir) % scattering_states % channels % threshold_energies )
         
         E_rel = E_rel/2.0_idp    ! Note: Channel thresholds from rsolve ar e in Rydberg
                                  !       we convert to Hartree here
          
         do i = 1, no_chan
         
            j = dm(ir) % scattering_states % channels % ichl(i)
            
            E_ion(j) = target_energy + E_rel(i)
            
         end do
         
         end associate
      
!        Allocate space for neutral state symmetry description.
!        Note: this is not output by rsolve, so would need entering by hand
!        currently not implemented, and nothing actually needs it at the moment.
         
         no_neutral_states =  size( dm(ir) % neutral_states % energies )
         allocate ( dm(ir) % neutral_states % stot(no_neutral_states) )
         allocate ( dm(ir) % neutral_states % mtot(no_neutral_states) )
         allocate ( dm(ir) % neutral_states % gutot(no_neutral_states) )
         
         dm(ir) % neutral_states % stot  = 0
         dm(ir) % neutral_states % mtot  = 0
         dm(ir) % neutral_states % gutot = 0
         
!        Convert partial wave transition moments from rsolve format
!        and store.

         call dm(ir) % convert_moments( re_pw_dipoles, im_pw_dipoles )
        
         dm(ir) % basis_type_electron = 'Slm' ! rsolve uses a basis of real spherical harmonics.
         dm(ir) % basis_type_photon   = 'Slm' ! rsolve uses a basis of real spherical harmonics.
     
      end do
      
      dipoles= dm(1)
      
      do ir = 2, no_scat_syms
         
         dipoles = dipoles + dm(ir)
      
      end do
      
      write( iwrite,  '(/, " Partial wave transition moments read. ")')
      
      call dipoles%print(iwrite,iprint)      
        
   end subroutine read_moments
   
!  Backwards compatability with previous version of dipelm
!  Most of the namelist varibales are no longer needed.
    
   subroutine read_old_DIPTRANS( legacy_input, first_IP, no_targ_states, ibound_state, ngrdproj, ngrdalign, &
   &                              euler_angle_limits, scat_angle_limits )
      use dipelm_defs, only: idp, maxprop_par, max_states_for_dipoles
      implicit none
      
      logical   :: legacy_input
      integer   :: max_lcontinuum, no_targ_states, ibound_state, ngrdproj(2), ngrdalign(3), &
      &            ipunits, iplane, lebedev, luexpoints, idistribution, lab_component, ienergy_step, & 
      &            iuse_calculated_IP, lu_photo_xsec, lu_photo_dcs, iorient_averaging, &
      &            ipara_or_perp_light, icalc_asymmetry_param
      
      real(idp) :: first_IP, degeneracy(max_states_for_dipoles), euler_angle_limits(6), &
      &            scat_angle_limits(4), experimental_IP(max_states_for_dipoles), &
      &            degeneracy_threshold, phase_factor
      
      character(len=80)  :: molecule_name
      character(len=120) :: name 
            
      integer :: ierr, ifail   
   
      namelist /DIPTRANS/  name,&
      &                    max_lcontinuum,&
      &                    no_targ_states,&
      &                    ngrdproj,&
      &                    ngrdalign,&
      &                    ibound_state,& 
      &                    first_IP,&
      &                    ipunits,&
      &                    iplane,&
      &                    degeneracy,&
      &                    molecule_name,&
      &                    euler_angle_limits,&
      &                    scat_angle_limits,& 
      &                    lebedev,&
      &                    luexpoints,&
      &                    idistribution,& 
      &                    lab_component,&
      &                    ienergy_step,&
      &                    experimental_IP, &
      &                    iuse_calculated_IP, &
      &                    degeneracy_threshold, &
      &                    lu_photo_xsec, &
      &                    lu_photo_dcs,  &
      &                    iorient_averaging,&
      &                    ipara_or_perp_light, &  !Set to 1 for parallel light, 2 for perpindicular light
      &                    phase_factor, &
      &                    icalc_asymmetry_param                       

      rewind(5)
      ierr=0
      read(5,nml=DIPTRANS, iostat=ierr)
      if (ierr .ne. 0) then
      
         return
      
      else
      
         legacy_input = .true.
         
      end if   

!     Default namelist DIPTRANS values
      lu_photo_xsec         = 2220
      lu_photo_dcs          = 100
      max_lcontinuum        = 4 
      no_targ_states        = 0  !Including degenerate states.
      ibound_state          = 1
      ngrdproj              = (/ 60,4 /)
      ngrdalign             = (/ 1,1,1 /)
      ipunits               = 1
      iplane                = 1 !1: z-x plane, 2: z-y plane photodipoles. Need to request 4 phi points, e.g. ngrdproj=60,4
      iorient_averaging     = 0
      ipara_or_perp_light   = 0
      iuse_calculated_IP    = 0
      phase_factor          = 1._idp
      icalc_asymmetry_param = 1
      first_IP              = 0.0_idp
      molecule_name         = 'mol'
      euler_angle_limits    = (/ 0._idp,0._idp,0._idp,360._idp,180._idp,360._idp /)
      scat_angle_limits     = (/ 0._idp,0._idp,180._idp,360._idp /)
      lebedev               = 0
      luexpoints            = 100
      idistribution         = 1
      lab_component         = 1 !  x=1, y=2, z=3
      ienergy_step          = 1
      experimental_IP       =-1.0_idp
      degeneracy_threshold  = 1.d-5
     
      rewind(5)
      ierr=0
      read(5,nml=DIPTRANS, iostat=ierr)
      
      if (ierr .ne. 0) then
         return
      end if
 
      return
          
   end subroutine

   !> Reads in the Aligned Distribution Moments from a formatted file. The file format is similar to the other UKRmol-out programs
   !> such as T-matrix file where the channels are stored in the header and the matrices follows.
   !> Includes the possibility to trigger auto-generation of AKQs using the string 'autogenerate_perfect_aligned_lmax_', see code.
   subroutine read_ADM_KQ(iwrite, ADM_KQ_file, KQ, AKQ)
      implicit none
!     Arguments
      integer, intent(in)              :: iwrite
      character(len=255), intent(in)   :: ADM_KQ_file
      integer, allocatable             :: KQ(:,:)
      complex(idp), allocatable        :: AKQ(:,:)

      integer :: lu, n_t, n_KQ, i, j, err
      real(idp), allocatable :: im(:)
      character(len=*), parameter :: hack = 'autogenerate_perfect_aligned_lmax_'

         write(iwrite,'(/," read_ADM_KQ: reading in Aligned Distribution Moments A_{K,Q}(t)")')
         write(iwrite,'(/," Input file: ",a255)') ADM_KQ_file

         i = index(ADM_KQ_file,hack)
         if (i > 0) then

            i = len(hack)+1
            j = len_trim(ADM_KQ_file)

            !extract lmax from the end of the input string
            read(ADM_KQ_file(i:j),*) lu

            n_t = 90

            call generate_perfect_aligned_ADM_KQ(iwrite, lu, n_t, KQ, AKQ)

            return

         endif

         open(newunit = lu, file = ADM_KQ_file, status = 'old', form = 'FORMATTED')

         read(lu,*) n_t, n_KQ

         allocate(KQ(2,n_KQ), AKQ(n_t,N_KQ), im(n_t), stat=err)
         if (err /= 0) then
            print *,'memory allocation failed',n_t,N_KQ
            stop 1
         endif

         write(iwrite,'(" Number of time-steps, number of (K,Q) pairs: ",i0,",",i0)') n_t, n_KQ

         do i = 1, n_KQ
            read(lu,*) KQ(1,i), KQ(2,i)
            write(iwrite,'(" K=",i0," Q=",i0)') KQ(1,i), KQ(2,i)
         enddo

         !Real part of the A_KQ coefficients
         do i = 1, n_KQ
            read(lu,*) AKQ(1:n_t,i)
         enddo

         !Imaginary part of the A_KQ coefficients
         do i = 1, n_KQ
            read(lu,*) im(1:n_t)

            do j = 1, n_t
               AKQ(j,i) = cmplx(real(AKQ(j,i),idp), im(j))
            enddo
         enddo

         write(iwrite,'(" A_{K,Q}(t) coefficients read-in successfully",/)')

         close(lu)

   end subroutine read_ADM_KQ

   !> Generates the Aligned Distribution Moments for the perfectly aligned case with molecular axis axis along the direction (theta,phi=0),
   !> where theta varies between 0 and pi/2, including the end points, on a grid of n_t points.
   !> In this case the AKQ moments are simply spherical harmonics 
   subroutine generate_perfect_aligned_ADM_KQ(iwrite, lmax, n_t, KQ, AKQ)
      implicit none
!     Arguments
      integer, intent(in)              :: iwrite, lmax, n_t
      integer, allocatable             :: KQ(:,:)
      complex(idp), allocatable        :: AKQ(:,:)

      integer                    :: n_KQ, i, err, l, m
      real(idp)                  :: theta
      real(idp), parameter       :: phi = 0.0_idp
      complex(idp), allocatable  :: Ylm(:)

         n_KQ = (lmax+1)**2

         write(iwrite,'(/," Auto-generating AKQs for phi = 0 and theta in the range [0,pi].")')
         write(iwrite,'(" Number of theta points = ",i0)') n_t
         write(iwrite,'(" L_max = ",i0)') lmax

         allocate(KQ(2,n_KQ), AKQ(n_t,N_KQ), stat=err)
         if (err /= 0) then
            print *,'memory allocation failed',n_t,N_KQ
            stop 1
         endif

         i = 0 
         do l = 0, lmax
            do m = -l, l
               i = i + 1 
               KQ(1,i) = l
               KQ(2,i) = m
            enddo
         enddo

         do i = 0, n_t-1

            theta = pi * i/real(n_t-1,idp)

            call a_sp_harm(lmax, theta, phi, Ylm)

            AKQ(i+1,1:N_KQ) = Ylm(1:n_KQ)

            deallocate(Ylm)

         enddo !i

         write(iwrite,'(" done")')

   end subroutine generate_perfect_aligned_ADM_KQ

   !> Reads in the time-steps for the aligned calculations.
   !> Includes the possibility to trigger auto-generation of time axis using the string 'autogenerate_time_', see code.
   subroutine read_time_file(iwrite, time_file, time)
      implicit none
!     Arguments
      integer, intent(in)              :: iwrite
      character(len=255), intent(in)   :: time_file
      real(idp), allocatable           :: time(:)

      integer :: lu, n_t, i, j, err
      real(idp), allocatable :: im(:)
      character(len=*), parameter :: hack = 'autogenerate_time_'

         write(iwrite,'(/," read_time_file: reading in the time-steps")')
         write(iwrite,'(/," Input file: ",a255)') time_file

         i = index(time_file,hack)
         if (i > 0) then

            i = len(hack)+1
            j = len_trim(time_file)

            !extract n_t from the end of the input string
            read(time_file(i:j),*) n_t

            write(iwrite,'(" n_t = ",i0)') n_t

            allocate(time(n_t), stat=err)
            if (err /= 0) then
               print *,'memory allocation failed',n_t
               stop 1
            endif

            !theta values saved in the time array
            do i = 0, n_t-1
               time(i+1) = pi * i/real(n_t-1,idp)
            enddo
            
            return
         endif

         open(newunit = lu, file = time_file, status = 'old', form = 'FORMATTED')

         read(lu,*) n_t

         allocate(time(n_t), stat=err)
         if (err /= 0) then
            print *,'memory allocation failed',n_t
            stop 1
         endif

         write(iwrite,'(" Number of time-steps: ",i0)') n_t

         do i = 1, n_t
            read(lu,*) time(i)
         enddo

         write(iwrite,'(" Time-steps read-in successfully",/)')

         close(lu)

   end subroutine read_time_file
   
   subroutine write_cross_section_and_beta_parameters( dipoles, select_ion_states, select_neutral_states,&
   &                                                   output_style, xsec, beta_1, beta_2l, beta_2c )   
      implicit none

!     Arguments
      type(moments)    :: dipoles
      integer          :: select_ion_states(:), select_neutral_states(:)      
      real(idp)        :: xsec(:,:,:), beta_1(:,:,:), beta_2l(:,:,:), beta_2c(:,:,:)
      integer          :: output_style
      
!     Local
      integer            :: lu_xsec, lu_beta1, lu_beta2l, lu_beta2c, ien, ine, ion, no_ion_states, &
      &                     no_neutral_states, no_energies, ion_state, neutral_state, lu_total_xsec 
      real(idp)          :: Ip, photon_energy     
      character(len=132) :: path
      character(len=5)   :: str_ion_state, str_neutral_state
      
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )
      
      no_ion_states     = size( select_ion_states  )
      no_neutral_states = size( select_neutral_states )      
      no_energies       = size( E_elec )

!>    @todo Finalize the file formats and names for cross sections and beta parameters 
      select case(output_style)
      case(0)
      
         open(newunit = lu_xsec,   file = 'photo_xsec'  ,  status = 'replace', form = 'FORMATTED')
         open(newunit = lu_beta1,  file = 'photo_beta_1c', status = 'replace', form = 'FORMATTED')
         open(newunit = lu_beta2l, file = 'photo_beta_2l', status = 'replace', form = 'FORMATTED')
         open(newunit = lu_beta2c, file = 'photo_beta_2c', status = 'replace', form = 'FORMATTED')
      
         open(newunit = lu_total_xsec,   file = 'photo_total_xsec'  ,  status = 'replace', form = 'FORMATTED') 
            
         do ine = 1, no_neutral_states  
            
            Ip = E_ion(1) - E_neut( select_neutral_states(ine) ) 
         
            do ien = 1, no_energies
      
               photon_energy = E_elec(ien) + Ip     

               write(lu_xsec,'(10000e20.5)')   photon_energy * Ha2eV, ( xsec(ien, ion, ine),    ion=1, no_ion_states )
            
               write(lu_beta1,'(10000e20.5)')  photon_energy * Ha2eV, ( beta_1(ien, ion, ine),  ion=1, no_ion_states )
            
               write(lu_beta2l,'(10000e20.5)') photon_energy * Ha2eV, ( beta_2l(ien, ion, ine), ion=1, no_ion_states )
            
               write(lu_beta2c,'(10000e20.5)') photon_energy * Ha2eV, ( beta_2c(ien, ion, ine), ion=1, no_ion_states )
               
               write(lu_total_xsec,'(10000e20.5)')   photon_energy * Ha2eV,  sum( xsec(ien, :, ine) )

            end do
            
               write(lu_xsec,  '("")')
               write(lu_beta1, '("")')
               write(lu_beta2l,'("")')
               write(lu_beta2c,'("")')
               write(lu_total_xsec,'("")')
            
         end do
        
      case(1)
               
         do ine = 1, no_neutral_states
         
            neutral_state = select_neutral_states(ine)
            
            Ip = E_ion(1) - E_neut( neutral_state )          
         
            do ion = 1, no_ion_states
            
               ion_state     = select_ion_states(ion)
            
               write(str_ion_state,    '(i5)')   ion_state
               write(str_neutral_state,'(i5)')   neutral_state

               path = 'photo_xsec_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_xsec, file=path, form="formatted", status="replace")
               
               path = 'photo_beta_1c_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_beta1, file=path, form="formatted", status="replace")  
                            
               path = 'photo_beta_2l_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_beta2l, file=path, form="formatted", status="replace")                        
         
               path = 'photo_beta_2c_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_beta2c, file=path, form="formatted", status="replace")
               
               path = 'photo_total_xsec_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_total_xsec, file=path, form="formatted", status="replace")                
               
               do ien = 1, no_energies
      
                  photon_energy = E_elec(ien) + Ip     

                  write(lu_xsec,'(10000e20.5)')   photon_energy * Ha2eV,  xsec(ien, ion, ine)
            
                  write(lu_beta1,'(10000e20.5)')  photon_energy * Ha2eV,  beta_1(ien, ion, ine)
            
                  write(lu_beta2l,'(10000e20.5)') photon_energy * Ha2eV,  beta_2l(ien, ion, ine)
            
                  write(lu_beta2c,'(10000e20.5)') photon_energy * Ha2eV,  beta_2c(ien, ion, ine)
                  
                  write(lu_total_xsec,'(10000e20.5)')   photon_energy * Ha2eV,  sum( xsec(ien, :, ine) )

               end do
            
               close(lu_xsec)
               close(lu_beta1)
               close(lu_beta2l)
               close(lu_beta2c)
               close(lu_total_xsec)
               
            end do
            
         end do                   
               
      end select
      
      
      
      end associate       
   
   end subroutine

   subroutine write_aligned_cross_section_and_beta_parameters( dipoles, select_ion_states, select_neutral_states, time,&
   &                                         en_start, en_end, select_photon_en, Lbig_max, cmp_xsec, beta_LQl, beta_LQc )
      implicit none

!     Arguments
      type(moments)    :: dipoles
      integer          :: select_ion_states(:), select_neutral_states(:), en_start, en_end
      complex(idp), allocatable :: cmp_xsec(:,:,:,:), beta_LQl(:,:,:,:,:), beta_LQc(:,:,:,:,:)
      real(idp)        :: time(:)
      integer          :: Lbig_max
      real(idp)        :: select_photon_en(:)
      
!     Local
      integer            :: lu_xsec, lu_beta1, lu_beta2l, lu_beta2c, ien, ine, ion, no_ion_states, &
      &                     no_neutral_states, no_energies, ion_state, neutral_state, lu_total_xsec, i
      real(idp)          :: Ip, photon_energy, el_en, delta, test, ph_min, ph_max
      character(len=132) :: path, path_l, path_c, path_en
      character(len=10)   :: str_ion_state, str_neutral_state, str_L, str_Q, str_en
      integer            :: LQ_ind, lu_betal, lu_betac, L, Q, t, n_t, n_LQ, lu_betal_all_en, lu_betac_all_en
      logical            :: linear_is_zero, circular_is_zero
      integer, allocatable :: select_en(:)
      
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )
      
      no_ion_states     = size( select_ion_states  )
      no_neutral_states = size( select_neutral_states )      
      no_energies       = size( E_elec )
      n_LQ              = (Lbig_max + 1)**2
      n_t               = size( cmp_xsec, 1)

      if (size(time) /= n_t) then
         print *,'error: incompatible time array and aligned data'
         stop 1
      endif

      allocate(select_en(size(select_photon_en)))

      do ine = 1, no_neutral_states
      
         neutral_state = select_neutral_states(ine)
         
         Ip = E_ion(1) - E_neut( neutral_state )          

         !find the photon energies closest to the selected photon energies
         ph_min = (E_elec(en_start) + Ip) * Ha2eV
         ph_max = (E_elec(en_end) + Ip) * Ha2eV
         do ien = en_start+1, en_end
            photon_energy = (E_elec(ien) + Ip) * Ha2eV
         enddo

         select_en = -1
         do i = 1, size(select_photon_en)
      
            delta = 10e6_idp

            if (select_photon_en(i) > ph_max .or. select_photon_en(i) < ph_min) cycle
      
            do ien = en_start+1, en_end
      
               photon_energy = (E_elec(ien) + Ip) * Ha2eV
               test = abs(photon_energy - select_photon_en(i))
      
               if (test > delta) then
                  select_en(i) = ien-1
                  exit
               else
                  delta = test
               endif
      
            enddo !ien
      
         enddo !i

         do ion = 1, no_ion_states
         
            ion_state     = select_ion_states(ion)
         
            write(*,'(/," ION state: ",i0)') ion_state

            write(str_ion_state,    '(i5)')   ion_state
            write(str_neutral_state,'(i5)')   neutral_state

            path = 'aligned_photo_xsec_ion_state_'//trim(adjustl(str_ion_state))//&
                  &'_neutral_state_'//trim(adjustl(str_neutral_state))
            open(newunit=lu_xsec, file=path, form="formatted", status="replace")

            write(*,'(" Selected photoelectron and photon energies:")')

            do ien = en_start, en_end

               photon_energy = (E_elec(ien) + Ip) * Ha2eV
               el_en = photon_energy - (E_ion(ion) - E_neut( ine )) * Ha2eV

               if (any(select_en == ien) .and. el_en >= 0.0_idp) then
                  write(*,'(1X,i0,1X,2(e25.15,1X))') ien, el_en, photon_energy
               endif

            enddo !ien  

            !cross section
            do ien = en_start, en_end
         
               photon_energy = (E_elec(ien) + Ip) * Ha2eV

               write(lu_xsec,'(20000e20.5)')   photon_energy,  (cmp_xsec(t, ien, ion, ine), t=1,n_t)

            end do

            close(lu_xsec)
           
            !angular distributions: beta parameters
            LQ_ind = 0
            do L = 0, Lbig_max
            do Q = -L, L

               LQ_ind = LQ_ind + 1

               !don't write the 0,0 component which is trivially 1
               if (LQ_ind == 1) cycle

               linear_is_zero   = .false.
               circular_is_zero = .false.

               if (all(abs(real(beta_LQl(:, :, LQ_ind, ion, ine),idp)) <= 10e-10_idp)) then
                  linear_is_zero = .true.
               endif

               if (all(abs(real(beta_LQc(:, :, LQ_ind, ion, ine),idp)) <= 10e-10_idp)) then
                  circular_is_zero = .true.
               endif

               if (linear_is_zero .and. circular_is_zero) cycle

               write(*,'(" Writing aligned PADs for (L,Q): ",i0,",",i0)') L,Q

               write(str_L,    '(i5)')   L
               write(str_Q,    '(i5)')   Q

               path_l = 'aligned_photo_beta_l_'//trim(adjustl(str_L))//'_'//trim(adjustl(str_Q))//&
                     &'_ion_state_'//trim(adjustl(str_ion_state))//&
                     &'_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_betal_all_en, file=path_l, form="formatted", status="replace")                        

               path_c = 'aligned_photo_beta_c_'//trim(adjustl(str_L))//'_'//trim(adjustl(str_Q))//&
                     &'_ion_state_'//trim(adjustl(str_ion_state))//&
                     &'_neutral_state_'//trim(adjustl(str_neutral_state))
               open(newunit=lu_betac_all_en, file=path_c, form="formatted", status="replace")                        

               do ien = en_start, en_end

                  photon_energy = (E_elec(ien) + Ip) * Ha2eV
                  el_en = photon_energy - (E_ion(ion) - E_neut( ine )) * Ha2eV

                  if (el_en < 0.0_idp) cycle

                  do t = 1, n_t
                     write(lu_betal_all_en,'(f8.4,1X,3(1X,e20.5))')  time(t), photon_energy,  beta_LQl(t, ien, LQ_ind, ion, ine)
                  enddo
                  write(lu_betal_all_en,'(" ")')

                  do t = 1, n_t
                     write(lu_betac_all_en,'(f8.4,1X,3(1X,e20.5))')  time(t), photon_energy,  beta_LQc(t, ien, LQ_ind, ion, ine)
                  enddo
                  write(lu_betac_all_en,'(" ")')

                  if (any(select_en == ien)) then

                     write(str_en,    '(f8.2)')   photon_energy
      
                     path = trim(adjustl(path_l))//'_en_'//trim(adjustl(str_en))
                     open(newunit=lu_betal, file=path, form="formatted", status="replace")                        
         
                     do t = 1, n_t
                        write(lu_betal,'(f8.4,1X,3e20.5)')  time(t), photon_energy,  beta_LQl(t, ien, LQ_ind, ion, ine)
                     enddo
      
                     path = trim(adjustl(path_c))//'_en_'//trim(adjustl(str_en))
                     open(newunit=lu_betac, file=path, form="formatted", status="replace")                        
               
                     do t = 1, n_t
                        write(lu_betac,'(f8.4,1X,3e20.5)')  time(t), photon_energy,  beta_LQc(t, ien, LQ_ind, ion, ine)
                     enddo

                  endif !(any(select_en) == ien)
            
               end do !ien

               close(lu_betal)
               close(lu_betac)

               close(lu_betal_all_en)
               close(lu_betac_all_en)
            enddo !Q
            enddo !L

         end do
         
      end do                   
      
      
      
      end associate       
   
   end subroutine
         
   subroutine write_MFDip( dipoles, electron_direction, steps_elec_direction, &
   &                       ion_state, neutral_state, MFDip)
      implicit none
   
      type(moments) :: dipoles
      integer       :: steps_elec_direction(2), ion_state, neutral_state
      real(idp)     :: electron_direction(4) 
      complex(idp)  :: MFDip(:,:,:)  
   
   
      integer                :: no_angles, no_components, no_energies, no_phi, no_theta, ien, lu_mfdip
      real(idp)              :: Ip
      real(idp), allocatable :: photon_energy(:), theta(:), phi(:)
      character(len=132) :: path
      character(len=5) :: str_ion_state, str_neutral_state

      no_theta             = steps_elec_direction(1)
      no_phi               = steps_elec_direction(2)
      no_angles            = size(MFDip,1)
      no_components        = size(MFDip,2)
      no_energies          = size(MFDip,3)
           
      allocate( photon_energy(no_energies) )
        
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )  

!     Create the photon energy grid     
   
      Ip = E_ion(1) - E_neut( neutral_state ) 
        
      do  ien = 1, no_energies

         photon_energy(ien) = E_elec(ien) + Ip  

      end do

      
      end associate 
      
!     Create the angular grid

      call  grid_theta_phi(steps_elec_direction, electron_direction, theta, phi)
               
      write(str_ion_state,    '(i5)')   ion_state
      write(str_neutral_state,'(i5)')   neutral_state

      path = 'photo_MFDip_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
       
      open(newunit=lu_mfdip, file=path, access="stream", status="replace")
      
      write(lu_mfdip) ion_state, neutral_state,                     &
      &               no_energies, photon_energy,                   &
      &               no_theta, theta,                              &
      &               no_phi, phi,                                  &                  
      &               no_angles, no_components, no_energies, MFDip
      
      close(lu_mfdip)
   
   end subroutine
      
   subroutine write_MFPAD( dipoles, electron_direction, steps_elec_direction, &
   &                       ion_state, neutral_state, MFPAD)
      implicit none
   
      type(moments) :: dipoles
      integer       :: steps_elec_direction(2), ion_state, neutral_state
      real(idp)     :: electron_direction(4) 
      real(idp)     :: MFPAD(:,:,:)  
   
   
      integer                :: no_angles, no_components, no_energies, no_phi, no_theta, ien, lu_mfpad
      real(idp)              :: Ip
      real(idp), allocatable :: photon_energy(:), theta(:), phi(:)
      character(len=132) :: path
      character(len=5) :: str_ion_state, str_neutral_state

      no_theta             = steps_elec_direction(1)
      no_phi               = steps_elec_direction(2)
      no_angles            = size(MFPAD,1)
      no_components        = size(MFPAD,2)
      no_energies          = size(MFPAD,3)
           
      allocate( photon_energy(no_energies) )
        
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )  

!     Create the photon energy grid     
   
      Ip = E_ion(1) - E_neut( neutral_state ) 
        
      do  ien = 1, no_energies

         photon_energy(ien) = E_elec(ien) + Ip  

      end do

      
      end associate 
      
!     Create the angular grid

      call  grid_theta_phi(steps_elec_direction, electron_direction, theta, phi)
               
      write(str_ion_state,    '(i5)')   ion_state
      write(str_neutral_state,'(i5)')   neutral_state

      path = 'photo_MFPAD_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
       
      open(newunit=lu_mfpad, file=path, access="stream", status="replace")
      
      write(lu_mfpad) ion_state, neutral_state,                     &
      &               no_energies, photon_energy,                   &
      &               no_theta, theta,                              &
      &               no_phi, phi,                                  &                  
      &               no_angles, no_components, no_energies, MFPAD
      
      close(lu_mfpad)
   
   end subroutine

   subroutine write_LFDip( dipoles, electron_direction, steps_elec_direction, grid_alpha, &
   &                       grid_beta, grid_gamma, ion_state, neutral_state, LFDip)
      implicit none
   
      type(moments) :: dipoles
      integer       :: steps_elec_direction(2), ion_state, neutral_state
      real(idp)     :: electron_direction(4), grid_alpha(:), grid_beta(:), grid_gamma(:) 
      complex(idp)  :: LFDip(:,:,:,:)  
   
   
      integer                :: no_angles, no_components, no_energies, no_phi, no_theta, ien, lu_lfdip, &
      &                         no_alpha, no_beta, no_gamma, no_euler, no_theta_phi
      real(idp)              :: Ip
      real(idp), allocatable :: photon_energy(:), theta(:), phi(:)
      character(len=132) :: path
      character(len=5) :: str_ion_state, str_neutral_state

      no_theta             = steps_elec_direction(1)
      no_phi               = steps_elec_direction(2)
      no_alpha             = size(grid_alpha)
      no_beta              = size(grid_beta)
      no_gamma             = size(grid_gamma)
      
      no_euler             = size(LFDip,1)
      no_theta_phi         = size(LFDip,2)
      no_components        = size(LFDip,3)
      no_energies          = size(LFDip,4)
           
      allocate( photon_energy(no_energies) )
        
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )  

!     Create the photon energy grid     
   
      Ip = E_ion(1) - E_neut( neutral_state ) 
        
      do  ien = 1, no_energies

         photon_energy(ien) = E_elec(ien) + Ip  

      end do

      
      end associate 
      
!     Create the angular grid

      call  grid_theta_phi(steps_elec_direction, electron_direction, theta, phi)
               
      write(str_ion_state,    '(i5)')   ion_state
      write(str_neutral_state,'(i5)')   neutral_state

      path = 'photo_LFDip_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
       
      open(newunit=lu_lfdip, file=path, access="stream", status="replace")
      
      write(lu_lfdip) ion_state, neutral_state,                     &
      &               no_energies, photon_energy,                   &
      &               no_alpha, grid_alpha,                          &
      &               no_beta, grid_beta,                            &  
      &               no_gamma, grid_gamma,                          &    
      &               no_theta, theta,                              &
      &               no_phi, phi,                                  &                  
      &               no_euler, no_theta_phi, no_components, no_energies, LFDip
      
      close(lu_lfdip)
   
   end subroutine
   
   subroutine write_LFPAD( dipoles, electron_direction, steps_elec_direction, grid_alpha, &
   &                       grid_beta, grid_gamma, ion_state, neutral_state, LFPAD)
      implicit none
   
      type(moments) :: dipoles
      integer       :: steps_elec_direction(2), ion_state, neutral_state
      real(idp)     :: electron_direction(4), grid_alpha(:), grid_beta(:), grid_gamma(:) 
      real(idp)     :: LFPAD(:,:,:,:)  
   
   
      integer                :: no_angles, no_components, no_energies, no_phi, no_theta, ien, lu_lfpad, &
      &                         no_alpha, no_beta, no_gamma, no_euler, no_theta_phi
      real(idp)              :: Ip
      real(idp), allocatable :: photon_energy(:), theta(:), phi(:)
      character(len=132) :: path
      character(len=5) :: str_ion_state, str_neutral_state

      no_theta             = steps_elec_direction(1)
      no_phi               = steps_elec_direction(2)
      no_alpha             = size(grid_alpha)
      no_beta              = size(grid_beta)
      no_gamma             = size(grid_gamma)
      
      no_euler             = size(LFPAD,1)
      no_theta_phi         = size(LFPAD,2)
      no_components        = size(LFPAD,3)
      no_energies          = size(LFPAD,4)
           
      allocate( photon_energy(no_energies) )
        
      associate( E_neut => dipoles % neutral_states % energies, &
      &          E_ion  => dipoles % scattering_states % channels % ion_states % energies, &
      &          E_elec => dipoles % scattering_states % energies )  

!     Create the photon energy grid     
   
      Ip = E_ion(1) - E_neut( neutral_state ) 
        
      do  ien = 1, no_energies

         photon_energy(ien) = E_elec(ien) + Ip  

      end do

      
      end associate 
      
!     Create the angular grid

      call  grid_theta_phi(steps_elec_direction, electron_direction, theta, phi)
               
      write(str_ion_state,    '(i5)')   ion_state
      write(str_neutral_state,'(i5)')   neutral_state

      path = 'photo_LFPAD_ion_state_'//trim(adjustl(str_ion_state))//'_neutral_state_'//trim(adjustl(str_neutral_state))
       
      open(newunit=lu_lfpad, file=path, access="stream", status="replace")
      
      write(lu_lfpad) ion_state, neutral_state,                     &
      &               no_energies, photon_energy,                   &
      &               no_alpha, grid_alpha,                          &
      &               no_beta, grid_beta,                            &  
      &               no_gamma, grid_gamma,                          &    
      &               no_theta, theta,                              &
      &               no_phi, phi,                                  &                  
      &               no_euler, no_theta_phi, no_components, no_energies, LFPAD
      
      close(lu_lfpad)
   
   end subroutine 
     
      
end module dipelm_io

