! Copyright 2019
!
! Alex G. Harvey with ontributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
module analytic_1p1_photon_pad
   use dipelmdefs, only: pi, eye, idp, icp, maxprop_par, alpha, convert_au_to_megabarns
   implicit none

   integer, parameter :: iwrite = 6
   integer, parameter :: max_intermediate_states = 10
   
   !todo these are only temporary for testing purposes
   logical :: use_dummy_dipoles = .false., bound_transition_chiral = .true., continuum_transition_chiral = .true.


contains

   subroutine driver(ifail)
      use dipelmprocs, only: transform_dipoles_to_complex_spherical_harmonic_basis, dcomp2i
      use angmom_procs, only: lm2i, CPHAZ
      USE IFPORT
      implicit none
      !integer, parameter :: max_intermediate_states = 10
      integer, intent(out) :: ifail

      !Arguments for get_dipoles
      integer :: no_dipole_components
      complex(kind=icp), allocatable :: dip_elm(:,:,:,:)
      character(len=1) :: dipole_component_order(maxprop_par)
      real(kind=idp), allocatable :: escat(:), evchl(:,:)
      integer, allocatable :: ichl(:,:), lvchl(:,:), mvchl(:,:)
      integer, dimension(:) :: lu_pw_dipoles(maxprop_par), no_channels(maxprop_par), &
     &                         mgvn(maxprop_par), stot(maxprop_par),gutot(maxprop_par)
      real(kind=idp), allocatable :: bound_state_energies(:)
      real(kind=idp) :: target_energy
      character(len=11) :: format_pw_dipoles
      integer, dimension(:) :: nset_pw_dipoles(maxprop_par)

      !Arguments for calculate_beta_two_photon
      integer :: lab_frame_p1, lab_frame_p2
      complex(kind=idp), allocatable :: beta(:,:), dip_full_csph(:,:,:,:,:)

      !namelist
      integer :: no_intermediate_states, index_initial_state(max_intermediate_states), index_final_state
      complex(kind=idp) :: amplitudes(1:3,1:max_intermediate_states)
      real(kind=idp) :: lab_frame_E_p2(3),ionization_potential

      !local
      integer :: ierr, maxpw, intermediate_state, no_partial_waves, no_scattering_energies, no_bound_states, no_targ_states
      integer :: i,j,k,l,m,en,ii,idcmp,itarget,iii
      complex(kind=idp), allocatable :: dip_full(:,:,:,:,:), coulomb_phase(:,:,:)
      real(kind=idp), allocatable :: etarg(:)
      real(kind=idp) :: k_final, eta, charge, d
 
      namelist /TWOPHOTONINP/ no_intermediate_states, amplitudes, index_initial_state, index_final_state, lab_frame_p1, charge, lab_frame_E_p2, ionization_potential, lab_frame_p2,&
                             &use_dummy_dipoles,bound_transition_chiral,continuum_transition_chiral

         no_intermediate_states = 0 !number of intermediate (initial) states
         index_initial_state = 0 !indices of the intermediate bound states
         index_final_state = 0 !index of the final cationic state
         amplitudes = 0.0_idp !excitation amplitudes corresponding to the bound states
         lab_frame_p1 = 2 !lab frame polarization of the first photon
         lab_frame_p2 = 2 !lab frame polarization of the second photon
         charge = 1.0_idp !charge of the residual cation
         lab_frame_E_p2 = (/0.0_idp,0.0_idp,0.0_idp/) !direction of the electric field vector for the 2nd photon (x,y,z) components; does not have to be normalized to 1
         ionization_potential = -1.0_idp !ionization potential in a.u.
         read(5,TWOPHOTONINP,ERR=100, iostat=ierr)
  100    if (ierr .gt. 0) then
            write (iwrite,'(5X,"Problems reading the namelist TWOPHOTONINP")')
            ifail=1000
            return
         end if

         if (no_intermediate_states > max_intermediate_states) then
            write(iwrite,'(5X,"no_intermediate_states > max_intermediate_states, recompile with a larger max_intermediate_states.")')
            ifail = 1100
            return
         elseif (no_intermediate_states .le. 0) then
            write(iwrite,'(5X,"no_intermediate_states .le. 0.")')
            ifail = 1200
            return
         endif

         if (minval(index_initial_state(1:no_intermediate_states)) .le. 0 .or. index_final_state .le. 0) then
            write(iwrite,'(5X,"at least one index_initial_state .le. 0 or index_final_state .le. 0.")')
            ifail = 1300
            return
         endif

         d = sqrt(dot_product(lab_frame_E_p2,lab_frame_E_p2))
         if (d .eq. 0.0_idp .and. abs(lab_frame_p2) > 1) then
            write(iwrite,'(5X,"on input lab_frame_E_p2 is zero and lab_frame_p2 is out of range")')
            ifail = 1400
            return
         endif
 
         if (ionization_potential .le. 0.0_idp) then
            write(iwrite,'(5X,"on input ionization_potential .le. 0")')
            ifail = 1500
            return
         endif

         do intermediate_state=1,no_intermediate_states

            !Read the dipoles for the current intermediate state from the disk: every intermediate state has its own DIPELMINP namelist and its own input dipole files.
            !todo extend this to allow intermediate states of different symmetries: this affects the channel numbers.
            call get_dipoles( dip_elm, ichl, evchl, lvchl, mvchl,no_channels, escat, no_dipole_components, &
                            &dipole_component_order,bound_state_energies,target_energy, lu_pw_dipoles,nset_pw_dipoles, format_pw_dipoles, ifail )

            if (ifail > 0) return
            maxpw = maxval(lvchl)
            no_partial_waves=(maxpw+1)**2
            no_scattering_energies = size(dip_elm,4)
            no_bound_states = size(dip_elm,2)
            no_targ_states = maxval(ichl)

            if (index_initial_state(intermediate_state) > no_bound_states) then
               stop "The in input value in index_initial_state exceeds the number of bound states available on the dipoles file"
            endif

            !     Determine target energies from channel energies
            !     ------------------------------------------------------------------
            
            allocate(etarg(no_targ_states))
            etarg=0.0_idp
            itarget=1
            
            do i=1,size(evchl,2)
               itarget=ichl(dcomp2i(dipole_component_order(1)),i)
               if(itarget .ne. 0) etarg(itarget)=evchl(dcomp2i(dipole_component_order(1)),i)/2.0_idp
            end do

            !     Construct $i^{-l}e^(i\sigma_l)$
            !     ------------------------------- 
            
            allocate(coulomb_phase(no_partial_waves,no_scattering_energies,no_targ_states))
            coulomb_phase=0.0_idp
            
            do itarget=1,no_targ_states
               do en=1,no_scattering_energies
                  do l=0,maxpw
                     do m=-l,l
                        if((escat(en)-etarg(itarget)) .gt. 0) then
            
                           k_final=sqrt(2*(escat(en)-etarg(itarget)))
                           eta=-charge/k_final
                           ii=lm2i(l,m)
                           coulomb_phase(ii,en,itarget)=(-eye)**(l)* exp(eye*CPHAZ(l,eta,6))
            
                        end if
                     end do     
                  end do
               end do
            end do

            !     Re-jig the moments matrix so that the partial wave index 
            !     corresponds to that used by the spherical harmonics, coulomb phase
            !     and rotation matrices. Multiply the dipoles by the Coulomb phase.
            !     ------------------------------------------------------------------

            allocate(dip_full(no_partial_waves,3,no_scattering_energies,no_targ_states,no_intermediate_states))
            dip_full=0.0_idp
            
            do en=1,no_scattering_energies
               do j=1,no_dipole_components
                  idcmp=dcomp2i(dipole_component_order(j))
            
                  do k=1,no_channels(j)
                     l=lvchl(idcmp,k)
                     m=mvchl(idcmp,k)
                     ii=lm2i(l,m)
                     itarget=ichl(idcmp,k)
            
                     !the dipoles for the bound state that we want are put in the last column of dip_full with index intermediate_state
                     dip_full(ii,idcmp,en,itarget,intermediate_state)= dip_elm(k,index_initial_state(intermediate_state),idcmp,en)*coulomb_phase(ii,en,itarget)!*(rand()-0.5) +eye*(rand()-0.5)
                  end do
            
               end do
            end do

            if (intermediate_state .eq. 1) then
               allocate(dip_full_csph(no_partial_waves,no_dipole_components,no_scattering_energies,no_targ_states,no_intermediate_states))
               dip_full_csph=0.0_idp
            endif

            !the transformed dipoles are put in the last column of dip_full_csph with index intermediate_state
            call transform_dipoles_to_complex_spherical_harmonic_basis(maxpw,intermediate_state,index_final_state,dip_full,dip_full_csph)

            if (intermediate_state < no_intermediate_states) then
               deallocate(dip_elm,dip_full,escat,evchl,ichl,lvchl,mvchl,etarg,coulomb_phase)
            else
               deallocate(dip_elm) !at the end of the loop we keep the channel info
            endif

         enddo !intermediate_state

         call calculate_beta_two_photon(no_intermediate_states,index_final_state,amplitudes(1:3,1:max_intermediate_states),dip_full_csph,lab_frame_p1,lab_frame_p2,lab_frame_E_p2,ionization_potential,escat,beta)

         call print_beta_two_photon(beta,.true.) !.true.)

   end subroutine driver

   !We assume that the dipoles for the intermediate states are put in columns with the same sequence numbers as the excitation amplitudes.
   subroutine calculate_beta_two_photon(no_intermediate_states,index_final_state,amplitudes,dipoles,lab_frame_p1,lab_frame_p2,lab_frame_E_p2,ionization_potential,escat,beta)
      use chiral_procs, only: make_dummy_dipoles
      use anglib, only: threej
      use angmom_procs, only: lm2i, CPHAZ
      implicit none
      !INPUT
      complex(kind=idp) :: amplitudes(1:3,1:max_intermediate_states) !mol frame excitation amplitudes for the first photon: (-1:1,1:no_intermediate_states)
      complex(kind=idp), allocatable :: dipoles(:,:,:,:,:) !mol frame photodipoles: lm,q,E,final_state,initial_state
      integer, intent(in) :: lab_frame_p1, lab_frame_p2 !lab frame polarization of the first photon and the second photon
      integer, intent(in) :: no_intermediate_states  !number of intermediate bound states
      integer, intent(in) :: index_final_state !indices of the bound states and index of the final cationic state
      real(kind=idp), intent(in) :: lab_frame_E_p2(3) !direction vector of the E field of the 2nd photon normalized to 1
      real(kind=idp), intent(in) :: ionization_potential, escat(size(dipoles,3)) !IP and electron energies
      !OUTPUT
      complex(kind=idp), allocatable :: beta(:,:) !beta parameters: E,KM

      integer, parameter :: max_k = 4
      integer :: km, maxpw, no_scattering_energies
      integer :: i,ip,ke,q1,q1p,K1,K2,l,lp,m,mp,q2,q2p,en,Mep,r,rp,lm,lpmp,M2,M2p,M1p,KM_p2,KM_m2
      complex(kind=idp) :: coupling(-max_k:max_k), pol_prod, pol_p2(-1:1)
      real(kind=idp) :: fac_llpke,coupling_sph_harm,tmp1,tmp2,tmp3, real_p2, real_m2, photon_factor(size(dipoles,3)), photon_energy,d, k_final, eta
      complex(kind=idp) :: ampl_product(-2:2,0:2,1:no_intermediate_states,1:no_intermediate_states), fac_llp, cf_plus, cf_minus, cf_m2, cf_p2
      real(kind=idp), parameter :: small = 10e-14_idp
      real(kind=idp), parameter :: rt_2 = sqrt(2.0_idp), inv_rt_2 = 1.0_idp/sqrt(2.0_idp)

         write(iwrite,'(//,5X,"calculate_beta_two_photon: beta parameters for 1+1 photon ionization.")')

         !Check input parameters
         if (.not.(allocated(dipoles))) stop "calculate_beta_two_photon: input array dipoles has not been allocated"
         if (abs(lab_frame_p1) > 1) stop "calculate_beta_two_photon: input lf_p1 out of range"

         if (index_final_state > size(dipoles,4)) stop "calculate_beta_two_photon: input index of the final state is out of range"

         d = sqrt(dot_product(lab_frame_E_p2,lab_frame_E_p2))
         if (d .eq. 0.0_idp .and. abs(lab_frame_p2) > 1) then
            write(iwrite,'(5X,"on input lab_frame_E_p2 is zero and lab_frame_p2 is out of range")')
            stop "error 1 in calculate_beta_two_photon"
         endif
 
         if (ionization_potential .le. 0.0_idp) then
            write(iwrite,'(5X,"on input ionization_potential .le. 0")')
            stop "error 2 in calculate_beta_two_photon"
         endif

         no_scattering_energies = size(dipoles,3)

         if (allocated(beta)) deallocate(beta)
         allocate(beta(no_scattering_energies,(max_k+1)**2)) !(energy,km)
         beta = 0.0_idp

         maxpw = sqrt(real(size(dipoles,1),idp))-1 !max continuum L
         if ((maxpw+1)**2 .ne. size(dipoles,1)) stop "incomplete pw data for dipoles"

         if (use_dummy_dipoles) then
            write(iwrite,'(5X,"Dummy dipoles will be used!")')
            write(iwrite,'(5X,"Bound transition chiral: ",l)') bound_transition_chiral
            write(iwrite,'(5X,"Continuum transition chiral: ",l)') continuum_transition_chiral
            maxpw = 2
            call make_dummy_dipoles(amplitudes,dipoles,maxpw,bound_transition_chiral,continuum_transition_chiral,no_intermediate_states)
            !todo bodge
            !dipoles(1,:,:,:,:) = 1.0_idp
            !dipoles(:,2-1,:,:,:) = 0.0_idp
            !dipoles(:,2+1,:,:,:) = 0.0_idp

            do en=1,no_scattering_energies
               photon_factor(en) = 1.0_idp
            enddo !en
         else
            do en=1,no_scattering_energies
               photon_energy = escat(en)+ionization_potential
               photon_factor(en) = 4*(pi**2)*alpha*photon_energy*convert_au_to_megabarns
            enddo !en
         endif

         write(iwrite,'(/,5X,"Polarization of the 1st photon is: ",i2)') lab_frame_p1
         write(iwrite,'(5X,"Direction of propagation of the 1st photon is z.")')

         !Construct the polarization function of the 2nd photon. If the vector
         !lab_frame_E_p2 is non-zero then this vector defines the direction of
         !polarization of the 2nd (linearly polarized) photon. Otherwise the
         !value lab_frame_p2 defines the polarization (linear or circular) of
         !the 2nd photon.
         pol_p2 = 0.0_idp
         if (d .eq. 0.0_idp) then
            !2nd photon polarized according to the value lab_frame_p2:
            pol_p2(lab_frame_p2) = 1.0_idp
            write(iwrite,'(5X,"Polarization of the 2nd photon is: ",i2)') lab_frame_p2
         else
            !2nd photon linearly polarized along the direction vector lab_frame_E_p2(x,y,z):
            pol_p2( 1) = -inv_rt_2*(lab_frame_E_p2(1) + eye*lab_frame_E_p2(2)) !-1/sqrt(2)*(x+i*y)
            pol_p2( 0) =  lab_frame_E_p2(3)
            pol_p2(-1) =  inv_rt_2*(lab_frame_E_p2(1) - eye*lab_frame_E_p2(2)) ! 1/sqrt(2)*(x-i*y)
            write(iwrite,'(5X,"2nd photon is linearly polarized in the direction:"," x=",e25.15," y=",e25.15," z=",e25.15)') lab_frame_E_p2(1:3)
         endif

         write(iwrite,'(5X,"Number of intermediate states: ",i)') no_intermediate_states
         write(iwrite,'(5X,"Number of scattering energies: ",i)') no_scattering_energies

         ampl_product = 0.0_idp
         do i=1,no_intermediate_states
         do ip=1,no_intermediate_states
            do q1=-1,1
            do q1p=-1,1
               M1p=q1-q1p
               do K1=0,2
                  tmp1 = threej(2*1,-2*q1,2*1,2*q1p,2*K1,2*M1p)
                  tmp2 = threej(2*1,-2*lab_frame_p1,2*1,2*lab_frame_p1,2*K1,0)
                  ampl_product(M1p,K1,ip,i) = ampl_product(M1p,K1,ip,i) + amplitudes(q1+2,i)*conjg(amplitudes(q1p+2,ip))*(-1)**(lab_frame_p1-q1)*(2*K1+1)*tmp1*tmp2
               enddo !K1
            enddo !q1p
            enddo !q1
         enddo !ip
         enddo !i

         do i=1,no_intermediate_states
         do ip=1,no_intermediate_states
            do Ke=0,4
               do l=0,maxpw
               do lp=0,maxpw
                  fac_llpke = sqrt((2*l+1.0_idp)*(2*lp+1.0_idp)*(2*Ke+1)/(4*pi))*threej(2*l,0,2*lp,0,2*Ke,0)
                  if (fac_llpke .eq. 0.0_idp) cycle
                  do m=-l,l
                  do mp=-lp,lp
                     Mep = m-mp
                     lm = l*l+l+m+1
                     lpmp = lp*lp+lp+mp+1
                     coupling_sph_harm = (-1)**(mp+Mep)*fac_llpke*threej(2*l,2*m,2*lp,-2*mp,2*Ke,-2*Mep)
                     if (coupling_sph_harm .eq. 0.0_idp) cycle
                     
                     do q2=-1,1
                     do q2p=-1,1
                        M2p=q2-q2p
                        coupling = 0.0_idp
                        do K2=0,2
                           tmp1 = threej(2*1 ,-2*q2, 2*1,2*q2p,2*K2, 2*M2p); if (tmp1 .eq. 0.0_idp) cycle
                           do r=-1,1
                           do rp=-1,1
                              M2 = r-rp
                              if (abs(M2) > Ke) cycle
                              KM = Ke*Ke+Ke+M2+1
                              pol_prod = pol_p2(r)*pol_p2(rp)*threej(2*1,-2*r,2*1,2*rp,2*K2,2*M2)
                              if (pol_prod .eq. 0.0_idp) cycle
                              do K1=0,2
                                 tmp2 = threej(2*K1,    0,2*K2,  2*M2,2*Ke,-2*M2);  if (tmp2 .eq. 0.0_idp) cycle
                                 do M1p=-2,2
                                    if (ampl_product(M1p,K1,ip,i) .eq. 0.0_idp) cycle
                                    tmp3 = threej(2*K1,2*M1p,2*K2, 2*M2p,2*Ke,-2*Mep); if (tmp3 .eq. 0.0_idp) cycle
                                    coupling(M2) = coupling(M2) + ampl_product(M1p,K1,ip,i)*pol_prod*(-1)**(r-q2)*(2*K2+1)*coupling_sph_harm*(-1)**(M2-Mep)*tmp1*tmp2*tmp3
                                 enddo !M1p
                              enddo !K1
                           enddo !rp
                           enddo !r
                        enddo !K2

                        do M2=-Ke,Ke
                           if (coupling(M2) .eq. 0.0_idp) cycle
                           KM = Ke*Ke+Ke+M2+1
                           if (use_dummy_dipoles) then
                              k_final = sqrt(2*(0.2))
                              eta     = -1.0d0/k_final
                              fac_llp = (eye)**(-l)*(eye)**lp*exp(eye*CPHAZ(l,eta,6))*exp(-eye*CPHAZ(lp,eta,6))
                           else
                              fac_llp = 1!*(eye)**(-l)*(eye)**lp
                           endif
                           do en=1,no_scattering_energies
                              !mol frame photodipoles: lm,q,E,final_state,initial_state
                              beta(en,KM) = beta(en,KM) + fac_llp*coupling(M2)*photon_factor(en)*dipoles(lm,2+q2,en,index_final_state,i)*conjg(dipoles(lpmp,2+q2p,en,index_final_state,ip))

                           enddo !en
                        enddo !M2

                     enddo !q2p
                     enddo !q2

                  enddo !mp
                  enddo !m
               enddo !lp
               enddo !l
            enddo !Ke
         enddo !ip
         enddo !i

!         en = 10
         do Ke=0,4
            do M2=0,Ke
               if (M2 .eq. 0) then
                  KM = ke*ke+ke+1
                  do en=1,no_scattering_energies
                     real_p2 = real(beta(en,KM))
                     if (abs(real_p2) .le. small) real_p2 = 0.0_idp
                     beta(en,KM) = real_p2
                     !if (real_p2 .ne. 0.0_idp) write(iwrite,'(2i4,2e25.15)') Ke,M2,real(beta(en,KM)) !X_{Ke,0}
                     if (abs(aimag(beta(en,KM))) > small) stop "error: complex observable case M=0"
                  enddo !en
               else !M2 > 0
                  KM_p2 = ke*ke+ke+m2+1
                  KM_m2 = ke*ke+ke-m2+1
                  do en=1,no_scattering_energies
                     cf_plus = beta(en,KM_p2)
                     cf_minus = beta(en,KM_m2)
                     !!write(iwrite,'(2i4,2e25.15)') Ke, M2,cf_plus
                     !!write(iwrite,'(2i4,2e25.15)') Ke,-M2,cf_minus
                     !rt_2 is there for normalization to ensure the coefficient corresponds to the normalized real spherical harmonic
                     cf_p2 = rt_2*( cf_plus+cf_minus) !X_{Ke,+2}
                     cf_m2 = rt_2*(-cf_plus+cf_minus) !X_{Ke,-2}
                     real_p2 = real(cf_p2)
                     real_m2 = aimag(cf_m2)
                     if (abs(real_p2) .le. small) real_p2 = 0.0_idp
                     if (abs(real_m2) .le. small) real_m2 = 0.0_idp
                     beta(en,KM_p2) = real_p2
                     beta(en,KM_m2) = real_m2
                     !!if (beta(en,KM_p2) .ne. 0.0_idp) write(iwrite,'(2i4,2e25.15)') Ke, M2,real(beta(en,KM_p2))
                     !!if (beta(en,KM_m2) .ne. 0.0_idp) write(iwrite,'(2i4,2e25.15)') Ke,-M2,real(beta(en,KM_m2))
                     if (abs(aimag(cf_p2)) > small) stop "error: complex observable case M=+2"
                     if (abs(real(cf_m2)) > small) stop "error: complex observable case M=-2"
                     !print *, "imag part ??", aimag(cf_p2)
                  enddo !en
               endif
            enddo
         enddo

   end subroutine calculate_beta_two_photon

   subroutine print_beta_two_photon(beta,energies_first)
      implicit none
      complex(kind=idp), intent(in) :: beta(:,:) !beta parameters: E,KM
      logical, intent(in) :: energies_first

      !local
      integer :: Ke,Me,en,no_scattering_energies,KM,KM_p2,KM_m2
      character(len=80) :: output_filename
      character(len=2) :: str_KM
      
         write(iwrite,'(//,5X,"print_beta_two_photon: print beta parameters for 1+1 photon ionization.")')

         if (use_dummy_dipoles) then
            no_scattering_energies = 1
         else
            no_scattering_energies = size(beta,1)
         endif

         if (energies_first) then !print in order energy,KM
            do Ke=0,4
               do Me=0,Ke
                  
                  write(str_KM,'(2i1)') Ke,Me

                  if (Me .eq. 0) then
                     KM = ke*ke+ke+1
                     output_filename ='beta'//'.'//trim(adjustl(str_KM))//'.dat'
                     open(unit=888+KM, file=trim(adjustl(output_filename)), form='formatted', access='sequential')
                     do en=1,no_scattering_energies
                        if (real(beta(en,KM)) .ne. 0.0_idp) write(iwrite,'(i,2i4,2e25.15)') en,Ke,Me,real(beta(en,KM))/real(beta(en,1)) !X_{Ke,0}
                        write(888+KM,'(i,2e25.15)') en, real(beta(en,KM))/real(beta(en,1))
                        
                     enddo !en
                     close(888+km)
                     
                  else if (abs(Me) .eq. 2 ) then! Me > 0
                     KM_p2 = ke*ke+ke+Me+1
                     KM_m2 = ke*ke+ke-Me+1
                     
                     output_filename ='beta'//'.'//trim(adjustl(str_KM))//'-.dat'
                     open(unit=888+KM_m2, file=trim(adjustl(output_filename)), form='formatted', access='sequential')
                     do en=1,no_scattering_energies
                        if (real(beta(en,KM_m2)) .ne. 0.0_idp) write(iwrite,'(i,2i4,2e25.15)') en,Ke,-Me,real(beta(en,KM_m2))/real(beta(en,1))
                        write(888+KM_m2,'(i, 2e25.15)') en, real(beta(en,KM_m2))/real(beta(en,1))
                     end do
                     close(888+KM_m2)
                     output_filename ='beta'//'.'//trim(adjustl(str_KM))//'+.dat'
                     open(unit=888+KM_p2, file=trim(adjustl(output_filename)), form='formatted', access='sequential')
                     
                     do en=1,no_scattering_energies                     
                        if (real(beta(en,KM_p2)) .ne. 0.0_idp) write(iwrite,'(i,2i4,2e25.15)') en,Ke,Me,real(beta(en,KM_p2))/real(beta(en,1))
                        write(888+KM_p2,'(i, 2e25.15)') en, real(beta(en,KM_p2))/real(beta(en,1))
                        
                     enddo !en
                     close(888+KM_p2)
                     
                     
                  endif
               enddo
            enddo
         else !print in order KM,energy
            do en=1,no_scattering_energies
               write(iwrite,'("Energy no. ",i)') en
               do Ke=0,4
                  do Me=0,Ke
                     if (Me .eq. 0) then
                        KM = ke*ke+ke+1
                        if (real(beta(en,KM)) .ne. 0.0_idp) write(iwrite,'("beta_LM",2i4,2e25.15)') Ke,Me,real(beta(en,KM)) !X_{Ke,0}
                     else !Me > 0
                        KM_p2 = ke*ke+ke+Me+1
                        KM_m2 = ke*ke+ke-Me+1
                        if (real(beta(en,KM_p2)) .ne. 0.0_idp) write(iwrite,'("beta_LM",2i4,2e25.15)') Ke, Me,real(beta(en,KM_p2))
                        if (real(beta(en,KM_m2)) .ne. 0.0_idp) write(iwrite,'("beta_LM",2i4,2e25.15)') Ke,-Me,real(beta(en,KM_m2))
                     endif
                  enddo
               enddo
            enddo
         endif

   end subroutine print_beta_two_photon

   subroutine get_dipoles( dip_elm, ichl, evchl, lvchl, mvchl,no_channels, escat, no_dipole_components, &
  &                       dipole_component_order,bound_state_energies,target_energy, lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, ifail )
      use photo_outerio, only: read_pw_dipoles, write_pw_dipoles
      use dipelmprocs, only: get_channel_info
      implicit none

 !    Arguments
      integer :: ifail 
      complex(kind=icp), allocatable :: dip_elm(:,:,:,:)          
      character(len=1) :: dipole_component_order(maxprop_par)                
      real(kind=idp), allocatable :: escat(:), evchl(:,:)
      integer, allocatable :: ichl(:,:), lvchl(:,:), mvchl(:,:)

!     Local variables
      integer, dimension(:) :: lu_pw_dipoles(maxprop_par), no_channels(maxprop_par), &
     &                         mgvn(maxprop_par), stot(maxprop_par), gutot(maxprop_par)
      integer :: point_group_index, ncomponents_assigned
      character(len=11) ::     format_pw_dipoles, test_formatted
      integer, dimension(:) :: nset_pw_dipoles(maxprop_par)
      character(len=80) ::     title
      character(len=3) ::      point_group
      character(len=1) ::      letter_box(3)=(/'y','z','x'/)
      
      real(kind=idp) :: target_energy
      real(kind=idp), allocatable :: escat_temp(:), evchl_temp(:), re_pw_dipoles(:,:,:,:), im_pw_dipoles(:,:,:,:),bound_state_energies(:)
      integer, allocatable :: ichl_temp(:), lvchl_temp(:), mvchl_temp(:), dip_comp_present(:), starg(:), mtarg(:), gtarg(:)

      integer :: lmax_property, maxchs, no_bound_states, no_scattering_energies, &
     &           no_dipole_components, i, j, k, idcmp, ienergy, icomponent, iprint

      integer :: ierr

      lu_pw_dipoles=0; no_channels=0; mgvn=0; stot=0; gutot=0; nset_pw_dipoles=0


      namelist /DIPELMINP/ lu_pw_dipoles, &
     &                     format_pw_dipoles, &
     &                     nset_pw_dipoles, &
     &                     iprint,          &
     &                     lmax_property, &
     &                     point_group ! possible values are 'C1','Cs','C2','Ci','C2v','C2h','D2','D2h' ! Deprecated

      ifail=0

!     Default namelist DIPELMINP values.
      iprint=1
      lmax_property=1                ! Maximum property number 1=dipoles 2= dipoles + quadrapoles      
      format_pw_dipoles ='UNFORMATTED'
      test_formatted ='FORMATTED'
      no_dipole_components=lmax_property**2+2*lmax_property
      point_group='' ! Point group is now deprecated

!     Read namelist
!     -------------
      read(5,DIPELMINP,ERR=200, iostat=ierr)
      dipole_component_order="" ! No longer needs to be entered in the namelist
  200 if (ierr .gt. 0) then
         write (iwrite,2000)
         ifail=2000
         return

      end if

!     Interrogate headers of partial wave dipoles to get information 
!     about the number of channels, check that all the dipole components are there, etc.
print *, lu_pw_dipoles
      call get_channel_info( lu_pw_dipoles, nset_pw_dipoles, format_pw_dipoles, mgvn, no_bound_states, &
     &                       no_dipole_components,dipole_component_order,letter_box, no_channels, &
     &                       no_scattering_energies, iwrite )
      maxchs=maxval(no_channels)
      allocate(escat(no_scattering_energies), evchl(no_dipole_components,maxchs), ichl(no_dipole_components,maxchs),&
     &         lvchl(no_dipole_components,maxchs),mvchl(no_dipole_components,maxchs))
      allocate( dip_elm(maxchs, no_bound_states, no_dipole_components, no_scattering_energies) )
      escat=0_idp;evchl=0._idp;ichl=0;lvchl=0;mvchl=0;dip_elm=0._idp

      do i=1,no_dipole_components
         call read_pw_dipoles( lu_pw_dipoles(i), nset_pw_dipoles(i), format_pw_dipoles, title, &
     &                         mgvn(i), stot(i), gutot(i),starg, mtarg, gtarg, ichl_temp, lvchl_temp, mvchl_temp,  &
     &                         evchl_temp, escat_temp, lmax_property, dip_comp_present,        &
     &                         bound_state_energies, target_energy, re_pw_dipoles, im_pw_dipoles,  &
     &                         iprint, iwrite, ifail )
 
         do ienergy=1, no_scattering_energies
            do icomponent=1,no_dipole_components
               if(dip_comp_present(icomponent).eq. 1) then
                  dip_elm(:,:,icomponent,ienergy)=transpose( cmplx(re_pw_dipoles(:,:,icomponent,ienergy),-im_pw_dipoles(:,:,icomponent,ienergy), kind=icp) )
                  ichl(  icomponent,1:no_channels(i) )=ichl_temp
                  lvchl( icomponent,1:no_channels(i) )=lvchl_temp
                  mvchl( icomponent,1:no_channels(i) )=mvchl_temp
                  evchl( icomponent,1:no_channels(i) )=evchl_temp
                  escat=escat_temp
                  
               end if
               
            end do
         IF(ANY(IsNaN(real(dip_elm(:,:,:,ienergy))))) dip_elm(:,:,:,ienergy)= dip_elm(:,:,:,ienergy-1)! remove NANs from dipoles (Danilo)   
         end do
         deallocate(ichl_temp,lvchl_temp,mvchl_temp,evchl_temp, escat_temp, re_pw_dipoles,im_pw_dipoles)
         
      end do
      
      return

!     format statements
 2000 format(/,5X,'Problems reading the namelist DIPELMINP',/)

   end subroutine get_dipoles

end module analytic_1p1_photon_pad
