! Copyright 2019
!
! Alex G. Harvey with ontributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!***************************************************************************************************
!
!  photo_outerio contains I/O routines needed for photo-ionization(/recombination) calculations
!  
!
!***************************************************************************************************
module photo_outerio

      integer, parameter :: idp=selected_real_kind(8)
      integer, parameter :: maxprop_par=8  
      real(kind=idp), parameter :: pi=3.1415926535897932_idp 
      integer, parameter ::        maxnuc=100
      real(kind=idp), parameter :: zero=0.0_idp, two=2.0_idp, amu=1822.832_idp
      character(len=5), parameter :: multipole_labels(8) = (/&
     & "Y    ","Z    ","X    ","XY   ","YZ   ","ZZ   ","XZ   ","XX-YY"/) 
     
contains

!     **********************************************************************************************
!
!     write_pw_dipoles: Write partial wave dipoles to file.
!
!     **********************************************************************************************
      subroutine write_pw_dipoles( lu_pw_dipoles, nset_pw_dipoles, form_pw_dipoles, title, mgvn,   &
     &                             stot, gutot, starg,mtarg,gtarg, ichl, lvchl, mvchl, evchl,escat, lmax_property,    &   
     &                             dip_comp_present,  bound_state_energies, target_energy,         &
     &                             re_pw_dipoles, im_pw_dipoles, iprnt, iwrite, ifail )
      implicit none
      
!     Arguments
      integer :: lu_pw_dipoles, nset_pw_dipoles, mgvn, stot, gutot, iprnt, iwrite, lmax_property, &
     &           ifail,dip_comp_present(:)
      character(len=11) ::  form_pw_dipoles
      character(len=80) ::  title
      integer :: ichl(:), lvchl(:), mvchl(:), starg(:), mtarg(:), gtarg(:)
      real(kind=idp) :: evchl(:), escat(:), re_pw_dipoles(:,:,:,:), im_pw_dipoles(:,:,:,:), bound_state_energies(:), target_energy

!     Local
      integer :: keydip, nchan, nbound, no_scat_energies, no_components, no_target_states
      integer :: ibound, ichan, icomponent, ienergy,icomp, i

      keydip=24

!     Determine array dimensions
      nbound           = size(re_pw_dipoles,1)
      nchan            = size(re_pw_dipoles,2)
      no_components    = size(re_pw_dipoles,3)
      no_scat_energies = size(re_pw_dipoles,4)
      no_target_states = size(mtarg)

      if(form_pw_dipoles .eq. 'UNFORMATTED') then
!        Write Set Header
!        ----------------      
         write(lu_pw_dipoles) keydip, nset_pw_dipoles, lmax_property
         write(lu_pw_dipoles) title
         write(lu_pw_dipoles) mgvn, stot, gutot, nchan,nbound, no_target_states, no_scat_energies, dip_comp_present !extra line
         write(lu_pw_dipoles) target_energy,bound_state_energies
         write(lu_pw_dipoles) starg, mtarg, gtarg      
         write(lu_pw_dipoles) ichl, lvchl, mvchl, evchl, escat

!        Write Set Body
!        --------------      
         write(lu_pw_dipoles) re_pw_dipoles
         write(lu_pw_dipoles) im_pw_dipoles

      else if (form_pw_dipoles .eq. 'FORMATTED') then
!        Write Set Header
!        ----------------         
         write(lu_pw_dipoles,'(3i4)') keydip,nset_pw_dipoles,lmax_property
         write(lu_pw_dipoles,'(" TITLE= ", a80)') title
         write(lu_pw_dipoles, '(3i4,4i8)') mgvn, stot, gutot, nchan, nbound, no_target_states, no_scat_energies
  
         do icomp=1,lmax_property*(lmax_property+2)     
            write(lu_pw_dipoles, '(i4)', advance='NO') dip_comp_present(icomp)
         end do
      
         write(lu_pw_dipoles, *) ""
         write(lu_pw_dipoles, '("  Target GS:", D18.8 )') target_energy
      
         write(lu_pw_dipoles, '("  Bound state     Energy")') 
         do ibound=1,nbound
            write(lu_pw_dipoles, '(i8,2x,D20.8 )') ibound, bound_state_energies(ibound)
         end do
         write(lu_pw_dipoles,*) ""
      
         write(lu_pw_dipoles, '("  Channel    Targ. 2S+1  I.Rep. g/u    l     m     Channel Thresholds")')
         write(lu_pw_dipoles, '("  -------------------------------------------------------------------"     )')
         do ichan=1,nchan
            write(lu_pw_dipoles, '(2i8,5i6,D20.8)') ichan, ichl(ichan),starg(ichl(ichan)),mtarg(ichl(ichan)),gtarg(ichl(ichan)),&
                                                    lvchl(ichan), mvchl(ichan), evchl(ichan)
         end do 
         write(lu_pw_dipoles,*) ""
         
!        Write Set Body
!        --------------      
         write(lu_pw_dipoles, '("  Channel     Electron Energy",7x, 8(a10, 10x, a10, 10x))') &
                               ("real "//multipole_labels(icomp), "imag "//multipole_labels(icomp), &
                                icomp=1,lmax_property*(lmax_property+2))
         write(lu_pw_dipoles, '("  --------------------------",8a40)') ("----------------------------------------", &
                                icomp=1,lmax_property*(lmax_property+2)) 
         do ibound=1, nbound
            do ichan=1,nchan

               do ienergy=1, no_scat_energies
                  write(lu_pw_dipoles, '(i8, 9D20.8)') ichan, escat(ienergy) , (re_pw_dipoles(ibound,ichan, icomp, ienergy), &
                                       im_pw_dipoles(ibound,ichan, icomp,ienergy), icomp=1,lmax_property*(lmax_property+2))
               end do
               write(lu_pw_dipoles,*) ""

            end do !ichan
         end do !ibound
         
      else
         stop "ERROR: form_pw_dipoles must be set to either 'FORMATTED' or 'UNFORMATTED'"
      
      end if

      end subroutine write_pw_dipoles

!     **********************************************************************************************
!
!     read_pw_dipoles: Read partial wave dipoles from file.
!
!     **********************************************************************************************
      subroutine read_pw_dipoles( lu_pw_dipoles, nset_pw_dipoles, form_pw_dipoles, title, mgvn,    &
     &                             stot, gutot, starg,mtarg,gtarg, ichl, lvchl, mvchl, evchl, escat,lmax_property,    &
     &                             dip_comp_present, bound_state_energies, target_energy,         &
     &                             re_pw_dipoles, im_pw_dipoles, iprint, iwrite, ifail )
      implicit none
      
!     Arguments
      integer :: lu_pw_dipoles, nset_pw_dipoles, mgvn, stot, gutot, iprint, iwrite, &
     &           ifail,lmax_property
      character(len=11) ::  form_pw_dipoles
      character(len=80) ::  title
      integer, allocatable :: ichl(:), lvchl(:), mvchl(:),dip_comp_present(:), starg(:), mtarg(:), gtarg(:)
      real(kind=idp), allocatable :: evchl(:), escat(:),re_pw_dipoles(:,:,:,:), im_pw_dipoles(:,:,:,:), bound_state_energies(:)
      real(kind=idp) :: target_energy
      
!     Local
      integer :: keydip, nchan, nbound, no_scat_energies, no_components, no_target_states, ierr, &
     &           ibound, ichan, ienergy, icomp, itemp1, itemp2, starg_temp, mtarg_temp, gtarg_temp
      logical :: exists
      character(len=8) :: str_temp1
      character(len=12) :: str_temp2
   
      keydip=24
      

      inquire(unit=lu_pw_dipoles, exist=exists)
      if (exists) then
         open (unit=lu_pw_dipoles, form=form_pw_dipoles, iostat=ierr, err=100)  
      else
         write(iwrite,'("ERROR: File unit ",i10, " does not exist.")') lu_pw_dipoles
         stop 
      end if
      
      open (unit=lu_pw_dipoles, form=form_pw_dipoles, iostat=ierr, err=100)   
      rewind(lu_pw_dipoles)  

      if(form_pw_dipoles .eq. 'UNFORMATTED') then
            
!        Read set header
!        ----------------
         read(lu_pw_dipoles) keydip, nset_pw_dipoles, lmax_property
         read(lu_pw_dipoles) title
         if ( allocated(dip_comp_present) ) deallocate(dip_comp_present)
         no_components=lmax_property**2+2*lmax_property 
         allocate(dip_comp_present(no_components))
         dip_comp_present=0     
      
         read(lu_pw_dipoles) mgvn, stot, gutot, nchan, nbound, no_target_states, no_scat_energies, dip_comp_present

!        Allocate space for channel info arrays
         if (allocated(ichl) )  deallocate(ichl)
         if (allocated(lvchl) ) deallocate(lvchl)
         if (allocated(mvchl) ) deallocate(mvchl)
         if (allocated(evchl) ) deallocate(evchl)
         if (allocated(escat) ) deallocate(escat)
         allocate( ichl(nchan), lvchl(nchan), mvchl(nchan), evchl(nchan), escat(no_scat_energies) )
         ichl=0;lvchl=0;mvchl=0;evchl=0;escat=0 
                  
         if (allocated(bound_state_energies) ) deallocate(bound_state_energies) 
         allocate(bound_state_energies(nbound)) 
         bound_state_energies=0._idp
         
         if (allocated(starg) ) deallocate(starg)
         if (allocated(mtarg) ) deallocate(mtarg)
         if (allocated(gtarg) ) deallocate(gtarg)
         allocate( starg(no_target_states), mtarg(no_target_states), gtarg(no_target_states) )
         starg=0; mtarg=0; gtarg=0 
      
         read(lu_pw_dipoles) target_energy,bound_state_energies
         read(lu_pw_dipoles) starg, mtarg, gtarg
         read(lu_pw_dipoles) ichl, lvchl, mvchl, evchl, escat
!        Read Set Body
!        --------------   
      
!        Allocate space for partial wave dipoles
         if (allocated(re_pw_dipoles) ) deallocate(re_pw_dipoles)
         if (allocated(im_pw_dipoles) ) deallocate(im_pw_dipoles)

         allocate( re_pw_dipoles(nbound, nchan, no_components, no_scat_energies), &
     &          im_pw_dipoles(nbound, nchan, no_components, no_scat_energies) ) 
         re_pw_dipoles=0;im_pw_dipoles=0
         
         read(lu_pw_dipoles) re_pw_dipoles
         read(lu_pw_dipoles) im_pw_dipoles
      
      else if (form_pw_dipoles .eq. 'FORMATTED') then

!        Read Set Header
!        ----------------         
         read(lu_pw_dipoles, *) keydip,nset_pw_dipoles,lmax_property
         read(lu_pw_dipoles,'(a8, a80)') str_temp1, title
         read(lu_pw_dipoles, *) mgvn, stot, gutot, nchan, nbound, no_target_states, no_scat_energies
         
         if ( allocated(dip_comp_present) ) deallocate(dip_comp_present)
         no_components=lmax_property**2+2*lmax_property 
         allocate(dip_comp_present(no_components))
         dip_comp_present=0      
         read(lu_pw_dipoles, *) dip_comp_present
         
!        Allocate space for channel info arrays
         if ( allocated(ichl ) ) deallocate(ichl)
         if ( allocated(lvchl) ) deallocate(lvchl)
         if ( allocated(mvchl) ) deallocate(mvchl)
         if ( allocated(evchl) ) deallocate(evchl)
         if ( allocated(escat) ) deallocate(escat)

         allocate( ichl(nchan), lvchl(nchan), mvchl(nchan), evchl(nchan), escat(no_scat_energies) )
         ichl=0;lvchl=0;mvchl=0;evchl=0;escat=0 
         if (allocated(bound_state_energies) ) deallocate(bound_state_energies) 
         allocate(bound_state_energies(nbound)) 
         bound_state_energies=0._idp
         if (allocated(starg) ) deallocate(starg)
         if (allocated(mtarg) ) deallocate(mtarg)
         if (allocated(gtarg) ) deallocate(gtarg)
         allocate( starg(no_target_states), mtarg(no_target_states), gtarg(no_target_states) )
         starg=0; mtarg=0; gtarg=0 
               
         read(lu_pw_dipoles, '(a12,D18.8)' ) str_temp2, target_energy
               
         read(lu_pw_dipoles, *) 
         do ibound=1,nbound
            read(lu_pw_dipoles, *) itemp1, bound_state_energies(ibound)
         end do
         read(lu_pw_dipoles, *)
      
         read(lu_pw_dipoles, *)
         read(lu_pw_dipoles, *)

         do ichan=1,nchan
            read(lu_pw_dipoles, *) itemp1, ichl(ichan),starg_temp,mtarg_temp,gtarg_temp, lvchl(ichan), mvchl(ichan), evchl(ichan)
            starg(ichl(ichan))=starg_temp
            mtarg(ichl(ichan))=mtarg_temp
            gtarg(ichl(ichan))=gtarg_temp
 
         end do 
         read(lu_pw_dipoles, *)
         
!        Read Set Body
!        -------------- 
         if (allocated(re_pw_dipoles) ) deallocate(re_pw_dipoles)
         if (allocated(im_pw_dipoles) ) deallocate(im_pw_dipoles)
         allocate( re_pw_dipoles(nbound, nchan, no_components, no_scat_energies), &
     &          im_pw_dipoles(nbound, nchan, no_components, no_scat_energies) ) 
         re_pw_dipoles=0;im_pw_dipoles=0
          
         read(lu_pw_dipoles, *) 
         read(lu_pw_dipoles, *) 
         do ibound=1, nbound
            do ichan=1,nchan
               do ienergy=1, no_scat_energies
                  read(lu_pw_dipoles, *) itemp1, escat(ienergy) , (re_pw_dipoles(ibound,ichan, icomp, ienergy), &
                                         im_pw_dipoles(ibound,ichan, icomp,ienergy), icomp=1,lmax_property*(lmax_property+2))
               end do
               read(lu_pw_dipoles, *)
            
            end do
         end do
         
      end if
            
      if (iprint.ge.1) then
!        Print Limited Header Info      
         write(iwrite, '(" PW DIPOLE FILE NO.= ",i10)') lu_pw_dipoles
         write(iwrite, '(" KEYDIP= ",i5, " NSET_PWD_DIPOLES= ",i5)') keydip, nset_pw_dipoles
         write(iwrite, '(" TITLE= ",a80)') title
         write(iwrite, '(" MGVN= ",i5, " STOT= ",i5, " GUTOT= ",i5, " NCHAN= ",i5, " NBOUND= ",i5, " NO_SCAT_ENERGIES= ",i5)' ) &
                        mgvn, stot, gutot, nchan, nbound, no_scat_energies
         write(iwrite, '(" DIP_COMP_PRESENT= ",8i5)') dip_comp_present
         write(iwrite, '(" TARGET STATE ENERGY 1  = ", d20.5)') target_energy
         do ibound=1,nbound
            write(iwrite, '(" BOUND STATE ENERGY ",i3, " = ", d20.5)') ibound, bound_state_energies(ibound)
         end do
      end if
      
      if (iprint.ge.2) then
!        Print all Header Info
         write(iwrite,'(3i4)') keydip,nset_pw_dipoles,lmax_property
         write(iwrite,'(" TITLE= ", a80)') title
    
         write(iwrite, '(3i4,4i8)') mgvn, stot, gutot, nchan, nbound, no_target_states, no_scat_energies
  
         do icomp=1,lmax_property*(lmax_property+2)     
            write(iwrite, '(i4)', advance='NO') dip_comp_present(icomp)
         end do
      
         write(iwrite, *) ""
         write(iwrite, '("  Target GS:", D18.8 )') target_energy
      
         write(iwrite, '("  Bound state     Energy")') 
         do ibound=1,nbound
            write(iwrite, '(i8,2x,D20.8 )') ibound, bound_state_energies(ibound)
         end do
         write(iwrite,*) ""
      
         write(iwrite, '("  Channel    Targ. 2S+1  I.Rep. g/u    l     m     Channel Thresholds")')
         write(iwrite, '("  -------------------------------------------------------------------"     )')
         do ichan=1,nchan
            write(iwrite, '(2i8,5i6,D20.8)') ichan, ichl(ichan),starg(ichl(ichan)),mtarg(ichl(ichan)),gtarg(ichl(ichan)), &
                                             lvchl(ichan), mvchl(ichan), evchl(ichan)
         end do 
         write(iwrite,*) ""
      end if
         
      if (iprint .ge. 3)  then
!        Print Body      
         write(iwrite, '("  Channel     Electron Energy",7x, 8(a10, 10x, a10, 10x))') &
                        ("real "//multipole_labels(icomp), "imag "//multipole_labels(icomp), &
                         icomp=1,lmax_property*(lmax_property+2))
         write(iwrite, '("  --------------------------",8a40)') ("----------------------------------------", &
                        icomp=1,lmax_property*(lmax_property+2)) 
         do ibound=1, nbound
            do ichan=1,nchan
               do ienergy=1, no_scat_energies
                  write(iwrite, '(i8, 9D20.8)') ichan, escat(ienergy) , (re_pw_dipoles(ibound,ichan, icomp, ienergy),&
                                 im_pw_dipoles(ibound,ichan, icomp,ienergy), icomp=1,lmax_property*(lmax_property+2))
            
               end do
               write(iwrite,*) ""
            
            end do
         end do
      end if
      
      close(lu_pw_dipoles)
      
      return
  100 stop "ERROR: Reading partial wave dipoles"   
      
      end subroutine read_pw_dipoles
            
!     **********************************************************************************************
!
!     read_transdip2: A wrapper for PHOTO_READM: Reads inner region dipoles output by CDENPROP.
!      ZM changed to read-in only the dipoles corresponding to the first nstat_neut Bra states.
!
!     **********************************************************************************************      
      subroutine read_transdip2(iwrite, iutdip, ntset, nstat_neut, nstat, lmax_property,prop,dip_comp_present,&
                                bound_state_energies, ifail)
      implicit none
      integer :: iutdip, ntset, nstat, nstat_neut
      real(kind=idp),allocatable, dimension(:,:,:) :: prop,prop_temp
      integer :: iwrite, nnuc, iprnt, nold, ifail, iposit, nuccen, lmax_property, dip_comp_present(:)
      real(kind=idp) :: R,rmass,bound_state_energies(:)
      real(kind=idp), allocatable, dimension(:) :: charg
      real(kind=idp), allocatable, dimension(:,:) :: geonuc
      integer :: i,j,k,maxprop
      nold=0

      if (nstat_neut .le. 0) stop "in read_transdip2 the number of Bra states requested is .le. 0."
      bound_state_energies=0._idp

      maxprop = lmax_property**2+2*lmax_property !number of properties (not including overlaps)
      iposit = 0 !ZM this is for electrons; TODO this should be an argument to read_transdip2
      write(iwrite,'("Assuming scattering particle is electron.")')
      call photo_readm( iwrite, iutdip, ntset, R, nstat_neut, nstat, nnuc, nuccen, charg, geonuc, rmass, lmax_property, & 
     &                  prop,dip_comp_present,bound_state_energies, iprnt,nold,ifail,iposit)

      if (ifail /= 0) then
         write (iwrite, '(A,I0)') 'Error while reading transition dipole moments from unit ', iutdip
         stop 1
      end if

!     Unpack triangular to symmetric matrix
      allocate(prop_temp(nstat,nstat_neut,maxprop))
      prop_temp=0_idp

      do k=1, maxprop
         prop_temp(:,:,k)=transpose(prop(:,:,k))
         do j=1,nstat_neut
            do i=1, nstat
               prop_temp(i,j,k)=0_idp
            end do
         enddo
      end do

      do i=1,nstat_neut
         prop(i,1:nstat,1:maxprop)= prop(i,1:nstat,1:maxprop)+prop_temp(1:nstat,i,1:maxprop)
      enddo

      end subroutine read_transdip2

!     **********************************************************************************************
!
!     PHOTO_READM: READS MOMENTS ATTACHED TO UNIT LUTARG
!     (Adapted from READM which reads DENPROP output)
!  
!	   Definition of input variables - alex
!
!	   IWRITE  =  Write messages to this unit. I 
!	   LUTARG  =  Properties file. I
!	   NTSET   =  Set no? I
!	   R       =  Internuclear sep? Doesnt look like its used, leftover from diatomic code?
!	   nstat   =  Number of CI states. I
!     nstat_neut   =  Number of Bra states for which the properties are required. We assume nstat_neut > 0.
!	   NNUC    =  No. of nuclei (no. of geometry records). I
!	   NUCCEN  =  Seems to be set to NNUC+1. I
!	   CHARG   =  Charge on each nuclear centre. I(NNUC)
!	   GEONUC  =  x,y,z coord of each nuclei. D(3,NNUC)
!	   RMASS   =  Reduced mass. D
!	   lmax_property   =  Maximum L value for property operator. I 
!	   prop    =  Matrix containing properties D(nstat, nstat, maxprop)
!	   eshift  =  CI vector energy shift. D(nstat)
!	   IPRNT   =  Print flag. I
!	   NOLD    =  Indicates whether target prop. file is new or old style. I [0:1]
!	   IFAIL   =  Error Flag. I
!	   IPOSIT  =  Positron run Flag. I
!
!     THE DUMPFILE IS DIVIDED INTO DATASETS BY HEADER CARDS BEGINNING
!     WITH KEY = 6
!     Old style files with KEY = 9 can also be read
!
!     KEY = RECORD KEY = FIRST FIELD (I1)
!     GTARG is retained for compatiblity with other modules
!
!     **********************************************************************************************
      subroutine photo_readm( iwrite, lutarg, ntset, r, nstat_neut, nstat, nnuc, nuccen, charg, geonuc, rmass, &
     &                        lmax_property, prop,dip_comp_present,bound_state_energies, iprnt, nold, ifail, iposit )
      implicit none

!     Arguments
      integer :: iwrite, lutarg, ntset, nstat, nnuc, nuccen, lmax_property, iprnt, nold, ifail, &
     &           iposit, dip_comp_present(:), nstat_neut
      real(kind=idp) :: r,rmass,bound_state_energies(:)
      real(kind=idp),allocatable :: geonuc(:,:),charg(:),prop(:,:,:)     

!     Local variables
      integer :: ipass, key ,keyh, iset, nrecs, nstat1, nmom, isw, i, j, k, it, iseq, icharg, itg, &
     &           im, ikmax, it1, it2, iq, lop, mop, maxprop
      real(kind=idp) :: trpi, chargesign, x, y, z, dnx
      character(len=3) :: catom
      character(len=26) :: head
      integer,dimension(:) :: inx(8)
      integer,allocatable,dimension(:) :: ip, irp
      real(kind=idp) ,dimension(:) :: rmoi(3), amnuc(maxnuc)    
      equivalence (inx(1),key)
      dip_comp_present=0
      ifail = 0

      rewind lutarg
      trpi = two*sqrt(pi)   

!ZM fixed sign for electrons/positrons: -1 for electrons, +1 for positrons
      chargesign = -1.0
      if (iposit .ne. 0) chargesign = 1.0
!     ----- SEARCH DUMPFILE FOR REQUIRED DATASET
      IPASS = 1
   17 read(lutarg,11,end=40) key     !read the key 
      if(key .ne. 6) go to 21      !and check its a properties file
      backspace lutarg

      ! read the first line to determine property file format (see molecular_properties_data::read_properties in UKRmol-in)
      call readmh(lutarg, keyh, iset, nrecs, nnuc, nstat1, nmom, isw, rmoi)

      nstat=nstat1
!     allocate arrays
      maxprop = lmax_property**2+2*lmax_property
      allocate( prop(nstat_neut,nstat,maxprop), geonuc(3,nnuc), charg(nnuc), ip(nstat), irp(nstat) )

!     -----------------------------------------
!     SET HEADER    ...    KEY = 6 :
!     FIELD
!       2   SET NUMBER
!       3   NUMBER OF RECORDS IN SET
!       4   NUMBER OF GEOMETRY RECORDS ( NNUC )
!       5   NUMBER OF RECORDS OF TARGET DATA
!       6   NUMBER OF RECORDS OF MOMENT DATA
!       7   MOMENT TYPE SWITCH, ISW
!       8   Rotational constant AX (a.u.)
!       9   Rotational constant BY (a.u.)
!      10   Rotational constant CZ (a.u.)
!     -----------------------------------------
      if(nstat.gt.nstat1 .or. (ntset.ne.0.and.iset.ne.ntset)) then

!     ---THIS IS NOT THE REQUIRED DATA SET SO SKIP REMAINING RECORDS
         if(iprnt.ne.0) write(iwrite,98) ntset, nstat, iset, nstat1

         do 19 j=1,inx(3)
            read(lutarg,11,end=21)
   19    continue
         go to 17
      endif

!     -------------------------------------------------------
!     READ DATA DEFINING MOLECULAR GEOMETRY   ...   KEY = 8 :
!     FIELD
!       2   NUCLEAR SEQUENCE NUMBER (I)
!       3   Name of atom
!       4   NUCLEAR CHARGE
!       5   NUCLEAR MASS ( IN ATOMIC UNITS )
!       6
!       7
!       8
!       9   NUCLEAR POSITION, GEONUC(I)
!      10   26-CHARACTER HEADER FIELD
!     --------------------------------------------------------
      do 8 it=1,nnuc
         read(lutarg,1102,err=21) key, iseq, catom, icharg, rmass, x, y, z
         if(key .ne. 8) go to 21
         write(iwrite,120) it, catom, icharg, x, y, z, rmass
         charg(iseq) = icharg
         amnuc(iseq) = rmass
         geonuc(1,iseq) = x
         geonuc(2,iseq) = y
         geonuc(3,iseq) = z
    8 continue

      nuccen=nnuc+1
      ikmax = 0

      do 10 itg=1,nstat1
         read (lutarg, *, err=21, end=21) (inx(i),i=1,8), dnx
         if(key .ne. 5) go to 21
         ! save the neutral state energies here
         if (itg .le. nstat_neut) bound_state_energies(itg)= dnx
   10 continue

!     ------------------------------------------------------------------
!     RDTMOM DATA    ...    KEY = 1
!
!     FIELD
!       2   STATE INDEX          (I)
!       3   |M(I)|
!       4   STATE INDEX          (J)
!       5   |M(J)|
!       6   OPERATOR CENTER INDEX  KOP
!       7   OPERATOR L-VALUE       LOP
!       8   OPERATOR |M|-VALUE     MOP
!       9   TRANSITION MOMENT IN AU, ISW CONVENTION DETERMINED BY HEADER
!     ------------------------------------------------------------------

      do i=1,nstat_neut
         do j=1,nstat1
            do k=1,maxprop
               prop(i,j,k) = zero
            end do
         end do
      end do

      write(iwrite,20)
      do 60 im=1,nmom
         read (lutarg, *, err=21, end=21) (inx(i),i=1,8), dnx
         if(key .ne. 1) go to 21
         it1=inx(2)
         it2=inx(4)
         lop=inx(7)
         mop = inx(8)
         !ZM transfer properties only for the Bra states that we want
         if (it1 .le. nstat_neut) then
            if(lop .lt. 1 .or. lop .gt. lmax_property) go to 60
            if(inx(6).ne.nuccen) go to 21
            iq = lop*lop+lop+mop   ! indexing function for the properties 1st dipole has index iq=1
            dip_comp_present(iq)=1 ! Indicates which dipole components we have
            prop(it1,it2,iq) = chargesign * dnx
         endif

   60 continue
      return

   40 if(ipass.eq.1) then
         rewind lutarg
         ipass = ipass+1
         go to 17
      else
         write(iwrite,23)ntset,r
   23    FORMAT(' UNABLE TO FIND REQUIRED TARGET DATA SET',I5,F10.3)
         stop
      endif
      return

   21 write(iwrite,66)
   66 format(' ERROR in target properties data')
      write(iwrite,111) (inx(i),i=1,8),dnx,head
      ifail = 1
      return

   98 format(' Required',2i5,5x,'Skipped',2i5,f10.6)
   96 format(6i5,f10.5)
   18 format(/' Reduced mass',f9.1,' au')
  188 format(/' Target rotational constants (a.u.)',3d15.6)
  189 format(/' Nuclear centres '/4x,'Atom',2x,'Charge ', &
     & 5x,'X',9x,'Y',9x,'Z',9x,'Mass')
  120 format(i3,2x,a3,3x,i3,3x,3f10.6,2x,f10.6)
   20 format(/' Transition moments')
   95 format('Target states on properties file are not in energy order')
  196 format(3i5,f20.10)
   97 format(/' Target states '/6x,' Irrep ','Spin',5x,' Energy (au)')
   11 format(i1,i8,i3,i8,4i3,d20.12,2x,a26)
  111 format(i1,i8,i3,i8,4i3,d20.12,2x,a26)
 1100 format(i1,6i3,1x,3d20.12)
! 1101 format(i1,i3,i6,i3,i4,i6,i3,1x,3d20.12)
 1101 format(i1,i3,i9,i3,i4,i9,i3,1x,3d20.12)
 1103 format(i1,i3,i9,i9,i9,i9,i3,1x,3d20.12)
 1102 format(i1,i3,a3,i3,f10.4,3f20.10)
      end  subroutine photo_readm


!     **********************************************************************************************
!
!     WRITSH  writes header of Scattering Coefficients(SC) file on unit 
!     LUSCT.
!     If NSSET=1 on input then SC are written as first set,
!              0 then written at end of information (eoi).
!     On output NSSET holds the actual set number
!
!     **********************************************************************************************
      SUBROUTINE WRITSH(LUSCT, NSSET, SFORM, TITLE, MGVN, STOT, GUTOT, NCHAN, ICHL, LVCHL, MVCHL, &
     &                  EVCHL,NSTAT,NSCAT,nesc,RR,IPRNT,IWRITE,IFAIL)

      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11)  SFORM
      CHARACTER(LEN=80)  TITLE
      DIMENSION ichl(nchan),LVCHL(nchan),MVCHL(nchan),EVCHL(nchan)
      DATA KEYSC/88/ !Find out if there is a convention for this
      SAVE
!
!     position file at eoi or end of set number NSSET
      NSET=NSSET
      CALL GETSET(LUSCT,NSET,KEYSC,SFORM,IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(IWRITE,666)NSSET,LUSCT
        RETURN
      ENDIF
      IF(NSSET.NE.1)  NSET=NSET+1
      WRITE(IWRITE,16) NSET,LUSCT
      NSSET=NSET
      NINFO = 3
!
!     write header
      IF(SFORM.EQ.'FORMATTED')  THEN
         TEMP=dble(NSTAT+2)/10.0d0 ! need to change
         NDATA1=(NSTAT+2)/10
         IF (TEMP.GT.NDATA1) NDATA1=NDATA1+1
         TEMP=dble(NCHAN)/10.0d0
         NDATA2=NCHAN/10
         IF (TEMP.GT.NDATA2) NDATA2=NDATA2+1
         NDATA = (NDATA1+NDATA2)*NSCAT
         NREC = NDATA+NINFO
         WRITE(LUSCT,10) KEYSC,NSET,NREC,NINFO,NDATA
         WRITE(LUSCT,13) TITLE
         WRITE(LUSCT,10) NSCAT,MGVN,STOT,GUTOT,NSTAT,NCHAN,nesc
         WRITE(LUSCT,11) RR
      ELSE
         NDATA = NSCAT*2 ! need to change
         NREC = NINFO+NDATA
         WRITE(LUSCT) KEYSC,NSET,NREC,NINFO,NDATA
         WRITE(LUSCT) TITLE
         WRITE(LUSCT) NSCAT,MGVN,STOT,GUTOT,NSTAT,NCHAN,nesc
         WRITE(LUSCT) RR
         write(lusct) ichl, lvchl, mvchl, evchl
!         write(lusct) (ICHL(i), i=1,nchan)
!         write(lusct) (LVCHL(i), i=1,nchan)
!         write(lusct) (MVCHL(i), i=1,nchan)
!         write(lusct) (EVCHL(i), i=1,nchan)
      ENDIF
!
      IF(IPRNT.NE.0)  THEN
         WRITE(IWRITE,14)
         WRITE(IWRITE,100) KEYSC,NSET,NREC,NINFO,NDATA
         WRITE(IWRITE,130) TITLE
         WRITE(IWRITE,100) NSCAT,MGVN,STOT,GUTOT,NSTAT,NCHAN,nesc
         WRITE(IWRITE,101) RR
         DO I=1,NCHAN
          WRITE(IWRITE,140) ICHL(i),LVCHL(i),MVCHL(i),EVCHL(i)
         END DO
      ENDIF
!
      RETURN
   10 FORMAT(10I5)
   11 FORMAT(10F20.13)
   13 FORMAT(A80)
   14 FORMAT(/' Header on LUSCT')
  100 FORMAT(1X,10I5)
  101 FORMAT(1X,10F20.6)
  130 FORMAT(1X,A80)
  140 FORMAT(3I7, F20.10)
  666 FORMAT(/' UNABLE TO FIND SCATTERING STATE SET ',I0,' ON UNIT ',I0)
   16 FORMAT(/' Scattering state coefficients will be written to set ',I0,' on unit ',I0)
      END SUBROUTINE WRITSH
      
!     **********************************************************************************************
!
!     WRITSC  writes body of Scattering Coefficients(SC) file on unit LUSCT.
!
!     **********************************************************************************************     
      SUBROUTINE WRITSC(LUSCT,SFORM,ESCAT,NCHAN,NSTAT,NESC,AR,AI)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)

      INTEGER LUSCT,NCHAN,NSTAT, NESC
      CHARACTER(LEN=11) SFORM
      DIMENSION AR(NSTAT,nchan,nesc), AI(NSTAT,nchan,nesc)
      DIMENSION escat(nesc)
      integer I,K,n

      if (SFORM.EQ.'FORMATTED')THEN
         do n=1,NESC
            do I = 1,nchan
               WRITE(LUSCT,101) ESCAT(n),I
               WRITE(LUSCT,103) (AR(K,I,n),K=1,NSTAT)
               WRITE(LUSCT,102) ESCAT(n),I
               WRITE(LUSCT,103) (AI(K,I,n),K=1,NSTAT)
            end do
         end do
      else
         do n=1,NESC
            do I=1,nchan
               WRITE(LUSCT) ESCAT(n),I,(AR(K,I,n),K=1,NSTAT)
               WRITE(LUSCT) ESCAT(n),I,(AI(K,I,n),K=1,NSTAT)
            end do
         end do
      endif
      RETURN
!
 101  FORMAT(F20.10,I5,'   Real Part')
 102  FORMAT(F20.10,I5,'   Imaginary Part')
 103  FORMAT(10F20.13)
 104  FORMAT(/'Energy ',F12.8,' is too close to R-matrix pole',I3)

      END SUBROUTINE WRITSC
      
end module photo_outerio
