! Copyright 2019
!
! Alex G. Harvey with ontributions from Danilo S. Brambila and Zdenek Masin.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      module compak_defs
      implicit none

!     Parameters
      integer, parameter :: icp=selected_real_kind(15)
      integer, parameter :: idp=selected_real_kind(8)
!       integer, parameter :: icp=selected_real_kind(19)
!       integer, parameter :: idp=selected_real_kind(12)
      integer, parameter :: keych=10, keyrm=11
      real(kind=idp), parameter ::  pi=3.1415927_idp, ryd=0.073500_idp
      complex(kind=icp), parameter :: eye=(0,1.0d0)

      end module compak_defs

      module compak_procs
      
      contains

!     ******************************************************************
!
!     Print a complex matrix.
!
!     ******************************************************************
      subroutine print_matrix(lunit,n,m, A)
      use compak_defs
      implicit none
!     Arguments
      integer :: n,m, lunit
      real(kind=idp) :: A(n,m)
!       integer, optional :: isdiagonal
!     Local
      integer :: i,j
      character(len=40) :: str_format
      character(len=6) :: colfmt
      intent(in) :: A
   
      colfmt='D20.5'
      

!       call create_col_format(str_format,m,colfmt,3)
!       write(lunit, *) str_format
!       if (present(isdiagonal)) then
!           if (isdiagonal .eq. 1) then
! !           do i=1, min(n,m)
!             write(lunit,'(100(D20.5,x))')  (A(j,j), j=1,min(n,m))
! !           end do
!           else
!             do i=1, n
!               write(lunit,'(100(D20.5,x))')  (A(i,j), j=1,m)
!             end do
!           end if
!       else
        do i=1, n
          write(lunit,'(100(D20.5,1x))')  (A(i,j), j=1,m)
        end do
!       end if

      end subroutine print_matrix

      subroutine zprint_matrix(lunit,n,m, A)
      use compak_defs
      implicit none
!     Arguments
      integer :: n,m, lunit
      complex(kind=icp) :: A(n,m)
!     Local
      integer :: i,j
      character(len=40) :: str_format
      character(len=6) :: colfmt
      colfmt='d20.5'

!       call create_col_format(str_format,2*m,colfmt,3)
!       write(lunit, *) str_format
write(lunit,*) 'start'
      do i=1, n
         write(lunit,'(200(D20.5,1x))')  (A(i,j), j=1,m)
      end do

      end subroutine zprint_matrix

      subroutine create_col_format(fstring,ncols,colfmt,ncolspc)
      implicit none
!     Creates a format statement, fstring,  with a definable number of
!     columns, ncols, column format, colfmt, and spacing, ncolspc. 
      integer, intent(in) :: ncols,ncolspc
!       character(len=30), intent(inout) :: fstring,gstring
      character(len=40), intent(inout) :: fstring
      character(len=6), intent(in) :: colfmt
!     local variables
      character(len=3) :: str_ncols,str_ncolspc

      write( str_ncols, '(i3)' ) ncols
      write( str_ncolspc, '(i3)' ) ncolspc
!       fstring='('//str_ncols//'('//colfmt//','//' "," '//','//str_ncolspc//'x))'

      fstring='('//str_ncols//'('//colfmt//','//' "," '//','//str_ncolspc//'x))'
 
      end subroutine create_col_format       



!     ******************************************************************
!
!     Calculate wavefunction coefficients using equation 2
!
!     Equation A= E^-1 W^T R^-1  F
!
!     ******************************************************************
      subroutine calc_wavefunction_coefs_eq2(ibctyp,gamma,nopen,noc,nchan,nocsf,ntarg,en,etarg,eig,wamp,rmat,fx_plusminus,&
                                             fxp_plusminus,akr,aki)
      use blas_lapack_gbl, only: blasint
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: ibctyp,nopen,noc,nocsf,nchan,ntarg
      real(kind=idp) :: en,gamma
      complex(kind=icp) :: fx_plusminus(noc,nopen),fxp_plusminus(noc,nopen)
      real(kind=idp) :: etarg(ntarg),eig(nocsf),wamp(nchan,nocsf),rmat(nchan,nchan),akr(nocsf,nopen),aki(nocsf,nopen)

!     Local variables
      integer :: ierr,i, info
      real(kind=idp) :: bamp_norm
      integer(blasint),allocatable :: ipiv(:)
      real(kind=idp),allocatable :: matrix_identity(:,:), temp_matrix(:,:), rmat_inverse(:,:), wampt_x_rmat_inverse(:,:)
      complex(kind=icp),allocatable :: delta_pole(:), tmp(:)


!     Create matrix delta_pole = 1/(E_k-E) where E_k is an R-matrix pole
      allocate(delta_pole(nocsf), tmp(nocsf), stat=ierr)
      if(ierr .ne. 0) stop 'ERROR: allocation error'
      delta_pole=0.0_idp

!     Complex version with i* gamma added to denominator
      do i = 1, nocsf
         delta_pole(i)=1._idp/cmplx(EIG(i)-ETARG(1)-(EN/2._idp),ibctyp*gamma,kind=icp)
      end do 

      bamp_norm=sqrt(2.0)

!     Invert R-matrix
      allocate(matrix_identity(nchan,nchan))
      matrix_identity=0.0_idp

      do i=1,nchan
         matrix_identity(i,i)=1.0_idp
      end do

      info=0
      allocate(ipiv(nchan),temp_matrix(nchan,nchan))
      ipiv=0 
      temp_matrix=rmat
      
      call getrf(temp_matrix,ipiv)

      allocate(rmat_inverse(nchan,nchan))
      rmat_inverse=matrix_identity
      call getrs(temp_matrix,ipiv,rmat_inverse)

!     Calculate wavefunction ocefficients
      allocate(wampt_x_rmat_inverse(nocsf,noc))
      wampt_x_rmat_inverse=0._idp

      call gemm(wamp, rmat_inverse, wampt_x_rmat_inverse, 'T')

      call gemm(wampt_x_rmat_inverse, real(fx_plusminus, idp), akr)
      call gemm(wampt_x_rmat_inverse, aimag(fx_plusminus), aki)

      !$omp parallel do default(shared) private(i) firstprivate(tmp)
      do i = 1, nocsf
         tmp = cmplx(akr(i, 1:nopen), aki(i, 1:nopen), icp)
         tmp = tmp * bamp_norm * (0.5_idp) * delta_pole(i)
         akr(i, 1:nopen) = real(tmp, idp)
         aki(i, 1:nopen) = aimag(tmp)
      end do
      !$omp end parallel do

      end subroutine calc_wavefunction_coefs_eq2
!     ******************************************************************
!
!     Calculate wavefunction coefficients using equation 1
!
!     A=(1/2a) E^-1 W^T [a F' -b F], note: b=0 
!
!     ******************************************************************
      subroutine calc_wavefunction_coefs_eq1(ibctyp,gamma,nopen,noc,nchan,nocsf,ntarg,en,etarg,eig,wamp,fx_plusminus,fxp_plusminus,&
                                             akr,aki)
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: ibctyp,nopen,noc,nocsf,nchan,ntarg
      real(kind=idp) :: en,gamma
      complex(kind=icp) :: fx_plusminus(noc,nopen),fxp_plusminus(noc,nopen)
      real(kind=idp) :: etarg(ntarg),eig(nocsf),wamp(nchan,nocsf), akr(nocsf,nopen), aki(nocsf,nopen)

!     Local variables
      integer :: ierr,i,j
      real(kind=idp) :: bamp_norm
      complex(kind=icp), allocatable :: delta_pole(:), tmp(:)


!     Create matrix delta_pole = 1/(E_k-E) where E_k is an R-matrix pole
      allocate(delta_pole(nocsf), tmp(nopen), stat=ierr)
      if(ierr .ne. 0) stop 'ERROR: allocation error'
      delta_pole=0.0_idp

!     Complex version with i* gamma added to denominator
      do i = 1, nocsf
         delta_pole(i)=1._idp/cmplx(EIG(i)-ETARG(1)-(EN/2._idp),ibctyp*gamma,kind=icp)
      end do

      bamp_norm=sqrt(2.0)

      call gemm(wamp, real(fxp_plusminus(1:noc,1:nopen), idp), akr(1:nocsf, 1:nopen), 'T')
      call gemm(wamp, aimag(fxp_plusminus(1:noc,1:nopen)), aki(1:nocsf, 1:nopen), 'T')

      !$omp parallel do default(shared) private(i) firstprivate(tmp)
      do i = 1, nocsf
         tmp = cmplx(akr(i, 1:nopen), aki(i, 1:nopen), icp)
         tmp = tmp * bamp_norm * (0.5_idp) * delta_pole(i)
         akr(i, 1:nopen) = real(tmp, idp)
         aki(i, 1:nopen) = aimag(tmp)
      end do
      !$omp end parallel do

      end subroutine calc_wavefunction_coefs_eq1
!     ******************************************************************
!
!     Calculate wavefunction coefficients using equation 1
!
!     A=(1/2a) E^-1 W^T [a F' -b F], note: b=0 
!
!     ******************************************************************
      subroutine calc_partial_wave_dipoles_eq1( ibctyp, gamma, nopen, noc, nchan, nocsf, ntarg, en,&
     &                                          etarg, eig, inner_dipoles, wamp, fx_plusminus, & 
     &                                          fxp_plusminus, pw_dipoles)
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: ibctyp, nopen, noc, nocsf, nchan, ntarg
      real(kind=idp) :: en, gamma
      complex(kind=icp) :: fx_plusminus(noc,nopen),fxp_plusminus(noc,nopen),pw_dipoles(:,:,:)
      real(kind=idp) :: etarg(ntarg), eig(nocsf), wamp(nchan,nocsf), inner_dipoles(:,:,:)

!     Local variables
      integer :: ierr,i,j, icomponent, no_components, no_bound_states
      real(kind=idp) :: bamp_norm
      complex(kind=icp) :: z
      real(kind=idp),allocatable :: re_delta_pole(:), im_delta_pole(:), re_DE(:,:), im_DE(:,:), re_DEwT(:,:), im_DEwT(:,:)
      complex(kind=icp),allocatable :: DEwT(:,:)

!     Create matrix delta_pole = 1/(E_k-E) where E_k is an R-matrix pole
      allocate (re_delta_pole(nocsf), im_delta_pole(nocsf), stat=ierr)
      if (ierr /= 0) stop 'ERROR: Allocation error 1 in calc_partial_wave_dipoles_eq1'

!     Complex version with i* gamma added to denominator
      do i = 1, nocsf
         z = 1._idp / cmplx(EIG(i)-ETARG(1)-(EN/2._idp), ibctyp*gamma, kind=icp)
         re_delta_pole(i) = real(z, idp)
         im_delta_pole(i) = aimag(z)
      end do

      no_bound_states=size(inner_dipoles,1)
      no_components=size(inner_dipoles,3)

      allocate (re_DE(no_bound_states, nocsf), im_DE(no_bound_states, nocsf), &
                re_DEwT(no_bound_states, noc), im_DEwT(no_bound_states, noc), DEwT(no_bound_states, noc), stat=ierr)
      if (ierr /= 0) stop 'ERROR: Allocation error 2 in calc_partial_wave_dipoles_eq1'

      bamp_norm=sqrt(2.0_idp)

      do icomponent=1,no_components

         ! multiply inner transition dipoles with R-matrix poles
         do i = 1, nocsf
            re_DE(:, i) = inner_dipoles(:, i, icomponent) * re_delta_pole(i)
            if (ibctyp*gamma /= 0) then
               im_DE(:, i) = inner_dipoles(:, i, icomponent) * im_delta_pole(i)
            end if
         end do

         ! contract with boundary amplitudes
         re_DEwT = 0
         im_DEwT = 0
         call gemm(re_DE, wamp, re_DEwT, 'N', 'T')
         if (ibctyp*gamma /= 0) then
            call gemm(im_DE, wamp, im_DEwT, 'N', 'T')
         end if
         DEwT = cmplx(re_DEwT, im_DEwT, icp)

         ! contract with boundary condition
         call gemm(DEwT, fxp_plusminus(1:noc,1:nopen), pw_dipoles(:,1:nopen,icomponent))

      end do

      pw_dipoles=bamp_norm*(0.5_idp)*pw_dipoles

      end subroutine calc_partial_wave_dipoles_eq1

! !     ******************************************************************
! !
! !     Apply scattering or photionisation boundary conditions
! !     (Outgoing or Incoming wave solutions respectively)
! !
! !     ******************************************************************
!       subroutine apply_boundary_conditions(nopen,noc,nchan,en,echl,isfullk,complex_fkmat,boundary_conditions,fx,fxp,fx_plusminus,fxp_plusminus)
!       use compak_defs
!       use lapack95_compak
!       use blas95_compak
!       implicit none
! 
! !     Argument variables
!       integer :: nopen,noc,nchan,isfullk
!       real(kind=idp) :: en
!       real(kind=idp) :: echl(nchan),fx(nchan,nchan,2),fxp(nchan,nchan,2)
!       complex(kind=icp) :: boundary_conditions(nopen,nopen),fx_plusminus(noc, nopen),fxp_plusminus(noc, nopen),complex_fkmat(nchan,nchan)
! 
! !     Local variables
!       integer :: i,j
!       real(kind=idp), allocatable :: energy(:),dmomentum(:),&
!      &                     root_momentum(:),root_momentum_matrix(:,:)
! 
!       complex(kind=icp), allocatable :: complex_fx(:,:,:),complex_fxp(:,:,:), &
!      &                     fx_times_fkmat_open(:,:),fxp_times_fkmat_open(:,:),&
!      &                     fx_times_fkmat_closed(:,:),fxp_times_fkmat_closed(:,:)
! 
!       intent(in) :: nopen,noc,nchan,complex_fkmat,boundary_conditions,fx,fxp
!       intent(out) :: fx_plusminus,fxp_plusminus
! 
! !     Create matrix k_{ii}= k_{i}^(1/2) where k is the momentum of the
! !     (outgoing or incoming?) electron momentum.
! 
!       allocate(energy(nchan),dmomentum(nchan),root_momentum(nchan))
!       energy = en-echl
!       dmomentum = sqrt(abs(energy))
!       root_momentum=sqrt(dmomentum)
! 
!       allocate(root_momentum_matrix(nchan,nchan))
!       do i = 1, nchan
!          root_momentum_matrix(i,i)=1._idp/root_momentum(i)
!       end do
! 
! !     Applying boundary conditions for open channels
!       allocate(fx_times_fkmat_open(nopen,nopen),fxp_times_fkmat_open(nopen,nopen))
!       fx_times_fkmat_open=0.0_idp;fxp_times_fkmat_open=0.0_idp
! 
!       allocate(complex_fx(nchan,nchan,2),complex_fxp(nchan,nchan,2))
!       complex_fx=fx
!       complex_fxp=fxp
! 
!       call gemm(complex_fx(1:nopen,1:nopen,2),complex_fkmat(1:nopen,1:nopen), fx_times_fkmat_open)
!       call gemm(complex_fxp(1:nopen,1:nopen,2),complex_fkmat(1:nopen,1:nopen), fxp_times_fkmat_open)
!    
!       fx_times_fkmat_open=fx_times_fkmat_open+complex_fx(1:nopen,1:nopen,1)
!       fxp_times_fkmat_open=fxp_times_fkmat_open+complex_fxp(1:nopen,1:nopen,1)
!       call gemm(fx_times_fkmat_open,boundary_conditions,fx_plusminus(1:nopen,1:nopen))
!       call gemm(fxp_times_fkmat_open,boundary_conditions,fxp_plusminus(1:nopen,1:nopen))
! 
!       if (isfullk .eq. 1) then
! !     Construct closed channels
! 
!          allocate(fx_times_fkmat_closed(nchan,nopen),fxp_times_fkmat_closed(nchan,nopen))
!          fx_times_fkmat_closed=0._idp;fxp_times_fkmat_closed=0._idp
! 
! !        using matmul
! !          fx_times_fkmat_closed(nopen+1:nchan,1:nopen)=matmul(complex_fx(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen))
! !          fxp_times_fkmat_closed(nopen+1:nchan,1:nopen)=matmul(complex_fxp(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen))
! 
! !          fx_plusminus(nopen+1:nchan,1:nopen)=matmul( fx_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions)
! !          fxp_plusminus(nopen+1:nchan,1:nopen)=matmul( fxp_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions)
! 
! !        using blas95_compak
!          call gemm(complex_fx(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen),fx_times_fkmat_closed(nopen+1:nchan,1:nopen))
!          call gemm(complex_fxp(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen),fxp_times_fkmat_closed(nopen+1:nchan,1:nopen))
! 
!          call gemm(fx_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions,fx_plusminus(nopen+1:nchan,1:nopen))
!          call gemm(fxp_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions,fxp_plusminus(nopen+1:nchan,1:nopen))
! 
!       end if
! 
! 
! 
! 
! !     Apply normalisation factor
!       fx_plusminus=fx_plusminus*sqrt(2.0_idp/pi)!*root_momentum(1)
!       fxp_plusminus=fxp_plusminus*sqrt(2.0_idp/pi)!*root_momentum(1)
! 
!       end subroutine apply_boundary_conditions

!     ******************************************************************
!
!     Apply scattering or photionisation boundary conditions
!     (Outgoing or Incoming wave solutions respectively)
!
!     ******************************************************************
      subroutine apply_boundary_conditions(nopen,noc,nchan,en,echl,isfullk,complex_fkmat,boundary_conditions,fx,fxp,fx_plusminus,&
                                           fxp_plusminus)
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: nopen,noc,nchan,isfullk
      real(kind=idp) :: en
      real(kind=idp) :: echl(nchan),fx(nchan,nchan,2),fxp(nchan,nchan,2)
      complex(kind=icp) :: boundary_conditions(nopen,nopen),fx_plusminus(noc, nopen),fxp_plusminus(noc, nopen),&
                           complex_fkmat(nchan,nchan)

!     Local variables
      integer :: i,j
      real(kind=idp), allocatable :: energy(:),dmomentum(:),&
     &                     root_momentum(:),root_momentum_matrix(:,:)

      complex(kind=icp), allocatable :: complex_fx(:,:,:),complex_fxp(:,:,:), &
     &                     fx_times_fkmat_open(:,:),fxp_times_fkmat_open(:,:),&
     &                     fx_times_fkmat_closed(:,:),fxp_times_fkmat_closed(:,:), &
     &                     fx_times_fkmat_all(:,:),fxp_times_fkmat_all(:,:)

      intent(in) :: nopen,noc,nchan,complex_fkmat,boundary_conditions,fx,fxp
      intent(out) :: fx_plusminus,fxp_plusminus

!     Create matrix k_{ii}= k_{i}^(1/2) where k is the momentum of the
!     (outgoing or incoming?) electron momentum.

      allocate(energy(nchan),dmomentum(nchan),root_momentum(nchan))
      energy = en-echl
      dmomentum = sqrt(abs(energy))
      root_momentum=sqrt(dmomentum)

      allocate(root_momentum_matrix(nchan,nchan))
      do i = 1, nchan
         root_momentum_matrix(i,i)=1._idp/root_momentum(i)
      end do

!     Applying boundary conditions for open channels
      allocate(fx_times_fkmat_open(nopen,nopen),fxp_times_fkmat_open(nopen,nopen))
      fx_times_fkmat_open=0.0_idp;fxp_times_fkmat_open=0.0_idp

      allocate(fx_times_fkmat_all(nchan,nopen),fxp_times_fkmat_all(nchan,nopen))
      fx_times_fkmat_all=0.0_idp;fxp_times_fkmat_all=0.0_idp


      allocate(complex_fx(nchan,nchan,2),complex_fxp(nchan,nchan,2))
      complex_fx=fx
      complex_fxp=fxp

!       write(888,*) " fx_1"
!       call print_matrix(888,nchan,nchan,fx(:,:,1))
!       write(888,*) " fx_2"
!       call print_matrix(888,nchan,nchan,fx(:,:,2))
!       write(888,*) " fxp_1"
!       call print_matrix(888,nchan,nchan,fxp(:,:,1))
!       write(888,*) " fxp_2"
!       call print_matrix(888,nchan,nchan,fxp(:,:,2))

      call gemm(complex_fx(1:nchan,1:nchan,2),complex_fkmat(1:nchan,1:nopen), fx_times_fkmat_all)
      call gemm(complex_fxp(1:nchan,1:nchan,2),complex_fkmat(1:nchan,1:nopen), fxp_times_fkmat_all)

      fx_times_fkmat_all=fx_times_fkmat_all+complex_fx(1:nchan,1:nopen,1)
      fxp_times_fkmat_all=fxp_times_fkmat_all+complex_fxp(1:nchan,1:nopen,1)

!       write(888,*) " fx_times_fkmat_all"
!       call zprint_matrix(888,nchan,nopen,fx_times_fkmat_all)
!       write(888,*) " fxp_times_fkmat_all"
!       call zprint_matrix(888,nchan,nopen,fxp_times_fkmat_all)
!TEST: New backprop scheme where the K matrix is premultiplied before propagation
!       fx_times_fkmat_all=fx(:,:,1)
!       fxp_times_fkmat_all=fxp(:,:,1)


      call gemm(fx_times_fkmat_all,boundary_conditions,fx_plusminus(1:nchan,1:nopen))
      call gemm(fxp_times_fkmat_all,boundary_conditions,fxp_plusminus(1:nchan,1:nopen))

      if (isfullk .eq. 1) then
!     Construct closed channels

!          allocate(fx_times_fkmat_closed(nchan,nopen),fxp_times_fkmat_closed(nchan,nopen))
!          fx_times_fkmat_closed=0._idp;fxp_times_fkmat_closed=0._idp
! 
! !        using matmul
! !          fx_times_fkmat_closed(nopen+1:nchan,1:nopen)=matmul(complex_fx(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen))
! !          fxp_times_fkmat_closed(nopen+1:nchan,1:nopen)=matmul(complex_fxp(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen))
! 
! !          fx_plusminus(nopen+1:nchan,1:nopen)=matmul( fx_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions)
! !          fxp_plusminus(nopen+1:nchan,1:nopen)=matmul( fxp_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions)
! 
! !        using blas95_compak
!          call gemm(complex_fx(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen),fx_times_fkmat_closed(nopen+1:nchan,1:nopen))
!          call gemm(complex_fxp(nopen+1:nchan,nopen+1:nchan,2),complex_fkmat(nopen+1:nchan,1:nopen),fxp_times_fkmat_closed(nopen+1:nchan,1:nopen))
! 
!          call gemm(fx_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions,fx_plusminus(nopen+1:nchan,1:nopen))
!          call gemm(fxp_times_fkmat_closed(nopen+1:nchan,1:nopen),boundary_conditions,fxp_plusminus(nopen+1:nchan,1:nopen))

      end if




!     Apply normalisation factor
      fx_plusminus=fx_plusminus*sqrt(2.0_idp/pi)!*root_momentum(1)
      fxp_plusminus=fxp_plusminus*sqrt(2.0_idp/pi)!*root_momentum(1)
!       write(888,*) " fx_plusminus real"
!       call print_matrix(888,nchan,nopen,real(fx_plusminus))
!       write(888,*) " fx_plusminus imag"
!       call print_matrix(888,nchan,nopen,imag(fx_plusminus))
!       write(888,*) " fxp_plusminus real"
!       call print_matrix(888,nchan,nopen,real(fxp_plusminus))
!       write(888,*) " fxp_plusminus imag"
!       call print_matrix(888,nchan,nopen,imag(fxp_plusminus))
!       write(888,*) " fx_plusminus"
!       call zprint_matrix(888,nchan,nopen,fx_plusminus)
!       write(888,*) " fxp_plusminus"
!       call zprint_matrix(888,nchan,nopen,fxp_plusminus)

      end subroutine apply_boundary_conditions
!     ******************************************************************
!
!     Create the scattering or photionisation boundary conditions
!     (Outgoing or Incoming wave solutions respectively)
!
!     ******************************************************************
      subroutine create_boundary_conditions(ibctyp,nopen,nchan,complex_fkmat,boundary_conditions)
      use blas_lapack_gbl, only: blasint
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: ibctyp,nopen,nchan
      complex(kind=icp), dimension(:,:) :: boundary_conditions(nopen,nopen), complex_fkmat(nchan,nchan)

!     Local variables
      integer :: info,i,j
      integer(blasint), allocatable :: ipiv(:)
      complex(kind=icp),allocatable,dimension(:,:)  :: zidentity_plusminus_iK, temp_matrix,matrix_identity

      intent(in) :: ibctyp,nopen
      intent(inout) :: complex_fkmat
      intent(out) :: boundary_conditions

!     Create the identity matrix
      allocate(matrix_identity(nopen,nopen))
      matrix_identity=0._idp

      do i=1,nopen
         matrix_identity(i,i)=1.0_idp
      end do

!     Create (I +/- iK)
      allocate(zidentity_plusminus_iK(nopen,nopen))
      zidentity_plusminus_iK=matrix_identity+ibctyp*eye*complex_fkmat(1:nopen,1:nopen)

!     Inverting [1 +/- iK] using blas/lapack routines
      info=0
      boundary_conditions=matrix_identity

      allocate(ipiv(nopen),temp_matrix(nopen,nopen))
      ipiv=0 
      temp_matrix= zidentity_plusminus_iK
      
      call getrf(temp_matrix,ipiv)

      call getrs(temp_matrix,ipiv,boundary_conditions)

! write(31111,*) boundary_conditions
! stop
      end subroutine create_boundary_conditions

      subroutine apply_boundary_conditions_no_inversion(ibctyp,nopen,noc,nchan,en,echl,isfullk,complex_fkmat,fx,fxp,fx_plusminus,&
                                                        fxp_plusminus)
      use blas_lapack_gbl, only: blasint
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: nopen,noc,nchan,isfullk,ibctyp
      real(kind=idp) :: en
      real(kind=idp) :: echl(nchan),fx(nchan,nchan,2),fxp(nchan,nchan,2)
      complex(kind=icp) :: fx_plusminus(noc, nopen),fxp_plusminus(noc, nopen),complex_fkmat(nchan,nchan)

!     Local variables
      integer :: i,j, info
      integer(blasint), allocatable :: ipiv(:)
      real(kind=idp), allocatable :: energy(:),dmomentum(:),&
     &                     root_momentum(:),root_momentum_matrix(:,:)

      complex(kind=icp), allocatable :: complex_fx(:,:,:),complex_fxp(:,:,:), &
     &                     fx_times_fkmat_all(:,:),fxp_times_fkmat_all(:,:),&
     &                     fx_times_fkmat_closed(:,:),fxp_times_fkmat_closed(:,:), &
     &                     fx_times_fkmat_all_trans(:,:),fxp_times_fkmat_all_trans(:,:)

      complex(kind=icp), dimension(:,:) :: boundary_conditions(nopen,nopen)

      complex(kind=icp),allocatable,dimension(:,:)  :: zidentity_plusminus_iK,&
     &                     temp_matrix, matrix_identity

      intent(in) :: nopen,noc,nchan,complex_fkmat,fx,fxp
      intent(out) :: fx_plusminus,fxp_plusminus


!     Create the identity matrix
      allocate(matrix_identity(nopen,nopen))
      matrix_identity=0._idp

      do i=1,nopen
         matrix_identity(i,i)=1.0_idp
      end do

!     Create (I +/- iK)
      allocate(zidentity_plusminus_iK(nopen,nopen))
      zidentity_plusminus_iK=matrix_identity+ibctyp*eye*complex_fkmat(1:nopen,1:nopen)

!     Inverting [1 +/- iK] using blas/lapack routines
      info=0
      boundary_conditions=zidentity_plusminus_iK


      allocate(ipiv(nopen),temp_matrix(nopen,nopen))
      ipiv=0 
      temp_matrix= zidentity_plusminus_iK
      
      call getrf(temp_matrix,ipiv)

 !     Create matrix k_{ii}= k_{i}^(1/2) where k is the momentum of the
!     (outgoing or incoming?) electron momentum.

      allocate(energy(nchan),dmomentum(nchan),root_momentum(nchan))
      energy = en-echl
      dmomentum = sqrt(abs(energy))
      root_momentum=sqrt(dmomentum)

      allocate(root_momentum_matrix(nchan,nchan))
      do i = 1, nchan
         root_momentum_matrix(i,i)=1._idp/root_momentum(i)
      end do

!     Applying boundary conditions for open channels
      allocate(fx_times_fkmat_all(nchan,nopen),fxp_times_fkmat_all(nchan,nopen))
      fx_times_fkmat_all=0.0_idp;fxp_times_fkmat_all=0.0_idp

      allocate(complex_fx(nchan,nchan,2),complex_fxp(nchan,nchan,2))
      complex_fx=fx
      complex_fxp=fxp

      call gemm(complex_fx(1:nchan,1:nchan,2),complex_fkmat(1:nchan,1:nopen), fx_times_fkmat_all)
      call gemm(complex_fxp(1:nchan,1:nchan,2),complex_fkmat(1:nchan,1:nopen), fxp_times_fkmat_all)
   
      fx_times_fkmat_all=fx_times_fkmat_all+complex_fx(1:nchan,1:nopen,1)
      fxp_times_fkmat_all=fxp_times_fkmat_all+complex_fxp(1:nchan,1:nopen,1)



!       call gemm(fx_times_fkmat_all,boundary_conditions,fx_plusminus(1:nchan,1:nopen))
!       call gemm(fxp_times_fkmat_all,boundary_conditions,fxp_plusminus(1:nchan,1:nopen))


      allocate(fx_times_fkmat_all_trans(nopen,nchan),fxp_times_fkmat_all_trans(nopen,nchan))
      fx_times_fkmat_all_trans=0.0_idp;fxp_times_fkmat_all_trans=0.0_idp

      fx_times_fkmat_all_trans=transpose(fx_times_fkmat_all)
      fxp_times_fkmat_all_trans=transpose(fxp_times_fkmat_all)

      call getrs(temp_matrix,ipiv,fx_times_fkmat_all_trans,'T')
      call getrs(temp_matrix,ipiv,fxp_times_fkmat_all_trans,'T')

      fx_plusminus(1:nchan,1:nopen)=transpose(fx_times_fkmat_all_trans)
      fxp_plusminus(1:nchan,1:nopen)=transpose(fxp_times_fkmat_all_trans)

!     Apply normalisation factor
      fx_plusminus=fx_plusminus*sqrt(2.0_idp/pi)!*root_momentum(1)
      fxp_plusminus=fxp_plusminus*sqrt(2.0_idp/pi)!*root_momentum(1)
     
!       write(888,*) " fx_plusminus real"
!       call print_matrix(888,nchan,nopen,real(fx_plusminus))
!       write(888,*) " fx_plusminus imag"
!       call print_matrix(888,nchan,nopen,imag(fx_plusminus))
!       write(888,*) " fxp_plusminus real"
!       call print_matrix(888,nchan,nopen,real(fxp_plusminus))
!       write(888,*) " fxp_plusminus imag"
!       call print_matrix(888,nchan,nopen,imag(fxp_plusminus))
      end subroutine apply_boundary_conditions_no_inversion

      subroutine kmat_smooth(nchan,bsto,nopen,f,fp,complex_rmat,complex_akmat,complex_fkmat)
      use blas_lapack_gbl, only: blasint
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none
!
!     ******************************************************************
!
!     k-matrix calculation
!
!     nchan        number of channels
!     bsto         logarithmic derivative/matching radius
!     nopen        number of open channels
!     f, fp        external region solutions and derivatives
!                  (assumed in correct locations)
!     rmat         internal region r-matrix
!
!     output :
!     akmat        k-matrix
!     fkmat          the full k-matrix including closed channels
!              needed for compak - alexh 11.11.10
!
!     aa,bb        work space, each of length nchan*nchan
!     x            work space, of length 2*nchan
!
!     ******************************************************************
!
   

!     Argument variables
      integer :: nchan,nopen
      real(kind=idp), dimension(:) :: bsto(nchan)
      real(kind=idp), dimension(:,:,:) :: f(nchan,nchan,2),fp(nchan,nchan,2)
      complex(kind=icp), dimension(:,:) :: complex_rmat(nchan,nchan)
      complex(kind=icp), dimension(:,:) ::complex_akmat(nchan,nchan),complex_fkmat(nchan,nchan)

!     Local variables
      integer :: i,j,k,info
      complex(kind=icp) :: df
      integer(blasint), allocatable, dimension(:) :: ipiv
      complex(kind=icp), allocatable, dimension(:,:) :: aa,bb

!
      allocate(aa(nchan,nchan),bb(nchan,nchan))
      complex_akmat=0._idp;complex_fkmat=0._idp;aa=0._idp;bb=0._idp

      do j=1,nchan
         do i=1,nchan
            aa(i,j) = f(i,j,2)
         end do
      end do
!
      do j=1,nchan
         do k=1,nchan
            df = fp(k,j,2)-bsto(k)*f(k,j,2)
            do i=1,nchan
               aa(i,j) = aa(i,j)-complex_rmat(i,k)*df
            end do
         end do
      end do
!
      do j=1,nopen
         do i=1,nchan
            bb(i,j)=-f(i,j,1)
         end do
      end do
!
      do j=1,nopen
         do k=1,nchan
            df = fp(k,j,1)-bsto(k)*f(k,j,1)
            do i=1,nchan
               bb(i,j)=bb(i,j)+complex_rmat(i,k)*df
            end do
         end do
      end do

      if(nchan.eq.1) then
        bb(1,1)=bb(1,1)/aa(1,1)
      else
!         call ma01a(aa,bb,nchan,nopen,0,nchan,nchan,x,x(nchan+1))

      info=0
      allocate(ipiv(nchan))
      ipiv=0

      call getrf(aa,ipiv)

      call getrs(aa,ipiv,bb)



      endif
!
      complex_fkmat=bb
      complex_akmat(1:nopen,1:nopen) = bb(1:nopen,1:nopen)
!
      return
931   format(i5,3x,i5,3x,2(d15.5,3x))
      end subroutine kmat_smooth

      subroutine construct_complex_rmatrix(nchan,nocsf,wamp,eig,scattering_energy,ibctyp,gamma,complex_rmat)
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none
!
!     ******************************************************************
!
!      complex_vrmat2 creates the complex R-matrix
!
!     ******************************************************************
!
!     input parameters are
!
!     ******************************************************************
!

!     Argument variables
      integer :: nchan, nocsf,ibctyp
      real(kind=idp) :: scattering_energy,gamma
      real(kind=idp), dimension(:) :: eig(nocsf)
      real(kind=idp), dimension(:,:) :: wamp(nchan,nocsf)
      complex(kind=icp), dimension(:,:) :: complex_rmat(nchan,nchan)

!     Local variables
      integer :: i
      real(kind=idp), allocatable, dimension(:,:) :: wampt
      complex(kind=icp), allocatable, dimension(:,:) :: delta_pole,complex_wamp, complex_wampt,wamp_x_delta_pole

      allocate(delta_pole(nocsf,nocsf))
      delta_pole=0._idp

      do i = 1, nocsf
         delta_pole(i,i)=1._idp/cmplx(eig(i)-scattering_energy,ibctyp*gamma,kind=icp)
      end do 

      allocate(wampt(nocsf,nchan))
      wampt=0.0_idp

      wampt=transpose(wamp)

      allocate(complex_wamp(nchan,nocsf), complex_wampt(nocsf,nchan),wamp_x_delta_pole(nchan,nocsf))
      complex_wamp=wamp
      complex_wampt=wampt

      call gemm(complex_wamp,delta_pole,wamp_x_delta_pole)
      call gemm(wamp_x_delta_pole,complex_wampt,complex_rmat)
      return
!
      end subroutine construct_complex_rmatrix

      subroutine read_averaged_kmatrix_header(luavkmat,nchan,no_energy_points)
      use compak_defs
      implicit none
      integer :: luavkmat,nchan,no_energy_points

      read(luavkmat) nchan,no_energy_points
      close(luavkmat)
      end subroutine read_averaged_kmatrix_header

      subroutine read_averaged_kmatrix_body(luavkmat,nchan,no_energy_points,kmatrix)
      use compak_defs
      implicit none
      integer :: luavkmat,nchan,no_energy_points
      complex(kind=icp), dimension(:,:,:) :: kmatrix(nchan,nchan,no_energy_points)

      integer :: nchan_temp, no_energy_points_temp

      rewind(luavkmat)
      read(luavkmat) nchan_temp,no_energy_points_temp
      read(luavkmat) kmatrix
      close(luavkmat)
      end subroutine read_averaged_kmatrix_body

!     ******************************************************************
!     
!     COMPAK: The main driver routine for constructing wavefuntion
!             coefficients.
!
!     INPUT:
!
!     nchan = Number of channels
!     nopen = Number of open channels
!     fx    = Solution to reduced radial equation
!     fxp   = Derivative of solution to reduced radial equation
!     nocsf = Number of configurations in CI expansion = Number of 
!             inner regions hamiltonian solutions (CI vectors)
!     eig   = 
!     wamp  = R-matrix boundary amplitudes
!     akmat = The open-open part of the K-matrix
!     fkmat = The full K-matrix including closed channels
!     etarg = The target energies
!     echl  = The channel energies
!     en    = The scattering energy
!     ntarg = The number of target states
!     rmat  = The R-matrix
!     rmatr = The R-matrix radius
!
!     OUTPUT:
!
!     ifail = 0 for succesful completion of compak, 1 otherwise
!     akr   = The real part of the wavefunction coefficients
!     aki   = The imaginary part of the wavfunction coefficients
!
!     ******************************************************************
      subroutine compak( nchan,nopen,fx,fxp, nocsf,eig,wamp,ifail,akmat, &
     &                   fkmat,etarg,echl,nrk,en,ntarg,rmat,rmatr, akr, aki, &
     &                   inner_dipoles, re_pw_dipoles, im_pw_dipoles)
      use compak_defs
      use lapack95_compak
      use blas95_compak
      implicit none

!     Argument variables
      integer :: nchan,nopen,nocsf,ifail,ntarg,nrk
      real(kind=idp) ::  en,rmatr
      real(kind=idp), dimension(:) :: eig(nocsf),echl(nchan),etarg(ntarg)       
      real(kind=idp), dimension(:,:) :: wamp(nchan,nocsf),akmat(nchan,nchan),fkmat(nchan,nchan),rmat(nchan,nchan)
      real(kind=idp), optional :: akr(nocsf,nchan), aki(nocsf,nchan)
      real(kind=idp), optional :: inner_dipoles(:,:,:), re_pw_dipoles(:,:,:), im_pw_dipoles(:,:,:)
      real(kind=idp), dimension(:,:,:) :: fx(nchan,nchan,2),fxp(nchan,nchan,2)

      intent(in) :: nchan,nopen,nocsf,ntarg,en,rmatr,eig,echl,etarg,wamp,rmat,fx,fxp,nrk
      intent(inout) :: akmat,fkmat
      intent(out) :: ifail,akr, aki

!     To potentially become argument variables in the future
      integer :: iequation,ibctyp,isfullk,iaverage_type,luavkmat
      real(kind=idp) :: gamma

!     Local variables
      integer :: noc,i,j, nchan_k, no_energy_points, ien
      real(kind=idp) :: scattering_energy
      real(kind=idp), allocatable :: wavefunction_coefs(:,:,:)
      complex(kind=icp),allocatable :: boundary_conditions(:,:), fx_plusminus(:,:), &
     &                                 fxp_plusminus(:,:), kmatrix(:,:,:), &
     &                                 pw_dipoles(:,:,:)

!     for complex r-matrix
      complex(kind=icp),allocatable :: complex_rmat(:,:), complex_akmat(:,:), complex_fkmat(:,:)
      real(kind=idp), dimension(:) :: bsto(nchan) ! should be argument variable

!Timing
real(8) :: start, finish


      ifail=0
!     Construct and apply photionisation/scattering boundary conditions.
!     ------------------------------------------------------------------

!     The following variables should eventually go into the appropriate
!     namelist in rsolve. For the moment they must be set prior to
!     compilation.

!     iequation determines which equation is used to calculate the
!     wavefunction coefficients. 
      iequation=1      ! = 1 equation without inverse R-matrix
                       ! = 2 equation with inverse R-matrix

!     Gamma adds a small imaginary part to the denominatior when
!     constructing (E-E_k)^(-1). Not implemented in this version of
!     compak.
      gamma=0.00_idp
 
!     ibctyp determines the boundary conditons
      ibctyp=1         ! = -1 for scattering       (I-iK)
                       ! =  1 for photoionisation  (I+iK)

!     Determines whether the full or the open-open K-matrix is used.
!     Kept in for testing purposes.
      isfullk=1        ! = 0 use open-open K-matrix
                       ! = 1 use full K-matrix (open--open + open--closed)

!     Sets which type of averageing (or none) to be used

      iaverage_type=0  ! =0 no averaging
                       ! =1 T-matrix averaging
                       ! =2 Complex R-matrix averaging (not fully implemented)

      bsto=0

      if (isfullk .eq. 1) then
         noc=nchan
      else
         noc=nopen
      end if

      scattering_energy=ETARG(1)+(EN/2._idp)

      allocate(complex_rmat(nchan,nchan),complex_akmat(nchan,nchan),complex_fkmat(nchan,nchan))
      complex_rmat=0._idp;complex_akmat=0._idp;complex_fkmat=0._idp

      select case(iaverage_type)
      case(0)
!        No averaging
         complex_rmat=rmat
         complex_akmat=akmat
         complex_fkmat=fkmat
      case(1)
!        Read in averaged K-matrix produced by tkmat_average
        
         complex_rmat=rmat
         complex_akmat=akmat
         complex_fkmat=fkmat
!          complex_fkmat=transpose(fkmat)

         luavkmat=190

         call read_averaged_kmatrix_header(luavkmat,nchan_k,no_energy_points)
         allocate(kmatrix(nchan,nchan,no_energy_points))
         if(nchan_k .ne. nchan) stop ' ERROR: No. of channels differs in averaged K-matrix'
         call read_averaged_kmatrix_body(luavkmat,nchan_k,no_energy_points,kmatrix)
         ien=nrk+1
         complex_fkmat(1:nopen,1:nopen)=kmatrix(1:nopen,1:nopen,ien)
         deallocate(kmatrix)
      case(2)
!        Construct complex R-matrix     
         call construct_complex_rmatrix(nchan,nocsf,wamp,eig,scattering_energy,ibctyp,gamma,complex_rmat)
      
         call kmat_smooth(nchan,bsto,nopen,fx,fxp,complex_rmat,complex_akmat,complex_fkmat)

      end select

!  if (ien.eq.20) write(6,*) fkmat(1:nopen,1:nopen)

! !     First we enforce K-matrix symmetry (this is enforced later in the
! !     R-matrix codes as the matrix is written to file in triangular
! !     form)
!       do j=1,nopen
!          do i=1,j
!             fkmat(j,i)=fkmat(i,j)  
!          end do
!       end do

!       fkmat(1:nopen,1:nopen)=akmat(1:nopen,1:nopen)

!     Constructing the photoionisation/scattering (i.e. incoming/outgoing)
!     wave) boundary conditions

      allocate(boundary_conditions(nopen,nopen))
      boundary_conditions=0._idp

!write(699,*) ien
call cpu_time(start)      
      call create_boundary_conditions(ibctyp,nopen,nchan,complex_fkmat,boundary_conditions)
call cpu_time(finish)
!write(699,*) 'create_boundary_conditions'    
!write(699, '(f20.15)'),finish-start

!     Applying the boundary conditions to the solutions of the reduced
!     radial equations.
      allocate(fx_plusminus(noc, nopen),fxp_plusminus(noc, nopen))
      fx_plusminus=0._idp;fxp_plusminus=0._idp

call cpu_time(start)  
      call apply_boundary_conditions(nopen,noc,nchan,en,echl,isfullk,complex_fkmat,boundary_conditions,fx,fxp,fx_plusminus,&
                                     fxp_plusminus)
!       call apply_boundary_conditions_no_inversion(ibctyp,nopen,noc,nchan,en,echl,isfullk,complex_fkmat,fx,fxp,fx_plusminus,fxp_plusminus)
call cpu_time(finish)
!write(699,*) 'apply_boundary_conditions'          
!write(699, '(f20.15)'),finish-start

!     Calculate wavefunction coefficients.
!     ------------------------------------------------------------------
      if (present(akr) .and. present(aki)) then  
         akr=0.0_idp; aki=0.0_idp

call cpu_time(start)  
         select case(iequation)
         case(1)

            call calc_wavefunction_coefs_eq1(ibctyp,gamma,nopen,noc,nchan,nocsf,ntarg,en,etarg,&
     &                                       eig,wamp,fx_plusminus,fxp_plusminus,akr,aki)
         case(2)

            call calc_wavefunction_coefs_eq2(ibctyp,gamma,nopen,noc,nchan,nocsf,ntarg,en,etarg,&
     &                                  eig,wamp,rmat,fx_plusminus,fxp_plusminus,akr,aki)
         case default
            stop ' ERROR: allowed values of iequation are 1 and 2'
         end select
call cpu_time(finish)
write(699,*) 'calc_wavefunction_coefs'        
write(699, '(f20.15)') finish-start
      end if

!     Calculate partial wave dipoles
!     ------------------------------
      if (present(inner_dipoles) .and. present(re_pw_dipoles) .and. present(im_pw_dipoles)) then  
         allocate( pw_dipoles(size(re_pw_dipoles,1),size(re_pw_dipoles,2),size(re_pw_dipoles,3)) )
         pw_dipoles=0
call cpu_time(start)          
         select case(iequation)
         case(1)
     
            call calc_partial_wave_dipoles_eq1( ibctyp, gamma, nopen, noc, nchan, nocsf,    & 
     &                                          ntarg, en, etarg, eig, inner_dipoles, wamp, &
     &                                          fx_plusminus, fxp_plusminus, pw_dipoles)    
     
         case(2)

             stop ' ERROR: equation 2 not yet implemented for partial wave dipoles'
         case default
            stop ' ERROR: allowed values of iequation are 1 and 2'
         end select    

call cpu_time(finish)
!write(700,*) 'calc_partial_wave_dipoles'        
!write(700, '(f20.15)'),finish-start
            
         re_pw_dipoles=real(pw_dipoles)
         im_pw_dipoles=aimag(pw_dipoles)
         
         deallocate(pw_dipoles)
         
      end if
      
      deallocate(boundary_conditions,fx_plusminus,fxp_plusminus)

      end subroutine compak

      end module compak_procs
