! Copyright 2019
!
! Zdenek Masin with contributions from others (see the UK-AMOR website)                               
!
! This file is part of GBTOlib.
!
!     GBTOlib is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     GBTOlib is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  GBTOlib (in trunk/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
module basis_data_generic_gbl
   use precisn_gbl
   use const_gbl, only: line_len, no_header, level2, level3
   use symmetry_gbl
   use integral_storage_gbl
   use bspline_grid_gbl
   use mpi_gbl

   private

   public shell_data_obj, CGTO_shell_data_obj, BTO_shell_data_obj, orbital_data_obj, basis_data_generic_obj

   type, abstract :: shell_data_obj
      !> Normalization factor for the shell (this is typically independent of the m angular number).
      real(kind=cfp) :: norm = 0.0_cfp
      !> Total number of functions generated by this shell.
      integer :: number_of_functions = 0
   contains
      !> Returns 0 if the type-bound parameters are consistent. Returns a non-zero value if an error in the parameters has been found.
      procedure(check_intf), deferred :: check
      !> Calculates the value of norm.
      procedure(normalize_intf), deferred :: normalize
      !> Read-in the data for the basis function from a file. The file is assumed to be open for stream access. The input data are unit number and position in the file where the data start.
      procedure(read_intf), deferred :: read
      !> Writes the data for the basis function into a file. The file is assumed to be open for stream access. The input data are unit number and position in the file where the data should be written.
      procedure(write_intf), deferred :: write
      !> Prints the shell data to the unit for standard output.
      procedure(print_intf), deferred :: print
      !> Evaluates the shell of functions at an arbitrary point in space.
      procedure(eval_intf), deferred :: eval
      !> Returns unique name characterizing the type of the shell of functions.
      procedure(name_intf), deferred :: name
   end type shell_data_obj
 
   abstract interface
      function check_intf(this)
         import :: shell_data_obj
         class(shell_data_obj) :: this
         integer :: check_intf
      end function check_intf
   end interface
 
   abstract interface
      subroutine normalize_intf(this)
         import :: shell_data_obj
         class(shell_data_obj) :: this
      end subroutine normalize_intf
   end interface
 
   abstract interface
      subroutine read_intf(this,lunit,posit,pos_after_rw,version)
         import :: shell_data_obj, line_len
         class(shell_data_obj) :: this
         integer, intent(in) :: lunit, posit
         integer, intent(out) :: pos_after_rw
         character(len=line_len), intent(in) :: version
      end subroutine read_intf
   end interface

   abstract interface
      subroutine write_intf(this,lunit,posit,pos_after_rw)
         import :: shell_data_obj
         class(shell_data_obj) :: this
         integer, intent(in) :: lunit, posit
         integer, intent(out) :: pos_after_rw
      end subroutine write_intf
   end interface

   abstract interface
      subroutine print_intf(this)
         import :: shell_data_obj
         class(shell_data_obj) :: this
      end subroutine print_intf
   end interface
 
   abstract interface
      function eval_intf(this,r,n_points)
         import :: shell_data_obj, cfp
         class(shell_data_obj) :: this
         integer :: n_points
         real(kind=cfp) :: eval_intf(this%number_of_functions,n_points)
         real(kind=cfp), intent(in) :: r(1:3,n_points)
      end function eval_intf
   end interface

   abstract interface
      function name_intf(this)
         import :: shell_data_obj, line_len
         class(shell_data_obj) :: this
         character(len=line_len) :: name_intf
      end function name_intf
   end interface

   abstract interface
      function is_continuum_intf(this)
         import :: shell_data_obj
         class(shell_data_obj) :: this
         logical :: is_continuum_intf
      end function is_continuum_intf
   end interface

   type, extends(shell_data_obj) :: CGTO_shell_data_obj
      !> Angular momentum of the shell.
      integer :: l = -1
      !> Coordinates (in a.u.) of the center of the shell.
      real(kind=cfp) :: center(3) = 0.0_cfp
      !> Number of primitive GTOs (exponents) that build up the CGTO.
      integer :: number_of_primitives = 0
      !> Exponents, contraction coefficients and norms of the primitive GTOs. These arrays must have size number_of_primitives.
      real(kind=cfp), allocatable :: exponents(:), contractions(:), norms(:)
      !> If set to .true. then it will be assumed that basis functions from this shell have a non-zero value at the boundary (R-matrix radius).
      logical :: non_zero_at_boundary = .false.
   contains
      procedure :: check => check_CGTO_shell
      procedure :: normalize => normalize_CGTO_shell
      procedure :: read => read_CGTO_shell
      procedure :: write => write_CGTO_shell
      procedure :: print => print_CGTO_shell
      procedure :: eval => eval_CGTO_shell
      procedure :: name => name_CGTO_shell
      procedure :: make_space => make_space_CGTO_shell
      procedure :: is_continuum => is_continuum_CGTO_shell
      !> Estimates the radius of a range of primitives consituting the CGTO shell.
      procedure :: estimate_shell_radius => estimate_CGTO_radius
      procedure :: overlaps_with_BTO => CGTO_overlaps_with_BTO
      !> Returns the range of radial B-spline indices for which this CGTO shell overlaps with the corresponding B-splines.
      procedure :: get_overlapping_bspline_range
      !> Returns the distance of the shell center from the CMS.
      procedure :: r_CMS
   end type CGTO_shell_data_obj
 
   type, extends(shell_data_obj) :: BTO_shell_data_obj
      !> Angular momentum of the shell.
      integer :: l = -1
      !> Index of the radial B-spline determining the shape of the radial part of this shell.
      integer :: bspline_index = 0
      !> Parameters of the radial B-spline grid.
      type(bspline_grid_obj) :: bspline_grid
      !> If set to .true. then it will be assumed that basis functions from this shell have a non-zero value at the boundary (R-matrix radius).
      !> \warning This must be set to .true. for all continuum BTOs regardless of whether they actually are non-zero at boundary or not. This should be improved.
      logical :: non_zero_at_boundary = .false.
   contains
      procedure :: check => check_BTO_shell
      procedure :: normalize => normalize_BTO_shell
      procedure :: read => read_BTO_shell
      procedure :: write => write_BTO_shell
      procedure :: print => print_BTO_shell
      procedure :: eval => eval_BTO_shell
      procedure :: name => name_BTO_shell
      procedure :: is_continuum => is_continuum_BTO_shell
      procedure :: overlaps_with_BTO => BTO_overlaps_with_BTO
   end type BTO_shell_data_obj

   !> \todo Include the index within symmetry: this can be useful
   type, extends(shell_data_obj) :: orbital_data_obj
      !> Point group symmetry of the orbitals in this set.
      integer :: point_group = 0
      !> Ireducible representation of the orbitals in this set.
      integer :: irr = 0
      !> Energy of each orbital in a.u. (if available then it is set to a non-zero value). This array must be always allocated to size number_of_functions.
      real(kind=cfp), allocatable :: energy(:)
      !> Electron spin in each orbital (if available then spin alpha is indicated by spin=1. Spin beta is indicated by spin=2).
      integer, allocatable :: spin(:)
      !> Occupation number in each orbital (if available then it is set to a non-negative value).
      real(kind=cfp), allocatable :: occup(:)
      !> How many coefficients does each orbital have (this number is always the same for all orbitals of the same symmetry).
      integer :: number_of_coefficients = 0
      !> Orbital coefficients (in columns). The first dimension must have size number_of_coefficients. The second dimension must have size number_of_functions.
      real(kind=cfp), allocatable :: coefficients(:,:)
   contains
      procedure :: check => check_orbital_data
      procedure :: normalize => normalize_orbital_data
      procedure :: read => read_orbital_data
      procedure :: write => write_orbital_data
      procedure :: print => print_orbital_data
      procedure :: eval => eval_orbital_data
      procedure :: name => name_orbital_data
      !> Extends the orbital data with continuum orbitals assuming the atomic orbitals marked as continuum are centered on the CMS.
      procedure :: add_cms_continuum => add_cms_continuum_orbital_data
      procedure :: keep_first_n_orbitals
   end type orbital_data_obj

   type, abstract :: basis_data_generic_obj
       !> The symmetry specification is fully in control of the user.
       type(symmetry_obj) :: symmetry_data
       !> Total number of shells in the basis.
       integer :: number_of_shells = 0
       !> The number of functions generated by all shells in the basis.
       integer :: number_of_functions = 0
   contains
       !> Allocates space for a given number and type of shells of basis functions.
       procedure(init_bdg_intf), deferred :: init
       !> Finalizes the basis set.
       procedure(final_bdg_intf), deferred :: final
       !> Adds data for one shell into the basis set.
       procedure(add_shell_bdg_intf), deferred :: add_shell
       !> Calculates and stores 1-electron integrals for all pairs of shells in the basis.
       procedure(one_electron_integrals_bdg_intf), deferred :: one_electron_integrals
       !> Calculates and stores 2-electron integrals for all quartets of shells in the basis with possible exclusion
       !> of certain classes of integrals as specified on input in integral_options.
       procedure(two_electron_integrals_bdg_intf), deferred :: two_electron_integrals
       !> Calculates indices for 1-electron or 2-electron integrals given a list of pairs or quartets of basis functions and
       !> specifying the type of the integral to index. The two_p_continuum input variable is used only for AO integrals.
       procedure(integral_index_bdg_intf), deferred :: integral_index
       !> Writes the basis set data into the given file.
       procedure, non_overridable :: write => write_bdg
       !> Reads the basis set data from the given file.
       procedure, non_overridable :: read => read_bdg
       !> Prints the basis set data to stdout.
       procedure(print_bdg), deferred :: print
       !> Returns the character string identifying the basis set. This is used for input/output involving disk.
       procedure(get_basis_name_bdg_intf), deferred :: get_basis_name
       !> Returns the character string identifying the i-th shell in the basis set. This is used for input/output involving disk.
       procedure(get_shell_name_bdg_intf), deferred :: get_shell_name
       !> Returns the shell data for the i-th shell in the basis set. This is used for input/output involving disk.
       procedure(get_shell_data_bdg_intf), deferred :: get_shell_data
       !> Returns .true. if the basis set has been initialized.
       procedure(is_initialized_bdg_intf), deferred :: is_initialized
       !> For a given IRR it returns a list of logical values of size equal to the number of functions with that IRR in the basis.
       !> i-th element of the output array is set to .true. if the i-th function in the basis is a continuum function.
       procedure(get_continuum_flags_bdg_intf), deferred :: get_continuum_flags
       !> Calculates amplitudes for all functions in the basis up to the channel with the given maximum l-value of and radial distance from CMS.
       procedure(amplitudes_intf), deferred :: calculate_amplitudes
   end type basis_data_generic_obj
 
   abstract interface
      function init_bdg_intf(this,n,geometry)
         import :: basis_data_generic_obj, geometry_obj
         class(basis_data_generic_obj) :: this
         integer, intent(in) :: n
         class(geometry_obj), intent(in) :: geometry
         integer :: init_bdg_intf
      end function init_bdg_intf
   end interface
 
   abstract interface
      function final_bdg_intf(this)
         import :: basis_data_generic_obj
         class(basis_data_generic_obj) :: this
         integer :: final_bdg_intf
      end function final_bdg_intf
   end interface
 
   abstract interface
      subroutine add_shell_bdg_intf(this,shell_data)
         import :: basis_data_generic_obj, shell_data_obj
         class(basis_data_generic_obj) :: this
         class(shell_data_obj), intent(inout) :: shell_data
      end subroutine add_shell_bdg_intf
   end interface
 
   abstract interface
      subroutine one_electron_integrals_bdg_intf(this,integral_storage,integral_options)
         import :: basis_data_generic_obj, integral_options_obj,integral_storage_obj
         class(basis_data_generic_obj) :: this
         class(integral_options_obj), intent(in) :: integral_options
         class(integral_storage_obj), intent(inout) :: integral_storage
      end subroutine one_electron_integrals_bdg_intf
   end interface

   abstract interface
      subroutine two_electron_integrals_bdg_intf(this,integral_storage,integral_options)
         import :: basis_data_generic_obj, integral_options_obj,integral_storage_obj
         class(basis_data_generic_obj) :: this
         class(integral_options_obj), intent(in) :: integral_options
         class(integral_storage_obj), intent(inout) :: integral_storage
      end subroutine two_electron_integrals_bdg_intf
   end interface

   abstract interface
      function integral_index_bdg_intf(this,integral_type,bf_indices,two_p_continuum)
         import :: basis_data_generic_obj
         class(basis_data_generic_obj) :: this
         character(len=*), intent(in) :: integral_type
         integer, intent(in) :: bf_indices(:,:)
         logical, intent(in) :: two_p_continuum
         integer :: integral_index_bdg_intf(size(bf_indices,2))
      end function integral_index_bdg_intf
   end interface

   abstract interface
      function get_basis_name_bdg_intf(this)
         import :: basis_data_generic_obj, line_len
         class(basis_data_generic_obj) :: this
         character(len=line_len) :: get_basis_name_bdg_intf
      end function get_basis_name_bdg_intf
   end interface

   abstract interface
      subroutine print_bdg(this)
         import :: basis_data_generic_obj
         class(basis_data_generic_obj) :: this
      end subroutine print_bdg
   end interface

   abstract interface
      function get_shell_name_bdg_intf(this,i)
         import :: basis_data_generic_obj, line_len
         class(basis_data_generic_obj) :: this
         integer, intent(in) :: i
         character(len=line_len) :: get_shell_name_bdg_intf
      end function get_shell_name_bdg_intf
   end interface

   abstract interface
      subroutine get_shell_data_bdg_intf(this,i,shell_data)
         import :: basis_data_generic_obj, shell_data_obj
         class(basis_data_generic_obj) :: this
         integer, intent(in) :: i
         class(shell_data_obj), intent(out) :: shell_data
      end subroutine get_shell_data_bdg_intf
   end interface

   abstract interface
      function is_initialized_bdg_intf(this)
         import :: basis_data_generic_obj
         class(basis_data_generic_obj) :: this
         logical :: is_initialized_bdg_intf
      end function is_initialized_bdg_intf
   end interface

   abstract interface
      subroutine get_continuum_flags_bdg_intf(this,irr,list)
         import :: basis_data_generic_obj
         class(basis_data_generic_obj) :: this
         integer, intent(in) :: irr
         logical, allocatable :: list(:)
      end subroutine get_continuum_flags_bdg_intf
   end interface

   abstract interface
      subroutine amplitudes_intf(this,a,normalize_to_a,amplitudes,continuum_channels)
         import :: basis_data_generic_obj, cfp
         class(basis_data_generic_obj) :: this
         real(kind=cfp), intent(in) :: a
         logical, intent(in) :: normalize_to_a
         integer, allocatable :: continuum_channels(:,:)
         real(kind=cfp), allocatable :: amplitudes(:,:)
      end subroutine amplitudes_intf
   end interface

contains
 
   function check_CGTO_shell(this)
      use gto_routines_gbl, only: check_cgto_data
      implicit none
      class(CGTO_shell_data_obj) :: this
      integer :: check_CGTO_shell
 
         check_CGTO_shell = check_cgto_data(this % number_of_primitives, &
                                            this % l, &
                                            this % exponents, &
                                            this % contractions, &
                                            this % norms, &
                                            this % number_of_functions)
 
   end function check_CGTO_shell
 
   subroutine normalize_CGTO_shell(this)
      use gto_routines_gbl, only: normalize_cgto
      implicit none
      class(CGTO_shell_data_obj) :: this
 
      integer :: err
 
         err = this%check()
         if (err /= 0) then
            call xermsg ('CGTO_shell_data_obj', 'normalize', &
                         'Check has failed. See CGTO_shell_data_obj%check for details.', err, 1)
         end if
 
         call normalize_cgto(this%number_of_primitives,this%l,this%exponents,this%contractions,this%norms,this%norm)
 
   end subroutine normalize_CGTO_shell
 
   subroutine read_CGTO_shell(this,lunit,posit,pos_after_rw,version)
      use gto_routines_gbl, only: read_cgto
      implicit none
      class(CGTO_shell_data_obj) :: this
      integer, intent(in) :: lunit, posit
      integer, intent(out) :: pos_after_rw
      character(len=line_len), intent(in) :: version
 
      integer :: err
 
         call read_cgto (this % number_of_primitives, this % l, this % exponents, this % contractions, this % norms, &
                         this % norm, this % center, this % non_zero_at_boundary, this % number_of_functions, &
                         lunit, posit, pos_after_rw)
  
         err = this%check()
         if (err /= 0) then
            call xermsg ('CGTO_shell_data_obj', 'read', &
                         'Check has failed. See CGTO_shell_data_obj%check for details.', err, 1)
         end if
 
   end subroutine read_CGTO_shell
 
   subroutine write_CGTO_shell(this,lunit,posit,pos_after_rw)
      use gto_routines_gbl, only: write_cgto
      implicit none
      class(CGTO_shell_data_obj) :: this
      integer, intent(in) :: lunit, posit
      integer, intent(out) :: pos_after_rw
 
      integer :: err
 
         err = this%check()
         if (err /= 0) then
            call xermsg ('CGTO_shell_data_obj', 'write', 'Check has failed. See CGTO_shell_data_obj%check for details.', err, 1)
         end if

         call write_cgto (this % number_of_primitives, this % l, this % exponents, this % contractions, this % norms, &
                          this % norm, this % center, this % non_zero_at_boundary, this % number_of_functions, &
                          lunit, posit, pos_after_rw)
 
   end subroutine write_CGTO_shell
 
   subroutine print_CGTO_shell(this)
      use gto_routines_gbl, only: print_cgto_data
      implicit none
      class(CGTO_shell_data_obj) :: this
 
      integer :: err
 
         err = this%check()
         if (err /= 0) then
            call xermsg ('CGTO_shell_data_obj', 'print', 'Check has failed. See CGTO_shell_data_obj%check for details.', err, 1)
         end if
 
         call print_cgto_data (this % number_of_primitives, this % l, this % exponents, this % contractions, this % norms, &
                               this % norm, this % center, this % non_zero_at_boundary)
 
   end subroutine print_CGTO_shell
 
   function eval_CGTO_shell(this,r,n_points)
      use gto_routines_gbl, only: eval_cgto
      implicit none
      class(CGTO_shell_data_obj) :: this
      integer :: n_points
      real(kind=cfp) :: eval_CGTO_shell(this%number_of_functions,n_points)
      real(kind=cfp), intent(in) :: r(1:3,n_points)
 
      integer :: err
 
         err = this%check()
         if (err /= 0) then
            call xermsg ('CGTO_shell_data_obj', 'eval', 'Check has failed. See CGTO_shell_data_obj%check for details.', err, 1)
         end if
 
         call eval_cgto (r, n_points, this % number_of_primitives, this % l, this % exponents, this % contractions, &
                         this % norms, this % norm, this % center, eval_CGTO_shell)
 
   end function eval_CGTO_shell

   function name_CGTO_shell(this)
      implicit none
      class(CGTO_shell_data_obj) :: this
      character(len=line_len) :: name_CGTO_shell

         name_CGTO_shell = "CGTO shell"

   end function name_CGTO_shell

   subroutine make_space_CGTO_shell(this,number_of_primitives)
      implicit none
      class(CGTO_shell_data_obj) :: this
      integer, intent(in) :: number_of_primitives

      integer :: err

         if (number_of_primitives <= 0) then
            call xermsg ('CGTO_shell_data_obj', 'make_space_CGTO_shell', &
                         'On input the value of number_of_primitives was .le. 0.', 1, 1)
         end if

         if (allocated(this%exponents)) deallocate(this%exponents)
         if (allocated(this%contractions)) deallocate(this%contractions)
         if (allocated(this%norms)) deallocate(this%norms)

         allocate(this % exponents(number_of_primitives), &
                  this % contractions(number_of_primitives), &
                  this % norms(number_of_primitives), stat = err)
         if (err /= 0) call xermsg('CGTO_shell_data_obj','make_space_CGTO_shell','Memory allocation failed.',err,1)

         this%number_of_primitives = number_of_primitives

   end subroutine make_space_CGTO_shell

   function is_continuum_CGTO_shell(this)
      implicit none
      class(CGTO_shell_data_obj) :: this
      logical :: is_continuum_CGTO_shell

         if (this%non_zero_at_boundary) then
            is_continuum_CGTO_shell = .true.
            if (this%center(1) .ne. 0.0_cfp .or. this%center(2) .ne. 0.0_cfp .or. this%center(3) .ne. 0.0_cfp) then
               call xermsg ('CGTO_shell_data_obj', 'is_continuum_CGTO_shell', &
                            'CGTO shell is non-zero at boundary but is not centered on CMS', 1, 1)
            endif
         else
            is_continuum_CGTO_shell = .false.
         endif

   end function is_continuum_CGTO_shell

   subroutine estimate_CGTO_radius(this,c_first,c_last,threshold,radius,cms_sphere_rmin,cms_sphere_rmax)
      use phys_const_gbl, only: fourpi
      implicit none
      class(CGTO_shell_data_obj) :: this
      real(kind=cfp), intent(in) :: threshold
      integer, intent(in) :: c_first,c_last
      real(kind=cfp), intent(out) ::  radius,cms_sphere_rmin,cms_sphere_rmax

      real(kind=cfp) :: RA, dm, d, val, R, fac, r_sq, val_left, val_right, left, right
      integer :: i, err

         err = this%check()
         if (err /= 0) then
            call xermsg ('CGTO_shell_data_obj', 'estimate_CGTO_radius', 'CGTO_shell_data_obj%check failed with an error.', err, 1)
         end if

         if (c_first <= 0 .or. c_last <= 0) then
            call xermsg ('CGTO_shell_data_obj', 'estimate_CGTO_radius', 'c_first and/or c_last .le. 0.', 1, 1)
         end if
         if (c_first > this % number_of_primitives .or. c_last > this % number_of_primitives) then
            call xermsg ('CGTO_shell_data_obj', 'estimate_CGTO_radius', 'c_first and/or c_last > this%number_of_primitives.', 2, 1)
         end if

         RA = sqrt(dot_product(this%center,this%center))      !radial distance from CMS to the GTO center
         dm = sqrt(this%l/(2.0_cfp*minval(this%exponents(c_first:c_last)))) !radial distance from the GTO center for which the radial part of the GTO has maximum

         !Find the radial interval where the radial value lies for which the CGTO is smaller than threshold.
         left = dm
         right = dm + 1.0_cfp
         do
            val = rad_cgto(right,this,c_first,c_last)
            if (val > threshold) then
               right = right + 1.0_cfp
            else
               exit
            endif
         enddo

         !Use bisection on the interval r = [left;right] to determine the shell radius accurately.
         val_left = rad_cgto(left,this,c_first,c_last)-threshold
         val_right = rad_cgto(right,this,c_first,c_last)-threshold
         do
            radius = (right+left)/2.0_cfp
            val = rad_cgto(radius,this,c_first,c_last)-threshold

            !We don't need the result super accurately so relative precision 1e-5 should be enough for most applications.
            !todo put the threshold into const module.
            if (abs(val/threshold) .le. 1.0e-5_cfp) exit

            if (val*val_left > 0.0_cfp) then !sign is the same in [left;middle] so bisect the interval [middle,right]
               left = radius
               val_left = val
            else !sign changes in [left;middle] so bisect the interval [left,middle]
               right = radius
               val_right = val
            endif
         enddo

         !Spherical shell limits outside of which this CGTO shell can be neglected.
         cms_sphere_rmin = max(RA-radius,0.0_cfp)
         cms_sphere_rmax = RA+radius

   contains
      
      !todo move this function to gto_routines_mod
      function rad_cgto(radius,cgto_shell,c_first,c_last)
         implicit none
         class(CGTO_shell_data_obj), intent(in) :: cgto_shell
         real(kind=cfp) :: rad_cgto
         real(kind=cfp), intent(in) :: radius
         integer, intent(in) :: c_first,c_last

         integer :: i
         real(kind=cfp) :: fac, r_sq

            fac = cgto_shell%norm*sqrt(fourpi/(2*cgto_shell%l+1.0_cfp))*radius**cgto_shell%l
            r_sq = radius*radius
            rad_cgto = 0.0_cfp
            !c_first and c_last must be in the range [1,cgto_shell%number_of_primitives]
            if (c_first .eq. c_last) then
               rad_cgto = abs(cgto_shell%contractions(c_first))*cgto_shell%norms(c_first)*exp(-cgto_shell%exponents(c_first)*r_sq)
            else
               do i=min(c_first,cgto_shell%number_of_primitives),max(c_last,cgto_shell%number_of_primitives)
                  rad_cgto = rad_cgto + cgto_shell%contractions(i)*cgto_shell%norms(i)*exp(-cgto_shell%exponents(i)*r_sq)
               enddo
            endif
            rad_cgto = abs(rad_cgto)*fac

      end function rad_cgto

   end subroutine estimate_CGTO_radius

   function CGTO_overlaps_with_BTO(this,BTO_shell_data)
      implicit none
      class(CGTO_shell_data_obj),intent(in) :: this
      class(BTO_shell_data_obj),intent(in) :: BTO_shell_data
      logical :: CGTO_overlaps_with_BTO

         call xermsg ('CGTO_shell_data_obj', 'CGTO_overlaps_with_BTO', 'Not implemented yet', 1, 1)

         !If the function is non_zero_at boundary then we assume it spans the
         !whole radial range on which the B-splines are defined. Also we assume
         !that the CGTO always overlaps with a BTO that is inside the inner sphere with radius r=C.
         !todo replace the test on non_zero_at_boundary with a test on CGTO radius and the B-spline domain.
         !CGTO_overlaps_with_BTO = (this%non_zero_at_boundary .or. &
         !                         (BTO_shell_data%bspline_index .le. BTO_shell_data%bspline_grid%ind_last_before_C))

   end function CGTO_overlaps_with_BTO

   subroutine get_overlapping_bspline_range(this,bspline_grid,first_bspline,last_bspline)
      implicit none
      class(CGTO_shell_data_obj),intent(in) :: this
      class(bspline_grid_obj), intent(in) :: bspline_grid
      integer, intent(out) :: first_bspline,last_bspline

         first_bspline = 0
         last_bspline = 0

         call xermsg ('CGTO_shell_data_obj', 'get_overlapping_bspline_range', 'Not implemented yet', 1, 1)

         !todo replace the test on non_zero_at_boundary with a test on CGTO radius and the B-spline domain.
         !if (this%non_zero_at_boundary) then
         !   first_bspline = 1
         !   last_bspline = bspline_grid%n
         !else
         !   first_bspline = 1
         !   last_bspline = bspline_grid%ind_last_before_C
         !endif

   end subroutine get_overlapping_bspline_range

   function r_CMS(this)
      implicit none
      class(CGTO_shell_data_obj),intent(in) :: this
      real(kind=cfp) :: r_CMS

         r_CMS = sqrt(dot_product(this%center,this%center))

   end function

   function check_BTO_shell(this)
      implicit none
      class(BTO_shell_data_obj) :: this
      integer :: check_BTO_shell

         !This error function returns positive values
         check_BTO_shell = this%bspline_grid%check()
         
         if (this%bspline_index .le. 0 .or. this%bspline_index > this%bspline_grid%n) check_BTO_shell = -1
         if (this%l < 0) check_BTO_shell = -2

         if (this%number_of_functions .ne. 2*this%l+1) check_BTO_shell = -3
 
   end function check_BTO_shell
 
   subroutine normalize_BTO_shell(this)
      implicit none
      class(BTO_shell_data_obj) :: this

      integer :: err

         err = this%check()
         if (err /= 0) then
            call xermsg ('BTO_shell_data_obj', 'read', 'Check has failed. See BTO_shell_data_obj%check for details.', err, 1)
         end if

         this%norm = this%bspline_grid%normalize(this%bspline_index)
 
   end subroutine normalize_BTO_shell
 
   subroutine read_BTO_shell(this,lunit,posit,pos_after_rw,version)
      use bspline_grid_gbl, only: read_BTO
      implicit none
      class(BTO_shell_data_obj) :: this
      integer, intent(in) :: lunit, posit
      integer, intent(out) :: pos_after_rw
      character(len=line_len), intent(in) :: version

      integer :: err

         call read_BTO (this % bspline_grid, this % l, this % bspline_index, this % number_of_functions, this % norm, &
                        this % non_zero_at_boundary, lunit, posit, pos_after_rw, version)

         err = this%check()
         if (err /= 0) then
            call xermsg ('BTO_shell_data_obj', 'read', 'Check has failed. See BTO_shell_data_obj%check for details.', err, 1)
         end if
 
   end subroutine read_BTO_shell
 
   subroutine write_BTO_shell(this,lunit,posit,pos_after_rw)
      use bspline_grid_gbl, only: read_BTO
      implicit none
      class(BTO_shell_data_obj) :: this
      integer, intent(in) :: lunit, posit
      integer, intent(out) :: pos_after_rw

      integer :: err

         err = this%check()
         if (err /= 0) then
            call xermsg ('BTO_shell_data_obj', 'write', 'Check has failed. See BTO_shell_data_obj%check for details.', err, 1)
         end if

         call write_BTO (this % bspline_grid, this % l, this % bspline_index, this % number_of_functions, this % norm, &
                         this % non_zero_at_boundary, lunit, posit, pos_after_rw)
 
   end subroutine write_BTO_shell
 
   subroutine print_BTO_shell(this)
      use bspline_grid_gbl, only: print_BTO
      implicit none
      class(BTO_shell_data_obj) :: this

      integer :: err

         err = this%check()
         if (err /= 0) then
            call xermsg ('BTO_shell_data_obj', 'print', 'Check has failed. See BTO_shell_data_obj%check for details.', err, 1)
         end if

         call print_BTO(this%bspline_grid,this%l,this%bspline_index,this%number_of_functions,this%norm,this%non_zero_at_boundary)
 
   end subroutine print_BTO_shell
 
   function eval_BTO_shell(this,r,n_points)
      use special_functions_gbl, only: cfp_resh
      use bspline_base_gbl, only: bvalu
      implicit none
      class(BTO_shell_data_obj) :: this
      integer :: n_points
      real(kind=cfp) :: eval_BTO_shell(this%number_of_functions,n_points)
      real(kind=cfp), intent(in) :: r(1:3,n_points)

      real(kind=cfp) :: point(3), tmp, bspline_val, tmp_old
      real(kind=cfp) :: r1, r2, RH(-this%l:this%l,0:this%l+1)
      integer :: i, m, err

         call this%bspline_grid%bspline_range(this%bspline_index,r1,r2)
       
         tmp_old = -1.0_cfp 
         do i=1,n_points
            tmp = sqrt(dot_product(r(1:3,i),r(1:3,i)))
            if (tmp .ge. r1 .and. tmp .le. r2) then !If this radial point falls within the B-spline range then evaluate B(r)/r at that point.

               if (tmp .ne. tmp_old) then !evaluate the B-spline only if the radial distance has changed.
                  this%bspline_grid%bcoef = 0.0_cfp
                  this%bspline_grid%bcoef(this%bspline_index) = 1.0_cfp
                  bspline_val = this % norm * bvalu(this % bspline_grid % knots, &
                                                    this % bspline_grid % bcoef, &
                                                    this % bspline_grid % n,&
                                                    this % bspline_grid % order, 0, tmp, &
                                                    this % bspline_grid % inbv, &
                                                    this % bspline_grid % work)
                  if (tmp /= 0) then
                     bspline_val = bspline_val / tmp
                  else if (bspline_val /= 0) then
                     call xermsg('BTO_shell_data_obj', 'eval_BTO_shell', 'Attempt to evaluate B/r at r = 0 for B /= 0.', 1, 1)
                  end if
                  tmp_old = tmp
               endif

               !Evaluate the real spherical harmonic for the given point.
               call cfp_resh(RH,r(1,i),r(2,i),r(3,i),this%l)
               do m=-this%l,this%l
                  eval_BTO_shell(m+this%l+1,i) = bspline_val*RH(m,this%l)
               enddo
            else
               eval_BTO_shell(1:this%number_of_functions,i) = 0.0_cfp
            endif
         enddo

         !call xermsg('BTO_shell_data_obj','eval_BTO_shell','Not implemented yet.',1,1)
 
   end function eval_BTO_shell

   function name_BTO_shell(this)
      implicit none
      class(BTO_shell_data_obj) :: this
      character(len=line_len) :: name_BTO_shell

         name_BTO_shell = "BTO shell"

   end function name_BTO_shell

   function is_continuum_BTO_shell(this)
      implicit none
      class(BTO_shell_data_obj) :: this
      logical :: is_continuum_BTO_shell

         is_continuum_BTO_shell = .true.

   end function is_continuum_BTO_shell

   function BTO_overlaps_with_BTO(this,BTO_shell_data)
      implicit none
      class(BTO_shell_data_obj),intent(in) :: this
      class(BTO_shell_data_obj),intent(in) :: BTO_shell_data
      logical :: BTO_overlaps_with_BTO

      real(kind=cfp) :: r1my,r2my, r1,r2, s,e

         call this%bspline_grid%bspline_range(this%bspline_index,r1my,r2my)
         call BTO_shell_data%bspline_grid%bspline_range(BTO_shell_data%bspline_index,r1,r2)

         s = max(r1my,r1)
         e = min(r2my,r2)
         BTO_overlaps_with_BTO = (s < e)

   end function BTO_overlaps_with_BTO

   function check_orbital_data(this)
      use orbital_routines_gbl, only: check_orbital
      implicit none
      class(orbital_data_obj) :: this
      integer :: check_orbital_data
 
         check_orbital_data = check_orbital(this % number_of_functions, &
                                            this % number_of_coefficients, &
                                            this % coefficients, &
                                            this % spin, &
                                            this % energy, &
                                            this % occup, &
                                            this % irr, &
                                            this % point_group)

   end function check_orbital_data
 
   subroutine normalize_orbital_data(this)
      implicit none
      class(orbital_data_obj) :: this

         this%norm = 1.0_cfp
 
   end subroutine normalize_orbital_data
 
   subroutine read_orbital_data(this,lunit,posit,pos_after_rw,version)
      use orbital_routines_gbl, only: read_orbital
      implicit none
      class(orbital_data_obj) :: this
      integer, intent(in) :: lunit, posit
      integer, intent(out) :: pos_after_rw
      character(len=line_len), intent(in) :: version

      integer :: err

         call read_orbital (this % number_of_functions, &
                            this % number_of_coefficients, &
                            this % coefficients, &
                            this % spin, &
                            this % energy, &
                            this % occup, &
                            this % irr, &
                            this % point_group, &
                            this % norm, &
                            lunit, posit, pos_after_rw)

         err = this%check()
         if (err /= 0) call xermsg('orbital_data_obj','read','Check has failed. See orbital_data_obj%check for details.',err,1)
 
   end subroutine read_orbital_data
 
   subroutine write_orbital_data(this,lunit,posit,pos_after_rw)
      use orbital_routines_gbl, only: write_orbital
      implicit none
      class(orbital_data_obj) :: this
      integer, intent(in) :: lunit, posit
      integer, intent(out) :: pos_after_rw
      integer :: err

         err = this%check()
         if (err /= 0) call xermsg('orbital_data_obj','write','Check has failed. See orbital_data_obj%check for details.',err,1)

         call write_orbital (this % number_of_functions, &
                             this % number_of_coefficients, &
                             this % coefficients, &
                             this % spin, &
                             this % energy, &
                             this % occup, &
                             this % irr, &
                             this % point_group, &
                             this % norm, &
                             lunit, posit, pos_after_rw)
 
   end subroutine write_orbital_data
 
   subroutine print_orbital_data(this)
      implicit none
      class(orbital_data_obj) :: this
 
   end subroutine print_orbital_data
 
   function eval_orbital_data(this,r,n_points)
      implicit none
      class(orbital_data_obj) :: this
      integer :: n_points
      real(kind=cfp) :: eval_orbital_data(this%number_of_functions,n_points)
      real(kind=cfp), intent(in) :: r(1:3,n_points)

      integer :: err

         call xermsg('orbital_data_obj','eval_orbital_data','Not implemented yet.',err,1)
 
   end function eval_orbital_data

   function name_orbital_data(this)
      implicit none
      class(orbital_data_obj) :: this
      character(len=line_len) :: name_orbital_data

         name_orbital_data = "Orbital set"

   end function name_orbital_data

   subroutine add_cms_continuum_orbital_data(this,is_continuum)
      implicit none
      class(orbital_data_obj) :: this
      logical, intent(in) :: is_continuum(:)

      integer :: n, i, j, n_ao, n_total_orbitals, err, n_set
      type(orbital_data_obj) :: temp

         !The check will fail if there are no orbitals in the set so we'll switch it off since this can happen.
         !err = this%check()
         !if (err /= 0) call xermsg('orbital_data_obj','add_cms_continuum_orbital_data','Check has failed. See orbital_data_obj%check for details.',err,1)

         n_ao = size(is_continuum)

         !We need to have information on all AO in the basis.
         if (this % number_of_coefficients > n_ao) then
            call xermsg ('orbital_data_obj', 'add_cms_continuum_orbital_data', &
                         'The input data on AO must span the full basis in which the orbitals are given.', 2, 1)
         end if
  
         !First count the number of continuum AOs in this symmetry
         n = count(is_continuum)

         !Save the current orbital data into a temporary structure
         select type(this)
            type is (orbital_data_obj)
               temp = this
            class default
               !If the orbital_data_obj has been extended then it may include
               !additional data structures that would not get copied accross in the statment above.
               call xermsg ('orbital_data_obj', 'add_cms_continuum_orbital_data', &
                            'This routine cannot handle extended types of the class orbital_data_obj.', 3, 1)
         end select

         if (allocated(this%occup)) deallocate(this%occup)
         if (allocated(this%energy)) deallocate(this%energy)
         if (allocated(this%spin)) deallocate(this%spin)
         if (allocated(this%coefficients)) deallocate(this%coefficients)

         n_total_orbitals = n + this%number_of_functions
         allocate(this % occup(n_total_orbitals), &
                  this % energy(n_total_orbitals), &
                  this % spin(n_total_orbitals), &
                  this % coefficients(n_ao, n_total_orbitals), stat = err)
         if (err /= 0) call xermsg ('orbital_data_obj', 'add_cms_continuum_orbital_data', 'Memory allocation failed.', err, 1)
         this%coefficients = 0.0_cfp; this%occup = 0.0_cfp; this%energy = 0.0_cfp; this%spin = 0

         !Copy the old data into the resized structure
         if (temp%number_of_functions > 0) then
            this%occup(1:temp%number_of_functions) = temp%occup(1:temp%number_of_functions)
            this%energy(1:temp%number_of_functions) = temp%energy(1:temp%number_of_functions)
            this%spin(1:temp%number_of_functions) = temp%spin(1:temp%number_of_functions)
            this % coefficients(1 : temp % number_of_coefficients, 1 : temp % number_of_functions) &
                = temp % coefficients(1 : temp % number_of_coefficients, 1 : temp % number_of_functions)
         endif
         this%number_of_functions = n_total_orbitals
         this%number_of_coefficients = n_ao

         !Construct the initial coefficients for the continuum functions.
         n_set = 0
         j = 0
         do i=temp%number_of_functions+1,this%number_of_functions

            !Find the next AO continuum function and set its coefficient to 1.0_cfp
            do
               j = j + 1
               if (j > n_ao) exit
               if (is_continuum(j)) then
                  this%coefficients(j,i) = 1.0_cfp
                  n_set = n_set + 1
                  exit
               endif
            enddo !j

         enddo !i

         if (n_set /= n) then
            call xermsg ('orbital_data_obj','add_cms_continuum_orbital_data', &
                         'Error setting up initial coefficients of the continuum orbitals.', 4, 1)
         end if
         
   end subroutine add_cms_continuum_orbital_data

   subroutine keep_first_n_orbitals(this,nob,select_orbitals_by)
      use sort_gbl, only: cfp_sort_float_int_1d
      use phys_const_gbl, only: to_ev
      implicit none
      class(orbital_data_obj) :: this
      integer, intent(in) :: nob, select_orbitals_by

      type(orbital_data_obj) :: temp
      integer, allocatable :: order(:)
      real(kind=cfp), allocatable :: energy(:)
      integer :: err, i, cnt

         write(level3,'("--------->","orbital_data_obj:keep_first_n_orbitals")')

         write(level2,'("Symmetry: ",i1)') this%irr

         if (nob < 0) then
            call xermsg ('orbital_data_obj','keep_first_n_orbitals', &
                         'On input number of orbitals to keep was less than 0.', 1, 1)
         end if
         if (nob > this % number_of_functions) then
            print *, nob, this % number_of_functions
            call xermsg ('orbital_data_obj','keep_first_n_orbitals', &
                         'On input the number of orbitals to keep was greater than the total number of orbitals.', 2, 1)
         end if

         if (this%number_of_functions .eq. 0) then
            if (nob > 0) then
                call xermsg ('orbital_data_obj','keep_first_n_orbitals', &
                             'The number of orbitals is zero but on input the number of orbitals &
                             &to keep was greater than zero.', 3, 1)
            end if
            return !there is nothing to be done
         endif

         if (select_orbitals_by <= 0 .or. select_orbitals_by > 3) then
            call xermsg ('orbital_data_obj','keep_first_n_orbitals', &
                         'On input select_orbitals_by was out of range: the allowed values are 1, 2 or 3.', 4, 1)
         end if

         !If we make it here then we know that nob > 0 and this%number_of_functions > 0.

         !Save the current orbital data into a temporary structure
         select type(this)
            type is (orbital_data_obj)
               temp = this
            class default
               !If the orbital_data_obj has been extended then it may include
               !additional data structures that would not get copied accross in the statment above.
               call xermsg ('orbital_data_obj', 'keep_first_n_orbitals', &
                            'This routine cannot handle extended types of the class orbital_data_obj.', 5, 1)
         end select

         this%number_of_functions = nob

         if (allocated(this%occup)) deallocate(this%occup)
         if (allocated(this%energy)) deallocate(this%energy)
         if (allocated(this%spin)) deallocate(this%spin)
         if (allocated(this%coefficients)) deallocate(this%coefficients)

         allocate(this%occup(nob),this%energy(nob),this%spin(nob),this%coefficients(this%number_of_coefficients,nob),stat=err)
         if (err /= 0) call xermsg ('orbital_data_obj', 'keep_first_n_orbitals', 'Memory allocation 1 failed.', err, 1)
         this%coefficients = 0.0_cfp; this%occup = 0.0_cfp; this%energy = 0.0_cfp; this%spin = 0

         if (select_orbitals_by .eq. 1) then
            write(level2,'("Keeping only the first ",i5," orbitals.")') nob
   
            !Copy the old data into the resized structure
            this%occup(1:nob) = temp%occup(1:nob)
            this%energy(1:nob) = temp%energy(1:nob)
            this%spin(1:nob) = temp%spin(1:nob)
            this%coefficients(1:this%number_of_coefficients,1:nob) = temp%coefficients(1:this%number_of_coefficients,1:nob)
         elseif (select_orbitals_by >= 2) then
            allocate(energy(temp%number_of_functions),order(temp%number_of_functions),stat=err)
            if (err /= 0) call xermsg ('orbital_data_obj', 'keep_first_n_orbitals', 'Memory allocation 2 failed.', err, 1)

            if (select_orbitals_by == 2) then
               write(level2,'("Keeping only the first ",i5," lowest-energy orbitals.")') nob
               energy(1:temp%number_of_functions) = temp%energy(1:temp%number_of_functions)
            else
               write(level2,'("Keeping only the first ",i5," highest-occupation orbitals.")') nob
               energy(1:temp%number_of_functions) = -temp%occup(1:temp%number_of_functions)
            end if

            do i=1,temp%number_of_functions
               order(i) = i
            enddo

            call cfp_sort_float_int_1d(temp%number_of_functions,energy,order)

            cnt = 0
            do i=1,nob !temp%number_of_functions
!               if (order(i) .le. nob) then
                  cnt = cnt + 1
                  if (cnt > nob) call xermsg ('orbital_data_obj','keep_first_n_orbitals', 'Error in sort.', 6, 1)

                  !Copy the old data into the resized structure
                  this%occup(i) = temp%occup(order(i))
                  this%energy(i) = temp%energy(order(i))
                  this%spin(i) = temp%spin(order(i))
                  this%coefficients(1:this%number_of_coefficients,i) = temp%coefficients(1:this%number_of_coefficients,order(i))
                  write(level3,'("Orbital ",i5,".",i1," with energy ",e25.15," eV and occupation ",f0.5," has been retained as &
                                &orbital with sequence number: ",i5)') order(i),this%irr,this%energy(i)*to_ev,this%occup(i),i
!               endif
            enddo !i
         endif

         write(level3,'("<---------","orbital_data_obj:keep_first_n_orbitals")')

   end subroutine keep_first_n_orbitals

   subroutine write_bdg(this,path)
      implicit none
      class(basis_data_generic_obj) :: this
      character(len=*), intent(in) :: path

      type(data_file_obj) :: basis_data_file
      type(geometry_obj) :: geometry
      character(len=line_len) :: name
      type(CGTO_shell_data_obj) :: CGTO_shell_data
      type(BTO_shell_data_obj) :: BTO_shell_data
      type(orbital_data_obj) :: orbital_data
      integer :: i, j, lunit, first_record, last_record, pos_after_write

         if (.not. this % is_initialized()) then
            call xermsg ('basis_data_generic_obj', 'write', &
                         'The object has not been initialized or not all atomic shells have been added.', 1, 1)
         end if

         write(level3,'("--------->","basis_data_generic_obj:write")')

         write(level2,'(/,"Writing basis data into the file: ",a)') path

         call basis_data_file%open(path)
         lunit = basis_data_file%get_unit_no()

         !The record on the file is identified directly by the name of the object. Note that this implies that no more than one CGTO basis set can be present on the given file.
         name = this%get_basis_name()
         first_record = basis_data_file%start_record(name)

         write(level2,'(/,"Basis set for which data will be written: ",a)') trim(adjustl(name))
         call mpi_mod_barrier(i)

         !How many shells there are on the file
         if (myrank .eq. master) then
            write(lunit,pos=first_record,err=10) this%number_of_shells
            inquire(lunit,pos=last_record)
         endif

         !master ensures all processes know where the record ends
         call mpi_mod_bcast(last_record,master)

         !construct the geometry data structure and write it into the file
         call this%symmetry_data%get_geometry(geometry)
         call geometry%write(lunit,last_record,pos_after_write)
         last_record = pos_after_write
         write(level3,'(/,"Symmetry data written.")')

         !Write the shell data for each type of function:
         call mpi_mod_barrier(i)
         do i=1,this%number_of_shells

            !Write the string identifier of the shell type to be written out: this is needed to implement shell reading.
            if (myrank .eq. master) then
               name = this%get_shell_name(i)
               write(lunit) name
               inquire(lunit,pos=last_record)
            endif

            !master ensures all processes know the current position in the file
            !and the name of the shell to be written.
            call mpi_mod_bcast(name,master)
            call mpi_mod_bcast(last_record,master)

            !Write the shell data
            if (name .eq. CGTO_shell_data%name()) then
               call this%get_shell_data(i,CGTO_shell_data)
               call CGTO_shell_data%write(lunit,last_record,pos_after_write)
            elseif (name .eq. BTO_shell_data%name()) then
               call this%get_shell_data(i,BTO_shell_data)
               call BTO_shell_data%write(lunit,last_record,pos_after_write)
            elseif (name .eq. orbital_data%name()) then
               call this%get_shell_data(i,orbital_data)
               call orbital_data%write(lunit,last_record,pos_after_write)
            else
               print *,name
               call xermsg('basis_data_generic_obj','write_bdg','Unimplemented shell-type.',3,1)
            endif

            !master ensures all processes know where the record ends
            if (myrank .eq. master) last_record = pos_after_write
            call mpi_mod_bcast(last_record,master)

         enddo !i

         write(level3,'(/,"Data for ",i0," shells of functions written.")') this%number_of_shells

         name = this%get_basis_name()
         call basis_data_file%close_record(name,first_record,last_record)
         call basis_data_file%close

         call mpi_mod_barrier(i)

         write(level3,'("<---------","basis_data_generic_obj:write")')

         return

 10      call xermsg ('basis_data_generic_obj', 'write', 'Error writing the shell data from the file and position given.', 3, 1)

   end subroutine write_bdg
 
   subroutine read_bdg(this,path)
      implicit none
      class(basis_data_generic_obj) :: this
      character(len=*), intent(in) :: path

      type(geometry_obj) :: geometry
      character(len=line_len) :: name
      type(data_file_obj) :: basis_data_file
      type(CGTO_shell_data_obj) :: CGTO_shell_data
      type(BTO_shell_data_obj) :: BTO_shell_data
      type(orbital_data_obj) :: orbital_data
      integer :: err, lunit, first_record, last_record, pos_after_read, i, n

         if (this%is_initialized()) then
            err = this%final()
            if (err /= 0) then
                call xermsg ('basis_data_generic_obj', 'read_bdg', &
                             'Finalization has failed. See basis_data_generic_obj%final for details.', err, 1)
            end if
         endif

         write(level3,'("--------->","basis_data_generic_obj:read")')

         write(level2,'(/,"Reading basis data from the file: ",a)') path

         call basis_data_file%open(path)
         lunit = basis_data_file%get_unit_no()

         name = this%get_basis_name()
         err = basis_data_file%find_header(name,first_record,last_record)
         if (err /= 0) then
            call xermsg ('basis_data_generic_obj', 'read_bdg', &
                         'Error locating the header corresponding to the atomic basis data.', err, 1)
         end if

         !The following may occur if writing of the data record has not been finished or the record is corrupt.
         if (first_record <= 0 .or. last_record <= 0) then
            call xermsg ('basis_data_generic_obj', 'read_bdg', &
                         'The requested data record is missing or not complete.', 2, 1)
         end if

         !how many shells there are on the file
         if (myrank .eq. master) then
            read(lunit,pos=first_record,err=10) n
            inquire(lunit,pos=last_record)
         endif

         !master ensures all processes get the number of shells in the basis and the position in the file
         call mpi_mod_bcast(n,master)
         call mpi_mod_bcast(last_record,master)

         !Read symmetry data
         call geometry%read(lunit,last_record,pos_after_read,basis_data_file%identifier)
         last_record = pos_after_read
         write(level3,'(/,"Symmetry data read-in.")')

         err = this%init(n,geometry)
         if (err /= 0) then
            call xermsg ('basis_data_generic_obj', 'read_bdg', &
                         'Init has failed; see basis_data_generic_obj%init for details.', err, 1)
         end if

         !Read the shell data
         do i=1,n

            !Read the character string identifying the shell data that follow:
            if (myrank .eq. master) then
               read(lunit) name
               inquire(lunit,pos=last_record)
            endif

            !master ensures all processes get the position in the file and the name of the shell stored
            call mpi_mod_bcast(last_record,master)
            call mpi_mod_bcast(name,master)

            !Read and add the appropriate type of shell:
            if (name .eq. CGTO_shell_data%name()) then
               call CGTO_shell_data%read(lunit,last_record,pos_after_read,basis_data_file%identifier)
               call this%add_shell(CGTO_shell_data)
            elseif (name .eq. BTO_shell_data%name()) then
               call BTO_shell_data%read(lunit,last_record,pos_after_read,basis_data_file%identifier)
               call this%add_shell(BTO_shell_data)
            elseif (name .eq. orbital_data%name()) then
               call orbital_data%read(lunit,last_record,pos_after_read,basis_data_file%identifier)
               call this%add_shell(orbital_data)
            else
               call xermsg('basis_data_generic_obj','read_bdg','Unimplemented shell-type.',3,1)
            endif

            !master ensures all processes know where the record ends
            if (myrank .eq. master) last_record = pos_after_read
            call mpi_mod_bcast(last_record,master)

         enddo !i

         write(level3,'(/,"Data for ",i0," shells of functions read-in.")') this%number_of_shells

         call basis_data_file%close

         write(level3,'("<---------","basis_data_generic_obj:read")')

         return

 10      call xermsg ('basis_data_generic_obj', 'read_bdg', 'Error reading the shell data from the file and position given.', 4, 1)

   end subroutine read_bdg

end module basis_data_generic_gbl
