clear
%% Plane projector variance estimation on multiple qubits
% Variance optimisation for the special case of tensor projectors on a
% plane, as in the subspace spanned by only two of the three Paulis
%
% Input is given by the number of qubits to be considered, the number of
% projectors analysed and the number of cycles of the optimisation.
% Each of the optimised states is saved independently for ease of access,
% together with the corresponding optimal coefficients
% "Aggregator" results, such as the actual optimal variance and purity, are
% also separately saved for all the results


Ncycle = 15;    % number of repetitions of fminicon
density = 30;   % how many different angles are considered
offset = 0;

N = 5;   % total number of qubits
d = 2^N; % dimension of vector space
D = d^2; % dimension of *full* Hilbert Space 

%% POVM matrix
% Plane POVM, made up of X and Z eigenstates
dir = "data/";
dir_states = dir+"planeprojXZ_states/";
filename = "planeprojXZ_"+N;
ppovm = plane_povm(1,3); % let's take the plane case

full_povm = povm_tensor_cycle(ppovm, N);
[pcm, bm, em, DD] = coef_matrix (full_povm);

n = max(size(pcm)); % number of distinct effects in POVM
Ds = min(size(pcm)); % dimenstion of subspace

%% aggregate containers
% rather than selecting different columns, I'd rather have keywords in the
% same file
variance_vec     = zeros(density, 1);
purity_vec       = zeros(density,1);
entanglement_vec = zeros(density,1); % saves number of non-zero elements 


%% canonical example
obs = qubit(0, 0); % this is equivalent for every choice of theta
if N >1
    full_obs = tensor_obs(obs, N);
else
    full_obs = obs;
end
can_coef = em*bm'*full_obs(:);

[var_can, rho_flat_can] =var_opt_fixcoef(pcm, can_coef, Ncycle, bm);
rho_can = reshape(bm*rho_flat_can, [d, d]); % obtains density matrix in 
save(dir_states+filename+"std.mat", "rho_can");


%% direct variance optimisation cycle

for i =1:density
    j = i ;
    theta = j*pi/(density*2); % angle between 0 and \pi/2 (then symmetric)
    obs = qubit(theta, 0);
    if N >1
        full_obs = tensor_obs(obs, N);
    else
        full_obs = obs;
    end
    flat_obs = bm'*full_obs(:);

    [var, rho_flat, opt_coefs] = var_opt(pcm, flat_obs, Ncycle, bm);
    rho = reshape(bm*rho_flat, [d, d]); % obtains density matrix in proper basis
    s_eigs = schmidt_eig(rho); % calculate entanglement for the density matrix
    variance_vec(j) = var; % optimal variance
    purity_vec(j) = purity(rho);
    entanglement_vec(j,:) = nnz(round(s_eigs, 8));

    % for each observable (determined by the index and the density to 
    % recover theta) saves the optimal state
    save(dir_states+filename+"_th_"+j+"_"+density+".mat", "rho", "s_eigs","opt_coefs");

    % in order not to loose results if cycle breaks, file overwritten each time
    save(dir+filename+".mat","density","offset","variance_vec", "entanglement_vec","purity_vec");%,"var_can"
end


%%


