%% Optimal state for fixed coefficients
% Given a set of valid reconstruction coefficients, we look for the state
% which maximises the variance. This is an alternative *tighter* upper
% bound compared to the shadow norm (which is also state-independent for a
% given set of coefficients, but can be estimated directly rather than
% needing to find the optimal state)
% The difference compared to the full variance optimisation is that the
% coefficients are fixed, rather than choosing coefficients which are
% optimal for the choice of the state.
%
% Input:    - POVM coefficient matrix 
%           - set of fixed coefficients ( we don't need flatobs, since we 
%               the coefficients already guarantee the reconstruction)
%           - Number of optimisation cycles
%           - Basis matrix           
%
% Output:   - maxiaml value of variance achieved 
%           - optimal state (which maximises variance )


function [var, rho_flat] = var_opt_fixcoef(povm_cm, coefs, Ncycle, basis_mat)

    options = optimoptions('fmincon');
    options = optimoptions(options,'StepTolerance',1e-8,'MaxFunctionEvaluations',10e4);

    D = max(size(basis_mat));   % dimension of *full* observable space
    
    c2 = conj(coefs).*coefs;         % vector of |x|^2
    varfix_c = @(x)-real(probs_rand_const (x, povm_cm, basis_mat)'*c2)+real(probs_rand_const(x, povm_cm, basis_mat)'*coefs)^2;
    
   
    x0 = ones(D,1)/(D); % initialisation doesn't really matter…
    lb = -ones(D,1);
    ub = ones(D,1);

    x_opt = x0;
    optvar = varfix_c(x_opt);
    
    eps = 0.001;
    diff = eps*ones(D,1);

    for i=1:Ncycle
        
        x0 = x_opt + eps*rand(D,1).*diff;
        [x_c, fval] = fmincon(varfix_c, x0,[],[],[],[],lb,ub,[],options);
        
        if fval < optvar
            optvar = fval;
            x_opt = x_c;     
            diff = x0 - x_c;
        end
    end
    
    var = -optvar;
    
    rho_flat = rhoflat_construction(x_opt, basis_mat);
    
   
end

