%% Variance optimisation
% Estimation of optimal variance
%
% Input:    - POVM coefficient matrix
%           - flattened observable (in the same basis)
%           - Number of optimisation cycles
%           - Basis matrix
%
% Output:   - optimal variance
%           - optimal state (which maximises <A^2>)
%           - associated optimised coefficients


function [var, rho_flat, oc] = var_opt(povm_mat, flat_obs, Ncycle, basis_mat)

    st = 10e-8;
    mfe = 10e4; % maximum function evaluations for optimization
    eps = 1e-4; % small perturbation for optimization

    D = max(size(basis_mat)); % dimension of *full* observable space
    varfun_c = @(x)-state_opt_sn(povm_mat, flat_obs, rhoflat_construction(x, basis_mat)) + real(((rhoflat_construction(x, basis_mat)).'*flat_obs))^2;  % negative to guarantee convexity
    % reality of <A>^2 imposed to guarantee that function can handle it 
    % (but already guaranteed by the fact that x exactly reconstructs an
    % hermitian observable)
    
    options = optimoptions('fmincon');
    options = optimoptions(options,'StepTolerance',st,'MaxFunctionEvaluations',mfe);
    
    x0 = ones(D,1)/(D); % initialisation doesn't really matter…
    % doesn't directly represent density matrix, but can be built from it
    lb = -ones(D,1);
    ub = ones(D,1);

    x_opt = x0;
    optvar = varfun_c(x_opt);
    %eps = 0.0001;
    diff = eps*ones(D,1);

    for i=1:Ncycle
        
        x0 = x_opt + eps*rand(D,1).*diff; % fluctuation proportional to change
        
        [x_c, fval] = fmincon(varfun_c, x0,[],[],[],[],lb,ub,[],options);
        
        if fval < optvar
            optvar = fval;
            x_opt = x_c;  
            diff = x0 - x_opt;
        end
    end
    
    var = -optvar;
    
    rho_flat = rhoflat_construction(x_opt,basis_mat);
    oc = state_opt_invmat (povm_mat, rho_flat)*flat_obs;
   
end

