%% state-dependent shadow norm
% Returns the shadow norm with optimal coefficients for fixed state
% This expression is found by minimising the shadow norm in terms of the
% coefficients for a fixed state (i.e. probability distribution of POVM
% outcomes). We can use this function then for the optimisation (variance
% maximisation)

% Inputs are :  - coefficient matrix of the POVM
%               - flattened target observable (expressed in the same basis)
%               - flattened fixed state ( expressed in the same basis)

% skips a step, instead of estimating coefficients first and sn later
function sn = state_opt_sn (povm_mat, flat_obs, flat_rho)
    n = max(size(povm_mat)); % dimensions of problem (no. of effects, dimension of observable)
    p = round(povm_mat*flat_rho,10);

    % Safety check:
    % we assume here that all probs !=0; if state were orthogonal to some 
    % effect, we can add a small noise parameter. 
    % Calculation done only if there are zero elements (there shouldn't)
    nz = n - nnz(p);
    if nz > 0
        % noise parameter, taken as the smallest element present divided by
        % all elements (in order not to perturb too much)
        eps = min(p(p>0))/n; 
        for i=1:n
            % this is just a safety check, if for any reason this were non-zero
            if p(i)== 0
                p(i) = eps;
            else
                p(i) = p(i)- eps/nz;
            end
        end
    end
    
    % definition of inverse probabilities matrix (should be well-defined
    % now)
    dmat = diag(1./p); % matrix of inverse probabilities
    aalt = (flat_obs.'/(povm_mat.'*dmat*povm_mat)).';
    oc = dmat*povm_mat*aalt; 

    % explicit definition rather than shortened one (which only applies in
    % the real case)
    sn = real(oc'*diag(p)*oc); 
    % reality imposed just not to confuse the function, but guaranteed by
    % the conj(a)*a product
    
end