%% Density matrix construction from random values
% Given the issues with the constrained optimisation function, it is much
% simpler to find random values and rearrange them into valid density
% matrices for the computaiton rather than relying on the external
% constraint: this way, we are surely guaranteed to have these contraints
% satisfied for any solution found

% Input:    - vector of coefficients, which represent real and imaginary 
%           part of coefficients, built smartly so only to require actually 
%           only d^2 coefficients (using its Hermitianity)           
%           - basis matrix, to ensure the flattened density matrix is in
%           the proper subspace

function rho_flat = rhoflat_construction(x, bm)
    
    d = fix(sqrt(length(x))); % dimension of vector, corresponds to vector space dimension
    if nargin < 2 % standard case of full rank POVM
        bm = eye(d^2);
    end
    dd = min(size(bm));  % dimension of subspace

    
    xmat = reshape(x,d,d);    % matrix of coefficients (not ordered)
    % first d elements are taken as the *diagonal* 
    diam = diag(diag(xmat)); % bc it was hard to only extract diagonal matrix

    % then we consider coefs for *lower* diagonal to be real coefficients
    ret = tril(xmat,-1);
    % while *upper diagonal* are imaginary coefficients
    imt = tril(xmat',-1);

    % now we actually build a valid Hermitian matrix
    T = (diam+ret+1j*imt);
    TT = T*T';
    
    %{
    % additional factor 2 in coefficients
    d = fix(sqrt(length(x)/2)); % dimension of vector, corresponds to vector space dimension
    T = reshape(x(1:d^2),d,d)+ 1j*reshape(x(d^2+1:end),d,d);
    TT = T*T';
    %}
    
    % now to project into proper subspace and reshape as density matrix 
    % (to properly normalise the matrix)
    if dd < d           % dimension of subspace smaller than HS space
        TT = reshape(bm*bm'*TT(:),d,d);
    end
    
    rho = TT / trace(TT); % we do need this normalisation
    

    rho_flat = bm'*rho(:); % final projection into proper subspace

end

% This apparently doesn't work as well!