!***********************************************************************
!*                   GNU Lesser General Public License
!*
!* This file is part of the GFDL Flexible Modeling System (FMS) Coupler.
!*
!* FMS Coupler is free software: you can redistribute it and/or modify
!* it under the terms of the GNU Lesser General Public License as
!* published by the Free Software Foundation, either version 3 of the
!* License, or (at your option) any later version.
!*
!* FMS Coupler is distributed in the hope that it will be useful, but
!* WITHOUT ANY WARRANTY; without even the implied warranty of
!* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
!* General Public License for more details.
!*
!* You should have received a copy of the GNU Lesser General Public
!* License along with FMS Coupler.
!* If not, see <http://www.gnu.org/licenses/>.
!***********************************************************************
!
!> \mainpage
!!
!! \brief  coupler_main couples component models for atmosphere, ocean,
!! land and sea ice on independent grids.  It also controls the time integration.
!!
!! \author Bruce Wyman <Bruce.Wyman@noaa.gov>
!! \author V. Balaji <V.Balaji@noaa.gov>
!!
!! This version couples model components representing atmosphere, ocean, land
!! and sea ice on independent grids. Each model component is represented by a
!! data type giving the instantaneous model state.
!!
!! The component models are coupled to allow implicit vertical diffusion of
!! heat and moisture at the interfaces of the atmosphere, land, and ice models.
!! As a result, the atmosphere, land, and ice models all use the same time step.
!! The atmospheric model has been separated into down and up calls that
!! correspond to the down and up sweeps of the standard tridiagonal elimination.
!!
!! The ocean interface uses explicit mixing. Fluxes to and from the ocean must
!! be passed through the ice model. This includes atmospheric fluxes as well as
!! fluxes from the land to the ocean (runoff).
!!
!! This program contains the model's main time loop. Each iteration of the
!! main time loop is one coupled (slow) time step. Within this slow time step
!! loop is a fast time step loop, using the atmospheric time step, where the
!! tridiagonal vertical diffusion equations are solved. Exchange between sea
!! ice and ocean occurs once every slow timestep.
!!
!! \section coupler_namelists  Namelists
!!
!! The three components of coupler: coupler_main, flux_exchange_mod, and surface_flux_mod
!! are configured through three namelists
!! * \ref coupler_config "coupler_nml"
!! * \ref flux_exchange_config "flux_exchange_nml"
!! * \ref surface_flux_config "surface_flux_nml"
!!
!!
!! \note
!! -# If no value is set for current_date, start_date, or calendar (or default value
!!     specified) then the value from restart file "INPUT/coupler.res" will be used.
!!     If neither a namelist value or restart file value exist the program will fail.
!! -# The actual run length will be the sum of months, days, hours, minutes, and
!!     seconds. A run length of zero is not a valid option.
!! -# The run length must be an intergal multiple of the coupling timestep dt_cpld.
!!
!! \section Main Program Example
!!
!! ~~~~~~~~~~{.f90}
!! DO slow time steps (ocean)
!!    call flux_ocean_to_ice
!!
!!    call set_ice_surface_fields
!!
!!    DO fast time steps (atmos)
!!       call flux_calculation
!!
!!       call ATMOS_DOWN
!!
!!       call flux_down_from_atmos
!!
!!       call LAND_FAST
!!
!!       call ICE_FAST
!!
!!       call flux_up_to_atmos
!!
!!       call ATMOS_UP
!!    ENDDO
!!
!!    call ICE_SLOW
!!
!!    call flux_ice_to_ocean
!!
!!    call OCEAN
!! ENDDO
!! ~~~~~~~~~~

!> \page coupler_config Coupler Configuration
!!
!! coupler_main is configured via the coupler_nml namelist in the `input.nml` file.
!! The following table contains the available namelist variables.
!!
!! <table>
!!   <tr>
!!     <th>Variable Name</th>
!!     <th>Type</th>
!!     <th>Default Value</th>
!!     <th>Description</th>
!!   </tr>
!!   <tr>
!!     <td>current_date</td>
!!     <td>integer, dimension(6)</td>
!!     <td>(/0,0,0,0,0,0/)</td>
!!     <td>The date that the current integration starts with.</td>
!!   </tr>
!!   <tr>
!!     <td>force_date_from_namelist</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>Flag that determines whether the namelist variable current_date should
!!       override the date in the restart file INPUT/coupler.res. If the restart
!!       file does not exist then force_date_from_namelist has not effect, the
!!       value of current_date will be used.</td>
!!   </tr>
!!   <tr>
!!     <td>calendar</td>
!!     <td>character(len=17)</td>
!!     <td>''</td>
!!     <td>The calendar type used by the current integration. Valid values are
!!       consistent with the time_manager module: 'julian', 'noleap', or
!!       'thirty_day'. The value 'no_calendar' can not be used because the
!!       time_manager's date function are used. All values must be
!!       lowercase.</td>
!!   </tr>
!!   <tr>
!!     <td>months</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>The number of months that the current integration will be run for.</td>
!!   </tr>
!!   <tr>
!!     <td>days</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>The number of days that the current integration will be run for.</td>
!!   </tr>
!!   <tr>
!!     <td>hours</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>The number of hours that the current integration will be run for.</td>
!!   </tr>
!!   <tr>
!!     <td>minutes</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>The number of minutes that the current integration will be run for.</td>
!!   </tr>
!!   <tr>
!!     <td>seconds</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>The number of seconds that the current integration will be run for.</td>
!!   </tr>
!!   <tr>
!!     <td>dt_atmos</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>Atmospheric model time step in seconds, including the fast coupling
!!       with land and sea ice.</td>
!!   </tr>
!!   <tr>
!!     <td>dt_cpld</td>
!!     <td>integer</td>
!!     <td>0</td>
!!     <td>Time step in seconds for coupling between ocean and atmospheric models:
!!       must be an integral multiple of dt_atmos and dt_ocean. This is the
!!       "slow" timestep.</td>
!!   </tr>
!!   <tr>
!!     <td>do_atmos, do_ocean, do_ice, do_land, do_flux</td>
!!     <td>logical</td>
!!     <td>.TRUE.</td>
!!     <td>If true (default), that particular model component (atmos, etc.) is
!!       run. If false, the execution of that component is skipped. This is used
!!       when ALL the output fields sent by that component to the coupler have
!!       been overridden using the data_override feature. For advanced users
!!       only: if you're not sure, you should leave these values at TRUE.</td>
!!   </tr>
!!   <tr>
!!     <td>concurrent</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>If true, the ocean executes concurrently with the atmosphere-land-ocean
!!       on a separate set of PEs. If false (default), the execution is serial:
!!       call atmos... followed by call ocean... If using concurrent execution,
!!       you must set one of atmos_npes and ocean_npes, see below.</td>
!!   </tr>
!!   <tr>
!!     <td>do_concurrent_radiation</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>If true then radiation is done concurrently.</td>
!!   </tr>
!!   <tr>
!!     <td>atmos_npes, ocean_npes</td>
!!     <td>integer</td>
!!     <td>none</td>
!!     <td>If concurrent is set to true, we use these to set the list of PEs on
!!       which each component runs. At least one of them must be set to a number
!!       between 0 and NPES. If exactly one of these two is set non-zero, the
!!       other is set to the remainder from NPES. If both are set non-zero they
!!       must add up to NPES.</td>
!!   </tr>
!!   <tr>
!!     <td>atmos_nthreads, ocean_nthreads</td>
!!     <td>integer</td>
!!     <td>1</td>
!!     <td>We set here the number of OpenMP threads to use separately for each
!!       component.</td>
!!   </tr>
!!   <tr>
!!     <td>radiation_nthreads</td>
!!     <td>integer</td>
!!     <td>1</td>
!!     <td>Number of threads to use for the concurrent radiation
!!       when do_concurrent_radiation = .true., otherwise is equal
!!       to atmos_nthreads </td>
!!   </tr>
!!   <tr>
!!     <td>use_lag_fluxes</td>
!!     <td>logical</td>
!!     <td>.TRUE.</td>
!!     <td> If true, the ocean is forced with SBCs from one coupling timestep ago.
!!       If false, the ocean is forced with most recent SBCs.  For an old leapfrog
!!       MOM4 coupling with dt_cpld=dt_ocean, lag fluxes can be shown to be stable
!!       and current fluxes to be unconditionally unstable.  For dt_cpld>dt_ocean there
!!       is probably sufficient damping for MOM4.  For more modern ocean models (such as
!!       MOM5, GOLD or MOM6) that do not use leapfrog timestepping, use_lag_fluxes=.False.
!!       should be much more stable.</td>
!!   </tr>
!!   <tr>
!!     <td>concurrent_ice</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>If true, the slow sea-ice is forced with the fluxes that were used for
!!       the fast ice processes one timestep before.  When used in conjuction with
!!       setting slow_ice_with_ocean=true, this approach allows the atmosphere and
!!       ocean to run concurrently even if use_lag_fluxes=.FALSE., and it can be
!!       shown to ameliorate or eliminate several ice-ocean coupled instabilities.</td>
!!   </tr>
!!   <tr>
!!     <td>slow_ice_with_ocean</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>If true, the slow sea-ice is advanced on the ocean processors.  Otherwise
!!       the slow sea-ice processes are on the same PEs as the fast sea-ice.</td>
!!   </tr>
!!   <tr>
!!     <td>restart_interval</td>
!!     <td>integer, dimension(6)</td>
!!     <td>(/0,0,0,0,0,0/)</td>
!!     <td>The time interval that write out intermediate restart file. The format
!!       is (yr,mo,day,hr,min,sec). When restart_interval is all zero, no
!!       intermediate restart file will be written out.</td>
!!   </tr>
!!   <tr>
!!     <td>do_debug</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>If .TRUE. print additional debugging messages.</td>
!!   </tr>
!!   <tr>
!!     <td>do_chksum</td>
!!     <td>logical</td>
!!     <td>.FALSE.</td>
!!     <td>Turns on/off checksum of certain variables.</td>
!!   </tr>
!!   <tr>
!!     <td>do_endpoint_chksum</td>
!!     <td>logical</td>
!!     <td>.TRUE.</td>
!!     <td>Report checksums of the start and end states of certain variables.</td>
!!   </tr>
!! </table>
!!
!! \note
!! -# If no value is set for current_date, start_date, or calendar (or default value specified) then the value from
!!    restart file "INPUT/coupler.res" will be used. If neither a namelist value or restart file value exist the
!!    program will fail.
!! -# The actual run length will be the sum of months, days, hours, minutes, and seconds. A run length of zero is not a
!!    valid option.
!! -# The run length must be an intergal multiple of the coupling timestep dt_cpld.

!> \throw FATAL, "no namelist value for current_date"
!!     A namelist value for current_date must be given if no restart file for
!!     coupler_main (INPUT/coupler.res) is found.
!! \throw FATAL, "invalid namelist value for calendar"
!!     The value of calendar must be 'julian', 'noleap', or 'thirty_day'.
!!     See the namelist documentation.
!! \throw FATAL, "no namelist value for calendar"
!!     If no restart file is present, then a namelist value for calendar
!!     must be specified.
!! \throw FATAL, "initial time is greater than current time"
!!     If a restart file is present, then the namelist value for either
!!     current_date or start_date was incorrectly set.
!! \throw FATAL, "run length must be multiple of ocean time step"
!!     There must be an even number of ocean time steps for the requested run length.
!! \throw FATAL, "final time does not match expected ending time"
!!     This error should probably not occur because of checks done at initialization time.
program coupler_main

  use constants_mod,           only: constants_init

  use time_manager_mod,        only: time_type, set_calendar_type, set_time
  use time_manager_mod,        only: set_date, get_date, days_in_month, month_name
  use time_manager_mod,        only: operator(+), operator(-), operator (<)
  use time_manager_mod,        only: operator (>), operator ( /= ), operator ( / )
  use time_manager_mod,        only: operator (*), THIRTY_DAY_MONTHS, JULIAN
  use time_manager_mod,        only: NOLEAP, NO_CALENDAR, INVALID_CALENDAR
  use time_manager_mod,        only: date_to_string, increment_date
  use time_manager_mod,        only: operator(>=), operator(<=), operator(==)

  use fms_mod,                 only: open_namelist_file, field_exist, file_exist, check_nml_error
  use fms_mod,                 only: uppercase, error_mesg, write_version_number
  use fms_mod,                 only: fms_init, fms_end, stdout
  use fms_mod,                 only: read_data, write_data

  use fms_io_mod,              only: fms_io_exit
  use fms_io_mod,              only: restart_file_type, register_restart_field
  use fms_io_mod,              only: save_restart, restore_state

  use diag_manager_mod,        only: diag_manager_init, diag_manager_end, diag_grid_end
  use diag_manager_mod,        only: DIAG_OCEAN, DIAG_OTHER, DIAG_ALL, get_base_date
  use diag_manager_mod,        only: diag_manager_set_time_end

  use field_manager_mod,       only: MODEL_ATMOS, MODEL_LAND, MODEL_ICE

  use tracer_manager_mod,      only: tracer_manager_init, get_tracer_index
  use tracer_manager_mod,      only: get_number_tracers, get_tracer_names, NO_TRACER

  use coupler_types_mod,       only: coupler_types_init, coupler_1d_bc_type
  use coupler_types_mod,       only: coupler_type_write_chksums
  use coupler_types_mod,       only: coupler_type_register_restarts, coupler_type_restore_state

  use data_override_mod,       only: data_override_init

!
! model interfaces used to couple the component models:
!               atmosphere, land, ice, and ocean
!

  use atmos_model_mod,         only: atmos_model_init, atmos_model_end
  use atmos_model_mod,         only: update_atmos_model_dynamics
  use atmos_model_mod,         only: update_atmos_model_down
  use atmos_model_mod,         only: update_atmos_model_up
  use atmos_model_mod,         only: atmos_data_type
  use atmos_model_mod,         only: land_ice_atmos_boundary_type
  use atmos_model_mod,         only: atmos_data_type_chksum
  use atmos_model_mod,         only: lnd_ice_atm_bnd_type_chksum
  use atmos_model_mod,         only: lnd_atm_bnd_type_chksum
  use atmos_model_mod,         only: ice_atm_bnd_type_chksum
  use atmos_model_mod,         only: atmos_model_restart
  use atmos_model_mod,         only: update_atmos_model_radiation
  use atmos_model_mod,         only: update_atmos_model_state

  use land_model_mod,          only: land_model_init, land_model_end
  use land_model_mod,          only: land_data_type, atmos_land_boundary_type
  use land_model_mod,          only: update_land_model_fast, update_land_model_slow
  use land_model_mod,          only: atm_lnd_bnd_type_chksum
  use land_model_mod,          only: land_data_type_chksum
  use land_model_mod,          only: land_model_restart

  use ice_model_mod,           only: ice_model_init, share_ice_domains, ice_model_end, ice_model_restart
  use ice_model_mod,           only: update_ice_model_fast, set_ice_surface_fields
  use ice_model_mod,           only: ice_data_type, land_ice_boundary_type
  use ice_model_mod,           only: ocean_ice_boundary_type, atmos_ice_boundary_type
  use ice_model_mod,           only: ice_data_type_chksum, ocn_ice_bnd_type_chksum
  use ice_model_mod,           only: atm_ice_bnd_type_chksum, lnd_ice_bnd_type_chksum
  use ice_model_mod,           only: unpack_ocean_ice_boundary, exchange_slow_to_fast_ice
  use ice_model_mod,           only: ice_model_fast_cleanup, unpack_land_ice_boundary
  use ice_model_mod,           only: exchange_fast_to_slow_ice, update_ice_model_slow

  use ocean_model_mod,         only: update_ocean_model, ocean_model_init,  ocean_model_end
  use ocean_model_mod,         only: ocean_public_type, ocean_state_type, ice_ocean_boundary_type
  use ocean_model_mod,         only: ocean_model_restart
  use ocean_model_mod,         only: ocean_public_type_chksum, ice_ocn_bnd_type_chksum

  use combined_ice_ocean_driver, only: update_slow_ice_and_ocean, ice_ocean_driver_type
  use combined_ice_ocean_driver, only: ice_ocean_driver_init, ice_ocean_driver_end
!
! flux_ calls translate information between model grids - see flux_exchange.f90
!

  use flux_exchange_mod,       only: flux_exchange_init, gas_exchange_init, sfc_boundary_layer
  use flux_exchange_mod,       only: generate_sfc_xgrid, send_ice_mask_sic
  use flux_exchange_mod,       only: flux_down_from_atmos, flux_up_to_atmos
  use flux_exchange_mod,       only: flux_land_to_ice, flux_ice_to_ocean, flux_ocean_to_ice
  use flux_exchange_mod,       only: flux_ice_to_ocean_finish, flux_ocean_to_ice_finish
  use flux_exchange_mod,       only: flux_check_stocks, flux_init_stocks
  use flux_exchange_mod,       only: flux_ocean_from_ice_stocks, flux_ice_to_ocean_stocks
  use flux_exchange_mod,       only: flux_atmos_to_ocean, flux_ex_arrays_dealloc

  use atmos_tracer_driver_mod, only: atmos_tracer_driver_gather_data

  use mpp_mod,                 only: mpp_clock_id, mpp_clock_begin, mpp_clock_end, mpp_chksum
  use mpp_mod,                 only: mpp_init, mpp_pe, mpp_npes, mpp_root_pe, mpp_sync
  use mpp_mod,                 only: stderr, stdlog, mpp_error, NOTE, FATAL, WARNING
  use mpp_mod,                 only: mpp_set_current_pelist, mpp_declare_pelist
  use mpp_mod,                 only: input_nml_file

  use mpp_io_mod,              only: mpp_open, mpp_close, mpp_io_clock_on
  use mpp_io_mod,              only: MPP_NATIVE, MPP_RDONLY, MPP_DELETE

  use mpp_domains_mod,         only: mpp_broadcast_domain

  use memutils_mod,            only: print_memuse_stats
  use iso_fortran_env

  implicit none

!-----------------------------------------------------------------------

  character(len=128) :: version = '$Id$'
  character(len=128) :: tag = '$Name$'

!-----------------------------------------------------------------------
!---- model defined-types ----

  type (atmos_data_type) :: Atm
  type  (land_data_type) :: Land
  type   (ice_data_type) :: Ice
  ! allow members of ocean type to be aliased (ap)
  type (ocean_public_type), target :: Ocean
  type (ocean_state_type),  pointer :: Ocean_state => NULL()

  type(atmos_land_boundary_type)     :: Atmos_land_boundary
  type(atmos_ice_boundary_type)      :: Atmos_ice_boundary
  type(land_ice_atmos_boundary_type) :: Land_ice_atmos_boundary
  type(land_ice_boundary_type)       :: Land_ice_boundary
  type(ice_ocean_boundary_type)      :: Ice_ocean_boundary
  type(ocean_ice_boundary_type)      :: Ocean_ice_boundary
  type(ice_ocean_driver_type), pointer :: ice_ocean_driver_CS => NULL()

!-----------------------------------------------------------------------
! ----- coupled model time -----

  type (time_type) :: Time, Time_init, Time_end, &
                      Time_step_atmos, Time_step_cpld
  type(time_type) :: Time_atmos, Time_ocean
  type(time_type) :: Time_flux_ice_to_ocean, Time_flux_ocean_to_ice

  integer :: num_atmos_calls, na
  integer :: num_cpld_calls, nc

!------ for intermediate restart
  type(restart_file_type), dimension(:), pointer :: &
    Ice_bc_restart => NULL(), Ocn_bc_restart => NULL()
  integer                              :: num_ice_bc_restart=0, num_ocn_bc_restart=0
  type(time_type)                      :: Time_restart, Time_restart_current, Time_start
  character(len=32)                    :: timestamp

! ----- coupled model initial date -----

  integer :: date_init(6) = (/ 0, 0, 0, 0, 0, 0 /)
  integer :: calendar_type = INVALID_CALENDAR

!-----------------------------------------------------------------------
!------ namelist interface -------

  integer, dimension(6) :: restart_interval = (/ 0, 0, 0, 0, 0, 0/) !< The time interval that write out intermediate restart file.
                                                                    !! The format is (yr,mo,day,hr,min,sec).  When restart_interval
                                                                    !! is all zero, no intermediate restart file will be written out
  integer, dimension(6) :: current_date     = (/ 0, 0, 0, 0, 0, 0 /) !< The date that the current integration starts with.  (See
                                                                     !! force_date_from_namelist.)
  character(len=17) :: calendar = '                 ' !< The calendar type used by the current integration.  Valid values are
                                                      !! consistent with the time_manager module: 'julian', 'noleap', or 'thirty_day'.
                                                      !! The value 'no_calendar' cannot be used because the time_manager's date
                                                      !! functions are used.  All values must be lower case.
  logical :: force_date_from_namelist = .false.  !< Flag that determines whether the namelist variable current_date should override
                                                 !! the date in the restart file `INPUT/coupler.res`.  If the restart file does not
                                                 !! exist then force_date_from_namelist has no effect, the value of current_date
                                                 !! will be used.
  integer :: months=0  !< Number of months the current integration will be run
  integer :: days=0    !< Number of days the current integration will be run
  integer :: hours=0   !< Number of hours the current integration will be run
  integer :: minutes=0 !< Number of minutes the current integration will be run
  integer :: seconds=0 !< Number of seconds the current integration will be run
  integer :: dt_atmos = 0 !< Atmospheric model time step in seconds, including the fat coupling with land and sea ice
  integer :: dt_cpld  = 0 !< Time step in seconds for coupling between ocean and atmospheric models.  This must be an
                          !! integral multiple of dt_atmos and dt_ocean.  This is the "slow" timestep.
  integer :: atmos_npes=0 !< The number of MPI tasks to use for the atmosphere
  integer :: ocean_npes=0 !< The number of MPI tasks to use for the ocean
  integer :: ice_npes=0   !< The number of MPI tasks to use for the ice
  integer :: land_npes=0  !< The number of MPI tasks to use for the land
  integer :: atmos_nthreads=1 !< Number of OpenMP threads to use in the atmosphere
  integer :: ocean_nthreads=1 !< Number of OpenMP threads to use in the ocean
  integer :: radiation_nthreads=1 !< Number of threads to use for the radiation.
  logical :: do_atmos =.true. !< Indicates if this component should be executed.  If .FALSE., then execution is skipped.
                              !! This is used when ALL the output fields sent by this component to the coupler have been
                              !! overridden  using the data_override feature.  This is for advanced users only.
  logical :: do_land =.true. !< See do_atmos
  logical :: do_ice =.true.  !< See do_atmos
  logical :: do_ocean=.true. !< See do_atmos
  logical :: do_flux =.true. !< See do_atmos
  logical :: concurrent=.FALSE. !< If .TRUE., the ocean executes concurrently with the atmosphere-land-ice on a separate
                                !! set of PEs.  Concurrent should be .TRUE. if concurrent_ice is .TRUE.
                                !! If .FALSE., the execution is serial: call atmos... followed by call ocean...
  logical :: do_concurrent_radiation=.FALSE. !< If .TRUE. then radiation is done concurrently
  logical :: use_lag_fluxes=.TRUE.  !< If .TRUE., the ocean is forced with SBCs from one coupling timestep ago.
                                    !! If .FALSE., the ocean is forced with most recent SBCs.  For an old leapfrog
                                    !! MOM4 coupling with dt_cpld=dt_ocean, lag fluxes can be shown to be stable
                                    !! and current fluxes to be unconditionally unstable.  For dt_cpld>dt_ocean there
                                    !! is probably sufficient damping for MOM4.  For more modern ocean models (such as
                                    !! MOM5, GOLD or MOM6) that do not use leapfrog timestepping, use_lag_fluxes=.False.
                                    !! should be much more stable.
  logical :: concurrent_ice=.FALSE. !< If .TRUE., the slow sea-ice is forced with the fluxes that were used for the
                                    !! fast ice processes one timestep before.  When used in conjuction with setting
                                    !! slow_ice_with_ocean=.TRUE., this approach allows the atmosphere and
                                    !! ocean to run concurrently even if use_lag_fluxes=.FALSE., and it can
                                    !! be shown to ameliorate or eliminate several ice-ocean coupled instabilities.
  logical :: slow_ice_with_ocean=.FALSE. !< If true, the slow sea-ice is advanced on the ocean processors.  Otherwise
                                    !! the slow sea-ice processes are on the same PEs as the fast sea-ice.
  logical :: combined_ice_and_ocean=.FALSE. !< If true, there is a single call from the coupler to advance
                                    !! both the slow sea-ice and the ocean. slow_ice_with_ocean and
                                    !! concurrent_ice must both be true if combined_ice_and_ocean is true.
  logical :: do_chksum=.FALSE.      !! If .TRUE., do multiple checksums throughout the execution of the model.
  logical :: do_endpoint_chksum=.TRUE.  !< If .TRUE., do checksums of the initial and final states.
  logical :: do_debug=.FALSE.       !< If .TRUE. print additional debugging messages.
  integer :: check_stocks = 0 ! -1: never 0: at end of run only n>0: every n coupled steps
  logical :: use_hyper_thread = .false.
  integer :: ncores_per_node = 0
  logical :: debug_affinity = .false.

  namelist /coupler_nml/ current_date, calendar, force_date_from_namelist,         &
                         months, days, hours, minutes, seconds, dt_cpld, dt_atmos, &
                         do_atmos, do_land, do_ice, do_ocean, do_flux,             &
                         atmos_npes, ocean_npes, ice_npes, land_npes,              &
                         atmos_nthreads, ocean_nthreads, radiation_nthreads,       &
                         concurrent, do_concurrent_radiation, use_lag_fluxes,      &
                         check_stocks, restart_interval, do_debug, do_chksum,      &
                         use_hyper_thread, ncores_per_node, debug_affinity,        &
                         concurrent_ice, slow_ice_with_ocean, do_endpoint_chksum,  &
                         combined_ice_and_ocean

  integer :: initClock, mainClock, termClock

  integer :: newClock0, newClock1, newClock2, newClock3, newClock4, newClock5, newClock7
  integer :: newClock6f, newClock6s, newClock6e, newClock10f, newClock10s, newClock10e
  integer :: newClock8, newClock9, newClock11, newClock12, newClock13, newClock14, newClocka
  integer :: newClockb, newClockc, newClockd, newClocke, newClockf, newClockg, newClockh, newClocki
  integer :: newClockj, newClockk, newClockl

  integer :: id_atmos_model_init, id_land_model_init, id_ice_model_init
  integer :: id_ocean_model_init, id_flux_exchange_init

  character(len=80) :: text
  character(len=48), parameter                    :: mod_name = 'coupler_main_mod'

  integer :: outunit
  integer :: ensemble_id = 1
  integer, allocatable :: ensemble_pelist(:, :)
  integer, allocatable :: slow_ice_ocean_pelist(:)
  integer :: conc_nthreads = 1
  integer :: omp_get_thread_num, omp_get_num_threads
  real :: omp_get_wtime
  real :: dsec, omp_sec(2)=0.0, imb_sec(2)=0.0

!#######################################################################
  INTEGER :: i, status, arg_count
  CHARACTER(len=256) :: executable_name, arg, fredb_id

#ifdef FREDB_ID
#define xstr(s) str(s)
#define str(s) #s
  fredb_id = xstr(FREDB_ID)
#else
#warning "FREDB_ID not defined. Continuing as normal."
  fredb_id = 'FREDB_ID was not defined (e.g. -DFREDB_ID=...) during preprocessing'
#endif

  arg_count = command_argument_count()
  DO i=0, arg_count
     CALL get_command_argument(i, arg, status=status)
     if (status .ne. 0) then
        write (error_unit,*) 'get_command_argument failed: status = ', status, ' arg = ', i
        stop 1
     end if

     if (i .eq. 0) then
       executable_name = arg
     else if (arg == '--fredb_id') then
        write (output_unit,*) TRIM(fredb_id)
        stop
     end if
  END DO

  if (arg_count .ge. 1) then
    write (error_unit,*) 'Usage: '//TRIM(executable_name)//' [--fredb_id]'
    stop 1
  end if

  call mpp_init()
!these clocks are on the global pelist
  initClock = mpp_clock_id( 'Initialization' )
  call mpp_clock_begin(initClock)

  call fms_init
  call constants_init

  call coupler_init
  if (do_chksum) call coupler_chksum('coupler_init+', 0)

  call mpp_set_current_pelist()

  call mpp_clock_end (initClock) !end initialization

  call mpp_clock_begin(mainClock) !begin main loop

!-----------------------------------------------------------------------
!------ ocean/slow-ice integration loop ------

  if (check_stocks >= 0) then
    call mpp_set_current_pelist()
    call flux_init_stocks(Time, Atm, Land, Ice, Ocean_state)
  endif

  if (Atm%pe) then
    call mpp_set_current_pelist(Atm%pelist)
    newClock1 = mpp_clock_id( 'generate_sfc_xgrid' )
  endif
  if (Ice%slow_ice_PE .or. Ocean%is_ocean_pe) then
     call mpp_set_current_pelist(slow_ice_ocean_pelist)
    newClock2 = mpp_clock_id( 'flux_ocean_to_ice' )
    newClock3 = mpp_clock_id( 'flux_ice_to_ocean' )
  endif
  if (Atm%pe) then
    call mpp_set_current_pelist(Atm%pelist)
    newClock5 = mpp_clock_id( 'ATM' )
    newClock7  = mpp_clock_id( ' ATM: atmos loop' )
    newClocka  = mpp_clock_id( '  A-L: atmos_tracer_driver_gather_data' )
    newClockb  = mpp_clock_id( '  A-L: sfc_boundary_layer' )
    newClockl  = mpp_clock_id( '  A-L: update_atmos_model_dynamics')
    if (.not. do_concurrent_radiation) then
      newClockj  = mpp_clock_id( '  A-L: serial radiation' )
    endif
    newClockc  = mpp_clock_id( '  A-L: update_atmos_model_down' )
    newClockd  = mpp_clock_id( '  A-L: flux_down_from_atmos' )
    newClocke  = mpp_clock_id( '  A-L: update_land_model_fast' )
    newClockf  = mpp_clock_id( '  A-L: update_ice_model_fast' )
    newClockg  = mpp_clock_id( '  A-L: flux_up_to_atmos' )
    newClockh  = mpp_clock_id( '  A-L: update_atmos_model_up' )
    if (do_concurrent_radiation) then
      newClockj  = mpp_clock_id( '  A-L: concurrent radiation' )
      newClocki  = mpp_clock_id( '  A-L: concurrent atmos' )
    endif
    newClockk  = mpp_clock_id( '  A-L: update_atmos_model_state')
    newClock8  = mpp_clock_id( ' ATM: update_land_model_slow' )
    newClock9  = mpp_clock_id( ' ATM: flux_land_to_ice' )
  endif
  if (Ice%pe) then
    if (Ice%fast_ice_pe) call mpp_set_current_pelist(Ice%fast_pelist)
    newClock6f = mpp_clock_id( ' Ice: set_ice_surface fast' )
    newClock10f = mpp_clock_id( ' Ice: update_ice_model_slow fast' )

    if (Ice%slow_ice_pe) call mpp_set_current_pelist(Ice%slow_pelist)
    newClock6s = mpp_clock_id( ' Ice: set_ice_surface slow' )
    newClock10s = mpp_clock_id( ' Ice: update_ice_model_slow slow' )
    newClock11 = mpp_clock_id( ' Ice: flux_ice_to_ocean_stocks' )

    call mpp_set_current_pelist(Ice%pelist)
    newClock6e = mpp_clock_id( ' Ice: set_ice_surface exchange' )
    newClock10e = mpp_clock_id( ' Ice: update_ice_model_slow exchange' )
  endif
  if (Ocean%is_ocean_pe) then
    call mpp_set_current_pelist(Ocean%pelist)
    newClock12 = mpp_clock_id( 'OCN' )
  endif
  call mpp_set_current_pelist()
  newClock4 = mpp_clock_id( 'flux_check_stocks' )
  newClock13 = mpp_clock_id( 'intermediate restart' )
  newClock14 = mpp_clock_id( 'final flux_check_stocks' )

  do nc = 1, num_cpld_calls
    if (do_chksum) call coupler_chksum('top_of_coupled_loop+', nc)
    call mpp_set_current_pelist()

    if (do_chksum) then
      if (Atm%pe) then
        call mpp_set_current_pelist(Atm%pelist)
        call atmos_ice_land_chksum('MAIN_LOOP-', nc, Atm, Land, Ice, &
                  Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
      endif
      if (Ocean%is_ocean_pe) then
        call mpp_set_current_pelist(Ocean%pelist)
        call ocean_chksum('MAIN_LOOP-', nc, Ocean, Ice_ocean_boundary)
      endif
      call mpp_set_current_pelist()
    endif

    ! Calls to flux_ocean_to_ice and flux_ice_to_ocean are all PE communication
    ! points when running concurrently. The calls are placed next to each other in
    ! concurrent mode to avoid multiple synchronizations within the main loop.
    ! With concurrent_ice, these only occur on the ocean PEs.
    if (Ice%slow_ice_PE .or. Ocean%is_ocean_pe) then
      ! If the slow ice is on a subset of the ocean PEs, use the ocean PElist.
       call mpp_set_current_pelist(slow_ice_ocean_pelist)
      call mpp_clock_begin(newClock2)
       !Redistribute quantities from Ocean to Ocean_ice_boundary
       !Ice intent is In.
       !Ice is used only for accessing Ice%area and knowing if we are on an Ice pe
      call flux_ocean_to_ice( Time, Ocean, Ice, Ocean_ice_boundary )
      Time_flux_ocean_to_ice = Time
      call mpp_clock_end(newClock2)

      ! Update Ice_ocean_boundary; the first iteration is supplied by restarts
      if (use_lag_fluxes) then
        call mpp_clock_begin(newClock3)
        call flux_ice_to_ocean( Time, Ice, Ocean, Ice_ocean_boundary )
        Time_flux_ice_to_ocean = Time
        call mpp_clock_end(newClock3)
      endif
    endif

    if (do_chksum) then
      call coupler_chksum('flux_ocn2ice+', nc)
      if (Atm%pe) then
        call mpp_set_current_pelist(Atm%pelist)
        call atmos_ice_land_chksum('fluxocn2ice+', nc, Atm, Land, Ice, &
                  Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
      endif
      if (Ocean%is_ocean_pe) then
        call mpp_set_current_pelist(Ocean%pelist)
        call ocean_public_type_chksum('fluxocn2ice+', nc, Ocean)
      endif
      call mpp_set_current_pelist()
    endif

    ! To print the value of frazil heat flux at the right time the following block
    ! needs to sit here rather than at the end of the coupler loop.
    if (check_stocks > 0) then
      call mpp_clock_begin(newClock4)
      if (check_stocks*((nc-1)/check_stocks) == nc-1 .AND. nc > 1) then
        call mpp_set_current_pelist()
        call flux_check_stocks(Time=Time, Atm=Atm, Lnd=Land, Ice=Ice, Ocn_state=Ocean_state)
      endif
      call mpp_clock_end(newClock4)
    endif

    if (do_ice .and. Ice%pe) then
      if (Ice%slow_ice_pe) then
        call mpp_set_current_pelist(Ice%slow_pelist)
        call mpp_clock_begin(newClock6s)

        ! This may do data override or diagnostics on Ice_ocean_boundary.
        call flux_ocean_to_ice_finish( Time_flux_ocean_to_ice, Ice, Ocean_Ice_Boundary )

        call unpack_ocean_ice_boundary( Ocean_ice_boundary, Ice )
        if (do_chksum) call slow_ice_chksum('update_ice_slow+', nc, Ice, Ocean_ice_boundary)
        call mpp_clock_end(newClock6s)
      endif

      ! This could be a point where the model is serialized if the fast and
      ! slow ice are on different PEs.
      if (.not.Ice%shared_slow_fast_PEs) call mpp_set_current_pelist(Ice%pelist)
      call mpp_clock_begin(newClock6e)
      call exchange_slow_to_fast_ice(Ice)
      call mpp_clock_end(newClock6e)

      if (concurrent_ice) then
        ! This call occurs all ice PEs.
        call mpp_clock_begin(newClock10e)
        call exchange_fast_to_slow_ice(Ice)
        call mpp_clock_end(newClock10e)
      endif

      if (Ice%fast_ice_pe) then
        if (.not.Ice%shared_slow_fast_PEs) call mpp_set_current_pelist(Ice%fast_pelist)
        call mpp_clock_begin(newClock6f)
        call set_ice_surface_fields(Ice)
        call mpp_clock_end(newClock6f)
      endif
    endif

    if (Atm%pe) then
      if (.NOT.(do_ice .and. Ice%pe) .OR. (ice_npes .NE. atmos_npes)) &
         call mpp_set_current_pelist(Atm%pelist)

      call mpp_clock_begin(newClock5)
      if (do_chksum) call atmos_ice_land_chksum('set_ice_surface+', nc, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
      call mpp_clock_begin(newClock1)
      call generate_sfc_xgrid( Land, Ice )
      call mpp_clock_end(newClock1)

      call send_ice_mask_sic(Time)

      !-----------------------------------------------------------------------
      !   ------ atmos/fast-land/fast-ice integration loop -------

      call mpp_clock_begin(newClock7)
      do na = 1, num_atmos_calls
        if (do_chksum) call atmos_ice_land_chksum('top_of_atmos_loop-', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)

        Time_atmos = Time_atmos + Time_step_atmos

        if (do_atmos) then
          call mpp_clock_begin(newClocka)
          call atmos_tracer_driver_gather_data(Atm%fields, Atm%tr_bot)
          call mpp_clock_end(newClocka)
        endif

        if (do_flux) then
          call mpp_clock_begin(newClockb)
          call sfc_boundary_layer( REAL(dt_atmos), Time_atmos, &
               Atm, Land, Ice, Land_ice_atmos_boundary )
          if (do_chksum)  call atmos_ice_land_chksum('sfc+', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          call mpp_clock_end(newClockb)
        endif

!$OMP   PARALLEL  &
!$OMP&    NUM_THREADS(conc_nthreads)  &
!$OMP&    DEFAULT(NONE)  &
!$OMP&    PRIVATE(conc_nthreads) &
!$OMP&    SHARED(atmos_nthreads, radiation_nthreads, nc, na, num_atmos_calls, atmos_npes, land_npes, ice_npes) &
!$OMP&    SHARED(Time_atmos, Atm, Land, Ice, Land_ice_atmos_boundary, Atmos_land_boundary, Atmos_ice_boundary) &
!$OMP&    SHARED(Ocean_ice_boundary) &
!$OMP&    SHARED(do_debug, do_chksum, do_atmos, do_land, do_ice, do_concurrent_radiation, omp_sec, imb_sec) &
!$OMP&    SHARED(newClockc, newClockd, newClocke, newClockf, newClockg, newClockh, newClocki, newClockj, newClockl)
!$      if (omp_get_thread_num() == 0) then
!$OMP     PARALLEL &
!$OMP&      NUM_THREADS(1) &
!$OMP&      DEFAULT(NONE) &
!$OMP&      PRIVATE(dsec) &
!$OMP&      SHARED(atmos_nthreads, radiation_nthreads, nc, na, num_atmos_calls, atmos_npes, land_npes, ice_npes) &
!$OMP&      SHARED(Time_atmos, Atm, Land, Ice, Land_ice_atmos_boundary, Atmos_land_boundary, Atmos_ice_boundary) &
!$OMP&      SHARED(Ocean_ice_boundary) &
!$OMP&      SHARED(do_debug, do_chksum, do_atmos, do_land, do_ice, do_concurrent_radiation, omp_sec, imb_sec) &
!$OMP&      SHARED(newClockc, newClockd, newClocke, newClockf, newClockg, newClockh, newClocki, newClockj, newClockl) 
!$        call omp_set_num_threads(atmos_nthreads)
!$        dsec=omp_get_wtime()
          if (do_concurrent_radiation) call mpp_clock_begin(newClocki)

          !      ---- atmosphere dynamics ----
          if (do_atmos) then
            call mpp_clock_begin(newClockl)
            call update_atmos_model_dynamics( Atm )
            call mpp_clock_end(newClockl)
          endif
          if (do_chksum) call atmos_ice_land_chksum('update_atmos_model_dynamics', (nc-1)*num_atmos_calls+na, &
                 Atm, Land, Ice, Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          if (do_debug)  call print_memuse_stats( 'update dyn')

          !      ---- SERIAL atmosphere radiation ----
          if (.not.do_concurrent_radiation) then
            call mpp_clock_begin(newClockj)
            call update_atmos_model_radiation( Land_ice_atmos_boundary, Atm )
            call mpp_clock_end(newClockj)
          endif
          if (do_chksum) call atmos_ice_land_chksum('update_atmos_model_radiation(ser)', (nc-1)*num_atmos_calls+na, &
                 Atm, Land, Ice, Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          if (do_debug)  call print_memuse_stats( 'update serial rad')

          !      ---- atmosphere down ----
          if (do_atmos) then
            call mpp_clock_begin(newClockc)
            call update_atmos_model_down( Land_ice_atmos_boundary, Atm )
            call mpp_clock_end(newClockc)
          endif
          if (do_chksum) call atmos_ice_land_chksum('update_atmos_down+', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          if (do_debug)  call print_memuse_stats( 'update down')

          call mpp_clock_begin(newClockd)
          call flux_down_from_atmos( Time_atmos, Atm, Land, Ice, &
                                     Land_ice_atmos_boundary, &
                                     Atmos_land_boundary, &
                                     Atmos_ice_boundary )
          call mpp_clock_end(newClockd)
          if (do_chksum) call atmos_ice_land_chksum('flux_down_from_atmos+', (nc-1)*num_atmos_calls+na, Atm, Land, &
                 Ice, Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)

          !      --------------------------------------------------------------
          !      ---- land model ----
          call mpp_clock_begin(newClocke)
          if (do_land .AND. land%pe) then
            if (land_npes .NE. atmos_npes) call mpp_set_current_pelist(Land%pelist)
            call update_land_model_fast( Atmos_land_boundary, Land )
          endif
          if (land_npes .NE. atmos_npes) call mpp_set_current_pelist(Atm%pelist)
          call mpp_clock_end(newClocke)
          if (do_chksum) call atmos_ice_land_chksum('update_land_fast+', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          if (do_debug)  call print_memuse_stats( 'update land')

          !      ---- ice model ----
          call mpp_clock_begin(newClockf)
          if (do_ice .AND. Ice%fast_ice_pe) then
            if (ice_npes .NE. atmos_npes)call mpp_set_current_pelist(Ice%fast_pelist)
            call update_ice_model_fast( Atmos_ice_boundary, Ice )
          endif
          if (ice_npes .NE. atmos_npes) call mpp_set_current_pelist(Atm%pelist)
          call mpp_clock_end(newClockf)
          if (do_chksum) call atmos_ice_land_chksum('update_ice_fast+', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          if (do_debug)  call print_memuse_stats( 'update ice')

          !      --------------------------------------------------------------
          !      ---- atmosphere up ----
          call mpp_clock_begin(newClockg)
          call flux_up_to_atmos( Time_atmos, Land, Ice, Land_ice_atmos_boundary, &
                                 Atmos_land_boundary, Atmos_ice_boundary )
          call mpp_clock_end(newClockg)
          if (do_chksum) call atmos_ice_land_chksum('flux_up2atmos+', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)

          call mpp_clock_begin(newClockh)
          if (do_atmos) &
            call update_atmos_model_up( Land_ice_atmos_boundary, Atm)
          call mpp_clock_end(newClockh)
          if (do_chksum) call atmos_ice_land_chksum('update_atmos_up+', (nc-1)*num_atmos_calls+na, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
          if (do_debug)  call print_memuse_stats( 'update up')

          call flux_atmos_to_ocean(Time_atmos, Atm, Atmos_ice_boundary, Ice)

          call flux_ex_arrays_dealloc

          !--------------
          if (do_concurrent_radiation) call mpp_clock_end(newClocki)
!$        omp_sec(1) = omp_sec(1) + (omp_get_wtime() - dsec)
!$OMP END PARALLEL
!$      endif
!$      if (omp_get_thread_num() == max(0,omp_get_num_threads()-1)) then
          !      ---- atmosphere radiation ----
          if (do_concurrent_radiation) then
!$OMP PARALLEL &
!$OMP&      NUM_THREADS(1) &
!$OMP&      DEFAULT(NONE) &
!$OMP&      PRIVATE(dsec) &
!$OMP&      SHARED(Atm, Land, Ice, Land_ice_atmos_boundary, Atmos_ice_boundary, Ocean_ice_boundary, Atmos_land_boundary) &
!$OMP&      SHARED(do_chksum, do_debug, omp_sec, num_atmos_calls, na, radiation_nthreads) &
!$OMP&      SHARED(newClockj)
!$          call omp_set_num_threads(radiation_nthreads)
!$          dsec=omp_get_wtime()
            call mpp_clock_begin(newClockj)
            call update_atmos_model_radiation( Land_ice_atmos_boundary, Atm )
            call mpp_clock_end(newClockj)
!$          omp_sec(2) = omp_sec(2) + (omp_get_wtime() - dsec)
!---CANNOT PUT AN MPP_CHKSUM HERE AS IT REQUIRES THE ABILITY TO HAVE TWO DIFFERENT OPENMP THREADS
!---INSIDE OF MPI AT THE SAME TIME WHICH IS NOT CURRENTLY ALLOWED
!           if (do_chksum) call atmos_ice_land_chksum('update_atmos_model_radiation(conc)', (nc-1)*num_atmos_calls+na, &
!                   Atm, Land, Ice, Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
            if (do_debug)  call print_memuse_stats( 'update concurrent rad')
!$OMP END PARALLEL
          endif
!$      endif
!$      imb_sec(omp_get_thread_num()+1) = imb_sec(omp_get_thread_num()+1) - omp_get_wtime()
!$OMP END PARALLEL
!$      imb_sec(1) = imb_sec(1) + omp_get_wtime()
!$      if (do_concurrent_radiation) imb_sec(2) = imb_sec(2) + omp_get_wtime()
!$      call omp_set_num_threads(atmos_nthreads+(conc_nthreads-1)*radiation_nthreads)

        call mpp_clock_begin(newClockk)
        call update_atmos_model_state( Atm )
        if (do_chksum) call atmos_ice_land_chksum('update_atmos_model_state+', (nc-1)*num_atmos_calls+na, Atm, Land, &
                  Ice,Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
        if (do_debug)  call print_memuse_stats( 'update state')
        call mpp_clock_end(newClockk)

      enddo ! end of na (fast loop)

      call mpp_clock_end(newClock7)

      call mpp_clock_begin(newClock8)
      !   ------ end of atmospheric time step loop -----
      if (do_land .AND. Land%pe) then
        if (land_npes .NE. atmos_npes) call mpp_set_current_pelist(Land%pelist)
        call update_land_model_slow(Atmos_land_boundary,Land)
      endif
      if (land_npes .NE. atmos_npes) call mpp_set_current_pelist(Atm%pelist)
      !-----------------------------------------------------------------------
      call mpp_clock_end(newClock8)
      if (do_chksum) call atmos_ice_land_chksum('update_land_slow+', nc, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)

      !
      !     need flux call to put runoff and p_surf on ice grid
      !
      call mpp_clock_begin(newClock9)
      call flux_land_to_ice( Time, Land, Ice, Land_ice_boundary )
      call mpp_clock_end(newClock9)
      if (do_chksum) call atmos_ice_land_chksum('fluxlnd2ice+', nc, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)

      Atmos_ice_boundary%p = 0.0 ! call flux_atmos_to_ice_slow ?
      Time = Time_atmos
      call mpp_clock_end(newClock5)
    endif                     !Atm%pe block

    if(Atm%pe) then
     call mpp_clock_begin(newClock5) !Ice is still using ATM pelist and need to be included in ATM clock
                                        !ATM clock is used for load-balancing the coupled models 
    endif
    if (do_ice .and. Ice%pe) then

      if (Ice%fast_ice_PE) then
           if (ice_npes .NE. atmos_npes) call mpp_set_current_pelist(Ice%fast_pelist)
        call mpp_clock_begin(newClock10f)
       ! These two calls occur on whichever PEs handle the fast ice processess.
        call ice_model_fast_cleanup(Ice)

        call unpack_land_ice_boundary(Ice, Land_ice_boundary)
        call mpp_clock_end(newClock10f)
      endif

      if (.not.concurrent_ice) then
        ! This could be a point where the model is serialized.
        if (.not.Ice%shared_slow_fast_PEs) call mpp_set_current_pelist(Ice%pelist)
        ! This call occurs all ice PEs.
        call mpp_clock_begin(newClock10e)
        call exchange_fast_to_slow_ice(Ice)
        call mpp_clock_end(newClock10e)
      endif

      !   ------ slow-ice model ------

      ! This call occurs on whichever PEs handle the slow ice processess.
      if (Ice%slow_ice_PE .and. .not.combined_ice_and_ocean) then
        if (slow_ice_with_ocean) call mpp_set_current_pelist(Ice%slow_pelist)
        call mpp_clock_begin(newClock10s)
        call update_ice_model_slow(Ice)

        call mpp_clock_begin(newClock11)
        call flux_ice_to_ocean_stocks(Ice)
        call mpp_clock_end(newClock11)
        call mpp_clock_end(newClock10s)
      endif

      if (do_chksum) call slow_ice_chksum('update_ice_slow+', nc, Ice, Ocean_ice_boundary)
     endif  ! End of Ice%pe block

     if(Atm%pe) then
        call mpp_set_current_pelist(Atm%pelist)
        call mpp_clock_end(newClock5)
     endif

    ! Update Ice_ocean_boundary using the newly calculated fluxes.
    if ((concurrent_ice .OR. .NOT.use_lag_fluxes) .and. .not.combined_ice_and_ocean) then
      !this could serialize unless slow_ice_with_ocean is true.
      if ((.not.do_ice) .or. (.not.slow_ice_with_ocean)) &
        call mpp_set_current_pelist()

      if (Ice%slow_ice_PE .or. Ocean%is_ocean_pe) then
        ! If the slow ice is on a subset of the ocean PEs, use the ocean PElist.
        call mpp_set_current_pelist(slow_ice_ocean_pelist)
        call mpp_clock_begin(newClock3)
        call flux_ice_to_ocean( Time, Ice, Ocean, Ice_ocean_boundary )
        Time_flux_ice_to_ocean = Time
        call mpp_clock_end(newClock3)
      endif
    endif

    if (Ocean%is_ocean_pe) then
      call mpp_set_current_pelist(Ocean%pelist)
      call mpp_clock_begin(newClock12)

      ! This may do data override or diagnostics on Ice_ocean_boundary.
      call flux_ice_to_ocean_finish(Time_flux_ice_to_ocean, Ice_ocean_boundary)

      if (combined_ice_and_ocean) then
        call flux_ice_to_ocean_stocks(Ice)
        call update_slow_ice_and_ocean(ice_ocean_driver_CS, Ice, Ocean_state, Ocean, &
                      Ice_ocean_boundary, Time_ocean, Time_step_cpld )
      else
      if (do_chksum) call ocean_chksum('update_ocean_model-', nc, Ocean, Ice_ocean_boundary)
      ! update_ocean_model since fluxes don't change here

      if (do_ocean) &
        call update_ocean_model( Ice_ocean_boundary, Ocean_state,  Ocean, &
                                 Time_ocean, Time_step_cpld )
      endif

      if (do_chksum) call ocean_chksum('update_ocean_model+', nc, Ocean, Ice_ocean_boundary)
      ! Get stocks from "Ice_ocean_boundary" and add them to Ocean stocks.
      ! This call is just for record keeping of stocks transfer and
      ! does not modify either Ocean or Ice_ocean_boundary
      call flux_ocean_from_ice_stocks(Ocean_state, Ocean, Ice_ocean_boundary)

      Time_ocean = Time_ocean +  Time_step_cpld
      Time = Time_ocean

      call mpp_clock_end(newClock12)
    endif

    !--- write out intermediate restart file when needed.
    if (Time >= Time_restart) then
      Time_restart_current = Time
      Time_restart = increment_date(Time, restart_interval(1), restart_interval(2), &
           restart_interval(3), restart_interval(4), restart_interval(5), restart_interval(6) )
      timestamp = date_to_string(time_restart_current)
      outunit= stdout()
      write(outunit,*) '=> NOTE from program coupler: intermediate restart file is written and ', &
           trim(timestamp),' is appended as prefix to each restart file name'
      if (Atm%pe) then
        call atmos_model_restart(Atm, timestamp)
        call land_model_restart(timestamp)
        call ice_model_restart(Ice, timestamp)
      endif
      if (Ocean%is_ocean_pe) then
        call ocean_model_restart(Ocean_state, timestamp)
      endif
      call coupler_restart(Time, Time_restart_current, timestamp)
    endif

    !--------------
    if (do_chksum) call coupler_chksum('MAIN_LOOP+', nc)
    write( text,'(a,i6)' )'Main loop at coupling timestep=', nc
    call print_memuse_stats(text)
    outunit= stdout()
    if (mpp_pe() == mpp_root_pe() .and. Atm%pe .and. do_concurrent_radiation) then
      write(outunit,102) 'At coupling step ', nc,' of ',num_cpld_calls, &
           ' Atm & Rad (imbalance): ',omp_sec(1),' (',imb_sec(1),')  ',omp_sec(2),' (',imb_sec(2),')'
    endif
    omp_sec(:)=0.
    imb_sec(:)=0.
    call flush(outunit)

  enddo
102 FORMAT(A17,i5,A4,i5,A24,f10.4,A2,f10.4,A3,f10.4,A2,f10.4,A1)

  call mpp_set_current_pelist()
  call mpp_clock_begin(newClock14)
  if (check_stocks >= 0) then
    call mpp_set_current_pelist()
    call flux_check_stocks(Time=Time, Atm=Atm, Lnd=Land, Ice=Ice, Ocn_state=Ocean_state)
  endif
  call mpp_clock_end(newClock14)

  call mpp_set_current_pelist()
!-----------------------------------------------------------------------
  call mpp_clock_end(mainClock)
  call mpp_clock_begin(termClock)

  if (do_chksum) call coupler_chksum('coupler_end-', nc)
  call coupler_end

  call mpp_clock_end(termClock)

  call print_memuse_stats( 'Memory HiWaterMark', always=.TRUE. )
  call fms_end

!-----------------------------------------------------------------------

contains

!#######################################################################

!> \brief Initialize all defined exchange grids and all boundary maps
  subroutine coupler_init

    use ensemble_manager_mod, only : ensemble_manager_init, get_ensemble_id,ensemble_pelist_setup
    use ensemble_manager_mod, only : get_ensemble_size, get_ensemble_pelist


!
!-----------------------------------------------------------------------
!     local parameters
!-----------------------------------------------------------------------
!

    character(len=64), parameter    :: sub_name = 'coupler_init'
    character(len=256), parameter   :: error_header =                               &
         '==>Error from ' // trim(mod_name) // '(' // trim(sub_name) // '):'
    character(len=256), parameter   :: note_header =                                &
         '==>Note from ' // trim(mod_name) // '(' // trim(sub_name) // '):'

    integer :: unit,  ierr, io,    m, i, outunit, logunit, errunit
    integer :: date(6)
    type (time_type) :: Run_length
    character(len=9) :: month
    integer :: pe, npes

    integer :: ens_siz(6), ensemble_size

    integer :: atmos_pe_start=0, atmos_pe_end=0, &
               ocean_pe_start=0, ocean_pe_end=0
    integer :: n
    integer :: diag_model_subset=DIAG_ALL
    logical :: other_fields_exist
    character(len=256) :: err_msg
    integer :: date_restart(6)
    character(len=64)  :: filename, fieldname
    integer :: id_restart, l
    integer :: omp_get_thread_num, omp_get_num_threads
    integer :: get_cpu_affinity, base_cpu, base_cpu_r, adder
    character(len=8)  :: walldate
    character(len=10) :: walltime
    character(len=5)  :: wallzone
    integer           :: wallvalues(8)

    type(coupler_1d_bc_type), pointer :: &
      gas_fields_atm => NULL(), &  ! A pointer to the type describing the
              ! atmospheric fields that will participate in the gas fluxes.
      gas_fields_ocn => NULL(), &  ! A pointer to the type describing the ocean
              ! and ice surface fields that will participate in the gas fluxes.
      gas_fluxes => NULL()   ! A pointer to the type describing the
              ! atmosphere-ocean gas and tracer fluxes.
!-----------------------------------------------------------------------

    outunit = stdout()
    errunit = stderr()
    logunit = stdlog()

    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Entering coupler_init at '&
                       //trim(walldate)//' '//trim(walltime)
    endif

!----- write version to logfile -------
    call write_version_number(version, tag)

!----- read namelist -------

#ifdef INTERNAL_FILE_NML
    read (input_nml_file, coupler_nml, iostat=io)
    ierr = check_nml_error (io, 'coupler_nml')
#else
    unit = open_namelist_file()
    ierr=1; do while (ierr /= 0)
      read (unit, nml=coupler_nml, iostat=io, end=10)
      ierr = check_nml_error (io, 'coupler_nml')
    enddo
10  call mpp_close(unit)
#endif

!---- when concurrent is set true and mpp_io_nml io_clock_on is set true, the model
!---- will crash with error message "MPP_CLOCK_BEGIN: cannot change pelist context of a clock",
!---- so need to make sure it will not happen
    if (concurrent) then
      if (mpp_io_clock_on()) then
        call error_mesg ('program coupler', 'when coupler_nml variable concurrent is set to true, '// &
             'mpp_io_nml variable io_clock_non can not be set to true.', FATAL )
      endif
    endif

!---- ncores_per_node must be set when use_hyper_thread = .true.
    if (use_hyper_thread .and. ncores_per_node == 0) then
      call error_mesg ('program copuler', 'coupler_nml ncores_per_node must be set when use_hyper_thread=true', FATAL)
    endif

!----- read date and calendar type from restart file -----

    if (file_exist('INPUT/coupler.res')) then
!Balaji: currently written in binary, needs form=MPP_NATIVE
      call mpp_open( unit, 'INPUT/coupler.res', action=MPP_RDONLY )
      read( unit,*,err=999 )calendar_type
      read( unit,* )date_init
      read( unit,* )date
      goto 998 !back to fortran-4
!read old-style coupler.res
999   call mpp_close(unit)
      call mpp_open( unit, 'INPUT/coupler.res', action=MPP_RDONLY, form=MPP_NATIVE )
      read(unit)calendar_type
      read(unit)date
998   call mpp_close(unit)
    else
      force_date_from_namelist = .true.
    endif

!----- use namelist value (either no restart or override flag on) ---

    if ( force_date_from_namelist ) then

      if ( sum(current_date) <= 0 ) then
        call error_mesg ('program coupler',  &
             'no namelist value for base_date or current_date', FATAL)
      else
        date      = current_date
      endif

!----- override calendar type with namelist value -----

      select case( uppercase(trim(calendar)) )
      case( 'JULIAN' )
        calendar_type = JULIAN
      case( 'NOLEAP' )
        calendar_type = NOLEAP
      case( 'THIRTY_DAY' )
        calendar_type = THIRTY_DAY_MONTHS
      case( 'NO_CALENDAR' )
        calendar_type = NO_CALENDAR
      end select

    endif

    call set_calendar_type (calendar_type, err_msg)
    if (err_msg /= '') then
      call mpp_error(FATAL, 'ERROR in coupler_init: '//trim(err_msg))
    endif

    if (concurrent .AND. .NOT.(use_lag_fluxes .OR. concurrent_ice) ) &
      call mpp_error( WARNING, 'coupler_init: you have set concurrent=TRUE, &
            & use_lag_fluxes=FALSE, and concurrent_ice=FALSE &
            & in coupler_nml. When not using lag fluxes, components &
            & will synchronize at two points, and thus run serially.' )
    if (concurrent_ice .AND. .NOT.slow_ice_with_ocean ) call mpp_error(WARNING, &
           'coupler_init: concurrent_ice is true, but slow ice_with_ocean is &
           & false in coupler_nml.  These two flags should both be true to avoid &
           & effectively serializing the run.' )
    if (use_lag_fluxes .AND. concurrent_ice ) call mpp_error(WARNING, &
           'coupler_init: use_lag_fluxes and concurrent_ice are both true. &
           & These two coupling options are intended to be exclusive.' )

    !Check with the ensemble_manager module for the size of ensemble
    !and PE counts for each member of the ensemble.
    !
    !NOTE: ensemble_manager_init renames all the output files (restart and diagnostics)
    !      to show which ensemble member they are coming from.
    !      There also need to be restart files for each member of the ensemble in INPUT.
    !
    !NOTE: if the ensemble_size=1 the input/output files will not be renamed.
    !

    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Starting initializing ensemble_manager at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    call ensemble_manager_init() ! init pelists for ensembles
    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Finished initializing ensemble_manager at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    ens_siz = get_ensemble_size()
    ensemble_size = ens_siz(1)
    npes = ens_siz(2)

    !Check for the consistency of PE counts
    if (concurrent) then
!atmos_npes + ocean_npes must equal npes
      if (atmos_npes.EQ.0 ) atmos_npes = npes - ocean_npes
      if (ocean_npes.EQ.0 ) ocean_npes = npes - atmos_npes
!both must now be non-zero
      if (atmos_npes.EQ.0 .OR. ocean_npes.EQ.0 ) &
        call mpp_error( FATAL, 'coupler_init: atmos_npes or ocean_npes must be specified for concurrent coupling.' )
      if (atmos_npes+ocean_npes.NE.npes ) &
        call mpp_error( FATAL, 'coupler_init: atmos_npes+ocean_npes must equal npes for concurrent coupling.' )
    else                        !serial timestepping
      if ((atmos_npes.EQ.0) .and. (do_atmos .or. do_land .or. do_ice)) atmos_npes = npes
      if ((ocean_npes.EQ.0) .and. (do_ocean)) ocean_npes = npes
      if (max(atmos_npes,ocean_npes).EQ.npes) then !overlapping pelists
        ! do nothing
      else                    !disjoint pelists
        if (atmos_npes+ocean_npes.NE.npes ) call mpp_error( FATAL,  &
             'coupler_init: atmos_npes+ocean_npes must equal npes for serial coupling on disjoint pelists.' )
      endif
    endif

    if (land_npes == 0 ) land_npes = atmos_npes
    if (land_npes > atmos_npes) call mpp_error(FATAL, 'coupler_init: land_npes > atmos_npes')

    if (ice_npes  == 0 ) ice_npes  = atmos_npes
    if (ice_npes  > atmos_npes) call mpp_error(FATAL, 'coupler_init: ice_npes > atmos_npes')

    allocate( Atm%pelist  (atmos_npes) )
    allocate( Ocean%pelist(ocean_npes) )
    allocate( Land%pelist (land_npes) )
    allocate( Ice%fast_pelist(ice_npes) )

    !Set up and declare all the needed pelists
    call ensemble_pelist_setup(concurrent, atmos_npes, ocean_npes, land_npes, ice_npes, &
                               Atm%pelist, Ocean%pelist, Land%pelist, Ice%fast_pelist)

!set up affinities based on threads

    ensemble_id = get_ensemble_id()

    allocate(ensemble_pelist(1:ensemble_size,1:npes))
    call get_ensemble_pelist(ensemble_pelist)

    Atm%pe            = ANY(Atm%pelist   .EQ. mpp_pe())
    Ocean%is_ocean_pe = ANY(Ocean%pelist .EQ. mpp_pe())
    Land%pe           = ANY(Land%pelist  .EQ. mpp_pe())

    Ice%shared_slow_fast_PEs = .not.slow_ice_with_ocean
    ! This is where different settings would be applied if the fast and slow
    ! ice occurred on different PEs.
    if (Ice%shared_slow_fast_PEs) then
      ! Fast and slow ice processes occur on the same PEs.
      allocate( Ice%pelist  (ice_npes) )
      Ice%pelist(:) = Ice%fast_pelist(:)
      allocate( Ice%slow_pelist(ice_npes) )
      Ice%slow_pelist(:) = Ice%fast_pelist(:)
      if(concurrent) then
         allocate(slow_ice_ocean_pelist(ocean_npes+ice_npes))
         slow_ice_ocean_pelist(1:ice_npes) = Ice%slow_pelist(:)
         slow_ice_ocean_pelist(ice_npes+1:ice_npes+ocean_npes) = Ocean%pelist(:)
    else
         if(ice_npes .GE. ocean_npes) then
            allocate(slow_ice_ocean_pelist(ice_npes))
            slow_ice_ocean_pelist(:) = Ice%slow_pelist(:)
         else
            allocate(slow_ice_ocean_pelist(ocean_npes))
            slow_ice_ocean_pelist(:) = Ocean%pelist(:)
         endif
      endif
    else
      ! Fast ice processes occur a subset of the atmospheric PEs, while
      ! slow ice processes occur on the ocean PEs.
      allocate( Ice%slow_pelist(ocean_npes) )
      Ice%slow_pelist(:) = Ocean%pelist(:)
      allocate( Ice%pelist  (ice_npes+ocean_npes) )
      ! Set Ice%pelist() to be the union of Ice%fast_pelist and Ice%slow_pelist.
      Ice%pelist(1:ice_npes) = Ice%fast_pelist(:)
      Ice%pelist(ice_npes+1:ice_npes+ocean_npes) = Ocean%pelist(:)
      allocate(slow_ice_ocean_pelist(ocean_npes))
      slow_ice_ocean_pelist(:) = Ocean%pelist(:)
    endif

    Ice%fast_ice_pe = ANY(Ice%fast_pelist(:) .EQ. mpp_pe())
    Ice%slow_ice_pe = ANY(Ice%slow_pelist(:) .EQ. mpp_pe())
    Ice%pe = Ice%fast_ice_pe .OR. Ice%slow_ice_pe
    call mpp_declare_pelist(slow_ice_ocean_pelist)
    !Why is the following needed?
!$  call omp_set_dynamic(.FALSE.)
!$  call omp_set_nested(.TRUE.)
    if (Atm%pe) then
      call mpp_set_current_pelist( Atm%pelist )
!$    if (.not.do_concurrent_radiation) radiation_nthreads=atmos_nthreads
!$    if (do_concurrent_radiation) conc_nthreads=2
!$    call omp_set_num_threads(conc_nthreads)
!$    base_cpu = get_cpu_affinity()
!$OMP PARALLEL
!$    if (omp_get_thread_num() == 0) then
!$      call omp_set_num_threads(atmos_nthreads)
!$OMP PARALLEL private(adder)      !!!!atmos_nthreads nested parallel
!$      if (use_hyper_thread) then
!$        if (mod(omp_get_thread_num(),2) == 0) then
!$          adder = omp_get_thread_num()/2
!$        else
!$          adder = ncores_per_node + omp_get_thread_num()/2
!$        endif
!$      else
!$        adder = omp_get_thread_num()
!$      endif
!$      call set_cpu_affinity (base_cpu + adder)
!$      if (debug_affinity) then
!$        write(6,*) " atmos  ", get_cpu_affinity(), adder, omp_get_thread_num()
!$        call flush(6)
!$      endif
!$OMP END PARALLEL   !!!!end atmos_nthreads nested parallel
!$    endif
!$    if (omp_get_thread_num() == 1) then
!$      call omp_set_num_threads(radiation_nthreads)
!$      if (use_hyper_thread) then
!$        base_cpu_r = atmos_nthreads/2 + mod(atmos_nthreads,2)
!$      else
!$        base_cpu_r = atmos_nthreads
!$      endif
!$OMP PARALLEL private(adder)       !!!!radiation_nthreads nested parallel
!$      if (use_hyper_thread) then
!$        if (mod(omp_get_thread_num()+mod(atmos_nthreads,2),2) == 0) then
!$          adder = base_cpu_r + omp_get_thread_num()/2
!$        else
!$          adder = base_cpu_r + ncores_per_node + omp_get_thread_num()/2 - mod(atmos_nthreads,2)
!$        endif
!$      else
!$        adder = base_cpu_r + omp_get_thread_num()
!$      endif
!$      call set_cpu_affinity (base_cpu + adder)
!$      if (debug_affinity) then
!$        write(6,*) " rad    ", get_cpu_affinity(), base_cpu_r, adder, omp_get_thread_num()
!$        call flush(6)
!$      endif
!$OMP END PARALLEL   !!!!end radiation_nthreads nested parallel
!$    endif
!$OMP END PARALLEL
!$    call omp_set_num_threads(atmos_nthreads)
    endif

   !--- initialization clock
    if (Atm%pe) then
      call mpp_set_current_pelist(Atm%pelist)
      id_atmos_model_init = mpp_clock_id( '  Init: atmos_model_init ' )
    endif
    if (Land%pe) then
      call mpp_set_current_pelist(Land%pelist)
      id_land_model_init  = mpp_clock_id( '  Init: land_model_init ' )
    endif
    if (Ice%pe) then
      if (Ice%shared_slow_fast_PEs) then
        call mpp_set_current_pelist(Ice%pelist)
      elseif (Ice%fast_ice_pe) then
        call mpp_set_current_pelist(Ice%fast_pelist)
      elseif (Ice%slow_ice_pe) then
        call mpp_set_current_pelist(Ice%slow_pelist)
      else
        call mpp_error(FATAL, "All Ice%pes must be a part of Ice%fast_ice_pe or Ice%slow_ice_pe")
      endif
      id_ice_model_init   = mpp_clock_id( '  Init: ice_model_init ' )
    endif
    if (Ocean%is_ocean_pe) then
      call mpp_set_current_pelist(Ocean%pelist)
      id_ocean_model_init = mpp_clock_id( '  Init: ocean_model_init ' )
    endif
    call mpp_set_current_pelist(ensemble_pelist(ensemble_id,:))
    id_flux_exchange_init = mpp_clock_id( '  Init: flux_exchange_init' )

    call mpp_set_current_pelist()
    mainClock = mpp_clock_id( 'Main loop' )
    termClock = mpp_clock_id( 'Termination' )

    !Write out messages on root PEs
    if (mpp_pe().EQ.mpp_root_pe()) then
      write( text,'(a,2i6,a,i2.2)' )'Atmos PE range: ', Atm%pelist(1)  , Atm%pelist(atmos_npes)  ,&
           ' ens_', ensemble_id
      call mpp_error( NOTE, 'coupler_init: '//trim(text) )
      if (ocean_npes .gt. 0) then
        write( text,'(a,2i6,a,i2.2)' )'Ocean PE range: ', Ocean%pelist(1), Ocean%pelist(ocean_npes), &
             ' ens_', ensemble_id
        call mpp_error( NOTE, 'coupler_init: '//trim(text) )
      else
        write( text,'(a,i2.2)' )'Ocean PE range is not set (do_ocean=.false. and concurrent=.false.) for ens_', &
              ensemble_id
        call mpp_error( NOTE, 'coupler_init: '//trim(text) )
      endif
      write( text,'(a,2i6,a,i2.2)' )'Land PE range: ', Land%pelist(1)  , Land%pelist(land_npes)  ,&
           ' ens_', ensemble_id
      call mpp_error( NOTE, 'coupler_init: '//trim(text) )
      write( text,'(a,2i6,a,i2.2)' )'Ice PE range: ', Ice%pelist(1), Ice%pelist(ice_npes), &
           ' ens_', ensemble_id
      call mpp_error( NOTE, 'coupler_init: '//trim(text) )

      if (concurrent) then
        call mpp_error( NOTE, 'coupler_init: Running with CONCURRENT coupling.' )

        write( logunit,'(a)' )'Using concurrent coupling...'
        write( logunit,'(a,4i6)' ) &
              'atmos_pe_start, atmos_pe_end, ocean_pe_start, ocean_pe_end=', &
              Atm%pelist(1)  , Atm%pelist(atmos_npes), Ocean%pelist(1), Ocean%pelist(ocean_npes)
      else
        call mpp_error( NOTE, 'coupler_init: Running with SERIAL coupling.' )
      endif
      if (use_lag_fluxes) then
        call mpp_error( NOTE, 'coupler_init: Sending LAG fluxes to ocean.' )
      else
        call mpp_error( NOTE, 'coupler_init: Sending most recent fluxes to ocean.' )
      endif
      if (concurrent_ice) call mpp_error( NOTE, &
        'coupler_init: using lagged slow-ice coupling mode.')
      if (combined_ice_and_ocean) call mpp_error( NOTE, &
        'coupler_init: advancing the ocean and slow-ice in a single call.')
      if (combined_ice_and_ocean .and. .not.concurrent_ice) call mpp_error( FATAL, &
        'coupler_init: concurrent_ice must be true if combined_ice_and_ocean is true.')
      if (combined_ice_and_ocean .and. .not.slow_ice_with_ocean) call mpp_error( FATAL, &
        'coupler_init: slow_ice_with_ocean must be true if combined_ice_and_ocean is true.')
    endif

!----- write namelist to logfile -----
    if (mpp_pe() == mpp_root_pe() )write( logunit, nml=coupler_nml )

!----- write current/initial date actually used to logfile file -----

    if ( mpp_pe().EQ.mpp_root_pe() ) &
      write( logunit, 16 )date(1),trim(month_name(date(2))),date(3:6)
16  format ('  current date used = ',i4,1x,a,2i3,2(':',i2.2),' gmt')

!-----------------------------------------------------------------------
!------ initialize diagnostics manager ------

!jwd Fork here is somewhat dangerous. It relies on "no side effects" from
!    diag_manager_init. diag_manager_init or this section should be
!    re-architected to guarantee this or remove this assumption.
!    For instance, what follows assumes that get_base_date has the same
!    time for both Atm and Ocean pes. While this should be the case, the
!    possible error condition needs to be checked

    diag_model_subset=DIAG_ALL
    if (Atm%pe) then
      call mpp_set_current_pelist(Atm%pelist)
      if (atmos_npes /= npes) diag_model_subset = DIAG_OTHER  ! change diag_model_subset from DIAG_ALL
    elseif (Ocean%is_ocean_pe) then  ! Error check above for disjoint pelists should catch any problem
      call mpp_set_current_pelist(Ocean%pelist)
      ! The FMS diag manager has a convention that segregates files with "ocean"
      ! in their names from the other files to handle long diag tables.  This
      ! does not work if the ice is on the ocean PEs.
      if ((ocean_npes /= npes) .and. .not.slow_ice_with_ocean) &
        diag_model_subset = DIAG_OCEAN  ! change diag_model_subset from DIAG_ALL
    endif
    if ( mpp_pe() == mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Starting to initialize diag_manager at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    ! initialize diag_manager for processor subset output
    call diag_manager_init(DIAG_MODEL_SUBSET=diag_model_subset, TIME_INIT=date)
    call print_memuse_stats( 'diag_manager_init' )
    if ( mpp_pe() == mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Finished initializing diag_manager at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
!-----------------------------------------------------------------------
!------ reset pelist to "full group" ------

    call mpp_set_current_pelist()
!----- always override initial/base date with diag_manager value -----

    call get_base_date ( date_init(1), date_init(2), date_init(3), &
         date_init(4), date_init(5), date_init(6)  )

!----- use current date if no base date ------

    if ( date_init(1) == 0 ) date_init = date

!----- set initial and current time types ------

    Time_init = set_date (date_init(1), date_init(2), date_init(3), &
         date_init(4), date_init(5), date_init(6))

    Time      = set_date (date(1), date(2), date(3),  &
         date(4), date(5), date(6))

    Time_start = Time

!----- compute the ending time -----

    Time_end = Time
    do m=1,months
       Time_end = Time_end + set_time(0,days_in_month(Time_end))
    enddo
    Time_end   = Time_end + set_time(hours*3600+minutes*60+seconds, days)
    !Need to pass Time_end into diag_manager for multiple thread case.
    call diag_manager_set_time_end(Time_end)

    Run_length = Time_end - Time

!--- get the time that last intermediate restart file was written out.
    if (file_exist('INPUT/coupler.intermediate.res')) then
       call mpp_open(unit,'INPUT/coupler.intermediate.res',action=MPP_RDONLY)
       read(unit,*) date_restart
       call mpp_close(unit)
    else
       date_restart = date
    endif

    Time_restart_current = Time
    if (ALL(restart_interval ==0)) then
       Time_restart = increment_date(Time_end, 0, 0, 10, 0, 0, 0)   ! no intermediate restart
    else
       Time_restart = set_date(date_restart(1), date_restart(2), date_restart(3),  &
                               date_restart(4), date_restart(5), date_restart(6) )
       Time_restart = increment_date(Time_restart, restart_interval(1), restart_interval(2), &
            restart_interval(3), restart_interval(4), restart_interval(5), restart_interval(6) )
       if (Time_restart <= Time) call mpp_error(FATAL, &
            '==>Error from program coupler: The first intermediate restart time is no larger than the start time')
    endif

!-----------------------------------------------------------------------
!----- write time stamps (for start time and end time) ------

    call mpp_open( unit, 'time_stamp.out', nohdrs=.TRUE. )

    month = month_name(date(2))
    if ( mpp_pe().EQ.mpp_root_pe() ) write (unit,20) date, month(1:3)

    call get_date (Time_end, date(1), date(2), date(3),  &
                   date(4), date(5), date(6))
    month = month_name(date(2))
    if ( mpp_pe().EQ.mpp_root_pe() ) write (unit,20) date, month(1:3)

    call mpp_close(unit)

20  format (i6,5i4,2x,a3)

!-----------------------------------------------------------------------
!----- compute the time steps ------

    Time_step_cpld  = set_time (dt_cpld ,0)
    Time_step_atmos = set_time (dt_atmos,0)

!----- determine maximum number of iterations per loop ------

    num_cpld_calls  = Run_length      / Time_step_cpld
    num_atmos_calls = Time_step_cpld  / Time_step_atmos

!-----------------------------------------------------------------------
!------------------- some error checks ---------------------------------

!----- initial time cannot be greater than current time -------

    if ( Time_init > Time ) call error_mesg ('program coupler',  &
         'initial time is greater than current time', FATAL)

!----- make sure run length is a multiple of ocean time step ------

    if ( num_cpld_calls * Time_step_cpld  /= Run_length )  &
      call error_mesg ('program coupler',  &
         'run length must be multiple of coupled time step', FATAL)

! ---- make sure cpld time step is a multiple of atmos time step ----

    if ( num_atmos_calls * Time_step_atmos /= Time_step_cpld )  &
      call error_mesg ('program coupler',   &
         'cpld time step is not a multiple of the atmos time step', FATAL)

!
!       Initialize the tracer manager. This needs to be done on all PEs,
!       before the individual models are initialized.
!

    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Starting to initialize tracer_manager at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    call tracer_manager_init()
!   Initialize the gas-exchange fluxes so this information can be made
!   available to the individual components.
    call gas_exchange_init(gas_fields_atm, gas_fields_ocn, gas_fluxes)
    call coupler_types_init()
    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Finished initializing tracer_manager at '&
                       //trim(walldate)//' '//trim(walltime)
    endif



!-----------------------------------------------------------------------
!------ initialize component models ------
!------ grid info now comes from grid_spec file

    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Beginning to initialize component models at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    if (Atm%pe) then
        call mpp_set_current_pelist(Atm%pelist)
!---- atmosphere ----
        if (mpp_pe().EQ.mpp_root_pe()) then
          call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
          write(errunit,*) 'Starting to initialize atmospheric model at '&
                           //trim(walldate)//' '//trim(walltime)
        endif

        call mpp_clock_begin(id_atmos_model_init)

        call atmos_model_init( Atm, Time_init, Time, Time_step_atmos, &
                               do_concurrent_radiation)

        call mpp_clock_end(id_atmos_model_init)

        if (mpp_pe().EQ.mpp_root_pe()) then
          call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
          write(errunit,*) 'Finished initializing atmospheric model at '&
                           //trim(walldate)//' '//trim(walltime)
        endif
        call print_memuse_stats( 'atmos_model_init' )
        call data_override_init(Atm_domain_in = Atm%domain)
    endif
!---- land ----------
    if (Land%pe) then
      call mpp_set_current_pelist(Land%pelist)
      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Starting to initialize land model at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call mpp_clock_begin(id_land_model_init)
      call land_model_init( Atmos_land_boundary, Land, Time_init, Time, &
                            Time_step_atmos, Time_step_cpld )
      call mpp_clock_end(id_land_model_init)
      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Finished initializing land model at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call print_memuse_stats( 'land_model_init' )
      call data_override_init(Land_domain_in = Land%domain)
#ifndef _USE_LEGACY_LAND_
      call data_override_init(Land_domainUG_in = Land%ug_domain)
#endif
    endif
!---- ice -----------
    if (Ice%pe) then  ! This occurs for all fast or slow ice PEs.
      if (Ice%fast_ice_pe) then
        call mpp_set_current_pelist(Ice%fast_pelist)
      elseif (Ice%slow_ice_pe) then
        call mpp_set_current_pelist(Ice%slow_pelist)
      else
        call mpp_error(FATAL, "All Ice%pes must be a part of Ice%fast_ice_pe or Ice%slow_ice_pe")
      endif
      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Starting to initialize ice model at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call mpp_clock_begin(id_ice_model_init)
      call ice_model_init(Ice, Time_init, Time, Time_step_atmos, &
                           Time_step_cpld, Verona_coupler=.false., &
                          concurrent_ice=concurrent_ice, &
                          gas_fluxes=gas_fluxes, gas_fields_ocn=gas_fields_ocn )
      call mpp_clock_end(id_ice_model_init)

      ! This must be called using the union of the ice PE_lists.
      call mpp_set_current_pelist(Ice%pelist)
      call share_ice_domains(Ice)

      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Finished initializing ice model at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call print_memuse_stats( 'ice_model_init' )
      if (Ice%fast_ice_pe) then
        call mpp_set_current_pelist(Ice%fast_pelist)
        call data_override_init(Ice_domain_in = Ice%domain)
      endif
    endif

!---- ocean ---------
    if (Ocean%is_ocean_pe) then
      call mpp_set_current_pelist(Ocean%pelist)
      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Starting to initialize ocean model at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call mpp_clock_begin(id_ocean_model_init)
      call ocean_model_init( Ocean, Ocean_state, Time_init, Time, &
                             gas_fields_ocn=gas_fields_ocn  )
      call mpp_clock_end(id_ocean_model_init)

      if (concurrent) then
!$      call omp_set_num_threads(ocean_nthreads)
        call mpp_set_current_pelist( Ocean%pelist )
!$      base_cpu = get_cpu_affinity()
!$OMP PARALLEL private(adder)    
!$      if (use_hyper_thread) then
!$        if (mod(omp_get_thread_num(),2) == 0) then
!$          adder = omp_get_thread_num()/2
!$        else
!$          adder = ncores_per_node + omp_get_thread_num()/2
!$        endif
!$      else
!$        adder = omp_get_thread_num()
!$      endif
!$      call set_cpu_affinity (base_cpu + adder)
!$      if (debug_affinity) then
!$        write(6,*) " ocean  ", get_cpu_affinity(), adder, omp_get_thread_num()
!$        call flush(6)
!$      endif
!$OMP END PARALLEL  
      else
        ocean_nthreads = atmos_nthreads
!$      call omp_set_num_threads(ocean_nthreads)
      endif

      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Finished initializing ocean model at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call print_memuse_stats( 'ocean_model_init' )
      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Starting to initialize data_override at '&
                         //trim(walldate)//' '//trim(walltime)
      endif
      call data_override_init(Ocean_domain_in = Ocean%domain )
      if (mpp_pe().EQ.mpp_root_pe()) then
        call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
        write(errunit,*) 'Finished initializing data_override at '&
                         //trim(walldate)//' '//trim(walltime)
      endif

      if (combined_ice_and_ocean) &
        call ice_ocean_driver_init(ice_ocean_driver_CS, Time_init, Time)

    endif ! end of Ocean%is_ocean_pe

!---------------------------------------------
    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Finished initializing component models at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    call mpp_set_current_pelist(ensemble_pelist(ensemble_id,:))

    call mpp_broadcast_domain(Ice%domain)
    call mpp_broadcast_domain(Ice%slow_domain_NH)
    call mpp_broadcast_domain(Ocean%domain)
!-----------------------------------------------------------------------
!---- initialize flux exchange module ----
    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Starting to initialize flux_exchange at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
    call mpp_clock_begin(id_flux_exchange_init)
    call flux_exchange_init ( Time, Atm, Land, Ice, Ocean, Ocean_state,&
             atmos_ice_boundary, land_ice_atmos_boundary, &
             land_ice_boundary, ice_ocean_boundary, ocean_ice_boundary, &
         do_ocean, slow_ice_ocean_pelist, dt_atmos=dt_atmos, dt_cpld=dt_cpld)
    call mpp_set_current_pelist(ensemble_pelist(ensemble_id,:))
    call mpp_clock_end(id_flux_exchange_init)
    call mpp_set_current_pelist()
    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Finsihed initializing flux_exchange at '&
                       //trim(walldate)//' '//trim(walltime)
    endif

    Time_atmos = Time
    Time_ocean = Time

!
!       read in extra fields for the air-sea gas fluxes
!
    if ( Ice%slow_ice_pe ) then
      call mpp_set_current_pelist(Ice%slow_pelist)

      call coupler_type_register_restarts(Ice%ocean_fluxes, Ice_bc_restart, &
               num_ice_bc_restart, Ice%slow_domain_NH, ocean_restart=.false.)

      ! Restore the fields from the restart files
        do l = 1, num_ice_bc_restart
        call restore_state(Ice_bc_restart(l), directory='INPUT', &
                           nonfatal_missing_files=.true.)
        enddo

      ! Check whether the restarts were read successfully.
      call coupler_type_restore_state(Ice%ocean_fluxes, directory='INPUT', &
                                      test_by_field=.true.)
        endif

    if ( Ocean%is_ocean_pe ) then
      call mpp_set_current_pelist(Ocean%pelist)

      call coupler_type_register_restarts(Ocean%fields, Ocn_bc_restart, &
               num_ocn_bc_restart, Ocean%domain, ocean_restart=.true.)

      ! Restore the fields from the restart files
        do l = 1, num_ocn_bc_restart
        call restore_state(Ocn_bc_restart(l), directory='INPUT', &
                           nonfatal_missing_files=.true.)
        enddo

      ! Check whether the restarts were read successfully.
      call coupler_type_restore_state(Ocean%fields, directory='INPUT', &
                                      test_by_field=.true.)
        endif

    call mpp_set_current_pelist()

!-----------------------------------------------------------------------
!---- open and close dummy file in restart dir to check if dir exists --

    call mpp_open( unit, 'RESTART/file' )
    call mpp_close(unit, MPP_DELETE)

    ! Call to daig_grid_end to free up memory used during regional
    ! output setup
    CALL diag_grid_end()

!-----------------------------------------------------------------------
    if ( do_endpoint_chksum ) then
      if (Atm%pe) then
        call mpp_set_current_pelist(Atm%pelist)
        call atmos_ice_land_chksum('coupler_init+', 0, Atm, Land, Ice, &
                     Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
      endif
      if (Ice%slow_ice_PE) then
        call mpp_set_current_pelist(Ice%slow_pelist)
        call slow_ice_chksum('coupler_init+', 0, Ice, Ocean_ice_boundary)
      endif
      if (Ocean%is_ocean_pe) then
        call mpp_set_current_pelist(Ocean%pelist)
        call ocean_chksum('coupler_init+', 0, Ocean, Ice_ocean_boundary)
      endif
    endif

    call mpp_set_current_pelist()
    call print_memuse_stats('coupler_init')

    if (mpp_pe().EQ.mpp_root_pe()) then
      call DATE_AND_TIME(walldate, walltime, wallzone, wallvalues)
      write(errunit,*) 'Exiting coupler_init at '&
                       //trim(walldate)//' '//trim(walltime)
    endif
  end subroutine coupler_init

!#######################################################################

  subroutine coupler_end()

!-----------------------------------------------------------------------

    if ( do_endpoint_chksum ) then
      if (Atm%pe) then
        call mpp_set_current_pelist(Atm%pelist)
        call atmos_ice_land_chksum('coupler_end', 0, Atm, Land, Ice, &
                 Land_ice_atmos_boundary, Atmos_ice_boundary, Atmos_land_boundary)
      endif
      if (Ice%slow_ice_PE) then
        call mpp_set_current_pelist(Ice%slow_pelist)
        call slow_ice_chksum('coupler_end', 0, Ice, Ocean_ice_boundary)
      endif
      if (Ocean%is_ocean_pe) then
        call mpp_set_current_pelist(Ocean%pelist)
        call ocean_chksum('coupler_end', 0, Ocean, Ice_ocean_boundary)
      endif
    endif
    call mpp_set_current_pelist()

!----- check time versus expected ending time ----

    if (Time /= Time_end) call error_mesg ('program coupler',  &
         'final time does not match expected ending time', WARNING)

!-----------------------------------------------------------------------
!the call to fms_io_exit has been moved here
!this will work for serial code or concurrent (disjoint pelists)
!but will fail on overlapping but unequal pelists
    if (Ocean%is_ocean_pe) then
      call mpp_set_current_pelist(Ocean%pelist)
      call ocean_model_end (Ocean, Ocean_state, Time)
    endif
    if (Atm%pe) then
      call mpp_set_current_pelist(Atm%pelist)
      call atmos_model_end ( Atm )
    endif
    if (Land%pe) then
      call mpp_set_current_pelist(Land%pelist)
      call land_model_end (Atmos_land_boundary, Land)
    endif
    if (Ice%pe) then  ! This happens on all fast or slow ice PEs.
      if (Ice%slow_ice_PE) then
        call mpp_set_current_pelist(Ice%slow_pelist)
      else ! This must be a fast ice PE.
        call mpp_set_current_pelist(Ice%fast_pelist)
      endif
      call ice_model_end (Ice)
    endif

    !----- write restart file ------
    call coupler_restart(Time, Time_restart_current)

    call diag_manager_end (Time)
    call fms_io_exit
    call mpp_set_current_pelist()

!-----------------------------------------------------------------------

  end subroutine coupler_end

  !> \brief Writing restart file that contains running time and restart file writing time.
  subroutine coupler_restart(Time_run, Time_res, time_stamp)
    type(time_type),   intent(in)           :: Time_run, Time_res
    character(len=*), intent(in),  optional :: time_stamp
    character(len=128)                      :: file_run, file_res
    integer :: yr, mon, day, hr, min, sec, date(6), unit, n

    call mpp_set_current_pelist()

    ! write restart file
    if (present(time_stamp)) then
      file_run = 'RESTART/'//trim(time_stamp)//'.coupler.res'
      file_res = 'RESTART/'//trim(time_stamp)//'.coupler.intermediate.res'
    else
      file_run = 'RESTART/coupler.res'
      file_res = 'RESTART/coupler.intermediate.res'
    endif

    !----- compute current date ------
    call get_date (Time_run, date(1), date(2), date(3),  &
                   date(4), date(5), date(6))
    call mpp_open( unit, file_run, nohdrs=.TRUE. )
    if ( mpp_pe().EQ.mpp_root_pe()) then
       write( unit, '(i6,8x,a)' )calendar_type, &
            '(Calendar: no_calendar=0, thirty_day_months=1, julian=2, gregorian=3, noleap=4)'

       write( unit, '(6i6,8x,a)' )date_init, &
            'Model start time:   year, month, day, hour, minute, second'
       write( unit, '(6i6,8x,a)' )date, &
            'Current model time: year, month, day, hour, minute, second'
    endif
    call mpp_close(unit)

    if (Time_res > Time_start) then
      call mpp_open( unit, file_res, nohdrs=.TRUE. )
      if ( mpp_pe().EQ.mpp_root_pe()) then
        call get_date(Time_res ,yr,mon,day,hr,min,sec)
        write( unit, '(6i6,8x,a)' )yr,mon,day,hr,min,sec, &
             'Current intermediate restart time: year, month, day, hour, minute, second'
      endif
      call mpp_close(unit)
    endif

    if (Ocean%is_ocean_pe) then
      call mpp_set_current_pelist(Ocean%pelist)
      do n = 1, num_ocn_bc_restart
        call save_restart(Ocn_bc_restart(n), time_stamp)
      enddo
    endif
    if (Atm%pe) then
      call mpp_set_current_pelist(Atm%pelist)
      do n = 1, num_ice_bc_restart
        call save_restart(Ice_bc_restart(n), time_stamp)
      enddo
    endif

  end subroutine coupler_restart

!--------------------------------------------------------------------------

!> \brief Print out checksums for several atm, land and ice variables
  subroutine coupler_chksum(id, timestep)

    character(len=*), intent(in) :: id
    integer         , intent(in) :: timestep

    type :: tracer_ind_type
      integer :: atm, ice, lnd ! indices of the tracer in the respective models
    end type tracer_ind_type
    integer                            :: n_atm_tr, n_lnd_tr, n_exch_tr
    integer                            :: n_atm_tr_tot, n_lnd_tr_tot
    integer                            :: i, tr, n, m, outunit
    type(tracer_ind_type), allocatable :: tr_table(:)
    character(32) :: tr_name

    call get_number_tracers (MODEL_ATMOS, num_tracers=n_atm_tr_tot, &
                             num_prog=n_atm_tr)
    call get_number_tracers (MODEL_LAND, num_tracers=n_lnd_tr_tot, &
                             num_prog=n_lnd_tr)

    ! Assemble the table of tracer number translation by matching names of
    ! prognostic tracers in the atmosphere and surface models; skip all atmos.
    ! tracers that have no corresponding surface tracers.
    allocate(tr_table(n_atm_tr))
    n = 1
    do i = 1,n_atm_tr
      call get_tracer_names( MODEL_ATMOS, i, tr_name )
      tr_table(n)%atm = i
      tr_table(n)%ice = get_tracer_index ( MODEL_ICE,  tr_name )
      tr_table(n)%lnd = get_tracer_index ( MODEL_LAND, tr_name )
      if (tr_table(n)%ice/=NO_TRACER .or. tr_table(n)%lnd/=NO_TRACER) n = n+1
    enddo
    n_exch_tr = n-1

100 FORMAT("CHECKSUM::",A32," = ",Z20)
101 FORMAT("CHECKSUM::",A16,a,'%',a," = ",Z20)

    if (Atm%pe) then
      call mpp_set_current_pelist(Atm%pelist)

      outunit = stdout()
      write(outunit,*) 'BEGIN CHECKSUM(Atm):: ', id, timestep
      write(outunit,100) 'atm%t_bot', mpp_chksum(atm%t_bot)
      write(outunit,100) 'atm%z_bot', mpp_chksum(atm%z_bot)
      write(outunit,100) 'atm%p_bot', mpp_chksum(atm%p_bot)
      write(outunit,100) 'atm%u_bot', mpp_chksum(atm%u_bot)
      write(outunit,100) 'atm%v_bot', mpp_chksum(atm%v_bot)
      write(outunit,100) 'atm%p_surf', mpp_chksum(atm%p_surf)
      write(outunit,100) 'atm%gust', mpp_chksum(atm%gust)
      do tr = 1,n_exch_tr
         n = tr_table(tr)%atm
         if (n /= NO_TRACER) then
            call get_tracer_names( MODEL_ATMOS, tr_table(tr)%atm, tr_name )
            write(outunit,100) 'atm%'//trim(tr_name), mpp_chksum(Atm%tr_bot(:,:,n))
         endif
      enddo

      write(outunit,100) 'land%t_surf', mpp_chksum(land%t_surf)
      write(outunit,100) 'land%t_ca', mpp_chksum(land%t_ca)
      write(outunit,100) 'land%rough_mom', mpp_chksum(land%rough_mom)
      write(outunit,100) 'land%rough_heat', mpp_chksum(land%rough_heat)
      write(outunit,100) 'land%rough_scale', mpp_chksum(land%rough_scale)
      do tr = 1,n_exch_tr
        n = tr_table(tr)%lnd
        if (n /= NO_TRACER) then
          call get_tracer_names( MODEL_ATMOS, tr_table(tr)%atm, tr_name )
#ifndef _USE_LEGACY_LAND_
          write(outunit,100) 'land%'//trim(tr_name), mpp_chksum(Land%tr(:,:,n))
#else
          write(outunit,100) 'land%'//trim(tr_name), mpp_chksum(Land%tr(:,:,:,n))
#endif
        endif
      enddo

      write(outunit,100) 'ice%t_surf', mpp_chksum(ice%t_surf)
      write(outunit,100) 'ice%rough_mom', mpp_chksum(ice%rough_mom)
      write(outunit,100) 'ice%rough_heat', mpp_chksum(ice%rough_heat)
      write(outunit,100) 'ice%rough_moist', mpp_chksum(ice%rough_moist)
      write(outunit,*) 'STOP CHECKSUM(Atm):: ', id, timestep

    !endif

    !if (Ocean%is_ocean_pe) then
       !call mpp_set_current_pelist(Ocean%pelist)

      write(outunit,*) 'BEGIN CHECKSUM(Ice):: ', id, timestep
      call coupler_type_write_chksums(Ice%ocean_fields, outunit, 'ice%')
      write(outunit,*) 'STOP CHECKSUM(Ice):: ', id, timestep

    endif

    deallocate(tr_table)

    call mpp_set_current_pelist()

  end subroutine coupler_chksum

  !#######################################################################

!> \brief This subroutine calls subroutine that will print out checksums of the elements
!! of the appropriate type.
!!
!! For coupled models typically these types are not defined on all processors.
!! It is assumed that the appropriate pelist has been set before entering this routine.
!! This can be achieved in the following way.
!! ~~~~~~~~~~{.f90}
!! if (Atm%pe) then
!!    call mpp_set_current_pelist(Atm%pelist)
!!    call atmos_ice_land_chksum('MAIN_LOOP-', nc)
!! endif
!! ~~~~~~~~~~
!! If you are on the global pelist before you enter this routine using the above call,
!! you can return to the global pelist by invoking
!! ~~~~~~~~~~{.f90}
!! call mpp_set_current_pelist()
!! ~~~~~~~~~~
!! after you exit. This is only necessary if you need to return to the global pelist.
  subroutine atmos_ice_land_chksum(id, timestep, Atm, Land, Ice, &
                   Land_ice_atmos_boundary, Atmos_ice_boundary, &
                   Atmos_land_boundary)

    character(len=*), intent(in) :: id
    integer         , intent(in) :: timestep
    type (atmos_data_type), intent(in) :: Atm
    type  (land_data_type), intent(in) :: Land
    type   (ice_data_type), intent(in) :: Ice
    type(land_ice_atmos_boundary_type), intent(in) :: Land_ice_atmos_boundary
    type(atmos_ice_boundary_type), intent(in)      :: Atmos_ice_boundary
    type(atmos_land_boundary_type), intent(in)     :: Atmos_land_boundary

    call atmos_data_type_chksum(     id, timestep, Atm)
    call lnd_ice_atm_bnd_type_chksum(id, timestep, Land_ice_atmos_boundary)

    if (Ice%fast_ice_pe) then
      call mpp_set_current_pelist(Ice%fast_pelist)
      call ice_data_type_chksum(   id, timestep, Ice)
      call atm_ice_bnd_type_chksum(id, timestep, Atmos_ice_boundary)
    endif
    if (Land%pe) then
      call mpp_set_current_pelist(Land%pelist)
      call land_data_type_chksum(  id, timestep, Land)
      call atm_lnd_bnd_type_chksum(id, timestep, Atmos_land_boundary)
    endif

    call mpp_set_current_pelist(Atm%pelist)

  end subroutine atmos_ice_land_chksum

!> \brief This subroutine calls subroutine that will print out checksums of the elements
!! of the appropriate type.
!!
!! For coupled models typically these types are not defined on all processors.
!! It is assumed that the appropriate pelist has been set before entering this routine.
!! This can be achieved in the following way.
!! ~~~~~~~~~~{.f90}
!! if (Ice%slow_ice_pe) then
!!    call mpp_set_current_pelist(Ice%slow_pelist)
!!    call slow_ice_chksum('MAIN_LOOP-', nc)
!! endif
!! ~~~~~~~~~~
!! If you are on the global pelist before you enter this routine using the above call,
!! you can return to the global pelist by invoking
!! ~~~~~~~~~~{.f90}
!! call mpp_set_current_pelist()
!! ~~~~~~~~~~
!! after you exit. This is only necessary if you need to return to the global pelist.
  subroutine slow_ice_chksum(id, timestep, Ice, Ocean_ice_boundary)

    character(len=*), intent(in) :: id
    integer         , intent(in) :: timestep
    type(ice_data_type), intent(in) :: Ice
    type(ocean_ice_boundary_type), intent(in) :: Ocean_ice_boundary

    call ice_data_type_chksum(    id, timestep, Ice)
    call ocn_ice_bnd_type_chksum( id, timestep, Ocean_ice_boundary)

  end subroutine slow_ice_chksum


!> \brief This subroutine calls subroutine that will print out checksums of the elements
!! of the appropriate type.
!!
!! For coupled models typically these types are not defined on all processors.
!! It is assumed that the appropriate pelist has been set before entering this routine.
!! This can be achieved in the following way.
!! ~~~~~~~~~~{.f90}
!! if (Ocean%is_ocean_pe) then
!!    call mpp_set_current_pelist(Ocean%pelist)
!!    call ocean_chksum('MAIN_LOOP-', nc)
!! endif
!! ~~~~~~~~~~
!! If you are on the global pelist before you enter this routine using the above call,
!! you can return to the global pelist by invoking
!! ~~~~~~~~~~{.f90}
!! call mpp_set_current_pelist()
!! ~~~~~~~~~~
!! after you exit. This is only necessary if you need to return to the global pelist.
  subroutine ocean_chksum(id, timestep, Ocean, Ice_ocean_boundary)

    character(len=*), intent(in) :: id
    integer         , intent(in) :: timestep
    type (ocean_public_type), intent(in) :: Ocean
    type(ice_ocean_boundary_type), intent(in) :: Ice_ocean_boundary

    call ocean_public_type_chksum(id, timestep, Ocean)
    call ice_ocn_bnd_type_chksum( id, timestep, Ice_ocean_boundary)

  end subroutine ocean_chksum


end program coupler_main
