!> Main routine for lateral (along surface or neutral) diffusion of tracers
module MOM_tracer_hor_diff

! This file is part of MOM6. See LICENSE.md for the license.

use MOM_cpu_clock,                only : cpu_clock_id, cpu_clock_begin, cpu_clock_end
use MOM_cpu_clock,                only : CLOCK_MODULE, CLOCK_ROUTINE
use MOM_diag_mediator,            only : post_data, diag_ctrl
use MOM_diag_mediator,            only : register_diag_field, safe_alloc_ptr, time_type
use MOM_domains,                  only : sum_across_PEs, max_across_PEs
use MOM_domains,                  only : create_group_pass, do_group_pass, group_pass_type
use MOM_domains,                  only : pass_vector
use MOM_debugging,                only : hchksum, uvchksum
use MOM_diabatic_driver,          only : diabatic_CS
use MOM_EOS,                      only : calculate_density, EOS_type, EOS_domain
use MOM_error_handler,            only : MOM_error, FATAL, WARNING, MOM_mesg, is_root_pe
use MOM_error_handler,            only : MOM_set_verbosity, callTree_showQuery
use MOM_error_handler,            only : callTree_enter, callTree_leave, callTree_waypoint
use MOM_file_parser,              only : get_param, log_version, param_file_type
use MOM_grid,                     only : ocean_grid_type
use MOM_lateral_mixing_coeffs,    only : VarMix_CS
use MOM_MEKE_types,               only : MEKE_type
use MOM_neutral_diffusion,        only : neutral_diffusion_init, neutral_diffusion_end
use MOM_neutral_diffusion,        only : neutral_diffusion_CS
use MOM_neutral_diffusion,        only : neutral_diffusion_calc_coeffs, neutral_diffusion
use MOM_hor_bnd_diffusion,        only : hbd_CS, hor_bnd_diffusion_init
use MOM_hor_bnd_diffusion,        only : hor_bnd_diffusion, hor_bnd_diffusion_end
use MOM_tracer_registry,          only : tracer_registry_type, tracer_type, MOM_tracer_chksum
use MOM_unit_scaling,             only : unit_scale_type
use MOM_variables,                only : thermo_var_ptrs, vertvisc_type
use MOM_verticalGrid,             only : verticalGrid_type

implicit none ; private

#include <MOM_memory.h>

public tracer_hordiff, tracer_hor_diff_init, tracer_hor_diff_end

!> The control structure for along-layer and epineutral tracer diffusion
type, public :: tracer_hor_diff_CS ; private
  real    :: KhTr           !< The along-isopycnal tracer diffusivity [L2 T-1 ~> m2 s-1].
  real    :: KhTr_Slope_Cff !< The non-dimensional coefficient in KhTr formula [nondim]
  real    :: KhTr_min       !< Minimum along-isopycnal tracer diffusivity [L2 T-1 ~> m2 s-1].
  real    :: KhTr_max       !< Maximum along-isopycnal tracer diffusivity [L2 T-1 ~> m2 s-1].
  real    :: KhTr_passivity_coeff !< Passivity coefficient that scales Rd/dx (default = 0)
                                  !! where passivity is the ratio between along-isopycnal
                                  !! tracer mixing and thickness mixing [nondim]
  real    :: KhTr_passivity_min   !< Passivity minimum (default = 1/2) [nondim]
  real    :: ML_KhTR_scale        !< With Diffuse_ML_interior, the ratio of the
                                  !! truly horizontal diffusivity in the mixed
                                  !! layer to the epipycnal diffusivity [nondim].
  real    :: max_diff_CFL         !< If positive, locally limit the along-isopycnal
                                  !! tracer diffusivity to keep the diffusive CFL
                                  !! locally at or below this value [nondim].
  logical :: KhTr_use_vert_struct  !< If true, uses the equivalent barotropic structure
                                  !! as the vertical structure of tracer diffusivity.
  logical :: Diffuse_ML_interior  !< If true, diffuse along isopycnals between
                                  !! the mixed layer and the interior.
  logical :: check_diffusive_CFL  !< If true, automatically iterate the diffusion
                                  !! to ensure that the diffusive equivalent of
                                  !! the CFL limit is not violated.
  logical :: use_neutral_diffusion !< If true, use the neutral_diffusion module from within
                                   !! tracer_hor_diff.
  logical :: use_hor_bnd_diffusion !< If true, use the hor_bnd_diffusion module from within
                                   !! tracer_hor_diff.
  logical :: recalc_neutral_surf   !< If true, recalculate the neutral surfaces if CFL has been
                                   !! exceeded
  logical :: limit_bug             !< If true and the answer date is 20240330 or below, use a
                                   !! rotational symmetry breaking bug when limiting the tracer
                                   !! properties in tracer_epipycnal_ML_diff.
  integer :: answer_date           !< The vintage of the order of arithmetic to use for the tracer
                                   !! diffusion.  Values of 20240330 or below recover the answers
                                   !! from the original form of this code, while higher values use
                                   !! mathematically equivalent expressions that recover rotational symmetry
                                   !! when DIFFUSE_ML_TO_INTERIOR is true.
  type(neutral_diffusion_CS), pointer :: neutral_diffusion_CSp => NULL() !< Control structure for neutral diffusion.
  type(hbd_CS), pointer    :: hor_bnd_diffusion_CSp => NULL() !< Control structure for
                                                              !! horizontal boundary diffusion.
  type(diag_ctrl), pointer :: diag => NULL() !< A structure that is used to
                                   !! regulate the timing of diagnostic output.
  logical :: debug                 !< If true, write verbose checksums for debugging purposes.
  logical :: show_call_tree        !< Display the call tree while running. Set by VERBOSITY level.
  logical :: first_call = .true.   !< This is true until after the first call
  !>@{ Diagnostic IDs
  integer :: id_KhTr_u  = -1
  integer :: id_KhTr_v  = -1
  integer :: id_KhTr_h  = -1
  integer :: id_CFL     = -1
  integer :: id_khdt_x  = -1
  integer :: id_khdt_y  = -1
  !>@}

  type(group_pass_type) :: pass_t !< For group halo pass, used in both
                                  !! tracer_hordiff and tracer_epipycnal_ML_diff
end type tracer_hor_diff_CS

!> A type that can be used to create arrays of pointers to 2D arrays
type p2d
  real, dimension(:,:), pointer :: p => NULL() !< A pointer to a 2D array of reals [various]
end type p2d
!> A type that can be used to create arrays of pointers to 2D integer arrays
type p2di
  integer, dimension(:,:), pointer :: p => NULL() !< A pointer to a 2D array of integers
end type p2di

!>@{ CPU time clocks
integer :: id_clock_diffuse, id_clock_epimix, id_clock_pass, id_clock_sync
!>@}

contains

!> Compute along-coordinate diffusion of all tracers
!! using the diffusivity in CS%KhTr, or using space-dependent diffusivity.
!! Multiple iterations are used (if necessary) so that there is no limit
!! on the acceptable time increment.
subroutine tracer_hordiff(h, dt, MEKE, VarMix, visc, G, GV, US, CS, Reg, tv, do_online_flag, read_khdt_x, read_khdt_y)
  type(ocean_grid_type),      intent(inout) :: G       !< Grid type
  type(verticalGrid_type),    intent(in)    :: GV      !< ocean vertical grid structure
  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)), &
                              intent(in)    :: h       !< Layer thickness [H ~> m or kg m-2]
  real,                       intent(in)    :: dt      !< time step [T ~> s]
  type(MEKE_type),            intent(in)    :: MEKE    !< MEKE fields
  type(VarMix_CS),            intent(in)    :: VarMix  !< Variable mixing type
  type(vertvisc_type),        intent(in)    :: visc    !< Structure with vertical viscosities,
                                                       !! boundary layer properties and related fields
  type(unit_scale_type),      intent(in)    :: US      !< A dimensional unit scaling type
  type(tracer_hor_diff_CS),   pointer       :: CS      !< module control structure
  type(tracer_registry_type), pointer       :: Reg     !< registered tracers
  type(thermo_var_ptrs),      intent(in)    :: tv      !< A structure containing pointers to any available
                                                       !! thermodynamic fields, including potential temp and
                                                       !! salinity or mixed layer density. Absent fields have
                                                       !! NULL ptrs, and these may (probably will) point to
                                                       !! some of the same arrays as Tr does.  tv is required
                                                       !! for epipycnal mixing between mixed layer and the interior.
  ! Optional inputs for offline tracer transport
  logical,          optional, intent(in)    :: do_online_flag !< If present and true, do online
                                                       !! tracer transport with stored velocities.
  ! The next two arguments do not appear to be used anywhere.
  real, dimension(SZIB_(G),SZJ_(G)), &
                    optional, intent(in)    :: read_khdt_x !< If present, these are the zonal diffusivities
                                                       !! times a timestep from a previous run [L2 ~> m2]
  real, dimension(SZI_(G),SZJB_(G)), &
                    optional, intent(in)    :: read_khdt_y !< If present, these are the meridional diffusivities
                                                       !! times a timestep from a previous run [L2 ~> m2]


  real, dimension(SZI_(G),SZJ_(G)) :: &
    Ihdxdy, &     ! The inverse of the volume or mass of fluid in a layer in a
                  ! grid cell [H-1 L-2 ~> m-3 or kg-1].
    CFL, &        ! A diffusive CFL number for each cell [nondim].
    dTr           ! The change in a tracer's concentration, in units of concentration [Conc].

  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)+1)  :: Kh_h
                  ! The tracer diffusivity averaged to tracer points [L2 T-1 ~> m2 s-1].
  real, dimension(SZIB_(G),SZJ_(G)) :: &
    khdt_x        ! The value of Khtr*dt times the open face width divided by
                  ! the distance between adjacent tracer points [L2 ~> m2].
  real, dimension(SZI_(G),SZJB_(G)) :: &
    khdt_y        ! The value of Khtr*dt times the open face width divided by
                  ! the distance between adjacent tracer points [L2 ~> m2].
  real, dimension(SZIB_(G),SZJ_(G),SZK_(GV)+1) :: &
    Coef_x, &     ! The coefficients relating zonal tracer differences to time-integrated
                  ! fluxes, in [L2 ~> m2] for some schemes and [H L2 ~> m3 or kg] for others.
    Kh_u          ! Tracer mixing coefficient at u-points [L2 T-1 ~> m2 s-1].
  real, dimension(SZI_(G),SZJB_(G),SZK_(GV)+1) :: &
    Coef_y, &     ! The coefficients relating meridional tracer differences to time-integrated
                  ! fluxes, in [L2 ~> m2] for some schemes and [H L2 ~> m3 or kg] for others.
    Kh_v          ! Tracer mixing coefficient at u-points [L2 T-1 ~> m2 s-1].

  real :: khdt_max ! The local limiting value of khdt_x or khdt_y [L2 ~> m2].
  real :: max_CFL ! The global maximum of the diffusive CFL number [nondim]
  logical :: use_VarMix, Resoln_scaled, do_online, use_Eady
  integer :: i, j, k, m, is, ie, js, je, nz, ntr, itt, num_itts
  real :: I_numitts  ! The inverse of the number of iterations, num_itts [nondim]
  real :: scale      ! The fraction of khdt_x or khdt_y that is applied in this
                     ! layer for this iteration [nondim].
  real :: Idt        ! The inverse of the time step [T-1 ~> s-1].
  real :: h_neglect  ! A thickness that is so small it is usually lost
                     ! in roundoff and can be neglected [H ~> m or kg m-2].
  real :: Kh_loc     ! The local value of Kh [L2 T-1 ~> m2 s-1].
  real :: Res_Fn     ! The local value of the resolution function [nondim].
  real :: Rd_dx      ! The local value of deformation radius over grid-spacing [nondim].
  real :: normalize  ! normalization used for diagnostic Kh_h [nondim]; diffusivity averaged to h-points.

  is = G%isc ; ie = G%iec ; js = G%jsc ; je = G%jec ; nz = GV%ke

  do_online = .true.
  if (present(do_online_flag)) do_online = do_online_flag

  if (.not. associated(CS)) call MOM_error(FATAL, "MOM_tracer_hor_diff: "// &
       "register_tracer must be called before tracer_hordiff.")
  if (.not. associated(Reg)) call MOM_error(FATAL, "MOM_tracer_hor_diff: "// &
       "register_tracer must be called before tracer_hordiff.")
  if (Reg%ntr == 0 .or. (CS%KhTr <= 0.0 .and. .not. VarMix%use_variable_mixing)) return

  if (CS%show_call_tree) call callTree_enter("tracer_hordiff(), MOM_tracer_hor_diff.F90")

  call cpu_clock_begin(id_clock_diffuse)

  ntr = Reg%ntr
  Idt = 1.0 / dt
  h_neglect = GV%H_subroundoff

  if (CS%Diffuse_ML_interior .and. CS%first_call) then ; if (is_root_pe()) then
    do m=1,ntr ; if (associated(Reg%Tr(m)%df_x) .or. associated(Reg%Tr(m)%df_y)) &
      call MOM_error(WARNING, "tracer_hordiff: Tracer "//trim(Reg%Tr(m)%name)// &
          " has associated 3-d diffusive flux diagnostics.  These are not "//&
          "valid when DIFFUSE_ML_TO_INTERIOR is defined. Use 2-d tracer "//&
          "diffusion diagnostics instead to get accurate total fluxes.")
    enddo
  endif ; endif
  CS%first_call = .false.

  if (CS%debug) call MOM_tracer_chksum("Before tracer diffusion ", Reg, G)

  use_VarMix = .false. ; Resoln_scaled = .false. ; use_Eady = .false.
  if (VarMix%use_variable_mixing) then
    use_VarMix = VarMix%use_variable_mixing
    Resoln_scaled = VarMix%Resoln_scaled_KhTr
    use_Eady = CS%KhTr_Slope_Cff > 0.
    CS%KhTr_use_vert_struct = allocated(VarMix%khtr_struct)
  endif

  call cpu_clock_begin(id_clock_pass)
  do m=1,ntr
    call create_group_pass(CS%pass_t, Reg%Tr(m)%t(:,:,:), G%Domain)
  enddo
  call cpu_clock_end(id_clock_pass)

  if (CS%show_call_tree) call callTree_waypoint("Calculating diffusivity (tracer_hordiff)")

  if (do_online) then
    if (use_VarMix) then
      !$OMP parallel do default(shared) private(Kh_loc,Rd_dx)
      do j=js,je ; do I=is-1,ie
        Kh_loc = CS%KhTr
        if (use_Eady) Kh_loc = Kh_loc + CS%KhTr_Slope_Cff*VarMix%L2u(I,j)*VarMix%SN_u(I,j)
        if (allocated(MEKE%Kh)) &
          Kh_loc = Kh_loc + MEKE%KhTr_fac*sqrt(MEKE%Kh(i,j)*MEKE%Kh(i+1,j))
        if (CS%KhTr_max > 0.) Kh_loc = min(Kh_loc, CS%KhTr_max)
        if (Resoln_scaled) &
          Kh_loc = Kh_loc * 0.5*(VarMix%Res_fn_h(i,j) + VarMix%Res_fn_h(i+1,j))
        Kh_u(I,j,1) = max(Kh_loc, CS%KhTr_min)
        if (CS%KhTr_passivity_coeff>0.) then ! Apply passivity
          Rd_dx=0.5*( VarMix%Rd_dx_h(i,j)+VarMix%Rd_dx_h(i+1,j) ) ! Rd/dx at u-points
          Kh_loc = Kh_u(I,j,1)*max( CS%KhTr_passivity_min, CS%KhTr_passivity_coeff*Rd_dx )
          if (CS%KhTr_max > 0.) Kh_loc = min(Kh_loc, CS%KhTr_max) ! Re-apply max
          Kh_u(I,j,1) = max(Kh_loc, CS%KhTr_min) ! Re-apply min
        endif
      enddo ; enddo
      !$OMP parallel do default(shared) private(Kh_loc,Rd_dx)
      do J=js-1,je ;  do i=is,ie
        Kh_loc = CS%KhTr
        if (use_Eady) Kh_loc = Kh_loc + CS%KhTr_Slope_Cff*VarMix%L2v(i,J)*VarMix%SN_v(i,J)
        if (allocated(MEKE%Kh)) &
          Kh_loc = Kh_loc + MEKE%KhTr_fac*sqrt(MEKE%Kh(i,j)*MEKE%Kh(i,j+1))
        if (CS%KhTr_max > 0.) Kh_loc = min(Kh_loc, CS%KhTr_max)
        if (Resoln_scaled) &
          Kh_loc = Kh_loc * 0.5*(VarMix%Res_fn_h(i,j) + VarMix%Res_fn_h(i,j+1))
        Kh_v(i,J,1) = max(Kh_loc, CS%KhTr_min)
        if (CS%KhTr_passivity_coeff>0.) then ! Apply passivity
          Rd_dx = 0.5*( VarMix%Rd_dx_h(i,j)+VarMix%Rd_dx_h(i,j+1) ) ! Rd/dx at v-points
          Kh_loc = Kh_v(i,J,1)*max( CS%KhTr_passivity_min, CS%KhTr_passivity_coeff*Rd_dx )
          if (CS%KhTr_max > 0.) Kh_loc = min(Kh_loc, CS%KhTr_max) ! Re-apply max
          Kh_v(i,J,1) = max(Kh_loc, CS%KhTr_min) ! Re-apply min
        endif
      enddo ; enddo

      !$OMP parallel do default(shared)
      do j=js,je ; do I=is-1,ie
        khdt_x(I,j) = dt*(Kh_u(I,j,1)*(G%dy_Cu(I,j)*G%IdxCu(I,j)))
      enddo ; enddo
      !$OMP parallel do default(shared)
      do J=js-1,je ; do i=is,ie
        khdt_y(i,J) = dt*(Kh_v(i,J,1)*(G%dx_Cv(i,J)*G%IdyCv(i,J)))
      enddo ; enddo
    elseif (Resoln_scaled) then
      !$OMP parallel do default(shared) private(Res_fn)
      do j=js,je ; do I=is-1,ie
        Res_fn = 0.5 * (VarMix%Res_fn_h(i,j) + VarMix%Res_fn_h(i+1,j))
        Kh_u(I,j,1) = max(CS%KhTr * Res_fn, CS%KhTr_min)
        khdt_x(I,j) = dt*(CS%KhTr*(G%dy_Cu(I,j)*G%IdxCu(I,j))) * Res_fn
      enddo ; enddo
      !$OMP parallel do default(shared) private(Res_fn)
      do J=js-1,je ;  do i=is,ie
        Res_fn = 0.5*(VarMix%Res_fn_h(i,j) + VarMix%Res_fn_h(i,j+1))
        Kh_v(i,J,1) = max(CS%KhTr * Res_fn, CS%KhTr_min)
        khdt_y(i,J) = dt*(CS%KhTr*(G%dx_Cv(i,J)*G%IdyCv(i,J))) * Res_fn
      enddo ; enddo
    else  ! Use a simple constant diffusivity.
      if (CS%id_KhTr_u > 0) then
        !$OMP parallel do default(shared)
        do j=js,je ; do I=is-1,ie
          Kh_u(I,j,1) = CS%KhTr
          khdt_x(I,j) = dt*(CS%KhTr*(G%dy_Cu(I,j)*G%IdxCu(I,j)))
        enddo ; enddo
      else
        !$OMP parallel do default(shared)
        do j=js,je ; do I=is-1,ie
          khdt_x(I,j) = dt*(CS%KhTr*(G%dy_Cu(I,j)*G%IdxCu(I,j)))
        enddo ; enddo
      endif
      if (CS%id_KhTr_v > 0) then
        !$OMP parallel do default(shared)
        do J=js-1,je ;  do i=is,ie
          Kh_v(i,J,1) = CS%KhTr
          khdt_y(i,J) = dt*(CS%KhTr*(G%dx_Cv(i,J)*G%IdyCv(i,J)))
        enddo ; enddo
      else
        !$OMP parallel do default(shared)
        do J=js-1,je ;  do i=is,ie
          khdt_y(i,J) = dt*(CS%KhTr*(G%dx_Cv(i,J)*G%IdyCv(i,J)))
        enddo ; enddo
      endif
    endif ! VarMix

    if (CS%max_diff_CFL > 0.0) then
      if ((CS%id_KhTr_u > 0) .or. (CS%id_KhTr_h > 0)) then
        !$OMP parallel do default(shared) private(khdt_max)
        do j=js,je ; do I=is-1,ie
          khdt_max = 0.125*CS%max_diff_CFL * min(G%areaT(i,j), G%areaT(i+1,j))
          if (khdt_x(I,j) > khdt_max) then
            khdt_x(I,j) = khdt_max
            if (dt*(G%dy_Cu(I,j)*G%IdxCu(I,j)) > 0.0) &
              Kh_u(I,j,1) = khdt_x(I,j) / (dt*(G%dy_Cu(I,j)*G%IdxCu(I,j)))
          endif
        enddo ; enddo
      else
        !$OMP parallel do default(shared) private(khdt_max)
        do j=js,je ; do I=is-1,ie
          khdt_max = 0.125*CS%max_diff_CFL * min(G%areaT(i,j), G%areaT(i+1,j))
          khdt_x(I,j) = min(khdt_x(I,j), khdt_max)
        enddo ; enddo
      endif
      if ((CS%id_KhTr_v > 0) .or. (CS%id_KhTr_h > 0)) then
        !$OMP parallel do default(shared) private(khdt_max)
        do J=js-1,je ; do i=is,ie
          khdt_max = 0.125*CS%max_diff_CFL * min(G%areaT(i,j), G%areaT(i,j+1))
          if (khdt_y(i,J) > khdt_max) then
            khdt_y(i,J) = khdt_max
            if (dt*(G%dx_Cv(i,J)*G%IdyCv(i,J)) > 0.0) &
              Kh_v(i,J,1) = khdt_y(i,J) / (dt*(G%dx_Cv(i,J)*G%IdyCv(i,J)))
          endif
        enddo ; enddo
      else
        !$OMP parallel do default(shared) private(khdt_max)
        do J=js-1,je ; do i=is,ie
          khdt_max = 0.125*CS%max_diff_CFL * min(G%areaT(i,j), G%areaT(i,j+1))
          khdt_y(i,J) = min(khdt_y(i,J), khdt_max)
        enddo ; enddo
      endif
    endif

  else ! .not. do_online
    !$OMP parallel do default(shared)
    do j=js,je ; do I=is-1,ie
      khdt_x(I,j) = read_khdt_x(I,j)
    enddo ; enddo
    !$OMP parallel do default(shared)
    do J=js-1,je ;  do i=is,ie
      khdt_y(i,J) = read_khdt_y(i,J)
    enddo ; enddo
    call pass_vector(khdt_x, khdt_y, G%Domain)
  endif ! do_online

  if (CS%check_diffusive_CFL) then
    if (CS%show_call_tree) call callTree_waypoint("Checking diffusive CFL (tracer_hordiff)")
    max_CFL = 0.0
    do j=js,je ; do i=is,ie
      CFL(i,j) = 2.0*((khdt_x(I-1,j) + khdt_x(I,j)) + &
                      (khdt_y(i,J-1) + khdt_y(i,J))) * G%IareaT(i,j)
      if (max_CFL < CFL(i,j)) max_CFL = CFL(i,j)
    enddo ; enddo
    call cpu_clock_begin(id_clock_sync)
    call max_across_PEs(max_CFL)
    call cpu_clock_end(id_clock_sync)
    num_itts = max(1, ceiling(max_CFL - 4.0*EPSILON(max_CFL)))
    I_numitts = 1.0 / (real(num_itts))
    if (CS%id_CFL > 0) call post_data(CS%id_CFL, CFL, CS%diag, mask=G%mask2dT)
  elseif (CS%max_diff_CFL > 0.0) then
    num_itts = max(1, ceiling(CS%max_diff_CFL - 4.0*EPSILON(CS%max_diff_CFL)))
    I_numitts = 1.0 / (real(num_itts))
  else
    num_itts = 1 ; I_numitts = 1.0
  endif

  do m=1,ntr
    if (associated(Reg%Tr(m)%df_x)) then
      do k=1,nz ; do j=js,je ; do I=is-1,ie
        Reg%Tr(m)%df_x(I,j,k) = 0.0
      enddo ; enddo ; enddo
    endif
    if (associated(Reg%Tr(m)%df_y)) then
      do k=1,nz ; do J=js-1,je ; do i=is,ie
        Reg%Tr(m)%df_y(i,J,k) = 0.0
      enddo ; enddo ; enddo
    endif
    if (associated(Reg%Tr(m)%df2d_x)) then
      do j=js,je ; do I=is-1,ie ; Reg%Tr(m)%df2d_x(I,j) = 0.0 ; enddo ; enddo
    endif
    if (associated(Reg%Tr(m)%df2d_y)) then
      do J=js-1,je ; do i=is,ie ; Reg%Tr(m)%df2d_y(i,J) = 0.0 ; enddo ; enddo
    endif
  enddo

  if (CS%use_hor_bnd_diffusion) then

    if (CS%show_call_tree) call callTree_waypoint("Calling horizontal boundary diffusion (tracer_hordiff)")

    call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)

    do k=1,nz+1
      do J=js-1,je
        do i=is,ie
          Coef_y(i,J,K) = I_numitts * khdt_y(i,J)
        enddo
      enddo
    enddo
    do k=1,nz+1
      do j=js,je
        do I=is-1,ie
          Coef_x(I,j,K) = I_numitts * khdt_x(I,j)
        enddo
      enddo
    enddo
    if (CS%KhTr_use_vert_struct) then
      do K=2,nz+1
        do J=js-1,je
          do i=is,ie
            Coef_y(i,J,K) = Coef_y(i,J,1) * 0.5 * ( VarMix%khtr_struct(i,j,k-1) + VarMix%khtr_struct(i,j+1,k-1) )
          enddo
        enddo
      enddo
      do k=2,nz+1
        do j=js,je
          do I=is-1,ie
            Coef_x(I,j,K) = Coef_x(I,j,1) * 0.5 * ( VarMix%khtr_struct(i,j,k-1) + VarMix%khtr_struct(i+1,j,k-1) )
          enddo
        enddo
      enddo
    endif

    do itt=1,num_itts
      if (CS%show_call_tree) call callTree_waypoint("Calling horizontal boundary diffusion (tracer_hordiff)",itt)
      if (itt>1) then ! Update halos for subsequent iterations
        call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)
      endif
      call hor_bnd_diffusion(G, GV, US, h, Coef_x, Coef_y, I_numitts*dt, Reg, visc, &
                             CS%hor_bnd_diffusion_CSp)
    enddo ! itt
  endif

  if (CS%use_neutral_diffusion) then

    if (CS%show_call_tree) call callTree_waypoint("Calling neutral diffusion coeffs (tracer_hordiff)")

    call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)
    ! We are assuming that neutral surfaces do not evolve (much) as a result of multiple
    !horizontal diffusion iterations. Otherwise the call to neutral_diffusion_calc_coeffs()
    ! would be inside the itt-loop. -AJA

    if (associated(tv%p_surf)) then
      call neutral_diffusion_calc_coeffs(G, GV, US, h, tv%T, tv%S, visc, CS%neutral_diffusion_CSp, p_surf=tv%p_surf)
    else
      call neutral_diffusion_calc_coeffs(G, GV, US, h, tv%T, tv%S, visc, CS%neutral_diffusion_CSp)
    endif

    do k=1,nz+1
      do J=js-1,je
        do i=is,ie
          Coef_y(i,J,K) = I_numitts * khdt_y(i,J)
        enddo
      enddo
    enddo
    do k=1,nz+1
      do j=js,je
        do I=is-1,ie
          Coef_x(I,j,K) = I_numitts * khdt_x(I,j)
        enddo
      enddo
    enddo
    if (CS%KhTr_use_vert_struct) then
      do K=2,nz+1
        do J=js-1,je
          do i=is,ie
            Coef_y(i,J,K) = Coef_y(i,J,1) * 0.5 * ( VarMix%khtr_struct(i,j,k-1) + VarMix%khtr_struct(i,j+1,k-1) )
          enddo
        enddo
      enddo
      do k=2,nz+1
        do j=js,je
          do I=is-1,ie
            Coef_x(I,j,K) = Coef_x(I,j,1) * 0.5 * ( VarMix%khtr_struct(i,j,k-1) + VarMix%khtr_struct(i+1,j,k-1) )
          enddo
        enddo
      enddo
    endif

    do itt=1,num_itts
      if (CS%show_call_tree) call callTree_waypoint("Calling neutral diffusion (tracer_hordiff)",itt)
      if (itt>1) then ! Update halos for subsequent iterations
        call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)
        if (CS%recalc_neutral_surf) then
          if (associated(tv%p_surf)) then
            call neutral_diffusion_calc_coeffs(G, GV, US, h, tv%T, tv%S, visc, CS%neutral_diffusion_CSp, &
                                               p_surf=tv%p_surf)
          else
            call neutral_diffusion_calc_coeffs(G, GV, US, h, tv%T, tv%S, visc, CS%neutral_diffusion_CSp)
          endif
        endif
      endif
      call neutral_diffusion(G, GV,  h, Coef_x, Coef_y, I_numitts*dt, Reg, US, CS%neutral_diffusion_CSp)
    enddo ! itt

  else    ! following if not using neutral diffusion, but instead along-surface diffusion

    if (CS%show_call_tree) call callTree_waypoint("Calculating horizontal diffusion (tracer_hordiff)")
    do itt=1,num_itts
      call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)
      !$OMP parallel do default(shared) private(scale,Coef_y,Coef_x,Ihdxdy,dTr)
      do k=1,nz
        scale = I_numitts
        if (CS%Diffuse_ML_interior) then
          if (k<=GV%nkml) then
            if (CS%ML_KhTr_scale <= 0.0) cycle
            scale = I_numitts * CS%ML_KhTr_scale
          endif
          if ((k>GV%nkml) .and. (k<=GV%nk_rho_varies)) cycle
        endif

        do J=js-1,je ; do i=is,ie
          Coef_y(i,J,1) = ((scale * khdt_y(i,J))*2.0*(h(i,j,k)*h(i,j+1,k))) / &
                                                   (h(i,j,k)+h(i,j+1,k)+h_neglect)
        enddo ; enddo

        do j=js,je
          do I=is-1,ie
            Coef_x(I,j,1) = ((scale * khdt_x(I,j))*2.0*(h(i,j,k)*h(i+1,j,k))) / &
                                                     (h(i,j,k)+h(i+1,j,k)+h_neglect)
          enddo

          do i=is,ie
            Ihdxdy(i,j) = G%IareaT(i,j) / (h(i,j,k)+h_neglect)
          enddo
        enddo

        do m=1,ntr
          do j=js,je ; do i=is,ie
            dTr(i,j) = Ihdxdy(i,j) * &
              ( ((Coef_x(I-1,j,1) * (Reg%Tr(m)%t(i-1,j,k) - Reg%Tr(m)%t(i,j,k))) - &
                 (Coef_x(I,j,1) * (Reg%Tr(m)%t(i,j,k) - Reg%Tr(m)%t(i+1,j,k)))) + &
                ((Coef_y(i,J-1,1) * (Reg%Tr(m)%t(i,j-1,k) - Reg%Tr(m)%t(i,j,k))) - &
                 (Coef_y(i,J,1) * (Reg%Tr(m)%t(i,j,k) - Reg%Tr(m)%t(i,j+1,k)))) )
          enddo ; enddo
          if (associated(Reg%Tr(m)%df_x)) then ; do j=js,je ; do I=G%IscB,G%IecB
            Reg%Tr(m)%df_x(I,j,k) = Reg%Tr(m)%df_x(I,j,k) + Coef_x(I,j,1) &
                * (Reg%Tr(m)%t(i,j,k) - Reg%Tr(m)%t(i+1,j,k)) * Idt
          enddo ; enddo ; endif
          if (associated(Reg%Tr(m)%df_y)) then ; do J=G%JscB,G%JecB ; do i=is,ie
            Reg%Tr(m)%df_y(i,J,k) = Reg%Tr(m)%df_y(i,J,k) + Coef_y(i,J,1) &
                * (Reg%Tr(m)%t(i,j,k) - Reg%Tr(m)%t(i,j+1,k)) * Idt
          enddo ; enddo ; endif
          if (associated(Reg%Tr(m)%df2d_x)) then ; do j=js,je ; do I=G%IscB,G%IecB
            Reg%Tr(m)%df2d_x(I,j) = Reg%Tr(m)%df2d_x(I,j) + Coef_x(I,j,1) &
                * (Reg%Tr(m)%t(i,j,k) - Reg%Tr(m)%t(i+1,j,k)) * Idt
          enddo ; enddo ; endif
          if (associated(Reg%Tr(m)%df2d_y)) then ; do J=G%JscB,G%JecB ; do i=is,ie
            Reg%Tr(m)%df2d_y(i,J) = Reg%Tr(m)%df2d_y(i,J) + Coef_y(i,J,1) &
                * (Reg%Tr(m)%t(i,j,k) - Reg%Tr(m)%t(i,j+1,k)) * Idt
          enddo ; enddo ; endif
          do j=js,je ; do i=is,ie
            Reg%Tr(m)%t(i,j,k) = Reg%Tr(m)%t(i,j,k) + dTr(i,j)
          enddo ; enddo
        enddo

      enddo ! End of k loop.

      ! Do user controlled underflow of the tracer concentrations.
      do m=1,ntr ; if (Reg%Tr(m)%conc_underflow > 0.0) then
        !$OMP parallel do default(shared)
        do k=1,nz ; do j=js,je ; do i=is,ie
          if (abs(Reg%Tr(m)%t(i,j,k)) < Reg%Tr(m)%conc_underflow) Reg%Tr(m)%t(i,j,k) = 0.0
        enddo ; enddo ; enddo
      endif ; enddo

    enddo ! End of "while" loop.

  endif   ! endif for CS%use_neutral_diffusion
  call cpu_clock_end(id_clock_diffuse)


  if (CS%Diffuse_ML_interior) then
    if (CS%show_call_tree) call callTree_waypoint("Calling epipycnal_ML_diff (tracer_hordiff)")
    if (CS%debug) call MOM_tracer_chksum("Before epipycnal diff ", Reg, G)

    call cpu_clock_begin(id_clock_epimix)
    call tracer_epipycnal_ML_diff(h, dt, Reg%Tr, ntr, khdt_x, khdt_y, G, GV, US, &
                                  CS, tv, num_itts)
    call cpu_clock_end(id_clock_epimix)
  endif

  if (CS%debug) call MOM_tracer_chksum("After tracer diffusion ", Reg, G)

  ! post diagnostics for 2d tracer diffusivity
  if (CS%id_KhTr_u > 0) then
    do j=js,je ; do I=is-1,ie
      Kh_u(I,j,:) = G%mask2dCu(I,j)*Kh_u(I,j,1)
    enddo ; enddo
    if (CS%KhTr_use_vert_struct) then
      do K=2,nz+1
        do j=js,je
          do I=is-1,ie
            Kh_u(I,j,K) = Kh_u(I,j,1) * 0.5 * ( VarMix%khtr_struct(i,j,k-1) + VarMix%khtr_struct(i+1,j,k-1) )
          enddo
        enddo
      enddo
    endif
    !call post_data(CS%id_KhTr_u, Kh_u, CS%diag, is_static=.false., mask=G%mask2dCu)
    call post_data(CS%id_KhTr_u, Kh_u, CS%diag)
  endif
  if (CS%id_KhTr_v > 0) then
    do J=js-1,je ; do i=is,ie
      Kh_v(i,J,:) = G%mask2dCv(i,J)*Kh_v(i,J,1)
    enddo ; enddo
    if (CS%KhTr_use_vert_struct) then
      do K=2,nz+1
        do J=js-1,je
          do i=is,ie
            Kh_v(i,J,K) = Kh_v(i,J,1) * 0.5 * ( VarMix%khtr_struct(i,j,k-1) + VarMix%khtr_struct(i,j+1,k-1) )
          enddo
        enddo
      enddo
    endif
    !call post_data(CS%id_KhTr_v, Kh_v, CS%diag, is_static=.false., mask=G%mask2dCv)
    call post_data(CS%id_KhTr_v, Kh_v, CS%diag)
  endif
  if (CS%id_KhTr_h > 0) then
    Kh_h(:,:,:) = 0.0
    do j=js,je ; do I=is-1,ie
      Kh_u(I,j,1) = G%mask2dCu(I,j)*Kh_u(I,j,1)
    enddo ; enddo
    do J=js-1,je ; do i=is,ie
      Kh_v(i,J,1) = G%mask2dCv(i,J)*Kh_v(i,J,1)
    enddo ; enddo

    do j=js,je ; do i=is,ie
      normalize = 1.0 / ((G%mask2dCu(I-1,j)+G%mask2dCu(I,j)) + &
                         (G%mask2dCv(i,J-1)+G%mask2dCv(i,J)) + 1.0e-37)
      Kh_h(i,j,:) = normalize*G%mask2dT(i,j)*((Kh_u(I-1,j,1)+Kh_u(I,j,1)) + &
                                             (Kh_v(i,J-1,1)+Kh_v(i,J,1)))
      if (CS%KhTr_use_vert_struct) then
        do K=2,nz+1
          Kh_h(i,j,K) = normalize*G%mask2dT(i,j)*VarMix%khtr_struct(i,j,k-1)*((Kh_u(I-1,j,1)+Kh_u(I,j,1)) + &
                                                                            (Kh_v(i,J-1,1)+Kh_v(i,J,1)))
        enddo
      endif
    enddo ; enddo
    !call post_data(CS%id_KhTr_h, Kh_h, CS%diag, is_static=.false., mask=G%mask2dT)
    call post_data(CS%id_KhTr_h, Kh_h, CS%diag)
  endif

  if (CS%debug) then
    call uvchksum("After tracer diffusion khdt_[xy]", khdt_x, khdt_y, &
                  G%HI, haloshift=0, symmetric=.true., unscale=US%L_to_m**2, &
                  scalar_pair=.true.)
  endif

  if (CS%id_khdt_x > 0) call post_data(CS%id_khdt_x, khdt_x, CS%diag)
  if (CS%id_khdt_y > 0) call post_data(CS%id_khdt_y, khdt_y, CS%diag)

  if (CS%show_call_tree) call callTree_leave("tracer_hordiff()")

end subroutine tracer_hordiff

!> This subroutine does epipycnal diffusion of all tracers between the mixed
!! and buffer layers and the interior, using the diffusivity in CS%KhTr.
!! Multiple iterations are used (if necessary) so that there is no limit on the
!! acceptable time increment.
subroutine tracer_epipycnal_ML_diff(h, dt, Tr, ntr, khdt_epi_x, khdt_epi_y, G, &
                                    GV, US, CS, tv, num_itts)
  type(ocean_grid_type),                    intent(inout) :: G          !< ocean grid structure
  type(verticalGrid_type),                  intent(in)    :: GV         !< ocean vertical grid structure
  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)), intent(in)   :: h          !< layer thickness [H ~> m or kg m-2]
  real,                                     intent(in)    :: dt         !< time step [T ~> s]
  type(tracer_type),                        intent(inout) :: Tr(:)      !< tracer array
  integer,                                  intent(in)    :: ntr        !< number of tracers
  real, dimension(SZIB_(G),SZJ_(G)),        intent(in)    :: khdt_epi_x !< Zonal epipycnal diffusivity times
                                                           !! a time step and the ratio of the open face width over
                                                           !! the distance between adjacent tracer points [L2 ~> m2]
  real, dimension(SZI_(G),SZJB_(G)),        intent(in)    :: khdt_epi_y !< Meridional epipycnal diffusivity times
                                                           !! a time step and the ratio of the open face width over
                                                           !! the distance between adjacent tracer points [L2 ~> m2]
  type(unit_scale_type),                    intent(in)    :: US         !< A dimensional unit scaling type
  type(tracer_hor_diff_CS),                 intent(inout) :: CS         !< module control structure
  type(thermo_var_ptrs),                    intent(in)    :: tv         !< thermodynamic structure
  integer,                                  intent(in)    :: num_itts   !< number of iterations (usually=1)


  real, dimension(SZI_(G), SZJ_(G)) :: &
    Rml_max  ! The maximum coordinate density within the mixed layer [R ~> kg m-3].
  real, dimension(SZI_(G), SZJ_(G), max(1,GV%nk_rho_varies)) :: &
    rho_coord ! The coordinate density that is used to mix along [R ~> kg m-3].

  ! The naming mnemonic is a=above,b=below,L=Left,R=Right,u=u-point,v=v-point.
  ! These are 1-D arrays of pointers to 2-d arrays to minimize memory usage.
  type(p2d), dimension(SZJ_(G)) :: &
    deep_wt_Lu, deep_wt_Ru, &  ! The relative weighting of the deeper of a pair [nondim].
    hP_Lu, hP_Ru       ! The total thickness on each side for each pair [H ~> m or kg m-2].

  type(p2d), dimension(SZJB_(G)) :: &
    deep_wt_Lv, deep_wt_Rv, & ! The relative weighting of the deeper of a pair [nondim].
    hP_Lv, hP_Rv       ! The total thickness on each side for each pair [H ~> m or kg m-2].

  type(p2di), dimension(SZJ_(G)) :: &
    k0b_Lu, k0a_Lu, &  ! The original k-indices of the layers that participate
    k0b_Ru, k0a_Ru     ! in each pair of mixing at u-faces.
  type(p2di), dimension(SZJB_(G)) :: &
    k0b_Lv, k0a_Lv, &  ! The original k-indices of the layers that participate
    k0b_Rv, k0a_Rv     ! in each pair of mixing at v-faces.

  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)) :: &
    tr_flux_N, &      ! The tracer flux through the northern face [conc H L2 ~> conc m3 or conc kg]
    tr_flux_S, &      ! The tracer flux through the southern face [conc H L2 ~> conc m3 or conc kg]
    tr_flux_E, &      ! The tracer flux through the eastern face [conc H L2 ~> conc m3 or conc kg]
    tr_flux_W, &      ! The tracer flux through the western face [conc H L2 ~> conc m3 or conc kg]
    tr_flux_conv      ! The flux convergence of tracers [conc H L2 ~> conc m3 or conc kg]

  ! The following 3-d arrays were created in 2014 in MOM6 PR#12 to facilitate openMP threading
  ! on an i-loop, which might have been ill advised.
  real, dimension(SZI_(G),SZJB_(G),SZK_(GV)*2) :: &
    Tr_flux_3d, &     ! The tracer flux through pairings at meridional faces [conc H L2 ~> conc m3 or conc kg]
    Tr_adj_vert_L, &  ! Vertical adjustments to which layer the fluxes go into in the southern
                      ! columns at meridional face [conc H L2 ~> conc m3 or conc kg]
    Tr_adj_vert_R     ! Vertical adjustments to which layer the fluxes go into in the northern
                      ! columns at meridional face [conc H L2 ~> conc m3 or conc kg]

  real, dimension(SZI_(G),SZK_(GV), SZJ_(G)) :: &
    rho_srt, & ! The density of each layer of the sorted columns [R ~> kg m-3].
    h_srt      ! The thickness of each layer of the sorted columns [H ~> m or kg m-2].
  integer, dimension(SZI_(G),SZK_(GV), SZJ_(G)) :: &
    k0_srt     ! The original k-index that each layer of the sorted column corresponds to.

  real, dimension(SZK_(GV)) :: &
    h_demand_L, & ! The thickness in the left column that is demanded to match the thickness
                  ! in the counterpart [H ~> m or kg m-2].
    h_demand_R, & ! The thickness in the right column that is demanded to match the thickness
                  ! in the counterpart [H ~> m or kg m-2].
    h_used_L, &   ! The summed thickness from the left column that has actually been used [H ~> m or kg m-2]
    h_used_R, &   ! The summed thickness from the right columns that has actually been used [H ~> m or kg m-2]
    h_supply_frac_L, & ! The fraction of the demanded thickness that can actually be supplied
                       ! from a layer on the left [nondim].
    h_supply_frac_R    ! The fraction of the demanded thickness that can actually be supplied
                       ! from a layer on the right [nondim].
  integer, dimension(SZI_(G), SZJ_(G))  :: &
    num_srt, &   ! The number of layers that are sorted in each column.
    k_end_srt, & ! The maximum index in each column that might need to be
                 ! sorted, based on neighboring values of max_kRho
    max_kRho     ! The index of the layer whose target density is just denser
                 ! than the densest part of the mixed layer.
  integer, dimension(SZJ_(G))           :: &
    max_srt      ! The maximum value of num_srt in a k-row.
  integer, dimension(SZIB_(G), SZJ_(G)) :: &
    nPu          ! The number of epipycnal pairings at each u-point.
  integer, dimension(SZI_(G), SZJB_(G)) :: &
    nPv          ! The number of epipycnal pairings at each v-point.
  real :: h_exclude    ! A thickness that layers must attain to be considered
                       ! for inclusion in mixing [H ~> m or kg m-2].
  real :: Idt        ! The inverse of the time step [T-1 ~> s-1].
  real :: I_maxitt   ! The inverse of the maximum number of iterations [nondim]
  real :: rho_pair, rho_a, rho_b  ! Temporary densities [R ~> kg m-3].
  real :: Tr_min_face  ! The minimum tracer concentration associated with a pairing [Conc]
  real :: Tr_max_face  ! The maximum tracer concentration associated with a pairing [Conc]
  real :: Tr_La, Tr_Lb ! The 2 left-side tracer concentrations that might be associated with a pairing [Conc]
  real :: Tr_Ra, Tr_Rb ! The 2 right-side tracer concentrations that might be associated with a pairing [Conc]
  real :: Tr_av_L    ! The average tracer concentrations on the left side of a pairing [Conc].
  real :: Tr_av_R    ! The average tracer concentrations on the right side of a pairing [Conc].
  real :: Tr_flux    ! The tracer flux from left to right in a pair [conc H L2 ~> conc m3 or conc kg].
  real :: Tr_adj_vert  ! A downward vertical adjustment to Tr_flux between the two cells that
                     ! make up one side of the pairing [conc H L2 ~> conc m3 or conc kg].
  real :: h_L, h_R   ! Thicknesses to the left and right [H ~> m or kg m-2].
  real :: wt_a, wt_b ! Fractional weights of layers above and below [nondim].
  real :: vol        ! A cell volume or mass [H L2 ~> m3 or kg].

  ! The total number of pairings is usually much less than twice the number of layers, but
  ! the memory in these 1-d columns of pairings can be allocated generously for safety.
  integer, dimension(SZK_(GV)*2) :: &
    kbs_Lp, &   ! The sorted indices of the Left and Right columns for
    kbs_Rp      ! each pairing.
  logical, dimension(SZK_(GV)*2) :: &
    left_set, &  ! If true, the left or right point determines the density of
    right_set    ! of the trio.  If densities are exactly equal, both are true.

  real :: tmp    ! A temporary variable used in swaps [various]
  real :: p_ref_cv(SZI_(G)) ! The reference pressure for the coordinate density [R L2 T-2 ~> Pa]

  integer, dimension(2) :: EOSdom ! The i-computational domain for the equation of state
  integer :: k_max, k_min, k_test, itmp
  integer :: i, j, k, k2, m, is, ie, js, je, nz, nkmb
  integer :: isd, ied, jsd, jed, IsdB, IedB, k_size
  integer :: kL, kR, kLa, kLb, kRa, kRb, nP, itt, ns, max_itt
  integer :: PEmax_kRho

  is = G%isc ; ie = G%iec ; js = G%jsc ; je = G%jec ; nz = GV%ke
  isd = G%isd ; ied = G%ied ; jsd = G%jsd ; jed = G%jed
  IsdB = G%IsdB ; IedB = G%IedB
  Idt = 1.0 / dt
  nkmb = GV%nk_rho_varies

  if (num_itts <= 1) then
    max_itt = 1 ; I_maxitt = 1.0
  else
    max_itt = num_itts ; I_maxitt = 1.0 / (real(max_itt))
  endif

  do i=is-2,ie+2 ; p_ref_cv(i) = tv%P_Ref ; enddo
  EOSdom(:) = EOS_domain(G%HI,halo=2)

  call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)
  ! Determine which layers the mixed- and buffer-layers map into...
  !$OMP parallel do default(shared)
  do k=1,nkmb ; do j=js-2,je+2
    call calculate_density(tv%T(:,j,k), tv%S(:,j,k), p_ref_cv, rho_coord(:,j,k), &
                           tv%eqn_of_state, EOSdom)
  enddo ; enddo

  do j=js-2,je+2 ; do i=is-2,ie+2
    Rml_max(i,j) = rho_coord(i,j,1)
    num_srt(i,j) = 0 ; max_kRho(i,j) = 0
  enddo ; enddo
  do k=2,nkmb ; do j=js-2,je+2 ; do i=is-2,ie+2
    if (Rml_max(i,j) < rho_coord(i,j,k)) Rml_max(i,j) = rho_coord(i,j,k)
  enddo ; enddo ; enddo

  !   Use bracketing and bisection to find the k-level that the densest of the
  ! mixed and buffer layer corresponds to, such that:
  !     GV%Rlay(max_kRho-1) < Rml_max <= GV%Rlay(max_kRho)
  !$OMP parallel do default(shared) private(k_min,k_max,k_test)
  do j=js-2,je+2 ; do i=is-2,ie+2 ; if (G%mask2dT(i,j) > 0.0) then
    if ((Rml_max(i,j) > GV%Rlay(nz)) .or. (nkmb+1 > nz)) then ; max_kRho(i,j) = nz+1
    elseif ((Rml_max(i,j) <= GV%Rlay(nkmb+1)) .or. (nkmb+2 > nz)) then ; max_kRho(i,j) = nkmb+1
    else
      k_min = nkmb+2 ; k_max = nz
      do
        k_test = (k_min + k_max) / 2
        if (Rml_max(i,j) <= GV%Rlay(k_test-1)) then ; k_max = k_test-1
        elseif (GV%Rlay(k_test) < Rml_max(i,j)) then ; k_min = k_test+1
        else ; max_kRho(i,j) = k_test ; exit ; endif

        if (k_min == k_max) then ; max_kRho(i,j) = k_max ; exit ; endif
      enddo
    endif
  endif ; enddo ; enddo

  PEmax_kRho = 0
  do j=js-1,je+1 ; do i=is-1,ie+1
    k_end_srt(i,j) = max(max_kRho(i,j), max_kRho(i-1,j), max_kRho(i+1,j), &
                         max_kRho(i,j-1), max_kRho(i,j+1))
    if (PEmax_kRho < k_end_srt(i,j)) PEmax_kRho = k_end_srt(i,j)
  enddo ; enddo
  if (PEmax_kRho > nz) PEmax_kRho = nz ! PEmax_kRho could have been nz+1.

  h_exclude = 10.0*(GV%Angstrom_H + GV%H_subroundoff)
  !$OMP parallel default(shared) private(ns,tmp,itmp)
  !$OMP do
  do j=js-1,je+1
    do k=1,nkmb ; do i=is-1,ie+1 ; if (G%mask2dT(i,j) > 0.0) then
      if (h(i,j,k) > h_exclude) then
        num_srt(i,j) = num_srt(i,j) + 1 ; ns = num_srt(i,j)
        k0_srt(i,ns,j) = k
        rho_srt(i,ns,j) = rho_coord(i,j,k)
        h_srt(i,ns,j) = h(i,j,k)
      endif
    endif ; enddo ; enddo
    do k=nkmb+1,PEmax_kRho ; do i=is-1,ie+1 ; if (G%mask2dT(i,j) > 0.0) then
      if ((k<=k_end_srt(i,j)) .and. (h(i,j,k) > h_exclude)) then
        num_srt(i,j) = num_srt(i,j) + 1 ; ns = num_srt(i,j)
        k0_srt(i,ns,j) = k
        rho_srt(i,ns,j) = GV%Rlay(k)
        h_srt(i,ns,j) = h(i,j,k)
      endif
    endif ; enddo ; enddo
  enddo
  ! Sort each column by increasing density.  This should already be close,
  ! and the size of the arrays are small, so straight insertion is used.
  !$OMP do
  do j=js-1,je+1 ; do i=is-1,ie+1
    do k=2,num_srt(i,j) ; if (rho_srt(i,k,j) < rho_srt(i,k-1,j)) then
      ! The last segment needs to be shuffled earlier in the list.
      do k2 = k,2,-1 ; if (rho_srt(i,k2,j) >= rho_srt(i,k2-1,j)) exit
        itmp = k0_srt(i,k2-1,j) ; k0_srt(i,k2-1,j) = k0_srt(i,k2,j) ; k0_srt(i,k2,j) = itmp
        tmp = rho_srt(i,k2-1,j) ; rho_srt(i,k2-1,j) = rho_srt(i,k2,j) ; rho_srt(i,k2,j) = tmp
        tmp = h_srt(i,k2-1,j) ; h_srt(i,k2-1,j) = h_srt(i,k2,j) ; h_srt(i,k2,j) = tmp
      enddo
    endif ; enddo
  enddo ; enddo
  !$OMP do
  do j=js-1,je+1
    max_srt(j) = 0
    do i=is-1,ie+1 ; max_srt(j) = max(max_srt(j), num_srt(i,j)) ; enddo
  enddo
  !$OMP end parallel

  do j=js,je
    k_size = max(2*max_srt(j),1)
    allocate(deep_wt_Lu(j)%p(IsdB:IedB,k_size))
    allocate(deep_wt_Ru(j)%p(IsdB:IedB,k_size))
    allocate(hP_Lu(j)%p(IsdB:IedB,k_size))
    allocate(hP_Ru(j)%p(IsdB:IedB,k_size))
    allocate(k0a_Lu(j)%p(IsdB:IedB,k_size))
    allocate(k0a_Ru(j)%p(IsdB:IedB,k_size))
    allocate(k0b_Lu(j)%p(IsdB:IedB,k_size))
    allocate(k0b_Ru(j)%p(IsdB:IedB,k_size))
  enddo

!$OMP parallel do default(none) shared(is,ie,js,je,G,num_srt,rho_srt,k0b_Lu,k0_srt, &
!$OMP                                  k0b_Ru,k0a_Lu,k0a_Ru,deep_wt_Lu,deep_wt_Ru,  &
!$OMP                                  h_srt,nkmb,nPu,hP_Lu,hP_Ru)                  &
!$OMP                          private(h_demand_L,h_used_L,h_demand_R,h_used_R,     &
!$OMP                                  kR,kL,nP,rho_pair,kbs_Lp,kbs_Rp,rho_a,rho_b, &
!$OMP                                  wt_b,left_set,right_set,h_supply_frac_R,     &
!$OMP                                  h_supply_frac_L)
  do j=js,je ; do I=is-1,ie ; if (G%mask2dCu(I,j) > 0.0) then
    ! Set up the pairings for fluxes through the zonal faces.

    do k=1,num_srt(i,j)   ; h_demand_L(k) = 0.0 ; h_used_L(k) = 0.0 ; enddo
    do k=1,num_srt(i+1,j) ; h_demand_R(k) = 0.0 ; h_used_R(k) = 0.0 ; enddo

    ! First merge the left and right lists into a single, sorted list.

    !   Discard any layers that are lighter than the lightest in the other
    ! column.  They can only participate in mixing as the lighter part of a
    ! pair of points.
    if (rho_srt(i,1,j) < rho_srt(i+1,1,j)) then
      kR = 1
      do kL=2,num_srt(i,j) ; if (rho_srt(i,kL,j) >= rho_srt(i+1,1,j)) exit ; enddo
    elseif (rho_srt(i+1,1,j) < rho_srt(i,1,j)) then
      kL = 1
      do kR=2,num_srt(i+1,j) ; if (rho_srt(i+1,kR,j) >= rho_srt(i,1,j)) exit ; enddo
    else
      kL = 1 ; kR = 1
    endif
    nP = 0
    do ! Loop to accumulate pairs of columns.
      if ((kL > num_srt(i,j)) .or. (kR > num_srt(i+1,j))) exit

      if (rho_srt(i,kL,j) > rho_srt(i+1,kR,j)) then
      ! The right point is lighter and defines the density for this trio.
        nP = nP+1 ; k = nP
        rho_pair = rho_srt(i+1,kR,j)

        k0b_Lu(j)%p(I,k) = k0_srt(i,kL,j) ; k0b_Ru(j)%p(I,k) = k0_srt(i+1,kR,j)
        k0a_Lu(j)%p(I,k) = k0_srt(i,kL-1,j) ; k0a_Ru(j)%p(I,k) = k0b_Ru(j)%p(I,k)
        kbs_Lp(k) = kL ; kbs_Rp(k) = kR

        rho_a = rho_srt(i,kL-1,j) ; rho_b = rho_srt(i,kL,j)
        wt_b = 1.0 ; if (abs(rho_a - rho_b) > abs(rho_pair - rho_a)) &
          wt_b = (rho_pair - rho_a) / (rho_b - rho_a)
        deep_wt_Lu(j)%p(I,k) = wt_b ; deep_wt_Ru(j)%p(I,k) = 1.0

        h_demand_L(kL) = h_demand_L(kL) + 0.5*h_srt(i+1,kR,j) * wt_b
        h_demand_L(kL-1) = h_demand_L(kL-1) + 0.5*h_srt(i+1,kR,j) * (1.0-wt_b)

        kR = kR+1 ; left_set(k) = .false. ; right_set(k) = .true.
      elseif (rho_srt(i,kL,j) < rho_srt(i+1,kR,j)) then
      ! The left point is lighter and defines the density for this trio.
        nP = nP+1 ; k = nP
        rho_pair = rho_srt(i,kL,j)
        k0b_Lu(j)%p(I,k) = k0_srt(i,kL,j) ; k0b_Ru(j)%p(I,k) = k0_srt(i+1,kR,j)
        k0a_Lu(j)%p(I,k) = k0b_Lu(j)%p(I,k) ; k0a_Ru(j)%p(I,k) = k0_srt(i+1,kR-1,j)

        kbs_Lp(k) = kL ; kbs_Rp(k) = kR

        rho_a = rho_srt(i+1,kR-1,j) ; rho_b = rho_srt(i+1,kR,j)
        wt_b = 1.0 ; if (abs(rho_a - rho_b) > abs(rho_pair - rho_a)) &
          wt_b = (rho_pair - rho_a) / (rho_b - rho_a)
        deep_wt_Lu(j)%p(I,k) = 1.0 ; deep_wt_Ru(j)%p(I,k) = wt_b

        h_demand_R(kR) = h_demand_R(kR) + 0.5*h_srt(i,kL,j) * wt_b
        h_demand_R(kR-1) = h_demand_R(kR-1) + 0.5*h_srt(i,kL,j) * (1.0-wt_b)

        kL = kL+1 ; left_set(k) = .true. ; right_set(k) = .false.
      elseif ((k0_srt(i,kL,j) <= nkmb) .or. (k0_srt(i+1,kR,j) <= nkmb)) then
        ! The densities are exactly equal and one layer is above the interior.
        nP = nP+1 ; k = nP
        k0b_Lu(j)%p(I,k) = k0_srt(i,kL,j) ; k0b_Ru(j)%p(I,k) = k0_srt(i+1,kR,j)
        k0a_Lu(j)%p(I,k) = k0b_Lu(j)%p(I,k) ; k0a_Ru(j)%p(I,k) = k0b_Ru(j)%p(I,k)
        kbs_Lp(k) = kL ; kbs_Rp(k) = kR
        deep_wt_Lu(j)%p(I,k) = 1.0 ; deep_wt_Ru(j)%p(I,k) = 1.0

        h_demand_L(kL) = h_demand_L(kL) + 0.5*h_srt(i+1,kR,j)
        h_demand_R(kR) = h_demand_R(kR) + 0.5*h_srt(i,kL,j)

        kL = kL+1 ; kR = kR+1 ; left_set(k) = .true. ; right_set(k) = .true.
      else ! The densities are exactly equal and in the interior.
        ! Mixing in this case has already occurred, so accumulate the thickness
        ! demanded for that mixing and skip onward.
        h_demand_L(kL) = h_demand_L(kL) + 0.5*h_srt(i+1,kR,j)
        h_demand_R(kR) = h_demand_R(kR) + 0.5*h_srt(i,kL,j)

        kL = kL+1 ; kR = kR+1
      endif
    enddo ! Loop to accumulate pairs of columns.
    nPu(I,j) = nP ! This is the number of active pairings.

    ! Determine what fraction of the thickness "demand" can be supplied.
    do k=1,num_srt(i+1,j)
      h_supply_frac_R(k) = 1.0
      if (h_demand_R(k) > 0.5*h_srt(i+1,k,j)) &
        h_supply_frac_R(k) = 0.5*h_srt(i+1,k,j) / h_demand_R(k)
    enddo
    do k=1,num_srt(i,j)
      h_supply_frac_L(k) = 1.0
      if (h_demand_L(k) > 0.5*h_srt(i,k,j)) &
        h_supply_frac_L(k) = 0.5*h_srt(i,k,j) / h_demand_L(k)
    enddo

    !  Distribute the "exported" thicknesses proportionately.
    do k=1,nPu(I,j)
      kL = kbs_Lp(k) ; kR = kbs_Rp(k)
      hP_Lu(j)%p(I,k) = 0.0 ; hP_Ru(j)%p(I,k) = 0.0
      if (left_set(k)) then ! Add the contributing thicknesses on the right.
        if (deep_wt_Ru(j)%p(I,k) < 1.0) then
          hP_Ru(j)%p(I,k) = 0.5*h_srt(i,kL,j) * min(h_supply_frac_R(kR), h_supply_frac_R(kR-1))
          wt_b = deep_wt_Ru(j)%p(I,k)
          h_used_R(kR-1) = h_used_R(kR-1) + (1.0 - wt_b)*hP_Ru(j)%p(I,k)
          h_used_R(kR) = h_used_R(kR) + wt_b*hP_Ru(j)%p(I,k)
        else
          hP_Ru(j)%p(I,k) = 0.5*h_srt(i,kL,j) * h_supply_frac_R(kR)
          h_used_R(kR) = h_used_R(kR) + hP_Ru(j)%p(I,k)
        endif
      endif
      if (right_set(k)) then ! Add the contributing thicknesses on the left.
        if (deep_wt_Lu(j)%p(I,k) < 1.0) then
          hP_Lu(j)%p(I,k) = 0.5*h_srt(i+1,kR,j) * min(h_supply_frac_L(kL), h_supply_frac_L(kL-1))
          wt_b = deep_wt_Lu(j)%p(I,k)
          h_used_L(kL-1) = h_used_L(kL-1) + (1.0 - wt_b)*hP_Lu(j)%p(I,k)
          h_used_L(kL) = h_used_L(kL) + wt_b*hP_Lu(j)%p(I,k)
        else
          hP_Lu(j)%p(I,k) = 0.5*h_srt(i+1,kR,j) * h_supply_frac_L(kL)
          h_used_L(kL) = h_used_L(kL) + hP_Lu(j)%p(I,k)
        endif
      endif
    enddo

    !   The left-over thickness (at least half the layer thickness) is now
    ! added to the thicknesses of the importing columns.
    do k=1,nPu(I,j)
      if (left_set(k)) hP_Lu(j)%p(I,k) = hP_Lu(j)%p(I,k) + &
                           (h_srt(i,kbs_Lp(k),j) - h_used_L(kbs_Lp(k)))
      if (right_set(k)) hP_Ru(j)%p(I,k) = hP_Ru(j)%p(I,k) + &
                            (h_srt(i+1,kbs_Rp(k),j) - h_used_R(kbs_Rp(k)))
    enddo

  endif ; enddo ; enddo ! i- & j- loops over zonal faces.

  do J=js-1,je
    k_size = max(max_srt(j)+max_srt(j+1),1)
    allocate(deep_wt_Lv(J)%p(isd:ied,k_size))
    allocate(deep_wt_Rv(J)%p(isd:ied,k_size))
    allocate(hP_Lv(J)%p(isd:ied,k_size))
    allocate(hP_Rv(J)%p(isd:ied,k_size))
    allocate(k0a_Lv(J)%p(isd:ied,k_size))
    allocate(k0a_Rv(J)%p(isd:ied,k_size))
    allocate(k0b_Lv(J)%p(isd:ied,k_size))
    allocate(k0b_Rv(J)%p(isd:ied,k_size))
  enddo

!$OMP parallel do default(none) shared(is,ie,js,je,G,num_srt,rho_srt,k0b_Lv,k0b_Rv, &
!$OMP                                  k0_srt,k0a_Lv,k0a_Rv,deep_wt_Lv,deep_wt_Rv,  &
!$OMP                                  h_srt,nkmb,nPv,hP_Lv,hP_Rv)                  &
!$OMP                          private(h_demand_L,h_used_L,h_demand_R,h_used_R,     &
!$OMP                                  kR,kL,nP,rho_pair,kbs_Lp,kbs_Rp,rho_a,rho_b, &
!$OMP                                  wt_b,left_set,right_set,h_supply_frac_R,     &
!$OMP                                  h_supply_frac_L)
  do J=js-1,je ; do i=is,ie ; if (G%mask2dCv(i,J) > 0.0) then
    ! Set up the pairings for fluxes through the meridional faces.

    do k=1,num_srt(i,j)   ; h_demand_L(k) = 0.0 ; h_used_L(k) = 0.0 ; enddo
    do k=1,num_srt(i,j+1) ; h_demand_R(k) = 0.0 ; h_used_R(k) = 0.0 ; enddo

    ! First merge the left and right lists into a single, sorted list.

    !   Discard any layers that are lighter than the lightest in the other
    ! column.  They can only participate in mixing as the lighter part of a
    ! pair of points.
    if (rho_srt(i,1,j) < rho_srt(i,1,j+1)) then
      kR = 1
      do kL=2,num_srt(i,j) ; if (rho_srt(i,kL,j) >= rho_srt(i,1,j+1)) exit ; enddo
    elseif (rho_srt(i,1,j+1) < rho_srt(i,1,j)) then
      kL = 1
      do kR=2,num_srt(i,j+1) ; if (rho_srt(i,kR,j+1) >= rho_srt(i,1,j)) exit ; enddo
    else
      kL = 1 ; kR = 1
    endif
    nP = 0
    do ! Loop to accumulate pairs of columns.
      if ((kL > num_srt(i,j)) .or. (kR > num_srt(i,j+1))) exit

      if (rho_srt(i,kL,j) > rho_srt(i,kR,j+1)) then
      ! The right point is lighter and defines the density for this trio.
        nP = nP+1 ; k = nP
        rho_pair = rho_srt(i,kR,j+1)

        k0b_Lv(J)%p(i,k) = k0_srt(i,kL,j)   ; k0b_Rv(J)%p(i,k) = k0_srt(i,kR,j+1)
        k0a_Lv(J)%p(i,k) = k0_srt(i,kL-1,j) ; k0a_Rv(J)%p(i,k) = k0b_Rv(J)%p(i,k)
        kbs_Lp(k) = kL ; kbs_Rp(k) = kR

        rho_a = rho_srt(i,kL-1,j) ; rho_b = rho_srt(i,kL,j)
        wt_b = 1.0 ; if (abs(rho_a - rho_b) > abs(rho_pair - rho_a)) &
          wt_b = (rho_pair - rho_a) / (rho_b - rho_a)
        deep_wt_Lv(J)%p(i,k) = wt_b ; deep_wt_Rv(J)%p(i,k) = 1.0

        h_demand_L(kL) = h_demand_L(kL) + 0.5*h_srt(i,kR,j+1) * wt_b
        h_demand_L(kL-1) = h_demand_L(kL-1) + 0.5*h_srt(i,kR,j+1) * (1.0-wt_b)

        kR = kR+1 ; left_set(k) = .false. ; right_set(k) = .true.
      elseif (rho_srt(i,kL,j) < rho_srt(i,kR,j+1)) then
      ! The left point is lighter and defines the density for this trio.
        nP = nP+1 ; k = nP
        rho_pair = rho_srt(i,kL,j)
        k0b_Lv(J)%p(i,k) = k0_srt(i,kL,j) ; k0b_Rv(J)%p(i,k) = k0_srt(i,kR,j+1)
        k0a_Lv(J)%p(i,k) = k0b_Lv(J)%p(i,k) ; k0a_Rv(J)%p(i,k) = k0_srt(i,kR-1,j+1)

        kbs_Lp(k) = kL ; kbs_Rp(k) = kR

        rho_a = rho_srt(i,kR-1,j+1) ; rho_b = rho_srt(i,kR,j+1)
        wt_b = 1.0 ; if (abs(rho_a - rho_b) > abs(rho_pair - rho_a)) &
          wt_b = (rho_pair - rho_a) / (rho_b - rho_a)
        deep_wt_Lv(J)%p(i,k) = 1.0 ; deep_wt_Rv(J)%p(i,k) = wt_b

        h_demand_R(kR) = h_demand_R(kR) + 0.5*h_srt(i,kL,j) * wt_b
        h_demand_R(kR-1) = h_demand_R(kR-1) + 0.5*h_srt(i,kL,j) * (1.0-wt_b)

        kL = kL+1 ; left_set(k) = .true. ; right_set(k) = .false.
      elseif ((k0_srt(i,kL,j) <= nkmb) .or. (k0_srt(i,kR,j+1) <= nkmb)) then
        ! The densities are exactly equal and one layer is above the interior.
        nP = nP+1 ; k = nP
        k0b_Lv(J)%p(i,k) = k0_srt(i,kL,j) ; k0b_Rv(J)%p(i,k) = k0_srt(i,kR,j+1)
        k0a_Lv(J)%p(i,k) = k0b_Lv(J)%p(i,k)  ; k0a_Rv(J)%p(i,k) = k0b_Rv(J)%p(i,k)
        kbs_Lp(k) = kL ; kbs_Rp(k) = kR
        deep_wt_Lv(J)%p(i,k) = 1.0 ; deep_wt_Rv(J)%p(i,k) = 1.0

        h_demand_L(kL) = h_demand_L(kL) + 0.5*h_srt(i,kR,j+1)
        h_demand_R(kR) = h_demand_R(kR) + 0.5*h_srt(i,kL,j)

        kL = kL+1 ; kR = kR+1 ; left_set(k) = .true. ; right_set(k) = .true.
      else ! The densities are exactly equal and in the interior.
        ! Mixing in this case has already occurred, so accumulate the thickness
        ! demanded for that mixing and skip onward.
        h_demand_L(kL) = h_demand_L(kL) + 0.5*h_srt(i,kR,j+1)
        h_demand_R(kR) = h_demand_R(kR) + 0.5*h_srt(i,kL,j)

        kL = kL+1 ; kR = kR+1
      endif
    enddo ! Loop to accumulate pairs of columns.
    nPv(i,J) = nP ! This is the number of active pairings.

    ! Determine what fraction of the thickness "demand" can be supplied.
    do k=1,num_srt(i,j+1)
      h_supply_frac_R(k) = 1.0
      if (h_demand_R(k) > 0.5*h_srt(i,k,j+1)) &
        h_supply_frac_R(k) = 0.5*h_srt(i,k,j+1) / h_demand_R(k)
    enddo
    do k=1,num_srt(i,j)
      h_supply_frac_L(k) = 1.0
      if (h_demand_L(k) > 0.5*h_srt(i,k,j)) &
        h_supply_frac_L(k) = 0.5*h_srt(i,k,j) / h_demand_L(k)
    enddo

    !  Distribute the "exported" thicknesses proportionately.
    do k=1,nPv(i,J)
      kL = kbs_Lp(k) ; kR = kbs_Rp(k)
      hP_Lv(J)%p(i,k) = 0.0 ; hP_Rv(J)%p(i,k) = 0.0
      if (left_set(k)) then ! Add the contributing thicknesses on the right.
        if (deep_wt_Rv(J)%p(i,k) < 1.0) then
          hP_Rv(J)%p(i,k) = 0.5*h_srt(i,kL,j) * min(h_supply_frac_R(kR), h_supply_frac_R(kR-1))
          wt_b = deep_wt_Rv(J)%p(i,k)
          h_used_R(kR-1) = h_used_R(kR-1) + (1.0 - wt_b) * hP_Rv(J)%p(i,k)
          h_used_R(kR) = h_used_R(kR) + wt_b * hP_Rv(J)%p(i,k)
        else
          hP_Rv(J)%p(i,k) = 0.5*h_srt(i,kL,j) * h_supply_frac_R(kR)
          h_used_R(kR) = h_used_R(kR) + hP_Rv(J)%p(i,k)
        endif
      endif
      if (right_set(k)) then ! Add the contributing thicknesses on the left.
        if (deep_wt_Lv(J)%p(i,k) < 1.0) then
          hP_Lv(J)%p(i,k) = 0.5*h_srt(i,kR,j+1) * min(h_supply_frac_L(kL), h_supply_frac_L(kL-1))
          wt_b = deep_wt_Lv(J)%p(i,k)
          h_used_L(kL-1) = h_used_L(kL-1) + (1.0 - wt_b) * hP_Lv(J)%p(i,k)
          h_used_L(kL) = h_used_L(kL) + wt_b * hP_Lv(J)%p(i,k)
        else
          hP_Lv(J)%p(i,k) = 0.5*h_srt(i,kR,j+1) * h_supply_frac_L(kL)
          h_used_L(kL) = h_used_L(kL) + hP_Lv(J)%p(i,k)
        endif
      endif
    enddo

    !   The left-over thickness (at least half the layer thickness) is now
    ! added to the thicknesses of the importing columns.
    do k=1,nPv(i,J)
      if (left_set(k)) hP_Lv(J)%p(i,k) = hP_Lv(J)%p(i,k) + &
                            (h_srt(i,kbs_Lp(k),j) - h_used_L(kbs_Lp(k)))
      if (right_set(k)) hP_Rv(J)%p(i,k) = hP_Rv(J)%p(i,k) + &
                             (h_srt(i,kbs_Rp(k),j+1) - h_used_R(kbs_Rp(k)))
    enddo


  endif ; enddo ; enddo ! i- & j- loops over meridional faces.

  ! The tracer-specific calculations start here.

  do itt=1,max_itt

    if (itt > 1) then ! The halos have already been filled if itt==1.
      call do_group_pass(CS%pass_t, G%Domain, clock=id_clock_pass)
    endif

    do m=1,ntr
      ! Zero out tracer tendencies.
      if (CS%answer_date <= 20240330) then
        tr_flux_conv(:,:,:) = 0.0
      else
        tr_flux_N(:,:,:) = 0.0 ; tr_flux_S(:,:,:) = 0.0
        tr_flux_E(:,:,:) = 0.0 ; tr_flux_W(:,:,:) = 0.0
      endif
      tr_flux_3d(:,:,:) = 0.0
      tr_adj_vert_R(:,:,:) = 0.0 ; tr_adj_vert_L(:,:,:) = 0.0

      !$OMP parallel do default(shared) private(Tr_min_face,Tr_max_face,kLa,kLb,kRa,kRb,Tr_La, &
      !$OMP                                     Tr_Lb,Tr_Ra,Tr_Rb,Tr_av_L,wt_b,Tr_av_R,h_L,h_R, &
      !$OMP                                     Tr_flux,Tr_adj_vert,wt_a,vol)
      do j=js,je ; do I=is-1,ie ; if (G%mask2dCu(I,j) > 0.0) then
        ! Determine the fluxes through the zonal faces.

        ! Find the acceptable range of tracer concentration around this face.
        if (nPu(I,j) >= 1) then
          Tr_min_face = min(Tr(m)%t(i,j,1), Tr(m)%t(i+1,j,1))
          Tr_max_face = max(Tr(m)%t(i,j,1), Tr(m)%t(i+1,j,1))
          do k=2,nkmb
            Tr_min_face = min(Tr_min_face, Tr(m)%t(i,j,k), Tr(m)%t(i+1,j,k))
            Tr_max_face = max(Tr_max_face, Tr(m)%t(i,j,k), Tr(m)%t(i+1,j,k))
          enddo

          ! Include the next two layers denser than the densest buffer layer.
          kLa = nkmb+1 ; if (max_kRho(i,j) < nz+1) kLa = max_kRho(i,j)
          kLb = kLa ; if (max_kRho(i,j) < nz) kLb = max_kRho(i,j)+1
          kRa = nkmb+1 ; if (max_kRho(i+1,j) < nz+1) kRa = max_kRho(i+1,j)
          kRb = kRa ; if (max_kRho(i+1,j) < nz) kRb = max_kRho(i+1,j)+1
          Tr_La = Tr_min_face ; Tr_Lb = Tr_La ; Tr_Ra = Tr_La ; Tr_Rb = Tr_La
          if (h(i,j,kLa) > h_exclude) Tr_La = Tr(m)%t(i,j,kLa)
          if ((CS%answer_date <= 20240330) .and. CS%limit_bug) then
            if (h(i,j,kLb) > h_exclude) Tr_La = Tr(m)%t(i,j,kLb)
          else
            if (h(i,j,kLb) > h_exclude) Tr_Lb = Tr(m)%t(i,j,kLb)
          endif
          if (h(i+1,j,kRa) > h_exclude) Tr_Ra = Tr(m)%t(i+1,j,kRa)
          if (h(i+1,j,kRb) > h_exclude) Tr_Rb = Tr(m)%t(i+1,j,kRb)
          Tr_min_face = min(Tr_min_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)
          Tr_max_face = max(Tr_max_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)

          ! Include all points in diffusive pairings at this face.
          do k=1,nPu(I,j)
            Tr_Lb = Tr(m)%t(i,j,k0b_Lu(j)%p(I,k))
            Tr_Rb = Tr(m)%t(i+1,j,k0b_Ru(j)%p(I,k))
            Tr_La = Tr_Lb ; Tr_Ra = Tr_Rb
            if (deep_wt_Lu(j)%p(I,k) < 1.0) Tr_La = Tr(m)%t(i,j,k0a_Lu(j)%p(I,k))
            if (deep_wt_Ru(j)%p(I,k) < 1.0) Tr_Ra = Tr(m)%t(i+1,j,k0a_Ru(j)%p(I,k))
            Tr_min_face = min(Tr_min_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)
            Tr_max_face = max(Tr_max_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)
          enddo
        endif

        do k=1,nPu(I,j)
          kLb = k0b_Lu(j)%p(I,k) ; Tr_Lb = Tr(m)%t(i,j,kLb) ; Tr_av_L = Tr_Lb
          if (deep_wt_Lu(j)%p(I,k) < 1.0) then
            kLa = k0a_Lu(j)%p(I,k) ; Tr_La = Tr(m)%t(i,j,kLa)
            wt_b = deep_wt_Lu(j)%p(I,k)
            Tr_av_L = wt_b*Tr_Lb + (1.0-wt_b)*Tr_La
          endif

          kRb = k0b_Ru(j)%p(I,k) ; Tr_Rb = Tr(m)%t(i+1,j,kRb) ; Tr_av_R = Tr_Rb
          if (deep_wt_Ru(j)%p(I,k) < 1.0) then
            kRa = k0a_Ru(j)%p(I,k) ; Tr_Ra = Tr(m)%t(i+1,j,kRa)
            wt_b = deep_wt_Ru(j)%p(I,k)
            Tr_av_R = wt_b*Tr_Rb + (1.0-wt_b)*Tr_Ra
          endif

          h_L = hP_Lu(j)%p(I,k) ; h_R = hP_Ru(j)%p(I,k)
          if (CS%answer_date <= 20240330) then
            Tr_flux = I_maxitt * khdt_epi_x(I,j) * (Tr_av_L - Tr_av_R) * &
                      ((2.0 * h_L * h_R) / (h_L + h_R))
          else
            Tr_flux = I_maxitt * ((2.0 * h_L * h_R) / (h_L + h_R)) * &
                      khdt_epi_x(I,j) * (Tr_av_L - Tr_av_R)
          endif

          if (deep_wt_Lu(j)%p(I,k) >= 1.0) then
            if (CS%answer_date <= 20240330) then
              tr_flux_conv(i,j,kLb) = tr_flux_conv(i,j,kLb) - Tr_flux
            else
              tr_flux_E(i,j,kLb) = tr_flux_E(i,j,kLb) + Tr_flux
            endif
          else
            Tr_adj_vert = 0.0
            wt_b = deep_wt_Lu(j)%p(I,k) ; wt_a = 1.0 - wt_b
            vol = hP_Lu(j)%p(I,k) * G%areaT(i,j)

            !   Ensure that the tracer flux does not drive the tracer values
            ! outside of the range Tr_min_face <= Tr <= Tr_max_face, or if it
            ! does that the concentration in both contributing pieces exceed
            ! this range equally. With down-gradient fluxes and the initial tracer
            ! concentrations determining the valid range, the latter condition
            ! only enters for large values of the effective diffusive CFL number.
            if (Tr_flux > 0.0) then
              if (Tr_La < Tr_Lb) then ; if (vol*(Tr_La-Tr_min_face) < Tr_flux) &
                Tr_adj_vert = -wt_a * min(Tr_flux - vol * (Tr_La-Tr_min_face), &
                                          (vol*wt_b) * (Tr_Lb - Tr_La))
              else ; if (vol*(Tr_Lb-Tr_min_face) < Tr_flux) &
                Tr_adj_vert = wt_b * min(Tr_flux - vol * (Tr_Lb-Tr_min_face), &
                                         (vol*wt_a) * (Tr_La - Tr_Lb))
              endif
            elseif (Tr_flux < 0.0) then
              if (Tr_La > Tr_Lb) then ; if (vol * (Tr_max_face-Tr_La) < -Tr_flux) &
                Tr_adj_vert = wt_a * min(-Tr_flux - vol * (Tr_max_face-Tr_La), &
                                         (vol*wt_b) * (Tr_La - Tr_Lb))
              else ; if (vol*(Tr_max_face-Tr_Lb) < -Tr_flux) &
                Tr_adj_vert = -wt_b * min(-Tr_flux - vol * (Tr_max_face-Tr_Lb), &
                                          (vol*wt_a)*(Tr_Lb - Tr_La))
              endif
            endif

            if (CS%answer_date <= 20240330) then
              tr_flux_conv(i,j,kLa) = tr_flux_conv(i,j,kLa) - (wt_a*Tr_flux + Tr_adj_vert)
              tr_flux_conv(i,j,kLb) = tr_flux_conv(i,j,kLb) - (wt_b*Tr_flux - Tr_adj_vert)
            else
              tr_flux_E(i,j,kLa) = tr_flux_E(i,j,kLa) + (wt_a*Tr_flux + Tr_adj_vert)
              tr_flux_E(i,j,kLb) = tr_flux_E(i,j,kLb) + (wt_b*Tr_flux - Tr_adj_vert)
            endif
          endif

          if (deep_wt_Ru(j)%p(I,k) >= 1.0) then
            if (CS%answer_date <= 20240330) then
              tr_flux_conv(i+1,j,kRb) = tr_flux_conv(i+1,j,kRb) + Tr_flux
            else
              tr_flux_W(i+1,j,kRb) = tr_flux_W(i+1,j,kRb) + Tr_flux
            endif
          else
            Tr_adj_vert = 0.0
            wt_b = deep_wt_Ru(j)%p(I,k) ; wt_a = 1.0 - wt_b
            vol = hP_Ru(j)%p(I,k) * G%areaT(i+1,j)

            !   Ensure that the tracer flux does not drive the tracer values
            ! outside of the range Tr_min_face <= Tr <= Tr_max_face, or if it
            ! does that the concentration in both contributing pieces exceed
            ! this range equally. With down-gradient fluxes and the initial tracer
            ! concentrations determining the valid range, the latter condition
            ! only enters for large values of the effective diffusive CFL number.
            if (Tr_flux < 0.0) then
              if (Tr_Ra < Tr_Rb) then ; if (vol * (Tr_Ra-Tr_min_face) < -Tr_flux) &
                Tr_adj_vert = -wt_a * min(-Tr_flux - vol * (Tr_Ra-Tr_min_face), &
                                          (vol*wt_b) * (Tr_Rb - Tr_Ra))
              else ; if (vol*(Tr_Rb-Tr_min_face) < (-Tr_flux)) &
                Tr_adj_vert = wt_b * min(-Tr_flux - vol * (Tr_Rb-Tr_min_face), &
                                         (vol*wt_a) * (Tr_Ra - Tr_Rb))
              endif
            elseif (Tr_flux > 0.0) then
              if (Tr_Ra > Tr_Rb) then ; if (vol * (Tr_max_face-Tr_Ra) < Tr_flux) &
                Tr_adj_vert = wt_a * min(Tr_flux - vol * (Tr_max_face-Tr_Ra), &
                                         (vol*wt_b) * (Tr_Ra - Tr_Rb))
              else ; if (vol*(Tr_max_face-Tr_Rb) < Tr_flux) &
                Tr_adj_vert = -wt_b * min(Tr_flux - vol * (Tr_max_face-Tr_Rb), &
                                          (vol*wt_a)*(Tr_Rb - Tr_Ra))
              endif
            endif

            if (CS%answer_date <= 20240330) then
              tr_flux_conv(i+1,j,kRa) = tr_flux_conv(i+1,j,kRa) + (wt_a*Tr_flux - Tr_adj_vert)
              tr_flux_conv(i+1,j,kRb) = tr_flux_conv(i+1,j,kRb) + (wt_b*Tr_flux + Tr_adj_vert)
            else
              tr_flux_W(i+1,j,kRa) = tr_flux_W(i+1,j,kRa) + (wt_a*Tr_flux - Tr_adj_vert)
              tr_flux_W(i+1,j,kRb) = tr_flux_W(i+1,j,kRb) + (wt_b*Tr_flux + Tr_adj_vert)
            endif
          endif
          if (associated(Tr(m)%df2d_x)) &
            Tr(m)%df2d_x(I,j) = Tr(m)%df2d_x(I,j) + Tr_flux * Idt
        enddo ! Loop over pairings at faces.
      endif ; enddo ; enddo ! i- & j- loops over zonal faces.

      !$OMP parallel do default(shared) private(Tr_min_face,Tr_max_face,kLa,kLb,kRa,kRb,             &
      !$OMP                                  Tr_La,Tr_Lb,Tr_Ra,Tr_Rb,Tr_av_L,wt_b,Tr_av_R,        &
      !$OMP                                  h_L,h_R,Tr_flux,Tr_adj_vert,wt_a,vol)
      do J=js-1,je ; do i=is,ie ; if (G%mask2dCv(i,J) > 0.0) then
        ! Determine the fluxes through the meridional faces.

        ! Find the acceptable range of tracer concentration around this face.
        if (nPv(i,J) >= 1) then
          Tr_min_face = min(Tr(m)%t(i,j,1), Tr(m)%t(i,j+1,1))
          Tr_max_face = max(Tr(m)%t(i,j,1), Tr(m)%t(i,j+1,1))
          do k=2,nkmb
            Tr_min_face = min(Tr_min_face, Tr(m)%t(i,j,k), Tr(m)%t(i,j+1,k))
            Tr_max_face = max(Tr_max_face, Tr(m)%t(i,j,k), Tr(m)%t(i,j+1,k))
          enddo

          ! Include the next two layers denser than the densest buffer layer.
          kLa = nkmb+1 ; if (max_kRho(i,j) < nz+1) kLa = max_kRho(i,j)
          kLb = kLa ; if (max_kRho(i,j) < nz) kLb = max_kRho(i,j)+1
          kRa = nkmb+1 ; if (max_kRho(i,j+1) < nz+1) kRa = max_kRho(i,j+1)
          kRb = kRa ; if (max_kRho(i,j+1) < nz) kRb = max_kRho(i,j+1)+1
          Tr_La = Tr_min_face ; Tr_Lb = Tr_La ; Tr_Ra = Tr_La ; Tr_Rb = Tr_La
          if (h(i,j,kLa) > h_exclude) Tr_La = Tr(m)%t(i,j,kLa)
          if ((CS%answer_date <= 20240330) .and. CS%limit_bug) then
            if (h(i,j,kLb) > h_exclude) Tr_La = Tr(m)%t(i,j,kLb)
          else
            if (h(i,j,kLb) > h_exclude) Tr_Lb = Tr(m)%t(i,j,kLb)
          endif
          if (h(i,j+1,kRa) > h_exclude) Tr_Ra = Tr(m)%t(i,j+1,kRa)
          if (h(i,j+1,kRb) > h_exclude) Tr_Rb = Tr(m)%t(i,j+1,kRb)
          Tr_min_face = min(Tr_min_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)
          Tr_max_face = max(Tr_max_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)

          ! Include all points in diffusive pairings at this face.
          do k=1,nPv(i,J)
            Tr_Lb = Tr(m)%t(i,j,k0b_Lv(J)%p(i,k)) ; Tr_Rb = Tr(m)%t(i,j+1,k0b_Rv(J)%p(i,k))
            Tr_La = Tr_Lb ; Tr_Ra = Tr_Rb
            if (deep_wt_Lv(J)%p(i,k) < 1.0) Tr_La = Tr(m)%t(i,j,k0a_Lv(J)%p(i,k))
            if (deep_wt_Rv(J)%p(i,k) < 1.0) Tr_Ra = Tr(m)%t(i,j+1,k0a_Rv(J)%p(i,k))
            Tr_min_face = min(Tr_min_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)
            Tr_max_face = max(Tr_max_face, Tr_La, Tr_Lb, Tr_Ra, Tr_Rb)
          enddo
        endif

        do k=1,nPv(i,J)
          kLb = k0b_Lv(J)%p(i,k) ; Tr_Lb = Tr(m)%t(i,j,kLb) ; Tr_av_L = Tr_Lb
          if (deep_wt_Lv(J)%p(i,k) < 1.0) then
            kLa = k0a_Lv(J)%p(i,k) ; Tr_La = Tr(m)%t(i,j,kLa)
            wt_b = deep_wt_Lv(J)%p(i,k)
            Tr_av_L = wt_b * Tr_Lb + (1.0-wt_b) * Tr_La
          endif

          kRb = k0b_Rv(J)%p(i,k) ; Tr_Rb = Tr(m)%t(i,j+1,kRb) ; Tr_av_R = Tr_Rb
          if (deep_wt_Rv(J)%p(i,k) < 1.0) then
            kRa = k0a_Rv(J)%p(i,k) ; Tr_Ra = Tr(m)%t(i,j+1,kRa)
            wt_b = deep_wt_Rv(J)%p(i,k)
            Tr_av_R = wt_b * Tr_Rb + (1.0-wt_b) * Tr_Ra
          endif

          h_L = hP_Lv(J)%p(i,k) ; h_R = hP_Rv(J)%p(i,k)
          Tr_flux = I_maxitt * ((2.0 * h_L * h_R) / (h_L + h_R)) * &
                    khdt_epi_y(i,J) * (Tr_av_L - Tr_av_R)
          Tr_flux_3d(i,J,k) = Tr_flux

          if (deep_wt_Lv(J)%p(i,k) < 1.0) then
            Tr_adj_vert = 0.0
            wt_b = deep_wt_Lv(J)%p(i,k) ; wt_a = 1.0 - wt_b
            vol = hP_Lv(J)%p(i,k) * G%areaT(i,j)

            !   Ensure that the tracer flux does not drive the tracer values
            ! outside of the range Tr_min_face <= Tr <= Tr_max_face.
            if (Tr_flux > 0.0) then
              if (Tr_La < Tr_Lb) then ; if (vol * (Tr_La-Tr_min_face) < Tr_flux) &
                Tr_adj_vert = -wt_a * min(Tr_flux - vol * (Tr_La-Tr_min_face), &
                                          (vol*wt_b) * (Tr_Lb - Tr_La))
              else ; if (vol*(Tr_Lb-Tr_min_face) < Tr_flux) &
                Tr_adj_vert = wt_b * min(Tr_flux - vol * (Tr_Lb-Tr_min_face), &
                                         (vol*wt_a) * (Tr_La - Tr_Lb))
              endif
            elseif (Tr_flux < 0.0) then
              if (Tr_La > Tr_Lb) then ; if (vol * (Tr_max_face-Tr_La) < -Tr_flux) &
                Tr_adj_vert = wt_a * min(-Tr_flux - vol * (Tr_max_face-Tr_La), &
                                         (vol*wt_b) * (Tr_La - Tr_Lb))
              else ; if (vol*(Tr_max_face-Tr_Lb) < -Tr_flux) &
                Tr_adj_vert = -wt_b * min(-Tr_flux - vol * (Tr_max_face-Tr_Lb), &
                                          (vol*wt_a)*(Tr_Lb - Tr_La))
              endif
            endif
            Tr_adj_vert_L(i,J,k) = Tr_adj_vert
          endif

          if (deep_wt_Rv(J)%p(i,k) < 1.0) then
            Tr_adj_vert = 0.0
            wt_b = deep_wt_Rv(J)%p(i,k) ; wt_a = 1.0 - wt_b
            vol = hP_Rv(J)%p(i,k) * G%areaT(i,j+1)

            !   Ensure that the tracer flux does not drive the tracer values
            ! outside of the range Tr_min_face <= Tr <= Tr_max_face.
            if (Tr_flux < 0.0) then
              if (Tr_Ra < Tr_Rb) then ; if (vol * (Tr_Ra-Tr_min_face) < -Tr_flux) &
                Tr_adj_vert = -wt_a * min(-Tr_flux - vol * (Tr_Ra-Tr_min_face), &
                                          (vol*wt_b) * (Tr_Rb - Tr_Ra))
              else ; if (vol*(Tr_Rb-Tr_min_face) < (-Tr_flux)) &
                Tr_adj_vert = wt_b * min(-Tr_flux - vol * (Tr_Rb-Tr_min_face), &
                                         (vol*wt_a) * (Tr_Ra - Tr_Rb))
              endif
            elseif (Tr_flux > 0.0) then
              if (Tr_Ra > Tr_Rb) then ; if (vol * (Tr_max_face-Tr_Ra) < Tr_flux) &
                Tr_adj_vert = wt_a * min(Tr_flux - vol * (Tr_max_face-Tr_Ra), &
                                         (vol*wt_b) * (Tr_Ra - Tr_Rb))
              else ; if (vol*(Tr_max_face-Tr_Rb) < Tr_flux) &
                Tr_adj_vert = -wt_b * min(Tr_flux - vol * (Tr_max_face-Tr_Rb), &
                                          (vol*wt_a)*(Tr_Rb - Tr_Ra))
              endif
            endif
            Tr_adj_vert_R(i,J,k) = Tr_adj_vert
          endif
          if (associated(Tr(m)%df2d_y)) &
            Tr(m)%df2d_y(i,J) = Tr(m)%df2d_y(i,J) + Tr_flux * Idt
        enddo ! Loop over pairings at faces.
      endif ; enddo ; enddo ! i- & j- loops over meridional faces.

      !$OMP parallel do default(shared) private(kLa,kLb,kRa,kRb,wt_b,wt_a)
      do J=js-1,je ; do i=is,ie ; if (G%mask2dCv(i,J) > 0.0) then
        ! The non-stride-1 loop order here is to facilitate openMP threading. However, it might be
        ! suboptimal when openMP threading is not used, at which point it might be better to fuse
        ! this loop with those that precede it and thereby eliminate the need for three 3-d arrays.
        if (CS%answer_date <= 20240330) then
          do k=1,nPv(i,J)
            kLb = k0b_Lv(J)%p(i,k); kRb = k0b_Rv(J)%p(i,k)
            if (deep_wt_Lv(J)%p(i,k) >= 1.0) then
              tr_flux_conv(i,j,kLb) = tr_flux_conv(i,j,kLb) - Tr_flux_3d(i,J,k)
            else
              kLa = k0a_Lv(J)%p(i,k)
              wt_b = deep_wt_Lv(J)%p(i,k) ; wt_a = 1.0 - wt_b
              tr_flux_conv(i,j,kLa) = tr_flux_conv(i,j,kLa) - (wt_a*Tr_flux_3d(i,J,k) + Tr_adj_vert_L(i,J,k))
              tr_flux_conv(i,j,kLb) = tr_flux_conv(i,j,kLb) - (wt_b*Tr_flux_3d(i,J,k) - Tr_adj_vert_L(i,J,k))
            endif
            if (deep_wt_Rv(J)%p(i,k) >= 1.0) then
              tr_flux_conv(i,j+1,kRb) = tr_flux_conv(i,j+1,kRb) + Tr_flux_3d(i,J,k)
            else
              kRa = k0a_Rv(J)%p(i,k)
              wt_b = deep_wt_Rv(J)%p(i,k) ; wt_a = 1.0 - wt_b
              tr_flux_conv(i,j+1,kRa) = tr_flux_conv(i,j+1,kRa) + &
                                              (wt_a*Tr_flux_3d(i,J,k) - Tr_adj_vert_R(i,J,k))
              tr_flux_conv(i,j+1,kRb) = tr_flux_conv(i,j+1,kRb) + &
                                              (wt_b*Tr_flux_3d(i,J,k) + Tr_adj_vert_R(i,J,k))
            endif
          enddo
        else
          do k=1,nPv(i,J)
            kLb = k0b_Lv(J)%p(i,k); kRb = k0b_Rv(J)%p(i,k)
            if (deep_wt_Lv(J)%p(i,k) >= 1.0) then
              tr_flux_N(i,j,kLb) = tr_flux_N(i,j,kLb) + Tr_flux_3d(i,J,k)
            else
              kLa = k0a_Lv(J)%p(i,k)
              wt_b = deep_wt_Lv(J)%p(i,k) ; wt_a = 1.0 - wt_b
              tr_flux_N(i,j,kLa) = tr_flux_N(i,j,kLa) + (wt_a*Tr_flux_3d(i,J,k) + Tr_adj_vert_L(i,J,k))
              tr_flux_N(i,j,kLb) = tr_flux_N(i,j,kLb) + (wt_b*Tr_flux_3d(i,J,k) - Tr_adj_vert_L(i,J,k))
            endif
            if (deep_wt_Rv(J)%p(i,k) >= 1.0) then
              tr_flux_S(i,j+1,kRb) = tr_flux_S(i,j+1,kRb) + Tr_flux_3d(i,J,k)
            else
              kRa = k0a_Rv(J)%p(i,k)
              wt_b = deep_wt_Rv(J)%p(i,k) ; wt_a = 1.0 - wt_b
              tr_flux_S(i,j+1,kRa) = tr_flux_S(i,j+1,kRa) + (wt_a*Tr_flux_3d(i,J,k) - Tr_adj_vert_R(i,J,k))
              tr_flux_S(i,j+1,kRb) = tr_flux_S(i,j+1,kRb) + (wt_b*Tr_flux_3d(i,J,k) + Tr_adj_vert_R(i,J,k))
            endif
          enddo
        endif
      endif ; enddo ; enddo

      if (CS%answer_date >= 20240331) then
        !$OMP parallel do default(shared)
        do k=1,PEmax_kRho ; do j=js,je ; do i=is,ie
          tr_flux_conv(i,j,k) = ((tr_flux_W(i,j,k) - tr_flux_E(i,j,k)) + &
                                 (tr_flux_S(i,j,k) - tr_flux_N(i,j,k)))
        enddo ; enddo ; enddo
      endif

      !$OMP parallel do default(shared)
      do k=1,PEmax_kRho ; do j=js,je ; do i=is,ie
        if ((G%mask2dT(i,j) > 0.0) .and. (h(i,j,k) > 0.0)) then
          Tr(m)%t(i,j,k) = Tr(m)%t(i,j,k) + tr_flux_conv(i,j,k) / (h(i,j,k)*G%areaT(i,j))
        endif
      enddo ; enddo ; enddo

      ! Do user controlled underflow of the tracer concentrations.
      if (Tr(m)%conc_underflow > 0.0) then
        !$OMP parallel do default(shared)
        do k=1,nz ; do j=js,je ; do i=is,ie
          if (abs(Tr(m)%t(i,j,k)) < Tr(m)%conc_underflow) Tr(m)%t(i,j,k) = 0.0
        enddo ; enddo ; enddo
      endif

    enddo ! Loop over tracers
  enddo ! Loop over iterations

  do j=js,je
    deallocate(deep_wt_Lu(j)%p) ; deallocate(deep_wt_Ru(j)%p)
    deallocate(Hp_Lu(j)%p)  ; deallocate(Hp_Ru(j)%p)
    deallocate(k0a_Lu(j)%p) ; deallocate(k0a_Ru(j)%p)
    deallocate(k0b_Lu(j)%p) ; deallocate(k0b_Ru(j)%p)
  enddo

  do J=js-1,je
    deallocate(deep_wt_Lv(J)%p) ; deallocate(deep_wt_Rv(J)%p)
    deallocate(Hp_Lv(J)%p)  ; deallocate(Hp_Rv(J)%p)
    deallocate(k0a_Lv(J)%p) ; deallocate(k0a_Rv(J)%p)
    deallocate(k0b_Lv(J)%p) ; deallocate(k0b_Rv(J)%p)
  enddo

end subroutine tracer_epipycnal_ML_diff


!> Initialize lateral tracer diffusion module
subroutine tracer_hor_diff_init(Time, G, GV, US, param_file, diag, EOS, diabatic_CSp, CS)
  type(time_type), target,    intent(in)    :: Time       !< current model time
  type(ocean_grid_type),      intent(in)    :: G          !< ocean grid structure
  type(verticalGrid_type),    intent(in)    :: GV         !< ocean vertical grid structure
  type(unit_scale_type),      intent(in)    :: US         !< A dimensional unit scaling type
  type(diag_ctrl), target,    intent(inout) :: diag       !< diagnostic control
  type(EOS_type),  target,    intent(in)    :: EOS        !< Equation of state CS
  type(diabatic_CS), pointer, intent(in)    :: diabatic_CSp !< Equation of state CS
  type(param_file_type),      intent(in)    :: param_file !< parameter file
  type(tracer_hor_diff_CS),   pointer       :: CS         !< horz diffusion control structure

  ! This include declares and sets the variable "version".
# include "version_variable.h"
  character(len=40)  :: mdl = "MOM_tracer_hor_diff" ! This module's name.
  integer :: default_answer_date

  if (associated(CS)) then
    call MOM_error(WARNING, "tracer_hor_diff_init called with associated control structure.")
    return
  endif
  allocate(CS)

  CS%diag => diag
  CS%show_call_tree = callTree_showQuery()

  ! Read all relevant parameters and write them to the model log.
  call log_version(param_file, mdl, version, "")
  call get_param(param_file, mdl, "KHTR", CS%KhTr, &
                 "The background along-isopycnal tracer diffusivity.", &
                 units="m2 s-1", default=0.0, scale=US%m_to_L**2*US%T_to_s)
!  call get_param(param_file, mdl, "KHTR_USE_EBT_STRUCT", CS%KhTh_use_ebt_struct, &
!                 "If true, uses the equivalent barotropic structure "//&
!                 "as the vertical structure of the tracer diffusivity.",&
!                 default=.false.)
  call get_param(param_file, mdl, "KHTR_SLOPE_CFF", CS%KhTr_Slope_Cff, &
                 "The scaling coefficient for along-isopycnal tracer "//&
                 "diffusivity using a shear-based (Visbeck-like) "//&
                 "parameterization.  A non-zero value enables this param.", &
                 units="nondim", default=0.0)
  call get_param(param_file, mdl, "KHTR_MIN", CS%KhTr_Min, &
                 "The minimum along-isopycnal tracer diffusivity.", &
                 units="m2 s-1", default=0.0, scale=US%m_to_L**2*US%T_to_s)
  call get_param(param_file, mdl, "KHTR_MAX", CS%KhTr_Max, &
                 "The maximum along-isopycnal tracer diffusivity.", &
                 units="m2 s-1", default=0.0, scale=US%m_to_L**2*US%T_to_s)
  call get_param(param_file, mdl, "KHTR_PASSIVITY_COEFF", CS%KhTr_passivity_coeff, &
                 "The coefficient that scales deformation radius over "//&
                 "grid-spacing in passivity, where passivity is the ratio "//&
                 "between along isopycnal mixing of tracers to thickness mixing. "//&
                 "A non-zero value enables this parameterization.", &
                 units="nondim", default=0.0)
  call get_param(param_file, mdl, "KHTR_PASSIVITY_MIN", CS%KhTr_passivity_min, &
                 "The minimum passivity which is the ratio between "//&
                 "along isopycnal mixing of tracers to thickness mixing.", &
                 units="nondim", default=0.5)
  call get_param(param_file, mdl, "DIFFUSE_ML_TO_INTERIOR", CS%Diffuse_ML_interior, &
                 "If true, enable epipycnal mixing between the surface "//&
                 "boundary layer and the interior.", default=.false.)
  call get_param(param_file, mdl, "CHECK_DIFFUSIVE_CFL", CS%check_diffusive_CFL, &
                 "If true, use enough iterations the diffusion to ensure "//&
                 "that the diffusive equivalent of the CFL limit is not "//&
                 "violated.  If false, always use the greater of 1 or "//&
                 "MAX_TR_DIFFUSION_CFL iteration.", default=.false.)
  call get_param(param_file, mdl, "MAX_TR_DIFFUSION_CFL", CS%max_diff_CFL, &
                 "If positive, locally limit the along-isopycnal tracer "//&
                 "diffusivity to keep the diffusive CFL locally at or "//&
                 "below this value.  The number of diffusive iterations "//&
                 "is often this value or the next greater integer.", &
                 units="nondim", default=-1.0)
  call get_param(param_File, mdl, "RECALC_NEUTRAL_SURF", CS%recalc_neutral_surf, &
                 "If true, then recalculate the neutral surfaces if the \n"//&
                 "diffusive CFL is exceeded. If false, assume that the  \n"//&
                 "positions of the surfaces do not change \n", default=.false.)
  call get_param(param_file, mdl, "DEFAULT_ANSWER_DATE", default_answer_date, &
                 "This sets the default value for the various _ANSWER_DATE parameters.", &
                 default=99991231, do_not_log=.true.)
  call get_param(param_file, mdl, "HOR_DIFF_ANSWER_DATE", CS%answer_date, &
                 "The vintage of the order of arithmetic to use for the tracer diffusion.  "//&
                 "Values of 20240330 or below recover the answers from the original form of the "//&
                 "along-isopycnal mixed layer to interior mixing code, while higher values use "//&
                 "mathematically equivalent expressions that recover rotational symmetry "//&
                 "when DIFFUSE_ML_TO_INTERIOR is true.", &
                 default=20240101, do_not_log=.not.CS%Diffuse_ML_interior)
                 !### Change the default later to default_answer_date.
  call get_param(param_file, mdl, "HOR_DIFF_LIMIT_BUG", CS%limit_bug, &
                 "If true and the answer date is 20240330 or below, use a rotational symmetry "//&
                 "breaking bug when limiting the tracer properties in tracer_epipycnal_ML_diff.", &
                 default=.true., do_not_log=((.not.CS%Diffuse_ML_interior).or.(CS%answer_date>=20240331)))
  CS%ML_KhTR_scale = 1.0
  if (CS%Diffuse_ML_interior) then
    call get_param(param_file, mdl, "ML_KHTR_SCALE", CS%ML_KhTR_scale, &
                 "With Diffuse_ML_interior, the ratio of the truly "//&
                 "horizontal diffusivity in the mixed layer to the "//&
                 "epipycnal diffusivity.  The valid range is 0 to 1.", &
                 units="nondim", default=1.0)
  endif

  CS%use_neutral_diffusion = neutral_diffusion_init(Time, G, GV, US, param_file, diag, EOS, &
                                                    diabatic_CSp, CS%neutral_diffusion_CSp )
  if (CS%use_neutral_diffusion .and. CS%Diffuse_ML_interior) call MOM_error(FATAL, "MOM_tracer_hor_diff: "// &
       "USE_NEUTRAL_DIFFUSION and DIFFUSE_ML_TO_INTERIOR are mutually exclusive!")
  CS%use_hor_bnd_diffusion = hor_bnd_diffusion_init(Time, G, GV, US, param_file, diag, diabatic_CSp, &
                                                    CS%hor_bnd_diffusion_CSp)
  if (CS%use_hor_bnd_diffusion .and. CS%Diffuse_ML_interior) call MOM_error(FATAL, "MOM_tracer_hor_diff: "// &
       "USE_HORIZONTAL_BOUNDARY_DIFFUSION and DIFFUSE_ML_TO_INTERIOR are mutually exclusive!")

  call get_param(param_file, mdl, "DEBUG", CS%debug, default=.false.)

  id_clock_diffuse = cpu_clock_id('(Ocean diffuse tracer)',          grain=CLOCK_MODULE)
  id_clock_epimix  = cpu_clock_id('(Ocean epipycnal diffuse tracer)',grain=CLOCK_MODULE)
  id_clock_pass    = cpu_clock_id('(Ocean tracer halo updates)',     grain=CLOCK_ROUTINE)
  id_clock_sync    = cpu_clock_id('(Ocean tracer global synch)',     grain=CLOCK_ROUTINE)

  CS%id_KhTr_u = -1
  CS%id_KhTr_v = -1
  CS%id_KhTr_h = -1
  CS%id_CFL    = -1

  CS%id_KhTr_u = register_diag_field('ocean_model', 'KHTR_u', diag%axesCui, Time, &
      'Epipycnal tracer diffusivity at zonal faces of tracer cell', 'm2 s-1', conversion=US%L_to_m**2*US%s_to_T)
  CS%id_KhTr_v = register_diag_field('ocean_model', 'KHTR_v', diag%axesCvi, Time, &
      'Epipycnal tracer diffusivity at meridional faces of tracer cell', 'm2 s-1', conversion=US%L_to_m**2*US%s_to_T)
  CS%id_KhTr_h = register_diag_field('ocean_model', 'KHTR_h', diag%axesTi, Time, &
      'Epipycnal tracer diffusivity at tracer cell center', 'm2 s-1', conversion=US%L_to_m**2*US%s_to_T, &
      cmor_field_name='diftrelo',                                                &
      cmor_standard_name= 'ocean_tracer_epineutral_laplacian_diffusivity',       &
      cmor_long_name = 'Ocean Tracer Epineutral Laplacian Diffusivity')

  CS%id_khdt_x = register_diag_field('ocean_model', 'KHDT_x', diag%axesCu1, Time, &
      'Epipycnal tracer diffusivity operator at zonal faces of tracer cell', 'm2', conversion=US%L_to_m**2)
  CS%id_khdt_y = register_diag_field('ocean_model', 'KHDT_y', diag%axesCv1, Time, &
      'Epipycnal tracer diffusivity operator at meridional faces of tracer cell', 'm2', conversion=US%L_to_m**2)
  if (CS%check_diffusive_CFL) then
    CS%id_CFL = register_diag_field('ocean_model', 'CFL_lateral_diff', diag%axesT1, Time,&
       'Grid CFL number for lateral/neutral tracer diffusion', 'nondim')
  endif


end subroutine tracer_hor_diff_init

subroutine tracer_hor_diff_end(CS)
  type(tracer_hor_diff_CS), pointer :: CS !< module control structure

  call neutral_diffusion_end(CS%neutral_diffusion_CSp)
  call hor_bnd_diffusion_end(CS%hor_bnd_diffusion_CSp)
  if (associated(CS)) deallocate(CS)

end subroutine tracer_hor_diff_end


!> \namespace mom_tracer_hor_diff
!!
!! \section section_intro Introduction to the module
!!
!!    This module contains subroutines that handle horizontal
!!  diffusion (i.e., isoneutral or along layer) of tracers.
!!
!!    Each of the tracers are subject to Fickian along-coordinate
!!  diffusion if Khtr is defined and positive.  The tracer diffusion
!!  can use a suitable number of iterations to guarantee stability
!!  with an arbitrarily large time step.

end module MOM_tracer_hor_diff
