!> Reports integrated quantities for monitoring the model state
module MOM_sum_output

! This file is part of MOM6. See LICENSE.md for the license.

use iso_fortran_env, only : int64
use MOM_checksums,     only : is_NaN
use MOM_coms,          only : sum_across_PEs, PE_here, root_PE, num_PEs, max_across_PEs, field_chksum
use MOM_coms,          only : reproducing_sum, reproducing_sum_EFP, EFP_to_real, real_to_EFP
use MOM_coms,          only : EFP_type, operator(+), operator(-), assignment(=), EFP_sum_across_PEs
use MOM_error_handler, only : MOM_error, FATAL, WARNING, NOTE, is_root_pe
use MOM_file_parser,   only : get_param, log_param, log_version, param_file_type
use MOM_forcing_type,  only : forcing
use MOM_grid,          only : ocean_grid_type
use MOM_interface_heights, only : find_eta
use MOM_io,            only : create_MOM_file, reopen_MOM_file
use MOM_io,            only : MOM_infra_file, MOM_netcdf_file, MOM_field
use MOM_io,            only : file_exists, slasher, vardesc, var_desc, MOM_write_field
use MOM_io,            only : field_size, read_variable, read_attribute, open_ASCII_file, stdout
use MOM_io,            only : axis_info, set_axis_info, delete_axis_info, get_filename_appendix
use MOM_io,            only : attribute_info, set_attribute_info, delete_attribute_info
use MOM_io,            only : APPEND_FILE, SINGLE_FILE, WRITEONLY_FILE
use MOM_spatial_means, only : array_global_min_max
use MOM_time_manager,  only : time_type, get_time, get_date, set_time
use MOM_time_manager,  only : operator(+), operator(-), operator(*), operator(/)
use MOM_time_manager,  only : operator(/=), operator(<=), operator(>=), operator(<), operator(>)
use MOM_time_manager,  only : get_calendar_type, time_type_to_real, NO_CALENDAR
use MOM_tracer_flow_control, only : tracer_flow_control_CS, call_tracer_stocks
use MOM_unit_scaling,  only : unit_scale_type
use MOM_variables,     only : surface, thermo_var_ptrs
use MOM_verticalGrid,  only : verticalGrid_type

implicit none ; private

#include <MOM_memory.h>

public write_energy, accumulate_net_input
public MOM_sum_output_init, MOM_sum_output_end

! A note on unit descriptions in comments: MOM6 uses units that can be rescaled for dimensional
! consistency testing. These are noted in comments with units like Z, H, L, and T, along with
! their mks counterparts with notation like "a velocity [Z T-1 ~> m s-1]".  If the units
! vary with the Boussinesq approximation, the Boussinesq variant is given first.

integer, parameter :: NUM_FIELDS = 17 !< Number of diagnostic fields
character (*), parameter :: depth_chksum_attr = "bathyT_checksum"
                                      !< Checksum attribute name of G%bathyT
                                      !! over the compute domain
character (*), parameter :: area_chksum_attr = "mask2dT_areaT_checksum"
                                      !< Checksum attribute of name of
                                      !! G%mask2dT * G%areaT over the compute
                                      !! domain

!> A list of depths and corresponding globally integrated ocean area at each
!! depth and the ocean volume below each depth.
type :: Depth_List
  integer                         :: listsize  !< length of the list <= niglobal*njglobal + 1
  real, allocatable, dimension(:) :: depth     !< A list of depths [Z ~> m]
  real, allocatable, dimension(:) :: area      !< The cross-sectional area of the ocean at that depth [L2 ~> m2]
  real, allocatable, dimension(:) :: vol_below !< The ocean volume below that depth [Z L2 ~> m3]
end type Depth_List

!> The control structure for the MOM_sum_output module
type, public :: sum_output_CS ; private
  logical :: initialized = .false. !< True if this control structure has been initialized.

  type(Depth_List)              :: DL !< The sorted depth list.

  integer, allocatable, dimension(:) :: lH
                                !< This saves the entry in DL with a volume just
                                !! less than the volume of fluid below the interface.
  logical :: do_APE_calc        !<   If true, calculate the available potential energy of the
                                !! interfaces.  Disabling this reduces the memory footprint of
                                !! high-PE-count models dramatically.
  logical :: read_depth_list    !<   Read the depth list from a file if it exists
                                !! and write it if it doesn't.
  character(len=200) :: depth_list_file  !< The name of the depth list file.
  real    :: D_list_min_inc     !<  The minimum increment [Z ~> m], between the depths of the
                                !! entries in the depth-list file, 0 by default.
  logical :: require_depth_list_chksum
                                !< Require matching checksums in Depth_list.nc when reading
                                !! the file.
  logical :: update_depth_list_chksum
                                !< Automatically update the Depth_list.nc file if the
                                !! checksums are missing or do not match current values.
  logical :: use_temperature    !<   If true, temperature and salinity are state variables.
  type(EFP_type) :: fresh_water_in_EFP !< The total mass of fresh water added by surface fluxes on
                                  !! this PE since the last time that write_energy was called [kg].
  type(EFP_type) :: net_salt_in_EFP !< The total salt added by surface fluxes on this PE since
                                  !! the last time that write_energy was called [ppt kg].
  type(EFP_type) :: net_heat_in_EFP !<  The total heat added by surface fluxes on this PE since
                                  !! the last time that write_energy was called [J].
  type(EFP_type) :: heat_prev_EFP !<  The total amount of heat in the ocean the last
                                  !! time that write_energy was called [J].
  type(EFP_type) :: salt_prev_EFP !< The total amount of salt in the ocean the last
                                  !! time that write_energy was called [ppt kg].
  type(EFP_type) :: mass_prev_EFP !< The total ocean mass the last time that
                                  !! write_energy was called [kg].
  real    :: dt_in_T            !< The baroclinic dynamics time step [T ~> s].

  type(time_type) :: energysavedays            !< The interval between writing the energies
                                               !! and other integral quantities of the run.
  type(time_type) :: energysavedays_geometric  !< The starting interval for computing a geometric
                                               !! progression of time deltas between calls to
                                               !! write_energy. This interval will increase by a factor of 2.
                                               !! after each call to write_energy.
  logical         :: energysave_geometric      !< Logical to control whether calls to write_energy should
                                               !! follow a geometric progression
  type(time_type) :: write_energy_time         !< The next time to write to the energy file.
  type(time_type) :: geometric_end_time        !< Time at which to stop the geometric progression
                                               !! of calls to write_energy and revert to the standard
                                               !! energysavedays interval

  real    :: timeunit           !< The length of the units for the time axis and certain input parameters
                                !! including ENERGYSAVEDAYS [s].

  logical :: date_stamped_output !< If true, use dates (not times) in messages to stdout.
  type(time_type) :: Start_time !< The start time of the simulation.
                                ! Start_time is set in MOM_initialization.F90
  integer, pointer :: ntrunc => NULL() !< The number of times the velocity has been
                                !! truncated since the last call to write_energy.
  real    :: max_Energy         !< The maximum permitted energy per unit mass.  If there is
                                !! more energy than this, the model should stop [L2 T-2 ~> m2 s-2].
  integer :: maxtrunc           !< The number of truncations per energy save
                                !! interval at which the run is stopped.
  logical :: write_stocks       !< If true, write the integrated tracer amounts
                                !! to stdout when the energy files are written.
  logical :: write_min_max      !< If true, write the maximum and minimum values of temperature,
                                !! salinity and some tracer concentrations to stdout when the energy
                                !! files are written.
  logical :: write_min_max_loc  !< If true, write the locations of the maximum and minimum values
                                !! of temperature, salinity and some tracer concentrations to stdout
                                !! when the energy files are written.
  integer :: previous_calls = 0 !< The number of times write_energy has been called.
  integer :: prev_n = 0         !< The value of n from the last call.
  type(MOM_netcdf_file) :: fileenergy_nc !< The file handle for the netCDF version of the energy file.
  integer :: fileenergy_ascii   !< The unit number of the ascii version of the energy file.
  type(MOM_field), dimension(NUM_FIELDS+MAX_FIELDS_) :: &
             fields             !< fieldtype variables for the output fields.
  character(len=200) :: energyfile  !< The name of the energy file with path.
end type sum_output_CS

contains

!> MOM_sum_output_init initializes the parameters and settings for the MOM_sum_output module.
subroutine MOM_sum_output_init(G, GV, US, param_file, directory, ntrnc, &
                               Input_start_time, CS)
  type(ocean_grid_type),   intent(in)    :: G          !< The ocean's grid structure.
  type(verticalGrid_type), intent(in)    :: GV         !< The ocean's vertical grid structure.
  type(unit_scale_type),   intent(in)    :: US         !< A dimensional unit scaling type
  type(param_file_type),   intent(in)    :: param_file !< A structure to parse for run-time
                                                       !! parameters.
  character(len=*),        intent(in)    :: directory  !< The directory where the energy file goes.
  integer, target,         intent(inout) :: ntrnc      !< The integer that stores the number of times
                                                       !! the velocity has been truncated since the
                                                       !! last call to write_energy.
  type(time_type),         intent(in)    :: Input_start_time !< The start time of the simulation.
  type(Sum_output_CS),     pointer       :: CS         !< A pointer that is set to point to the
                                                       !! control structure for this module.
  ! Local variables
  real :: maxvel    ! The maximum permitted velocity [L T-1 ~> m s-1]
  ! This include declares and sets the variable "version".
# include "version_variable.h"
  character(len=40)  :: mdl = "MOM_sum_output" ! This module's name.
  character(len=200) :: energyfile  ! The name of the energy file.
  character(len=32) :: filename_appendix = '' ! FMS appendix to filename for ensemble runs

  if (associated(CS)) then
    call MOM_error(WARNING, "MOM_sum_output_init called with associated control structure.")
    return
  endif
  allocate(CS)

  CS%initialized = .true.

  ! Read all relevant parameters and write them to the model log.
  call log_version(param_file, mdl, version, "")
  call get_param(param_file, mdl, "CALCULATE_APE", CS%do_APE_calc, &
                 "If true, calculate the available potential energy of "//&
                 "the interfaces.  Setting this to false reduces the "//&
                 "memory footprint of high-PE-count models dramatically.", &
                 default=.true.)
  call get_param(param_file, mdl, "WRITE_STOCKS", CS%write_stocks, &
                 "If true, write the integrated tracer amounts to stdout "//&
                 "when the energy files are written.", default=.true.)
  call get_param(param_file, mdl, "ENABLE_THERMODYNAMICS", CS%use_temperature, &
                 "If true, Temperature and salinity are used as state "//&
                 "variables.", default=.true.)
  call get_param(param_file, mdl, "WRITE_TRACER_MIN_MAX", CS%write_min_max, &
                 "If true, write the maximum and minimum values of temperature, salinity and "//&
                 "some tracer concentrations to stdout when the energy files are written.", &
                 default=.false., do_not_log=.not.CS%write_stocks, debuggingParam=.true.)
  call get_param(param_file, mdl, "WRITE_TRACER_MIN_MAX_LOC", CS%write_min_max_loc, &
                 "If true, write the locations of the maximum and minimum values of "//&
                 "temperature, salinity and some tracer concentrations to stdout when the "//&
                 "energy files are written.", &
                 default=.false., do_not_log=.not.CS%write_min_max, debuggingParam=.true.)
  call get_param(param_file, mdl, "DT", CS%dt_in_T, &
                 "The (baroclinic) dynamics time step.", &
                 units="s", scale=US%s_to_T, fail_if_missing=.true.)
  call get_param(param_file, mdl, "MAXTRUNC", CS%maxtrunc, &
                 "The run will be stopped, and the day set to a very "//&
                 "large value if the velocity is truncated more than "//&
                 "MAXTRUNC times between energy saves.  Set MAXTRUNC to 0 "//&
                 "to stop if there is any truncation of velocities.", &
                 units="truncations save_interval-1", default=0)

  call get_param(param_file, mdl, "MAX_ENERGY", CS%max_Energy, &
                 "The maximum permitted average energy per unit mass; the "//&
                 "model will be stopped if there is more energy than "//&
                 "this.  If zero or negative, this is set to 10*MAXVEL^2.", &
                 units="m2 s-2", default=0.0, scale=US%m_s_to_L_T**2)
  if (CS%max_Energy <= 0.0) then
    call get_param(param_file, mdl, "MAXVEL", maxvel, &
                 "The maximum velocity allowed before the velocity "//&
                 "components are truncated.", units="m s-1", default=3.0e8, scale=US%m_s_to_L_T)
    CS%max_Energy = 10.0 * maxvel**2
    call log_param(param_file, mdl, "MAX_ENERGY as used", CS%max_Energy, &
                   units="m2 s-2", unscale=US%L_T_to_m_s**2)
  endif

  call get_param(param_file, mdl, "ENERGYFILE", energyfile, &
                 "The file to use to write the energies and globally "//&
                 "summed diagnostics.", default="ocean.stats")

  !query fms_io if there is a filename_appendix (for ensemble runs)
  call get_filename_appendix(filename_appendix)
  if (len_trim(filename_appendix) > 0) then
    energyfile = trim(energyfile) //'.'//trim(filename_appendix)
  endif

  CS%energyfile = trim(slasher(directory))//trim(energyfile)
  call log_param(param_file, mdl, "output_path/ENERGYFILE", CS%energyfile)
#ifdef STATSLABEL
  CS%energyfile = trim(CS%energyfile)//"."//trim(adjustl(STATSLABEL))
#endif

  call get_param(param_file, mdl, "DATE_STAMPED_STDOUT", CS%date_stamped_output, &
                 "If true, use dates (not times) in messages to stdout", &
                 default=.true.)
  ! Note that the units of CS%Timeunit are the MKS units of [s].
  call get_param(param_file, mdl, "TIMEUNIT", CS%Timeunit, &
                 "The time unit in seconds a number of input fields", &
                 units="s", default=86400.0)
  if (CS%Timeunit < 0.0) CS%Timeunit = 86400.0

  if (CS%do_APE_calc) then
    call get_param(param_file, mdl, "READ_DEPTH_LIST", CS%read_depth_list, &
                   "Read the depth list from a file if it exists or "//&
                   "create that file otherwise.", default=.false.)
    call get_param(param_file, mdl, "DEPTH_LIST_MIN_INC", CS%D_list_min_inc, &
                   "The minimum increment between the depths of the "//&
                   "entries in the depth-list file.", &
                   units="m", default=1.0E-10, scale=US%m_to_Z)
    if (CS%read_depth_list) then
      call get_param(param_file, mdl, "DEPTH_LIST_FILE", CS%depth_list_file, &
                   "The name of the depth list file.", default="Depth_list.nc")
      if (scan(CS%depth_list_file,'/') == 0) &
        CS%depth_list_file = trim(slasher(directory))//trim(CS%depth_list_file)

      call get_param(param_file, mdl, "REQUIRE_DEPTH_LIST_CHECKSUMS", &
                     CS%require_depth_list_chksum, &
                 "Require that matching checksums be in Depth_list.nc "//&
                 "when reading the file.", default=.true.)
      if (.not. CS%require_depth_list_chksum) &
        call get_param(param_file, mdl, "UPDATE_DEPTH_LIST_CHECKSUMS", &
                     CS%update_depth_list_chksum, &
                 "Automatically update the Depth_list.nc file if the "//&
                 "checksums are missing or do not match current values.", &
                 default=.false.)
    endif

    allocate(CS%lH(GV%ke))
    call depth_list_setup(G, GV, US, CS%DL, CS)
  else
    CS%DL%listsize = 1
  endif

  call get_param(param_file, mdl, "ENERGYSAVEDAYS",CS%energysavedays, &
                 "The interval in units of TIMEUNIT between saves of the "//&
                 "energies of the run and other globally summed diagnostics.",&
                 default=set_time(0,days=1), timeunit=CS%Timeunit)
  call get_param(param_file, mdl, "ENERGYSAVEDAYS_GEOMETRIC",CS%energysavedays_geometric, &
                 "The starting interval in units of TIMEUNIT for the first call "//&
                 "to save the energies of the run and other globally summed diagnostics. "//&
                 "The interval increases by a factor of 2. after each call to write_energy.",&
                 default=set_time(seconds=0), timeunit=CS%Timeunit)

  if ((time_type_to_real(CS%energysavedays_geometric) > 0.) .and. &
     (CS%energysavedays_geometric < CS%energysavedays)) then
         CS%energysave_geometric = .true.
  else
         CS%energysave_geometric = .false.
  endif

  CS%Start_time = Input_start_time
  CS%ntrunc => ntrnc

end subroutine MOM_sum_output_init

!> MOM_sum_output_end deallocates memory used by the MOM_sum_output module.
subroutine MOM_sum_output_end(CS)
  type(Sum_output_CS), pointer :: CS  !< The control structure returned by a
                                      !! previous call to MOM_sum_output_init.
  if (associated(CS)) then
    if (CS%do_APE_calc) then
      deallocate(CS%DL%depth, CS%DL%area, CS%DL%vol_below)
      deallocate(CS%lH)
    endif

    deallocate(CS)
  endif
end subroutine MOM_sum_output_end

!>  This subroutine calculates and writes the total model energy, the energy and
!! mass of each layer, and other globally integrated  physical quantities.
subroutine write_energy(u, v, h, tv, day, n, G, GV, US, CS, tracer_CSp, dt_forcing)
  type(ocean_grid_type),   intent(in)    :: G   !< The ocean's grid structure.
  type(verticalGrid_type), intent(in)    :: GV  !< The ocean's vertical grid structure.
  type(unit_scale_type),   intent(in)    :: US  !< A dimensional unit scaling type
  real, dimension(SZIB_(G),SZJ_(G),SZK_(GV)), &
                           intent(in)    :: u   !< The zonal velocity [L T-1 ~> m s-1].
  real, dimension(SZI_(G),SZJB_(G),SZK_(GV)), &
                           intent(in)    :: v   !< The meridional velocity [L T-1 ~> m s-1].
  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)), &
                           intent(in)    :: h   !< Layer thicknesses [H ~> m or kg m-2].
  type(thermo_var_ptrs),   intent(in)    :: tv  !< A structure pointing to various
                                                !! thermodynamic variables.
  type(time_type),         intent(in)    :: day !< The current model time.
  integer,                 intent(in)    :: n   !< The time step number of the
                                                !! current execution.
  type(Sum_output_CS),     pointer       :: CS  !< The control structure returned by a
                                                !! previous call to MOM_sum_output_init.
  type(tracer_flow_control_CS), pointer  :: tracer_CSp !< Control structure with the tree of
                                                !! all registered tracer packages
  type(time_type),  optional, intent(in) :: dt_forcing !< The forcing time step

  ! Local variables
  real :: eta(SZI_(G),SZJ_(G),SZK_(GV)+1) ! The height of interfaces [Z ~> m].
  real :: areaTm(SZI_(G),SZJ_(G)) ! A masked version of areaT [L2 ~> m2].
  real :: KE(SZK_(GV)) ! The total kinetic energy of a layer [J].
  real :: PE(SZK_(GV)+1)! The available potential energy of an interface [J].
  real :: KE_tot       ! The total kinetic energy [R Z L4 T-2 ~> J].
  real :: PE_tot       ! The total available potential energy [R Z L4 T-2 ~> J].
  real :: Z_0APE(SZK_(GV)+1) ! The uniform depth which overlies the same
                       ! volume as is below an interface [Z ~> m].
  real :: H_0APE(SZK_(GV)+1) ! A version of Z_0APE, converted to m, usually positive [m].
  real :: toten        ! The total kinetic & potential energies of
                       ! all layers [J] (i.e. kg m2 s-2).
  real :: En_mass      ! The total kinetic and potential energies divided by
                       ! the total mass of the ocean [m2 s-2].
  real :: vol_lay(SZK_(GV)) ! The volume of fluid in a layer [Z L2 ~> m3].
  real :: volbelow     ! The volume of all layers beneath an interface [Z L2 ~> m3].
  real :: mass_lay(SZK_(GV)) ! The mass of fluid in a layer [R Z L2 ~> kg].
  real :: mass_tot_RZL2   ! The total mass of the ocean [R Z L2 ~> kg].
  real :: mass_tot     ! The total mass of the ocean [kg].
  real :: vol_tot      ! The total ocean volume [Z L2 ~> m3].
  real :: mass_chg     ! The change in total ocean mass of fresh water since
                       ! the last call to this subroutine [kg].
  real :: mass_anom    ! The change in fresh water that cannot be accounted for
                       ! by the surface fluxes [kg].
  real :: Salt         ! The total amount of salt in the ocean [ppt kg].
  real :: Salt_chg     ! The change in total ocean salt since the last call
                       ! to this subroutine [ppt kg].
  real :: Salt_anom    ! The change in salt that cannot be accounted for by
                       ! the surface fluxes [ppt kg].
  real :: salin        ! The mean salinity of the ocean [ppt].
  real :: salin_anom   ! The change in total salt that cannot be accounted for by
                       ! the surface fluxes divided by total mass [ppt].
  real :: Heat         ! The total amount of Heat in the ocean [J].
  real :: Heat_chg     ! The change in total ocean heat since the last call to this subroutine [J].
  real :: Heat_anom    ! The change in heat that cannot be accounted for by the surface fluxes [J].
  real :: temp         ! The mean potential temperature of the ocean [degC].
  real :: temp_anom    ! The change in total heat that cannot be accounted for
                       ! by the surface fluxes, divided by the total heat
                       ! capacity of the ocean [degC].
  real :: hint         ! The deviation of an interface from H [Z ~> m].
  real :: hbot         ! 0 if the basin is deeper than H, or the
                       ! height of the basin depth over H otherwise [Z ~> m].
                       ! This makes PE only include real fluid.
  real :: hbelow       ! The depth of fluid in all layers beneath an interface [Z ~> m].
  type(EFP_type) :: &
    mass_EFP, &        ! The total mass of the ocean in extended fixed point form [kg].
    salt_EFP, &        ! The total amount of salt in the ocean in extended fixed point form [ppt kg].
    heat_EFP, &        ! The total amount of heat in the ocean in extended fixed point form [J].
    salt_chg_EFP, &    ! The change in total ocean salt since the last call to this subroutine [ppt kg].
    heat_chg_EFP, &    ! The change in total ocean heat since the last call to this subroutine [J].
    mass_chg_EFP, &    ! The change in total ocean mass of fresh water since
                       ! the last call to this subroutine [kg].
    salt_anom_EFP, &   ! The change in salt that cannot be accounted for by the surface
                       ! fluxes [ppt kg].
    heat_anom_EFP, &   ! The change in heat that cannot be accounted for by the surface fluxes [J].
    mass_anom_EFP      ! The change in fresh water that cannot be accounted for by the surface
                       ! fluxes [kg].
  type(EFP_type), dimension(5) :: EFP_list ! An array of EFP types for joint global sums.
  real :: CFL_Iarea    ! Direction-based inverse area used in CFL test [L-2 ~> m-2].
  real :: CFL_trans    ! A transport-based definition of the CFL number [nondim].
  real :: CFL_lin      ! A simpler definition of the CFL number [nondim].
  real :: max_CFL(2)   ! The maxima of the CFL numbers [nondim].
  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)) :: &
    tmp1               ! A temporary array used in reproducing sums [various]
  real, dimension(SZI_(G),SZJ_(G),SZK_(GV)+1) :: &
    PE_pt              ! The potential energy at each point [R Z L4 T-2 ~> J].
  real, dimension(SZI_(G),SZJ_(G)) :: &
    Temp_int, Salt_int ! Layer and cell integrated heat and salt [Q R Z L2 ~> J] and [g Salt].
  real :: RZL4_to_J    ! The combination of unit rescaling factors to convert the spatially integrated
                       ! kinetic or potential energies into mks units [T2 kg m2 R-1 Z-1 L-4 s-2 ~> 1]
  integer :: num_nc_fields  ! The number of fields that will actually go into
                            ! the NetCDF file.
  integer :: i, j, k, is, ie, js, je, nz, m, Isq, Ieq, Jsq, Jeq, isr, ier, jsr, jer
  integer :: li, lbelow, labove  ! indices of deep_area_vol, used to find Z_0APE.
                                 ! lbelow & labove are lower & upper limits for li
                                 ! in the search for the entry in lH to use.
  integer :: start_of_day, num_days
  real    :: reday  ! Time in units given by CS%Timeunit, but often [days]
  character(len=240) :: energypath_nc
  character(len=200) :: mesg
  character(len=32)  :: mesg_intro, time_units, day_str, n_str, date_str
  logical :: date_stamped
  type(time_type) :: dt_force ! A time_type version of the forcing timestep.

  real :: S_min   ! The global minimum unmasked value of the salinity [ppt]
  real :: S_max   ! The global maximum unmasked value of the salinity [ppt]
  real :: S_min_x ! The x-positions of the global salinity minima
                  ! in the units of G%geoLonT, often [degrees_E] or [km]
  real :: S_min_y ! The y-positions of the global salinity minima
                  ! in the units of G%geoLatT, often [degrees_N] or [km]
  real :: S_min_z ! The z-positions of the global salinity minima [layer]
  real :: S_max_x ! The x-positions of the global salinity maxima
                  ! in the units of G%geoLonT, often [degrees_E] or [km]
  real :: S_max_y ! The y-positions of the global salinity maxima
                  ! in the units of G%geoLatT, often [degrees_N] or [km]
  real :: S_max_z ! The z-positions of the global salinity maxima [layer]

  real :: T_min   ! The global minimum unmasked value of the temperature [degC]
  real :: T_max   ! The global maximum unmasked value of the temperature [degC]
  real :: T_min_x ! The x-positions of the global temperature minima
                  ! in the units of G%geoLonT, often [degreeT_E] or [km]
  real :: T_min_y ! The y-positions of the global temperature minima
                  ! in the units of G%geoLatT, often [degreeT_N] or [km]
  real :: T_min_z ! The z-positions of the global temperature minima [layer]
  real :: T_max_x ! The x-positions of the global temperature maxima
                  ! in the units of G%geoLonT, often [degreeT_E] or [km]
  real :: T_max_y ! The y-positions of the global temperature maxima
                  ! in the units of G%geoLatT, often [degreeT_N] or [km]
  real :: T_max_z ! The z-positions of the global temperature maxima [layer]


  ! The units of the tracer stock vary between tracers, with [conc] given explicitly by Tr_units.
  real :: Tr_stocks(MAX_FIELDS_) ! The total amounts of each of the registered tracers [kg conc]
  real :: Tr_min(MAX_FIELDS_)   ! The global minimum unmasked value of the tracers [conc]
  real :: Tr_max(MAX_FIELDS_)   ! The global maximum unmasked value of the tracers [conc]
  real :: Tr_min_x(MAX_FIELDS_) ! The x-positions of the global tracer minima
                                ! in the units of G%geoLonT, often [degrees_E] or [km]
  real :: Tr_min_y(MAX_FIELDS_) ! The y-positions of the global tracer minima
                                ! in the units of G%geoLatT, often [degrees_N] or [km]
  real :: Tr_min_z(MAX_FIELDS_) ! The z-positions of the global tracer minima [layer]
  real :: Tr_max_x(MAX_FIELDS_) ! The x-positions of the global tracer maxima
                                ! in the units of G%geoLonT, often [degrees_E] or [km]
  real :: Tr_max_y(MAX_FIELDS_) ! The y-positions of the global tracer maxima
                                ! in the units of G%geoLatT, often [degrees_N] or [km]
  real :: Tr_max_z(MAX_FIELDS_) ! The z-positions of the global tracer maxima [layer]
  logical :: Tr_minmax_avail(MAX_FIELDS_) ! A flag indicating whether the global minimum and
                                ! maximum information are available for each of the tracers
  character(len=40), dimension(MAX_FIELDS_) :: &
    Tr_names, &          ! The short names for each of the tracers
    Tr_units             ! The units for each of the tracers
  integer :: nTr_stocks  ! The total number of tracers in all registered tracer packages
  integer :: iyear, imonth, iday, ihour, iminute, isecond, itick ! For call to get_date()

 ! A description for output of each of the fields.
  type(vardesc) :: vars(NUM_FIELDS+MAX_FIELDS_)

  ! write_energy_time is the next integral multiple of energysavedays.
  dt_force = set_time(seconds=2) ; if (present(dt_forcing)) dt_force = dt_forcing
  if (CS%previous_calls == 0) then
    if (CS%energysave_geometric) then
      if (CS%energysavedays_geometric < CS%energysavedays) then
        CS%write_energy_time = day + CS%energysavedays_geometric
        CS%geometric_end_time = CS%Start_time + CS%energysavedays * &
          (1 + (day - CS%Start_time) / CS%energysavedays)
      else
        CS%write_energy_time = CS%Start_time + CS%energysavedays * &
          (1 + (day - CS%Start_time) / CS%energysavedays)
      endif
    else
      CS%write_energy_time = CS%Start_time + CS%energysavedays * &
        (1 + (day - CS%Start_time) / CS%energysavedays)
    endif
  elseif (day + (dt_force/2) < CS%write_energy_time) then
    return  ! Do not write this step
  else ! Determine the next write time before proceeding
    if (CS%energysave_geometric) then
      if (CS%write_energy_time + CS%energysavedays_geometric >= &
          CS%geometric_end_time) then
        CS%write_energy_time = CS%geometric_end_time
        CS%energysave_geometric = .false.  ! stop geometric progression
      else
        CS%write_energy_time = CS%write_energy_time + CS%energysavedays_geometric
      endif
      CS%energysavedays_geometric = CS%energysavedays_geometric*2
    else
      CS%write_energy_time = CS%write_energy_time + CS%energysavedays
    endif
  endif

  num_nc_fields = 17
  if (.not.CS%use_temperature) num_nc_fields = 11
  vars(1) = var_desc("Ntrunc","Nondim","Number of Velocity Truncations",'1','1')
  vars(2) = var_desc("En","Joules","Total Energy",'1','1')
  vars(3) = var_desc("APE","Joules","Total Interface APE",'1','i')
  vars(4) = var_desc("KE","Joules","Total Layer KE",'1','L')
  vars(5) = var_desc("H0","meter","Zero APE Depth of Interface",'1','i')
  vars(6) = var_desc("Mass_lay","kg","Total Layer Mass",'1','L')
  vars(7) = var_desc("Mass","kg","Total Mass",'1','1')
  vars(8) = var_desc("Mass_chg","kg","Total Mass Change between Entries",'1','1')
  vars(9) = var_desc("Mass_anom","kg","Anomalous Total Mass Change",'1','1')
  vars(10) = var_desc("max_CFL_trans","Nondim","Maximum finite-volume CFL",'1','1')
  vars(11) = var_desc("max_CFL_lin","Nondim","Maximum finite-difference CFL",'1','1')
  if (CS%use_temperature) then
    vars(12) = var_desc("Salt","kg","Total Salt",'1','1')
    vars(13) = var_desc("Salt_chg","kg","Total Salt Change between Entries",'1','1')
    vars(14) = var_desc("Salt_anom","kg","Anomalous Total Salt Change",'1','1')
    vars(15) = var_desc("Heat","Joules","Total Heat",'1','1')
    vars(16) = var_desc("Heat_chg","Joules","Total Heat Change between Entries",'1','1')
    vars(17) = var_desc("Heat_anom","Joules","Anomalous Total Heat Change",'1','1')
  endif

  is = G%isc ; ie = G%iec ; js = G%jsc ; je = G%jec ; nz = GV%ke
  Isq = G%IscB ; Ieq = G%IecB ; Jsq = G%JscB ; Jeq = G%JecB
  isr = is - (G%isd-1) ; ier = ie - (G%isd-1) ; jsr = js - (G%jsd-1) ; jer = je - (G%jsd-1)

  RZL4_to_J = US%RZL2_to_kg*US%L_T_to_m_s**2 ! Used to unscale energies

  if (.not.associated(CS)) call MOM_error(FATAL, &
         "write_energy: Module must be initialized before it is used.")

  if (.not.CS%initialized) call MOM_error(FATAL, &
         "write_energy: Module must be initialized before it is used.")

  do j=js,je ; do i=is,ie
    areaTm(i,j) = G%mask2dT(i,j)*G%areaT(i,j)
  enddo ; enddo

  tmp1(:,:,:) = 0.0
  do k=1,nz ; do j=js,je ; do i=is,ie
    tmp1(i,j,k) = h(i,j,k) * (GV%H_to_RZ*areaTm(i,j))
  enddo ; enddo ; enddo
  mass_tot_RZL2 = reproducing_sum(tmp1, isr, ier, jsr, jer, sums=mass_lay, EFP_sum=mass_EFP, unscale=US%RZL2_to_kg)

  if (GV%Boussinesq) then
    do k=1,nz ; vol_lay(k) = (1.0 / GV%Rho0) * mass_lay(k) ; enddo
  else
    if (CS%do_APE_calc) then
      call find_eta(h, tv, G, GV, US, eta, dZref=G%Z_ref)
      do k=1,nz ; do j=js,je ; do i=is,ie
        tmp1(i,j,k) = (eta(i,j,K)-eta(i,j,K+1)) * areaTm(i,j)
      enddo ; enddo ; enddo
      vol_tot = reproducing_sum(tmp1, isr, ier, jsr, jer, sums=vol_lay, unscale=US%Z_to_m*US%L_to_m**2)
    endif
  endif ! Boussinesq

  nTr_stocks = 0
  Tr_minmax_avail(:) = .false.
  if (CS%write_min_max .and. CS%write_min_max_loc) then
    call call_tracer_stocks(h, Tr_stocks, G, GV, US, tracer_CSp, stock_names=Tr_names, &
                            stock_units=Tr_units, num_stocks=nTr_stocks,&
                            got_min_max=Tr_minmax_avail, global_min=Tr_min, global_max=Tr_max, &
                            xgmin=Tr_min_x, ygmin=Tr_min_y, zgmin=Tr_min_z,&
                            xgmax=Tr_max_x, ygmax=Tr_max_y, zgmax=Tr_max_z)
  elseif (CS%write_min_max) then
    call call_tracer_stocks(h, Tr_stocks, G, GV, US, tracer_CSp, stock_names=Tr_names, &
                            stock_units=Tr_units, num_stocks=nTr_stocks,&
                            got_min_max=Tr_minmax_avail, global_min=Tr_min, global_max=Tr_max)
  else
    call call_tracer_stocks(h, Tr_stocks, G, GV, US, tracer_CSp, stock_names=Tr_names, &
                            stock_units=Tr_units, num_stocks=nTr_stocks)
  endif
  if (nTr_stocks > 0) then
    do m=1,nTr_stocks
      vars(num_nc_fields+m) = var_desc(Tr_names(m), units=Tr_units(m), &
                    longname=Tr_names(m), hor_grid='1', z_grid='1')
    enddo
    num_nc_fields = num_nc_fields + nTr_stocks
  endif

  if (CS%use_temperature .and. CS%write_stocks) then
    call array_global_min_max(tv%T, G, nz, T_min, T_max, &
                              T_min_x, T_min_y, T_min_z, T_max_x, T_max_y, T_max_z, unscale=US%C_to_degC)
    call array_global_min_max(tv%S, G, nz, S_min, S_max, &
                              S_min_x, S_min_y, S_min_z, S_max_x, S_max_y, S_max_z, unscale=US%S_to_ppt)
  endif

  if (CS%previous_calls == 0) then

    CS%mass_prev_EFP = mass_EFP
    CS%fresh_water_in_EFP = real_to_EFP(0.0)
    if (CS%use_temperature) then
      CS%net_salt_in_EFP = real_to_EFP(0.0)  ; CS%net_heat_in_EFP = real_to_EFP(0.0)
    endif

    !  Reopen or create a text output file, with an explanatory header line.
    if (is_root_pe()) then
      if (day > CS%Start_time) then
        call open_ASCII_file(CS%fileenergy_ascii, trim(CS%energyfile), action=APPEND_FILE)
      else
        call open_ASCII_file(CS%fileenergy_ascii, trim(CS%energyfile), action=WRITEONLY_FILE)
        if (abs(CS%timeunit - 86400.0) < 1.0) then
          if (CS%use_temperature) then
            write(CS%fileenergy_ascii,'("  Step,",7x,"Day,  Truncs,      &
                &Energy/Mass,      Maximum CFL,  Mean Sea Level,  Total Mass,  Mean Salin, &
                &Mean Temp, Frac Mass Err,   Salin Err,    Temp Err")')
            write(CS%fileenergy_ascii,'(12x,"[days]",17x,"[m2 s-2]",11x,"[Nondim]",7x,"[m]",13x,&
                &"[kg]",9x,"[PSU]",6x,"[degC]",7x,"[Nondim]",8x,"[PSU]",8x,"[degC]")')
          else
            write(CS%fileenergy_ascii,'("  Step,",7x,"Day,  Truncs,      &
                &Energy/Mass,      Maximum CFL,  Mean sea level,   Total Mass,    Frac Mass Err")')
            write(CS%fileenergy_ascii,'(12x,"[days]",17x,"[m2 s-2]",11x,"[Nondim]",8x,"[m]",13x,&
                &"[kg]",11x,"[Nondim]")')
          endif
        else
          if ((CS%timeunit >= 0.99) .and. (CS%timeunit < 1.01)) then
            time_units = "           [seconds]     "
          elseif ((CS%timeunit >= 3599.0) .and. (CS%timeunit < 3601.0)) then
            time_units = "            [hours]      "
          elseif ((CS%timeunit >= 86399.0) .and. (CS%timeunit < 86401.0)) then
            time_units = "             [days]      "
          elseif ((CS%timeunit >= 3.0e7) .and. (CS%timeunit < 3.2e7)) then
            time_units = "            [years]      "
          else
            write(time_units,'(9x,"[",es8.2," s]    ")') CS%timeunit
          endif

          if (CS%use_temperature) then
            write(CS%fileenergy_ascii,'("  Step,",7x,"Time, Truncs,      &
                &Energy/Mass,      Maximum CFL,  Mean Sea Level,  Total Mass,  Mean Salin, &
                &Mean Temp, Frac Mass Err,   Salin Err,    Temp Err")')
            write(CS%fileenergy_ascii,'(A25,10x,"[m2 s-2]",11x,"[Nondim]",7x,"[m]",13x,&
                &"[kg]",9x,"[PSU]",6x,"[degC]",7x,"[Nondim]",8x,"[PSU]",6x,&
                &"[degC]")') time_units
          else
            write(CS%fileenergy_ascii,'("  Step,",7x,"Time, Truncs,      &
                &Energy/Mass,      Maximum CFL,  Mean sea level,   Total Mass,    Frac Mass Err")')
            write(CS%fileenergy_ascii,'(A25,10x,"[m2 s-2]",11x,"[Nondim]",8x,"[m]",13x,&
                &"[kg]",11x,"[Nondim]")') time_units
          endif
        endif
      endif

      energypath_nc = trim(CS%energyfile) // ".nc"
      if (day > CS%Start_time) then
        call reopen_MOM_file(CS%fileenergy_nc, trim(energypath_nc), vars, &
            num_nc_fields, CS%fields, SINGLE_FILE, CS%timeunit, G=G, GV=GV)
      else
        call create_MOM_file(CS%fileenergy_nc, trim(energypath_nc), vars, &
            num_nc_fields, CS%fields, SINGLE_FILE, CS%timeunit, G=G, GV=GV)
      endif
    endif
  endif

  if (CS%do_APE_calc) then
    lbelow = 1 ; volbelow = 0.0
    do k=nz,1,-1
      volbelow = volbelow + vol_lay(k)
      if ((volbelow >= CS%DL%vol_below(CS%lH(k))) .and. &
          (volbelow < CS%DL%vol_below(CS%lH(k)+1))) then
        li = CS%lH(k)
      else
        labove=CS%DL%listsize
        li = (labove + lbelow) / 2
        do while (li > lbelow)
          if (volbelow < CS%DL%vol_below(li)) then ; labove = li
          else ; lbelow = li ; endif
          li = (labove + lbelow) / 2
        enddo
        CS%lH(k) = li
      endif
      lbelow = li
      Z_0APE(K) = CS%DL%depth(li) - (volbelow - CS%DL%vol_below(li)) / CS%DL%area(li)
    enddo
    Z_0APE(nz+1) = CS%DL%depth(2)

    !   Calculate the Available Potential Energy integrated over each interface.  With a nonlinear
    ! equation of state or with a bulk mixed layer this calculation is only approximate.
    ! With an ALE model this does not make sense and should be revisited.
    PE_pt(:,:,:) = 0.0
    if (GV%Boussinesq) then
      do j=js,je ; do i=is,ie
        hbelow = 0.0
        do K=nz,1,-1
          hbelow = hbelow + h(i,j,k) * GV%H_to_Z
          hint = Z_0APE(K) + (hbelow - (G%bathyT(i,j) + G%Z_ref))
          hbot = Z_0APE(K) - (G%bathyT(i,j) + G%Z_ref)
          hbot = (hbot + ABS(hbot)) * 0.5
          PE_pt(i,j,K) = (0.5 * areaTm(i,j)) * (GV%Rho0*GV%g_prime(K)) * &
                  (hint * hint - hbot * hbot)
        enddo
      enddo ; enddo
    elseif (GV%semi_Boussinesq) then
      do j=js,je ; do i=is,ie
        do K=nz,1,-1
          hint = Z_0APE(K) + eta(i,j,K)  ! eta and H_0 have opposite signs.
          hbot = max(Z_0APE(K) - (G%bathyT(i,j) + G%Z_ref), 0.0)
          PE_pt(i,j,K) = (0.5 * areaTm(i,j) * (GV%Rho0*GV%g_prime(K))) * &
                         (hint * hint - hbot * hbot)
        enddo
      enddo ; enddo
    else
      do j=js,je ; do i=is,ie
        do K=nz,2,-1
          hint = Z_0APE(K) + eta(i,j,K)  ! eta and H_0 have opposite signs.
          hbot = max(Z_0APE(K) - (G%bathyT(i,j) + G%Z_ref), 0.0)
          PE_pt(i,j,K) = (0.25 * areaTm(i,j) * &
                          ((GV%Rlay(k)+GV%Rlay(k-1))*GV%g_prime(K))) * &
                         (hint * hint - hbot * hbot)
        enddo
        hint = Z_0APE(1) + eta(i,j,1)  ! eta and H_0 have opposite signs.
        hbot = max(Z_0APE(1) - (G%bathyT(i,j) + G%Z_ref), 0.0)
        PE_pt(i,j,1) = (0.5 * areaTm(i,j) * (GV%Rlay(1)*GV%g_prime(1))) * &
                       (hint * hint - hbot * hbot)
      enddo ; enddo
    endif

    PE_tot = reproducing_sum(PE_pt, isr, ier, jsr, jer, sums=PE, unscale=RZL4_to_J)
    do k=1,nz+1 ; H_0APE(K) = US%Z_to_m*Z_0APE(K) ; enddo
  else
    PE_tot = 0.0
    do k=1,nz+1 ; PE(K) = 0.0 ; H_0APE(K) = 0.0 ; enddo
  endif

  ! Calculate the Kinetic Energy integrated over each layer.
  tmp1(:,:,:) = 0.0
  do k=1,nz ; do j=js,je ; do i=is,ie
    tmp1(i,j,k) = (0.25 * GV%H_to_RZ*(areaTm(i,j) * h(i,j,k))) * &
            (((u(I-1,j,k)**2) + (u(I,j,k)**2)) + ((v(i,J-1,k)**2) + (v(i,J,k)**2)))
  enddo ; enddo ; enddo

  KE_tot = reproducing_sum(tmp1, isr, ier, jsr, jer, sums=KE, unscale=RZL4_to_J)

  ! Use reproducing sums to do global integrals relate to the heat, salinity and water budgets.
  if (CS%use_temperature) then
    Temp_int(:,:) = 0.0 ; Salt_int(:,:) = 0.0
    do k=1,nz ; do j=js,je ; do i=is,ie
      Salt_int(i,j) = Salt_int(i,j) + tv%S(i,j,k) * (h(i,j,k)*(GV%H_to_RZ * areaTm(i,j)))
      Temp_int(i,j) = Temp_int(i,j) + (tv%C_p * tv%T(i,j,k)) * (h(i,j,k)*(GV%H_to_RZ * areaTm(i,j)))
    enddo ; enddo ; enddo
    salt_EFP = reproducing_sum_EFP(Salt_int, isr, ier, jsr, jer, only_on_PE=.true., &
                                   unscale=US%RZL2_to_kg*US%S_to_ppt)
    heat_EFP = reproducing_sum_EFP(Temp_int, isr, ier, jsr, jer, only_on_PE=.true., &
                                   unscale=US%RZL2_to_kg*US%Q_to_J_kg)

    ! Combining the sums avoids multiple blocking all-PE updates.
    EFP_list(1) = salt_EFP ;  EFP_list(2) = heat_EFP ; EFP_list(3) = CS%fresh_water_in_EFP
    EFP_list(4) = CS%net_salt_in_EFP ; EFP_list(5) = CS%net_heat_in_EFP
    call EFP_sum_across_PEs(EFP_list, 5)
    ! Return the globally summed values to the original variables.
    salt_EFP = EFP_list(1) ; heat_EFP = EFP_list(2) ; CS%fresh_water_in_EFP = EFP_list(3)
    CS%net_salt_in_EFP = EFP_list(4) ; CS%net_heat_in_EFP = EFP_list(5)

  else
    call EFP_sum_across_PEs(CS%fresh_water_in_EFP)
  endif

  ! Calculate the maximum CFL numbers.
  max_CFL(1:2) = 0.0
  do k=1,nz ; do j=js,je ; do I=Isq,Ieq
    CFL_Iarea = G%IareaT(i,j)
    if (u(I,j,k) < 0.0) &
      CFL_Iarea = G%IareaT(i+1,j)

    CFL_trans = abs(u(I,j,k) * CS%dt_in_T) * (G%dy_Cu(I,j) * CFL_Iarea)
    CFL_lin = abs(u(I,j,k) * CS%dt_in_T) * G%IdxCu(I,j)
    max_CFL(1) = max(max_CFL(1), CFL_trans)
    max_CFL(2) = max(max_CFL(2), CFL_lin)
  enddo ; enddo ; enddo
  do k=1,nz ; do J=Jsq,Jeq ; do i=is,ie
    CFL_Iarea = G%IareaT(i,j)
    if (v(i,J,k) < 0.0) &
      CFL_Iarea = G%IareaT(i,j+1)

    CFL_trans = abs(v(i,J,k) * CS%dt_in_T) * (G%dx_Cv(i,J) * CFL_Iarea)
    CFL_lin = abs(v(i,J,k) * CS%dt_in_T) * G%IdyCv(i,J)
    max_CFL(1) = max(max_CFL(1), CFL_trans)
    max_CFL(2) = max(max_CFL(2), CFL_lin)
  enddo ; enddo ; enddo

  call sum_across_PEs(CS%ntrunc)

  call max_across_PEs(max_CFL, 2)

  ! From this point onward, many of the calculations set or use variables in unscaled (mks) units.

  Salt = 0.0 ; Heat = 0.0
  if (CS%use_temperature) then
    Salt = EFP_to_real(salt_EFP)
    Heat = EFP_to_real(heat_EFP)
    if (CS%previous_calls == 0) then
      CS%salt_prev_EFP = salt_EFP ; CS%heat_prev_EFP = heat_EFP
    endif
    Salt_chg_EFP = Salt_EFP - CS%salt_prev_EFP
    Salt_chg = EFP_to_real(Salt_chg_EFP)
    Salt_anom_EFP = Salt_chg_EFP - CS%net_salt_in_EFP
    Salt_anom = EFP_to_real(Salt_anom_EFP)
    Heat_chg_EFP = Heat_EFP - CS%heat_prev_EFP
    Heat_chg = EFP_to_real(Heat_chg_EFP)
    Heat_anom_EFP = Heat_chg_EFP - CS%net_heat_in_EFP
    Heat_anom = EFP_to_real(Heat_anom_EFP)
  endif

  mass_chg_EFP = mass_EFP - CS%mass_prev_EFP
  mass_anom_EFP = mass_chg_EFP - CS%fresh_water_in_EFP
  mass_anom = EFP_to_real(mass_anom_EFP)
  if (CS%use_temperature .and. .not.GV%Boussinesq) then
    ! net_salt_input needs to be converted from ppt m s-1 to kg m-2 s-1.
    mass_anom = mass_anom - 0.001*EFP_to_real(CS%net_salt_in_EFP)
  endif
  mass_chg = EFP_to_real(mass_chg_EFP)

  mass_tot = US%RZL2_to_kg * mass_tot_RZL2
  if (CS%use_temperature) then
    salin = Salt / mass_tot
    salin_anom = Salt_anom / mass_tot
   ! salin_chg = Salt_chg / mass_tot
    temp = heat / (mass_tot*US%Q_to_J_kg*US%degC_to_C*tv%C_p)
    temp_anom = Heat_anom / (mass_tot*US%Q_to_J_kg*US%degC_to_C*tv%C_p)
  endif
  toten = RZL4_to_J * (KE_tot + PE_tot)
  En_mass = toten / mass_tot

  call get_time(day, start_of_day, num_days)
  date_stamped = (CS%date_stamped_output .and. (get_calendar_type() /= NO_CALENDAR))
  if (date_stamped) &
    call get_date(day, iyear, imonth, iday, ihour, iminute, isecond, itick)
  if (abs(CS%timeunit - 86400.0) < 1.0) then
    reday = REAL(num_days)+ (REAL(start_of_day)/86400.0)
    mesg_intro = "MOM Day "
  else
    reday = REAL(num_days)*(86400.0/CS%timeunit) + &
            REAL(start_of_day)/abs(CS%timeunit)
    mesg_intro = "MOM Time "
  endif
  if (reday < 1.0e8) then ;      write(day_str, '(F12.3)') reday
  elseif (reday < 1.0e11) then ; write(day_str, '(F15.3)') reday
  else ;                         write(day_str, '(ES15.9)') reday ; endif

  if     (n < 1000000)   then ; write(n_str, '(I6)')  n
  elseif (n < 10000000)  then ; write(n_str, '(I7)')  n
  elseif (n < 100000000) then ; write(n_str, '(I8)')  n
  else                        ; write(n_str, '(I10)') n ; endif

  if (date_stamped) then
    write(date_str,'("MOM Date",i7,2("/",i2.2)," ",i2.2,2(":",i2.2))') &
       iyear, imonth, iday, ihour, iminute, isecond
  else
    date_str = trim(mesg_intro)//trim(day_str)
  endif

  if (is_root_pe()) then  ! Only the root PE actually writes anything.
    if (CS%use_temperature) then
        write(stdout,'(A," ",A,": En ",ES12.6, ", MaxCFL ", F8.5, ", Mass ", &
                & ES18.12, ", Salt ", F15.11,", Temp ", F15.11)') &
          trim(date_str), trim(n_str), En_mass, max_CFL(1), mass_tot, salin, temp
    else
        write(stdout,'(A," ",A,": En ",ES12.6, ", MaxCFL ", F8.5, ", Mass ", &
                & ES18.12)') &
          trim(date_str), trim(n_str), En_mass, max_CFL(1), mass_tot
    endif

    if (CS%use_temperature) then
      write(CS%fileenergy_ascii,'(A,",",A,",", I6,", En ",ES22.16, &
                               &", CFL ", F8.5, ", SL ",&
                               &es11.4,", M ",ES11.5,", S",f8.4,", T",f8.4,&
                               &", Me ",ES9.2,", Se ",ES9.2,", Te ",ES9.2)') &
            trim(n_str), trim(day_str), CS%ntrunc, En_mass, max_CFL(1), &
            -H_0APE(1), mass_tot, salin, temp, mass_anom/mass_tot, salin_anom, &
            temp_anom
    else
      write(CS%fileenergy_ascii,'(A,",",A,",", I6,", En ",ES22.16, &
                                &", CFL ", F8.5, ", SL ",&
                                  &ES11.4,", Mass ",ES11.5,", Me ",ES9.2)') &
            trim(n_str), trim(day_str), CS%ntrunc, En_mass, max_CFL(1), &
            -H_0APE(1), mass_tot, mass_anom/mass_tot
    endif

    if (CS%ntrunc > 0) then
      write(stdout,'(A," Energy/Mass:",ES12.5," Truncations ",I0)') &
        trim(date_str), En_mass, CS%ntrunc
    endif

    if (CS%write_stocks) then
      write(stdout,'("    Total Energy: ",Z16.16,ES24.16)') toten, toten
      write(stdout,'("    Total Mass: ",ES24.16,", Change: ",ES24.16," Error: ",ES12.5," (",ES8.1,")")') &
            mass_tot, mass_chg, mass_anom, mass_anom/mass_tot
      if (CS%use_temperature) then
        if (Salt == 0.) then
          write(stdout,'("    Total Salt: ",ES24.16,", Change: ",ES24.16," Error: ",ES12.5)') &
              Salt*0.001, Salt_chg*0.001, Salt_anom*0.001
        else
          write(stdout,'("    Total Salt: ",ES24.16,", Change: ",ES24.16," Error: ",ES12.5," (",ES8.1,")")') &
              Salt*0.001, Salt_chg*0.001, Salt_anom*0.001, Salt_anom/Salt
        endif
        if (CS%write_min_max .and. CS%write_min_max_loc) then
          write(stdout,'(16X,"Salinity Global Min:",ES24.16,1X,"at: (",f7.2,",",f7.2,",",f8.2,")"  )') &
                S_min, S_min_x, S_min_y, S_min_z
          write(stdout,'(16X,"Salinity Global Max:",ES24.16,1X,"at: (",f7.2,",",f7.2,",",f8.2,")"  )') &
                S_max, S_max_x, S_max_y, S_max_z
        elseif (CS%write_min_max) then
          write(stdout,'(16X,"Salinity Global Min & Max:",ES24.16,1X,ES24.16)') S_min, S_max
        endif

        if (Heat == 0.) then
          write(stdout,'("    Total Heat: ",ES24.16,", Change: ",ES24.16," Error: ",ES12.5)') &
              Heat, Heat_chg, Heat_anom
        else
          write(stdout,'("    Total Heat: ",ES24.16,", Change: ",ES24.16," Error: ",ES12.5," (",ES8.1,")")') &
              Heat, Heat_chg, Heat_anom, Heat_anom/Heat
        endif
        if (CS%write_min_max .and. CS%write_min_max_loc) then
          write(stdout,'(16X,"Temperature Global Min:",ES24.16,1X,"at: (",f7.2,",",f7.2,",",f8.2,")"  )') &
                T_min, T_min_x, T_min_y, T_min_z
          write(stdout,'(16X,"Temperature Global Max:",ES24.16,1X,"at: (",f7.2,",",f7.2,",",f8.2,")"  )') &
                T_max, T_max_x, T_max_y, T_max_z
        elseif (CS%write_min_max) then
          write(stdout,'(16X,"Temperature Global Min & Max:",ES24.16,1X,ES24.16)') T_min, T_max
        endif
      endif
      do m=1,nTr_stocks

        write(stdout,'("      Total ",a,": ",ES24.16,1X,a)') &
              trim(Tr_names(m)), Tr_stocks(m), trim(Tr_units(m))

        if (CS%write_min_max .and. CS%write_min_max_loc .and. Tr_minmax_avail(m)) then
          write(stdout,'(18X,a," Global Min:",ES24.16,1X,"at: (",f7.2,",",f7.2,",",f8.2,")"  )') &
               trim(Tr_names(m)), Tr_min(m), Tr_min_x(m), Tr_min_y(m), Tr_min_z(m)
          write(stdout,'(18X,a," Global Max:",ES24.16,1X,"at: (",f7.2,",",f7.2,",",f8.2,")"  )') &
               trim(Tr_names(m)), Tr_max(m), Tr_max_x(m), Tr_max_y(m), Tr_max_z(m)
        elseif (CS%write_min_max .and. Tr_minmax_avail(m)) then
          write(stdout,'(18X,a," Global Min & Max:",ES24.16,1X,ES24.16)') &
               trim(Tr_names(m)), Tr_min(m), Tr_max(m)
        endif

      enddo
    endif

    call CS%fileenergy_nc%write_field(CS%fields(1), real(CS%ntrunc), reday)
    call CS%fileenergy_nc%write_field(CS%fields(2), toten, reday)
    do k=1,nz+1 ; PE(K) = RZL4_to_J*PE(K) ; enddo
    call CS%fileenergy_nc%write_field(CS%fields(3), PE, reday)
    do k=1,nz ; KE(k) = RZL4_to_J*KE(k) ; enddo
    call CS%fileenergy_nc%write_field(CS%fields(4), KE, reday)
    call CS%fileenergy_nc%write_field(CS%fields(5), H_0APE, reday)
    do k=1,nz ; mass_lay(k) = US%RZL2_to_kg*mass_lay(k) ; enddo
    call CS%fileenergy_nc%write_field(CS%fields(6), mass_lay, reday)

    call CS%fileenergy_nc%write_field(CS%fields(7), mass_tot, reday)
    call CS%fileenergy_nc%write_field(CS%fields(8), mass_chg, reday)
    call CS%fileenergy_nc%write_field(CS%fields(9), mass_anom, reday)
    call CS%fileenergy_nc%write_field(CS%fields(10), max_CFL(1), reday)
    call CS%fileenergy_nc%write_field(CS%fields(11), max_CFL(2), reday)
    if (CS%use_temperature) then
      call CS%fileenergy_nc%write_field(CS%fields(12), 0.001*Salt, reday)
      call CS%fileenergy_nc%write_field(CS%fields(13), 0.001*salt_chg, reday)
      call CS%fileenergy_nc%write_field(CS%fields(14), 0.001*salt_anom, reday)
      call CS%fileenergy_nc%write_field(CS%fields(15), Heat, reday)
      call CS%fileenergy_nc%write_field(CS%fields(16), heat_chg, reday)
      call CS%fileenergy_nc%write_field(CS%fields(17), heat_anom, reday)
      do m=1,nTr_stocks
        call CS%fileenergy_nc%write_field(CS%fields(17+m), Tr_stocks(m), reday)
      enddo
    else
      do m=1,nTr_stocks
        call CS%fileenergy_nc%write_field(CS%fields(11+m), Tr_stocks(m), reday)
      enddo
    endif

    call CS%fileenergy_nc%flush()
  endif  ! Only the root PE actually writes anything.

  if (is_NaN(En_mass)) then
    call MOM_error(FATAL, "write_energy : NaNs in total model energy forced model termination.")
  elseif (En_mass > US%L_T_to_m_s**2*CS%max_Energy) then
    write(mesg,'("Energy per unit mass of ",ES11.4," exceeds ",ES11.4)') &
                  En_mass, US%L_T_to_m_s**2*CS%max_Energy
    call MOM_error(FATAL, "write_energy : Excessive energy per unit mass forced model termination.")
  endif
  if (CS%ntrunc>CS%maxtrunc) then
    call MOM_error(FATAL, "write_energy : Ocean velocity has been truncated too many times.")
  endif
  CS%ntrunc = 0
  CS%previous_calls = CS%previous_calls + 1

  CS%mass_prev_EFP = mass_EFP ; CS%fresh_water_in_EFP = real_to_EFP(0.0)
  if (CS%use_temperature) then
    CS%salt_prev_EFP = Salt_EFP ; CS%net_salt_in_EFP = real_to_EFP(0.0)
    CS%heat_prev_EFP = Heat_EFP ; CS%net_heat_in_EFP = real_to_EFP(0.0)
  endif

end subroutine write_energy

!> This subroutine accumulates the net input of volume, salt and heat, through
!! the ocean surface for use in diagnosing conservation.
subroutine accumulate_net_input(fluxes, sfc_state, tv, dt, G, US, CS)
  type(forcing),         intent(in) :: fluxes !< A structure containing pointers to any possible
                                              !! forcing fields.  Unused fields are unallocated.
  type(surface),         intent(in) :: sfc_state !< A structure containing fields that
                                              !! describe the surface state of the ocean.
  type(thermo_var_ptrs), intent(in) :: tv     !< A structure pointing to various
                                              !! thermodynamic variables.
  real,                  intent(in) :: dt     !< The amount of time over which to average [T ~> s].
  type(ocean_grid_type), intent(in) :: G      !< The ocean's grid structure.
  type(unit_scale_type), intent(in) :: US     !< A dimensional unit scaling type
  type(Sum_output_CS),   pointer    :: CS     !< The control structure returned by a previous call
                                              !! to MOM_sum_output_init.
  ! Local variables
  real, dimension(SZI_(G),SZJ_(G)) :: &
    FW_in, &   ! The net fresh water input, integrated over a timestep [R Z L2 ~> kg].
    salt_in, & ! The total salt added by surface fluxes, integrated
               ! over a time step [ppt kg].
    heat_in    ! The total heat added by surface fluxes, integrated
               ! over a time step [Q R Z L2 ~> J].

  type(EFP_type) :: &
    FW_in_EFP,   &   ! The net fresh water input, integrated over a timestep
                     ! and summed over space [kg].
    salt_in_EFP, &   ! The total salt added by surface fluxes, integrated
                     ! over a time step and summed over space [R Z L2 ~> ppt kg].
    heat_in_EFP      ! The total heat added by boundary fluxes, integrated
                     ! over a time step and summed over space [J].

  integer :: i, j, is, ie, js, je, isr, ier, jsr, jer

  is = G%isc ; ie = G%iec ; js = G%jsc ; je = G%jec

  FW_in(:,:) = 0.0
  if (associated(fluxes%evap)) then
    if (associated(fluxes%lprec) .and. associated(fluxes%fprec)) then
      do j=js,je ; do i=is,ie
        FW_in(i,j) = dt*G%areaT(i,j)*(fluxes%evap(i,j) + &
            (((fluxes%lprec(i,j) + fluxes%vprec(i,j)) + fluxes%lrunoff(i,j)) + &
              (fluxes%fprec(i,j) + fluxes%frunoff(i,j))))
      enddo ; enddo
    else
      call MOM_error(WARNING, &
        "accumulate_net_input called with associated evap field, but no precip field.")
    endif
  endif

  if (associated(fluxes%seaice_melt)) then ; do j=js,je ; do i=is,ie
    FW_in(i,j) = FW_in(i,j) + dt * G%areaT(i,j) * fluxes%seaice_melt(i,j)
  enddo ; enddo ; endif

  salt_in(:,:) = 0.0 ; heat_in(:,:) = 0.0
  if (CS%use_temperature) then

    if (associated(fluxes%sw)) then ; do j=js,je ; do i=is,ie
      heat_in(i,j) = heat_in(i,j) + dt * G%areaT(i,j) * (fluxes%sw(i,j) + &
             (fluxes%lw(i,j) + (fluxes%latent(i,j) + fluxes%sens(i,j))))
    enddo ; enddo ; endif

    if (associated(fluxes%seaice_melt_heat)) then ; do j=js,je ; do i=is,ie
      heat_in(i,j) = heat_in(i,j) + dt * G%areaT(i,j) * &
                                    fluxes%seaice_melt_heat(i,j)
    enddo ; enddo ; endif

    ! smg: new code
    ! include heat content from water transport across ocean surface
!    if (associated(fluxes%heat_content_lprec)) then ; do j=js,je ; do i=is,ie
!      heat_in(i,j) = heat_in(i,j) + dt * G%areaT(i,j) * &
!         (fluxes%heat_content_lprec(i,j)   + (fluxes%heat_content_fprec(i,j)   &
!       + (fluxes%heat_content_lrunoff(i,j) + (fluxes%heat_content_frunoff(i,j) &
!       + (fluxes%heat_content_cond(i,j)    + (fluxes%heat_content_vprec(i,j)   &
!       +  fluxes%heat_content_massout(i,j)))))))
!    enddo ; enddo ; endif

    ! smg: old code
    if (associated(fluxes%heat_content_evap)) then
      do j=js,je ; do i=is,ie
        heat_in(i,j) = heat_in(i,j) + dt * G%areaT(i,j) * &
                       (fluxes%heat_content_evap(i,j) + fluxes%heat_content_lprec(i,j) + &
                        fluxes%heat_content_cond(i,j) + fluxes%heat_content_fprec(i,j) + &
                        fluxes%heat_content_lrunoff(i,j) + fluxes%heat_content_frunoff(i,j))
      enddo ; enddo
    elseif (associated(tv%TempxPmE)) then
      do j=js,je ; do i=is,ie
        heat_in(i,j) = heat_in(i,j) + (tv%C_p * G%areaT(i,j)) * tv%TempxPmE(i,j)
      enddo ; enddo
    elseif (associated(fluxes%evap)) then
      do j=js,je ; do i=is,ie
        heat_in(i,j) = heat_in(i,j) + (tv%C_p * sfc_state%SST(i,j)) * FW_in(i,j)
      enddo ; enddo
    endif

    ! The following heat sources may or may not be used.
    if (associated(tv%internal_heat)) then
      do j=js,je ; do i=is,ie
        heat_in(i,j) = heat_in(i,j) + (tv%C_p * G%areaT(i,j)) * tv%internal_heat(i,j)
      enddo ; enddo
    endif
    if (associated(tv%frazil)) then ; do j=js,je ; do i=is,ie
      heat_in(i,j) = heat_in(i,j) + G%areaT(i,j) * tv%frazil(i,j)
    enddo ; enddo ; endif
    if (associated(fluxes%heat_added)) then ; do j=js,je ; do i=is,ie
      heat_in(i,j) = heat_in(i,j) + dt*G%areaT(i,j) * fluxes%heat_added(i,j)
    enddo ; enddo ; endif
!    if (associated(sfc_state%sw_lost)) then ; do j=js,je ; do i=is,ie
!      sfc_state%sw_lost must be in units of [Q R Z ~> J m-2]
!      heat_in(i,j) = heat_in(i,j) - G%areaT(i,j) * sfc_state%sw_lost(i,j)
!    enddo ; enddo ; endif

    if (associated(fluxes%salt_flux)) then ; do j=js,je ; do i=is,ie
      ! integrate salt_flux in [R Z T-1 ~> kgSalt m-2 s-1] to give [ppt kg]
      salt_in(i,j) = dt * G%areaT(i,j)*(1000.0*fluxes%salt_flux(i,j))
    enddo ; enddo ; endif
  endif

  if ((CS%use_temperature) .or. associated(fluxes%lprec) .or. &
      associated(fluxes%evap)) then
    ! The on-PE sums are stored here, but the sums across PEs are deferred to
    ! the next call to write_energy to avoid extra barriers.
    isr = is - (G%isd-1) ; ier = ie - (G%isd-1) ; jsr = js - (G%jsd-1) ; jer = je - (G%jsd-1)
    FW_in_EFP   = reproducing_sum_EFP(FW_in,   isr, ier, jsr, jer, only_on_PE=.true., &
                                      unscale=US%RZL2_to_kg)
    heat_in_EFP = reproducing_sum_EFP(heat_in, isr, ier, jsr, jer, only_on_PE=.true., &
                                      unscale=US%RZL2_to_kg*US%Q_to_J_kg)
    salt_in_EFP = reproducing_sum_EFP(salt_in, isr, ier, jsr, jer, only_on_PE=.true., &
                                      unscale=US%RZL2_to_kg)

    CS%fresh_water_in_EFP = CS%fresh_water_in_EFP + FW_in_EFP
    CS%net_salt_in_EFP    = CS%net_salt_in_EFP    + salt_in_EFP
    CS%net_heat_in_EFP    = CS%net_heat_in_EFP    + heat_in_EFP
  endif

end subroutine accumulate_net_input

!>  This subroutine sets up an ordered list of depths, along with the
!! cross sectional areas at each depth and the volume of fluid deeper
!! than each depth.  This might be read from a previously created file
!! or it might be created anew.  (For now only new creation occurs.
subroutine depth_list_setup(G, GV, US, DL, CS)
  type(ocean_grid_type),   intent(in)    :: G   !< The ocean's grid structure
  type(verticalGrid_type), intent(in)    :: GV  !< The ocean's vertical grid structure.
  type(unit_scale_type),   intent(in)    :: US  !< A dimensional unit scaling type
  type(Depth_List),        intent(inout) :: DL  !< The list of depths, areas and volumes to set up
  type(Sum_output_CS),     pointer       :: CS  !< The control structure returned by a
                                                !! previous call to MOM_sum_output_init.
  ! Local variables
  logical :: valid_DL_read
  integer :: k

  if (CS%read_depth_list) then
    if (file_exists(CS%depth_list_file)) then
      if (CS%update_depth_list_chksum) then
        call read_depth_list(G, US, DL, CS%depth_list_file, &
          require_chksum=CS%require_depth_list_chksum, file_matches=valid_DL_read)
      else
        call read_depth_list(G, US, DL, CS%depth_list_file, require_chksum=CS%require_depth_list_chksum)
        valid_DL_read = .true. ! Otherwise there would have been a fatal error.
      endif
    else
      if (is_root_pe()) call MOM_error(NOTE, "depth_list_setup: "// &
        trim(CS%depth_list_file)//" does not exist.  Creating a new file.")
      valid_DL_read = .false.
    endif

    if (.not.valid_DL_read) then
      call create_depth_list(G, DL, CS%D_list_min_inc)
      call write_depth_list(G, US, DL, CS%depth_list_file)
    endif
  else
    call create_depth_list(G, DL, CS%D_list_min_inc)
  endif

  do k=1,GV%ke
    CS%lH(k) = DL%listsize-1
  enddo

end subroutine depth_list_setup

!>  create_depth_list makes an ordered list of depths, along with the cross
!! sectional areas at each depth and the volume of fluid deeper than each depth.
subroutine create_depth_list(G, DL, min_depth_inc)
  type(ocean_grid_type), intent(in)    :: G  !< The ocean's grid structure.
  type(Depth_List),      intent(inout) :: DL !< The list of depths, areas and volumes to create
  real,                  intent(in)    :: min_depth_inc !< The minimum increment between depths in the list [Z ~> m]

  ! Local variables
  real, dimension(G%Domain%niglobal*G%Domain%njglobal + 1) :: &
    Dlist, &  !< The global list of bottom depths [Z ~> m].
    AreaList  !< The global list of cell areas [L2 ~> m2].
  integer, dimension(G%Domain%niglobal*G%Domain%njglobal+1) :: &
    indx2     !< The position of an element in the original unsorted list.
  real    :: Dnow  !< The depth now being considered for sorting [Z ~> m].
  real    :: Dprev !< The most recent depth that was considered [Z ~> m].
  real    :: vol   !< The running sum of open volume below a depth [Z L2 ~> m3].
  real    :: area  !< The open area at the current depth [L2 ~> m2].
  real    :: D_list_prev !< The most recent depth added to the list [Z ~> m].
  logical :: add_to_list !< This depth should be included as an entry on the list.

  integer :: ir, indxt
  integer :: mls, list_size
  integer :: list_pos, i_global, j_global
  integer :: i, j, k, kl

  mls = G%Domain%niglobal*G%Domain%njglobal

! Need to collect the global data from compute domains to a 1D array for sorting.
  Dlist(:) = 0.0
  Arealist(:) = 0.0
  do j=G%jsc,G%jec ; do i=G%isc,G%iec
    ! Set global indices that start the global domain at 1 (Fortran convention).
    j_global = j + G%jdg_offset - (G%jsg-1)
    i_global = i + G%idg_offset - (G%isg-1)

    list_pos = (j_global-1)*G%Domain%niglobal + i_global
    Dlist(list_pos) = G%bathyT(i,j) + G%Z_ref
    Arealist(list_pos) = G%mask2dT(i,j) * G%areaT(i,j)
  enddo ; enddo

  ! These sums reproduce across PEs because the arrays are only nonzero on one PE.
  call sum_across_PEs(Dlist, mls+1)
  call sum_across_PEs(Arealist, mls+1)

  do j=1,mls+1 ; indx2(j) = j ; enddo
  k = mls / 2  + 1 ; ir = mls
  do
    if (k > 1) then
      k = k - 1
      indxt = indx2(k)
      Dnow = Dlist(indxt)
    else
      indxt = indx2(ir)
      Dnow = Dlist(indxt)
      indx2(ir) = indx2(1)
      ir = ir - 1
      if (ir == 1) then ; indx2(1) = indxt ; exit ; endif
    endif
    i=k ; j=k*2
    do ; if (j > ir) exit
      if (j < ir .AND. Dlist(indx2(j)) < Dlist(indx2(j+1))) j = j + 1
      if (Dnow < Dlist(indx2(j))) then ; indx2(i) = indx2(j) ; i = j ; j = j + i
      else ; j = ir+1 ; endif
    enddo
    indx2(i) = indxt
  enddo

!  At this point, the lists should perhaps be culled to save memory.
! Culling: (1) identical depths (e.g. land) - take the last one.
!          (2) the topmost and bottommost depths are always saved.
!          (3) very close depths
!          (4) equal volume changes.

  ! Count the unique elements in the list.
  D_list_prev = Dlist(indx2(mls))
  list_size = 2
  do k=mls-1,1,-1
    if (Dlist(indx2(k)) < D_list_prev-min_depth_inc) then
      list_size = list_size + 1
      D_list_prev = Dlist(indx2(k))
    endif
  enddo

  DL%listsize = list_size+1
  allocate(DL%depth(DL%listsize), DL%area(DL%listsize), DL%vol_below(DL%listsize))

  vol = 0.0 ; area = 0.0
  Dprev = Dlist(indx2(mls))
  D_list_prev = Dprev

  kl = 0
  do k=mls,1,-1
    i = indx2(k)
    vol = vol + area * (Dprev - Dlist(i))
    area = area + AreaList(i)

    add_to_list = .false.
    if ((kl == 0) .or. (k==1)) then
      add_to_list = .true.
    elseif (Dlist(indx2(k-1)) < D_list_prev-min_depth_inc) then
      add_to_list = .true.
      D_list_prev = Dlist(indx2(k-1))
    endif

    if (add_to_list) then
      kl = kl+1
      DL%depth(kl) = Dlist(i)
      DL%area(kl) = area
      DL%vol_below(kl) = vol
    endif
    Dprev = Dlist(i)
  enddo

  do while (kl+1 < DL%listsize)
    ! I don't understand why this is needed... RWH
    kl = kl+1
    DL%vol_below(kl) = DL%vol_below(kl-1) * 1.000001
    DL%area(kl) = DL%area(kl-1)
    DL%depth(kl) = DL%depth(kl-1)
  enddo

  DL%vol_below(DL%listsize) = DL%vol_below(DL%listsize-1) * 1000.0
  DL%area(DL%listsize) = DL%area(DL%listsize-1)
  DL%depth(DL%listsize) = DL%depth(DL%listsize-1)

end subroutine create_depth_list

!> This subroutine writes out the depth list to the specified file.
subroutine write_depth_list(G, US, DL, filename)
  type(ocean_grid_type), intent(in) :: G   !< The ocean's grid structure.
  type(unit_scale_type), intent(in) :: US  !< A dimensional unit scaling type
  type(Depth_List),      intent(in) :: DL  !< The list of depths, areas and volumes to write
  character(len=*),      intent(in) :: filename !< The path to the depth list file to write.

  ! Local variables
  type(vardesc), dimension(:), allocatable :: &
    vars          ! Types that described the staggering and metadata for the fields
  type(MOM_field), dimension(:), allocatable :: &
    fields        ! Types with metadata about the variables that will be written
  type(axis_info), dimension(:), allocatable :: &
    extra_axes    ! Descriptors for extra axes that might be used
  type(attribute_info), dimension(:), allocatable :: &
    global_atts   ! Global attributes and their values
  type(MOM_netcdf_file) :: IO_handle   ! The I/O handle of the fileset
  character(len=16) :: depth_chksum, area_chksum

  ! All ranks are required to compute the global checksum
  call get_depth_list_checksums(G, US, depth_chksum, area_chksum)

  if (.not.is_root_pe()) return

  allocate(vars(3))
  allocate(fields(3))
  allocate(extra_axes(1))
  allocate(global_atts(2))

  call set_axis_info(extra_axes(1), "list", ax_size=DL%listsize)
  vars(1) = var_desc("depth", "m", "Sorted depth", '1', dim_names=(/"list"/), fixed=.true.)
  vars(2) = var_desc("area", "m2", "Open area at depth", '1', dim_names=(/"list"/), fixed=.true.)
  vars(3) = var_desc("vol_below", "m3", "Open volume below depth", '1', dim_names=(/"list"/), fixed=.true.)
  call set_attribute_info(global_atts(1), depth_chksum_attr, depth_chksum)
  call set_attribute_info(global_atts(2), area_chksum_attr, area_chksum)

  call create_MOM_file(IO_handle, filename, vars, 3, fields, SINGLE_FILE, &
      extra_axes=extra_axes, global_atts=global_atts)
  call MOM_write_field(IO_handle, fields(1), DL%depth, scale=US%Z_to_m)
  call MOM_write_field(IO_handle, fields(2), DL%area, scale=US%L_to_m**2)
  call MOM_write_field(IO_handle, fields(3), DL%vol_below, scale=US%Z_to_m*US%L_to_m**2)

  call delete_axis_info(extra_axes)
  call delete_attribute_info(global_atts)
  deallocate(vars, extra_axes, fields, global_atts)
  call IO_handle%close()
end subroutine write_depth_list

!> This subroutine reads in the depth list from the specified file
!! and allocates the memory within and sets up DL.
subroutine read_depth_list(G, US, DL, filename, require_chksum, file_matches)
  type(ocean_grid_type), intent(in)    :: G   !< The ocean's grid structure
  type(unit_scale_type), intent(in)    :: US  !< A dimensional unit scaling type
  type(Depth_List),      intent(inout) :: DL  !< The list of depths, areas and volumes
  character(len=*),      intent(in)    :: filename !< The path to the depth list file to read.
  logical,               intent(in)    :: require_chksum !< If true, missing or mismatched depth
                                              !! and area checksums result in a fatal error.
  logical, optional,     intent(out)   :: file_matches !< If present, this indicates whether the file
                                              !! has been read with matching depth and area checksums

  ! Local variables
  character(len=240) :: var_msg
  integer :: list_size, ndim, sizes(4)
  character(len=:), allocatable :: depth_file_chksum, area_file_chksum
  character(len=16) :: depth_grid_chksum, area_grid_chksum
  logical :: depth_att_found, area_att_found

  ! Check bathymetric consistency between this configuration and the depth list file.
  call read_attribute(filename, depth_chksum_attr, depth_file_chksum, found=depth_att_found)
  call read_attribute(filename, area_chksum_attr, area_file_chksum, found=area_att_found)

  if ((.not.depth_att_found) .or. (.not.area_att_found)) then
    var_msg = trim(filename) // " checksums are missing;"
    if (require_chksum) then
      call MOM_error(FATAL, trim(var_msg) // " aborting.")
    elseif (present(file_matches)) then
      call MOM_error(WARNING, trim(var_msg) // " updating file.")
      file_matches = .false.
      return
    else
      call MOM_error(WARNING, trim(var_msg) // " some diagnostics may not be reproducible.")
    endif
  else
    call get_depth_list_checksums(G, US, depth_grid_chksum, area_grid_chksum)

    if ((trim(depth_grid_chksum) /= trim(depth_file_chksum)) .or. &
        (trim(area_grid_chksum) /= trim(area_file_chksum)) ) then
      var_msg = trim(filename) // " checksums do not match;"
      if (require_chksum) then
        call MOM_error(FATAL, trim(var_msg) // " aborting.")
      elseif (present(file_matches)) then
        call MOM_error(WARNING, trim(var_msg) // " updating file.")
        file_matches = .false.
        return
      else
        call MOM_error(WARNING, trim(var_msg) // " some diagnostics may not be reproducible.")
      endif
    endif
  endif
  if (allocated(area_file_chksum)) deallocate(area_file_chksum)
  if (allocated(depth_file_chksum)) deallocate(depth_file_chksum)

  ! Get the length of the list.
  call field_size(filename, "depth", sizes, ndims=ndim)
  if (ndim /= 1) call MOM_ERROR(FATAL, "MOM_sum_output read_depth_list: depth in "//&
                                trim(filename)//" has too many or too few dimensions.")
  list_size = sizes(1)

  DL%listsize = list_size
  allocate(DL%depth(list_size), DL%area(list_size), DL%vol_below(list_size))

  call read_variable(filename, "depth", DL%depth, scale=US%m_to_Z)
  call read_variable(filename, "area", DL%area, scale=US%m_to_L**2)
  call read_variable(filename, "vol_below", DL%vol_below, scale=US%m_to_Z*US%m_to_L**2)

  if (present(file_matches)) file_matches = .true.

end subroutine read_depth_list


!> Return the checksums required to verify DEPTH_LIST_FILE contents.
!!
!! This function computes checksums for the bathymetry (G%bathyT) and masked
!! area (mask2dT * areaT) fields of the model grid G, which are used to compute
!! the depth list.  A difference in checksum indicates that a different method
!! was used to compute the grid data, and that any results using the depth
!! list, such as APE, will not be reproducible.
!!
!! Checksums are saved as hexadecimal strings, in order to avoid potential
!! datatype issues with netCDF attributes.
subroutine get_depth_list_checksums(G, US, depth_chksum, area_chksum)
  type(ocean_grid_type), intent(in) :: G          !< Ocean grid structure
  type(unit_scale_type), intent(in) :: US         !< A dimensional unit scaling type
  character(len=16), intent(out) :: depth_chksum  !< Depth checksum hexstring
  character(len=16), intent(out) :: area_chksum   !< Area checksum hexstring

  integer :: i, j
  real, allocatable :: field(:,:)  ! A temporary array for output converted to MKS units [m] or [m2]

  allocate(field(G%isc:G%iec, G%jsc:G%jec))

  ! Depth checksum
  do j=G%jsc,G%jec ; do i=G%isc,G%iec
    field(i,j) = US%Z_to_m*(G%bathyT(i,j) + G%Z_ref)
  enddo ; enddo
  write(depth_chksum, '(Z16)') field_chksum(field(:,:))

  ! Area checksum
  do j=G%jsc,G%jec ; do i=G%isc,G%iec
    field(i,j) = G%mask2dT(i,j) * US%L_to_m**2*G%areaT(i,j)
  enddo ; enddo
  write(area_chksum, '(Z16)') field_chksum(field(:,:))

  deallocate(field)
end subroutine get_depth_list_checksums

!> \namespace mom_sum_output
!!
!! By Robert Hallberg, April 1994 - June 2002
!!
!!   This file contains the subroutine (write_energy) that writes
!! horizontally integrated quantities, such as energies and layer
!! volumes, and other summary information to an output file.  Some
!! of these quantities (APE or resting interface height) are defined
!! relative to the global histogram of topography.  The subroutine
!! that compiles that histogram (depth_list_setup) is also included
!! in this file.
!!
!!   In addition, if the number of velocity truncations since the
!! previous call to write_energy exceeds maxtrunc or the total energy
!! exceeds a very large threshold, a fatal termination is triggered.

end module MOM_sum_output
