!> Provides integrals of density
module MOM_density_integrals

! This file is part of MOM6. See LICENSE.md for the license.

use MOM_EOS,              only : EOS_type
use MOM_EOS,              only : EOS_quadrature, EOS_domain
use MOM_EOS,              only : analytic_int_density_dz
use MOM_EOS,              only : analytic_int_specific_vol_dp
use MOM_EOS,              only : calculate_density
use MOM_EOS,              only : calculate_spec_vol
use MOM_EOS,              only : calculate_specific_vol_derivs
use MOM_EOS,              only : average_specific_vol
use MOM_error_handler,    only : MOM_error, FATAL, WARNING, MOM_mesg
use MOM_hor_index,        only : hor_index_type
use MOM_string_functions, only : uppercase
use MOM_variables,        only : thermo_var_ptrs
use MOM_unit_scaling,     only : unit_scale_type
use MOM_verticalGrid,     only : verticalGrid_type

implicit none ; private

#include <MOM_memory.h>

public int_density_dz
public int_density_dz_generic_pcm
public int_density_dz_generic_plm
public int_density_dz_generic_ppm
public int_specific_vol_dp
public int_spec_vol_dp_generic_pcm
public int_spec_vol_dp_generic_plm
public avg_specific_vol
public find_depth_of_pressure_in_cell
public diagnose_mass_weight_Z, diagnose_mass_weight_p

contains

!> Calls the appropriate subroutine to calculate analytical and nearly-analytical
!! integrals in z across layers of pressure anomalies, which are
!! required for calculating the finite-volume form pressure accelerations in a
!! Boussinesq model.
subroutine int_density_dz(T, S, z_t, z_b, rho_ref, rho_0, G_e, HI, EOS, US, dpa, &
                          intz_dpa, intx_dpa, inty_dpa, bathyT, SSH, dz_neglect, MassWghtInterp, Z_0p)
  type(hor_index_type), intent(in)  :: HI  !< Ocean horizontal index structures for the arrays
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T   !< Potential temperature referenced to the surface [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S   !< Salinity [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: z_t !< Height at the top of the layer in depth units [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: z_b !< Height at the bottom of the layer [Z ~> m]
  real,                 intent(in)  :: rho_ref !< A mean density [R ~> kg m-3], that is
                                           !! subtracted out to reduce the magnitude of each of the
                                           !! integrals.
  real,                 intent(in)  :: rho_0 !< A density [R ~> kg m-3], that is used
                                           !! to calculate the pressure (as p~=-z*rho_0*G_e)
                                           !! used in the equation of state.
  real,                 intent(in)  :: G_e !< The Earth's gravitational acceleration
                                           !! [L2 Z-1 T-2 ~> m s-2]
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US  !< A dimensional unit scaling type
  real, dimension(SZI_(HI),SZJ_(HI)), &
                      intent(inout) :: dpa !< The change in the pressure anomaly
                                           !! across the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
            optional, intent(inout) :: intz_dpa !< The integral through the thickness of the
                                           !! layer of the pressure anomaly relative to the
                                           !! anomaly at the top of the layer [R L2 Z T-2 ~> Pa m]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
            optional, intent(inout) :: intx_dpa !< The integral in x of the difference between
                                          !! the pressure anomaly at the top and bottom of the
                                          !! layer divided by the x grid spacing [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJB_(HI)), &
            optional, intent(inout) :: inty_dpa !< The integral in y of the difference between
                                          !! the pressure anomaly at the top and bottom of the
                                          !! layer divided by the y grid spacing [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: bathyT !< The depth of the bathymetry [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: SSH !< The sea surface height [Z ~> m]
  real,       optional, intent(in)  :: dz_neglect !< A minuscule thickness change [Z ~> m]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals
  real, dimension(HI%isd:HI%ied,HI%jsd:HI%jed), &
              optional, intent(in)  :: Z_0p !< The height at which the pressure is 0 [Z ~> m]

  if (EOS_quadrature(EOS)) then
    call int_density_dz_generic_pcm(T, S, z_t, z_b, rho_ref, rho_0, G_e, HI, EOS, US, dpa, &
                                    intz_dpa, intx_dpa, inty_dpa, bathyT, SSH, dz_neglect, MassWghtInterp, Z_0p=Z_0p)
  else
    call analytic_int_density_dz(T, S, z_t, z_b, rho_ref, rho_0, G_e, HI, EOS, dpa, &
                                 intz_dpa, intx_dpa, inty_dpa, bathyT, SSH, dz_neglect, MassWghtInterp, Z_0p=Z_0p)
  endif

end subroutine int_density_dz


!> Calculates (by numerical quadrature) integrals of pressure anomalies across layers, which
!! are required for calculating the finite-volume form pressure accelerations in a Boussinesq model.
subroutine int_density_dz_generic_pcm(T, S, z_t, z_b, rho_ref, rho_0, G_e, HI, &
                                      EOS, US, dpa, intz_dpa, intx_dpa, inty_dpa, bathyT, SSH, &
                                      dz_neglect, MassWghtInterp, use_inaccurate_form, Z_0p)
  type(hor_index_type), intent(in)  :: HI  !< Horizontal index type for input variables.
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T  !< Potential temperature of the layer [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S  !< Salinity of the layer [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: z_t !< Height at the top of the layer in depth units [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: z_b !< Height at the bottom of the layer [Z ~> m]
  real,                 intent(in)  :: rho_ref !< A mean density [R ~> kg m-3], that is
                                          !! subtracted out to reduce the magnitude
                                          !! of each of the integrals.
  real,                 intent(in)  :: rho_0 !< A density [R ~> kg m-3], that is used
                                          !! to calculate the pressure (as p~=-z*rho_0*G_e)
                                          !! used in the equation of state.
  real,                 intent(in)  :: G_e !< The Earth's gravitational acceleration
                                          !! [L2 Z-1 T-2 ~> m s-2]
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US !< A dimensional unit scaling type
  real, dimension(SZI_(HI),SZJ_(HI)), &
                      intent(inout) :: dpa !< The change in the pressure anomaly
                                          !! across the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
            optional, intent(inout) :: intz_dpa !< The integral through the thickness of the
                                          !! layer of the pressure anomaly relative to the
                                          !! anomaly at the top of the layer [R L2 Z T-2 ~> Pa m]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
            optional, intent(inout) :: intx_dpa !< The integral in x of the difference between
                                          !! the pressure anomaly at the top and bottom of the
                                          !! layer divided by the x grid spacing [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJB_(HI)), &
            optional, intent(inout) :: inty_dpa !< The integral in y of the difference between
                                          !! the pressure anomaly at the top and bottom of the
                                          !! layer divided by the y grid spacing [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: bathyT !< The depth of the bathymetry [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: SSH !< The sea surface height [Z ~> m]
  real,       optional, intent(in)  :: dz_neglect !< A minuscule thickness change [Z ~> m]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals
  logical,    optional, intent(in)  :: use_inaccurate_form !< If true, uses an inaccurate form of
                                          !! density anomalies, as was used prior to March 2018.
  real, dimension(HI%isd:HI%ied,HI%jsd:HI%jed), &
              optional, intent(in)  :: Z_0p  !< The height at which the pressure is 0 [Z ~> m]

  ! Local variables
  real :: T5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Temperatures along a line of subgrid locations [C ~> degC]
  real :: S5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Salinities along a line of subgrid locations [S ~> ppt]
  real :: p5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Pressures along a line of subgrid locations [R L2 T-2 ~> Pa]
  real :: r5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Densities anomalies along a line of subgrid locations [R ~> kg m-3]
  real :: T15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Temperatures at an array of subgrid locations [C ~> degC]
  real :: S15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Salinities at an array of subgrid locations [S ~> ppt]
  real :: p15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Pressures at an array of subgrid locations [R L2 T-2 ~> Pa]
  real :: r15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Densities at an array of subgrid locations [R ~> kg m-3]
  real :: rho_anom   ! The depth averaged density anomaly [R ~> kg m-3]
  real, parameter :: C1_90 = 1.0/90.0  ! A rational constant [nondim]
  real :: GxRho      ! The product of the gravitational acceleration and reference density [R L2 Z-1 T-2 ~> Pa m-1]
  real :: I_Rho      ! The inverse of the Boussinesq density [R-1 ~> m3 kg-1]
  real :: dz         ! The layer thickness [Z ~> m]
  real :: dz_x(5,HI%iscB:HI%iecB) ! Layer thicknesses along an x-line of subgrid locations [Z ~> m]
  real :: dz_y(5,HI%isc:HI%iec)   ! Layer thicknesses along a y-line of subgrid locations [Z ~> m]
  real :: z0pres(HI%isd:HI%ied,HI%jsd:HI%jed) ! The height at which the pressure is zero [Z ~> m]
  real :: hWght      ! A pressure-thickness below topography [Z ~> m]
  real :: hL, hR     ! Pressure-thicknesses of the columns to the left and right [Z ~> m]
  real :: iDenom     ! The inverse of the denominator in the weights [Z-2 ~> m-2]
  real :: hWt_LL, hWt_LR ! hWt_LA is the weighted influence of A on the left column [nondim]
  real :: hWt_RL, hWt_RR ! hWt_RA is the weighted influence of A on the right column [nondim]
  real :: wt_L, wt_R ! The linear weights of the left and right columns [nondim]
  real :: wtT_L, wtT_R ! The weights for tracers from the left and right columns [nondim]
  real :: intz(5)    ! The gravitational acceleration times the integrals of density
                     ! with height at the 5 sub-column locations [R L2 T-2 ~> Pa]
  logical :: do_massWeight ! Indicates whether to do mass weighting near bathymetry
  logical :: top_massWeight ! Indicates whether to do mass weighting the sea surface
  logical :: use_rho_ref ! Pass rho_ref to the equation of state for more accurate calculation
                         ! of density anomalies.
  integer, dimension(2) :: EOSdom_h5  ! The 5-point h-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_q15 ! The 3x5-point q-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_h15 ! The 3x5-point h-point i-computational domain for the equation of state
  integer :: is, ie, js, je, Isq, Ieq, Jsq, Jeq, i, j, m, n, pos

  ! These array bounds work for the indexing convention of the input arrays, but
  ! on the computational domain defined for the output arrays.
  Isq = HI%IscB ; Ieq = HI%IecB
  Jsq = HI%JscB ; Jeq = HI%JecB
  is = HI%isc ; ie = HI%iec
  js = HI%jsc ; je = HI%jec

  GxRho = G_e * rho_0
  I_Rho = 1.0 / rho_0
  if (present(Z_0p)) then
    do j=Jsq,Jeq+1 ; do i=Isq,Ieq+1
      z0pres(i,j) = Z_0p(i,j)
    enddo ; enddo
  else
    z0pres(:,:) = 0.0
  endif
  use_rho_ref = .true.
  if (present(use_inaccurate_form)) then
    if (use_inaccurate_form) use_rho_ref = .not. use_inaccurate_form
  endif

  do_massWeight = .false. ; top_massWeight = .false.
  if (present(MassWghtInterp)) then
    do_massWeight = BTEST(MassWghtInterp, 0) ! True for odd values
    top_massWeight = BTEST(MassWghtInterp, 1) ! True if the 2 bit is set
    if (do_massWeight .and. .not.present(bathyT)) call MOM_error(FATAL, &
        "int_density_dz_generic: bathyT must be present if near-bottom mass weighting is in use.")
    if (top_massWeight .and. .not.present(SSH)) call MOM_error(FATAL, &
        "int_density_dz_generic: SSH must be present if near-surface mass weighting is in use.")
    if ((do_massWeight .or. top_massWeight) .and. .not.present(dz_neglect)) call MOM_error(FATAL, &
        "int_density_dz_generic: dz_neglect must be present if mass weighting is in use.")
  endif

  ! Set the loop ranges for equation of state calculations at various points.
  EOSdom_h5(1) = 1 ; EOSdom_h5(2) = 5*(Ieq-Isq+2)
  EOSdom_q15(1) = 1 ; EOSdom_q15(2) = 15*(Ieq-Isq+1)
  EOSdom_h15(1) = 1 ; EOSdom_h15(2) = 15*(HI%iec-HI%isc+1)

  do j=Jsq,Jeq+1
    do i=Isq,Ieq+1
      dz = z_t(i,j) - z_b(i,j)
      do n=1,5
        T5(i*5+n) = T(i,j) ; S5(i*5+n) = S(i,j)
        p5(i*5+n) = -GxRho*((z_t(i,j) - z0pres(i,j)) - 0.25*real(n-1)*dz)
      enddo
    enddo

    if (use_rho_ref) then
      call calculate_density(T5, S5, p5, r5, EOS, EOSdom_h5, rho_ref=rho_ref)
    else
      call calculate_density(T5, S5, p5, r5, EOS, EOSdom_h5)
    endif

    do i=Isq,Ieq+1
      ! Use Boole's rule to estimate the pressure anomaly change.
      rho_anom = C1_90*(7.0*(r5(i*5+1)+r5(i*5+5)) + 32.0*(r5(i*5+2)+r5(i*5+4)) + 12.0*r5(i*5+3))
      if (.not.use_rho_ref) rho_anom = rho_anom - rho_ref
      dz = z_t(i,j) - z_b(i,j)
      dpa(i,j) = G_e*dz*rho_anom
      ! Use a Boole's-rule-like fifth-order accurate estimate of the double integral of
      ! the pressure anomaly.
      if (present(intz_dpa)) intz_dpa(i,j) = 0.5*G_e*dz**2 * &
            (rho_anom - C1_90*(16.0*(r5(i*5+4)-r5(i*5+2)) + 7.0*(r5(i*5+5)-r5(i*5+1))) )
    enddo
  enddo

  if (present(intx_dpa)) then ; do j=js,je
    do I=Isq,Ieq
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation of
      ! T & S along the top and bottom integrals, akin to thickness weighting.
      hWght = 0.0
      if (do_massWeight) &
        hWght = max(0., -bathyT(i,j)-z_t(i+1,j), -bathyT(i+1,j)-z_t(i,j))
      if (top_massWeight) &
        hWght = max(hWght, z_b(i+1,j)-SSH(i,j), z_b(i,j)-SSH(i+1,j))
      if (hWght > 0.) then
        hL = (z_t(i,j) - z_b(i,j)) + dz_neglect
        hR = (z_t(i+1,j) - z_b(i+1,j)) + dz_neglect
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
        hWt_LL = (hWght*hL + hR*hL) * iDenom ; hWt_LR = (hWght*hR) * iDenom
        hWt_RR = (hWght*hR + hR*hL) * iDenom ; hWt_RL = (hWght*hL) * iDenom
      else
        hWt_LL = 1.0 ; hWt_LR = 0.0 ; hWt_RR = 1.0 ; hWt_RL = 0.0
      endif

      do m=2,4
        ! T, S, and z are interpolated in the horizontal.  The z interpolation
        ! is linear, but for T and S it may be thickness weighted.
        wt_L = 0.25*real(5-m) ; wt_R = 1.0-wt_L
        wtT_L = (wt_L*hWt_LL) + (wt_R*hWt_RL) ; wtT_R = (wt_L*hWt_LR) + (wt_R*hWt_RR)
        dz_x(m,i) = (wt_L*(z_t(i,j) - z_b(i,j))) + (wt_R*(z_t(i+1,j) - z_b(i+1,j)))
        pos = i*15+(m-2)*5
        T15(pos+1) = (wtT_L*T(i,j)) + (wtT_R*T(i+1,j))
        S15(pos+1) = (wtT_L*S(i,j)) + (wtT_R*S(i+1,j))
        p15(pos+1) = -GxRho * ((wt_L*(z_t(i,j)-z0pres(i,j))) + (wt_R*(z_t(i+1,j)-z0pres(i+1,j))))
        do n=2,5
          T15(pos+n) = T15(pos+1) ; S15(pos+n) = S15(pos+1)
          p15(pos+n) = p15(pos+n-1) + GxRho*0.25*dz_x(m,i)
        enddo
      enddo
    enddo

    if (use_rho_ref) then
      call calculate_density(T15, S15, p15, r15, EOS, EOSdom_q15, rho_ref=rho_ref)
    else
      call calculate_density(T15, S15, p15, r15, EOS, EOSdom_q15)
    endif

    do I=Isq,Ieq
      intz(1) = dpa(i,j) ; intz(5) = dpa(i+1,j)
      ! Use Boole's rule to estimate the pressure anomaly change.
      if (use_rho_ref) then
        do m=2,4
          pos = i*15+(m-2)*5
          intz(m) = (G_e*dz_x(m,i)*(C1_90*( 7.0*(r15(pos+1)+r15(pos+5)) + &
                                           32.0*(r15(pos+2)+r15(pos+4)) + &
                                           12.0*r15(pos+3)) ))
        enddo
      else
        do m=2,4
          pos = i*15+(m-2)*5
          intz(m) = (G_e*dz_x(m,i)*(C1_90*( 7.0*(r15(pos+1)+r15(pos+5)) + &
                                           32.0*(r15(pos+2)+r15(pos+4)) + &
                                           12.0*r15(pos+3)) - rho_ref ))
        enddo
      endif
      ! Use Boole's rule to integrate the bottom pressure anomaly values in x.
      intx_dpa(i,j) = C1_90*(7.0*(intz(1)+intz(5)) + 32.0*(intz(2)+intz(4)) + &
                             12.0*intz(3))
    enddo
  enddo ; endif

  if (present(inty_dpa)) then ; do J=Jsq,Jeq
    do i=is,ie
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation of
      ! T & S along the top and bottom integrals, akin to thickness weighting.
      hWght = 0.0
      if (do_massWeight) &
        hWght = max(0., -bathyT(i,j)-z_t(i,j+1), -bathyT(i,j+1)-z_t(i,j))
      if (top_massWeight) &
        hWght = max(hWght, z_b(i,j+1)-SSH(i,j), z_b(i,j)-SSH(i,j+1))
      if (hWght > 0.) then
        hL = (z_t(i,j) - z_b(i,j)) + dz_neglect
        hR = (z_t(i,j+1) - z_b(i,j+1)) + dz_neglect
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
        hWt_LL = (hWght*hL + hR*hL) * iDenom ; hWt_LR = (hWght*hR) * iDenom
        hWt_RR = (hWght*hR + hR*hL) * iDenom ; hWt_RL = (hWght*hL) * iDenom
      else
        hWt_LL = 1.0 ; hWt_LR = 0.0 ; hWt_RR = 1.0 ; hWt_RL = 0.0
      endif

      do m=2,4
        ! T, S, and z are interpolated in the horizontal.  The z interpolation
        ! is linear, but for T and S it may be thickness weighted.
        wt_L = 0.25*real(5-m) ; wt_R = 1.0-wt_L
        wtT_L = (wt_L*hWt_LL) + (wt_R*hWt_RL) ; wtT_R = (wt_L*hWt_LR) + (wt_R*hWt_RR)
        dz_y(m,i) = (wt_L*(z_t(i,j) - z_b(i,j))) + (wt_R*(z_t(i,j+1) - z_b(i,j+1)))
        pos = i*15+(m-2)*5
        T15(pos+1) = (wtT_L*T(i,j)) + (wtT_R*T(i,j+1))
        S15(pos+1) = (wtT_L*S(i,j)) + (wtT_R*S(i,j+1))
        p15(pos+1) = -GxRho * ((wt_L*(z_t(i,j)-z0pres(i,j))) + (wt_R*(z_t(i,j+1)-z0pres(i,j+1))))
        do n=2,5
          T15(pos+n) = T15(pos+1) ; S15(pos+n) = S15(pos+1)
          p15(pos+n) = p15(pos+n-1) + GxRho*0.25*dz_y(m,i)
        enddo
      enddo
    enddo

    if (use_rho_ref) then
      call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                             r15(15*HI%isc+1:), EOS, EOSdom_h15, rho_ref=rho_ref)
    else
      call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                             r15(15*HI%isc+1:), EOS, EOSdom_h15)
    endif

    do i=is,ie
      intz(1) = dpa(i,j) ; intz(5) = dpa(i,j+1)
      ! Use Boole's rule to estimate the pressure anomaly change.
      do m=2,4
        pos = i*15+(m-2)*5
        if (use_rho_ref) then
          intz(m) = (G_e*dz_y(m,i)*(C1_90*(7.0*(r15(pos+1)+r15(pos+5)) + &
                                          32.0*(r15(pos+2)+r15(pos+4)) + &
                                          12.0*r15(pos+3)) ))
        else
          intz(m) = (G_e*dz_y(m,i)*(C1_90*(7.0*(r15(pos+1)+r15(pos+5)) + &
                                          32.0*(r15(pos+2)+r15(pos+4)) + &
                                          12.0*r15(pos+3)) - rho_ref ))
        endif
      enddo
      ! Use Boole's rule to integrate the values.
      inty_dpa(i,j) = C1_90*(7.0*(intz(1)+intz(5)) + 32.0*(intz(2)+intz(4)) + &
                                       12.0*intz(3))
    enddo
  enddo ; endif
end subroutine int_density_dz_generic_pcm


!> Compute pressure gradient force integrals by quadrature for the case where
!! T and S are linear profiles.
subroutine int_density_dz_generic_plm(k, tv, T_t, T_b, S_t, S_b, e, rho_ref, &
                                      rho_0, G_e, dz_subroundoff, bathyT, HI, GV, EOS, US, use_stanley_eos, dpa, &
                                      intz_dpa, intx_dpa, inty_dpa, MassWghtInterp, &
                                      use_inaccurate_form, Z_0p)
  integer,              intent(in)  :: k   !< Layer index to calculate integrals for
  type(hor_index_type), intent(in)  :: HI  !< Ocean horizontal index structures for the input arrays
  type(verticalGrid_type), intent(in) :: GV !< Vertical grid structure
  type(thermo_var_ptrs), intent(in) :: tv  !< Thermodynamic variables
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: T_t !< Potential temperature at the cell top [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: T_b !< Potential temperature at the cell bottom [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: S_t !< Salinity at the cell top [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: S_b !< Salinity at the cell bottom [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)+1), &
                        intent(in)  :: e   !< Height of interfaces [Z ~> m]
  real,                 intent(in)  :: rho_ref !< A mean density [R ~> kg m-3], that is subtracted
                                           !! out to reduce the magnitude of each of the integrals.
  real,                 intent(in)  :: rho_0 !< A density [R ~> kg m-3], that is used to calculate
                                           !! the pressure (as p~=-z*rho_0*G_e) used in the equation of state.
  real,                 intent(in)  :: G_e !< The Earth's gravitational acceleration [L2 Z-1 T-2 ~> m s-2]
  real,                 intent(in)  :: dz_subroundoff !< A minuscule thickness change [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: bathyT !< The depth of the bathymetry [Z ~> m]
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US !< A dimensional unit scaling type
  logical,              intent(in) :: use_stanley_eos !< If true, turn on Stanley SGS T variance parameterization
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(inout) :: dpa !< The change in the pressure anomaly across the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intz_dpa !< The integral through the thickness of the layer of
                                           !! the pressure anomaly relative to the anomaly at the
                                           !! top of the layer [R L2 Z T-2 ~> Pa m]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intx_dpa !< The integral in x of the difference between the
                                           !! pressure anomaly at the top and bottom of the layer
                                           !! divided by the x grid spacing [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJB_(HI)), &
              optional, intent(inout) :: inty_dpa !< The integral in y of the difference between the
                                           !! pressure anomaly at the top and bottom of the layer
                                           !! divided by the y grid spacing [R L2 T-2 ~> Pa]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals
  logical,    optional, intent(in)  :: use_inaccurate_form !< If true, uses an inaccurate form of
                                           !! density anomalies, as was used prior to March 2018.
  real, dimension(HI%isd:HI%ied,HI%jsd:HI%jed), &
              optional, intent(in)  :: Z_0p !< The height at which the pressure is 0 [Z ~> m]

! This subroutine calculates (by numerical quadrature) integrals of
! pressure anomalies across layers, which are required for calculating the
! finite-volume form pressure accelerations in a Boussinesq model.  The one
! potentially dodgy assumption here is that rho_0 is used both in the denominator
! of the accelerations, and in the pressure used to calculated density (the
! latter being -z*rho_0*G_e).  These two uses could be separated if need be.
!
! It is assumed that the salinity and temperature profiles are linear in the
! vertical. The top and bottom values within each layer are provided and
! a linear interpolation is used to compute intermediate values.

  ! Local variables
  real :: T5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Temperatures along a line of subgrid locations [C ~> degC]
  real :: S5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Salinities along a line of subgrid locations [S ~> ppt]
  real :: T25((5*HI%iscB+1):(5*(HI%iecB+2))) ! SGS temperature variance along a line of subgrid
                                             ! locations [C2 ~> degC2]
  real :: TS5((5*HI%iscB+1):(5*(HI%iecB+2))) ! SGS temp-salt covariance along a line of subgrid
                                             ! locations [C S ~> degC ppt]
  real :: S25((5*HI%iscB+1):(5*(HI%iecB+2))) ! SGS salinity variance along a line of subgrid locations [S2 ~> ppt2]
  real :: p5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Pressures along a line of subgrid locations [R L2 T-2 ~> Pa]
  real :: r5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Densities anomalies along a line of subgrid
                                             ! locations [R ~> kg m-3]
  real :: u5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Densities anomalies along a line of subgrid locations
                                             ! (used for inaccurate form) [R ~> kg m-3]
  real :: T15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Temperatures at an array of subgrid locations [C ~> degC]
  real :: S15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Salinities at an array of subgrid locations [S ~> ppt]
  real :: T215((15*HI%iscB+1):(15*(HI%iecB+1))) ! SGS temperature variance along a line of subgrid
                                                ! locations [C2 ~> degC2]
  real :: TS15((15*HI%iscB+1):(15*(HI%iecB+1))) ! SGS temp-salt covariance along a line of subgrid
                                                ! locations [C S ~> degC ppt]
  real :: S215((15*HI%iscB+1):(15*(HI%iecB+1))) ! SGS salinity variance along a line of subgrid
                                                ! locations [S2 ~> ppt2]
  real :: p15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Pressures at an array of subgrid locations [R L2 T-2 ~> Pa]
  real :: r15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Densities at an array of subgrid locations [R ~> kg m-3]
  real :: wt_t(5), wt_b(5)          ! Top and bottom weights [nondim]
  real :: rho_anom                  ! A density anomaly [R ~> kg m-3]
  real :: w_left, w_right           ! Left and right weights [nondim]
  real :: intz(5)    ! The gravitational acceleration times the integrals of density
                     ! with height at the 5 sub-column locations [R L2 T-2 ~> Pa]
  real, parameter :: C1_90 = 1.0/90.0  ! A rational constant [nondim]
  real :: GxRho      ! The product of the gravitational acceleration and reference density [R L2 Z-1 T-2 ~> Pa m-1]
  real :: I_Rho      ! The inverse of the Boussinesq density [R-1 ~> m3 kg-1]
  real :: dz(HI%iscB:HI%iecB+1)   ! Layer thicknesses at tracer points [Z ~> m]
  real :: dz_x(5,HI%iscB:HI%iecB) ! Layer thicknesses along an x-line of subgrid locations [Z ~> m]
  real :: dz_y(5,HI%isc:HI%iec)   ! Layer thicknesses along a y-line of subgrid locations [Z ~> m]
  real :: massWeightToggle          ! A non-dimensional toggle factor for near-bottom mass weighting (0 or 1) [nondim]
  real :: TopWeightToggle           ! A non-dimensional toggle factor for near-surface mass weighting (0 or 1) [nondim]
  real :: Ttl, Tbl, Ttr, Tbr        ! Temperatures at the velocity cell corners [C ~> degC]
  real :: Stl, Sbl, Str, Sbr        ! Salinities at the velocity cell corners [S ~> ppt]
  real :: z0pres(HI%isd:HI%ied,HI%jsd:HI%jed) ! The height at which the pressure is zero [Z ~> m]
  real :: hWght                     ! A topographically limited thickness weight [Z ~> m]
  real :: hWghtTop                  ! An ice draft limited thickness weight [Z ~> m]
  real :: hL, hR                    ! Thicknesses to the left and right [Z ~> m]
  real :: iDenom                    ! The denominator of the thickness weight expressions [Z-2 ~> m-2]
  logical :: use_rho_ref ! Pass rho_ref to the equation of state for more accurate calculation
                         ! of density anomalies.
  logical :: use_varT, use_varS, use_covarTS ! Logicals for SGS variances fields
  integer, dimension(2) :: EOSdom_h5  ! The 5-point h-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_q15 ! The 3x5-point q-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_h15 ! The 3x5-point h-point i-computational domain for the equation of state
  integer :: Isq, Ieq, Jsq, Jeq, i, j, m, n, pos

  Isq = HI%IscB ; Ieq = HI%IecB ; Jsq = HI%JscB ; Jeq = HI%JecB

  GxRho = G_e * rho_0
  I_Rho = 1.0 / rho_0
  if (present(Z_0p)) then
    do j=Jsq,Jeq+1 ; do i=Isq,Ieq+1
      z0pres(i,j) = Z_0p(i,j)
    enddo ; enddo
  else
    z0pres(:,:) = 0.0
  endif
  massWeightToggle = 0. ; TopWeightToggle = 0.
  if (present(MassWghtInterp)) then
    if (BTEST(MassWghtInterp, 0)) massWeightToggle = 1.
    if (BTEST(MassWghtInterp, 1)) TopWeightToggle = 1.
  endif
  use_rho_ref = .true.
  if (present(use_inaccurate_form)) use_rho_ref = .not. use_inaccurate_form

  use_varT = .false. !ensure initialized
  use_covarTS = .false.
  use_varS = .false.
  if (use_stanley_eos) then
    use_varT = associated(tv%varT)
    use_covarTS = associated(tv%covarTS)
    use_varS = associated(tv%varS)
  endif

  T25(:) = 0.
  TS5(:) = 0.
  S25(:) = 0.
  T215(:) = 0.
  TS15(:) = 0.
  S215(:) = 0.

  do n = 1, 5
    wt_t(n) = 0.25 * real(5-n)
    wt_b(n) = 1.0 - wt_t(n)
  enddo

  ! Set the loop ranges for equation of state calculations at various points.
  EOSdom_h5(1) = 1 ; EOSdom_h5(2) = 5*(Ieq-Isq+2)
  EOSdom_q15(1) = 1 ; EOSdom_q15(2) = 15*(Ieq-Isq+1)
  EOSdom_h15(1) = 1 ; EOSdom_h15(2) = 15*(HI%iec-HI%isc+1)

  ! 1. Compute vertical integrals
  do j=Jsq,Jeq+1
    do i = Isq,Ieq+1
      dz(i) = e(i,j,K) - e(i,j,K+1)
      do n=1,5
        p5(i*5+n) = -GxRho*((e(i,j,K) - z0pres(i,j)) - 0.25*real(n-1)*dz(i))
        ! Salinity and temperature points are linearly interpolated
        S5(i*5+n) = wt_t(n) * S_t(i,j,k) + wt_b(n) * S_b(i,j,k)
        T5(i*5+n) = wt_t(n) * T_t(i,j,k) + wt_b(n) * T_b(i,j,k)
      enddo
      if (use_varT) T25(i*5+1:i*5+5) = tv%varT(i,j,k)
      if (use_covarTS) TS5(i*5+1:i*5+5) = tv%covarTS(i,j,k)
      if (use_varS) S25(i*5+1:i*5+5) = tv%varS(i,j,k)
    enddo
    if (use_Stanley_eos) then
      call calculate_density(T5, S5, p5, T25, TS5, S25, r5, EOS, EOSdom_h5, rho_ref=rho_ref)
    else
      if (use_rho_ref) then
        call calculate_density(T5, S5, p5, r5, EOS, EOSdom_h5, rho_ref=rho_ref)
      else
        call calculate_density(T5, S5, p5, r5, EOS, EOSdom_h5)
        u5(:) = r5(:) - rho_ref
      endif
    endif

    if (use_rho_ref) then
      do i=Isq,Ieq+1
        ! Use Boole's rule to estimate the pressure anomaly change.
        rho_anom = C1_90*(7.0*(r5(i*5+1)+r5(i*5+5)) + 32.0*(r5(i*5+2)+r5(i*5+4)) + 12.0*r5(i*5+3))
        dpa(i,j) = G_e*dz(i)*rho_anom
        if (present(intz_dpa)) then
          ! Use a Boole's-rule-like fifth-order accurate estimate of
          ! the double integral of the pressure anomaly.
          intz_dpa(i,j) = 0.5*G_e*dz(i)**2 * &
                  (rho_anom - C1_90*(16.0*(r5(i*5+4)-r5(i*5+2)) + 7.0*(r5(i*5+5)-r5(i*5+1))) )
        endif
      enddo
    else
      do i=Isq,Ieq+1
        ! Use Boole's rule to estimate the pressure anomaly change.
        rho_anom = C1_90*(7.0*(r5(i*5+1)+r5(i*5+5)) + 32.0*(r5(i*5+2)+r5(i*5+4)) + 12.0*r5(i*5+3)) &
                   - rho_ref
        dpa(i,j) = G_e*dz(i)*rho_anom
        if (present(intz_dpa)) then
          ! Use a Boole's-rule-like fifth-order accurate estimate of
          ! the double integral of the pressure anomaly.
          intz_dpa(i,j) = 0.5*G_e*dz(i)**2 * &
                  (rho_anom - C1_90*(16.0*(u5(i*5+4)-u5(i*5+2)) + 7.0*(u5(i*5+5)-u5(i*5+1))) )
        endif
      enddo
    endif
  enddo ! end loops on j

  ! 2. Compute horizontal integrals in the x direction
  if (present(intx_dpa)) then ; do j=HI%jsc,HI%jec
    do I=Isq,Ieq
      ! Corner values of T and S
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation
      ! of T,S along the top and bottom integrals, almost like thickness
      ! weighting.
      ! Note: To work in terrain following coordinates we could offset
      ! this distance by the layer thickness to replicate other models.
      hWght = massWeightToggle * &
              max(0., -bathyT(i,j)-e(i+1,j,K), -bathyT(i+1,j)-e(i,j,K))
      ! CY: The below code just uses top interface, which may be bad in high res open ocean
      ! We want something like if (pa(i+1,k+1)<pa(i,1)) or (pa(i+1,1) <pa(i,k+1)) then...
      ! but pressures are not passed through to this submodule, and tv just has surface press.
      !if ((p(i+1,j,k+1)<p(i,j,1)).or.(tv%p(i+1,j,k+1)<tv%p(i,j,1))) then
      hWghtTop = TopWeightToggle * &
              max(0., e(i+1,j,K+1)-e(i,j,1), e(i,j,K+1)-e(i+1,j,1))
      !else ! pressure criteria not activated
      !  hWghtTop = 0.
      !endif
      ! Set it to be max of the bottom and top hWghts:
      hWght = max(hWght, hWghtTop)
      if (hWght > 0.) then
        hL = (e(i,j,K) - e(i,j,K+1)) + dz_subroundoff
        hR = (e(i+1,j,K) - e(i+1,j,K+1)) + dz_subroundoff
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1./( hWght*(hR + hL) + hL*hR )
        Ttl = ( (hWght*hR)*T_t(i+1,j,k) + (hWght*hL + hR*hL)*T_t(i,j,k) ) * iDenom
        Ttr = ( (hWght*hL)*T_t(i,j,k) + (hWght*hR + hR*hL)*T_t(i+1,j,k) ) * iDenom
        Tbl = ( (hWght*hR)*T_b(i+1,j,k) + (hWght*hL + hR*hL)*T_b(i,j,k) ) * iDenom
        Tbr = ( (hWght*hL)*T_b(i,j,k) + (hWght*hR + hR*hL)*T_b(i+1,j,k) ) * iDenom
        Stl = ( (hWght*hR)*S_t(i+1,j,k) + (hWght*hL + hR*hL)*S_t(i,j,k) ) * iDenom
        Str = ( (hWght*hL)*S_t(i,j,k) + (hWght*hR + hR*hL)*S_t(i+1,j,k) ) * iDenom
        Sbl = ( (hWght*hR)*S_b(i+1,j,k) + (hWght*hL + hR*hL)*S_b(i,j,k) ) * iDenom
        Sbr = ( (hWght*hL)*S_b(i,j,k) + (hWght*hR + hR*hL)*S_b(i+1,j,k) ) * iDenom
      else
        Ttl = T_t(i,j,k); Tbl = T_b(i,j,k); Ttr = T_t(i+1,j,k); Tbr = T_b(i+1,j,k)
        Stl = S_t(i,j,k); Sbl = S_b(i,j,k); Str = S_t(i+1,j,k); Sbr = S_b(i+1,j,k)
      endif

      do m=2,4
        w_left = wt_t(m) ; w_right = wt_b(m)
        dz_x(m,i) = (w_left*(e(i,j,K) - e(i,j,K+1))) + (w_right*(e(i+1,j,K) - e(i+1,j,K+1)))

        ! Salinity and temperature points are linearly interpolated in
        ! the horizontal. The subscript (1) refers to the top value in
        ! the vertical profile while subscript (5) refers to the bottom
        ! value in the vertical profile.
        pos = i*15+(m-2)*5
        T15(pos+1) = (w_left*Ttl) + (w_right*Ttr)
        T15(pos+5) = (w_left*Tbl) + (w_right*Tbr)

        S15(pos+1) = (w_left*Stl) + (w_right*Str)
        S15(pos+5) = (w_left*Sbl) + (w_right*Sbr)

        p15(pos+1) = -GxRho * ((w_left*(e(i,j,K)-z0pres(i,j))) + (w_right*(e(i+1,j,K)-z0pres(i+1,j))))

        ! Pressure
        do n=2,5
          p15(pos+n) = p15(pos+n-1) + GxRho*0.25*dz_x(m,i)
        enddo

        ! Salinity and temperature (linear interpolation in the vertical)
        do n=2,4
          S15(pos+n) = wt_t(n) * S15(pos+1) + wt_b(n) * S15(pos+5)
          T15(pos+n) = wt_t(n) * T15(pos+1) + wt_b(n) * T15(pos+5)
        enddo
        if (use_varT) T215(pos+1:pos+5) = (w_left*tv%varT(i,j,k)) + (w_right*tv%varT(i+1,j,k))
        if (use_covarTS) TS15(pos+1:pos+5) = (w_left*tv%covarTS(i,j,k)) + (w_right*tv%covarTS(i+1,j,k))
        if (use_varS) S215(pos+1:pos+5) = (w_left*tv%varS(i,j,k)) + (w_right*tv%varS(i+1,j,k))
      enddo
    enddo

    if (use_stanley_eos) then
      call calculate_density(T15, S15, p15, T215, TS15, S215, r15, EOS, EOSdom_q15, rho_ref=rho_ref)
    else
      if (use_rho_ref) then
        call calculate_density(T15, S15, p15, r15, EOS, EOSdom_q15, rho_ref=rho_ref)
      else
        call calculate_density(T15, S15, p15, r15, EOS, EOSdom_q15)
      endif
    endif

    do I=Isq,Ieq
      intz(1) = dpa(i,j) ; intz(5) = dpa(i+1,j)

      ! Use Boole's rule to estimate the pressure anomaly change.
      if (use_rho_ref) then
        do m = 2,4
          pos = i*15+(m-2)*5
          intz(m) = (G_e*dz_x(m,i)*( C1_90*(7.0*(r15(pos+1)+r15(pos+5)) + 32.0*(r15(pos+2)+r15(pos+4)) + &
                            12.0*r15(pos+3)) ))
        enddo
      else
        do m = 2,4
          pos = i*15+(m-2)*5
          intz(m) = (G_e*dz_x(m,i)*( C1_90*(7.0*(r15(pos+1)+r15(pos+5)) + 32.0*(r15(pos+2)+r15(pos+4)) + &
                            12.0*r15(pos+3)) - rho_ref ))
        enddo
      endif
      ! Use Boole's rule to integrate the bottom pressure anomaly values in x.
      intx_dpa(I,j) = C1_90*(7.0*(intz(1)+intz(5)) + 32.0*(intz(2)+intz(4)) + &
                             12.0*intz(3))
    enddo
  enddo ; endif

  ! 3. Compute horizontal integrals in the y direction
  if (present(inty_dpa)) then ; do J=Jsq,Jeq
    do i=HI%isc,HI%iec
    ! Corner values of T and S
    ! hWght is the distance measure by which the cell is violation of
    ! hydrostatic consistency. For large hWght we bias the interpolation
    ! of T,S along the top and bottom integrals, almost like thickness
    ! weighting.
    ! Note: To work in terrain following coordinates we could offset
    ! this distance by the layer thickness to replicate other models.
      hWght = massWeightToggle * &
              max(0., -bathyT(i,j)-e(i,j+1,K), -bathyT(i,j+1)-e(i,j,K))
      ! CY: The below code just uses top interface, which may be bad in high res open ocean
      ! We want something like if (pa(j+1,k+1)<pa(j,1)) or (pa(j+1,1) <pa(i,j,k+1)) then...
      ! but pressures are not passed through to this submodule, and tv just has surface press.
      !if ((p(i,j+1,k+1)<p(i,j,1)).or.(tv%p(i,j+1,k+1)<tv%p(i,j,1))) then
      hWghtTop = TopWeightToggle * &
              max(0., e(i,j+1,K+1)-e(i,j,1), e(i,j,K+1)-e(i,j+1,1))
      !else ! pressure criteria not activated
      !  hWghtTop = 0.
      !endif
      ! Set it to be max of the bottom and top hWghts:
      hWght = max(hWght, hWghtTop)
      if (hWght > 0.) then
        hL = (e(i,j,K) - e(i,j,K+1)) + dz_subroundoff
        hR = (e(i,j+1,K) - e(i,j+1,K+1)) + dz_subroundoff
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1./( hWght*(hR + hL) + hL*hR )
        Ttl = ( (hWght*hR)*T_t(i,j+1,k) + (hWght*hL + hR*hL)*T_t(i,j,k) ) * iDenom
        Ttr = ( (hWght*hL)*T_t(i,j,k) + (hWght*hR + hR*hL)*T_t(i,j+1,k) ) * iDenom
        Tbl = ( (hWght*hR)*T_b(i,j+1,k) + (hWght*hL + hR*hL)*T_b(i,j,k) ) * iDenom
        Tbr = ( (hWght*hL)*T_b(i,j,k) + (hWght*hR + hR*hL)*T_b(i,j+1,k) ) * iDenom
        Stl = ( (hWght*hR)*S_t(i,j+1,k) + (hWght*hL + hR*hL)*S_t(i,j,k) ) * iDenom
        Str = ( (hWght*hL)*S_t(i,j,k) + (hWght*hR + hR*hL)*S_t(i,j+1,k) ) * iDenom
        Sbl = ( (hWght*hR)*S_b(i,j+1,k) + (hWght*hL + hR*hL)*S_b(i,j,k) ) * iDenom
        Sbr = ( (hWght*hL)*S_b(i,j,k) + (hWght*hR + hR*hL)*S_b(i,j+1,k) ) * iDenom
      else
        Ttl = T_t(i,j,k); Tbl = T_b(i,j,k); Ttr = T_t(i,j+1,k); Tbr = T_b(i,j+1,k)
        Stl = S_t(i,j,k); Sbl = S_b(i,j,k); Str = S_t(i,j+1,k); Sbr = S_b(i,j+1,k)
      endif

      do m=2,4
        w_left = wt_t(m) ; w_right = wt_b(m)
        dz_y(m,i) = (w_left*(e(i,j,K) - e(i,j,K+1))) + (w_right*(e(i,j+1,K) - e(i,j+1,K+1)))

        ! Salinity and temperature points are linearly interpolated in
        ! the horizontal. The subscript (1) refers to the top value in
        ! the vertical profile while subscript (5) refers to the bottom
        ! value in the vertical profile.
        pos = i*15+(m-2)*5
        T15(pos+1) = (w_left*Ttl) + (w_right*Ttr)
        T15(pos+5) = (w_left*Tbl) + (w_right*Tbr)

        S15(pos+1) = (w_left*Stl) + (w_right*Str)
        S15(pos+5) = (w_left*Sbl) + (w_right*Sbr)

        p15(pos+1) = -GxRho * ((w_left*(e(i,j,K)-z0pres(i,j))) + (w_right*(e(i,j+1,K)-z0pres(i,j+1))))

        ! Pressure
        do n=2,5
          p15(pos+n) = p15(pos+n-1) + GxRho*0.25*dz_y(m,i)
        enddo

        ! Salinity and temperature (linear interpolation in the vertical)
        do n=2,4
          S15(pos+n) = wt_t(n) * S15(pos+1) + wt_b(n) * S15(pos+5)
          T15(pos+n) = wt_t(n) * T15(pos+1) + wt_b(n) * T15(pos+5)
        enddo
        if (use_varT) T215(pos+1:pos+5) = (w_left*tv%varT(i,j,k)) + (w_right*tv%varT(i,j+1,k))
        if (use_covarTS) TS15(pos+1:pos+5) = (w_left*tv%covarTS(i,j,k)) + (w_right*tv%covarTS(i,j+1,k))
        if (use_varS) S215(pos+1:pos+5) = (w_left*tv%varS(i,j,k)) + (w_right*tv%varS(i,j+1,k))
      enddo
    enddo

    if (use_stanley_eos) then
      call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                             T215(15*HI%isc+1:), TS15(15*HI%isc+1:), S215(15*HI%isc+1:), &
                             r15(15*HI%isc+1:), EOS, EOSdom_h15, rho_ref=rho_ref)
    else
      if (use_rho_ref) then
        call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                               r15(15*HI%isc+1:), EOS, EOSdom_h15, rho_ref=rho_ref)
      else
        call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                               r15(15*HI%isc+1:), EOS, EOSdom_h15)
      endif
    endif

    do i=HI%isc,HI%iec
      intz(1) = dpa(i,j) ; intz(5) = dpa(i,j+1)

      ! Use Boole's rule to estimate the pressure anomaly change.
      if (use_rho_ref) then
        do m = 2,4
          pos = i*15+(m-2)*5
          intz(m) = (G_e*dz_y(m,i)*( C1_90*(7.0*(r15(pos+1)+r15(pos+5)) + &
                                           32.0*(r15(pos+2)+r15(pos+4)) + &
                                           12.0*r15(pos+3)) ))
        enddo
      else
        do m = 2,4
          pos = i*15+(m-2)*5
          intz(m) = (G_e*dz_y(m,i)*( C1_90*(7.0*(r15(pos+1)+r15(pos+5)) + &
                                           32.0*(r15(pos+2)+r15(pos+4)) + &
                                           12.0*r15(pos+3)) - rho_ref ))
        enddo
      endif
      ! Use Boole's rule to integrate the values.
      inty_dpa(i,J) = C1_90*(7.0*(intz(1)+intz(5)) + 32.0*(intz(2)+intz(4)) + &
                             12.0*intz(3))
    enddo
  enddo ; endif

end subroutine int_density_dz_generic_plm


!> Compute pressure gradient force integrals for layer "k" and the case where T and S
!! are parabolic profiles
subroutine int_density_dz_generic_ppm(k, tv, T_t, T_b, S_t, S_b, e, &
                                      rho_ref, rho_0, G_e, dz_subroundoff, bathyT, HI, GV, EOS, US, use_stanley_eos, &
                                      dpa, intz_dpa, intx_dpa, inty_dpa, MassWghtInterp, Z_0p)
  integer,              intent(in)  :: k   !< Layer index to calculate integrals for
  type(hor_index_type), intent(in)  :: HI  !< Ocean horizontal index structures for the input arrays
  type(verticalGrid_type), intent(in) :: GV !< Vertical grid structure
  type(thermo_var_ptrs), intent(in) :: tv  !< Thermodynamic variables
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: T_t !< Potential temperature at the cell top [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: T_b !< Potential temperature at the cell bottom [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: S_t !< Salinity at the cell top [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)), &
                        intent(in)  :: S_b !< Salinity at the cell bottom [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI),SZK_(GV)+1), &
                        intent(in)  :: e   !< Height of interfaces [Z ~> m]
  real,                 intent(in)  :: rho_ref !< A mean density [R ~> kg m-3], that is
                                           !! subtracted out to reduce the magnitude of each of the integrals.
  real,                 intent(in)  :: rho_0 !< A density [R ~> kg m-3], that is used to calculate
                                           !! the pressure (as p~=-z*rho_0*G_e) used in the equation of state.
  real,                 intent(in)  :: G_e !< The Earth's gravitational acceleration [L2 Z-1 T-2 ~> m s-2]
  real,                 intent(in)  :: dz_subroundoff !< A minuscule thickness change [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: bathyT !< The depth of the bathymetry [Z ~> m]
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US  !< A dimensional unit scaling type
  logical,              intent(in)  :: use_stanley_eos !< If true, turn on Stanley SGS T variance parameterization
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(inout) :: dpa !< The change in the pressure anomaly across the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intz_dpa !< The integral through the thickness of the layer of
                                           !! the pressure anomaly relative to the anomaly at the
                                           !! top of the layer [R L2 Z T-2 ~> Pa m]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intx_dpa !< The integral in x of the difference between the
                                           !! pressure anomaly at the top and bottom of the layer
                                           !! divided by the x grid spacing [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJB_(HI)), &
              optional, intent(inout) :: inty_dpa !< The integral in y of the difference between the
                                           !! pressure anomaly at the top and bottom of the layer
                                           !! divided by the y grid spacing [R L2 T-2 ~> Pa]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals
  real, dimension(HI%isd:HI%ied,HI%jsd:HI%jed), &
              optional, intent(in)  :: Z_0p !< The height at which the pressure is 0 [Z ~> m]

! This subroutine calculates (by numerical quadrature) integrals of
! pressure anomalies across layers, which are required for calculating the
! finite-volume form pressure accelerations in a Boussinesq model.  The one
! potentially dodgy assumption here is that rho_0 is used both in the denominator
! of the accelerations, and in the pressure used to calculated density (the
! latter being -z*rho_0*G_e).  These two uses could be separated if need be.
!
! It is assumed that the salinity and temperature profiles are parabolic in the
! vertical. The top and bottom values within each layer are provided and
! a parabolic interpolation is used to compute intermediate values.

  ! Local variables
  real :: T5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Temperatures along a line of subgrid locations [C ~> degC]
  real :: S5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Salinities along a line of subgrid locations [S ~> ppt]
  real :: T25((5*HI%iscB+1):(5*(HI%iecB+2))) ! SGS temperature variance along a line of subgrid
                                             ! locations [C2 ~> degC2]
  real :: TS5((5*HI%iscB+1):(5*(HI%iecB+2))) ! SGS temp-salt covariance along a line of subgrid
                                             ! locations [C S ~> degC ppt]
  real :: S25((5*HI%iscB+1):(5*(HI%iecB+2))) ! SGS salinity variance along a line of subgrid locations [S2 ~> ppt2]
  real :: p5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Pressures along a line of subgrid locations [R L2 T-2 ~> Pa]
  real :: r5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Densities anomalies along a line of subgrid
                                             ! locations [R ~> kg m-3]
  real :: T15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Temperatures at an array of subgrid locations [C ~> degC]
  real :: S15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Salinities at an array of subgrid locations [S ~> ppt]
  real :: T215((15*HI%iscB+1):(15*(HI%iecB+1))) ! SGS temperature variance along a line of subgrid
                                                ! locations [C2 ~> degC2]
  real :: TS15((15*HI%iscB+1):(15*(HI%iecB+1))) ! SGS temp-salt covariance along a line of subgrid
                                                ! locations [C S ~> degC ppt]
  real :: S215((15*HI%iscB+1):(15*(HI%iecB+1))) ! SGS salinity variance along a line of subgrid
                                                ! locations [S2 ~> ppt2]
  real :: p15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Pressures at an array of subgrid locations [R L2 T-2 ~> Pa]
  real :: r15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Densities at an array of subgrid locations [R ~> kg m-3]
  real :: wt_t(5), wt_b(5) ! Top and bottom weights [nondim]
  real :: rho_anom ! The integrated density anomaly [R ~> kg m-3]
  real :: w_left, w_right  ! Left and right weights [nondim]
  real :: intz(5) ! The gravitational acceleration times the integrals of density
                  ! with height at the 5 sub-column locations [R L2 T-2 ~> Pa]
  real, parameter :: C1_90 = 1.0/90.0  ! A rational constant [nondim]
  real :: GxRho ! The gravitational acceleration times density [R L2 Z-1 T-2 ~> kg m-2 s-2]
  real :: I_Rho ! The inverse of the Boussinesq density [R-1 ~> m3 kg-1]
  real :: dz ! Layer thicknesses at tracer points [Z ~> m]
  real :: dz_x(5,HI%iscB:HI%iecB) ! Layer thicknesses along an x-line of subgrid locations [Z ~> m]
  real :: dz_y(5,HI%isc:HI%iec)   ! Layer thicknesses along a y-line of subgrid locations [Z ~> m]
  real :: massWeightToggle        ! A non-dimensional toggle factor for near-bottom mass weighting (0 or 1) [nondim]
  real :: TopWeightToggle         ! A non-dimensional toggle factor for near-surface mass weighting (0 or 1) [nondim]
  real :: Ttl, Tbl, Tml, Ttr, Tbr, Tmr ! Temperatures at the velocity cell corners [C ~> degC]
  real :: Stl, Sbl, Sml, Str, Sbr, Smr ! Salinities at the velocity cell corners [S ~> ppt]
  real :: s6 ! PPM curvature coefficient for S [S ~> ppt]
  real :: t6 ! PPM curvature coefficient for T [C ~> degC]
  real :: T_top, T_mn, T_bot ! Left edge, cell mean and right edge values used in PPM reconstructions of T [C ~> degC]
  real :: S_top, S_mn, S_bot ! Left edge, cell mean and right edge values used in PPM reconstructions of S [S ~> ppt]
  real :: z0pres(HI%isd:HI%ied,HI%jsd:HI%jed) ! The height at which the pressure is zero [Z ~> m]
  real :: hWght  ! A topographically limited thickness weight [Z ~> m]
  real :: hWghtTop ! A surface displacement limited thickness weight [Z ~> m]
  real :: hL, hR ! Thicknesses to the left and right [Z ~> m]
  real :: iDenom ! The denominator of the thickness weight expressions [Z-2 ~> m-2]
  integer, dimension(2) :: EOSdom_h5  ! The 5-point h-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_q15 ! The 3x5-point q-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_h15 ! The 3x5-point h-point i-computational domain for the equation of state
  integer :: Isq, Ieq, Jsq, Jeq, i, j, m, n, pos
  logical :: use_PPM ! If false, assume zero curvature in reconstruction, i.e. PLM
  logical :: use_varT, use_varS, use_covarTS ! Logicals for SGS variances fields

  Isq = HI%IscB ; Ieq = HI%IecB ; Jsq = HI%JscB ; Jeq = HI%JecB

  GxRho = G_e * rho_0
  I_Rho = 1.0 / rho_0
  if (present(Z_0p)) then
    do j=Jsq,Jeq+1 ; do i=Isq,Ieq+1
      z0pres(i,j) = Z_0p(i,j)
    enddo ; enddo
  else
    z0pres(:,:) = 0.0
  endif
  massWeightToggle = 0. ; TopWeightToggle = 0.
  if (present(MassWghtInterp)) then
    if (BTEST(MassWghtInterp, 0)) massWeightToggle = 1.
    if (BTEST(MassWghtInterp, 1)) TopWeightToggle = 1.
  endif

  ! In event PPM calculation is bypassed with use_PPM=False
  s6 = 0.
  t6 = 0.
  use_PPM = .true. ! This is a place-holder to allow later re-use of this function

  use_varT = .false. !ensure initialized
  use_covarTS = .false.
  use_varS = .false.
  if (use_stanley_eos) then
    use_varT = associated(tv%varT)
    use_covarTS = associated(tv%covarTS)
    use_varS = associated(tv%varS)
  endif

  T25(:) = 0.
  TS5(:) = 0.
  S25(:) = 0.
  T215(:) = 0.
  TS15(:) = 0.
  S215(:) = 0.

  do n = 1, 5
    wt_t(n) = 0.25 * real(5-n)
    wt_b(n) = 1.0 - wt_t(n)
  enddo

  ! Set the loop ranges for equation of state calculations at various points.
  EOSdom_h5(1) = 1 ; EOSdom_h5(2) = 5*(Ieq-Isq+2)
  EOSdom_q15(1) = 1 ; EOSdom_q15(2) = 15*(Ieq-Isq+1)
  EOSdom_h15(1) = 1 ; EOSdom_h15(2) = 15*(HI%iec-HI%isc+1)

  ! 1. Compute vertical integrals
  do j=Jsq,Jeq+1
    do i=Isq,Ieq+1
      if (use_PPM) then
        ! Curvature coefficient of the parabolas
        s6 = 3.0 * ( 2.0*tv%S(i,j,k) - ( S_t(i,j,k) + S_b(i,j,k) ) )
        t6 = 3.0 * ( 2.0*tv%T(i,j,k) - ( T_t(i,j,k) + T_b(i,j,k) ) )
      endif
      dz = e(i,j,K) - e(i,j,K+1)
      do n=1,5
        p5(I*5+n) = -GxRho*((e(i,j,K) - z0pres(i,j)) - 0.25*real(n-1)*dz)
        ! Salinity and temperature points are reconstructed with PPM
        S5(I*5+n) = wt_t(n) * S_t(i,j,k) + wt_b(n) * ( S_b(i,j,k) + s6 * wt_t(n) )
        T5(I*5+n) = wt_t(n) * T_t(i,j,k) + wt_b(n) * ( T_b(i,j,k) + t6 * wt_t(n) )
      enddo
      if (use_stanley_eos) then
        if (use_varT) T25(I*5+1:I*5+5) = tv%varT(i,j,k)
        if (use_covarTS) TS5(I*5+1:I*5+5) = tv%covarTS(i,j,k)
        if (use_varS) S25(I*5+1:I*5+5) = tv%varS(i,j,k)
      endif
    enddo

    if (use_stanley_eos) then
      call calculate_density(T5, S5, p5, T25, TS5, S25, r5, EOS, EOSdom_h5, rho_ref=rho_ref)
    else
      call calculate_density(T5, S5, p5, r5, EOS, EOSdom_h5, rho_ref=rho_ref)
    endif

    do i=Isq,Ieq+1
      dz = e(i,j,K) - e(i,j,K+1)
      ! Use Boole's rule to estimate the pressure anomaly change.
      rho_anom = C1_90*(7.0*(r5(i*5+1)+r5(i*5+5)) + 32.0*(r5(i*5+2)+r5(i*5+4)) + 12.0*r5(i*5+3))
      dpa(i,j) = G_e*dz*rho_anom
      if (present(intz_dpa)) then
        ! Use a Boole's-rule-like fifth-order accurate estimate of
        ! the double integral of the pressure anomaly.
        intz_dpa(i,j) = 0.5*G_e*dz**2 * &
                        (rho_anom - C1_90*(16.0*(r5(i*5+4)-r5(i*5+2)) + 7.0*(r5(i*5+5)-r5(i*5+1))) )
      endif
    enddo ! end loop on i
  enddo ! end loop on j

  ! 2. Compute horizontal integrals in the x direction
  if (present(intx_dpa)) then ; do j=HI%jsc,HI%jec
    do I=Isq,Ieq
      ! Corner values of T and S
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation
      ! of T,S along the top and bottom integrals, almost like thickness
      ! weighting.
      ! Note: To work in terrain following coordinates we could offset
      ! this distance by the layer thickness to replicate other models.
      hWght = massWeightToggle * &
              max(0., -bathyT(i,j)-e(i+1,j,K), -bathyT(i+1,j)-e(i,j,K))
      hWghtTop = TopWeightToggle * &
              max(0., e(i+1,j,K+1)-e(i,j,1), e(i,j,K+1)-e(i+1,j,1))
      hWght = max(hWght, hWghtTop)
      if (hWght > 0.) then
        hL = (e(i,j,K) - e(i,j,K+1)) + dz_subroundoff
        hR = (e(i+1,j,K) - e(i+1,j,K+1)) + dz_subroundoff
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1./( hWght*(hR + hL) + hL*hR )
        Ttl = ( (hWght*hR)*T_t(i+1,j,k) + (hWght*hL + hR*hL)*T_t(i,j,k) ) * iDenom
        Tbl = ( (hWght*hR)*T_b(i+1,j,k) + (hWght*hL + hR*hL)*T_b(i,j,k) ) * iDenom
        Tml = ( (hWght*hR)*tv%T(i+1,j,k)+ (hWght*hL + hR*hL)*tv%T(i,j,k) ) * iDenom
        Ttr = ( (hWght*hL)*T_t(i,j,k) + (hWght*hR + hR*hL)*T_t(i+1,j,k) ) * iDenom
        Tbr = ( (hWght*hL)*T_b(i,j,k) + (hWght*hR + hR*hL)*T_b(i+1,j,k) ) * iDenom
        Tmr = ( (hWght*hL)*tv%T(i,j,k) + (hWght*hR + hR*hL)*tv%T(i+1,j,k) ) * iDenom
        Stl = ( (hWght*hR)*S_t(i+1,j,k) + (hWght*hL + hR*hL)*S_t(i,j,k) ) * iDenom
        Sbl = ( (hWght*hR)*S_b(i+1,j,k) + (hWght*hL + hR*hL)*S_b(i,j,k) ) * iDenom
        Sml = ( (hWght*hR)*tv%S(i+1,j,k) + (hWght*hL + hR*hL)*tv%S(i,j,k) ) * iDenom
        Str = ( (hWght*hL)*S_t(i,j,k) + (hWght*hR + hR*hL)*S_t(i+1,j,k) ) * iDenom
        Sbr = ( (hWght*hL)*S_b(i,j,k) + (hWght*hR + hR*hL)*S_b(i+1,j,k) ) * iDenom
        Smr = ( (hWght*hL)*tv%S(i,j,k) + (hWght*hR + hR*hL)*tv%S(i+1,j,k) ) * iDenom
      else
        Ttl = T_t(i,j,k); Tbl = T_b(i,j,k); Ttr = T_t(i+1,j,k); Tbr = T_b(i+1,j,k)
        Tml = tv%T(i,j,k); Tmr = tv%T(i+1,j,k)
        Stl = S_t(i,j,k); Sbl = S_b(i,j,k); Str = S_t(i+1,j,k); Sbr = S_b(i+1,j,k)
        Sml = tv%S(i,j,k); Smr = tv%S(i+1,j,k)
      endif

      do m=2,4
        w_left = wt_t(m) ; w_right = wt_b(m)

        ! Salinity and temperature points are linearly interpolated in
        ! the horizontal. The subscript (1) refers to the top value in
        ! the vertical profile while subscript (5) refers to the bottom
        ! value in the vertical profile.
        T_top = (w_left*Ttl) + (w_right*Ttr)
        T_mn = (w_left*Tml) + (w_right*Tmr)
        T_bot = (w_left*Tbl) + (w_right*Tbr)

        S_top = (w_left*Stl) + (w_right*Str)
        S_mn = (w_left*Sml) + (w_right*Smr)
        S_bot = (w_left*Sbl) + (w_right*Sbr)

        ! Pressure
        dz_x(m,i) = (w_left*(e(i,j,K) - e(i,j,K+1))) + (w_right*(e(i+1,j,K) - e(i+1,j,K+1)))

        pos = i*15+(m-2)*5
        p15(pos+1) = -GxRho * ((w_left*(e(i,j,K)-z0pres(i,j))) + (w_right*(e(i+1,j,K)-z0pres(i+1,j))))
        do n=2,5
          p15(pos+n) = p15(pos+n-1) + GxRho*0.25*dz_x(m,i)
        enddo

        ! Parabolic reconstructions in the vertical for T and S
        if (use_PPM) then
          ! Coefficients of the parabolas
          s6 = 3.0 * ( 2.0*S_mn - ( S_top + S_bot ) )
          t6 = 3.0 * ( 2.0*T_mn - ( T_top + T_bot ) )
        endif
        do n=1,5
          S15(pos+n) = wt_t(n) * S_top + wt_b(n) * ( S_bot + s6 * wt_t(n) )
          T15(pos+n) = wt_t(n) * T_top + wt_b(n) * ( T_bot + t6 * wt_t(n) )
        enddo
        if (use_stanley_eos) then
          if (use_varT) T215(pos+1:pos+5) = (w_left*tv%varT(i,j,k)) + (w_right*tv%varT(i+1,j,k))
          if (use_covarTS) TS15(pos+1:pos+5) = (w_left*tv%covarTS(i,j,k)) + (w_right*tv%covarTS(i+1,j,k))
          if (use_varS) S215(pos+1:pos+5) = (w_left*tv%varS(i,j,k)) + (w_right*tv%varS(i+1,j,k))
        endif
        if (use_stanley_eos) then
          call calculate_density(T5, S5, p5, T25, TS5, S25, r5, EOS, rho_ref=rho_ref)
        else
          call calculate_density(T5, S5, p5, r5, EOS, rho_ref=rho_ref)
        endif
      enddo
    enddo

    if (use_stanley_eos) then
      call calculate_density(T15, S15, p15, T215, TS15, S215, r15, EOS, EOSdom_q15, rho_ref=rho_ref)
    else
      call calculate_density(T15, S15, p15, r15, EOS, EOSdom_q15, rho_ref=rho_ref)
    endif

    do I=Isq,Ieq
      do m=2,4
        pos = i*15+(m-2)*5
        ! Use Boole's rule to estimate the pressure anomaly change.
        intz(m) = (G_e*dz_x(m,i)*(C1_90*( 7.0*(r15(pos+1)+r15(pos+5)) + &
                                         32.0*(r15(pos+2)+r15(pos+4)) + &
                                         12.0*r15(pos+3)) ))
      enddo ! m
      intz(1) = dpa(i,j) ; intz(5) = dpa(i+1,j)

      ! Use Boole's rule to integrate the bottom pressure anomaly values in x.
      intx_dpa(I,j) = C1_90*(7.0*(intz(1)+intz(5)) + 32.0*(intz(2)+intz(4)) + 12.0*intz(3))

    enddo
  enddo ; endif

  ! 3. Compute horizontal integrals in the y direction
  if (present(inty_dpa)) then ; do J=Jsq,Jeq
    do i=HI%isc,HI%iec
      ! Corner values of T and S
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation
      ! of T,S along the top and bottom integrals, almost like thickness
      ! weighting.
      ! Note: To work in terrain following coordinates we could offset
      ! this distance by the layer thickness to replicate other models.
      hWght = massWeightToggle * &
              max(0., -bathyT(i,j)-e(i,j+1,K), -bathyT(i,j+1)-e(i,j,K))
      hWghtTop = TopWeightToggle * &
              max(0., e(i,j+1,K+1)-e(i,j,1), e(i,j,K+1)-e(i,j+1,1))
      hWght = max(hWght, hWghtTop)
      if (hWght > 0.) then
        hL = (e(i,j,K) - e(i,j,K+1)) + dz_subroundoff
        hR = (e(i,j+1,K) - e(i,j+1,K+1)) + dz_subroundoff
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1./( hWght*(hR + hL) + hL*hR )
        Ttl = ( (hWght*hR)*T_t(i,j+1,k) + (hWght*hL + hR*hL)*T_t(i,j,k) ) * iDenom
        Tbl = ( (hWght*hR)*T_b(i,j+1,k) + (hWght*hL + hR*hL)*T_b(i,j,k) ) * iDenom
        Tml = ( (hWght*hR)*tv%T(i,j+1,k)+ (hWght*hL + hR*hL)*tv%T(i,j,k) ) * iDenom
        Ttr = ( (hWght*hL)*T_t(i,j,k) + (hWght*hR + hR*hL)*T_t(i,j+1,k) ) * iDenom
        Tbr = ( (hWght*hL)*T_b(i,j,k) + (hWght*hR + hR*hL)*T_b(i,j+1,k) ) * iDenom
        Tmr = ( (hWght*hL)*tv%T(i,j,k) + (hWght*hR + hR*hL)*tv%T(i,j+1,k) ) * iDenom
        Stl = ( (hWght*hR)*S_t(i,j+1,k) + (hWght*hL + hR*hL)*S_t(i,j,k) ) * iDenom
        Sbl = ( (hWght*hR)*S_b(i,j+1,k) + (hWght*hL + hR*hL)*S_b(i,j,k) ) * iDenom
        Sml = ( (hWght*hR)*tv%S(i,j+1,k)+ (hWght*hL + hR*hL)*tv%S(i,j,k) ) * iDenom
        Str = ( (hWght*hL)*S_t(i,j,k) + (hWght*hR + hR*hL)*S_t(i,j+1,k) ) * iDenom
        Sbr = ( (hWght*hL)*S_b(i,j,k) + (hWght*hR + hR*hL)*S_b(i,j+1,k) ) * iDenom
        Smr = ( (hWght*hL)*tv%S(i,j,k) + (hWght*hR + hR*hL)*tv%S(i,j+1,k) ) * iDenom
      else
        Ttl = T_t(i,j,k); Tbl = T_b(i,j,k); Ttr = T_t(i,j+1,k); Tbr = T_b(i,j+1,k)
        Tml = tv%T(i,j,k); Tmr = tv%T(i,j+1,k)
        Stl = S_t(i,j,k); Sbl = S_b(i,j,k); Str = S_t(i,j+1,k); Sbr = S_b(i,j+1,k)
        Sml = tv%S(i,j,k); Smr = tv%S(i,j+1,k)
      endif

      do m=2,4
        w_left = wt_t(m) ; w_right = wt_b(m)

        ! Salinity and temperature points are linearly interpolated in
        ! the horizontal. The subscript (1) refers to the top value in
        ! the vertical profile while subscript (5) refers to the bottom
        ! value in the vertical profile.
        T_top = (w_left*Ttl) + (w_right*Ttr)
        T_mn = (w_left*Tml) + (w_right*Tmr)
        T_bot = (w_left*Tbl) + (w_right*Tbr)

        S_top = (w_left*Stl) + (w_right*Str)
        S_mn = (w_left*Sml) + (w_right*Smr)
        S_bot = (w_left*Sbl) + (w_right*Sbr)

        ! Pressure
        dz_y(m,i) = (w_left*(e(i,j,K) - e(i,j,K+1))) + (w_right*(e(i,j+1,K) - e(i,j+1,K+1)))

        pos = i*15+(m-2)*5
        p15(pos+1) = -GxRho * ((w_left*(e(i,j,K)-z0pres(i,j))) + (w_right*(e(i,j+1,K)-z0pres(i,j+1))))
        do n=2,5
          p15(pos+n) = p15(pos+n-1) + GxRho*0.25*dz_y(m,i)
        enddo

        ! Parabolic reconstructions in the vertical for T and S
        if (use_PPM) then
          ! Coefficients of the parabolas
          s6 = 3.0 * ( 2.0*S_mn - ( S_top + S_bot ) )
          t6 = 3.0 * ( 2.0*T_mn - ( T_top + T_bot ) )
        endif
        do n=1,5
          S15(pos+n) = wt_t(n) * S_top + wt_b(n) * ( S_bot + s6 * wt_t(n) )
          T15(pos+n) = wt_t(n) * T_top + wt_b(n) * ( T_bot + t6 * wt_t(n) )
        enddo

        if (use_stanley_eos) then
          if (use_varT) T215(pos+1:pos+5) = (w_left*tv%varT(i,j,k)) + (w_right*tv%varT(i,j+1,k))
          if (use_covarTS) TS15(pos+1:pos+5) = (w_left*tv%covarTS(i,j,k)) + (w_right*tv%covarTS(i,j+1,k))
          if (use_varS) S215(pos+1:pos+5) = (w_left*tv%varS(i,j,k)) + (w_right*tv%varS(i,j+1,k))
        endif
      enddo
    enddo

    if (use_stanley_eos) then
      call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                             T215(15*HI%isc+1:), TS15(15*HI%isc+1:), S215(15*HI%isc+1:), &
                             r15(15*HI%isc+1:), EOS, EOSdom_h15, rho_ref=rho_ref)
    else
      call calculate_density(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                             r15(15*HI%isc+1:), EOS, EOSdom_h15, rho_ref=rho_ref)
    endif

    do i=HI%isc,HI%iec
      do m=2,4
        ! Use Boole's rule to estimate the pressure anomaly change.
        pos = i*15+(m-2)*5
        intz(m) = (G_e*dz_y(m,i)*(C1_90*( 7.0*(r15(pos+1)+r15(pos+5)) + &
                                         32.0*(r15(pos+2)+r15(pos+4)) + &
                                         12.0*r15(pos+3)) ))
      enddo ! m
      intz(1) = dpa(i,j) ; intz(5) = dpa(i,j+1)

      ! Use Boole's rule to integrate the bottom pressure anomaly values in y.
      inty_dpa(i,J) = C1_90*(7.0*(intz(1)+intz(5)) + 32.0*(intz(2)+intz(4)) + 12.0*intz(3))
    enddo
  enddo ; endif

end subroutine int_density_dz_generic_ppm

!> Calls the appropriate subroutine to calculate analytical and nearly-analytical
!! integrals in pressure across layers of geopotential anomalies, which are
!! required for calculating the finite-volume form pressure accelerations in a
!! non-Boussinesq model.  There are essentially no free assumptions, apart from the
!! use of Boole's rule to do the horizontal integrals, and from a truncation in the
!! series for log(1-eps/1+eps) that assumes that |eps| < 0.34.
subroutine int_specific_vol_dp(T, S, p_t, p_b, alpha_ref, HI, EOS, US, &
                               dza, intp_dza, intx_dza, inty_dza, halo_size, &
                               bathyP, P_surf, dP_tiny, MassWghtInterp)
  type(hor_index_type), intent(in)  :: HI  !< The horizontal index structure
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T   !< Potential temperature referenced to the surface [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S   !< Salinity [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_t !< Pressure at the top of the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_b !< Pressure at the bottom of the layer [R L2 T-2 ~> Pa]
  real,                 intent(in)  :: alpha_ref !< A mean specific volume that is subtracted out
                            !! to reduce the magnitude of each of the integrals [R-1 ~> m3 kg-1]
                            !! The calculation is mathematically identical with different values of
                            !! alpha_ref, but this reduces the effects of roundoff.
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US  !< A dimensional unit scaling type
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(inout) :: dza !< The change in the geopotential anomaly across
                            !! the layer [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intp_dza !< The integral in pressure through the layer of the
                            !! geopotential anomaly relative to the anomaly at the bottom of the
                            !! layer [R L4 T-4 ~> Pa m2 s-2]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intx_dza !< The integral in x of the difference between the
                            !! geopotential anomaly at the top and bottom of the layer divided by
                            !! the x grid spacing [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJB_(HI)), &
              optional, intent(inout) :: inty_dza !< The integral in y of the difference between the
                            !! geopotential anomaly at the top and bottom of the layer divided by
                            !! the y grid spacing [L2 T-2 ~> m2 s-2]
  integer,    optional, intent(in)  :: halo_size !< The width of halo points on which to calculate dza.
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: bathyP  !< The pressure at the bathymetry [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: P_surf !< The pressure at the ocean surface [R L2 T-2 ~> Pa]
  real,       optional, intent(in)  :: dP_tiny !< A minuscule pressure change with
                            !! the same units as p_t [R L2 T-2 ~> Pa]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals

  if (EOS_quadrature(EOS)) then
    call int_spec_vol_dp_generic_pcm(T, S, p_t, p_b, alpha_ref, HI, EOS, US, &
                                     dza, intp_dza, intx_dza, inty_dza, halo_size, &
                                     bathyP, P_surf, dP_tiny, MassWghtInterp)
  else
    call analytic_int_specific_vol_dp(T, S, p_t, p_b, alpha_ref, HI, EOS, &
                                      dza, intp_dza, intx_dza, inty_dza, halo_size, &
                                      bathyP, P_surf, dP_tiny, MassWghtInterp)
  endif

end subroutine int_specific_vol_dp


!>   This subroutine calculates integrals of specific volume anomalies in
!! pressure across layers, which are required for calculating the finite-volume
!! form pressure accelerations in a non-Boussinesq model.  There are essentially
!! no free assumptions, apart from the use of Boole's rule quadrature to do the integrals.
subroutine int_spec_vol_dp_generic_pcm(T, S, p_t, p_b, alpha_ref, HI, EOS, US, dza, &
                                       intp_dza, intx_dza, inty_dza, halo_size, &
                                       bathyP, P_surf, dP_neglect, MassWghtInterp)
  type(hor_index_type), intent(in)  :: HI !< A horizontal index type structure.
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T  !< Potential temperature of the layer [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S  !< Salinity of the layer [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_t !< Pressure atop the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_b !< Pressure below the layer [R L2 T-2 ~> Pa]
  real,                 intent(in)  :: alpha_ref !< A mean specific volume that is subtracted out
                            !! to reduce the magnitude of each of the integrals [R-1 ~> m3 kg-1]
                            !! The calculation is mathematically identical with different values of
                            !! alpha_ref, but alpha_ref alters the effects of roundoff, and
                            !! answers do change.
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US !< A dimensional unit scaling type
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(inout) :: dza !< The change in the geopotential anomaly
                            !! across the layer [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intp_dza !< The integral in pressure through the layer of
                            !! the geopotential anomaly relative to the anomaly at the bottom of the
                            !! layer [R L4 T-4 ~> Pa m2 s-2]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intx_dza  !< The integral in x of the difference between
                            !! the geopotential anomaly at the top and bottom of the layer divided
                            !! by the x grid spacing [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJB_(HI)), &
              optional, intent(inout) :: inty_dza  !< The integral in y of the difference between
                            !! the geopotential anomaly at the top and bottom of the layer divided
                            !! by the y grid spacing [L2 T-2 ~> m2 s-2]
  integer,    optional, intent(in)  :: halo_size !< The width of halo points on which to calculate dza.
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: bathyP !< The pressure at the bathymetry [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: P_surf !< The pressure at the ocean surface [R L2 T-2 ~> Pa]
  real,       optional, intent(in)  :: dP_neglect !< A minuscule pressure change with
                                           !! the same units as p_t [R L2 T-2 ~> Pa]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals

!   This subroutine calculates analytical and nearly-analytical integrals in
! pressure across layers of geopotential anomalies, which are required for
! calculating the finite-volume form pressure accelerations in a non-Boussinesq
! model.  There are essentially no free assumptions, apart from the use of
! Boole's rule to do the horizontal integrals, and from a truncation in the
! series for log(1-eps/1+eps) that assumes that |eps| < 0.34.

  ! Local variables
  real :: T5((5*HI%isd+1):(5*(HI%ied+2)))  ! Temperatures along a line of subgrid locations [C ~> degC]
  real :: S5((5*HI%isd+1):(5*(HI%ied+2)))  ! Salinities along a line of subgrid locations [S ~> ppt]
  real :: p5((5*HI%isd+1):(5*(HI%ied+2)))  ! Pressures along a line of subgrid locations [R L2 T-2 ~> Pa]
  real :: a5((5*HI%isd+1):(5*(HI%ied+2)))  ! Specific volumes anomalies along a line of subgrid
                                           ! locations [R-1 ~> m3 kg-3]
  real :: T15((15*HI%isd+1):(15*(HI%ied+1))) ! Temperatures at an array of subgrid locations [C ~> degC]
  real :: S15((15*HI%isd+1):(15*(HI%ied+1))) ! Salinities at an array of subgrid locations [S ~> ppt]
  real :: p15((15*HI%isd+1):(15*(HI%ied+1))) ! Pressures at an array of subgrid locations [R L2 T-2 ~> Pa]
  real :: a15((15*HI%isd+1):(15*(HI%ied+1))) ! Specific volumes at an array of subgrid locations [R ~> kg m-3]
  real :: alpha_anom ! The depth averaged specific density anomaly [R-1 ~> m3 kg-1]
  real :: dp         ! The pressure change through a layer [R L2 T-2 ~> Pa]
  real :: dp_x(5,SZIB_(HI)) ! The pressure change through a layer along an x-line of subgrid locations [Z ~> m]
  real :: dp_y(5,SZI_(HI))  ! The pressure change through a layer along a y-line of subgrid locations [Z ~> m]
  real :: hWght      ! A pressure-thickness below topography [R L2 T-2 ~> Pa]
  real :: hL, hR     ! Pressure-thicknesses of the columns to the left and right [R L2 T-2 ~> Pa]
  real :: iDenom     ! The inverse of the denominator in the weights [T4 R-2 L-4 ~> Pa-2]
  real :: hWt_LL, hWt_LR ! hWt_LA is the weighted influence of A on the left column [nondim]
  real :: hWt_RL, hWt_RR ! hWt_RA is the weighted influence of A on the right column [nondim]
  real :: wt_L, wt_R ! The linear weights of the left and right columns [nondim]
  real :: wtT_L, wtT_R ! The weights for tracers from the left and right columns [nondim]
  real :: intp(5)    ! The integrals of specific volume with pressure at the
                     ! 5 sub-column locations [L2 T-2 ~> m2 s-2]
  logical :: do_massWeight ! Indicates whether to do mass weighting.
  logical :: top_massWeight ! Indicates whether to do mass weighting the sea surface
  logical :: massWeight_bug ! If true, use an incorrect expression to determine where to apply mass weighting
  real, parameter :: C1_90 = 1.0/90.0  ! A rational constant [nondim]
  integer, dimension(2) :: EOSdom_h5  ! The 5-point h-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_q15 ! The 3x5-point q-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_h15 ! The 3x5-point h-point i-computational domain for the equation of state
  integer :: Isq, Ieq, Jsq, Jeq, ish, ieh, jsh, jeh, i, j, m, n, pos, halo

  Isq = HI%IscB ; Ieq = HI%IecB ; Jsq = HI%JscB ; Jeq = HI%JecB
  halo = 0 ; if (present(halo_size)) halo = MAX(halo_size,0)
  ish = HI%isc-halo ; ieh = HI%iec+halo ; jsh = HI%jsc-halo ; jeh = HI%jec+halo
  if (present(intx_dza)) then ; ish = MIN(Isq,ish) ; ieh = MAX(Ieq+1,ieh); endif
  if (present(inty_dza)) then ; jsh = MIN(Jsq,jsh) ; jeh = MAX(Jeq+1,jeh); endif

  do_massWeight = .false. ; massWeight_bug = .false. ; top_massWeight = .false.
  if (present(MassWghtInterp)) then
    do_massWeight = BTEST(MassWghtInterp, 0) ! True for odd values
    top_massWeight = BTEST(MassWghtInterp, 1) ! True if the 2 bit is set
    massWeight_bug = BTEST(MassWghtInterp, 3) ! True if the 8 bit is set
    if (do_massWeight .and. .not.present(bathyP)) call MOM_error(FATAL, &
        "int_spec_vol_dp_generic_pcm: bathyP must be present if near-bottom mass weighting is in use.")
    if (top_massWeight .and. .not.present(P_surf)) call MOM_error(FATAL, &
        "int_spec_vol_dp_generic_pcm: P_surf must be present if near-surface mass weighting is in use.")
    if ((do_massWeight .or. top_massWeight) .and. .not.present(dP_neglect)) call MOM_error(FATAL, &
        "int_spec_vol_dp_generic_pcm: dP_neglect must be present if mass weighting is in use.")
  endif

  ! Set the loop ranges for equation of state calculations at various points.
  EOSdom_h5(1) = 1 ; EOSdom_h5(2) = 5*(ieh-ish+1)
  EOSdom_q15(1) = 1 ; EOSdom_q15(2) = 15*(Ieq-Isq+1)
  EOSdom_h15(1) = 1 ; EOSdom_h15(2) = 15*(HI%iec-HI%isc+1)

  do j=jsh,jeh
    do i=ish,ieh
      dp = p_b(i,j) - p_t(i,j)
      pos = 5*i
      do n=1,5
        T5(pos+n) = T(i,j) ; S5(pos+n) = S(i,j)
        p5(pos+n) = p_b(i,j) - 0.25*real(n-1)*dp
      enddo
    enddo

    call calculate_spec_vol(T5(5*ish+1:), S5(5*ish+1:), p5(5*ish+1:), a5(5*ish+1:), EOS, &
                            EOSdom_h5, spv_ref=alpha_ref)

    do i=ish,ieh
      dp = p_b(i,j) - p_t(i,j)
      ! Use Boole's rule to estimate the interface height anomaly change.
      pos = 5*i
      alpha_anom = C1_90*(7.0*(a5(pos+1)+a5(pos+5)) + 32.0*(a5(pos+2)+a5(pos+4)) + 12.0*a5(pos+3))
      dza(i,j) = dp*alpha_anom
      ! Use a Boole's-rule-like fifth-order accurate estimate of the double integral of
      ! the interface height anomaly.
      if (present(intp_dza)) intp_dza(i,j) = 0.5*dp**2 * &
            (alpha_anom - C1_90*(16.0*(a5(pos+4)-a5(pos+2)) + 7.0*(a5(pos+5)-a5(pos+1))) )
    enddo
  enddo

  if (present(intx_dza)) then ; do j=HI%jsc,HI%jec
    do I=Isq,Ieq
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation of
      ! T & S along the top and bottom integrals, akin to thickness weighting.
      hWght = 0.0
      if (do_massWeight .and. massWeight_bug) then
        hWght = max(0., bathyP(i,j)-p_t(i+1,j), bathyP(i+1,j)-p_t(i,j))
      elseif (do_massWeight) then
        hWght = max(0., p_t(i+1,j)-bathyP(i,j), p_t(i,j)-bathyP(i+1,j))
      endif
      if (top_massWeight) &
        hWght = max(hWght, P_surf(i,j)-p_b(i+1,j), P_surf(i+1,j)-p_b(i,j))
      if (hWght > 0.) then
        hL = (p_b(i,j) - p_t(i,j)) + dP_neglect
        hR = (p_b(i+1,j) - p_t(i+1,j)) + dP_neglect
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
        hWt_LL = (hWght*hL + hR*hL) * iDenom ; hWt_LR = (hWght*hR) * iDenom
        hWt_RR = (hWght*hR + hR*hL) * iDenom ; hWt_RL = (hWght*hL) * iDenom
      else
        hWt_LL = 1.0 ; hWt_LR = 0.0 ; hWt_RR = 1.0 ; hWt_RL = 0.0
      endif

      do m=2,4
        wt_L = 0.25*real(5-m) ; wt_R = 1.0-wt_L
        wtT_L = (wt_L*hWt_LL) + (wt_R*hWt_RL) ; wtT_R = (wt_L*hWt_LR) + (wt_R*hWt_RR)
        pos = i*15+(m-2)*5

        ! T, S, and p are interpolated in the horizontal.  The p interpolation
        ! is linear, but for T and S it may be thickness weighted.
        p15(pos+1) = (wt_L*p_b(i,j)) + (wt_R*p_b(i+1,j))
        dp_x(m,I) = (wt_L*(p_b(i,j) - p_t(i,j))) + (wt_R*(p_b(i+1,j) - p_t(i+1,j)))
        T15(pos+1) = (wtT_L*T(i,j)) + (wtT_R*T(i+1,j))
        S15(pos+1) = (wtT_L*S(i,j)) + (wtT_R*S(i+1,j))

        do n=2,5
          T15(pos+n) = T15(pos+1) ; S15(pos+n) = S15(pos+1)
          p15(pos+n) = p15(pos+n-1) - 0.25*dp_x(m,I)
        enddo
      enddo
    enddo

    call calculate_spec_vol(T15(15*Isq+1:), S15(15*Isq+1:), p15(15*Isq+1:), &
                            a15(15*Isq+1:), EOS, EOSdom_q15, spv_ref=alpha_ref)

    do I=Isq,Ieq
      intp(1) = dza(i,j) ; intp(5) = dza(i+1,j)
      ! Use Boole's rule to estimate the interface height anomaly change.
      do m=2,4
        pos = i*15+(m-2)*5
        intp(m) = (dp_x(m,I)*( C1_90*(7.0*(a15(pos+1)+a15(pos+5)) + 32.0*(a15(pos+2)+a15(pos+4)) + &
                                  12.0*a15(pos+3)) ))
      enddo
      ! Use Boole's rule to integrate the interface height anomaly values in x.
      intx_dza(i,j) = C1_90*(7.0*(intp(1)+intp(5)) + 32.0*(intp(2)+intp(4)) + &
                             12.0*intp(3))
    enddo
  enddo ; endif

  if (present(inty_dza)) then ; do J=Jsq,Jeq
    do i=HI%isc,HI%iec
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation of
      ! T & S along the top and bottom integrals, akin to thickness weighting.
      hWght = 0.0
      if (do_massWeight .and. massWeight_bug) then
        hWght = max(0., bathyP(i,j)-p_t(i,j+1), bathyP(i,j+1)-p_t(i,j))
      elseif (do_massWeight) then
        hWght = max(0., p_t(i,j+1)-bathyP(i,j), p_t(i,j)-bathyP(i,j+1))
      endif
      if (top_massWeight) &
        hWght = max(hWght, P_surf(i,j)-p_b(i,j+1), P_surf(i,j+1)-p_b(i,j))
      if (hWght > 0.) then
        hL = (p_b(i,j) - p_t(i,j)) + dP_neglect
        hR = (p_b(i,j+1) - p_t(i,j+1)) + dP_neglect
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
        hWt_LL = (hWght*hL + hR*hL) * iDenom ; hWt_LR = (hWght*hR) * iDenom
        hWt_RR = (hWght*hR + hR*hL) * iDenom ; hWt_RL = (hWght*hL) * iDenom
      else
        hWt_LL = 1.0 ; hWt_LR = 0.0 ; hWt_RR = 1.0 ; hWt_RL = 0.0
      endif

      do m=2,4
        wt_L = 0.25*real(5-m) ; wt_R = 1.0-wt_L
        wtT_L = (wt_L*hWt_LL) + (wt_R*hWt_RL) ; wtT_R = (wt_L*hWt_LR) + (wt_R*hWt_RR)
        pos = i*15+(m-2)*5

        ! T, S, and p are interpolated in the horizontal.  The p interpolation
        ! is linear, but for T and S it may be thickness weighted.
        p15(pos+1) = (wt_L*p_b(i,j)) + (wt_R*p_b(i,j+1))
        dp_y(m,i) = (wt_L*(p_b(i,j) - p_t(i,j))) + (wt_R*(p_b(i,j+1) - p_t(i,j+1)))
        T15(pos+1) = (wtT_L*T(i,j)) + (wtT_R*T(i,j+1))
        S15(pos+1) = (wtT_L*S(i,j)) + (wtT_R*S(i,j+1))
        do n=2,5
          T15(pos+n) = T15(pos+1) ; S15(pos+n) = S15(pos+1)
          p15(pos+n) = p15(pos+n-1) - 0.25*dp_y(m,i)
        enddo
      enddo
    enddo

    call calculate_spec_vol(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                            a15(15*HI%isc+1:), EOS, EOSdom_h15, spv_ref=alpha_ref)

    do i=HI%isc,HI%iec

      intp(1) = dza(i,j) ; intp(5) = dza(i,j+1)
      ! Use Boole's rule to estimate the interface height anomaly change.
      do m=2,4
        pos = i*15+(m-2)*5
        intp(m) = (dp_y(m,i)*( C1_90*(7.0*(a15(pos+1)+a15(pos+5)) + 32.0*(a15(pos+2)+a15(pos+4)) + &
                                     12.0*a15(pos+3)) ))
      enddo
      ! Use Boole's rule to integrate the interface height anomaly values in y.
      inty_dza(i,j) = C1_90*(7.0*(intp(1)+intp(5)) + 32.0*(intp(2)+intp(4)) + &
                             12.0*intp(3))
    enddo
  enddo ; endif

end subroutine int_spec_vol_dp_generic_pcm

!>   This subroutine calculates integrals of specific volume anomalies in
!! pressure across layers, which are required for calculating the finite-volume
!! form pressure accelerations in a non-Boussinesq model.  There are essentially
!! no free assumptions, apart from the use of Boole's rule quadrature to do the integrals.
subroutine int_spec_vol_dp_generic_plm(T_t, T_b, S_t, S_b, p_t, p_b, alpha_ref, &
                             dP_neglect, bathyP, HI, EOS, US, dza, &
                             intp_dza, intx_dza, inty_dza, P_surf, MassWghtInterp)
  type(hor_index_type), intent(in)  :: HI !< A horizontal index type structure.
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T_t  !< Potential temperature at the top of the layer [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T_b  !< Potential temperature at the bottom of the layer [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S_t  !< Salinity at the top the layer [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S_b  !< Salinity at the bottom the layer [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_t !< Pressure atop the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_b !< Pressure below the layer [R L2 T-2 ~> Pa]
  real,                 intent(in)  :: alpha_ref !< A mean specific volume that is subtracted out
                            !! to reduce the magnitude of each of the integrals [R-1 ~> m3 kg-1]
                            !! The calculation is mathematically identical with different values of
                            !! alpha_ref, but alpha_ref alters the effects of roundoff, and
                            !! answers do change.
  real,                 intent(in)  :: dP_neglect !<!< A miniscule pressure change with
                                             !! the same units as p_t [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: bathyP !< The pressure at the bathymetry [R L2 T-2 ~> Pa]
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in) :: US !< A dimensional unit scaling type
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(inout) :: dza !< The change in the geopotential anomaly
                            !! across the layer [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intp_dza !< The integral in pressure through the layer of
                            !! the geopotential anomaly relative to the anomaly at the bottom of the
                            !! layer [R L4 T-4 ~> Pa m2 s-2]
  real, dimension(SZIB_(HI),SZJ_(HI)), &
              optional, intent(inout) :: intx_dza  !< The integral in x of the difference between
                            !! the geopotential anomaly at the top and bottom of the layer divided
                            !! by the x grid spacing [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJB_(HI)), &
              optional, intent(inout) :: inty_dza  !< The integral in y of the difference between
                            !! the geopotential anomaly at the top and bottom of the layer divided
                            !! by the y grid spacing [L2 T-2 ~> m2 s-2]
  real, dimension(SZI_(HI),SZJ_(HI)), &
              optional, intent(in)  :: P_surf !< The pressure at the ocean surface [R L2 T-2 ~> Pa]
  integer,    optional, intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                            !! mass weighting to interpolate T/S in integrals

!   This subroutine calculates analytical and nearly-analytical integrals in
! pressure across layers of geopotential anomalies, which are required for
! calculating the finite-volume form pressure accelerations in a non-Boussinesq
! model.  There are essentially no free assumptions, apart from the use of
! Boole's rule to do the horizontal integrals, and from a truncation in the
! series for log(1-eps/1+eps) that assumes that |eps| < 0.34.

  real :: T5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Temperatures along a line of subgrid locations [C ~> degC]
  real :: S5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Salinities along a line of subgrid locations [S ~> ppt]
  real :: p5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Pressures along a line of subgrid locations [R L2 T-2 ~> Pa]
  real :: a5((5*HI%iscB+1):(5*(HI%iecB+2)))  ! Specific volumes anomalies along a line of subgrid
                                             ! locations [R-1 ~> m3 kg-3]
  real :: T15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Temperatures at an array of subgrid locations [C ~> degC]
  real :: S15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Salinities at an array of subgrid locations [S ~> ppt]
  real :: p15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Pressures at an array of subgrid locations [R L2 T-2 ~> Pa]
  real :: a15((15*HI%iscB+1):(15*(HI%iecB+1))) ! Specific volumes at an array of subgrid locations [R ~> kg m-3]
  real :: wt_t(5), wt_b(5) ! Weights of top and bottom values at quadrature points [nondim]
  real :: T_top, T_bot ! Horizontally interpolated temperature at the cell top and bottom [C ~> degC]
  real :: S_top, S_bot ! Horizontally interpolated salinity at the cell top and bottom [S ~> ppt]
  real :: P_top, P_bot ! Horizontally interpolated pressure at the cell top and bottom [R L2 T-2 ~> Pa]

  real :: alpha_anom ! The depth averaged specific density anomaly [R-1 ~> m3 kg-1]
  real :: dp         ! The pressure change through a layer [R L2 T-2 ~> Pa]
  real :: dp_90(2:4,SZIB_(HI)) ! The pressure change through a layer divided by 90 [R L2 T-2 ~> Pa]
  real :: hWght      ! A pressure-thickness below topography [R L2 T-2 ~> Pa]
  real :: hL, hR     ! Pressure-thicknesses of the columns to the left and right [R L2 T-2 ~> Pa]
  real :: iDenom     ! The inverse of the denominator in the weights [T4 R-2 L-4 ~> Pa-2]
  real :: hWt_LL, hWt_LR ! hWt_LA is the weighted influence of A on the left column [nondim]
  real :: hWt_RL, hWt_RR ! hWt_RA is the weighted influence of A on the right column [nondim]
  real :: wt_L, wt_R ! The linear weights of the left and right columns [nondim]
  real :: wtT_L, wtT_R ! The weights for tracers from the left and right columns [nondim]
  real :: intp(5)    ! The integrals of specific volume with pressure at the
                     ! 5 sub-column locations [L2 T-2 ~> m2 s-2]
  real, parameter :: C1_90 = 1.0/90.0  ! A rational constant [nondim]
  logical :: do_massWeight ! Indicates whether to do mass weighting.
  logical :: top_massWeight ! Indicates whether to do mass weighting the sea surface
  logical :: massWeight_bug ! If true, use an incorrect expression to determine where to apply mass weighting
  integer, dimension(2) :: EOSdom_h5  ! The 5-point h-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_q15 ! The 3x5-point q-point i-computational domain for the equation of state
  integer, dimension(2) :: EOSdom_h15 ! The 3x5-point h-point i-computational domain for the equation of state
  integer :: Isq, Ieq, Jsq, Jeq, i, j, m, n, pos

  Isq = HI%IscB ; Ieq = HI%IecB ; Jsq = HI%JscB ; Jeq = HI%JecB

  do_massWeight = .false. ; massWeight_bug = .false. ; top_massWeight = .false.
  if (present(MassWghtInterp)) then
    do_massWeight = BTEST(MassWghtInterp, 0) ! True for odd values
    top_massWeight = BTEST(MassWghtInterp, 1) ! True if the 2 bit is set
    massWeight_bug = BTEST(MassWghtInterp, 3) ! True if the 8 bit is set
    if (top_massWeight .and. .not.present(P_surf)) call MOM_error(FATAL, &
        "int_spec_vol_dp_generic_plm: P_surf must be present if near-surface mass weighting is in use.")
  endif

  do n = 1, 5 ! Note that these are reversed from int_density_dz.
    wt_t(n) = 0.25 * real(n-1)
    wt_b(n) = 1.0 - wt_t(n)
  enddo

  ! Set the loop ranges for equation of state calculations at various points.
  EOSdom_h5(1) = 1 ; EOSdom_h5(2) = 5*(Ieq-Isq+2)
  EOSdom_q15(1) = 1 ; EOSdom_q15(2) = 15*(Ieq-Isq+1)
  EOSdom_h15(1) = 1 ; EOSdom_h15(2) = 15*(HI%iec-HI%isc+1)

  ! 1. Compute vertical integrals
  do j=Jsq,Jeq+1
    do i=Isq,Ieq+1
      do n=1,5 ! T, S and p are linearly interpolated in the vertical.
        p5(i*5+n) = wt_t(n) * p_t(i,j) + wt_b(n) * p_b(i,j)
        S5(i*5+n) = wt_t(n) * S_t(i,j) + wt_b(n) * S_b(i,j)
        T5(i*5+n) = wt_t(n) * T_t(i,j) + wt_b(n) * T_b(i,j)
      enddo
    enddo
    call calculate_spec_vol(T5, S5, p5, a5, EOS, EOSdom_h5, spv_ref=alpha_ref)
    do i=Isq,Ieq+1
      ! Use Boole's rule to estimate the interface height anomaly change.
      dp = p_b(i,j) - p_t(i,j)
      alpha_anom = C1_90*((7.0*(a5(i*5+1)+a5(i*5+5)) + 32.0*(a5(i*5+2)+a5(i*5+4))) + 12.0*a5(i*5+3))
      dza(i,j) = dp*alpha_anom
      ! Use a Boole's-rule-like fifth-order accurate estimate of the double integral of
      ! the interface height anomaly.
      if (present(intp_dza)) intp_dza(i,j) = 0.5*dp**2 * &
            (alpha_anom - C1_90*(16.0*(a5(i*5+4)-a5(i*5+2)) + 7.0*(a5(i*5+5)-a5(i*5+1))) )
    enddo
  enddo

  ! 2. Compute horizontal integrals in the x direction
  if (present(intx_dza)) then ; do j=HI%jsc,HI%jec
    do I=Isq,Ieq
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation
      ! of T,S along the top and bottom integrals, almost like thickness
      ! weighting. Note: To work in terrain following coordinates we could
      ! offset this distance by the layer thickness to replicate other models.
      hWght = 0.0
      if (do_massWeight .and. massWeight_bug) then
        hWght = max(0., bathyP(i,j)-p_t(i+1,j), bathyP(i+1,j)-p_t(i,j))
      elseif (do_massWeight) then
        hWght = max(0., p_t(i+1,j)-bathyP(i,j), p_t(i,j)-bathyP(i+1,j))
      endif
      if (top_massWeight) &
        hWght = max(hWght, P_surf(i,j)-p_b(i+1,j), P_surf(i+1,j)-p_b(i,j))
      if (hWght > 0.) then
        hL = (p_b(i,j) - p_t(i,j)) + dP_neglect
        hR = (p_b(i+1,j) - p_t(i+1,j)) + dP_neglect
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
        hWt_LL = (hWght*hL + hR*hL) * iDenom ; hWt_LR = (hWght*hR) * iDenom
        hWt_RR = (hWght*hR + hR*hL) * iDenom ; hWt_RL = (hWght*hL) * iDenom
      else
        hWt_LL = 1.0 ; hWt_LR = 0.0 ; hWt_RR = 1.0 ; hWt_RL = 0.0
      endif

      do m=2,4
        wt_L = 0.25*real(5-m) ; wt_R = 1.0-wt_L
        wtT_L = (wt_L*hWt_LL) + (wt_R*hWt_RL) ; wtT_R = (wt_L*hWt_LR) + (wt_R*hWt_RR)

        ! T, S, and p are interpolated in the horizontal.  The p interpolation
        ! is linear, but for T and S it may be thickness weighted.
        P_top = (wt_L*p_t(i,j)) + (wt_R*p_t(i+1,j))
        P_bot = (wt_L*p_b(i,j)) + (wt_R*p_b(i+1,j))
        T_top = (wtT_L*T_t(i,j)) + (wtT_R*T_t(i+1,j))
        T_bot = (wtT_L*T_b(i,j)) + (wtT_R*T_b(i+1,j))
        S_top = (wtT_L*S_t(i,j)) + (wtT_R*S_t(i+1,j))
        S_bot = (wtT_L*S_b(i,j)) + (wtT_R*S_b(i+1,j))
        dp_90(m,I) = C1_90*(P_bot - P_top)

        ! Salinity, temperature and pressure with linear interpolation in the vertical.
        pos = i*15+(m-2)*5
        do n=1,5
          p15(pos+n) = wt_t(n) * P_top + wt_b(n) * P_bot
          S15(pos+n) = wt_t(n) * S_top + wt_b(n) * S_bot
          T15(pos+n) = wt_t(n) * T_top + wt_b(n) * T_bot
        enddo
      enddo
    enddo

    call calculate_spec_vol(T15, S15, p15, a15, EOS, EOSdom_q15, spv_ref=alpha_ref)

    do I=Isq,Ieq
      intp(1) = dza(i,j) ; intp(5) = dza(i+1,j)
      do m=2,4
        ! Use Boole's rule to estimate the interface height anomaly change.
        ! The integrals at the ends of the segment are already known.
        pos = I*15+(m-2)*5
        intp(m) = (dp_90(m,I)*((7.0*(a15(pos+1)+a15(pos+5)) + &
                                32.0*(a15(pos+2)+a15(pos+4))) + 12.0*a15(pos+3) ))
      enddo
      ! Use Boole's rule to integrate the interface height anomaly values in x.
      intx_dza(I,j) = C1_90*((7.0*(intp(1)+intp(5)) + 32.0*(intp(2)+intp(4))) + &
                             12.0*intp(3))
    enddo
  enddo ; endif

  ! 3. Compute horizontal integrals in the y direction
  if (present(inty_dza)) then ; do J=Jsq,Jeq
    do i=HI%isc,HI%iec
      ! hWght is the distance measure by which the cell is violation of
      ! hydrostatic consistency. For large hWght we bias the interpolation
      ! of T,S along the top and bottom integrals, like thickness weighting.
      hWght = 0.0
      if (do_massWeight .and. massWeight_bug) then
        hWght = max(0., bathyP(i,j)-p_t(i,j+1), bathyP(i,j+1)-p_t(i,j))
      elseif (do_massWeight) then
        hWght = max(0., p_t(i,j+1)-bathyP(i,j), p_t(i,j)-bathyP(i,j+1))
      endif
      if (top_massWeight) &
        hWght = max(hWght, P_surf(i,j)-p_b(i,j+1), P_surf(i,j+1)-p_b(i,j))
      if (hWght > 0.) then
        hL = (p_b(i,j) - p_t(i,j)) + dP_neglect
        hR = (p_b(i,j+1) - p_t(i,j+1)) + dP_neglect
        hWght = hWght * ( (hL-hR)/(hL+hR) )**2
        iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
        hWt_LL = (hWght*hL + hR*hL) * iDenom ; hWt_LR = (hWght*hR) * iDenom
        hWt_RR = (hWght*hR + hR*hL) * iDenom ; hWt_RL = (hWght*hL) * iDenom
      else
        hWt_LL = 1.0 ; hWt_LR = 0.0 ; hWt_RR = 1.0 ; hWt_RL = 0.0
      endif

      do m=2,4
        wt_L = 0.25*real(5-m) ; wt_R = 1.0-wt_L
        wtT_L = (wt_L*hWt_LL) + (wt_R*hWt_RL) ; wtT_R = (wt_L*hWt_LR) + (wt_R*hWt_RR)

        ! T, S, and p are interpolated in the horizontal.  The p interpolation
        ! is linear, but for T and S it may be thickness weighted.
        P_top = (wt_L*p_t(i,j)) + (wt_R*p_t(i,j+1))
        P_bot = (wt_L*p_b(i,j)) + (wt_R*p_b(i,j+1))
        T_top = (wtT_L*T_t(i,j)) + (wtT_R*T_t(i,j+1))
        T_bot = (wtT_L*T_b(i,j)) + (wtT_R*T_b(i,j+1))
        S_top = (wtT_L*S_t(i,j)) + (wtT_R*S_t(i,j+1))
        S_bot = (wtT_L*S_b(i,j)) + (wtT_R*S_b(i,j+1))
        dp_90(m,i) = C1_90*(P_bot - P_top)

        ! Salinity, temperature and pressure with linear interpolation in the vertical.
        pos = i*15+(m-2)*5
        do n=1,5
          p15(pos+n) = wt_t(n) * P_top + wt_b(n) * P_bot
          S15(pos+n) = wt_t(n) * S_top + wt_b(n) * S_bot
          T15(pos+n) = wt_t(n) * T_top + wt_b(n) * T_bot
        enddo
      enddo
    enddo

    call calculate_spec_vol(T15(15*HI%isc+1:), S15(15*HI%isc+1:), p15(15*HI%isc+1:), &
                            a15(15*HI%isc+1:), EOS, EOSdom_h15, spv_ref=alpha_ref)

    do i=HI%isc,HI%iec
      intp(1) = dza(i,j) ; intp(5) = dza(i,j+1)
      do m=2,4
        ! Use Boole's rule to estimate the interface height anomaly change.
        ! The integrals at the ends of the segment are already known.
        pos = i*15+(m-2)*5
        intp(m) = (dp_90(m,i) * ((7.0*(a15(pos+1)+a15(pos+5)) + &
                                  32.0*(a15(pos+2)+a15(pos+4))) + 12.0*a15(pos+3)))
      enddo
      ! Use Boole's rule to integrate the interface height anomaly values in x.
      inty_dza(i,J) = C1_90*((7.0*(intp(1)+intp(5)) + 32.0*(intp(2)+intp(4))) + &
                             12.0*intp(3))
    enddo
  enddo ; endif

end subroutine int_spec_vol_dp_generic_plm


!> Diagnose the fractional mass weighting in a layer that might be used with a Boussinesq calculation.
subroutine diagnose_mass_weight_Z(z_t, z_b, bathyT, SSH, dz_neglect, MassWghtInterp, HI, &
                                  MassWt_u, MassWt_v)
  type(hor_index_type), intent(in)  :: HI !< A horizontal index type structure.
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: z_t !< Height at the top of the layer in depth units [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: z_b !< Height at the bottom of the layer [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: bathyT !< The depth of the bathymetry [Z ~> m]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: SSH !< The sea surface height [Z ~> m]
  real,                 intent(in)  :: dz_neglect !< A minuscule thickness change [Z ~> m]
  integer,              intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals
  real, dimension(SZIB_(HI),SZJ_(HI)), &
                        intent(inout) :: MassWt_u  !< The fractional mass weighting at u-points [nondim]
  real, dimension(SZI_(HI),SZJB_(HI)), &
                        intent(inout) :: MassWt_v  !< The fractional mass weighting at v-points [nondim]

  ! Local variables
  real :: hWght      ! A pressure-thickness below topography [Z ~> m]
  real :: hL, hR     ! Pressure-thicknesses of the columns to the left and right [Z ~> m]
  real :: iDenom     ! The inverse of the denominator in the weights [Z-2 ~> m-2]
  logical :: do_massWeight  ! Indicates whether to do mass weighting near bathymetry
  logical :: top_massWeight ! Indicates whether to do mass weighting the sea surface
  integer :: Isq, Ieq, Jsq, Jeq, i, j

  Isq = HI%IscB ; Ieq = HI%IecB
  Jsq = HI%JscB ; Jeq = HI%JecB

  do_massWeight = BTEST(MassWghtInterp, 0) ! True for odd values
  top_massWeight = BTEST(MassWghtInterp, 1) ! True if the 2 bit is set

  ! Calculate MassWt_u
  do j=HI%jsc,HI%jec ; do I=Isq,Ieq
    ! hWght is the distance measure by which the cell is violation of
    ! hydrostatic consistency. For large hWght we bias the interpolation
    ! of T,S along the top and bottom integrals, like thickness weighting.
    hWght = 0.0
    if (do_massWeight) &
      hWght = max(0., -bathyT(i,j)-z_t(i+1,j), -bathyT(i+1,j)-z_t(i,j))
    if (top_massWeight) &
      hWght = max(hWght, z_b(i+1,j)-SSH(i,j), z_b(i,j)-SSH(i+1,j))
    if (hWght > 0.) then
      hL = (z_t(i,j) - z_b(i,j)) + dz_neglect
      hR = (z_t(i+1,j) - z_b(i+1,j)) + dz_neglect
      hWght = hWght * ( (hL-hR)/(hL+hR) )**2
      iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
      MassWt_u(I,j) = (hWght*hR + hWght*hL) * iDenom
    else
      MassWt_u(I,j) = 0.0
    endif
  enddo ; enddo

  ! Calculate MassWt_v
  do J=Jsq,Jeq ; do i=HI%isc,HI%iec
    ! hWght is the distance measure by which the cell is violation of
    ! hydrostatic consistency. For large hWght we bias the interpolation
    ! of T,S along the top and bottom integrals, like thickness weighting.
    hWght = 0.0
    if (do_massWeight) &
      hWght = max(0., -bathyT(i,j)-z_t(i,j+1), -bathyT(i,j+1)-z_t(i,j))
    if (top_massWeight) &
      hWght = max(hWght, z_b(i,j+1)-SSH(i,j), z_b(i,j)-SSH(i,j+1))
    if (hWght > 0.) then
      hL = (z_t(i,j) - z_b(i,j)) + dz_neglect
      hR = (z_t(i,j+1) - z_b(i,j+1)) + dz_neglect
      hWght = hWght * ( (hL-hR)/(hL+hR) )**2
      iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
      MassWt_v(i,J) = (hWght*hR + hWght*hL) * iDenom
    else
      MassWt_v(i,J) = 0.0
    endif
  enddo ; enddo

end subroutine diagnose_mass_weight_Z


!> Diagnose the fractional mass weighting in a layer that might be used with a non-Boussinesq calculation.
subroutine diagnose_mass_weight_p(p_t, p_b, bathyP, P_surf, dP_neglect, MassWghtInterp, HI, &
                                  MassWt_u, MassWt_v)
  type(hor_index_type), intent(in)  :: HI !< A horizontal index type structure.
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_t !< Pressure atop the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_b !< Pressure below the layer [R L2 T-2 ~> Pa]
  real,                 intent(in)  :: dP_neglect !<!< A miniscule pressure change with
                                           !! the same units as p_t [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: bathyP !< The pressure at the bathymetry [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: P_surf !< The pressure at the ocean surface [R L2 T-2 ~> Pa]
  integer,              intent(in)  :: MassWghtInterp !< A flag indicating whether and how to use
                                           !! mass weighting to interpolate T/S in integrals
  real, dimension(SZIB_(HI),SZJ_(HI)), &
                        intent(inout) :: MassWt_u  !< The fractional mass weighting at u-points [nondim]
  real, dimension(SZI_(HI),SZJB_(HI)), &
                        intent(inout) :: MassWt_v  !< The fractional mass weighting at v-points [nondim]

  ! Local variables
  real :: hWght      ! A pressure-thickness below topography [R L2 T-2 ~> Pa]
  real :: hL, hR     ! Pressure-thicknesses of the columns to the left and right [R L2 T-2 ~> Pa]
  real :: iDenom     ! The inverse of the denominator in the weights [T4 R-2 L-4 ~> Pa-2]
  logical :: do_massWeight ! Indicates whether to do mass weighting.
  logical :: top_massWeight ! Indicates whether to do mass weighting the sea surface
  logical :: massWeight_bug ! If true, use an incorrect expression to determine where to apply mass weighting
  integer :: Isq, Ieq, Jsq, Jeq, i, j

  Isq = HI%IscB ; Ieq = HI%IecB
  Jsq = HI%JscB ; Jeq = HI%JecB

  do_massWeight = BTEST(MassWghtInterp, 0) ! True for odd values
  top_massWeight = BTEST(MassWghtInterp, 1) ! True if the 2 bit is set
  massWeight_bug = BTEST(MassWghtInterp, 3) ! True if the 8 bit is set

  ! Calculate MassWt_u
  do j=HI%jsc,HI%jec ; do I=Isq,Ieq
    ! hWght is the distance measure by which the cell is violation of
    ! hydrostatic consistency. For large hWght we bias the interpolation
    ! of T,S along the top and bottom integrals, like thickness weighting.
    hWght = 0.0
    if (do_massWeight .and. massWeight_bug) then
      hWght = max(0., bathyP(i,j)-p_t(i+1,j), bathyP(i+1,j)-p_t(i,j))
    elseif (do_massWeight) then
      hWght = max(0., p_t(i+1,j)-bathyP(i,j), p_t(i,j)-bathyP(i+1,j))
    endif
    if (top_massWeight) &
      hWght = max(hWght, P_surf(i,j)-p_b(i+1,j), P_surf(i+1,j)-p_b(i,j))
    if (hWght > 0.) then
      hL = (p_b(i,j) - p_t(i,j)) + dP_neglect
      hR = (p_b(i+1,j) - p_t(i+1,j)) + dP_neglect
      hWght = hWght * ( (hL-hR)/(hL+hR) )**2
      iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
      MassWt_u(I,j) = (hWght*hR + hWght*hL) * iDenom
    else
      MassWt_u(I,j) = 0.0
    endif
  enddo ; enddo

  ! Calculate MassWt_v
  do J=Jsq,Jeq ; do i=HI%isc,HI%iec
    ! hWght is the distance measure by which the cell is violation of
    ! hydrostatic consistency. For large hWght we bias the interpolation
    ! of T,S along the top and bottom integrals, like thickness weighting.
    hWght = 0.0
    if (do_massWeight .and. massWeight_bug) then
      hWght = max(0., bathyP(i,j)-p_t(i,j+1), bathyP(i,j+1)-p_t(i,j))
    elseif (do_massWeight) then
      hWght = max(0., p_t(i,j+1)-bathyP(i,j), p_t(i,j)-bathyP(i,j+1))
    endif
    if (top_massWeight) &
      hWght = max(hWght, P_surf(i,j)-p_b(i,j+1), P_surf(i,j+1)-p_b(i,j))
    if (hWght > 0.) then
      hL = (p_b(i,j) - p_t(i,j)) + dP_neglect
      hR = (p_b(i,j+1) - p_t(i,j+1)) + dP_neglect
      hWght = hWght * ( (hL-hR)/(hL+hR) )**2
      iDenom = 1.0 / ( hWght*(hR + hL) + hL*hR )
      MassWt_v(i,J) = (hWght*hR + hWght*hL) * iDenom
    else
      MassWt_v(i,J) = 0.0
    endif
  enddo ; enddo

end subroutine diagnose_mass_weight_p

!> Find the depth at which the reconstructed pressure matches P_tgt
subroutine find_depth_of_pressure_in_cell(T_t, T_b, S_t, S_b, z_t, z_b, P_t, P_tgt, &
                       rho_ref, G_e, EOS, US, P_b, z_out, z_tol)
  real,                  intent(in)  :: T_t !< Potential temperature at the cell top [C ~> degC]
  real,                  intent(in)  :: T_b !< Potential temperature at the cell bottom [C ~> degC]
  real,                  intent(in)  :: S_t !< Salinity at the cell top [S ~> ppt]
  real,                  intent(in)  :: S_b !< Salinity at the cell bottom [S ~> ppt]
  real,                  intent(in)  :: z_t !< Absolute height of top of cell [Z ~> m]   (Boussinesq ????)
  real,                  intent(in)  :: z_b !< Absolute height of bottom of cell [Z ~> m]
  real,                  intent(in)  :: P_t !< Anomalous pressure of top of cell, relative
                                            !! to g*rho_ref*z_t [R L2 T-2 ~> Pa]
  real,                  intent(in)  :: P_tgt !< Target pressure at height z_out, relative
                                            !! to g*rho_ref*z_out [R L2 T-2 ~> Pa]
  real,                  intent(in)  :: rho_ref !< Reference density with which calculation
                                            !! are anomalous to [R ~> kg m-3]
  real,                  intent(in)  :: G_e !< Gravitational acceleration [L2 Z-1 T-2 ~> m s-2]
  type(EOS_type),        intent(in)  :: EOS !< Equation of state structure
  type(unit_scale_type), intent(in)  :: US  !< A dimensional unit scaling type
  real,                  intent(out) :: P_b !< Pressure at the bottom of the cell [R L2 T-2 ~> Pa]
  real,                  intent(out) :: z_out !< Absolute depth at which anomalous pressure = p_tgt [Z ~> m]
  real,                  intent(in)  :: z_tol !< The tolerance in finding z_out [Z ~> m]

  ! Local variables
  real :: dp    ! Pressure thickness of the layer [R L2 T-2 ~> Pa]
  real :: F_guess, F_l, F_r  ! Fractional positions [nondim]
  real :: GxRho ! The product of the gravitational acceleration and reference density [R L2 Z-1 T-2 ~> Pa m-1]
  real :: Pa, Pa_left, Pa_right, Pa_tol ! Pressure anomalies, P = integral of g*(rho-rho_ref) dz [R L2 T-2 ~> Pa]
  integer :: m  ! A counter for how many iterations have been done in the while loop
  character(len=240) :: msg

  GxRho = G_e * rho_ref

  ! Anomalous pressure difference across whole cell
  dp = frac_dp_at_pos(T_t, T_b, S_t, S_b, z_t, z_b, rho_ref, G_e, 1.0, EOS)

  P_b = P_t + dp ! Anomalous pressure at bottom of cell

  if (P_tgt <= P_t ) then
    z_out = z_t
    return
  endif

  if (P_tgt >= P_b) then
    z_out = z_b
    return
  endif

  F_l = 0.
  Pa_left = P_t - P_tgt ! Pa_left < 0
  F_r = 1.
  Pa_right = P_b - P_tgt ! Pa_right > 0
  Pa_tol = GxRho * z_tol

  F_guess = F_l - Pa_left / (Pa_right - Pa_left) * (F_r - F_l)
  Pa = Pa_right - Pa_left ! To get into iterative loop
  m = 0 ! Reset the counter for the loop to be zero
  do while ( abs(Pa) > Pa_tol )

    m = m + 1
    if (m > 30) then ! Call an error, because convergence to the tolerance has not been achieved
     write(msg,*) Pa_left,Pa,Pa_right,P_t-P_tgt,P_b-P_tgt
     call MOM_error(FATAL, 'find_depth_of_pressure_in_cell completes too many iterations: '//msg)
    endif
    z_out = z_t + ( z_b - z_t ) * F_guess
    Pa = frac_dp_at_pos(T_t, T_b, S_t, S_b, z_t, z_b, rho_ref, G_e, F_guess, EOS) - ( P_tgt - P_t )

    if (Pa<Pa_left) then
      write(msg,*) Pa_left,Pa,Pa_right,P_t-P_tgt,P_b-P_tgt
      call MOM_error(FATAL, 'find_depth_of_pressure_in_cell out of bounds negative: /n'//msg)
    elseif (Pa<0.) then
      Pa_left = Pa
      F_l = F_guess
    elseif (Pa>Pa_right) then
      write(msg,*) Pa_left,Pa,Pa_right,P_t-P_tgt,P_b-P_tgt
      call MOM_error(FATAL, 'find_depth_of_pressure_in_cell out of bounds positive: /n'//msg)
    elseif (Pa>0.) then
      Pa_right = Pa
      F_r = F_guess
    else ! Pa == 0
      return
    endif
    F_guess = F_l - Pa_left / (Pa_right - Pa_left) * (F_r - F_l)

  enddo

end subroutine find_depth_of_pressure_in_cell

!> Calculate the average in situ specific volume across layers
subroutine avg_specific_vol(T, S, p_t, dp, HI, EOS, SpV_avg, halo_size)
  type(hor_index_type), intent(in)  :: HI  !< The horizontal index structure
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: T   !< Potential temperature of the layer [C ~> degC]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: S   !< Salinity of the layer [S ~> ppt]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: p_t !< Pressure at the top of the layer [R L2 T-2 ~> Pa]
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(in)  :: dp  !< Pressure change in the layer [R L2 T-2 ~> Pa]
  type(EOS_type),       intent(in)  :: EOS !< Equation of state structure
  real, dimension(SZI_(HI),SZJ_(HI)), &
                        intent(inout) :: SpV_avg !< The vertical average specific volume
                                           !! in the layer [R-1 ~> m3 kg-1]
  integer,    optional, intent(in)  :: halo_size !< The number of halo points in which to work.

  ! Local variables
  integer, dimension(2) :: EOSdom ! The i-computational domain for the equation of state
  integer :: jsh, jeh, j, halo

  halo = 0 ; if (present(halo_size)) halo = MAX(halo_size,0)
  jsh = HI%jsc-halo ; jeh = HI%jec+halo

  EOSdom(:) = EOS_domain(HI, halo_size)
  do j=jsh,jeh
    call average_specific_vol(T(:,j), S(:,j), p_t(:,j), dp(:,j), SpV_avg(:,j), EOS, EOSdom)
  enddo

end subroutine avg_specific_vol

!> Returns change in anomalous pressure change from top to non-dimensional
!! position pos between z_t and z_b [R L2 T-2 ~> Pa]
real function frac_dp_at_pos(T_t, T_b, S_t, S_b, z_t, z_b, rho_ref, G_e, pos, EOS)
  real,           intent(in)  :: T_t !< Potential temperature at the cell top [C ~> degC]
  real,           intent(in)  :: T_b !< Potential temperature at the cell bottom [C ~> degC]
  real,           intent(in)  :: S_t !< Salinity at the cell top [S ~> ppt]
  real,           intent(in)  :: S_b !< Salinity at the cell bottom [S ~> ppt]
  real,           intent(in)  :: z_t !< The geometric height at the top of the layer [Z ~> m]
  real,           intent(in)  :: z_b !< The geometric height at the bottom of the layer [Z ~> m]
  real,           intent(in)  :: rho_ref !< A mean density [R ~> kg m-3], that is subtracted out to
                                     !! reduce the magnitude of each of the integrals.
  real,           intent(in)  :: G_e !< The Earth's gravitational acceleration [L2 Z-1 T-2 ~> m s-2]
  real,           intent(in)  :: pos !< The fractional vertical position, 0 to 1 [nondim]
  type(EOS_type), intent(in)  :: EOS !< Equation of state structure

  ! Local variables
  real, parameter :: C1_90 = 1.0/90.0  ! A rational constant [nondim]
  real :: dz                 ! Distance from the layer top [Z ~> m]
  real :: top_weight, bottom_weight ! Fractional weights at quadrature points [nondim]
  real :: rho_ave            ! Average density [R ~> kg m-3]
  real, dimension(5) :: T5   ! Temperatures at quadrature points [C ~> degC]
  real, dimension(5) :: S5   ! Salinities at quadrature points [S ~> ppt]
  real, dimension(5) :: p5   ! Pressures at quadrature points [R L2 T-2 ~> Pa]
  real, dimension(5) :: rho5 ! Densities at quadrature points [R ~> kg m-3]
  integer :: n

  do n=1,5
    ! Evaluate density at five quadrature points
    bottom_weight = 0.25*real(n-1) * pos
    top_weight = 1.0 - bottom_weight
    ! Salinity and temperature points are linearly interpolated
    S5(n) = top_weight * S_t + bottom_weight * S_b
    T5(n) = top_weight * T_t + bottom_weight * T_b
    p5(n) = ( top_weight * z_t + bottom_weight * z_b ) * ( G_e * rho_ref )
  enddo
  call calculate_density(T5, S5, p5, rho5, EOS)
  rho5(:) = rho5(:) !- rho_ref ! Work with anomalies relative to rho_ref

  ! Use Boole's rule to estimate the average density
  rho_ave = C1_90*(7.0*(rho5(1)+rho5(5)) + 32.0*(rho5(2)+rho5(4)) + 12.0*rho5(3))

  dz = ( z_t - z_b ) * pos
  frac_dp_at_pos = G_e * dz * rho_ave
end function frac_dp_at_pos

end module MOM_density_integrals

!> \namespace mom_density_integrals
!!
