import numpy as np
import onnxruntime as ort

# --- Configuration ---
MODEL_PATH = "model/noise_model.onnx"

# IMPORTANT: You must use the SAME stats calculated during training
# Look at your training logs for the specific numbers, or load them from a config file.
# These are placeholder values based on typical sensor data ranges.
DATA_MIN = np.array([-3.51, 3.0, 1.0, 1.0], dtype=np.float32) # [Noise, Cars, Motos, People]
DATA_MAX = np.array([55.43, 601.0, 8.0, 858.0], dtype=np.float32)

# The index of the target column (Noise Level) to denormalize the output
TARGET_IDX = 0 
TRAIN_WINDOW = 30  # The model expects 30 past time steps
NUM_FEATURES = 4   # Noise, Cars, Motos, People

def normalize(data, data_min, data_max):
    """Scales data to 0-1 range used by the model."""
    return (data - data_min) / (data_max - data_min + 1e-8)

def denormalize_value(value, target_min, target_max):
    """Converts 0-1 model output back to real decibels."""
    return value * (target_max - target_min) + target_min

def main():
    # 1. Load the ONNX Model
    print(f"Loading model from {MODEL_PATH}...")
    session = ort.InferenceSession(MODEL_PATH)
    
    # Get input/output names (standard practice for ONNX)
    input_name = session.get_inputs()[0].name
    output_name = session.get_outputs()[0].name

    # 2. Prepare Dummy Input Data
    # In a real app, this would be your last 30 readings from the sensors.
    # Shape: (Batch_Size=1, Sequence_Length=30, Features=4)
    print("Generating dummy sensor input (30 time steps)...")
    
    # Let's simulate varying noise and traffic
    raw_input = np.random.uniform(
        low=[40, 0, 0, 0], 
        high=[80, 5, 2, 10], 
        size=(1, TRAIN_WINDOW, NUM_FEATURES)
    ).astype(np.float32)

    # 3. Preprocessing (Normalization)
    # The model expects values between 0 and 1
    input_tensor = normalize(raw_input, DATA_MIN, DATA_MAX).astype(np.float32)

    # 4. Run Inference
    print("Running prediction...")
    # The input dictionary maps the model's input name to your data
    result = session.run([output_name], {input_name: input_tensor})
    
    # The result is a list containing the output arrays
    predicted_normalized = result[0][0][0] # Extract the single scalar value

    # 5. Post-processing (Denormalization)
    prediction_db = denormalize_value(
        predicted_normalized, 
        target_min=DATA_MIN[TARGET_IDX], 
        target_max=DATA_MAX[TARGET_IDX]
    )

    print("-" * 30)
    print(f"Predicted Noise Level: {prediction_db:.2f} dB")
    print("-" * 30)

if __name__ == "__main__":
    main()
