"""FERC 1 specific transformation parameters.

These constants are used to construct :mod:`pydantic` models, which are validated and
used to control the various data transformations. The definitions of those models can be
found in :mod:`pudl.transform.classes` and :mod:`pudl.transform.ferc1`
"""

from datetime import date

import numpy as np
import pandas as pd

##############################################################################
# Unit converstion parameters
##############################################################################
PERPOUND_TO_PERSHORTTON = {
    "multiplier": 2000.0,
    "from_unit": r"_per_lb",
    "to_unit": r"_per_ton",
}
"""Parameters for converting from inverse pounds to inverse short tons."""

CENTS_TO_DOLLARS = {
    "multiplier": 0.01,
    "from_unit": r"_cents",
    "to_unit": r"_usd",
}
"""Parameters for converting from cents to dollars."""

CENTS_PERMMBTU_TO_USD_PERMMBTU = {
    "multiplier": 0.01,
}
"""Parameters for converting from cents per mmbtu to dollars per mmbtu."""


PERCF_TO_PERMCF = {
    "multiplier": 1000.0,
    "from_unit": r"_per_cf",
    "to_unit": r"_per_mcf",
}
"""Parameters for converting from inverse cubic feet to inverse 1000s of cubic feet."""

PERGALLON_TO_PERBARREL = {
    "multiplier": 42.0,
    "from_unit": r"_per_gal",
    "to_unit": r"_per_bbl",
}
"""Parameters for converting from inverse gallons to inverse barrels."""

PERKW_TO_PERMW = {
    "multiplier": 1000.0,
    "from_unit": r"_per_kw",
    "to_unit": r"_per_mw",
}
"""Parameters for converting column units from per kW to per MW."""

PERKWH_TO_PERMWH = {
    "multiplier": 1000.0,
    "from_unit": r"_per_kwh",
    "to_unit": r"_per_mwh",
}
"""Parameters for converting column units from per kWh to per MWh."""

KW_TO_MW = {
    "multiplier": 1e-3,
    "from_unit": r"_kw",
    "to_unit": r"_mw",
}
"""Parameters for converting column units from kW to MW."""

KWH_TO_MWH = {
    "multiplier": 1e-3,
    "from_unit": r"_kwh",
    "to_unit": r"_mwh",
}
"""Parameters for converting column units from kWh to MWh."""

BTU_TO_MMBTU = {
    "multiplier": 1e-6,
    "from_unit": r"_btu",
    "to_unit": r"_mmbtu",
}
"""Parameters for converting column units from BTU to MMBTU."""

PERBTU_TO_PERMMBTU = {
    "multiplier": 1e6,
    "from_unit": r"_per_btu",
    "to_unit": r"_per_mmbtu",
}
"""Parameters for converting column units from BTU to MMBTU."""

BTU_PERKWH_TO_MMBTU_PERMWH = {
    "multiplier": (1e-6 * 1000.0),
    "from_unit": r"_btu_per_kwh",
    "to_unit": r"_mmbtu_per_mwh",
}
"""Parameters for converting column units from BTU/kWh to MMBTU/MWh."""


##############################################################################
# Valid ranges to impose on various columns
##############################################################################
VALID_PLANT_YEARS = {
    "lower_bound": 1850,
    "upper_bound": date.today().year,
}
"""Valid range of years for power plant construction."""

VALID_COAL_MMBTU_PER_TON = {
    "lower_bound": 6.5,
    "upper_bound": 29.0,
}
"""Valid range for coal heat content, taken from the EIA-923 instructions.

Lower bound is for waste coal. Upper bound is for bituminous coal.
https://www.eia.gov/survey/form/eia_923/instructions.pdf
"""

VALID_COAL_USD_PER_MMBTU = {
    "lower_bound": 0.5,
    "upper_bound": 7.5,
}
"""Historical coal price range from the EIA-923 Fuel Receipts and Costs table."""

VALID_GAS_MMBTU_PER_MCF = {
    "lower_bound": 0.3,
    "upper_bound": 3.3,
}
"""Valid range for gaseous fuel heat content, taken from the EIA-923 instructions.

Lower bound is for landfill gas. Upper bound is for "other gas".  Blast furnace gas
(which has very low heat content) is effectively excluded.
https://www.eia.gov/survey/form/eia_923/instructions.pdf
"""

VALID_GAS_USD_PER_MMBTU = {
    "lower_bound": 1.0,
    "upper_bound": 35.0,
}
"""Historical natural gas price range from the EIA-923 Fuel Receipts and Costs table."""

VALID_OIL_MMBTU_PER_BBL = {
    "lower_bound": 3.0,
    "upper_bound": 6.9,
}
"""Valid range for petroleum fuels heat content, taken from the EIA-923 instructions.

Lower bound is for waste oil. Upper bound is for residual fuel oil.
https://www.eia.gov/survey/form/eia_923/instructions.pdf
"""

VALID_OIL_USD_PER_MMBTU = {
    "lower_bound": 5.0,
    "upper_bound": 33.0,
}
"""Historical petroleum price range from the EIA-923 Fuel Receipts and Costs table."""

##############################################################################
# Unit Corrections
##############################################################################
COAL_COST_PER_MMBTU_CORRECTIONS = {
    "data_col": "fuel_cost_per_mmbtu",
    "cat_col": "fuel_type_code_pudl",
    "cat_val": "coal",
    "valid_range": VALID_COAL_USD_PER_MMBTU,
    "unit_conversions": [
        CENTS_TO_DOLLARS,
    ],
}
GAS_COST_PER_MMBTU_CORRECTIONS = {
    "data_col": "fuel_cost_per_mmbtu",
    "cat_col": "fuel_type_code_pudl",
    "cat_val": "gas",
    "valid_range": VALID_GAS_USD_PER_MMBTU,
    "unit_conversions": [
        CENTS_TO_DOLLARS,
    ],
}
OIL_COST_PER_MMBTU_CORRECTIONS = {
    "data_col": "fuel_cost_per_mmbtu",
    "cat_col": "fuel_type_code_pudl",
    "cat_val": "oil",
    "valid_range": VALID_OIL_USD_PER_MMBTU,
    "unit_conversions": [
        CENTS_TO_DOLLARS,
    ],
}
FUEL_COST_PER_MMBTU_CORRECTIONS = [
    COAL_COST_PER_MMBTU_CORRECTIONS,
    GAS_COST_PER_MMBTU_CORRECTIONS,
    OIL_COST_PER_MMBTU_CORRECTIONS,
]

COAL_MMBTU_PER_UNIT_CORRECTIONS = {
    "data_col": "fuel_mmbtu_per_unit",
    "cat_col": "fuel_type_code_pudl",
    "cat_val": "coal",
    "valid_range": VALID_COAL_MMBTU_PER_TON,
    "unit_conversions": [
        PERPOUND_TO_PERSHORTTON,
        BTU_TO_MMBTU,
    ],
}
GAS_MMBTU_PER_UNIT_CORRECTIONS = {
    "data_col": "fuel_mmbtu_per_unit",
    "cat_col": "fuel_type_code_pudl",
    "cat_val": "gas",
    "valid_range": VALID_GAS_MMBTU_PER_MCF,
    "unit_conversions": [
        PERCF_TO_PERMCF,
        BTU_TO_MMBTU,
    ],
}
OIL_MMBTU_PER_UNIT_CORRECTIONS = {
    "data_col": "fuel_mmbtu_per_unit",
    "cat_col": "fuel_type_code_pudl",
    "cat_val": "oil",
    "valid_range": VALID_OIL_MMBTU_PER_BBL,
    "unit_conversions": [
        PERGALLON_TO_PERBARREL,
        BTU_TO_MMBTU,  # Why was this omitted in the old corrections?
    ],
}
FUEL_MMBTU_PER_UNIT_CORRECTIONS = [
    COAL_MMBTU_PER_UNIT_CORRECTIONS,
    GAS_MMBTU_PER_UNIT_CORRECTIONS,
    OIL_MMBTU_PER_UNIT_CORRECTIONS,
]

##############################################################################
# String normalizations
##############################################################################
FERC1_STRING_NORM = {
    "remove_chars": r"$?={}\x02\x00",
    "nullable": False,
}

##############################################################################
# Common invalid plant names
##############################################################################
INVALID_PLANT_NAMES = [pd.NA, np.nan, "not applicable", "0", "", "-"]
"""Invalid plant names which appear in multiple plant tables."""

##############################################################################
# String categorizations
##############################################################################
PLANT_FUNCTION_CATEGORIES = {
    "categories": {
        "intangible": {
            "intangible",
            "ferc:IntangiblePlantMember",
        },
        "steam_production": {
            "steam_production",
            "ferc:SteamProductionPlantMember",
        },
        "nuclear_production": {
            "nuclear_production",
            "ferc:NuclearProductionPlantMember",
        },
        "hydraulic_production_conventional": {
            "hydraulic_production_conventional",
            "ferc:HydraulicProductionPlantConventionalMember",
        },
        "hydraulic_production_pumped_storage": {
            "hydraulic_production_pumped_storage",
            "ferc:HydraulicProductionPlantPumpedStorageMember",
        },
        "other_production": {
            "other_production",
            "ferc:OtherProductionPlantMember",
        },
        "transmission": {
            "transmission",
            "ferc:TransmissionPlantMember",
        },
        "distribution": {
            "distribution",
            "ferc:DistributionPlantMember",
        },
        "regional_transmission_and_market_operation": {
            "regional_transmission_and_market_operation",
            "ferc:RegionalTransmissionAndMarketOperationMember",
        },
        "general": {
            "general",
            "ferc:GeneralPlantMember",
        },
        "common_plant_electric": {
            "common_plant_electric",
            "ferc:CommonPlantElectricMember",
        },
        "total": {"total"},
    }
}

UTILITY_TYPE_CATEGORIES = {
    "categories": {
        "electric": {"electric", "ferc:ElectricUtilityMember"},
        "gas": {"gas", "ferc:GasUtilityMember"},
        "common": {"common", "ferc:CommonUtilityMember"},
        "other": {"other", "ferc:OtherUtilityMember"},
        "total": {"total"},
    }
}

PLANT_STATUS = {
    "categories": {
        "in_service": {
            "in_service",
            "ferc:ElectricPlantInServiceMember",
        },
        "future": {
            "future",
            "ferc:ElectricPlantHeldForFutureUseMember",
        },
        "leased": {
            "leased",
            "ferc:ElectricPlantLeasedToOthersMember",
        },
        "total": {
            "total",
        },
    }
}

FUEL_CATEGORIES: dict[str, set[str]] = {
    "categories": {
        "coal": {
            "bit coal",
            "bit-coal",
            "ciak",
            "coa",
            "coal",
            "coal & gas",
            "coal & oil",
            "coal (sb)",
            "coal - subbit",
            "coal and gas",
            "coal bit",
            "coal mcf",
            "coal tons",
            "coal unit #3",
            "coal&coke",
            "coal(sb)",
            "coal-lig",
            "coal-lignite",
            "coal-sub",
            "coal-sub bit",
            "coal-subb",
            "coal-subbit",
            "coal-subbitum",
            "coal-u3",
            "coal-unit #3",
            "coal. gas",
            "coal.oil",
            "coal/gas",
            "coal/oil",
            "coals",
            "coke",
            "fossil",
            "lignite",
            "lignite/coal",
            "other-steam expenses:",
            "pet. coke",
            "petcoke",
            "steam plants",
            "steam-common production",
            "steam-common production plant",
            "tons",
            "steam:",
            "steam heating plant",
            "steam plant",
        },
        "oil": {
            "# 2  oil",
            "# 2 oil",
            "# 6 oil",
            "#1 diesel",
            "#1 uls diesel",
            "#1 fuel oil",
            "#1 kerosene",
            "#1 ls diesel",
            "#1 oil",
            "#2",
            "#2 diesel",
            "#2 fue oil",
            "#2 fuel",
            "#2 fuel oil",
            "#2 ls diesel",
            "#2 uls diesel",
            "#2 oil",
            "#2 oil-diesel",
            "#2oil diesel",
            "#6",
            "#6 & used",
            "#6 hvy oil",
            "#6 oil",
            "2",
            "2 oil",
            "6",
            "barrels",
            "bbl",
            "bbl.",
            "biofuel",
            "black start diesel units",
            "black start diesel units:",
            "blend oil",
            "blended oil",
            "bunker oil",
            "comp oil/gas",
            "dielel oil",
            "dielsel oil",
            "diesal",
            "diesel",
            "diesel fuel",
            "diesel generators",
            "diesel oil",
            "diesel plant",
            "diesel plants",
            "diesel plants:",
            "diesel:",
            "diesel-low",
            "diesel-low s",
            "diesel-low s or gas",
            "diesel-ultra low s",
            "diesel/",
            "diesel/compo",
            "diesel/compos",
            "diesle-low s",
            "fuel oil",
            "fo",
            "heavy oil",
            "jet",
            "jet fuel",
            "jet oil",
            "kero",
            "light oil",
            "montauk diesel",
            "nno. 2 oil",
            "no 2",
            "no. 2 diesel",
            "no 2 oil",
            "no 2. oil",
            "no 6 oil",
            "no. 2",
            "no. 2  oil",
            "no. 2 oil",
            "no. 2 oils",
            "no. 6 oil",
            "no.2 oil",
            "no.2 oil gas",
            "no.6 oil",
            "no.6& used",
            "oial",
            "oil",
            "oil #6",
            "oil (#2)",
            "oil (#6)",
            "oil (6)",
            "oil (used)",
            "oil - 2",
            "oil / gas",
            "oil 6",
            "oil bbl",
            "oil bbls",
            "oil bls",
            "oil un 1 & 2",
            "oil(#2)",
            "oil-2",
            "oil-6",
            "oil-8",
            "oil-diesel",
            "oil-fuel",
            "oil-gas",
            "oil-kero",
            "oil-u1&2",
            "oil-unit #1",
            "oil-unit1&2",
            "oil.",
            "oil, nat .gas",
            "oil, nat. gas",
            "oil,n.gas",
            "oil,nat.gas",
            "oil/diesel",
            "oil/gas",
            "oill",
            "oils",
            "oiul",
            "oli",
            "pil",
            "resid. oil",
            "residual oil",
            "tall oil",
            "used oil",
        },
        "gas": {
            "# 6 gas",
            "#6 gas",
            "* propane",
            "blast gaa",
            "blast gas",
            "coal.gas",
            "coke gas",
            "coke oven gas",
            "dinner lake gas",
            "g",
            "ga",
            "gac",
            "gas",
            "gas #2016",
            "gas & oil",
            "gas - oil",
            "gas / oil",
            "gas ct",
            "gas expander",
            "gas mcf",
            "gas plant",
            "gas turbine",
            "gas turbine plant",
            "gas turbine plants",
            "gas turbine:",
            "gas turbines",
            "gas turbines:",
            "gas, fuel oil",
            "gas, oil",
            "gas-oil",
            "gas.oil",
            "gas/oil",
            "gas/fuel oil",
            "gas`",
            "gass",
            "lp gas",
            "mcf",
            "methane",
            "methane gas",
            "n. gas",
            "nat  gas",
            "nat gas",
            "nat. gas",
            "nat.gas",
            "natl gas",
            "natuaral gas",
            "natural",
            "natural  gas",
            "natural gas",
            "natural gas/fuel oil",
            "ng",
            "ng, fo",
            "prop",
            "prop.",
            "propane",
            "propane **",
            "propane**",
            "syngas",
            "syngass",
            "waste heat/gas",
        },
        "solar": {
            "community solar gardens",
            "photovoltaic",
            "photo voltaic generating plants:",
            "proj dvlpmnt div solar photovoltaic project:",
            "sixth street solar",
            "solar",
            "solar array",
            "solar electric",
            "solar photovoltaic",
            "solar photovoltiac",
            "solar plant",
            "solar plants",
            "solar plants:",
            "solar:",
            "other production - solar",
            "customer hosted solar:",
            "solar farms",
        },
        "wind": {
            "rosiere wind turbines",
            "wind",
            "wind turbine",
            "wind turbine plant",
            "wind turbine plants",
            "wind turbine plants:",
            "wind turbine:",
            "wind turbines",
            "wind:",
        },
        "hydro": {
            "additional hydro plant",
            "conventional hydro",
            "hydraulic",
            "hydraulic (1):",
            "hydraulic:",
            "hydro",
            "hyrdo",
            "hydro -",
            "hydro : licensed proj. no.",
            "hydro electric",
            "hydro lic project no.",
            "hydro lic. project no.",
            "hydro license",
            "hydro license no.",
            "hydro plant:",
            "hydro plants",
            "hydro plants:",
            "hydro plants: licensed proj no.",
            "hydro plants: licensed proj. no.",
            "hydro-electric",
            "hydro-neal shoals",
            "hydro:",
            "hydro: lic project no.",
            "hydro: lic. project no.",
            "hydro: license no.",
            "hydroelectric",
            "hydroelectric : licensed proj. no.",
            "hydroelectric generating plants",
            "hydroelectric generating plants :",
            "hydroelectric generating plants:",
            "hydroelectric: licensed proj. no.",
            "other hydro assets (resvr., strm gages, etc.)",
            "other hydro assets (resvr.,strm gages, etc.)",
            "other hydro costs:",
            "research and development costs (hydros)",
            "run of river",
            "run-of-river",
            "run-of-rr",
            "run-of-rv",
            "small hydroelectric plants:",
            "small hydroelectric generating plants:",
            "unallocated hydros",
            "unallocated - hydros",
            "water",
            "lewiston canal facilities:",
            "hydro plants hydro plants",
            "neal shoals-hydro license",
        },
        "nuclear": {
            "nuclear",
            "grams of uran",
            "grams of",
            "grams of  ura",
            "grams",
            "nucleur",
            "nulear",
            "nucl",
            "nucleart",
            "nucelar",
            "gr.uranium",
            "grams of urm",
            "nuclear (9)",
            "nulcear",
            "nuc",
            "gr. uranium",
            "uranium",
            "nuclear mw da",
            "grams of ura",
            "nucvlear",
            "nuclear (1)",
        },
        "waste": {
            "biomass",
            "landfill",
            "landfill gas",
            "landfill:",
            "rdf",
            "rdf, gas",
            "refuse",
            "switchgrass",
            "tdf",
            "tire",
            "tire chips",
            "tire refuse",
            "tires",
            "tires/refuse",
            "waste",
            "waste oil",
            "wood",
            "wood chips",
            "wood waste",
            "woodchips",
            "woodships",
        },
        # This should really be NA but we require a fuel_type_code_pudl
        "other": {
            "",
            "# 6 & used",
            "*",
            "/#2a",
            "0",
            "1",
            "120976",
            "20",
            "3",
            "77509",
            "a",
            "a, b",
            "a,b",
            "all",
            "all fuel",
            "all fuels",
            "all fuels =",
            "all fuels are",
            "all fuels to",
            "alt. fuel",
            "alt. fuels",
            "alternative",
            "at right",
            "average",
            "avg",
            "avg.",
            "bio fuel",
            "blo",
            "blo bls",
            "bng",
            "c",
            "c omposite",
            "comb",
            "comb.",
            "combined",
            "comp",
            "compos.",
            "composit",
            "composite",
            "compsite",
            "fuel cell:",
            "fuel cell",
            "furfural",
            "kagv",
            "lime",
            "m",
            "mbtu's",
            "mbtus",
            "mw days",
            "n.a.",
            "n/a",
            "na",
            "none",
            "other",
            "purch steam",
            "purch. steam",
            "rrm",
            "steam",
            "the right are",
            "toal",
            "total",
            "total pc",
            "total pr crk",
            "total prairie",
            "waste heat",
            "watson bng",
            "—",
            "-",
            "--",
            "---------",
            '"',
            "storage/pipelines",
            "hydrogen",
        },
        "na_category": {
            "na_category",
            "internal combustion",
            "internal combustion:",
            "internal combustion plants:",
            "gold creek internal combustion:",
        },
    }
}
"""A mapping a canonical fuel name to a set of strings which are used to represent that
fuel in the FERC Form 1 Reporting.

Case is ignored, as all fuel strings are converted to lower case in the data set.
"""

FUEL_UNIT_CATEGORIES: dict[str, set[str]] = {
    "categories": {
        "ton": {
            "ton",
            "toms",
            "taons",
            "tones",
            "col-tons",
            "toncoaleq",
            "coal",
            "tons coal eq",
            "coal-tons",
            "tons",
            "tons coal",
            "coal-ton",
            "tires-tons",
            "coal tons -2 ",
            "oil-tons",
            "coal tons 200",
            "ton-2000",
            "coal tons",
            "coal tons -2",
            "coal-tone",
            "tire-ton",
            "tire-tons",
            "ton coal eqv",
            "tos",
            "coal tons - 2",
            "c. t.",
            "c.t.",
            "t",
            "toncoalequiv",
            "coal - ton",
            "toons",
        },
        "mcf": {
            "mcf",
            "mcf's",
            "mcfs",
            "mcf.",
            "mcfe",
            "gas mcf",
            '"gas" mcf',
            "gas-mcf",
            "mfc",
            "mct",
            " mcf",
            "msfs",
            "mlf",
            "mscf",
            "mci",
            "mcl",
            "mcg",
            "m.cu.ft.",
            "kcf",
            "(mcf)",
            "mcf *(4)",
            "mcf00",
            "m.cu.ft..",
            "1000 c.f",
            "gas - mcf",
        },
        "bbl": {
            "bbl",
            "barrel",
            "bbls",
            "barrels",
            "bbrl",
            "bbl.",
            "bbls.",
            "oil 42 gal",
            "oil-barrels",
            "barrrels",
            "bbl-42 gal",
            "oil-barrel",
            "bb.",
            "barrells",
            "bar",
            "bbld",
            "oil- barrel",
            "barrels    .",
            "bbl .",
            "barels",
            "barrell",
            "berrels",
            "bb",
            "bbl.s",
            "oil-bbl",
            "bls",
            "bbl:",
            "barrles",
            "blb",
            "propane-bbl",
            "barriel",
            "berriel",
            "barrile",
            "(bbl.)",
            "barrel *(4)",
            "(4) barrel",
            "bbf",
            "blb.",
            "(bbl)",
            "bb1",
            "bbsl",
            "barrrel",
            "barrels 100%",
            "bsrrels",
            "bbl's",
            "*barrels",
            "oil - barrels",
            "oil 42 gal ba",
            "bll",
            "boiler barrel",
            "gas barrel",
            '"boiler" barr',
            '"gas" barrel',
            '"boiler"barre',
            '"boiler barre',
            "barrels .",
            "bariel",
            "brrels",
            "oil barrel",
            "barreks",
            "oil-bbls",
            "oil-bbs",
            "boe",
            "oil- bbl",
        },
        "mmbbl": {
            "mmbbl",
            "mmbbls",
            "mmboe",
        },
        "gal": {"gal", "gallons", "gal.", "gals", "gals.", "gallon", "galllons"},
        "kgal": {
            "kgal",
            "oil(1000 gal)",
            "oil(1000)",
            "oil (1000)",
            "oil(1000",
            "oil(1000ga)",
            "1000 gals",
            "1000 gal",
        },
        "grams": {
            "gram",
            "grams",
            "gm u",
            "grams u235",
            "grams u-235",
            "grams of uran",
            "grams: u-235",
            "grams:u-235",
            "grams:u235",
            "grams u308",
            "grams: u235",
            "grams of",
            "grams - n/a",
            "gms uran",
            "s e uo2 grams",
            "gms uranium",
            "grams of urm",
            "gms. of uran",
            "grams (100%)",
            "grams v-235",
            "se uo2 grams",
            "grams u",
            "g",
            "grams of uranium",
        },
        "kg": {
            "kg",
            "kg of uranium",
            "kg uranium",
            "kilg. u-235",
            "kg u-235",
            "kilograms-u23",
            "kilograms u-2",
            "kilograms",
            "kg of",
            "kg-u-235",
            "kilgrams",
            "kilogr. u235",
            "uranium kg",
            "kg uranium25",
            "kilogr. u-235",
            "kg uranium 25",
            "kilgr. u-235",
            "kguranium 25",
            "kg-u235",
            "kgm",
        },
        "klbs": {
            "klbs",
            "k lbs.",
            "k lbs",
            "1000 / lbs",
            "1000 lbs",
        },
        "mmbtu": {
            "mmbtu",
            "mmbtus",
            "mbtus",
            "(mmbtu)",
            "mmbtu's",
            "nuclear-mmbtu",
            "nuclear-mmbt",
            "mmbtul",
        },
        "btu": {
            "btu",
            "btus",
        },
        "mwdth": {
            "mwdth",
            "mwd therman",
            "mw days-therm",
            "mwd thrml",
            "mwd thermal",
            "mwd/mtu",
            "mw days",
            "mwd",
            "mw day",
            "dth",
            "mwdaysthermal",
            "mw day therml",
            "mw days thrml",
            "nuclear mwd",
            "mmwd",
            "mw day/therml",
            "mw days/therm",
            "mw days (th",
            "ermal)",
        },
        "mwhth": {
            "nwh therm",
            "mwhth",
            "mwh them",
            "mwh threm",
            "mwh therm",
            "mwh",
            "mwh therms.",
            "mwh term.uts",
            "mwh thermal",
            "mwh thermals",
            "mw hr therm",
            "mwh therma",
            "mwh therm.uts",
        },
        "na_category": {
            "na_category",
            "na",
            "",
            "1265",
            "mwh units",
            "composite",
            "therms",
            "n/a",
            "mbtu/kg",
            "uranium 235",
            "oil",
            "ccf",
            "2261",
            "uo2",
            "(7)",
            "oil #2",
            "oil #6",
            'a"',
            "dekatherm",
            "0",
            "nuclear",
            "gas",
            "62,679",
            "uranium",
            "oil/gas",
            "thermal",
            "(thermal)",
            "se uo2",
            "181679",
            "83",
            "3070",
            "248",
            "273976",
            "747",
            "-",
            "are total",
            "pr. creek",
            "decatherms",
            "uramium",
            ".",
            "total pr crk",
            ">>>>>>>>",
            "all",
            "total",
            "alternative-t",
            "oil-mcf",
            "3303671",
            "929",
            "7182175",
            "319",
            "1490442",
            "10881",
            "1363663",
            "7171",
            "1726497",
            "4783",
            "7800",
            "12559",
            "2398",
            "creek fuels",
            "propane-barre",
            "509",
            "barrels/mcf",
            "propane-bar",
            "4853325",
            "4069628",
            "1431536",
            "708903",
            "mcf/oil (1000",
            "344",
            "mcf / gallen",
            "none",
            "—",
        },
    }
}
"""A mapping of canonical fuel units (keys) to sets of strings representing those fuel
units (values)"""

PLANT_TYPE_CATEGORIES: dict[str, set[str]] = {
    "categories": {
        "steam": {
            "*resp. share steam",
            "(6) milwaukee county is a steam utility plant",
            "(steam)",
            "40% share steam",
            "40% share steamsteam (2)",
            "coal",
            "coal fired steam tur",
            "coal fired steam turbine",
            "coal-fired steam",
            "mpc 50% share steam",
            "mpc 50% shares steam",
            "mpc 50%share steam",
            "mpc40%share steam",
            "mpc50% share steam",
            "oil-fired steam",
            "other-steam expenses:",
            "resp share st note 3",
            "resp share st note3",
            "resp share stm note3",
            "resp. share steam",
            "respondent's sh-st",
            "respondent's sh.-st.",
            "steam",
            "steam (1)",
            "steam (1) & (2)",
            "steam (2)",
            "steam (3)",
            "steam (4)",
            "steam (5)",
            "steam (6)",
            "steam (7)",
            "steam (8)",
            "steam (a)",
            "steam (a,b)",
            "steam (incl i.c.)",
            "steam (leased)",
            "steam (note 1)",
            "steam (retired)",
            "steam (see note 1)",
            "steam (see note 1,",
            "steam (see note 3)",
            "steam (see notes)",
            "steam - 100%",
            "steam - 64%",
            "steam - 72%",
            "steam 1",
            "steam 100",
            "steam 4, 5",
            "steam a",
            "steam conventional",
            "steam fossil",
            "steam heating plant",
            "steam plant",
            "steam plants",
            "steam retired 2013",
            "steam turbine",
            "steam units 1 2 3",
            "steam units 1 and 2",
            "steam units 1, 2, 3",
            "steam units 1,2,3",
            "steam units 3 and 4",
            "steam units 4 & 6",
            "steam units 4 5",
            "steam units 4&5",
            "steam units 4&6",
            "steam units 4, 5",
            "steam- 64%",
            "steam- 72%",
            "steam-100%",
            "steam-64%",
            "steam-common production",
            "steam-common production plant",
            "steam-internal comb",
            "steam/fossil",
            "steam;retired - 2013",
            "steam:",
            "steams",
            "stean",
            "stream",
            "unit total steam",
            "unit total-steam",
            "steam/fuel oil",
        },
        "combustion_turbine": {
            "* gas turbine",
            "*gas turbine",
            "*gas turbine(note 3)",
            "*gas turbine(note3)",
            "add to gas turbine",
            "b-combustion turbine unit",
            "com turbine peaking",
            "comb turb peak units",
            "comb turb peaking",
            "comb turbine peaking",
            "comb. turb-gas oper",
            "comb. turb-gas oper.",
            "comb. turb.",
            "comb. turb. oil oper",
            "comb. turbine (a)",
            "comb. turine",
            "comb.turb-gas oper.",
            "comb.turb.gas/oil op",
            "comb.turb.oil oper",
            "comb.turb.oil oper.",
            "comb.turb.peak.units",
            "combusion turbine",
            "combusition turbine",
            "combustine turbine",
            "combustion and gas",
            "combustion burbine",
            "combustion tubine",
            "combustion turbin",
            "combustion turbine",
            "combustion_turbine",
            "combustionturbine",
            "combution turbin",
            "combution turbine",
            "combutsion turbine",
            "comubstion turbine",
            "conbustion turbine",
            "cumbustion turbine",
            "gas - turbine",
            "gas expander turbine",
            "gas plant",
            "gas trubine",
            "gas turb",
            "gas turb(see notes)",
            "gas turb, diesel",
            "gas turb, int. comb",
            "gas turb, int. comb.",
            "gas turb. (see note",
            "gas turb.(see note5)",
            "gas turb.(see notes)",
            "gas turbin (leasd)",
            "gas turbin (lease",
            "gas turbine",
            "gas turbine # 1",
            "gas turbine #1",
            "gas turbine (2)",
            "gas turbine (a)",
            "gas turbine (leasd)",
            "gas turbine (lease",
            "gas turbine (lease)",
            "gas turbine (leased",
            "gas turbine (leased)",
            "gas turbine (note 1)",
            "gas turbine (note 3)",
            "gas turbine (note1)",
            "gas turbine (see nos",
            "gas turbine - note 1",
            "gas turbine note 3,6",
            "gas turbine note 4,6",
            "gas turbine peaker",
            "gas turbine peaking",
            "gas turbine peakload",
            "gas turbine plant",
            "gas turbine plants",
            "gas turbine-simple",
            "gas turbine-unit 2",
            "gas turbine/int comb",
            "gas turbine/int. cm",
            "gas turbine/intcomb",
            "gas turbine:",
            "gas turbine; retired",
            "gas turbines",
            "gas turbines:",
            "gas turbinint comb",
            "gas turbne/int comb",
            "gas-turbine",
            "gasturbine",
            "gt",
            "gtg/gas",
            "i.c.e. /gas turbine",
            "i.c.e./gas tubine",
            "i.c.e./gas turbine",
            "i.c.e/ gas turbine",
            "i.c.e/gas turbine",
            "jet",
            "jet engine",
            "jet powered turbine",
            "jet turbine",
            "resp share gas note3",
            "simple cycle",
            "simple cycle turbine",
            "gas turbine/fuel oil",
        },
        "combined_cycle": {
            "Combined cycle",
            "cc",
            "cc / gas turb",
            "com cycle gas turb",
            "com. cyc",
            "com. cycle",
            "comb cyc",
            "comb cyc (a) (b)",
            "comb cycle",
            "comb cycle (a,b)",
            "comb cycle gas turb",
            "comb cycle/steam tur",
            "comb. cycle",
            "comb. cycle gas turb",
            "combine cycle",
            "combine cycle oper",
            "combined",
            "combined cycle",
            "combined cycle (a,b)",
            "combined cycle - 40%",
            "combined cycle ctg",
            "combined cycle oper",
            "combined cycle operation",
            "combined cycle opern",
            "combined cycle plant:",
            "combined turbine",
            "combined_cycle",
            "combinedcycle",
            "ctg steam gas",
            "ctg/steam -gas",
            "ctg/steam-gas",
            "gas & steam turbine",
            "gas / steam",
            "gas fired cc turbine",
            "gas steam",
            "gas trb & heat rec",
            "gas tur & comb cyc",
            "gas tur & comb cycl",
            "gas turb-combined cy",
            "gas turb. & heat rec",
            "gas turb/comb cycle",
            "gas turb/comb. cyc",
            "gas turb/cumb. cyc",
            "gas turb/cumbus cycl",
            "gas turbine / steam",
            "gas turbine-combine cycle",
            "gas turbine-combined cycle",
            "gas turbine/ steam",
            "gas turbine/steam",
            "gas/steam",
            "gas/steam comb cycle",
            "gas/steam comb. cycl",
            "gasturb/comb cycle",
            "igcc",
            "igcc/gas turbine",
            "ngcc",
            "st/gas turb comb cyc",
            "steam & cc",
            "steam & comb cycle",
            "steam & combined ce",
            "steam & gas turbine",
            "steam (comb. cycle)",
            "steam (comb. cycle)gas turbine/steam",
            "steam and cc",
            "steam cc",
            "steam comb cycle",
            "steam/gas turb.",
            "steam/gas turbine",
            "combined cycle-40%",
        },
        "nuclear": {
            "nuclear",
            "nuclear (3)",
            "steam(nuclear)",
            "nuclear(see note4)",
            "nuclear steam",
            "nuclear turbine",
            "nuclear - steam",
            "nuclear (a)(b)(c)",
            "nuclear (b)(c)",
            "* nuclear",
            "nuclear (b) (c)",
            "nuclear (see notes)",
            "steam (nuclear)",
            "* nuclear (note 2)",
            "nuclear (note 2)",
            "nuclear (see note 2)",
            "nuclear(see notes)",
            "nuclear-steam",
            "nuclear (see note 3)",
        },
        "geothermal": {"steam - geothermal", "steam_geothermal", "geothermal"},
        "internal_combustion": {
            "*int combustion (1)",
            "*int. combustion (1)",
            "*internal combust'n",
            "auke bay internal combustion",
            "auke bay internal combustion:",
            "available for specific diesel unit.",
            "available for specific diesel units.",
            "a-internal combustion unit",
            "black start diesel units",
            "black start diesel units:",
            "comb. cyc",
            "comb. turbine",
            "combustion",
            "conemaugh - internal combustion",
            "diesal",
            "diesel",
            "diesel engine",
            "diesel generators",
            "diesel plant",
            "diesel plants",
            "diesel plants:",
            "diesel:",
            "diesel turbine",
            "east hampton diesel",
            "gold creek internal combustion:",
            "ic",
            "int combust (note 1)",
            "int combust (note1)",
            "int combust - note 1",
            "int. combust (note1)",
            "int. combust - note1",
            "int. combustine",
            "int. combustion",
            "int. combustion (1)",
            "int.combustine",
            "interal",
            "internal conbustion",
            "interal combustion:",
            "internal combustion:",
            "internal",
            "internal comb",
            "internal comb recip",
            "internal comb.",
            "internal combust.",
            "internal combustion",
            "internal combustion :",
            "internal combustion auxiliary",
            "internal combustion peaking units",
            "internal combustion plants:",
            "internal combustion reciprocating",
            "internal combustion-diesel",
            "internal combustion: (emergency standby",
            "internal conbustion:",
            "internal_combustion",
            "international combustion",
            "internl combustion",
            "lemon creek internal combustion",
            "lemon creek internal combustion:",
            "montauk diesel",
            "other / internal combustion :",
            "other/internal combustion",
            "reciprocating engine",
            "shawville - internal combustion",
            "steam internal comb",
        },
        "wind": {
            "(7) montfort wind farm was purchased dec. 2012.",
            "rosiere wind turbines",
            "suisun wind power plant (r)",
            "wind",
            "wind energy",
            "wind turbine",
            "wind - turbine",
            "wind generation",
            "wind turbin",
            "wind turbine plant",
            "wind turbine plants",
            "wind turbine plants:",
            "wind turbine:",
            "wind turbines",
            "wind:",
        },
        "photovoltaic": {
            "community solar gardens",
            "other production - solar",
            "photovoltaic",
            "sixth street solar",
            "solar",
            "solar array",
            "solar electric",
            "solar photovoltaic",
            "solar photovoltiac",
            "solar plant",
            "solar plants",
            "solar plants:",
            "solar project",
            "solar:",
            "proj dvlpmnt div solar photovoltaic project:",
            "photo voltaic generating plants:",
            "customer hosted solar:",
        },
        "solar_thermal": {"solar thermal", "solar_thermal"},
        "fuel_cell": {
            "fuel cell",
            "fuel_cell",
            "fuel cell:",
        },
        "hydro": {
            "(1) the dixon hydro electric generating facility",
            "(6) excludes portion allocated to other water",
            "(hydroelectric facility)",
            "a - hydro license project no. 2315.",
            "a-hydro license project no. 2315",
            "additional hydro plant",
            "all other hydro misc. plant, resv., etc.",
            "all other hydro miscellaneous plant",
            "all other hydro plant, resv., misc., etc.",
            "conventional hydro",
            "hydraulic",
            "hydraulic (1):",
            "hydraulic race (project no. 2424)(note 1)",
            "hydraulic:",
            "hydro",
            "hydro -",
            "hydro : licensed proj. no.",
            "hydro electric",
            "hydro lic project no.",
            "hydro lic. project no.",
            "hydro license",
            "hydro license no.",
            "hydro plant:",
            "hydro plants",
            "hydro plants:",
            "hydro plants: licensed proj no.",
            "hydro plants: licensed proj. no.",
            "hydro-electric",
            "hydro-neal shoals",
            "hydro:",
            "hydro: lic project no.",
            "hydro: lic. project no.",
            "hydro: license no.",
            "hydroelectric",
            "hydroelectric : licensed proj. no.",
            "hydroelectric generating plants",
            "hydroelectric generating plants :",
            "hydroelectric generating plants:",
            "hydroelectric: licensed proj. no.",
            "hyrdo",
            "neal shoals-hydro license",
            "other hydro assets (resvr., strm gages, etc.)",
            "other hydro assets (resvr.,strm gages, etc.)",
            "other hydro costs:",
            "research and development costs (hydros)",
            "small hydroelectric generating plants:",
            "small hydroelectric plants:",
            "talcville hydro station(project no. 4402)(note 1",
            "the hydro plant above",
            "unallocated - hydros",
            "unallocated hydros",
        },
        "storage": {
            "daily storage",
            "storage",
            "storage (re-reg)",
            "storage - dailey",
            "storage - daily",
            "storage - weekly",
            "storage reservoir",
            "weekly storage",
        },
        "na_category": {
            "",
            "(left blank)",
            "(see note 5)",
            "(see note 7)",
            "--",
            "0",
            "0.0000",
            "combined heat/power",
            "common",
            "common expenses",
            "conventional",
            "expenses commom to",
            "expenses common to",
            "footnote",
            "fuel facilities",
            "fuel oil",
            "gas",
            "n/a",
            "na",
            "na_category",
            "non-applicable",
            "none",
            "none.",
            "not applicable",
            "not applicable.",
            "not in service",
            "not in service in",
            "oil",
            "other",
            "other generation",
            "other generation plt",
            "package",
            "peach bottom",
            "plant retired- 2013",
            "plants sold in 1999",
            "resp share (note 11)",
            "resp share - note 11",
            "resp share - note 5",
            "resp. share (note 2)",
            "resp. share (note 3)",
            "resp. share (note 4)",
            "resp. share (note 6)",
            "resp. share (note 8)",
            "resp. share (note 9)",
            "resp. share (note11)",
            "resp. share (note2)",
            "resp. share (note4)",
            "resp. share (note6)",
            "resp. share (note8)",
            "resp. share - note 2",
            "resp. share - note 4",
            "resp. share - note 6",
            "resp. share - note 7",
            "resp. share - note 8",
            "resp. share - note 9",
            "resp. share note 11",
            "respondent's share",
            "retired",
            "retired - 2013",
            "retired in 1978",
            "retired in 1981",
            "see pgs 402.1-402.3",
            "see pgs 403.1-403.9",
            "sold april 16, 1999",
            "sold may 07, 1999",
            "storage/pipelines",
            "unit total (note 2)",
            "unit total (note 3)",
            "unit total (note2)",
            "unit total (note3)",
            "waste heat",
            "—",
        },
    }
}
"""A mapping from canonical plant kinds (keys) to the associated freeform strings
(values) identified as being associated with that kind of plant in the FERC Form 1 raw
data.

There are many strings that weren't categorized, Solar and Solar Project were not
classified as these do not indicate if they are solar thermal or photovoltaic. Variants
on Steam (e.g. "steam 72" and "steam and gas") were classified based on additional
research of the plants on the Internet.
"""

PLANT_TYPE_CATEGORIES_HYDRO: dict[str, set[str]] = {
    "categories": {
        "hydro": {
            "conventional",
            "hydro",
            "hydro -",
            "hydro : licensed proj. no.",
            "hydro electric",
            "hydro lic project no.",
            "hydro license no.",
            "hydro plant:",
            "hydro plants",
            "hydro plants:",
            "hydro plants: licensed proj no.",
            "hydro plants: licensed proj. no.",
            "hydro-electric",
            "hydro:",
            "hydro: lic project no.",
            "hydro: lic. project no.",
            "hydro: license no.",
            "hydroelectric",
            "hydroelectric : licensed proj. no.",
            "hydroelectric generating plants",
            "hydroelectric generating plants :",
            "hydroelectric generating plants:",
            "hydroelectric: licensed proj. no.",
            "hyrdo",
            "small hydroelectric plants:",
            "unallocated hydros",
        },
        "run_of_river": {
            "run_of_river",
            "run ofriver",
            "r of r",
            "run - of - river",
            "run -of - river",
            "run of river",
            "run-of -river",
            "run-of river",
            "run-of-river",
        },
        "run_of_river_with_storage": {
            "run_of_river_with_storage",
            "run-of-river/storage",
            "r of r/ storage",
            "r of r/storage",
            "r or r/storage",
            "run of river-stge",
            "run of river-stora",
            "run of river-storage",
            "run-of river;storage",
            "run-of-river/stora",
            "run-of-river; sto",
            "run-of-river; stor",
            "run-of-river; stora",
            "run-of-river; storag",
            "run-of-river;stor",
            "run-of-river;stora",
            "run-of-river;storage",
        },
        "storage": {
            "daily storage",
            "storage",
            "storage (re-reg)",
            "storage - dailey",
            "storage - daily",
            "storage - weekly",
            "storage reservoir",
            "weekly storage",
        },
        "na_category": {
            "na_category",
            "(see note 7, page",
            "-",
            "---not applicable---",
            "--not applicable--",
            "00",
            "blank page",
            "gap common",
            "not used",
            "peaking",
            "schedule is n/a",
            "see footnote",
            "see note 7,p.403fn",
            "see note 7,pg 4031",
            "sold in 1999",
            "",
            "not applicable.",
            "gas turbine",
            "na",
            "not applicable",
            "0",
            "none",
            "none.",
            "n/a",
        },
    }
}
"""A mapping from canonical plant kinds (keys) to the associated freeform strings
(values) identified as being associated with that kind of plant in the FERC Form 1 Hydro
Plants data.

These are seperated out from the rest of the plant types due to the difference in
languaged used to refer to hydro vs. other types of plants. For example: "conventional"
in the context of a hydro plant means that it is conventional hydro-electric. In the
context of the steam table, however, it's unclear what conventional means.
"""

CONSTRUCTION_TYPE_CATEGORIES: dict[str, set[str]] = {
    "categories": {
        "outdoor": {
            "outdoor",
            "outdoor boiler",
            "full outdoor",
            "outdoor boilers",
            "outboilers",
            "fuel outdoor",
            "outdoors",
            "boiler outdoor& full",
            "boiler outdoor&full",
            "outdoor boiler& full",
            "full -outdoor",
            "outdoor steam",
            "ob",
            "outdoor automatic",
            "outdoor repower",
            "full outdoor boiler",
            "fo",
            "outdoor boiler & ful",
            "full-outdoor",
            "outoor",
            "outdoor  boiler&full",
            "boiler outdoor &full",
            "outdoor boiler &full",
            "boiler outdoor & ful",
            "outdoor-boiler",
            "outdoor - boiler",
            "outdoor const.",
            "4 outdoor boilers",
            "3 outdoor boilers",
            "full outdoors",
            "full oudoors",
            "outdoor (auto oper)",
            "outside boiler",
            "outdoor boiler&full",
            "outdoor hrsg",
            "outdoor-steel encl.",
            "boiler-outdr & full",
            "con.& full outdoor",
            "partial outdoor",
            "outdoor (auto. oper)",
            "outdoor (auto.oper)",
            "outdoor construction",
            "1 outdoor boiler",
            "2 outdoor boilers",
            "outdoor enclosure",
            "2 outoor boilers",
            "boiler outdr.& full",
            "boiler outdr. & full",
            "ful outdoor",
            "outdoor-steel enclos",
            "outdoor (auto oper.)",
            "con. & full outdoor",
            "outdore",
            "boiler & full outdor",
            "full & outdr boilers",
            "outodoor (auto oper)",
            "outdoor steel encl.",
            "full outoor",
            "boiler & outdoor ful",
            "otdr. blr. & f. otdr",
            "f.otdr & otdr.blr.",
            "oudoor (auto oper)",
            "outdoor constructin",
            "f. otdr. & otdr. blr",
            "outdoor boiler & fue",
            "outdoor boiler &fuel",
            "conventional; outdoor",
            "coven./outdoor",
            "conv & outdoor boiler",
        },
        "semioutdoor": {
            "semioutdoor",
            "more than 50% outdoors",
            "more than 50% outdoo",
            "more than 50% outdos",
            "over 50% outdoor",
            "over 50% outdoors",
            "semi-outdoor",
            "semi - outdoor",
            "semi outdoor",
            "semi-enclosed",
            "semi-outdoor boiler",
            "semi outdoor boiler",
            "semi- outdoor",
            "semi - outdoors",
            "semi -outdoor",
            "conven & semi-outdr",
            "conv & semi-outdoor",
            "conv & semi- outdoor",
            "convent. semi-outdr",
            "conv. semi outdoor",
            "conv(u1)/semiod(u2)",
            "conv u1/semi-od u2",
            "conv-one blr-semi-od",
            "convent semioutdoor",
            "conv. u1/semi-od u2",
            "conv - 1 blr semi od",
            "conv. ui/semi-od u2",
            "conv-1 blr semi-od",
            "conven. semi-outdoor",
            "conv semi-outdoor",
            "u1-conv./u2-semi-od",
            "u1-conv./u2-semi -od",
            "convent. semi-outdoo",
            "u1-conv. / u2-semi",
            "outdr & conventnl",
            "conven. full outdoor",
            "conv. & outdoor blr",
            "conv. & outdoor blr.",
            "conv. & outdoor boil",
            "conv. & outdr boiler",
            "conv. & out. boiler",
            "convntl,outdoor blr",
            "outdoor & conv.",
            "2 conv., 1 out. boil",
            "outdoor/conventional",
            "conv. boiler outdoor",
            "conv-one boiler-outd",
            "conventional outdoor",
            "conventional outdor",
            "conv. outdoor boiler",
            "conv.outdoor boiler",
            "conventional outdr.",
            "conven,outdoorboiler",
            "conven full outdoor",
            "conven,full outdoor",
            "1 out boil, 2 conv",
            "conv. & full outdoor",
            "conv. & outdr. boilr",
            "conv outdoor boiler",
            "convention. outdoor",
            "conv. sem. outdoor",
            "convntl, outdoor blr",
            "conv & outdoor boil",
            "conv & outdoor boil.",
            "outdoor & conv",
            "conv. broiler outdor",
            "1 out boilr, 2 conv",
            "conv.& outdoor boil.",
            "conven,outdr.boiler",
            "conven,outdr boiler",
            "outdoor & conventil",
            "1 out boilr 2 conv",
            "conv & outdr. boilr",
            "conven, full outdoor",
            "conven full outdr.",
            "conven, full outdr.",
            "conv/outdoor boiler",
            "convnt'l outdr boilr",
            "1 out boil 2 conv",
            "conv full outdoor",
            "conven, outdr boiler",
            "conventional/outdoor",
            "conv&outdoor boiler",
            "outdoor & convention",
            "conv & outdoor boilr",
            "conv & full outdoor",
            "convntl. outdoor blr",
            "conv - ob",
            "1conv'l/2odboilers",
            "2conv'l/1odboiler",
            "conv-ob",
            "conv.-ob",
            "1 conv/ 2odboilers",
            "2 conv /1 odboilers",
            "conv- ob",
            "conv -ob",
            "con sem outdoor",
            "cnvntl, outdr, boilr",
            "less than 50% outdoo",
            "less than 50% outdoors",
            "under 50% outdoor",
            "under 50% outdoors",
            "1cnvntnl/2odboilers",
            "2cnvntnl1/1odboiler",
            "con & ob",
            "combination (b)",
            "indoor & outdoor",
            "conven. blr. & full",
            "conv. & otdr. blr.",
            "combination",
            "indoor and outdoor",
            "conven boiler & full",
            "2conv'l/10dboiler",
            "4 indor/outdr boiler",
            "4 indr/outdr boilerr",
            "4 indr/outdr boiler",
            "indoor & outdoof",
            "convet. semi-outdoor",
        },
        "conventional": {
            "conventional",
            "conventional boiler",
            "conventional - boiler",
            "conv-b",
            "conventionall",
            "convention",
            "coventional",
            "conven full boiler",
            "c0nventional",
            "conventtional",
            "convential",
            "conventional bulb",
            "conventrional",
            "*conventional",
            "convetional",
            "conventioanl",
            "conventioinal",
            "conventaional",
            "indoor construction",
            "convenional",
            "conventional steam",
            "conventinal",
            "convntional",
            "conventionl",
            "conventionsl",
            "conventiional",
            "convntl steam plants",
            "indoor const.",
            "full indoor",
            "indoor",
            "indoor automatic",
            "indoor boiler",
            "indoor boiler and steam turbine",
            "(peak load) indoor",
            "conventionl,indoor",
            "conventionl, indoor",
            "conventional, indoor",
            "conventional;outdoor",
            "conven./outdoor",
            "conventional;semi-ou",
            "comb. cycle indoor",
            "comb cycle indoor",
            "3 indoor boiler",
            "2 indoor boilers",
            "1 indoor boiler",
            "2 indoor boiler",
            "3 indoor boilers",
            "fully contained",
            "conv - b",
            "conventional/boiler",
            "cnventional",
            "comb. cycle indooor",
            "sonventional",
            "ind enclosures",
            "conentional",
            "conventional - boilr",
            "indoor boiler and st",
            "conven",
            "conv. underground",
            "conven. underground",
            "conventional (a)",
        },
        "na_category": {
            "na_category",
            "na",
            "",
            "automatic operation",
            "comb. turb. installn",
            "comb. turb. instaln",
            "com. turb. installn",
            "n/a",
            "for detailed info.",
            "for detailed info",
            "combined cycle",
            "not applicable",
            "gas",
            "heated individually",
            "metal enclosure",
            "pressurized water",
            "nuclear",
            "jet engine",
            "gas turbine",
            "storage/pipelines",
            "0",
            "during 1994",
            "peaking - automatic",
            "gas turbine/int. cm",
            "2 oil/gas turbines",
            "wind",
            "package",
            "mobile",
            "auto-operated",
            "steam plants",
            "other production",
            "all nuclear plants",
            "other power gen.",
            "automatically operad",
            "automatically operd",
            "circ fluidized bed",
            "jet turbine",
            "gas turbne/int comb",
            "automatically oper.",
            "retired 1/1/95",
            "during 1995",
            "1996. plant sold",
            "reactivated 7/1/96",
            "gas turbine/int comb",
            "portable",
            "head individually",
            "automatic opertion",
            "peaking-automatic",
            "cycle",
            "full order",
            "circ. fluidized bed",
            "gas turbine/intcomb",
            "0.0000",
            "none",
            "2 oil / gas",
            "block & steel",
            "and 2000",
            "comb.turb. instaln",
            "automatic oper.",
            "pakage",
            "---",
            "—",
            "n/a (ct)",
            "comb turb instain",
            "ind encloures",
            "2 oil /gas turbines",
            "combustion turbine",
            "1970",
            "gas/oil turbines",
            "combined cycle steam",
            "pwr",
            "2 oil/ gas",
            "2 oil / gas turbines",
            "gas / oil turbines",
            "no boiler",
            "internal combustion",
            "gasturbine no boiler",
            "boiler",
            "tower -10 unit facy",
            "gas trubine",
            "4 gas/oil trubines",
            "2 oil/ 4 gas/oil tur",
            "5 gas/oil turbines",
            "tower 16",
            "2 on 1 gas turbine",
            "tower 23",
            "tower -10 unit",
            "tower - 101 unit",
            "3 on 1 gas turbine",
            "tower - 10 units",
            "tower - 165 units",
            "wind turbine",
            "fixed tilt pv",
            "tracking pv",
            "o",
            "wind trubine",
            "wind generator",
            "subcritical",
            "sucritical",
            "simple cycle",
            "simple & reciprocat",
            "solar",
            "pre-fab power plant",
            "prefab power plant",
            "prefab. power plant",
            "pump storage",
            "underground",
            "see page 402",
            "non-applicable",
            "duct burner",
            "see footnote",
            "simple and reciprocat",
            "--not applicable--",
            "---not applicable---",
            "not applicable.",
            "schedule is n/a",
            "none.",
            "see note",
            "not appicable",
            "pv tracking",
            "r of r/storage",
            "simple and reciprocating",
            "simple & reciprocating",
        },
    }
}
"""A dictionary of construction types (keys) and lists of construction type strings
associated with each type (values) from FERC Form 1.

There are many strings that weren't categorized, including crosses between conventional
and outdoor, PV, wind, combined cycle, and internal combustion. The lists are broken out
into the two types specified in Form 1: conventional and outdoor. These lists are
inclusive so that variants of conventional (e.g. "conventional full") and outdoor (e.g.
"outdoor full" and "outdoor hrsg") are included.
"""

##############################################################################
# Fully assembled set of FERC 1 transformation parameters
##############################################################################
TRANSFORM_PARAMS = {
    "core_ferc1__yearly_steam_plants_fuel_sched402": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "plant_name": "plant_name_ferc1",
                    "fuel": "fuel_type_code_pudl",
                    "fuel_unit": "fuel_units",
                    "fuel_avg_heat": "fuel_btu_per_unit",
                    "fuel_quantity": "fuel_consumed_units",
                    "fuel_cost_burned": "fuel_cost_per_unit_burned",
                    "fuel_cost_delvd": "fuel_cost_per_unit_delivered",
                    # Note: Original fuel_cost_btu is misleadingly named
                    "fuel_cost_btu": "fuel_cost_per_mmbtu",
                    "fuel_generaton": "fuel_btu_per_kwh",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "report_year": "report_year",
                    "row_number": "row_number",
                    "fuel_cost_kwh": "fuel_cost_per_kwh",
                    "spplmnt_num": "spplmnt_num",
                    "row_seq": "row_seq",
                }
            },
            "xbrl": {
                "columns": {
                    "plant_name_axis": "plant_name_ferc1",
                    "fuel_kind_axis": "fuel_type_code_pudl",
                    "fuel_unit": "fuel_units",
                    "fuel_burned_average_heat_content": "fuel_btu_per_unit",
                    "quantity_of_fuel_burned": "fuel_consumed_units",
                    "average_cost_of_fuel_per_unit_burned": "fuel_cost_per_unit_burned",
                    "average_cost_of_fuel_per_unit_as_delivered": "fuel_cost_per_unit_delivered",
                    "average_cost_of_fuel_burned_per_million_british_thermal_unit": "fuel_cost_per_mmbtu",
                    "average_british_thermal_unit_per_kilowatt_hour_net_generation": "fuel_btu_per_kwh",
                    "average_cost_of_fuel_burned_per_kilowatt_hour_net_generation": "fuel_cost_per_kwh",
                    "report_year": "report_year",
                    "fuel_kind": "fuel_kind",
                    "end_date": "end_date",
                    "entity_id": "utility_id_ferc1_xbrl",
                    "start_date": "start_date",
                }
            },
        },
        "categorize_strings": {
            "fuel_type_code_pudl": FUEL_CATEGORIES,
            "fuel_units": FUEL_UNIT_CATEGORIES,
        },
        "convert_units": {
            "fuel_btu_per_unit": BTU_TO_MMBTU,
            "fuel_btu_per_kwh": BTU_PERKWH_TO_MMBTU_PERMWH,
            "fuel_cost_per_kwh": PERKWH_TO_PERMWH,
        },
        "normalize_strings": {
            "plant_name_ferc1": FERC1_STRING_NORM,
            "fuel_type_code_pudl": FERC1_STRING_NORM,
            "fuel_units": FERC1_STRING_NORM,
        },
        "correct_units": [
            COAL_MMBTU_PER_UNIT_CORRECTIONS,
            GAS_MMBTU_PER_UNIT_CORRECTIONS,
            OIL_MMBTU_PER_UNIT_CORRECTIONS,
            COAL_COST_PER_MMBTU_CORRECTIONS,
            GAS_COST_PER_MMBTU_CORRECTIONS,
            OIL_COST_PER_MMBTU_CORRECTIONS,
        ],
        "drop_invalid_rows": [
            {
                "invalid_values": [0, pd.NA, np.nan, ""],
                "required_valid_cols": [
                    "fuel_consumed_units",
                    "fuel_cost_per_mmbtu",
                    "fuel_cost_per_mwh",
                    "fuel_cost_per_unit_delivered",
                    "fuel_cost_per_unit_burned",
                    "fuel_mmbtu_per_unit",
                    "fuel_units",
                ],
            },
            {
                "invalid_values": INVALID_PLANT_NAMES
                + ["must 123", "must 456", "ant1-3", "elk 1-3"],
                "required_valid_cols": ["plant_name_ferc1"],
            },
        ],
    },
    "core_ferc1__yearly_steam_plants_sched402": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "cost_structure": "capex_structures",
                    "expns_misc_power": "opex_misc_power",
                    "plant_name": "plant_name_ferc1",
                    "plnt_capability": "plant_capability_mw",
                    "expns_plants": "opex_plants",
                    "expns_misc_steam": "opex_misc_steam",
                    "cost_per_kw": "capex_per_kw",
                    "when_not_limited": "not_water_limited_capacity_mw",
                    "asset_retire_cost": "asset_retirement_cost",
                    "expns_steam_othr": "opex_steam_other",
                    "expns_transfer": "opex_transfer",
                    "expns_engnr": "opex_engineering",
                    "avg_num_of_emp": "avg_num_employees",
                    "cost_of_plant_to": "capex_total",
                    "expns_rents": "opex_rents",
                    "tot_prdctn_expns": "opex_production_total",
                    "plant_kind": "plant_type",
                    "respondent_id": "utility_id_ferc1_dbf",
                    "expns_operations": "opex_operations",
                    "cost_equipment": "capex_equipment",
                    "type_const": "construction_type",
                    "plant_hours": "plant_hours_connected_while_generating",
                    "expns_coolants": "opex_coolants",
                    "expns_fuel": "opex_fuel",
                    "when_limited": "water_limited_capacity_mw",
                    "expns_kwh": "opex_per_kwh",
                    "expns_allowances": "opex_allowances",
                    "expns_steam": "opex_steam",
                    "yr_const": "construction_year",
                    "yr_installed": "installation_year",
                    "expns_boiler": "opex_boiler",
                    "peak_demand": "peak_demand_mw",
                    "cost_land": "capex_land",
                    "tot_capacity": "capacity_mw",
                    "net_generation": "net_generation_kwh",
                    "expns_electric": "opex_electric",
                    "expns_structures": "opex_structures",
                    "report_year": "report_year",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "spplmnt_num": "spplmnt_num",
                    "row_seq": "row_seq",
                },
            },
            "xbrl": {
                "columns": {
                    "cost_of_structures_and_improvements_steam_production": "capex_structures",
                    "miscellaneous_steam_power_expenses": "opex_misc_power",
                    "plant_name_axis": "plant_name_ferc1",
                    "net_continuous_plant_capability": "plant_capability_mw",
                    "maintenance_of_electric_plant_steam_power_generation": "opex_plants",
                    "maintenance_of_miscellaneous_steam_plant": "opex_misc_steam",
                    "cost_per_kilowatt_of_installed_capacity": "capex_per_kw",
                    "net_continuous_plant_capability_not_limited_by_condenser_water": "not_water_limited_capacity_mw",
                    "asset_retirement_costs_steam_production": "asset_retirement_cost",
                    "steam_from_other_sources": "opex_steam_other",
                    "steam_transferred_credit": "opex_transfer",
                    "maintenance_supervision_and_engineering_steam_power_generation": "opex_engineering",
                    "plant_average_number_of_employees": "avg_num_employees",
                    "cost_of_plant": "capex_total",
                    "rents_steam_power_generation": "opex_rents",
                    "power_production_expenses_steam_power": "opex_production_total",
                    "plant_kind": "plant_type",
                    "operation_supervision_and_engineering_expense": "opex_operations",
                    "cost_of_equipment_steam_production": "capex_equipment",
                    "plant_construction_type": "construction_type",
                    "plant_hours_connected_to_load": "plant_hours_connected_while_generating",
                    "coolants_and_water": "opex_coolants",
                    "fuel_steam_power_generation": "opex_fuel",
                    "net_continuous_plant_capability_limited_by_condenser_water": "water_limited_capacity_mw",
                    "expenses_per_net_kilowatt_hour": "opex_per_kwh",
                    "allowances": "opex_allowances",
                    "steam_expenses_steam_power_generation": "opex_steam",
                    "year_plant_originally_constructed": "construction_year",
                    "year_last_unit_of_plant_installed": "installation_year",
                    "maintenance_of_boiler_plant_steam_power_generation": "opex_boiler",
                    "net_peak_demand_on_plant": "peak_demand_mw",
                    "cost_of_land_and_land_rights_steam_production": "capex_land",
                    "installed_capacity_of_plant": "capacity_mw",
                    "net_generation_excluding_plant_use": "net_generation_kwh",
                    "electric_expenses_steam_power_generation": "opex_electric",
                    "maintenance_of_structures_steam_power_generation": "opex_structures",
                    "report_year": "report_year",
                    "order_number": "order_number",
                    "plant_name": "plant_name",
                    "entity_id": "utility_id_ferc1_xbrl",
                    "end_date": "end_date",
                    "start_date": "start_date",
                    "date": "date",
                },
            },
        },
        "normalize_strings": {
            "plant_name_ferc1": FERC1_STRING_NORM,
            "construction_type": FERC1_STRING_NORM,
            "plant_type": FERC1_STRING_NORM,
        },
        "nullify_outliers": {
            "construction_year": VALID_PLANT_YEARS,
            "installation_year": VALID_PLANT_YEARS,
        },
        "categorize_strings": {
            "construction_type": CONSTRUCTION_TYPE_CATEGORIES,
            "plant_type": PLANT_TYPE_CATEGORIES,
        },
        "convert_units": {
            "capex_per_kw": PERKW_TO_PERMW,
            "opex_per_kwh": PERKWH_TO_PERMWH,
            "net_generation_kwh": KWH_TO_MWH,
        },
        "spot_fix_values": [
            {
                "idx_cols": ["record_id"],
                "fix_cols": ["plant_name_ferc1"],
                "expect_unique": True,
                "spot_fixes": [
                    ("f1_steam_1999_12_72_0_1", "clifty creek"),
                    ("f1_steam_2010_12_306_0_1", "harrison county"),
                    ("f1_steam_1997_12_230_0_1", "hermiston generating"),
                    ("f1_steam_1998_12_64_0_1", "hardee power station"),
                    ("f1_steam_2015_12_276_0_1", "state line"),
                    ("f1_steam_2014_12_276_0_1", "state line"),
                    ("f1_steam_2003_12_62_2_3", "pea ridge"),
                    ("f1_steam_2003_12_62_2_2", "smith"),
                    ("f1_steam_2000_12_204_0_1", "seabrook"),
                    ("f1_steam_2001_12_204_0_1", "seabrook"),
                ],
            },
            {
                "idx_cols": ["record_id"],
                "fix_cols": ["capacity_mw"],
                "expect_unique": True,
                "spot_fixes": [
                    (
                        "steam_electric_generating_plant_statistics_large_plants_402_2021_c000290_cliffside",
                        1530.5,
                    ),
                ],
            },
        ],
        "drop_invalid_rows": [
            {
                "invalid_values": [0, "0", pd.NA, np.nan, "", "none"],
                "allowed_invalid_cols": [
                    "record_id",
                    "utility_id_ferc1",
                    "utility_id_ferc1_dbf",
                    "utility_id_ferc1_xbrl",
                    "plant_name_ferc1",
                    "report_year",
                    "date",
                    "start_date",
                    "end_date",
                    "order_number",
                    "plant_name",
                    "plant_type",
                ],
            },
            {
                "invalid_values": INVALID_PLANT_NAMES,
                "required_valid_cols": ["plant_name_ferc1"],
            },
        ],
    },
    "core_ferc1__yearly_hydroelectric_plants_sched406": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "plant_name": "plant_name_ferc1",
                    "project_no": "project_num",
                    "yr_const": "construction_year",
                    "plant_kind": "plant_type",
                    "plant_const": "construction_type",
                    "yr_installed": "installation_year",
                    "tot_capacity": "capacity_mw",
                    "peak_demand": "peak_demand_mw",
                    "plant_hours": "plant_hours_connected_while_generating",
                    "favorable_cond": "net_capacity_favorable_conditions_mw",
                    "adverse_cond": "net_capacity_adverse_conditions_mw",
                    "avg_num_of_emp": "avg_num_employees",
                    "net_generation": "net_generation_kwh",
                    "cost_of_land": "capex_land",
                    "cost_structure": "capex_structures",
                    "cost_facilities": "capex_facilities",
                    "cost_equipment": "capex_equipment",
                    "cost_roads": "capex_roads",
                    "cost_plant_total": "capex_total",
                    "cost_per_kw": "cost_per_kw",
                    "expns_operations": "opex_operations",
                    "expns_water_pwr": "opex_water_for_power",
                    "expns_hydraulic": "opex_hydraulic",
                    "expns_electric": "opex_electric",
                    "expns_generation": "opex_generation_misc",
                    "expns_rents": "opex_rents",
                    "expns_structures": "opex_structures",
                    "expns_dams": "opex_dams",
                    "expns_plant": "opex_plant",
                    "expns_misc_plant": "opex_misc_plant",
                    "expns_kwh": "opex_per_kwh",
                    "expns_engnr": "opex_engineering",
                    "expns_total": "opex_total",
                    "asset_retire_cost": "asset_retirement_cost",
                    "report_year": "report_year",
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                },
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "plant_name": "plant_name_ferc1",
                    "order_number": "order_number",
                    "report_year": "report_year",
                    "date": "date",
                    "start_date": "start_date",
                    "end_date": "end_date",
                    "licensed_project_number_and_hydroelectric_plant_name_axis": "project_num_and_plant_name_ferc1",
                    "installed_capacity_of_plant": "capacity_mw",
                    "cost_of_plant": "capex_total",
                    "equipment_costs_hydroelectric_production": "capex_equipment",
                    "cost_of_reservoirs_dams_and_waterways_hydroelectric_production": "capex_facilities",
                    "cost_of_structures_and_improvements_hydroelectric_production": "capex_structures",  # should we rename this as capex_structures_and_improvements?
                    "asset_retirement_costs_hydroelectric_production": "asset_retirement_cost",  # should we rename this to capex_asset_retirement?
                    "cost_of_land_and_land_rights_hydroelectric_production": "capex_land",
                    "cost_of_roads_railroads_and_bridges_hydroelectric_production": "capex_roads",
                    "licensed_project_number": "project_num",
                    "net_plant_capability_under_most_favorable_operating_conditions": "net_capacity_favorable_conditions_mw",
                    "plant_construction_type": "construction_type",
                    "net_generation_excluding_plant_use": "net_generation_kwh",
                    "expenses_per_net_kilowatt_hour": "opex_per_kwh",
                    "cost_per_kilowatt_of_installed_capacity": "capex_per_kw",
                    "maintenance_of_structures_hydraulic_power_generation": "opex_structures",
                    "maintenance_supervision_and_engineering_hydraulic_power_generation": "opex_engineering",
                    "maintenance_of_miscellaneous_hydraulic_plant": "opex_misc_plant",
                    "hydraulic_expenses": "opex_hydraulic",
                    "water_for_power": "opex_water_for_power",
                    "electric_expenses_hydraulic_power_generation": "opex_electric",
                    "maintenance_of_electric_plant_hydraulic_power_generation": "opex_plant",
                    "plant_hours_connected_to_load": "plant_hours_connected_while_generating",
                    "power_production_expenses_hydraulic_power": "opex_total",
                    "miscellaneous_hydraulic_power_generation_expenses": "opex_generation_misc",
                    "net_peak_demand_on_plant": "peak_demand_mw",
                    "year_last_unit_of_plant_installed": "installation_year",
                    "plant_average_number_of_employees": "avg_num_employees",
                    "plant_kind": "plant_type",
                    "year_plant_originally_constructed": "construction_year",
                    "maintenance_of_reservoirs_dams_and_waterways": "opex_dams",
                    "rents_hydraulic_power_generation": "opex_rents",
                    "net_plant_capability_under_most_adverse_operating_conditions": "net_capacity_adverse_conditions_mw",
                    "operation_supervision_and_engineering_expense": "opex_operations",
                },
            },
        },
        "normalize_strings": {
            "plant_name_ferc1": FERC1_STRING_NORM,
            "construction_type": FERC1_STRING_NORM,
            "plant_type": FERC1_STRING_NORM,
        },
        "nullify_outliers": {
            "construction_year": VALID_PLANT_YEARS,
            "installation_year": VALID_PLANT_YEARS,
        },
        "categorize_strings": {
            "construction_type": CONSTRUCTION_TYPE_CATEGORIES,
            "plant_type": PLANT_TYPE_CATEGORIES_HYDRO,
        },
        "convert_units": {
            "capex_per_kw": PERKW_TO_PERMW,
            "opex_per_kwh": PERKWH_TO_PERMWH,
            "net_generation_kwh": KWH_TO_MWH,
        },
        "strip_non_numeric_values": {"project_num": {"strip_non_numeric_values": True}},
        "drop_invalid_rows": [
            {
                "invalid_values": [0, "0", pd.NA, np.nan, "", "none", "—"],
                "allowed_invalid_cols": [
                    "record_id",
                    "utility_id_ferc1",
                    "utility_id_ferc1_dbf",
                    "utility_id_ferc1_xbrl",
                    "plant_name_ferc1",
                    "project_num_and_plant_name_ferc1",
                    "report_year",
                    "date",
                    "start_date",
                    "end_date",
                    "order_number",
                    "project_num",
                    "plant_type",
                    "construction_type",
                ],
            },
            {
                "invalid_values": INVALID_PLANT_NAMES,
                "required_valid_cols": ["plant_name_ferc1"],
            },
        ],
    },
    "core_ferc1__yearly_small_plants_sched410": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "plant_name": "plant_name_ferc1",
                    "yr_constructed": "construction_year",
                    "capacity_rating": "capacity_mw",
                    "net_demand": "peak_demand_mw",
                    "net_generation": "net_generation_kwh",
                    "plant_cost": "capex_total",
                    "plant_cost_mw": "capex_per_mw",
                    "operation": "opex_operations",
                    "expns_fuel": "opex_fuel",
                    "expns_maint": "opex_maintenance",
                    "kind_of_fuel": "fuel_type",
                    "fuel_cost": "fuel_cost_per_mmbtu",
                    "report_year": "report_year",
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                }
            },
            "xbrl": {
                "columns": {
                    "report_year": "report_year",
                    "entity_id": "utility_id_ferc1_xbrl",
                    "plant_name": "plant_name_ferc1",
                    "start_date": "start_date",
                    "end_date": "end_date",
                    "date": "date",
                    "generation_type": "plant_type",
                    "fuel_cost_per_mmbtus": "fuel_cost_per_mmbtu",
                    "fuel_kind": "fuel_type",
                    "year_plant_originally_constructed": "construction_year",
                    "maintenance_production_expenses": "opex_maintenance",
                    "operating_expenses_excluding_fuel": "opex_operations",
                    "fuel_production_expenses": "opex_fuel",
                    "order_number": "order_number",
                    "net_generation_excluding_plant_use": "net_generation_kwh",
                    "net_peak_demand_on_plant": "peak_demand_mw",
                    "plant_cost_per_mw": "capex_per_mw",
                    "installed_capacity_of_plant": "capacity_mw",
                    "cost_of_plant": "capex_total",
                    "generating_plant_statistics_axis": "generating_plant_statistics",
                },
            },
        },
        "normalize_strings": {
            "plant_name_ferc1": FERC1_STRING_NORM,
            "fuel_type": FERC1_STRING_NORM,
            "plant_type": FERC1_STRING_NORM,
        },
        "nullify_outliers": {
            "construction_year": VALID_PLANT_YEARS,
        },
        "convert_units": {
            "fuel_cost_per_mmbtu": CENTS_PERMMBTU_TO_USD_PERMMBTU,
            "net_generation_kwh": KWH_TO_MWH,
        },
        "categorize_strings": {
            "fuel_type": FUEL_CATEGORIES,
            "fuel_type_from_header": FUEL_CATEGORIES,
            "plant_type": PLANT_TYPE_CATEGORIES,
            "plant_type_from_header": PLANT_TYPE_CATEGORIES,
        },
        "drop_invalid_rows": [
            {"invalid_values": ["header", "note"], "required_valid_cols": ["row_type"]},
            {
                "invalid_values": [0, "0", pd.NA, np.nan, "", "none"],
                "allowed_invalid_cols": [
                    "record_id",
                    "utility_id_ferc1",
                    "utility_id_ferc1_dbf",
                    "utility_id_ferc1_xbrl",
                    "plant_name_ferc1",
                    "generating_plant_statistics",
                    "report_year",
                    "construction_year",
                    "date",
                    "start_date",
                    "end_date",
                    "order_number",
                    "plant_type",
                    "fuel_type",
                    "license_id_ferc1",
                    "notes",
                    "row_type",
                ],
            },
            {
                "invalid_values": INVALID_PLANT_NAMES,
                "required_valid_cols": ["plant_name_ferc1"],
            },
        ],
    },
    "core_ferc1__yearly_plant_in_service_sched204": {
        "rename_columns_ferc1": {
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "ferc_account_label",
                    "starting_balance": "starting_balance",
                    "additions": "additions",
                    "retirements": "retirements",
                    "adjustments": "adjustments",
                    "transfers": "transfers",
                    "ending_balance": "ending_balance",
                }
            },
            "dbf": {
                "columns": {
                    "report_year": "report_year",
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                    "begin_yr_bal": "starting_balance",
                    "addition": "additions",
                    "retirements": "retirements",
                    "adjustments": "adjustments",
                    "transfers": "transfers",
                    "yr_end_bal": "ending_balance",
                    "xbrl_factoid": "ferc_account_label",
                }
            },
            "instant_xbrl": {
                "columns": {
                    "communication_equipment": "communication_equipment_general_plant",
                    "office_furniture_and_equipment": "office_furniture_and_equipment_general_plant",
                    "structures_and_improvements": "structures_and_improvements_general_plant",
                    "tools_shop_and_garage_equipment": "tools_shop_and_garage_equipment_general_plant",
                    "miscellaneous_equipment": "miscellaneous_equipment_general_plant",
                    "transportation_equipment": "transportation_equipment_general_plant",
                    "laboratory_equipment": "laboratory_equipment_general_plant",
                    "land_and_land_rights": "land_and_land_rights_general_plant",
                    "stores_equipment": "stores_equipment_general_plant",
                    "power_operated_equipment": "power_operated_equipment_general_plant",
                    "other_tangible_property": "other_tangible_property_general_plant",
                    "installations_on_customers_premises_distribution_plant": "installations_on_customer_premises_distribution_plant",
                    "fuel_holders_producers_and_accessories_other_production": "fuel_holders_products_and_accessories_other_production",
                    "structures_and_improvement_nuclear_production": "structures_and_improvements_nuclear_production",
                    "leased_property_on_customers_premises_distribution_plant": "leased_property_on_customer_premises_distribution_plant",
                }
            },
            "duration_xbrl": {
                "columns": {
                    "energy_storage_equipment_production_other_production_additions": "energy_storage_equipment_other_production_additions",
                    "energy_storage_equipment_production_other_production_adjustments": "energy_storage_equipment_other_production_adjustments",
                    "energy_storage_equipment_production_other_production_retirements": "energy_storage_equipment_other_production_retirements",
                    "energy_storage_equipment_production_other_production_transfers": "energy_storage_equipment_other_production_transfers",
                    "asset_retirement_costs_for_regional_transmission_and_market_operations_regional_transmission_and_market_operation_plant_additions": "asset_retirement_costs_for_regional_transmission_and_market_operation_plant_regional_transmission_and_market_operation_plant_additions",
                    "asset_retirement_costs_for_regional_transmission_and_market_operations_regional_transmission_and_market_operation_plant_adjustments": "asset_retirement_costs_for_regional_transmission_and_market_operation_plant_regional_transmission_and_market_operation_plant_adjustments",
                    "asset_retirement_costs_for_regional_transmission_and_market_operations_regional_transmission_and_market_operation_plant_retirements": "asset_retirement_costs_for_regional_transmission_and_market_operation_plant_regional_transmission_and_market_operation_plant_retirements",
                    "asset_retirement_costs_for_regional_transmission_and_market_operations_regional_transmission_and_market_operation_plant_transfers": "asset_retirement_costs_for_regional_transmission_and_market_operation_plant_regional_transmission_and_market_operation_plant_transfers",
                }
            },
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": ["entity_id", "report_year", "sched_table_name"],
                "value_types": [
                    "starting_balance",
                    "additions",
                    "retirements",
                    "transfers",
                    "adjustments",
                    "ending_balance",
                ],
                "expected_drop_cols": 2,
                "stacked_column_name": "xbrl_factoid",
            }
        },
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "electric", "is_dimension": True},
                "plant_status": {"column_value": "in_service", "is_dimension": True},
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "ferc_account_label"},
            "on": "ferc_account_label",
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_plant_in_srvce"]},
        # Known issue with reporting of construction in progress not classified in classified fields of table.
        "reconcile_table_calculations": {
            "column_to_check": "ending_balance",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {"error_frequency": 0.08},
                    "utility_id_ferc1": {
                        "error_frequency": 0.038,
                        "relative_error_magnitude": 0.51,  # worst utility: 444
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_pumped_storage_plants_sched408": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "report_year": "report_year",
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                    "plant_name": "plant_name_ferc1",
                    "tot_capacity": "capacity_mw",
                    "project_no": "project_num",
                    "plant_kind": "construction_type",
                    "peak_demand": "peak_demand_mw",
                    "yr_const": "construction_year",
                    "yr_installed": "installation_year",
                    "plant_hours": "plant_hours_connected_while_generating",
                    "plant_capability": "plant_capability_mw",
                    "avg_num_of_emp": "avg_num_employees",
                    "cost_wheels": "capex_wheels_turbines_generators",
                    "cost_land": "capex_land",
                    "cost_structures": "capex_structures",
                    "cost_facilties": "capex_facilities",
                    "cost_electric": "capex_equipment_electric",
                    "cost_misc_eqpmnt": "capex_equipment_misc",
                    "cost_roads": "capex_roads",
                    "asset_retire_cost": "asset_retirement_cost",
                    "cost_of_plant": "capex_total",
                    "cost_per_kw": "capex_per_kw",
                    "expns_operations": "opex_operations",
                    "expns_water_pwr": "opex_water_for_power",
                    "expns_pump_strg": "opex_pumped_storage",
                    "expns_electric": "opex_electric",
                    "expns_misc_power": "opex_generation_misc",
                    "expns_rents": "opex_rents",
                    "expns_engneering": "opex_engineering",
                    "expns_structures": "opex_structures",
                    "expns_dams": "opex_dams",
                    "expns_plant": "opex_plant",
                    "expns_misc_plnt": "opex_misc_plant",
                    "expns_producton": "opex_production_before_pumping",
                    "expns_kwh": "opex_per_kwh",
                    "pumping_expenses": "opex_pumping",
                    "tot_prdctn_exns": "opex_total",
                    "energy_used": "energy_used_for_pumping_kwh",
                    "net_load": "net_load_kwh",
                    "net_generation": "net_generation_kwh",
                },
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "plant_name": "plant_name_ferc1",
                    "order_number": "order_number",
                    "report_year": "report_year",
                    "date": "date",
                    "start_date": "start_date",
                    "end_date": "end_date",
                    "asset_retirement_costs_pumped_storage_plant": "asset_retirement_cost",
                    "cost_of_accessory_electric_equipment_pumped_storage_plant": "capex_equipment_electric",
                    "cost_of_land_and_land_rights_pumped_storage_plant": "capex_land",
                    "cost_of_miscellaneous_power_plant_equipment_pumped_storage_plant": "capex_equipment_misc",
                    "cost_of_plant": "capex_total",
                    "cost_of_reservoirs_dams_and_waterways_pumped_storage_plant": "capex_facilities",
                    "cost_of_roads_railroads_and_bridges_pumped_storage_plant": "capex_roads",
                    "cost_of_structures_and_improvements_pumped_storage_plant": "capex_structures",
                    "cost_of_water_wheels_turbines_and_generators_pumped_storage_plant": "capex_wheels_turbines_generators",
                    "cost_per_kilowatt_of_installed_capacity": "capex_per_kw",
                    "electric_expenses_pumped_storage_plant": "opex_electric",
                    "energy_used_for_pumping": "energy_used_for_pumping_kwh",
                    "expenses_per_net_kilowatt_hour": "opex_per_kwh",
                    "expenses_per_net_kilowatt_hour_generation_and_pumping": "opex_generation_and_pumping_per_kwh",
                    "installed_capacity_of_plant": "capacity_mw",
                    "licensed_project_number": "project_num",
                    "licensed_project_number_and_pumped_storage_plant_name_axis": "project_num_and_plant_name_ferc1",
                    "maintenance_of_electric_plant_pumped_storage_plant": "opex_plant",
                    "maintenance_of_miscellaneous_pumped_storage_plant": "opex_misc_plant",
                    "maintenance_of_reservoirs_dams_and_waterways_pumped_storage_plant": "opex_dams",
                    "maintenance_of_structures_pumped_storage_plant": "opex_structures",
                    "maintenance_supervision_and_engineering_pumped_storage_plant": "opex_engineering",
                    "miscellaneous_pumped_storage_power_generation_expenses": "opex_generation_misc",
                    "net_continuous_plant_capability": "plant_capability_mw",
                    "net_generation_excluding_plant_use": "net_generation_kwh",
                    "net_output_for_load": "net_load_kwh",
                    "net_peak_demand_on_plant": "peak_demand_mw",
                    "operation_supervision_and_engineering_expense": "opex_operations",
                    "plant_average_number_of_employees": "avg_num_employees",
                    "plant_construction_type": "construction_type",
                    "plant_hours_connected_to_load": "plant_hours_connected_while_generating",
                    "power_production_expense_before_pumping_expenses": "opex_production_before_pumping",
                    "power_production_expenses_pumped_storage_plant": "opex_total",
                    "pumped_storage_expenses": "opex_pumped_storage",
                    "pumping_expenses": "opex_pumping",
                    "rents_pumped_storage_plant": "opex_rents",
                    "water_for_power": "opex_water_for_power",
                    "year_last_unit_of_plant_installed": "installation_year",
                    "year_plant_originally_constructed": "construction_year",
                },
            },
        },
        "normalize_strings": {
            "plant_name_ferc1": FERC1_STRING_NORM,
            "construction_type": FERC1_STRING_NORM,
        },
        "nullify_outliers": {
            "construction_year": VALID_PLANT_YEARS,
            "installation_year": VALID_PLANT_YEARS,
        },
        "categorize_strings": {
            "construction_type": CONSTRUCTION_TYPE_CATEGORIES,
        },
        "convert_units": {
            "capex_per_kw": PERKW_TO_PERMW,
            "opex_per_kwh": PERKWH_TO_PERMWH,
            "opex_generation_and_pumping_per_kwh": PERKWH_TO_PERMWH,
            "net_generation_kwh": KWH_TO_MWH,
            "net_load_kwh": KWH_TO_MWH,
            "energy_used_for_pumping_kwh": KWH_TO_MWH,
        },
        "strip_non_numeric_values": {"project_num": {"strip_non_numeric_values": True}},
        "drop_invalid_rows": [
            {
                "invalid_values": [0, "0", pd.NA, np.nan, "", "none", "—"],
                "allowed_invalid_cols": [
                    "record_id",
                    "utility_id_ferc1",
                    "utility_id_ferc1_dbf",
                    "utility_id_ferc1_xbrl",
                    "plant_name_ferc1",
                    "report_year",
                    "date",
                    "start_date",
                    "end_date",
                    "project_num",
                    "construction_type",
                ],
            },
            {
                "invalid_values": INVALID_PLANT_NAMES,
                "required_valid_cols": ["plant_name_ferc1"],
            },
        ],
    },
    "core_ferc1__yearly_purchased_power_and_exchanges_sched326": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "report_prd": "report_prd",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "spplmnt_num": "spplmnt_num",
                    "report_year": "report_year",
                    "respondent_id": "utility_id_ferc1_dbf",
                    "athrty_co_name": "seller_name",
                    "sttstcl_clssfctn": "purchase_type_code",
                    "rtsched_trffnbr": "tariff",
                    "avgmth_bill_dmnd": "billing_demand_mw",
                    "avgmth_ncp_dmnd": "non_coincident_peak_demand_mw",
                    "avgmth_cp_dmnd": "coincident_peak_demand_mw",
                    "mwh_purchased": "purchased_mwh",
                    "mwh_recv": "received_mwh",
                    "mwh_delvd": "delivered_mwh",
                    "dmnd_charges": "demand_charges",
                    "erg_charges": "energy_charges",
                    "othr_charges": "other_charges",
                    "settlement_tot": "total_settlement",
                }
            },
            "xbrl": {
                "columns": {
                    "end_date": "end_date",
                    "start_date": "start_date",
                    "report_year": "report_year",
                    "order_number": "order_number",
                    "entity_id": "utility_id_ferc1_xbrl",
                    "average_monthly_billing_demand": "billing_demand_mw",
                    "average_monthly_coincident_peak_demand": "coincident_peak_demand_mw",
                    "average_monthly_non_coincident_peak_demand": "non_coincident_peak_demand_mw",
                    "demand_charges_of_purchased_power": "demand_charges",
                    "energy_charges_of_purchased_power": "energy_charges",
                    "energy_delivered_through_power_exchanges": "delivered_mwh",
                    "energy_received_through_power_exchanges": "received_mwh",
                    "megawatt_hours_purchased_for_energy_storage": "purchased_storage_mwh",
                    "megawatt_hours_purchased_other_than_storage": "purchased_other_than_storage_mwh",
                    "name_of_company_or_public_authority_providing_purchased_power": "seller_name",
                    "other_charges_of_purchased_power": "other_charges",
                    "purchased_power_axis": "purchased_power_axis",  # new probably pk col
                    "rate_schedule_tariff_number": "tariff",
                    "settlement_of_power": "total_settlement",
                    "statistical_classification_code": "purchase_type_code",
                }
            },
        },
        "normalize_strings": {
            "seller_name": FERC1_STRING_NORM,
        },
        "strip_non_numeric_values": {
            "billing_demand_mw": {"strip_non_numeric_values": True},
            "non_coincident_peak_demand_mw": {"strip_non_numeric_values": True},
            "coincident_peak_demand_mw": {"strip_non_numeric_values": True},
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [0, "0", pd.NA, np.nan, "", "none", "—"],
                "required_valid_cols": [
                    "purchased_mwh",
                    "received_mwh",
                    "delivered_mwh",
                    "demand_charges",
                    "energy_charges",
                    "other_charges",
                    "total_settlement",
                    "purchased_storage_mwh",
                    "purchased_other_than_storage_mwh",
                    "billing_demand_mw",
                    "non_coincident_peak_demand_mw",
                    "coincident_peak_demand_mw",
                ],
            }
        ],
    },
    "core_ferc1__yearly_transmission_lines_sched422": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "designation_from": "start_point",
                    "designation_to": "end_point",
                    "voltage_oper": "operating_voltage_kv",
                    "designed_voltage": "designed_voltage_kv",
                    "structure": "supporting_structure_type",
                    "length_dsgnt": "transmission_line_length_miles",
                    "length_another": "transmission_line_and_structures_length_miles",
                    "num_of_circuits": "num_transmission_circuits",
                    "conductor_size": "conductor_size_and_material",
                    "cost_land": "capex_land",
                    "cost_other": "capex_other",
                    "cost_total": "capex_total",
                    "expns_operations": "opex_operations",
                    "expns_maint": "opex_maintenance",
                    "expns_rents": "opex_rents",
                    "expns_total": "opex_total",
                    "report_prd": "report_prd",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "start_date": "start_date",
                    "end_date": "end_date",
                    "transmission_line_statistics_axis": "transmission_line_statistics_axis",
                    "length_for_transmission_lines_aggregated_with_other_structures": "transmission_line_and_structures_length_miles",
                    "overall_cost_of_transmission_line": "capex_total",
                    "number_of_transmission_circuits": "num_transmission_circuits",
                    "maintenance_expenses_of_transmission_line": "opex_maintenance",
                    "cost_of_land_and_land_rights_transmission_lines": "capex_land",
                    "operating_expenses_of_transmission_line": "opex_operations",
                    "designed_voltage_of_transmission_line": "designed_voltage_kv",
                    "supporting_structure_of_transmission_line_type": "supporting_structure_type",
                    "construction_and_other_costs_transmission_lines": "capex_other",
                    "operating_voltage_of_transmission_line": "operating_voltage_kv",
                    "transmission_line_end_point": "end_point",
                    "order_number": "order_number",
                    "size_of_conductor_and_material": "conductor_size_and_material",
                    "transmission_line_start_point": "start_point",
                    "rent_expenses_of_transmission_line": "opex_rents",
                    "length_for_stand_alone_transmission_lines": "transmission_line_length_miles",
                    "overall_expenses_of_transmission_line": "opex_total",
                    "report_year": "report_year",
                }
            },
        },
        "replace_with_na": {
            "conductor_size_and_material": {"replace_with_na": [""]},
            "supporting_structure_type": {"replace_with_na": [""]},
            "start_point": {"replace_with_na": [""]},
            "end_point": {"replace_with_na": [""]},
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [pd.NA, np.nan, "", None, "None"],
                "required_valid_cols": [
                    "operating_voltage_kv",
                    "designed_voltage_kv",
                    "transmission_line_length_miles",
                    "transmission_line_and_structures_length_miles",
                    "num_transmission_circuits",
                    "capex_land",
                    "capex_other",
                    "capex_total",
                    "opex_operations",
                    "opex_maintenance",
                    "opex_rents",
                    "opex_total",
                ],
            }
        ],
    },
    "core_ferc1__yearly_energy_sources_sched401": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "erg_src_mwh": "energy_mwh",
                    "erg_disp_mwh": "energy_disposition_mwh",
                    "report_prd": "report_prd",
                    "xbrl_factoid": "energy_source_type",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "energy_source_type",
                    "energy_source_mwh": "energy_mwh",
                }
            },
            "duration_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_energy_source_mwh"
                    for xbrl_col in [
                        # generation
                        "steam_generation",
                        "nuclear_generation",
                        "hydro_conventional_generation",
                        "hydro_pumped_storage_generation",
                        "other_energy_generation",
                        "pumping_energy",
                        "net_energy_generation",
                        "megawatt_hours_purchased_other_than_storage",
                        "megawatt_hours_purchased_for_energy_storage",
                        # exchanges
                        "energy_received_through_power_exchanges",
                        "energy_delivered_through_power_exchanges",
                        "net_energy_through_power_exchanges",
                        # transmission
                        "electric_power_wheeling_energy_received",
                        "electric_power_wheeling_energy_delivered",
                        "net_transmission_energy_for_others_electric_power_wheeling",
                        "transmission_losses_by_others_electric_power_wheeling",
                        # total
                        "sources_of_energy",
                    ]
                }
            },
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [pd.NA, np.nan, ""],
                "required_valid_cols": ["energy_mwh"],
            },
        ],
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": ["energy_source_mwh"],
                "expected_drop_cols": 10,
                "stacked_column_name": "xbrl_factoid",
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "energy_source_type"},
            "on": "energy_source_type",
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_elctrc_erg_acct"]},
    },
    "core_ferc1__yearly_energy_dispositions_sched401": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "erg_src_mwh": "energy_source_mwh",  # drop
                    "erg_disp_mwh": "energy_mwh",
                    "report_prd": "report_prd",
                    "xbrl_factoid": "energy_disposition_type",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "energy_disposition_type",
                    "energy_disposition_mwh": "energy_mwh",
                }
            },
            "duration_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_energy_disposition_mwh"
                    for xbrl_col in [
                        "energy_stored",
                        "energy_losses",
                        "megawatt_hours_sold_non_requirements_sales",
                        "megawatt_hours_sold_sales_to_ultimate_consumers",
                        "internal_use_energy",
                        "non_charged_energy",
                        "megawatt_hours_sold_requirements_sales",
                        # total
                        "disposition_of_energy",
                    ]
                }
            },
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [pd.NA, np.nan, ""],
                "required_valid_cols": ["energy_mwh"],
            },
        ],
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": ["energy_disposition_mwh"],
                "expected_drop_cols": 19,
                "stacked_column_name": "xbrl_factoid",
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "energy_disposition_type"},
            "on": "energy_disposition_type",
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_elctrc_erg_acct"]},
    },
    "core_ferc1__yearly_utility_plant_summary_sched200": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "other": "utility_type_other",
                    "xbrl_factoid": "utility_type",
                    # util plant
                    # in service
                    "service_plant": "utility_plant_in_service_classified_ending_balance",
                    "propundr_cptl_ls": "utility_plant_in_service_property_under_capital_leases_ending_balance",
                    "plnt_prchs_sold": "utility_plant_in_service_plant_purchased_or_sold_ending_balance",
                    "cmplt_const_ucls": "utility_plant_in_service_completed_construction_not_classified_ending_balance",
                    "xprmnt_plnt_ucls": "utility_plant_in_service_experimental_plant_unclassified_ending_balance",
                    "in_srvc_total": "utility_plant_in_service_classified_and_unclassified_ending_balance",
                    # rest of util plant
                    "leased_to_others": "utility_plant_leased_to_others_ending_balance",
                    "held_ftre_use": "utility_plant_held_for_future_use_ending_balance",
                    "const_wrk_prgrs": "construction_work_in_progress_ending_balance",
                    "acqstn_adjstmnt": "utility_plant_acquisition_adjustment_ending_balance",
                    "tot_utlty_plant": "utility_plant_and_construction_work_in_progress_ending_balance",
                    "accum_prvsn_dad": "accumulated_provision_for_depreciation_amortization_and_depletion_of_plant_utility_reported_ending_balance",
                    "net_utlty_plant": "utility_plant_net_ending_balance",
                    # detail of accum deprish
                    # in service
                    "in_srvce_depr": "depreciation_utility_plant_in_service_ending_balance",
                    "amrtzd_dplt_nglr": "amortization_and_depletion_of_producing_natural_gas_land_and_land_rights_utility_plant_in_service_ending_balance",
                    "amrtzd_ugrndstrg": "amortization_of_underground_storage_land_and_land_rights_utility_plant_in_service_ending_balance",
                    "amrtz_utlty_plnt": "amortization_of_other_utility_plant_utility_plant_in_service_ending_balance",
                    "tot_in_service": "depreciation_amortization_and_depletion_utility_plant_in_service_ending_balance",
                    # leased to others
                    "leased_othr_depr": "depreciation_utility_plant_leased_to_others_ending_balance",
                    "amrtz_dplt": "amortization_and_depletion_utility_plant_leased_to_others_ending_balance",
                    "tot_leased_othr": "depreciation_amortization_and_depletion_utility_plant_leased_to_others_ending_balance",
                    # held for future use
                    "depr_ftre_use": "depreciation_utility_plant_held_for_future_use_ending_balance",
                    "amortization": "amortization_utility_plant_held_for_future_use_ending_balance",
                    "total_ftre_use": "depreciation_and_amortization_utility_plant_held_for_future_use_ending_balance",
                    # rest of details of acum deprish
                    "abndn_leases": "abandonment_of_leases_ending_balance",
                    "amrtzplnt_acqstn": "amortization_of_plant_acquisition_adjustment_ending_balance",
                    "tot_accum_prvsn": "accumulated_provision_for_depreciation_amortization_and_depletion_of_plant_utility_ending_balance",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "utility_type_axis": "utility_type",
                    "xbrl_factoid": "utility_plant_asset_type",
                }
            },
            "instant_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_ending_balance"
                    for xbrl_col in [
                        "depreciation_amortization_and_depletion_utility_plant_in_service",
                        "depreciation_and_amortization_utility_plant_held_for_future_use",
                        "amortization_of_plant_acquisition_adjustment",
                        "depreciation_utility_plant_in_service",
                        "utility_plant_in_service_classified_and_unclassified",
                        "utility_plant_and_construction_work_in_progress",
                        "depreciation_utility_plant_leased_to_others",
                        "utility_plant_in_service_classified",
                        "depreciation_amortization_and_depletion_utility_plant_leased_to_others",
                        "depreciation_utility_plant_held_for_future_use",
                        "abandonment_of_leases",
                        "utility_plant_net",
                        "amortization_utility_plant_held_for_future_use",
                        "amortization_and_depletion_utility_plant_leased_to_others",
                        "accumulated_provision_for_depreciation_amortization_and_depletion_of_plant_utility",
                        "utility_plant_in_service_property_under_capital_leases",
                        "utility_plant_acquisition_adjustment",
                        "utility_plant_leased_to_others",
                        "utility_plant_held_for_future_use",
                        "amortization_of_other_utility_plant_utility_plant_in_service",
                        "utility_plant_in_service_completed_construction_not_classified",
                        "utility_plant_in_service_plant_purchased_or_sold",
                        "construction_work_in_progress",
                        "utility_plant_in_service_experimental_plant_unclassified",
                    ]
                }
                | {
                    "amortization_and_depletion_of_producing_natural_gas_land_and_land_rightsutility_plant_in_service": "amortization_and_depletion_of_producing_natural_gas_land_and_land_rights_utility_plant_in_service_ending_balance",
                    "amortization_of_underground_storage_land_and_land_rightsutility_plant_in_service": "amortization_of_underground_storage_land_and_land_rights_utility_plant_in_service_ending_balance",
                }
            },
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [pd.NA, np.nan, ""],
                "required_valid_cols": ["ending_balance"],
            },
        ],
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "utility_type_axis",
                    "sched_table_name",
                ],
                "value_types": ["ending_balance"],
                "expected_drop_cols": 1,
                "stacked_column_name": "xbrl_factoid",
            },
            "dbf": {
                "idx_cols": [
                    "report_year",
                    "record_id",
                    "utility_id_ferc1",
                    "utility_type",
                    "utility_type_other",
                ],
                "value_types": ["ending_balance"],
                "expected_drop_cols": 1,
                "stacked_column_name": "utility_plant_asset_type",
            },
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "utility_plant_asset_type"},
            "on": "utility_plant_asset_type",
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_utltyplnt_smmry"]},
        "categorize_strings": {
            "utility_type": {
                "categories": UTILITY_TYPE_CATEGORIES["categories"]
                | {
                    "other2": {"other2", "ferc:OtherUtility2Member"},
                    "other3": {"other3", "ferc:OtherUtility3Member"},
                }
            },
        },
        "normalize_strings": {
            "utility_type_other": FERC1_STRING_NORM,
        },
        "replace_with_na": {
            "utility_type_other": {"replace_with_na": [""]},
        },
        "reconcile_table_calculations": {
            "column_to_check": "ending_balance",
            "subdimension_column": "utility_type",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {
                        "error_frequency": 0.016,
                        "relative_error_magnitude": 0.039,
                    },
                    "report_year": {
                        "error_frequency": 0.28,  # worst year: 2021
                        "null_calculated_value_frequency": 1.0,
                        "relative_error_magnitude": 0.47,  # worst year: 2021
                    },
                    "xbrl_factoid": {
                        "error_frequency": 0.16,
                        "relative_error_magnitude": 0.2,
                        "null_calculated_value_frequency": 1.0,
                    },  # worst xbrl_factoid: utility_plant_and_construction_work_in_progress
                    "utility_id_ferc1": {
                        "error_frequency": 0.28,  # worst utility: 444
                        "relative_error_magnitude": 0.21,
                        "null_calculated_value_frequency": 1.0,
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_balance_sheet_assets_sched110": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "xbrl_factoid": "asset_type",
                    "begin_yr_balance": "ending_balance",
                    "end_yr_balance": "starting_balance",
                    "end_qtr_bal": "end_qtr_bal",
                    "pri_yr_q4_bal": "pri_yr_q4_bal",
                }
            },
            "instant_xbrl": {
                "columns": {
                    "preliminary_natural_gas_survey_and_investigation_charges_and_other_preliminary_survey_and_investigation_charges": "preliminary_natural_gas_survey_and_investigation_charges"
                },
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "asset_type",
                }
            },
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": ["starting_balance", "ending_balance"],
                "expected_drop_cols": 0,
                "stacked_column_name": "xbrl_factoid",
            },
        },
        "drop_duplicate_rows_dbf": {
            "data_columns": ["ending_balance", "starting_balance"],
            "table_name": "core_ferc1__yearly_balance_sheet_assets_sched110",
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_comp_balance_db"]},
        "assign_quarterly_data_to_yearly_dbf": {
            "quarterly_to_yearly_column_map": {
                "pri_yr_q4_bal": "starting_balance",
                "end_qtr_bal": "ending_balance",
            },
            "quarterly_filed_years": range(2005, 2021),
        },
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "total", "is_dimension": True}
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "asset_type"},
            "on": "asset_type",
        },
        "reconcile_table_calculations": {
            "column_to_check": "ending_balance",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {"error_frequency": 0.00042},
                },
            },
        },
    },
    "core_ferc1__yearly_balance_sheet_liabilities_sched110": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "begin_yr_balance": "starting_balance",
                    "end_yr_balance": "ending_balance",
                    "report_prd": "report_prd",
                    "end_qtr_bal": "end_qtr_bal",
                    "pri_yr_q4_bal": "pri_yr_q4_bal",
                    "xbrl_factoid": "liability_type",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "liability_type",
                }
            },
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": ["starting_balance", "ending_balance"],
                "expected_drop_cols": 0,
                "stacked_column_name": "xbrl_factoid",
            }
        },
        "drop_duplicate_rows_dbf": {
            "data_columns": ["ending_balance", "starting_balance"],
            "table_name": "core_ferc1__yearly_balance_sheet_liabilities_sched110",
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_bal_sheet_cr"]},
        "assign_quarterly_data_to_yearly_dbf": {
            "quarterly_to_yearly_column_map": {
                "pri_yr_q4_bal": "starting_balance",
                "end_qtr_bal": "ending_balance",
            },
            "quarterly_filed_years": range(2005, 2021),
        },
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "total", "is_dimension": True}
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "liability_type"},
            "on": "liability_type",
        },
        "reconcile_table_calculations": {
            "column_to_check": "ending_balance",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {
                        "error_frequency": 0.0022,
                        "relative_error_magnitude": 0.016,
                    },
                    "report_year": {
                        "error_frequency": 0.03,
                        "relative_error_magnitude": 0.5,
                    },
                    "xbrl_factoid": {
                        "error_frequency": 0.069,
                        "null_calculated_value_frequency": 1.0,
                    },
                    "utility_id_ferc1": {
                        "error_frequency": 0.077,
                        "relative_error_magnitude": 0.51,
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_depreciation_summary_sched336": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "depr_expn": "depreciation_expense_dollar_value",
                    "depr_asset_retire": "depreciation_expense_asset_retirement_dollar_value",
                    "limterm_elc_plnt": "amortization_limited_term_electric_plant_dollar_value",
                    "othr_elc_plnt": "amortization_other_electric_plant_dollar_value",
                    "total": "depreciation_amortization_total_dollar_value",
                    "xbrl_factoid": "plant_function",
                    "report_prd": "report_prd",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "ferc_account_label",
                }
            },
            "duration_xbrl": {
                "columns": {
                    "functional_classification_axis": "plant_function",
                    "depreciation_expense_excluding_amortization_of_acquisition_adjustments": "depreciation_expense_dollar_value",
                    "depreciation_expense_for_asset_retirement_costs_excluding_amortizationg_of_acquisition_adjustments": "depreciation_expense_asset_retirement_dollar_value",
                    "amortization_of_limited_term_plant_or_property": "amortization_limited_term_electric_plant_dollar_value",
                    "amortization_of_other_electric_plant": "amortization_other_electric_plant_dollar_value",
                    "depreciation_and_amortization": "depreciation_amortization_total_dollar_value",
                }
            },
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "plant_function",
                    "sched_table_name",
                ],
                "value_types": [
                    "dollar_value",
                ],
                "expected_drop_cols": 3,
                "stacked_column_name": "xbrl_factoid",
            },
            "dbf": {
                "idx_cols": [
                    "report_year",
                    "record_id",
                    "utility_id_ferc1",
                    "plant_function",
                ],
                "value_types": ["dollar_value"],
                "expected_drop_cols": 1,
                "stacked_column_name": "ferc_account_label",
            },
        },
        "categorize_strings": {"plant_function": PLANT_FUNCTION_CATEGORIES},
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_dacs_epda"]},
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "electric", "is_dimension": True}
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "ferc_account_label"},
            "on": "ferc_account_label",
        },
        "reconcile_table_calculations": {
            "column_to_check": "dollar_value",
            "subdimension_column": "plant_function",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {"null_calculated_value_frequency": 0.84},
                    "report_year": {"null_calculated_value_frequency": 1.0},
                    "utility_id_ferc1": {
                        "error_frequency": 0.23,
                        "relative_error_magnitude": 1.0,
                        "null_calculated_value_frequency": 1.0,
                    },  # worst utility: 331
                },
            },
        },
    },
    "core_ferc1__yearly_operating_revenues_sched300": {
        "rename_columns_ferc1": {
            "duration_xbrl": {
                "columns": {
                    f"megawatt_hours_sold_{col}": f"{col}_sales_mwh"
                    for col in [
                        "residential_sales",
                        "small_or_commercial",
                        "large_or_industrial",
                        "public_street_and_highway_lighting",
                        "other_sales_to_public_authorities",
                        "sales_to_railroads_and_railways",
                        "interdepartmental_sales",
                        "sales_to_ultimate_consumers",
                        "sales_for_resale",
                        "sales_of_electricity",
                        "provision_for_rate_refunds",
                        "revenues_net_of_provision_for_refunds",
                    ]
                }
                | {
                    f"average_number_of_customers_per_month_{col}": f"{col}_avg_customers_per_month"
                    for col in [
                        "residential_sales",
                        "small_or_commercial",
                        "large_or_industrial",
                        "public_street_and_highway_lighting",
                        "other_sales_to_public_authorities",
                        "sales_to_railroads_and_railways",
                        "interdepartmental_sales",
                        "sales_to_ultimate_consumers",
                        "sales_for_resale",
                        "sales_of_electricity",
                        "provision_for_rate_refunds",
                        "revenues_net_of_provision_for_refunds",
                    ]
                }
                | {
                    col: f"{col}_electric_operating_revenue"
                    for col in [
                        "residential_sales",
                        # Note: the small_or_commercial/large_or_industrial columns are already prefixed
                        "public_street_and_highway_lighting",
                        "other_sales_to_public_authorities",
                        "sales_to_railroads_and_railways",
                        "interdepartmental_sales",
                        "sales_to_ultimate_consumers",
                        "sales_for_resale",
                        "sales_of_electricity",
                        "provision_for_rate_refunds",
                        "revenues_net_of_provision_for_refunds",
                        "forfeited_discounts",
                        "miscellaneous_service_revenues",
                        "sales_of_water_and_water_power",
                        "rent_from_electric_property",
                        "interdepartmental_rents",
                        "other_electric_revenue",
                        "revenues_from_transmission_of_electricity_of_others",
                        "regional_transmission_service_revenues",
                        "miscellaneous_revenue",
                        "other_miscellaneous_operating_revenues",
                        "other_operating_revenues",
                        "electric_operating_revenues",
                    ]
                }
                | {
                    f"{col}_sales_electric_operating_revenue": f"{col}_electric_operating_revenue"
                    for col in ["small_or_commercial", "large_or_industrial"]
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "electric_operating_revenue": "dollar_value",
                }
            },
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "xbrl_factoid": "revenue_type",
                    "rev_amt_crnt_yr": "dollar_value",
                    "mwh_sold_crnt_yr": "sales_mwh",
                    "avg_cstmr_crntyr": "avg_customers_per_month",
                }
            },
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": [
                    "electric_operating_revenue",
                    "sales_mwh",
                    "avg_customers_per_month",
                ],
                "expected_drop_cols": 2,
                "stacked_column_name": "revenue_type",
            },
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_elctrc_oper_rev"]},
        "select_dbf_rows_by_category": {
            "column_name": "revenue_type",
            "select_by_xbrl_categories": True,
            "len_expected_categories_to_drop": 4,
        },
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "electric", "is_dimension": True}
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "revenue_type"},
            "on": "revenue_type",
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [pd.NA, np.nan, ""],
                "required_valid_cols": [
                    "dollar_value",
                    "sales_mwh",
                    "avg_customers_per_month",
                ],
            },
        ],
        "reconcile_table_calculations": {
            "column_to_check": "dollar_value",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {"error_frequency": 0.0031},
                    "xbrl_factoid": {
                        "error_frequency": 0.04,
                        "null_calculated_value_frequency": 1.0,
                        "relative_error_magnitude": 0.03,
                    },  # worst xbrl_factoid: other_operating_revenues
                    "utility_id_ferc1": {
                        "error_frequency": 0.29,
                        "relative_error_magnitude": 0.51,
                        "null_calculated_value_frequency": 1.0,
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_retained_earnings_sched118": {
        "rename_columns_ferc1": {
            "duration_xbrl": {
                "columns": {
                    "retained_earnings_primary_contra_account_affected": "retained_earnings_contra_primary_account_affected"
                }
                | {
                    col: f"{col}_contra_primary_account_affected"
                    for col in [
                        "adjustments_to_retained_earnings_debit",
                        "balance_transferred_from_income",
                        "appropriations_of_retained_earnings",
                        "adjustments_to_retained_earnings_credit",
                        "dividends_declared_common_stock",
                        "dividends_declared_preferred_stock",
                        "changes_unappropriated_undistributed_subsidiary_earnings_credits",
                    ]
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "contra_primary_account_affected": "amount",
                }
            },
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "xbrl_factoid": "earnings_type",
                    "prev_balance": "starting_balance",
                }
            },
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": [
                    "starting_balance",
                    "ending_balance",
                    "contra_primary_account_affected",
                ],
                "expected_drop_cols": 2,
                "stacked_column_name": "earnings_type",
            },
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_retained_erng"]},
        "select_dbf_rows_by_category": {
            "column_name": "earnings_type",
            "select_by_xbrl_categories": True,
            "additional_categories": [
                "unappropriated_retained_earnings_previous_year",
                "unappropriated_undistributed_subsidiary_earnings_previous_year",
            ],
            "len_expected_categories_to_drop": 6,
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "total", "is_dimension": True}
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "earnings_type"},
            "on": "earnings_type",
        },
        "strip_non_numeric_values": {"amount": {"strip_non_numeric_values": True}},
        "reconcile_table_calculations": {
            "column_to_check": "ending_balance",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {"error_frequency": 0.08},
                    "report_year": {
                        "error_frequency": 0.09,
                        "relative_error_magnitude": 1.0,
                    },
                    "xbrl_factoid": {
                        "error_frequency": 0.034,
                        "relative_error_magnitude": 0.22,
                    },
                    "utility_id_ferc1": {
                        "error_frequency": 0.056,
                        "relative_error_magnitude": 0.26,
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_income_statements_sched114": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "xbrl_factoid": "income_type",
                    "current_yr_total": "total_dollar_value",
                    "cy_elctrc_total": "electric_dollar_value",
                    "cy_gas_total": "gas_dollar_value",
                    "cy_other_total_1": "other1_dollar_value",
                    "cy_other_total_2": "other2_dollar_value",
                    "cy_other_total_3": "other3_dollar_value",
                    "cy_other_total_4": "other4_dollar_value",
                    "cy_other_total": "other_total_dollar_value",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "utility_type_axis": "utility_type",
                    "xbrl_factoid": "income_type",
                }
            },
            "duration_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_dollar_value"
                    for xbrl_col in [
                        "accretion_expense",
                        "equity_in_earnings_of_subsidiary_companies",
                        "miscellaneous_nonoperating_income",
                        "amortization_of_conversion_expenses",
                        "regulatory_credits",
                        "expenditures_for_certain_civic_political_and_related_activities",
                        "income_taxes_other",
                        "other_interest_expense",
                        "provisions_for_deferred_income_taxes_utility_operating_income",
                        "losses_from_disposition_of_service_company_plant",
                        "gains_from_disposition_of_allowances",
                        "investment_tax_credit_adjustments_nonutility_operations",
                        "net_extraordinary_items",
                        "amortization_of_property_losses_unrecovered_plant_and_regulatory_study_costs",
                        "depreciation_expense_for_asset_retirement_costs",
                        "depreciation_expense",
                        "interest_on_debt_to_associated_companies",
                        "amortization_of_debt_discount_and_expense",
                        "amortization_of_electric_plant_acquisition_adjustments",
                        "provision_for_deferred_income_taxes_credit_operating_income",
                        "taxes_on_other_income_and_deductions",
                        "interest_on_long_term_debt",
                        "gains_from_disposition_of_plant",
                        "regulatory_debits",
                        "costs_and_expenses_of_merchandising_jobbing_and_contract_work",
                        "penalties",
                        "expenses_of_nonutility_operations",
                        "income_taxes_operating_income",
                        "allowance_for_other_funds_used_during_construction",
                        "other_income_deductions",
                        "amortization_and_depletion_of_utility_plant",
                        "investment_tax_credits",
                        "other_income",
                        "donations",
                        "investment_tax_credit_adjustments",
                        "allowance_for_borrowed_funds_used_during_construction_credit",
                        "losses_from_disposition_of_allowances",
                        "income_taxes_extraordinary_items",
                        "provision_for_deferred_income_taxes_other_income_and_deductions",
                        "miscellaneous_amortization",
                        "net_utility_operating_income",
                        "operation_expense",
                        "net_income_loss",
                        "income_taxes_utility_operating_income_other",
                        "nonoperating_rental_income",
                        "net_other_income_and_deductions",
                        "amortization_of_loss_on_reacquired_debt",
                        "amortization_of_gain_on_reacquired_debt_credit",
                        "loss_on_disposition_of_property",
                        "taxes_other_than_income_taxes_other_income_and_deductions",
                        "life_insurance",
                        "income_before_extraordinary_items",
                        "net_interest_charges",
                        "maintenance_expense",
                        "revenues_from_merchandising_jobbing_and_contract_work",
                        "other_deductions",
                        "operating_revenues",
                        "provision_for_deferred_income_taxes_credit_other_income_and_deductions",
                        "extraordinary_items_after_taxes",
                        "revenues_from_nonutility_operations",
                        "taxes_other_than_income_taxes_utility_operating_income",
                        "utility_operating_expenses",
                        "gain_on_disposition_of_property",
                        "extraordinary_deductions",
                        "interest_and_dividend_income",
                        "income_taxes_federal",
                        "extraordinary_income",
                        "amortization_of_premium_on_debt_credit",
                    ]
                }
            },
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "utility_type_axis",
                    "sched_table_name",
                ],
                "value_types": ["dollar_value"],
                "expected_drop_cols": 2,
                "stacked_column_name": "xbrl_factoid",
            },
            "dbf": {
                "idx_cols": [
                    "utility_id_ferc1",
                    "report_year",
                    "income_type",
                    "record_id",
                ],
                "value_types": ["dollar_value"],
                "expected_drop_cols": 11,
                "stacked_column_name": "utility_type",
            },
        },
        "drop_duplicate_rows_dbf": {
            "data_columns": ["dollar_value"],
            "table_name": "core_ferc1__yearly_income_statements_sched114",
        },
        "align_row_numbers_dbf": {
            "dbf_table_names": ["f1_income_stmnt", "f1_incm_stmnt_2"]
        },
        "categorize_strings": {
            "utility_type": {
                "categories": UTILITY_TYPE_CATEGORIES["categories"]
                | {
                    "total": {"total", "other_total"},
                    "other": {"other", "ferc:OtherUtilityMember"},
                    "other1": {"other1"},
                    "other2": {"other2"},
                    "other3": {"other3"},
                    "other4": {"other4"},
                }
            },
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "income_type"},
            "on": "income_type",
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [pd.NA, np.nan, ""],
                "required_valid_cols": ["dollar_value"],
            },
        ],
        "reconcile_table_calculations": {
            "column_to_check": "dollar_value",
            "subdimension_column": "utility_type",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {
                        "error_frequency": 0.023,
                        "null_calculated_value_frequency": 0.74,
                    },
                    "xbrl_factoid": {
                        "error_frequency": 0.51,
                        "relative_error_magnitude": 0.82,
                        "null_calculated_value_frequency": 1.0,
                    },  # worst xbrl_factoid: net_utility_operating_income
                    "report_year": {
                        "error_frequency": 0.026,
                        "null_calculated_value_frequency": 1.0,
                    },
                    "utility_id_ferc1": {
                        "error_frequency": 0.17,
                        "relative_error_magnitude": 0.89,
                        "null_calculated_value_frequency": 1.0,
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_depreciation_changes_sched219": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "total_cde": "total_dollar_value",
                    "future_plant": "future_dollar_value",
                    "leased_plant": "leased_dollar_value",
                    "electric_plant": "in_service_dollar_value",
                    "xbrl_factoid": "depreciation_type",
                }
            },
            "instant_xbrl": {
                "columns": {
                    "accumulated_provision_for_depreciation_of_electric_utility_plant_ending_balance": "ending_balance_dollar_value",
                    "accumulated_provision_for_depreciation_of_electric_utility_plant_starting_balance": "starting_balance_dollar_value",
                }
            },
            "duration_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_dollar_value"
                    for xbrl_col in [
                        "book_cost_of_asset_retirement_costs",
                        "book_cost_of_retired_plant",
                        "cost_of_removal_of_plant",
                        "depreciation_expense_excluding_adjustments",
                        "depreciation_expense_for_asset_retirement_costs",
                        "depreciation_provision",
                        "expenses_of_electric_plant_leased_to_others",
                        "net_charges_for_retired_plant",
                        "other_accounts",
                        "other_adjustments_to_accumulated_depreciation",
                        "other_clearing_accounts",
                        "salvage_value_of_retired_plant",
                        "transportation_expenses_clearing",
                    ]
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "electric_plant_classification_axis": "plant_status",
                    "utility_type_axis": "utility_type",
                }
            },
        },
        "categorize_strings": {
            "utility_type": UTILITY_TYPE_CATEGORIES,
            "plant_status": PLANT_STATUS,
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_accumdepr_prvsn"]},
        "wide_to_tidy": {
            "dbf": {
                "idx_cols": [
                    "utility_id_ferc1",
                    "report_year",
                    "depreciation_type",
                    "record_id",
                ],
                "value_types": ["dollar_value"],
                "expected_drop_cols": 2,
                "stacked_column_name": "plant_status",
            },
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "electric_plant_classification_axis",
                    "utility_type_axis",
                    "sched_table_name",
                ],
                "value_types": ["dollar_value"],
                "expected_drop_cols": 2,
                "stacked_column_name": "depreciation_type",
            },
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "depreciation_type"},
            "on": "depreciation_type",
        },
        "select_dbf_rows_by_category": {
            "column_name": "depreciation_type",
            "select_by_xbrl_categories": True,
            "len_expected_categories_to_drop": 12,
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "reconcile_table_calculations": {
            "column_to_check": "dollar_value",
            # Note: this table does not currently get exploded. It will require
            # additional debugging at a later date.
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {
                        "error_frequency": 0.06,
                        "relative_error_magnitude": 0.045,
                    },
                    "report_year": {
                        "error_frequency": 0.079,  # worst year: 2021
                        "relative_error_magnitude": 0.073,
                    },
                    "utility_id_ferc1": {
                        "error_frequency": 0.14,
                        "relative_error_magnitude": 0.66,
                    },
                    "xbrl_factoid": {
                        "error_frequency": 0.59,
                        "relative_error_magnitude": 0.1,
                    },  # worst xbrl_factoid: ending_balance
                },
            },
        },
    },
    "core_ferc1__yearly_depreciation_by_function_sched219": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "total_cde": "total_ending_balance",
                    "future_plant": "future_ending_balance",
                    "leased_plant": "leased_ending_balance",
                    "electric_plant": "in_service_ending_balance",
                    "xbrl_factoid": "plant_function",
                }
            },
            "instant_xbrl": {
                "columns": {
                    "accumulated_depreciation_steam_production_ending_balance": "steam_production_ending_balance",
                    "accumulated_depreciation_nuclear_production_ending_balance": "nuclear_production_ending_balance",
                    "accumulated_depreciation_hydraulic_production_conventional_ending_balance": "hydraulic_production_conventional_ending_balance",
                    "accumulated_depreciation_hydraulic_production_pumped_storage_ending_balance": "hydraulic_production_pumped_storage_ending_balance",
                    "accumulated_depreciation_other_production_ending_balance": "other_production_ending_balance",
                    "accumulated_depreciation_transmission_ending_balance": "transmission_ending_balance",
                    "accumulated_depreciation_distribution_ending_balance": "distribution_ending_balance",
                    "accumulated_depreciation_general_ending_balance": "general_ending_balance",
                    "accumulated_depreciation_regional_transmission_and_market_operation_ending_balance": "regional_transmission_and_market_operation_ending_balance",
                    "accumulated_provision_for_depreciation_of_electric_utility_plant_ending_balance": "total_ending_balance",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "electric_plant_classification_axis": "plant_status",
                    "utility_type_axis": "utility_type",
                }
            },
        },
        "categorize_strings": {
            "utility_type": UTILITY_TYPE_CATEGORIES,
            "plant_status": PLANT_STATUS,
            "plant_function": PLANT_FUNCTION_CATEGORIES,
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_accumdepr_prvsn"]},
        "wide_to_tidy": {
            "dbf": {
                "idx_cols": [
                    "utility_id_ferc1",
                    "report_year",
                    "plant_function",
                    "record_id",
                ],
                "value_types": ["ending_balance"],
                "expected_drop_cols": 2,
                "stacked_column_name": "plant_status",
            },
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "report_year",
                    "electric_plant_classification_axis",
                    "utility_type_axis",
                    "sched_table_name",
                ],
                "value_types": ["ending_balance"],
                "expected_drop_cols": 10,
                "stacked_column_name": "plant_function",
            },
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "depreciation_type"},
            "on": "depreciation_type",
        },
        "select_dbf_rows_by_category": {
            "column_name": "plant_function",
            "select_by_xbrl_categories": False,
            "additional_categories": [
                "distribution",
                "general",
                "hydraulic_production_conventional",
                "hydraulic_production_pumped_storage",
                "nuclear_production",
                "other_production",
                "regional_transmission_and_market_operation",
                "steam_production",
                "transmission",
                "total",
            ],
            "len_expected_categories_to_drop": 17,
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "reconcile_table_calculations": {
            "column_to_check": "ending_balance",
            "subdimension_column": "plant_status",
            "subdimension_merge_validation": "many_to_many",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {
                        "error_frequency": 0.034,
                        "null_reported_value_frequency": 0.76,
                    },
                    "xbrl_factoid": {"error_frequency": 0.034},
                    "report_year": {
                        "error_frequency": 0.1,
                        "relative_error_magnitude": 0.92,
                    },  # worst year: 2021
                    "utility_id_ferc1": {
                        "error_frequency": 0.51,
                        "relative_error_magnitude": 1.0,
                    },  # worst utility: 331
                }
            },
        },
    },
    "core_ferc1__yearly_cash_flows_sched120": {
        "rename_columns_ferc1": {
            "instant_xbrl": {
                "columns": {
                    "cash_and_cash_equivalents_ending_balance": "ending_balance_amount",
                    "cash_and_cash_equivalents_starting_balance": "starting_balance_amount",
                }
            },
            "duration_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_amount"
                    for xbrl_col in [
                        "net_increase_decrease_in_allowances_held_for_speculation_investing_activities",
                        "net_increase_decrease_in_other_regulatory_assets_operating_activities",
                        "gross_additions_to_nonutility_plant_investing_activities",
                        "net_increase_decrease_in_payables_and_accrued_expenses_operating_activities",
                        "payments_for_retirement_of_common_stock_financing_activities",
                        "depreciation_and_depletion",
                        "proceeds_from_disposal_of_noncurrent_assets",
                        "gross_additions_to_utility_plant_less_nuclear_fuel_investing_activities",
                        "gross_additions_to_nuclear_fuel_investing_activities",
                        "gross_additions_to_common_utility_plant_investing_activities",
                        "net_increase_decrease_in_inventory_investing_activities",
                        "net_increase_decrease_in_payables_and_accrued_expenses_investing_activities",
                        "dividends_on_common_stock",
                        "cash_provided_by_outside_sources",
                        "noncash_adjustments_to_cash_flows_from_operating_activities",
                        "cash_flows_provided_from_used_in_investment_activities",
                        "dividends_on_preferred_stock",
                        "net_increase_decrease_in_receivables_operating_activities",
                        "other_retirements_of_balances_impacting_cash_flows_from_financing_activities",
                        "other_adjustments_by_outside_sources_to_cash_flows_from_financing_activities",
                        "other_adjustments_to_cash_flows_from_investment_activities",
                        "net_increase_decrease_in_inventory_operating_activities",
                        "other_adjustments_to_cash_flows_from_operating_activities",
                        "collections_on_loans",
                        "net_increase_decrease_in_other_regulatory_liabilities_operating_activities",
                        "net_increase_decrease_in_allowances_inventory_operating_activities",
                        "net_increase_decrease_in_cash_and_cash_equivalents_abstract",
                        "net_cash_flow_from_operating_activities",
                        "disposition_of_investments_in_and_advances_to_associated_and_subsidiary_companies_abstract",
                        "proceeds_from_issuance_of_common_stock_financing_activities",
                        "net_income_loss",
                        "allowance_for_other_funds_used_during_construction_operating_activities",
                        "payments_for_retirement_of_preferred_stock_financing_activities",
                        "net_decrease_in_short_term_debt",
                        "net_increase_decrease_in_receivables_investing_activities",
                        "payments_for_retirement_abstract",
                        "investment_tax_credit_adjustments_net",
                        "acquisition_of_other_noncurrent_assets",
                        "investments_in_and_advances_to_associated_and_subsidiary_companies",
                        "undistributed_earnings_from_subsidiary_companies_operating_activities",
                        "allowance_for_other_funds_used_during_construction_investing_activities",
                        "other_adjustments_to_cash_flows_from_financing_activities",
                        "proceeds_from_issuance_of_long_term_debt_financing_activities",
                        "deferred_income_taxes_net",
                        "loans_made_or_purchased",
                        "proceeds_from_issuance_of_preferred_stock_financing_activities",
                        "cash_outflows_for_plant",
                        "contributions_and_advances_from_associated_and_subsidiary_companies",
                        "disposition_of_investments_in_and_advances_to_associated_and_subsidiary_companies",
                        "purchase_of_investment_securities",
                        "cash_flows_provided_from_used_in_financing_activities",
                        "proceeds_from_sales_of_investment_securities",
                        "payments_for_retirement_of_long_term_debt_financing_activities",
                        "net_increase_in_short_term_debt",
                        "other_construction_and_acquisition_of_plant_investment_activities",
                    ]
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                }
            },
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                    "xbrl_factoid": "amount_type",
                    "amounts": "amount",
                    "prev_amounts": "starting_balance",
                }
            },
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True
        },
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "end_date",
                    "start_date",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": ["amount"],
                "stacked_column_name": "amount_type",
            }
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_cash_flow"]},
        "select_dbf_rows_by_category": {
            "column_name": "amount_type",
            "select_by_xbrl_categories": True,
            "additional_categories": [
                "net_increase_decrease_in_cash_and_cash_equivalents"
            ],
            "len_expected_categories_to_drop": 5,
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "amount_type"},
            "on": "amount_type",
        },
    },
    "core_ferc1__yearly_operating_expenses_sched320": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                    "crnt_yr_amt": "dollar_value",
                    "xbrl_factoid": "expense_type",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "xbrl_factoid": "expense_type",
                }
            },
            "duration_xbrl": {
                "columns": {
                    xbrl_col: f"{xbrl_col}_dollar_value"
                    for xbrl_col in [
                        "administrative_and_general_expenses",
                        "administrative_and_general_operation_expense",
                        "administrative_and_general_salaries",
                        "administrative_expenses_transferred_credit",
                        "advertising_expenses",
                        "allowances",
                        "ancillary_services_market_administration",
                        "capacity_market_administration",
                        "coolants_and_water",
                        "customer_account_expenses",
                        "customer_assistance_expenses",
                        "customer_installations_expenses",
                        "customer_records_and_collection_expenses",
                        "customer_service_and_information_expenses",
                        "day_ahead_and_real_time_market_administration",
                        "demonstrating_and_selling_expenses",
                        "distribution_expenses",
                        "distribution_maintenance_expense_electric",
                        "distribution_operation_expenses_electric",
                        "duplicate_charges_credit",
                        "electric_expenses_hydraulic_power_generation",
                        "electric_expenses_nuclear_power_generation",
                        "electric_expenses_steam_power_generation",
                        "employee_pensions_and_benefits",
                        "franchise_requirements",
                        "fuel",
                        "fuel_steam_power_generation",
                        "general_advertising_expenses",
                        "generation_expenses",
                        "generation_interconnection_studies",
                        "hydraulic_expenses",
                        "hydraulic_power_generation_maintenance_expense",
                        "hydraulic_power_generation_operations_expense",
                        "informational_and_instructional_advertising_expenses",
                        "injuries_and_damages",
                        "load_dispatch_monitor_and_operate_transmission_system",
                        "load_dispatch_reliability",
                        "load_dispatch_transmission_service_and_scheduling",
                        "load_dispatching",
                        "maintenance_of_boiler_plant_steam_power_generation",
                        "maintenance_of_communication_equipment_electric_transmission",
                        "maintenance_of_communication_equipment_regional_market_expenses",
                        "maintenance_of_computer_hardware",
                        "maintenance_of_computer_hardware_transmission",
                        "maintenance_of_computer_software",
                        "maintenance_of_computer_software_transmission",
                        "maintenance_of_electric_plant_hydraulic_power_generation",
                        "maintenance_of_electric_plant_nuclear_power_generation",
                        "maintenance_of_electric_plant_steam_power_generation",
                        "maintenance_of_energy_storage_equipment",
                        "maintenance_of_energy_storage_equipment_other_power_generation",
                        "maintenance_of_energy_storage_equipment_transmission",
                        "maintenance_of_general_plant",
                        "maintenance_of_generating_and_electric_plant",
                        "maintenance_of_line_transformers",
                        "maintenance_of_meters",
                        "maintenance_of_miscellaneous_distribution_plant",
                        "maintenance_of_miscellaneous_hydraulic_plant",
                        "maintenance_of_miscellaneous_market_operation_plant",
                        "maintenance_of_miscellaneous_nuclear_plant",
                        "maintenance_of_miscellaneous_other_power_generation_plant",
                        "maintenance_of_miscellaneous_regional_transmission_plant",
                        "maintenance_of_miscellaneous_steam_plant",
                        "maintenance_of_miscellaneous_transmission_plant",
                        "maintenance_of_overhead_lines",
                        "maintenance_of_overhead_lines_transmission",
                        "maintenance_of_reactor_plant_equipment_nuclear_power_generation",
                        "maintenance_of_reservoirs_dams_and_waterways",
                        "maintenance_of_station_equipment",
                        "maintenance_of_station_equipment_transmission",
                        "maintenance_of_street_lighting_and_signal_systems",
                        "maintenance_of_structures",
                        "maintenance_of_structures_and_improvements_regional_market_expenses",
                        "maintenance_of_structures_distribution_expense",
                        "maintenance_of_structures_hydraulic_power_generation",
                        "maintenance_of_structures_nuclear_power_generation",
                        "maintenance_of_structures_steam_power_generation",
                        "maintenance_of_structures_transmission_expense",
                        "maintenance_of_underground_lines",
                        "maintenance_of_underground_lines_transmission",
                        "maintenance_supervision_and_engineering",
                        "maintenance_supervision_and_engineering_electric_transmission_expenses",
                        "maintenance_supervision_and_engineering_hydraulic_power_generation",
                        "maintenance_supervision_and_engineering_nuclear_power_generation",
                        "maintenance_supervision_and_engineering_other_power_generation",
                        "maintenance_supervision_and_engineering_steam_power_generation",
                        "market_facilitation_monitoring_and_compliance_services",
                        "market_monitoring_and_compliance",
                        "meter_expenses",
                        "meter_reading_expenses",
                        "miscellaneous_customer_accounts_expenses",
                        "miscellaneous_customer_service_and_informational_expenses",
                        "miscellaneous_distribution_expenses",
                        "miscellaneous_general_expenses",
                        "miscellaneous_hydraulic_power_generation_expenses",
                        "miscellaneous_nuclear_power_expenses",
                        "miscellaneous_other_power_generation_expenses",
                        "miscellaneous_sales_expenses",
                        "miscellaneous_steam_power_expenses",
                        "miscellaneous_transmission_expenses",
                        "nuclear_fuel_expense",
                        "nuclear_power_generation_maintenance_expense",
                        "nuclear_power_generation_operations_expense",
                        "office_supplies_and_expenses",
                        "operation_of_energy_storage_equipment",
                        "operation_of_energy_storage_equipment_distribution",
                        "operation_of_energy_storage_equipment_transmission_expense",
                        "operation_supervision",
                        "operation_supervision_and_engineering_distribution_expense",
                        "operation_supervision_and_engineering_electric_transmission_expenses",
                        "operation_supervision_and_engineering_hydraulic_power_generation",
                        "operation_supervision_and_engineering_nuclear_power_generation",
                        "operation_supervision_and_engineering_other_power_generation",
                        "operation_supervision_and_engineering_steam_power_generation",
                        "operations_and_maintenance_expenses_electric",
                        "other_expenses_other_power_supply_expenses",
                        "other_power_generation_maintenance_expense",
                        "other_power_generation_operations_expense",
                        "other_power_supply_expense",
                        "outside_services_employed",
                        "overhead_line_expense",
                        "overhead_line_expenses",
                        "power_production_expenses",
                        "power_production_expenses_hydraulic_power",
                        "power_production_expenses_nuclear_power",
                        "power_production_expenses_other_power",
                        "power_production_expenses_steam_power",
                        "power_purchased_for_storage_operations",
                        "property_insurance",
                        "purchased_power",
                        "regional_market_expenses",
                        "regional_market_maintenance_expense",
                        "regional_market_operation_expense",
                        "regulatory_commission_expenses",
                        "reliability_planning_and_standards_development",
                        "reliability_planning_and_standards_development_services",
                        "rents_administrative_and_general_expense",
                        "rents_distribution_expense",
                        "rents_hydraulic_power_generation",
                        "rents_nuclear_power_generation",
                        "rents_other_power_generation",
                        "rents_regional_market_expenses",
                        "rents_steam_power_generation",
                        "rents_transmission_electric_expense",
                        "sales_expenses",
                        "scheduling_system_control_and_dispatch_services",
                        "station_expenses_distribution",
                        "station_expenses_transmission_expense",
                        "steam_expenses_nuclear_power_generation",
                        "steam_expenses_steam_power_generation",
                        "steam_from_other_sources",
                        "steam_from_other_sources_nuclear_power_generation",
                        "steam_power_generation_maintenance_expense",
                        "steam_power_generation_operations_expense",
                        "steam_transferred_credit",
                        "steam_transferred_credit_nuclear_power_generation",
                        "street_lighting_and_signal_system_expenses",
                        "supervision_customer_account_expenses",
                        "supervision_customer_service_and_information_expenses",
                        "supervision_sales_expense",
                        "system_control_and_load_dispatching_electric",
                        "transmission_expenses",
                        "transmission_maintenance_expense_electric",
                        "transmission_of_electricity_by_others",
                        "transmission_operation_expense",
                        "transmission_rights_market_administration",
                        "transmission_service_studies",
                        "uncollectible_accounts",
                        "underground_line_expenses",
                        "underground_line_expenses_transmission_expense",
                        "water_for_power",
                    ]
                }
            },
        },
        "align_row_numbers_dbf": {"dbf_table_names": ["f1_elc_op_mnt_expn"]},
        "wide_to_tidy": {
            "xbrl": {
                "idx_cols": [
                    "entity_id",
                    "end_date",
                    "start_date",
                    "report_year",
                    "sched_table_name",
                ],
                "value_types": ["dollar_value"],
                "stacked_column_name": "xbrl_factoid",
            }
        },
        "drop_duplicate_rows_dbf": {
            "data_columns": ["dollar_value"],
            "table_name": "core_ferc1__yearly_operating_expenses_sched320",
        },
        "add_columns_with_uniform_values": {
            "columns_to_add": {
                "utility_type": {"column_value": "electric", "is_dimension": True}
            }
        },
        "merge_xbrl_metadata": {
            "rename_columns": {"xbrl_factoid": "expense_type"},
            "on": "expense_type",
        },
        "reconcile_table_calculations": {
            "column_to_check": "dollar_value",
            "group_metric_checks": {
                "group_metric_tolerances": {
                    "ungrouped": {"relative_error_magnitude": 0.002},
                    "report_year": {"relative_error_magnitude": 0.042},
                    "xbrl_factoid": {
                        "error_frequency": 0.018,
                        "relative_error_magnitude": 0.028,
                    },
                    "utility_id_ferc1": {
                        "error_frequency": 0.017,
                        "relative_error_magnitude": 0.066,
                    },
                },
            },
        },
    },
    "core_ferc1__yearly_other_regulatory_liabilities_sched278": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "spplmnt_num": "spplmnt_num",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "row_prvlg": "row_prvlg",
                    "report_prd": "report_prd",
                    "beg_yr_bal": "starting_balance",
                    "end_yr_bal": "ending_balance",
                    "dr_acct_num": "account_detail",
                    "dr_amount": "decrease_in_other_regulatory_liabilities",
                    "credits": "increase_in_other_regulatory_liabilities",
                    "dsc_purp": "description",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "increase_in_other_regulatory_liabilities": "increase_in_other_regulatory_liabilities",
                    "decrease_in_other_regulatory_liabilities": "decrease_in_other_regulatory_liabilities",
                    "description_and_purpose_of_other_regulatory_liabilities": "description",
                    "other_regulatory_liabilities_description_of_credited_account_number_for_debit_adjustment": "account_detail",
                    "other_regulatory_liabilities_ending_balance": "ending_balance",
                    "other_regulatory_liabilities_starting_balance": "starting_balance",
                    "other_regulatory_liability_axis": "other_regulatory_liability_axis",
                }
            },
        },
        "unstack_balances_to_report_year_instant_xbrl": {
            "unstack_balances_to_report_year": True,
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [0, pd.NA, np.nan, ""],
                "required_valid_cols": [
                    "ending_balance",
                    "starting_balance",
                    "increase_in_other_regulatory_liabilities",
                    "decrease_in_other_regulatory_liabilities",
                ],
            },
        ],
    },
    "core_ferc1__yearly_sales_by_rate_schedules_sched304": {
        "rename_columns_ferc1": {
            "dbf": {
                "columns": {
                    "respondent_id": "utility_id_ferc1_dbf",
                    "report_year": "report_year",
                    "report_prd": "report_prd",
                    "row_prvlg": "row_prvlg",
                    "row_number": "row_number",
                    "row_seq": "row_seq",
                    "spplmnt_num": "spplmnt_num",
                    "sched_num_ttl": "rate_schedule_description",
                    "mwh_sold": "sales_mwh",
                    "revenue": "dollar_value",
                    "avg_num_cstmr": "avg_customers_per_month",
                    "kwh_sale_cstmr": "kwh_per_customer",
                    "revenue_kwh_sold": "revenue_per_kwh",
                }
            },
            "xbrl": {
                "columns": {
                    "entity_id": "utility_id_ferc1_xbrl",
                    "report_year": "report_year",
                    "description_of_number_and_title_of_rate_schedule": "rate_schedule_description",
                    "average_number_of_customers_per_month_residential_sales": "residential_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_residential_sales_billed": "residential_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_small_or_commercial": "commercial_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_small_or_commercial_billed": "commercial_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_large_or_industrial": "industrial_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_large_or_industrial_billed": "industrial_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_commercial_and_industrial_sales": "commercial_and_industrial_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_commercial_and_industrial_sales_billed": "commercial_and_industrial_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_public_street_and_highway_lighting": "public_lighting_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_public_street_and_highway_lighting_billed": "public_lighting_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_other_sales_to_public_authorities": "public_authorities_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_other_sales_to_public_authorities_billed": "public_authorities_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_sales_to_railroads_and_railways": "railroads_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_sales_to_railroads_and_railways_billed": "railroads_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_interdepartmental_sales": "interdepartmental_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_interdepartmental_sales_billed": "interdepartmental_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_provision_for_rate_refunds": "provision_for_rate_refund_total_avg_customers_per_month",
                    "average_number_of_customers_per_month_provision_for_rate_refunds_billed": "provision_for_rate_refund_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_sales_of_electricity_by_rate_schedules_unbilled": "total_unbilled_avg_customers_per_month",
                    "average_number_of_customers_per_month_sales_of_electricity_by_rate_schedules": "total_billed_avg_customers_per_month",
                    "average_number_of_customers_per_month_sales_of_electricity_by_rate_schedules_including_unbilled_revenue": "total_total_avg_customers_per_month",
                    "average_kilowatt_hours_of_sales_per_customer_sales_of_electricity_by_rate_schedules": "total_billed_kwh_per_customer",
                    "average_kilowatt_hours_of_sales_per_customer_sales_of_electricity_by_rate_schedules_including_unbilled_revenue": "total_total_kwh_per_customer",  # maybe we don't need this b/c calculated?
                    "average_kilowatt_hours_of_sales_per_customer_sales_of_electricity_by_rate_schedules_unbilled": "total_unbilled_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_residential_sales": "residential_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_residential_sales_billed": "residential_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_small_or_commercial_sales": "commercial_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_small_or_commercial_sales_billed": "commercial_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_large_or_industrial_sales": "industrial_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_large_or_industrial_sales_billed": "industrial_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_commercial_and_industrial_sales": "commercial_and_industrial_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_commercial_and_industrial_sales_billed": "commercial_and_industrial_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_public_street_and_highway_lighting": "public_lighting_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_public_street_and_highway_lighting_billed": "public_lighting_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_other_sales_to_public_authorities": "public_authorities_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_other_sales_to_public_authorities_billed": "public_authorities_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_sales_to_railroads_and_railways": "railroads_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_sales_to_railroads_and_railways_billed": "railroads_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_interdepartmental_sales": "interdepartmental_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_interdepartmental_sales_billed": "interdepartmental_billed_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_provision_for_rate_refunds": "provision_for_rate_refund_total_kwh_per_customer",
                    "kilowatt_hours_of_sales_per_customer_provision_for_rate_refunds_billed": "provision_for_rate_refund_billed_kwh_per_customer",
                    "revenue_per_kilowatt_hour_sold_residential_sales": "residential_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_residential_sales_billed": "residential_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_residential_sales_unbilled": "residential_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_small_or_commercial_sales": "commercial_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_small_or_commercial_sales_billed": "commercial_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_small_or_commercial_sales_unbilled": "commercial_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_large_or_industrial_sales": "industrial_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_large_or_industrial_sales_billed": "industrial_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_large_or_industrial_sales_unbilled": "industrial_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_commercial_and_industrial_sales": "commercial_and_industrial_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_commercial_and_industrial_sales_billed": "commercial_and_industrial_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_commercial_and_industrial_sales_unbilled": "commercial_and_industrial_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_public_street_and_highway_lighting": "public_lighting_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_public_street_and_highway_lighting_billed": "public_lighting_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_public_street_and_highway_lighting_unbilled": "public_lighting_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_other_sales_to_public_authorities": "public_authorities_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_other_sales_to_public_authorities_billed": "public_authorities_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_other_sales_to_public_authorities_unbilled": "public_authorities_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_sales_to_railroads_and_railways": "railroads_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_sales_to_railroads_and_railways_billed": "railroads_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_sales_to_railroads_and_railways_unbilled": "railroads_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_interdepartmental_sales": "interdepartmental_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_interdepartmental_sales_billed": "interdepartmental_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_interdepartmental_sales_unbilled": "interdepartmental_unbilled_revenue_per_kwh",
                    "average_revenue_per_kilowatt_hour_sold_sales_of_electricity_by_rate_schedules": "total_billed_revenue_per_kwh",
                    "average_revenue_per_kilowatt_hour_sold_sales_of_electricity_by_rate_schedules_including_unbilled_revenue": "total_total_revenue_per_kwh",
                    "average_revenue_per_kilowatt_hour_sold_sales_of_electricity_by_rate_schedules_unbilled": "total_unbilled_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_provision_for_rate_refunds": "provision_for_rate_refund_total_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_provision_for_rate_refunds_billed": "provision_for_rate_refund_billed_revenue_per_kwh",
                    "revenue_per_kilowatt_hour_sold_provision_for_rate_refunds_unbilled": "provision_for_rate_refund_unbilled_revenue_per_kwh",
                    "megawatt_hours_sold_residential_sales": "residential_total_sales_mwh",
                    "megawatt_hours_sold_residential_sales_billed": "residential_billed_sales_mwh",
                    "megawatt_hours_sold_residential_sales_unbilled": "residential_unbilled_sales_mwh",
                    "megawatt_hours_sold_small_or_commercial": "commercial_total_sales_mwh",
                    "megawatt_hours_sold_small_or_commercial_billed": "commercial_billed_sales_mwh",
                    "megawatt_hours_sold_small_or_commercial_unbilled": "commercial_unbilled_sales_mwh",
                    "megawatt_hours_sold_large_or_industrial": "industrial_total_sales_mwh",
                    "megawatt_hours_sold_large_or_industrial_billed": "industrial_billed_sales_mwh",
                    "megawatt_hours_sold_large_or_industrial_unbilled": "industrial_unbilled_sales_mwh",
                    "megawatt_hours_sold_commercial_and_industrial_sales": "commercial_and_industrial_total_sales_mwh",
                    "megawatt_hours_sold_commercial_and_industrial_sales_billed": "commercial_and_industrial_billed_sales_mwh",
                    "megawatt_hours_sold_commercial_and_industrial_sales_unbilled": "commercial_and_industrial_unbilled_sales_mwh",
                    "megawatt_hours_sold_public_street_and_highway_lighting": "public_lighting_total_sales_mwh",
                    "megawatt_hours_sold_public_street_and_highway_lighting_billed": "public_lighting_billed_sales_mwh",
                    "megawatt_hours_sold_public_street_and_highway_lighting_unbilled": "public_lighting_unbilled_sales_mwh",
                    "megawatt_hours_sold_other_sales_to_public_authorities": "public_authorities_total_sales_mwh",
                    "megawatt_hours_sold_other_sales_to_public_authorities_billed": "public_authorities_billed_sales_mwh",
                    "megawatt_hours_sold_other_sales_to_public_authorities_unbilled": "public_authorities_unbilled_sales_mwh",
                    "megawatt_hours_sold_sales_to_railroads_and_railways": "railroads_total_sales_mwh",
                    "megawatt_hours_sold_sales_to_railroads_and_railways_billed": "railroads_billed_sales_mwh",
                    "megawatt_hours_sold_sales_to_railroads_and_railways_unbilled": "railroads_unbilled_sales_mwh",
                    "megawatt_hours_sold_interdepartmental_sales": "interdepartmental_total_sales_mwh",
                    "megawatt_hours_sold_interdepartmental_sales_billed": "interdepartmental_billed_sales_mwh",
                    "megawatt_hours_sold_interdepartmental_sales_unbilled": "interdepartmental_unbilled_sales_mwh",
                    "megawatt_hours_of_electricity_sold_by_rate_schedules": "total_billed_sales_mwh",
                    "megawatt_hours_of_electricity_sold_by_rate_schedules_including_unbilled_revenue": "total_total_sales_mwh",
                    "megawatt_hours_of_electricity_sold_by_rate_schedules_unbilled": "total_unbilled_sales_mwh",
                    "megawatt_hours_sold_provision_for_rate_refunds": "provision_for_rate_refund_total_sales_mwh",
                    "megawatt_hours_sold_provision_for_rate_refunds_billed": "provision_for_rate_refund_billed_sales_mwh",
                    "megawatt_hours_sold_provision_for_rate_refunds_unbilled": "provision_for_rate_refund_unbilled_sales_mwh",
                    "residential_sales": "residential_total_dollar_value",
                    "residential_sales_billed": "residential_billed_dollar_value",
                    "residential_sales_unbilled": "residential_unbilled_dollar_value",
                    "small_or_commercial_sales_electric_operating_revenue": "commercial_total_dollar_value",
                    "small_or_commercial_sales_electric_operating_revenue_billed": "commercial_billed_dollar_value",
                    "small_or_commercial_sales_electric_operating_revenue_unbilled": "commercial_unbilled_dollar_value",
                    "large_or_industrial_sales_electric_operating_revenue": "industrial_total_dollar_value",
                    "large_or_industrial_sales_electric_operating_revenue_billed": "industrial_billed_dollar_value",
                    "large_or_industrial_sales_electric_operating_revenue_unbilled": "industrial_unbilled_dollar_value",
                    "commercial_and_industrial_sales": "commercial_and_industrial_total_dollar_value",
                    "commercial_and_industrial_sales_billed": "commercial_and_industrial_billed_dollar_value",
                    "commercial_and_industrial_sales_unbilled": "commercial_and_industrial_unbilled_dollar_value",
                    "public_street_and_highway_lighting": "public_lighting_total_dollar_value",
                    "public_street_and_highway_lighting_billed": "public_lighting_billed_dollar_value",
                    "public_street_and_highway_lighting_unbilled": "public_lighting_unbilled_dollar_value",
                    "other_sales_to_public_authorities": "public_authorities_total_dollar_value",
                    "other_sales_to_public_authorities_billed": "public_authorities_billed_dollar_value",
                    "other_sales_to_public_authorities_unbilled": "public_authorities_unbilled_dollar_value",
                    "sales_to_railroads_and_railways": "railroads_total_dollar_value",
                    "sales_to_railroads_and_railways_billed": "railroads_billed_dollar_value",
                    "sales_to_railroads_and_railways_unbilled": "railroads_unbilled_dollar_value",
                    "interdepartmental_sales_billed": "interdepartmental_billed_dollar_value",
                    "interdepartmental_sales_unbilled": "interdepartmental_unbilled_dollar_value",
                    "interdepartmental_sales": "interdepartmental_total_dollar_value",
                    "provision_for_rate_refunds": "provision_for_rate_refund_total_dollar_value",
                    "provision_for_rate_refunds_billed": "provision_for_rate_refund_billed_dollar_value",
                    "provision_for_rate_refunds_unbilled": "provision_for_rate_refund_unbilled_dollar_value",
                    "revenue_from_sales_of_electricity_by_rate_schedules_including_unbilled_revenue": "total_total_dollar_value",
                    "revenue_from_sales_of_electricity_by_rate_schedules": "total_billed_dollar_value",
                    "revenue_from_sales_of_electricity_by_rate_schedules_unbilled": "total_unbilled_dollar_value",
                }
            },
        },
        "combine_axis_columns_xbrl": {
            "axis_columns_to_combine": [
                "residential_sales_axis",
                "commercial_sales_axis",
                "industrial_sales_axis",
                "public_street_and_highway_lighting_axis",
                "other_sales_to_public_authorities_axis",
                "sales_to_railroads_and_railways_axis",
                "interdepartmental_sales_axis",
                "provision_for_rate_refunds_axis",
            ],
            "new_axis_column_name": "sales_axis",
        },
        "wide_to_tidy": {
            "xbrl": [
                {
                    "idx_cols": [
                        "utility_id_ferc1_xbrl",
                        "report_year",
                        "start_date",
                        "end_date",
                        "rate_schedule_description",
                        "sched_table_name",
                        "sales_axis",
                    ],
                    "value_types": [
                        "billed_avg_customers_per_month",
                        "unbilled_avg_customers_per_month",
                        "total_avg_customers_per_month",
                        "billed_kwh_per_customer",
                        "unbilled_kwh_per_customer",
                        "total_kwh_per_customer",
                        "billed_revenue_per_kwh",
                        "unbilled_revenue_per_kwh",
                        "total_revenue_per_kwh",
                        "billed_sales_mwh",
                        "unbilled_sales_mwh",
                        "total_sales_mwh",
                        "billed_dollar_value",
                        "unbilled_dollar_value",
                        "total_dollar_value",
                    ],
                    "stacked_column_name": "rate_schedule_type",
                    "expected_drop_cols": 1,
                },
                {
                    "idx_cols": [
                        "utility_id_ferc1_xbrl",
                        "report_year",
                        "start_date",
                        "end_date",
                        "rate_schedule_description",
                        "sched_table_name",
                        "rate_schedule_type",
                        "sales_axis",
                    ],
                    "value_types": [
                        "avg_customers_per_month",
                        "kwh_per_customer",
                        "revenue_per_kwh",
                        "sales_mwh",
                        "dollar_value",
                    ],
                    "stacked_column_name": "billing_status",
                    "expected_drop_cols": 0,
                },
            ]
        },
        "drop_invalid_rows": [
            {
                "invalid_values": [np.nan, pd.NA, "", None],
                "required_valid_cols": [
                    "sales_mwh",
                    "dollar_value",
                    "avg_customers_per_month",
                    "kwh_per_customer",
                    "revenue_per_kwh",
                ],
            }
        ],
    },
}
"""The full set of parameters used to transform the FERC Form 1 data.

Each item in the dictionary can be used to instantiate a
:class:`pudl.transform.ferc1.Ferc1TableTransformParams` object appropriate for
transforming the table identified by that item's key.
"""
