"""A basic CLI to autogenerate dbt data test configurations."""

import json
from collections import namedtuple
from dataclasses import dataclass
from difflib import unified_diff
from pathlib import Path
from typing import Any

import click
import duckdb
import pandas as pd
import yaml
from deepdiff import DeepDiff
from pydantic import BaseModel

from pudl.dbt_wrapper import DBT_DIR, build_with_context, dagster_to_dbt_selection
from pudl.logging_helpers import configure_root_logger, get_logger
from pudl.metadata.classes import PUDL_PACKAGE
from pudl.workspace.setup import PudlPaths

logger = get_logger(__name__)


ALL_TABLES = [r.name for r in PUDL_PACKAGE.resources]


def _prettier_yaml_dumps(yaml_contents: dict[str, Any]) -> str:
    """Dump YAML to string that Prettier likes."""

    class PrettierCompatibleDumper(yaml.Dumper):
        """Custom Dumper that indents lists like prettier does.

        Default dumper behavior::

            foo:
            - name: a
            - name: b

        prettier behavior::

            foo:
              - name: a
              - name: b

        NOTE (2025-07-10): this code was generated by LLM and Does The
        Right Thing. It appears to do so by forcing the ``indentless`` parameter to
        always be ``False``. If at any point this stops working we should convert
        to using ``ruamel`` which has much more customizable formatting with actual
        documentation.
        """

        def increase_indent(self, flow=False, indentless=False):
            return super().increase_indent(flow, False)

    return yaml.dump(
        yaml_contents,
        default_flow_style=False,
        Dumper=PrettierCompatibleDumper,
        indent=2,
        sort_keys=False,
        width=float("inf"),
    )


class DbtColumn(BaseModel):
    """Define yaml structure of a dbt column."""

    name: str
    description: str | None = None
    data_tests: list | None = None
    meta: dict | None = None
    tags: list[str] | None = None

    def add_column_tests(self, column_tests: list) -> "DbtColumn":
        """Add data tests to columns in dbt config."""
        data_tests = self.data_tests if self.data_tests is not None else []
        return self.model_copy(update={"data_tests": data_tests + column_tests})


class DbtTable(BaseModel):
    """Define yaml structure of a dbt table."""

    name: str
    description: str | None = None
    data_tests: list | None = None
    columns: list[DbtColumn]
    meta: dict | None = None
    tags: list[str] | None = None
    config: dict | None = None  # only for models

    def add_source_tests(self, source_tests: list) -> "DbtTable":
        """Add data tests to source in dbt config."""
        data_tests = self.data_tests if self.data_tests is not None else []
        return self.model_copy(update={"data_tests": data_tests + source_tests})

    def add_column_tests(self, column_tests: dict[str, list]) -> "DbtTable":
        """Add data tests to columns in dbt config."""
        columns = {column.name: column for column in self.columns}
        columns.update(
            {
                name: columns[name].add_column_tests(tests)
                for name, tests in column_tests.items()
            }
        )

        return self.model_copy(update={"columns": list(columns.values())})

    @classmethod
    def from_table_name(cls, table_name: str) -> "DbtTable":
        """Construct configuration defining table from PUDL metadata."""
        return cls(
            name=table_name,
            columns=[
                DbtColumn(name=f.name)
                for f in PUDL_PACKAGE.get_resource(table_name).schema.fields
            ],
        )


class DbtSource(BaseModel):
    """Define basic dbt yml structure to add a pudl table as a dbt source."""

    name: str = "pudl"
    tables: list[DbtTable]
    description: str | None = None
    meta: dict | None = None

    def add_source_tests(self, source_tests: list) -> "DbtSource":
        """Add data tests to source in dbt config."""
        return self.model_copy(
            update={"tables": [self.tables[0].add_source_tests(source_tests)]}
        )

    def add_column_tests(self, column_tests: dict[str, list]) -> "DbtSource":
        """Add data tests to columns in dbt config."""
        return self.model_copy(
            update={"tables": [self.tables[0].add_column_tests(column_tests)]}
        )


class DbtSchema(BaseModel):
    """Define basic structure of a dbt models yaml file."""

    version: int = 2
    sources: list[DbtSource]
    models: list[DbtTable] | None = None

    def add_source_tests(
        self, source_tests: list, model_name: str | None = None
    ) -> "DbtSchema":
        """Add data tests to source in dbt config."""
        if model_name is None:
            schema = self.model_copy(
                update={"sources": [self.sources[0].add_source_tests(source_tests)]}
            )
        else:
            models = {model.name: model for model in self.models}
            models[model_name] = models[model_name].add_source_tests(source_tests)
            schema = self.model_copy(update={"models": list(models.values())})

        return schema

    def add_column_tests(
        self, column_tests: dict[str, list], model_name: str | None = None
    ) -> "DbtSchema":
        """Add data tests to columns in dbt config."""
        if model_name is None:
            schema = self.model_copy(
                update={"sources": [self.sources[0].add_column_tests(column_tests)]}
            )
        else:
            models = {model.name: model for model in self.models}
            models[model_name] = models[model_name].add_column_tests(column_tests)
            schema = self.model_copy(update={"models": list(models.values())})

        return schema

    @classmethod
    def from_table_name(cls, table_name: str) -> "DbtSchema":
        """Construct configuration defining table from PUDL metadata."""
        return cls(
            sources=[
                DbtSource(
                    tables=[DbtTable.from_table_name(table_name)],
                )
            ],
        )

    @classmethod
    def from_yaml(cls, schema_path: Path) -> "DbtSchema":
        """Load a DbtSchema object from a YAML file."""
        with schema_path.open("r") as schema_yaml:
            return cls.model_validate(yaml.safe_load(schema_yaml))

    def to_yaml(self, schema_path: Path):
        """Write DbtSchema object to YAML file."""
        with schema_path.open("w") as schema_file:
            yaml_output = _prettier_yaml_dumps(self.model_dump(exclude_none=True))
            schema_file.write(yaml_output)

    def merge_metadata_from(self, old_schema: "DbtSchema") -> "DbtSchema":
        """Merge metadata from an old schema into this one, preferring new values where present."""
        if self.models:
            raise ValueError(
                "Merging metadata is not allowed when models are defined in the schema."
            )

        def merge_pydantic_model(
            new_model: BaseModel, old_model: BaseModel
        ) -> BaseModel:
            """Recursively merge fields of two pydantic models."""
            if not old_model:
                return new_model

            merged_data = {}
            for field_name, _field_info in new_model.model_fields.items():
                new_value = getattr(new_model, field_name, None)
                old_value = getattr(old_model, field_name, None)

                if isinstance(new_value, BaseModel):
                    merged_data[field_name] = merge_pydantic_model(new_value, old_value)
                elif (
                    isinstance(new_value, list)
                    and new_value
                    and isinstance(new_value[0], BaseModel)
                ):
                    # Merge lists of models by matching names
                    old_map = {m.name: m for m in old_value or []}
                    merged_list = [
                        merge_pydantic_model(m, old_map.get(m.name)) for m in new_value
                    ]
                    merged_data[field_name] = merged_list
                else:
                    merged_data[field_name] = (
                        new_value if new_value is not None else old_value
                    )

            return new_model.model_copy(update=merged_data)

        # Merge sources
        old_sources_map = {s.name: s for s in old_schema.sources or []}
        new_sources = [
            merge_pydantic_model(src, old_sources_map.get(src.name))
            for src in self.sources
        ]

        return self.model_copy(update={"sources": new_sources})


def schema_has_removals_or_modifications(
    old_schema: dict,
    new_schema: dict,
) -> bool:
    """Check if schema changes include removals or modifications.

    Ignores:
    * Column removals with no metadata (only {"name"} or empty dict)
    * Column renames (values_changed on ['name'])
    """
    diff = DeepDiff(
        old_schema,
        new_schema,
        ignore_order=True,
        verbose_level=2,
        view="tree",
        threshold_to_diff_deeper=0,
    )

    change_keys = {
        "values_changed",
        "type_changes",
        "dictionary_item_removed",
        "iterable_item_removed",
        "attribute_deleted",
    }

    for key in change_keys:
        if key not in diff:
            continue

        items = diff[key]

        for obj in items.values() if isinstance(items, dict) else items:
            path_str = obj.path() if hasattr(obj, "path") else str(obj)
            removed_val = getattr(obj, "t1", None)

            if (
                key in ("dictionary_item_removed", "iterable_item_removed")
                and "['columns']" in path_str
                and isinstance(removed_val, dict)
                and (not removed_val or set(removed_val.keys()) == {"name"})
            ):
                continue  # ignore empty column drop

            if key == "values_changed" and path_str.endswith("['name']"):
                continue  # ignore renames

            return True  # anything else is a modification/removal

    return False


def _log_schema_diff(old_schema: DbtSchema, new_schema: DbtSchema):
    """Print colored summary of schema changes."""
    summary = _schema_diff_summary(old_schema, new_schema)
    click.echo("Proposed schema changes:")
    for line in summary:
        if line.startswith("- "):
            click.secho(line, fg="red")
        elif line.startswith("+ "):
            click.secho(line, fg="green")
        else:
            click.secho(line)


def _schema_diff_summary(old_schema: DbtSchema, new_schema: DbtSchema):
    """Return a summary of schema changes based on YAML output."""
    stripped = (
        line.rstrip()
        for line in unified_diff(
            _prettier_yaml_dumps(old_schema.model_dump(exclude_none=True)).split("\n"),
            _prettier_yaml_dumps(new_schema.model_dump(exclude_none=True)).split("\n"),
            fromfile="old_schema",
            tofile="new_schema",
        )
    )
    return [line for line in stripped if line != ""]


def get_data_source(table_name: str) -> str:
    """Return the data source element of the table's name."""
    return table_name.strip("_").split("_")[1]


UpdateResult = namedtuple("UpdateResult", ["success", "message"])


def _get_local_table_path(table_name):
    return str(PudlPaths().parquet_path(table_name))


def _get_model_path(table_name: str, data_source: str) -> Path:
    return DBT_DIR / "models" / data_source / table_name


def _get_row_count_csv_path() -> Path:
    return DBT_DIR / "seeds" / "etl_full_row_counts.csv"


def _get_existing_row_counts() -> pd.DataFrame:
    return pd.read_csv(
        _get_row_count_csv_path(),
        dtype={"partition": "string", "table_name": "string"},
    ).fillna(value="")


def _calculate_row_counts(
    table_name: str,
    partition_expr: str | None = None,
) -> pd.DataFrame:
    table_path = _get_local_table_path(table_name)

    if partition_expr is None:
        partition_expr_sql = "''"
        group_by_clause = ""
    else:
        partition_expr_sql = partition_expr
        group_by_clause = f"GROUP BY {partition_expr_sql}"

    row_count_query = f"""
SELECT
    CAST(COALESCE(CAST({partition_expr_sql} AS VARCHAR), '') AS VARCHAR) AS partition,
    COUNT(*) AS row_count
FROM '{table_path}' {group_by_clause}
    """  # noqa: S608

    new_row_counts = (
        duckdb.sql(row_count_query)
        .df()
        .assign(table_name=table_name)
        .astype({"partition": "string", "table_name": "string"})
        .loc[:, ["table_name", "partition", "row_count"]]
    )

    return new_row_counts


def _combine_row_counts(existing: pd.DataFrame, new: pd.DataFrame) -> pd.DataFrame:
    return (
        pd.concat([existing, new])
        .drop_duplicates(subset=["partition", "table_name"], keep="last")
        .sort_values(["table_name", "partition"])
    )


def _write_row_counts(row_counts: pd.DataFrame):
    csv_path = _get_row_count_csv_path()
    row_counts.to_csv(csv_path, index=False)


def update_row_counts(
    table_name: str,
    data_source: str,
    clobber: bool = False,
) -> UpdateResult:
    """Generate updated row counts per partition and write to csv file within dbt project."""
    model_path = _get_model_path(table_name, data_source)
    schema_path = model_path / "schema.yml"
    schema = DbtSchema.from_yaml(schema_path)
    table = schema.sources[0].tables[0]

    partition_expressions = _extract_row_count_partitions(table)
    if len(partition_expressions) > 1:
        raise ValueError(
            f"Only a single row counts test per table is supported, "
            f"but found {len(partition_expressions)} in {table_name}."
        )

    has_test = bool(partition_expressions)
    existing = _get_existing_row_counts()
    has_existing_row_counts = table_name in existing["table_name"].to_numpy()

    if not has_test and not has_existing_row_counts:
        return UpdateResult(
            success=True,
            message=f"No row count test defined for {table_name}, and no row counts found, nothing to update.",
        )

    if not has_test and not clobber:
        return UpdateResult(
            success=False,
            message=f"Row counts exist for {table_name}, but no row count test is defined. Use clobber to remove.",
        )

    if has_existing_row_counts and not clobber:
        return UpdateResult(
            success=False,
            message=f"Row counts for {table_name} already exist. Use clobber to overwrite.",
        )

    # At this point, we know row counts can be written: either because no row counts
    # exist yet, or because clobber is set.

    # In any case, we remove the old row counts for the table we are refreshing
    filtered = existing[existing["table_name"] != table_name]
    old = existing[existing["table_name"] == table_name]

    # At this point, there is no test defined but there are row counts, and overwrite is allowed, so
    # we want to remove the row counts for this table
    if not has_test:
        # Remove outdated entry
        _write_row_counts(filtered)
        return UpdateResult(
            success=True,
            message=f"Removed {len(existing) - len(filtered)} outdated row counts for {table_name} (no test defined).",
        )

    partition_expr = partition_expressions[0]  # TODO: support multiple partitions
    new = _calculate_row_counts(table_name, partition_expr)

    # Make old and new row counts comparable so we can detect changes
    row_count_idx = ["table_name", "partition"]
    if (
        old.sort_values(by=row_count_idx)
        .reset_index(drop=True)
        .equals(new.sort_values(by=row_count_idx).reset_index(drop=True))
    ):
        return UpdateResult(
            success=True,
            message=f"Row counts for {table_name} are unchanged.",
        )

    # Finally, we reach the case where there are actual row counts to update:
    combined = _combine_row_counts(filtered, new)
    _write_row_counts(combined)

    return UpdateResult(
        success=True,
        message=f"Successfully updated row counts for {table_name}, partitioned by {partition_expr}.",
    )


def update_table_schema(
    table_name: str,
    data_source: str,
    clobber: bool = False,
) -> UpdateResult:
    """Generate and write out a schema.yaml file defining a new or updated table."""
    model_path = _get_model_path(table_name, data_source)
    schema_path = model_path / "schema.yml"

    generated_schema = DbtSchema.from_table_name(table_name)

    # If we are updating an existing model, augment the newly generated schema with
    # whatever additional information is available from the old dbt schema.
    if model_path.exists() and not clobber:
        # Load existing schema
        old_schema = DbtSchema.from_yaml(schema_path)
        # Replace new schema with the generated schema but copying missing metadata
        # from the old schema
        new_schema = generated_schema.merge_metadata_from(old_schema)

        if schema_has_removals_or_modifications(
            old_schema.model_dump(exclude_none=True),
            new_schema.model_dump(exclude_none=True),
        ):
            _log_schema_diff(old_schema, new_schema)
            # TODO 2025-07-11: perhaps we can integrate this `git add -p` flow directly
            # into the --clobber command?
            return UpdateResult(
                success=False,
                message=f"dbt configuration for table {table_name} has "
                "information that would be deleted. Update manually, or: "
                "1) re-run with --clobber, 2) run `git add dbt/models -p` "
                "and follow the instructions for recovering the deleted info",
            )
    else:
        # Either there is no existing schema, or --clobber is set. Use generated schema as is.
        new_schema = generated_schema

    # By the time we've gotten here, either:
    # * We've merged the old and new schema (if old schema existed, and we're updating)
    # * We're clobbering an existing schema entirely (clobber is True)
    # * We're creating a new schema from scratch (model_path did not exist)
    model_path.mkdir(parents=True, exist_ok=True)
    new_schema.to_yaml(schema_path)

    return UpdateResult(
        success=True,
        message=f"Wrote schema config for table {table_name} at {schema_path}.",
    )


def _log_update_result(result: UpdateResult):
    if result.success:
        logger.info(result.message)
    else:
        logger.error(result.message)


def _extract_row_count_partitions(table: DbtTable) -> list[str | None]:
    """Extract partition columns from check_row_counts_per_partition tests in a DbtTable."""
    partitions: list[str | None] = []

    if table.data_tests:
        for test in table.data_tests:
            if "check_row_counts_per_partition" not in test:
                continue
            if not isinstance(test, dict):
                raise ValueError(f"Row counts test expected to be a dictionary: {test}")
            test_def = test.get("check_row_counts_per_partition")
            if isinstance(test_def, dict):
                partitions.append(test_def.get("arguments", {}).get("partition_expr"))

    return partitions


@dataclass
class TableUpdateArgs:
    """Define a single class to collect the args for all table update commands."""

    tables: list[str]
    schema: bool = False
    row_counts: bool = False
    clobber: bool = False


@click.command
@click.argument(
    "tables",
    nargs=-1,
)
@click.option(
    "--schema/--no-schema",
    default=False,
    help="Update source table schema.yml configs.",
)
@click.option(
    "--row-counts/--no-row-counts",
    default=False,
    help="Update source table row count expectations.",
)
@click.option(
    "--clobber/--no-clobber",
    default=False,
    help="Overwrite existing table schema config and row counts. Otherwise, the script will fail if destructive changes are made.",
)
def update_tables(
    tables: list[str],
    clobber: bool,
    schema: bool,
    row_counts: bool,
):
    """Add or update dbt schema configs and row count expectations for PUDL tables.

    The ``tables`` argument can be a single table name, a list of table names, or
    'all'. If 'all' the script will update configurations for for all PUDL tables.

    If ``--clobber`` is set, existing configurations for tables will be overwritten.
    if this does not result in deletions.
    """
    args = TableUpdateArgs(
        tables=list(tables),
        schema=schema,
        row_counts=row_counts,
        clobber=clobber,
    )

    tables = args.tables
    if "all" in tables:
        tables = ALL_TABLES
    elif len(bad_tables := [name for name in tables if name not in ALL_TABLES]) > 0:
        raise RuntimeError(
            f"The following table(s) could not be found in PUDL metadata: {bad_tables}"
        )

    for table_name in tables:
        data_source = get_data_source(table_name)
        if args.schema:
            _log_update_result(
                update_table_schema(
                    table_name=table_name,
                    data_source=data_source,
                    clobber=args.clobber,
                )
            )
        if args.row_counts:
            _log_update_result(
                update_row_counts(
                    table_name=table_name,
                    data_source=data_source,
                    clobber=args.clobber,
                )
            )


@click.command()
@click.option(
    "--select",
    help="dbt selector for the asset(s) you want to validate. Syntax "
    "documentation at https://docs.getdbt.com/reference/node-selection/syntax",
)
@click.option(
    "--asset-select",
    "-a",
    help=(
        "*DAGSTER* selector for the asset(s) you want to validate. "
        "This gets translated into a dbt selection. For example, you can "
        "use '+key:\"out_eia__yearly_generators\"' to validate "
        "out_eia_yearly_generators and its upstream assets. Syntax "
        "documentation at https://docs.dagster.io/guides/build/assets/asset-selection-syntax/reference "
    ),
)
@click.option(
    "--exclude",
    help="dbt selector for the asset(s) you want to exclude from validation. Syntax "
    "documentation at https://docs.getdbt.com/reference/node-selection/syntax",
)
@click.option(
    "--dry-run/--no-dry-run",
    default=False,
    help="If dry, will print out the parameters we would pass to dbt, but not "
    "actually run the validation tests. Defaults to not-dry.",
)
def validate(
    select: str | None = None,
    asset_select: str | None = None,
    exclude: str | None = None,
    dry_run: bool = False,
) -> None:
    """Validate a selection of dbt nodes.

    Wraps the ``dbt build`` command line so we can annotate the result with the
    actual data that was returned from the test query.

    Understands how to translate Dagster asset selection syntax into dbt node
    selections via the --asset-select flag.

    Default behavior if you do not pass `--asset-select` or `--select` is to
    validate everything.

    Usage examples:

    Run all the checks for one asset:

        $ dbt_helper validate --asset-select "key:out_eia__yearly_generators"

    Run the checks for one specific dbt node:

        $ dbt_helper validate --select "source:pudl_dbt.pudl.out_eia__yearly_generators"

    Run checks for an asset and all its upstream dependencies:

        $ dbt_helper validate --asset-select "+key:out_eia__yearly_generators"

    Exclude the row count tests:

        $ dbt_helper validate --asset-select "+key:out_eia__yearly_generators" --exclude "*check_row_counts*"



    """
    if select is not None:
        if asset_select is not None:
            raise click.UsageError(
                "You can't pass --select and --asset-select at the same time."
            )
        node_selection = select
    else:
        if asset_select is not None:
            node_selection = dagster_to_dbt_selection(asset_select)
        else:
            node_selection = "*"

    build_params = {
        "node_selection": node_selection,
        "node_exclusion": exclude,
        "dbt_target": "etl-full",
    }

    if dry_run:
        logger.info(
            f"Dry run - would build with these params: {json.dumps(build_params)}"
        )
        return

    test_result = build_with_context(**build_params)
    if not test_result.success:
        raise AssertionError(
            f"failure contexts:\n{test_result.format_failure_contexts()}"
        )


@click.group(
    context_settings={"help_option_names": ["-h", "--help"]},
)
def dbt_helper():
    """Script for auto-generating dbt configuration and migrating existing tests.

    This CLI currently provides the following sub-commands:

    update-tables: which can update or create a dbt table (model) schema.yml file under
    the ``dbt/models`` repo. These configuration files tell dbt about the structure of
    the table and what data tests are specified for it. The script can also generate or
    update the expected row counts for existing tables, assuming they have been
    materialized to parquet files and are sitting in your $PUDL_OUTPUT directory.

    validate: run validation tests for a selection of dbt nodes.

    Run ``dbt_helper {command} --help`` for detailed usage on each command.
    """


dbt_helper.add_command(update_tables)
dbt_helper.add_command(validate)


if __name__ == "__main__":
    configure_root_logger()
    dbt_helper()
