from __future__ import annotations

from dataclasses import dataclass
from typing import Any, Dict, List, Optional, Set
from datetime import datetime, timezone

from .policy import PolicyPack, load_policy_pack

ALLOWED_RECEIPT_TYPES = {"DATA","TRAIN","EVAL","DEPLOY","OPERATE","TRANSFER"}

@dataclass
class ReceiptResult:
    decision: str  # PASS|FAIL|HOLD
    reason_codes: List[str]
    detail: str = ""

def _parse_utc(ts: str) -> Optional[datetime]:
    if not isinstance(ts, str):
        return None
    try:
        return datetime.fromisoformat(ts.replace('Z', '+00:00')).astimezone(timezone.utc)
    except Exception:
        return None

def _deterministic_now_from_cs(cs: Dict[str, Any]) -> Optional[datetime]:
    # Deterministic replay reference time:
    # Prefer audit_window.end_utc when present (vector suite convention).
    aw = cs.get('audit_window')
    if isinstance(aw, dict):
        end_utc = aw.get('end_utc')
        dt = _parse_utc(end_utc)
        if dt is not None:
            return dt
    return None

def _get_registry_snapshot_refs(cs: Dict[str, Any]) -> List[str]:
    bindings = cs.get('bindings') if isinstance(cs, dict) else None
    if isinstance(bindings, dict):
        refs = bindings.get('registry_snapshot_refs')
        if isinstance(refs, list):
            return [r for r in refs if isinstance(r, str) and r.strip()]
    return []

def verify_receipt(receipt: Dict[str, Any], policy: PolicyPack, now: Optional[datetime]) -> ReceiptResult:
    required = ['receipt_type','action_class','scope','policy','evidence_handles','decision','reason_codes','issued_utc']
    missing = [k for k in required if k not in receipt]
    if missing:
        return ReceiptResult('HOLD', ['EVIDENCE_MISSING'], f"missing_fields={','.join(missing)}")

    rtype = receipt.get('receipt_type')
    if rtype not in ALLOWED_RECEIPT_TYPES:
        return ReceiptResult('FAIL', ['POLICY_MISMATCH'], f"unknown_receipt_type={rtype}")

    # Policy binding
    pol = receipt.get('policy') or {}
    if pol.get('policy_id') != policy.policy_id or pol.get('version') != policy.version:
        return ReceiptResult('FAIL', ['POLICY_MISMATCH'], 'policy_id_or_version_mismatch')

    # Scope + action class admissibility
    if receipt.get('action_class') not in policy.allowed_action_classes:
        return ReceiptResult('FAIL', ['SCOPE_MISMATCH'], 'action_class_not_allowed')
    if receipt.get('scope') not in policy.allowed_scopes:
        return ReceiptResult('FAIL', ['SCOPE_MISMATCH'], 'scope_not_allowed')
    if rtype not in policy.allowed_receipt_types:
        return ReceiptResult('FAIL', ['SCOPE_MISMATCH'], 'receipt_type_not_allowed')

    # Evidence presence
    handles = receipt.get('evidence_handles')
    if not isinstance(handles, list) or len(handles) == 0:
        return ReceiptResult('HOLD', ['EVIDENCE_MISSING'], 'no_evidence_handles')

    # Freshness (optional, deterministic)
    if policy.freshness_max_age_s is not None:
        issued = _parse_utc(receipt.get('issued_utc'))
        if issued is None or now is None:
            return ReceiptResult('HOLD', ['NONDETERMINISM_DETECTED'], 'freshness_time_unparseable')
        age_s = int((now - issued).total_seconds())
        if age_s > int(policy.freshness_max_age_s):
            return ReceiptResult('HOLD', ['EVIDENCE_STALE'], f"age_s={age_s} max_age_s={policy.freshness_max_age_s}")

    # If the receipt itself claims HOLD, propagate.
    if receipt.get('decision') == 'HOLD':
        rc = receipt.get('reason_codes') or ['NONDETERMINISM_DETECTED']
        return ReceiptResult('HOLD', list(rc), 'receipt_claims_hold')

    return ReceiptResult('PASS', [], 'admissible_mvp')

@dataclass
class PackResult:
    decision: str
    reason_codes: List[str]
    findings: List[str]

def verify_pack(pack: Dict[str, Any]) -> PackResult:
    # Required top-level fields for the vector suite
    for k in ['conformance_statement','policy_pack','receipts']:
        if k not in pack:
            return PackResult('HOLD', ['EVIDENCE_MISSING'], [f"missing_pack_field={k}"])

    cs = pack.get('conformance_statement') or {}
    for k in ['rfc_version','claimed_level','covered_receipt_types','covered_control_points','declared_scope','declared_action_class']:
        if k not in cs:
            return PackResult('HOLD', ['EVIDENCE_MISSING'], [f"missing_conformance_statement_field={k}"])

    policy = load_policy_pack(pack.get('policy_pack') or {})
    claimed_level = cs.get('claimed_level')
    require_enforcement = claimed_level in ('L2','L3')

    # Optional: require registry snapshot reference(s) (non-normative; supports replay stability guidance).
    if policy.require_reason_code_registry_snapshot and claimed_level in ('L1','L2','L3'):
        refs = _get_registry_snapshot_refs(cs)
        if not refs:
            return PackResult('HOLD', ['REGISTRY_SNAPSHOT_MISSING'], [
                'Required registry_snapshot_refs missing or empty in conformance_statement.bindings.'
            ])

    now = _deterministic_now_from_cs(cs)

    receipts = pack.get('receipts')
    if not isinstance(receipts, list) or not receipts:
        return PackResult('HOLD', ['EVIDENCE_MISSING'], ['no_receipts'])

    covered_cps = set([c.strip().lower() for c in (cs.get('covered_control_points') or []) if isinstance(c, str) and c.strip()])
    if require_enforcement and not covered_cps:
        return PackResult('HOLD', ['EVIDENCE_MISSING'], ['L2/L3 claimed but covered_control_points empty.'])

    findings: List[str] = []
    reasons: Set[str] = set()

    # Coverage gap check (L2+)
    if require_enforcement and covered_cps:
        present_cps = set([(r.get('control_point') or '').strip().lower() for r in receipts if isinstance(r, dict)])
        present_cps = {cp for cp in present_cps if cp}
        for cp in sorted(covered_cps):
            if cp not in present_cps:
                findings.append(f"COVERAGE_GAP: missing receipt evidence for claimed control_point '{cp}'.")
                reasons.add('COVERAGE_GAP')

    for r in receipts:
        if not isinstance(r, dict):
            findings.append("HOLD ['EVIDENCE_MISSING'] receipt_not_object")
            reasons.add('EVIDENCE_MISSING')
            continue

        rr = verify_receipt(r, policy, now)
        if rr.decision != 'PASS':
            findings.append(f"{r.get('receipt_type','?')}: {rr.decision} {rr.reason_codes} {rr.detail}")
            for rc in (rr.reason_codes or []):
                if isinstance(rc, str) and rc:
                    reasons.add(rc)
            continue

        if require_enforcement:
            cp = (r.get('control_point') or '').strip().lower()
            if not cp:
                findings.append(f"{r.get('receipt_type','?')}: HOLD ['EVIDENCE_MISSING'] missing_control_point")
                reasons.add('EVIDENCE_MISSING')
                continue
            if covered_cps and cp not in covered_cps:
                findings.append(f"{r.get('receipt_type','?')}: FAIL ['SCOPE_MISMATCH'] control_point_not_covered")
                reasons.add('SCOPE_MISMATCH')
                continue

            permit = r.get('permit')
            if not isinstance(permit, dict) or not permit.get('permit_id'):
                findings.append(f"{r.get('receipt_type','?')}: HOLD ['EVIDENCE_MISSING'] missing_permit")
                reasons.add('EVIDENCE_MISSING')
                continue
            if permit.get('scope') and permit.get('scope') != r.get('scope'):
                findings.append(f"{r.get('receipt_type','?')}: FAIL ['SCOPE_MISMATCH'] permit_scope_mismatch")
                reasons.add('SCOPE_MISMATCH')
                continue

    # Aggregate
    if any('FAIL' in f for f in findings) or ('COVERAGE_GAP' in reasons):
        if 'COVERAGE_GAP' in reasons and not any('FAIL' in f for f in findings):
            findings.insert(0, "FAIL ['COVERAGE_GAP'] Claimed control-point coverage is not evidenced by receipts in the pack.")
        return PackResult('FAIL', sorted(reasons) if reasons else ['COVERAGE_GAP'], findings)

    if findings:
        if not reasons:
            reasons.add('EVIDENCE_MISSING')
        return PackResult('HOLD', sorted(reasons), findings)

    return PackResult('PASS', [], ['Conformance pack replay PASS (MVP).'])
