import argparse
import json
from pathlib import Path
from typing import Any, Dict

from .verify import verify_pack

def _load_json(path: str) -> Dict[str, Any]:
    p = Path(path)
    return json.loads(p.read_text(encoding='utf-8'))

def cmd_verify_pack(args: argparse.Namespace) -> int:
    pack = _load_json(args.pack_json)
    res = verify_pack(pack)
    print(json.dumps({
        'decision': res.decision,
        'reason_codes': res.reason_codes,
        'findings': res.findings,
    }, indent=2))
    return 0 if res.decision == 'PASS' else 1

def cmd_run_vectors(args: argparse.Namespace) -> int:
    base = Path(args.vectors_dir)
    ok = 0
    fail = 0
    for child in sorted([p for p in base.iterdir() if p.is_dir()]):
        pack_path = child / 'pack.json'
        exp_path = child / 'expected.json'
        if not pack_path.exists() or not exp_path.exists():
            continue

        pack = json.loads(pack_path.read_text(encoding='utf-8'))
        exp = json.loads(exp_path.read_text(encoding='utf-8'))

        res = verify_pack(pack)
        got_decision = res.decision
        got_reasons = list(res.reason_codes or [])

        expected_decision = exp.get('decision')
        expected_reasons = exp.get('reason_codes')

        ok_this = (got_decision == expected_decision)
        if ok_this and isinstance(expected_reasons, list):
            ok_this = (got_reasons == list(expected_reasons))

        if ok_this:
            print(f"[OK]   {child}")
            ok += 1
        else:
            msg = f"[FAIL] {child} expected_decision={expected_decision} got_decision={got_decision}"
            if isinstance(expected_reasons, list):
                msg += f" | expected_reason_codes={expected_reasons} got_reason_codes={got_reasons}"
            print(msg)
            fail += 1

    return 0 if fail == 0 else 1

def main() -> int:
    ap = argparse.ArgumentParser(prog='kc_independent_verifier')
    sub = ap.add_subparsers(dest='cmd', required=True)

    ap_v = sub.add_parser('verify-pack', help='Verify a conformance pack JSON.')
    ap_v.add_argument('pack_json')
    ap_v.set_defaults(func=cmd_verify_pack)

    ap_r = sub.add_parser('run-vectors', help='Run a directory of vectors (each contains pack.json + expected.json).')
    ap_r.add_argument('vectors_dir')
    ap_r.set_defaults(func=cmd_run_vectors)

    args = ap.parse_args()
    return int(args.func(args))

if __name__ == '__main__':
    raise SystemExit(main())
