from dataclasses import dataclass
from typing import Dict, Any, List, Optional, Tuple
from .policy import PolicyPack
from .reason_codes import HOLD_IS_FAIL_CLOSED

ALLOWED_RECEIPT_TYPES = {"DATA","TRAIN","EVAL","DEPLOY","OPERATE","TRANSFER"}

@dataclass
class VerifyResult:
    decision: str  # PASS|FAIL|HOLD
    reason_codes: List[str]
    notes: Optional[str] = None

def _now_epoch_s(now_epoch_s: Optional[int]) -> int:
    import time
    return int(now_epoch_s if now_epoch_s is not None else time.time())

def verify_receipt(receipt: Dict[str, Any], policy: PolicyPack, now_epoch_s: Optional[int]=None) -> VerifyResult:
    # Minimal required fields per RFC
    required = ["receipt_type","action_class","scope","policy","evidence_handles","decision","reason_codes","issued_utc"]
    for k in required:
        if k not in receipt:
            return VerifyResult("HOLD", ["EVIDENCE_MISSING"], f"Missing receipt field: {k}")
    rt = receipt["receipt_type"]
    if rt not in ALLOWED_RECEIPT_TYPES:
        return VerifyResult("FAIL", ["POLICY_MISMATCH"], f"Unknown receipt_type: {rt}")
    # policy id/version binding
    pol = receipt["policy"]
    if pol.get("policy_id") != policy.policy_id or pol.get("version") != policy.version:
        return VerifyResult("FAIL", ["POLICY_MISMATCH"], "Policy id/version mismatch.")
    # scope/action class admissibility
    if receipt["action_class"] not in policy.allowed_action_classes or receipt["scope"] not in policy.allowed_scopes:
        return VerifyResult("FAIL", ["SCOPE_MISMATCH"], "Scope/action_class not allowed by policy.")
    if rt not in policy.allowed_receipt_types:
        return VerifyResult("FAIL", ["SCOPE_MISMATCH"], "Receipt type not allowed by policy.")
    # evidence presence
    handles = receipt.get("evidence_handles", [])
    if not isinstance(handles, list) or len(handles) == 0:
        return VerifyResult("HOLD", ["EVIDENCE_MISSING"], "No evidence handles.")
    # freshness (optional)
    if policy.freshness_max_age_s is not None:
        # issued_utc must be ISO8601; parse roughly
        from datetime import datetime, timezone
        try:
            dt = datetime.fromisoformat(receipt["issued_utc"].replace("Z","+00:00"))
            age = _now_epoch_s(now_epoch_s) - int(dt.replace(tzinfo=timezone.utc).timestamp())
            if age > policy.freshness_max_age_s:
                return VerifyResult("HOLD", ["EVIDENCE_STALE"], f"Receipt age {age}s exceeds max {policy.freshness_max_age_s}s.")
        except Exception:
            return VerifyResult("HOLD", ["NONDETERMINISM_DETECTED"], "Could not parse issued_utc deterministically.")
    # If receipt claims HOLD, treat as HOLD
    claimed = receipt.get("decision")
    if claimed == "HOLD":
        return VerifyResult("HOLD", list(receipt.get("reason_codes") or ["NONDETERMINISM_DETECTED"]), "Receipt claims HOLD.")
    # Otherwise PASS
    return VerifyResult("PASS", [], "Receipt admissible under policy (MVP).")
