from dataclasses import dataclass
from typing import List, Dict, Any, Optional

@dataclass(frozen=True)
class PolicyPack:
    policy_id: str
    version: str
    allowed_receipt_types: List[str]
    allowed_action_classes: List[str]
    allowed_scopes: List[str]
    freshness_max_age_s: Optional[int] = None  # optional policy freshness rule

    # Optional (non-normative): require a resolvable registry snapshot reference for replay.
    # This supports RFC guidance on registry snapshot references for interpreting identifiers/reason codes.
    require_reason_code_registry_snapshot: bool = False

def load_policy_pack(obj: Dict[str, Any]) -> PolicyPack:
    required = ["policy_id","version","allowed_receipt_types","allowed_action_classes","allowed_scopes"]
    for k in required:
        if k not in obj:
            raise ValueError(f"PolicyPack missing required field: {k}")
    return PolicyPack(
        policy_id=obj["policy_id"],
        version=obj["version"],
        allowed_receipt_types=list(obj["allowed_receipt_types"]),
        allowed_action_classes=list(obj["allowed_action_classes"]),
        allowed_scopes=list(obj["allowed_scopes"]),
        freshness_max_age_s=obj.get("freshness_max_age_s"),
        require_reason_code_registry_snapshot=bool(obj.get("require_reason_code_registry_snapshot", False)),
    )
