from dataclasses import dataclass
from typing import Dict, Any, List, Optional, Set

from .policy import load_policy_pack, PolicyPack
from .verify import verify_receipt

# Public-safe, non-normative reference verifier utilities.
# The RFC is the normative specification.

CONTROL_POINTS_MIN = {
    "compute admission",
    "artifact registry/promotion",
    "transfer/egress",
    "externalization boundary",
    "deployment change",
}

@dataclass
class PackResult:
    decision: str  # PASS|FAIL|HOLD
    reason_codes: List[str]
    findings: List[str]

def _deterministic_now_epoch_s_from_cs(cs: Dict[str, Any]) -> Optional[int]:
    """Deterministic replay reference time.
    Convention: use conformance_statement.audit_window.end_utc when present.
    """
    aw = cs.get("audit_window") if isinstance(cs, dict) else None
    end_utc = aw.get("end_utc") if isinstance(aw, dict) else None
    if isinstance(end_utc, str):
        try:
            from datetime import datetime, timezone
            dt = datetime.fromisoformat(end_utc.replace("Z", "+00:00"))
            return int(dt.replace(tzinfo=timezone.utc).timestamp())
        except Exception:
            return None
    return None

def _get_registry_snapshot_refs(cs: Dict[str, Any]) -> List[str]:
    bindings = cs.get("bindings") if isinstance(cs, dict) else None
    if isinstance(bindings, dict):
        refs = bindings.get("registry_snapshot_refs")
        if isinstance(refs, list):
            return [r for r in refs if isinstance(r, str) and r.strip()]
    return []

def verify_conformance_pack(pack: Dict[str, Any]) -> PackResult:
    findings: List[str] = []
    reasons: Set[str] = set()

    # Required top-level
    for k in ["conformance_statement", "policy_pack", "receipts"]:
        if k not in pack:
            return PackResult("HOLD", ["EVIDENCE_MISSING"], [f"Missing pack field: {k}"])

    cs = pack["conformance_statement"]
    for k in ["rfc_version", "claimed_level", "covered_receipt_types", "covered_control_points", "declared_scope", "declared_action_class"]:
        if k not in cs:
            return PackResult("HOLD", ["EVIDENCE_MISSING"], [f"Missing conformance_statement field: {k}"])

    policy: PolicyPack = load_policy_pack(pack["policy_pack"])
    claimed_level = cs.get("claimed_level")

    # Deterministic replay reference time
    now_epoch_s = _deterministic_now_epoch_s_from_cs(cs)

    # Optional: require registry snapshot reference(s) (public-safe; supports RFC replay stability guidance).
    if policy.require_reason_code_registry_snapshot and claimed_level in ("L1", "L2", "L3"):
        refs = _get_registry_snapshot_refs(cs)
        if not refs:
            return PackResult(
                "HOLD",
                ["REGISTRY_SNAPSHOT_MISSING"],
                ["Required registry_snapshot_refs missing or empty in conformance_statement.bindings."],
            )

    receipts = pack["receipts"]
    if not isinstance(receipts, list) or not receipts:
        return PackResult("HOLD", ["EVIDENCE_MISSING"], ["No receipts in pack."])

    require_enforcement = claimed_level in ("L2", "L3")

    covered_cps = set(
        [c.strip().lower() for c in cs.get("covered_control_points", []) if isinstance(c, str) and c.strip()]
    )

    if require_enforcement and not covered_cps:
        return PackResult("HOLD", ["EVIDENCE_MISSING"], ["L2/L3 claimed but covered_control_points empty."])

    # (B) Control-point coverage check (L2+): claimed control points MUST be evidenced by at least one receipt.
    if require_enforcement and covered_cps:
        present_cps = set(
            [(r.get("control_point") or "").strip().lower() for r in receipts if isinstance(r, dict)]
        )
        present_cps = {cp for cp in present_cps if cp}
        for cp in sorted(covered_cps):
            if cp not in present_cps:
                findings.append(f"COVERAGE_GAP: missing receipt evidence for claimed control_point '{cp}'.")
                reasons.add("COVERAGE_GAP")

    # Per-receipt checks (MVP semantics)
    for r in receipts:
        if not isinstance(r, dict):
            findings.append("HOLD ['EVIDENCE_MISSING'] Receipt is not an object.")
            reasons.add("EVIDENCE_MISSING")
            continue

        res = verify_receipt(r, policy, now_epoch_s=now_epoch_s)
        if res.decision != "PASS":
            findings.append(
                f"{r.get('receipt_type','?')}: {res.decision} {res.reason_codes} {res.notes or ''}".strip()
            )
            for rc in (res.reason_codes or []):
                if isinstance(rc, str) and rc:
                    reasons.add(rc)
            continue

        if require_enforcement:
            cp = (r.get("control_point") or "").strip().lower()
            if not cp:
                findings.append(f"{r.get('receipt_type','?')}: HOLD ['EVIDENCE_MISSING'] Missing control_point for L2/L3 claim.")
                reasons.add("EVIDENCE_MISSING")
                continue
            if covered_cps and cp not in covered_cps:
                findings.append(f"{r.get('receipt_type','?')}: FAIL ['SCOPE_MISMATCH'] control_point '{cp}' not in covered_control_points.")
                reasons.add("SCOPE_MISMATCH")
                continue

            permit = r.get("permit")
            if not isinstance(permit, dict) or not permit.get("permit_id"):
                findings.append(f"{r.get('receipt_type','?')}: HOLD ['EVIDENCE_MISSING'] Missing permit evidence for L2/L3 claim.")
                reasons.add("EVIDENCE_MISSING")
                continue
            if permit.get("scope") and permit.get("scope") != r.get("scope"):
                findings.append(f"{r.get('receipt_type','?')}: FAIL ['SCOPE_MISMATCH'] Permit scope mismatch.")
                reasons.add("SCOPE_MISMATCH")
                continue

    # Aggregate decision
    if any("FAIL" in f for f in findings) or ("COVERAGE_GAP" in reasons):
        # Coverage gaps and explicit FAIL findings invalidate the claim.
        if "COVERAGE_GAP" in reasons and not any("FAIL" in f for f in findings):
            findings.insert(0, "FAIL ['COVERAGE_GAP'] Claimed control-point coverage is not evidenced by receipts in the pack.")
        return PackResult("FAIL", sorted(reasons) if reasons else ["COVERAGE_GAP"], findings)

    if findings:
        # Remaining findings are HOLD-class.
        if not reasons:
            reasons.add("EVIDENCE_MISSING")
        return PackResult("HOLD", sorted(reasons), findings)

    return PackResult("PASS", [], ["Conformance pack replay PASS (MVP)."])
