import argparse, json, sys, os
from .policy import load_policy_pack
from .verify import verify_receipt
from .pack import verify_conformance_pack

def _load(path: str):
    with open(path, "r", encoding="utf-8") as f:
        return json.load(f)

def cmd_verify_receipt(args):
    receipt = _load(args.receipt)
    policy = load_policy_pack(_load(args.policy))
    res = verify_receipt(receipt, policy)
    print(json.dumps(res.__dict__, indent=2))

def cmd_verify_pack(args):
    pack = _load(args.pack)
    res = verify_conformance_pack(pack)
    print(json.dumps(res.__dict__, indent=2))

def cmd_run_vectors(args):
    base = args.dir
    failures = 0
    for root, _, files in os.walk(base):
        if "expected.json" in files and "pack.json" in files:
            pack = _load(os.path.join(root, "pack.json"))
            exp = _load(os.path.join(root, "expected.json"))
            got = verify_conformance_pack(pack).__dict__

            ok = True
            if got.get("decision") != exp.get("decision"):
                ok = False

            # Optional stronger check: expected reason_codes (if present) must match.
            if ok and isinstance(exp.get("reason_codes"), list):
                if list(got.get("reason_codes") or []) != list(exp.get("reason_codes") or []):
                    ok = False

            if not ok:
                failures += 1
                msg = f"[FAIL] {root}: expected decision={exp.get('decision')} got decision={got.get('decision')}"
                if isinstance(exp.get("reason_codes"), list):
                    msg += f" | expected reason_codes={exp.get('reason_codes')} got reason_codes={got.get('reason_codes')}"
                print(msg)
            else:
                print(f"[OK]   {root}")
    if failures:
        sys.exit(1)

def main():
    p = argparse.ArgumentParser(prog="kc-verifier")
    sp = p.add_subparsers(dest="cmd", required=True)

    p1 = sp.add_parser("verify-receipt")
    p1.add_argument("receipt")
    p1.add_argument("--policy", required=True)
    p1.set_defaults(func=cmd_verify_receipt)

    p2 = sp.add_parser("verify-pack")
    p2.add_argument("pack")
    p2.set_defaults(func=cmd_verify_pack)

    p3 = sp.add_parser("run-vectors")
    p3.add_argument("dir")
    p3.set_defaults(func=cmd_run_vectors)

    args = p.parse_args()
    args.func(args)

if __name__ == "__main__":
    main()
