// SPDX-FileCopyrightText: 2014 Istituto Nazionale di Fisica Nucleare
//
// SPDX-License-Identifier: Apache-2.0

package org.italiangrid.storm.webdav.test.authz.pdp;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationRequest.newAuthorizationRequest;
import static org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationResult.Decision.DENY;
import static org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationResult.Decision.NOT_APPLICABLE;
import static org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationResult.Decision.PERMIT;
import static org.mockito.Mockito.lenient;
import static org.mockito.Mockito.when;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.MappingMatch;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import org.eclipse.jetty.ee11.servlet.ServletPathMapping;
import org.italiangrid.storm.webdav.authz.VOMSFQANAuthority;
import org.italiangrid.storm.webdav.authz.VOMSVOAuthority;
import org.italiangrid.storm.webdav.authz.pdp.DefaultPathAuthorizationPdp;
import org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationPolicy;
import org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationPolicyRepository;
import org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationResult;
import org.italiangrid.storm.webdav.authz.pdp.principal.Anyone;
import org.italiangrid.storm.webdav.authz.pdp.principal.AuthorityHolder;
import org.italiangrid.storm.webdav.oauth.authority.JwtClientAuthority;
import org.italiangrid.storm.webdav.oauth.authority.JwtGroupAuthority;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.http.HttpMethod;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.web.servlet.util.matcher.PathPatternRequestMatcher;

@ExtendWith(MockitoExtension.class)
class AuthzPdpTests {

  public static final String TEST_ISSUER = "https://test.example";
  public static final String TEST2_ISSUER = "https://test2.example";
  public static final String AUTHORIZED_JWT_CLIENT_ID = "1234";
  public static final String UNAUTHORIZED_JWT_CLIENT_ID = "5678";

  @Mock PathAuthorizationPolicyRepository repo;

  @Mock HttpServletRequest request;

  @Mock Authentication authentication;

  @Mock ServletPathMapping servletPathMapping;

  @InjectMocks DefaultPathAuthorizationPdp pdp;

  @SuppressWarnings("unchecked")
  private <T> Set<T> authorities(GrantedAuthority... authorities) {
    return (Set<T>) Set.of(authorities);
  }

  @BeforeEach
  void setup() {
    lenient().when(request.getRequestURI()).thenReturn("/");
    lenient().when(request.getHttpServletMapping()).thenReturn(servletPathMapping);
    lenient().when(servletPathMapping.getMappingMatch()).thenReturn(MappingMatch.DEFAULT);
    lenient().when(repo.getPolicies()).thenReturn(Collections.emptyList());
  }

  @Test
  void notApplicableWithEmptyPolicies() {

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(NOT_APPLICABLE));
  }

  @Test
  void denyPolicyApplied() {

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(DENY));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(denyAllPolicy));
  }

  @Test
  void firstApplicablePolicyApplied() {

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    PathAuthorizationPolicy permitAllPolicy =
        PathAuthorizationPolicy.builder()
            .withPermit()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(permitAllPolicy, denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(permitAllPolicy));
  }

  @Test
  void oauthGroupHolderPolicyNotAppliedAsItDoesNotMatchPath() {

    PathAuthorizationPolicy oauthTestPolicy =
        PathAuthorizationPolicy.builder()
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new JwtGroupAuthority(TEST_ISSUER, "/test")))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .build();

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(oauthTestPolicy, denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(DENY));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(denyAllPolicy));
  }

  @Test
  void oauthGroupHolderPolicyNotAppliedDueToWrongGroup() {

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtGroupAuthority(TEST_ISSUER, "/test/subgroup")));

    PathAuthorizationPolicy oauthTestPolicy =
        PathAuthorizationPolicy.builder()
            .withDescription("Allow GET on /test/** to members of /test group")
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new JwtGroupAuthority(TEST_ISSUER, "/test")))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .build();

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDescription("Deny all")
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(oauthTestPolicy, denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    when(request.getMethod()).thenReturn("GET");
    when(request.getRequestURI()).thenReturn("/test/ciccio");

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(DENY));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(denyAllPolicy));
  }

  @Test
  void oauthGroupHolderPolicyNotAppliedDueToWrongIssuer() {

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtGroupAuthority(TEST2_ISSUER, "/test")));

    PathAuthorizationPolicy oauthTestPolicy =
        PathAuthorizationPolicy.builder()
            .withDescription("Allow GET on /test/** to members of /test group")
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new JwtGroupAuthority(TEST_ISSUER, "/test")))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .build();

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(oauthTestPolicy, denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);
    when(request.getMethod()).thenReturn("GET");
    when(request.getRequestURI()).thenReturn("/test/ciccio");

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(DENY));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(denyAllPolicy));
  }

  @Test
  void oauthGroupHolderPolicyApplied() {

    when(request.getRequestURI()).thenReturn("/test/file0");
    when(request.getMethod()).thenReturn("GET");

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtGroupAuthority(TEST_ISSUER, "/test")));

    PathAuthorizationPolicy oauthTestPolicy =
        PathAuthorizationPolicy.builder()
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new JwtGroupAuthority(TEST_ISSUER, "/test")))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .build();

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(oauthTestPolicy, denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(oauthTestPolicy));
  }

  @Test
  void oauthClientHolderPolicyApplied() {

    when(request.getRequestURI()).thenReturn("/test/file0");
    when(request.getMethod()).thenReturn("GET");

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtClientAuthority(TEST_ISSUER, AUTHORIZED_JWT_CLIENT_ID)));

    PathAuthorizationPolicy oauthTestPolicy =
        PathAuthorizationPolicy.builder()
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(
                    new JwtClientAuthority(TEST_ISSUER, AUTHORIZED_JWT_CLIENT_ID)))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .build();

    PathAuthorizationPolicy denyAllPolicy =
        PathAuthorizationPolicy.builder()
            .withDeny()
            .withPrincipalMatcher(new Anyone())
            .withRequestMatcher(PathPatternRequestMatcher.withDefaults().matcher("/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(oauthTestPolicy, denyAllPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(oauthTestPolicy));

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtClientAuthority(TEST_ISSUER, UNAUTHORIZED_JWT_CLIENT_ID)));

    result = pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(DENY));
  }

  @Test
  void multiplePrincipalMatchersWorkAsExpected() {

    when(request.getRequestURI()).thenReturn("/test/file0");
    when(request.getMethod()).thenReturn("GET");

    PathAuthorizationPolicy multiplePrincipalsPolicy =
        PathAuthorizationPolicy.builder()
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new JwtGroupAuthority(TEST_ISSUER, "/test")))
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new VOMSFQANAuthority("/test/example")))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(multiplePrincipalsPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtGroupAuthority(TEST_ISSUER, "/test")));

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(multiplePrincipalsPolicy));

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtGroupAuthority(TEST_ISSUER, "/other")));

    result = pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(NOT_APPLICABLE));

    when(authentication.getAuthorities())
        .thenReturn(
            authorities(
                new VOMSVOAuthority("test"),
                new VOMSFQANAuthority("/test"),
                new VOMSFQANAuthority("/test/example")));

    result = pdp.authorizeRequest(newAuthorizationRequest(request, authentication));

    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(multiplePrincipalsPolicy));

    when(authentication.getAuthorities())
        .thenReturn(authorities(new SimpleGrantedAuthority("ANONYMOUS")));

    result = pdp.authorizeRequest(newAuthorizationRequest(request, authentication));

    assertThat(result.getDecision(), is(NOT_APPLICABLE));
  }

  @Test
  void multiplePathsMatchersWorkAsExpected() {

    when(request.getRequestURI()).thenReturn("/test/file0");
    when(request.getMethod()).thenReturn("GET");

    PathAuthorizationPolicy multiplePathsPolicy =
        PathAuthorizationPolicy.builder()
            .withPermit()
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new JwtGroupAuthority(TEST_ISSUER, "/test")))
            .withPrincipalMatcher(
                AuthorityHolder.fromAuthority(new VOMSFQANAuthority("/test/example")))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/test/**"))
            .withRequestMatcher(
                PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.GET, "/other/**"))
            .build();

    List<PathAuthorizationPolicy> policies = List.of(multiplePathsPolicy);
    when(repo.getPolicies()).thenReturn(policies);

    when(authentication.getAuthorities())
        .thenReturn(authorities(new JwtGroupAuthority(TEST_ISSUER, "/test")));

    PathAuthorizationResult result =
        pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(multiplePathsPolicy));

    when(request.getRequestURI()).thenReturn("/other/file0");

    result = pdp.authorizeRequest(newAuthorizationRequest(request, authentication));
    assertThat(result.getDecision(), is(PERMIT));
    assertThat(result.getPolicy().isPresent(), is(true));
    assertThat(result.getPolicy().get(), is(multiplePathsPolicy));

    when(request.getRequestURI()).thenReturn("/yet-another");
    result = pdp.authorizeRequest(newAuthorizationRequest(request, authentication));

    assertThat(result.getDecision(), is(NOT_APPLICABLE));
  }
}
