// SPDX-FileCopyrightText: 2014 Istituto Nazionale di Fisica Nucleare
//
// SPDX-License-Identifier: Apache-2.0

package org.italiangrid.storm.webdav.spring.web;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import org.italiangrid.storm.webdav.authn.ErrorPageAuthenticationEntryPoint;
import org.italiangrid.storm.webdav.authn.PrincipalHelper;
import org.italiangrid.storm.webdav.authz.SAPermission;
import org.italiangrid.storm.webdav.authz.VOMSAuthenticationFilter;
import org.italiangrid.storm.webdav.authz.VOMSAuthenticationProvider;
import org.italiangrid.storm.webdav.authz.managers.ConsensusBasedManager;
import org.italiangrid.storm.webdav.authz.managers.FineGrainedAuthzManager;
import org.italiangrid.storm.webdav.authz.managers.FineGrainedCopyMoveAuthzManager;
import org.italiangrid.storm.webdav.authz.managers.LocalAuthzManager;
import org.italiangrid.storm.webdav.authz.managers.MacaroonAuthzManager;
import org.italiangrid.storm.webdav.authz.managers.UnanimousDelegatedManager;
import org.italiangrid.storm.webdav.authz.managers.WlcgScopeAuthzCopyMoveManager;
import org.italiangrid.storm.webdav.authz.managers.WlcgScopeAuthzManager;
import org.italiangrid.storm.webdav.authz.pdp.LocalAuthorizationPdp;
import org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationPdp;
import org.italiangrid.storm.webdav.authz.pdp.WlcgStructuredPathAuthorizationPdp;
import org.italiangrid.storm.webdav.authz.util.ReadonlyHttpMethodMatcher;
import org.italiangrid.storm.webdav.config.OAuthProperties;
import org.italiangrid.storm.webdav.config.ServiceConfigurationProperties;
import org.italiangrid.storm.webdav.config.StorageAreaConfiguration;
import org.italiangrid.storm.webdav.config.StorageAreaInfo;
import org.italiangrid.storm.webdav.oauth.StormJwtAuthenticationConverter;
import org.italiangrid.storm.webdav.server.PathResolver;
import org.italiangrid.storm.webdav.server.servlet.PreAuthenticatedFilter;
import org.italiangrid.storm.webdav.server.servlet.WebDAVMethod;
import org.italiangrid.storm.webdav.tpc.LocalURLService;
import org.italiangrid.storm.webdav.web.PathConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.web.error.ErrorPage;
import org.springframework.boot.web.error.ErrorPageRegistrar;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.InsufficientAuthenticationException;
import org.springframework.security.authorization.AuthorizationManager;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.HeadersConfigurer.HstsConfig;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.access.AccessDeniedHandlerImpl;
import org.springframework.security.web.access.expression.WebExpressionAuthorizationManager;
import org.springframework.security.web.access.intercept.RequestAuthorizationContext;
import org.springframework.security.web.authentication.AnonymousAuthenticationFilter;
import org.springframework.security.web.authentication.logout.LogoutFilter;
import org.springframework.security.web.firewall.HttpFirewall;
import org.springframework.security.web.firewall.RequestRejectedException;
import org.springframework.security.web.firewall.RequestRejectedHandler;
import org.springframework.security.web.firewall.StrictHttpFirewall;
import org.springframework.web.filter.ForwardedHeaderFilter;

@Configuration
@EnableMethodSecurity(proxyTargetClass = true)
public class SecurityConfig {

  private static final Logger LOG = LoggerFactory.getLogger(SecurityConfig.class);

  private static final List<String> ALLOWED_METHODS;

  static {
    ALLOWED_METHODS = new ArrayList<>();
    ALLOWED_METHODS.add(HttpMethod.HEAD.name());
    ALLOWED_METHODS.add(HttpMethod.GET.name());
    ALLOWED_METHODS.add(HttpMethod.POST.name());
    ALLOWED_METHODS.add(HttpMethod.PUT.name());
    ALLOWED_METHODS.add(HttpMethod.DELETE.name());
    ALLOWED_METHODS.add(HttpMethod.OPTIONS.name());
    ALLOWED_METHODS.add(HttpMethod.PATCH.name());
    ALLOWED_METHODS.add(WebDAVMethod.PROPFIND.name());
    ALLOWED_METHODS.add(WebDAVMethod.PROPPATCH.name());
    ALLOWED_METHODS.add(WebDAVMethod.MKCOL.name());
    ALLOWED_METHODS.add(WebDAVMethod.COPY.name());
    ALLOWED_METHODS.add(WebDAVMethod.MOVE.name());
    ALLOWED_METHODS.add(WebDAVMethod.LOCK.name());
    ALLOWED_METHODS.add(WebDAVMethod.UNLOCK.name());
  }

  @Autowired OAuthProperties oauthProperties;

  @Autowired StorageAreaConfiguration saConfiguration;

  @Autowired ServiceConfigurationProperties serviceConfigurationProperties;

  @Autowired PathResolver pathResolver;

  @Autowired
  @Qualifier("vomsAuthenticationFilter")
  VOMSAuthenticationFilter vomsFilter;

  @Autowired LocalURLService localURLService;

  @Autowired PathAuthorizationPdp fineGrainedAuthzPdp;

  @Autowired PrincipalHelper principalHelper;

  @Bean
  HttpFirewall allowWebDAVMethodsFirewall() {

    StrictHttpFirewall firewall = new StrictHttpFirewall();
    firewall.setAllowedHttpMethods(ALLOWED_METHODS);
    return firewall;
  }

  @Bean
  SecurityFilterChain filterChain(
      HttpSecurity http,
      VOMSAuthenticationProvider vomsProvider,
      StormJwtAuthenticationConverter authConverter,
      @Value("${storm.nginx.enabled}") boolean nginxEnabled,
      PrincipalHelper helper) {

    if (nginxEnabled) {
      http.addFilterBefore(new ForwardedHeaderFilter(), LogoutFilter.class);
    }
    http.addFilterAfter(
        new PreAuthenticatedFilter(nginxEnabled, helper), AnonymousAuthenticationFilter.class);
    http.authenticationProvider(vomsProvider).addFilter(vomsFilter);

    if (serviceConfigurationProperties.getAuthz().isDisabled()) {
      LOG.warn("AUTHORIZATION DISABLED: this shouldn't be used in production!");
      http.authorizeHttpRequests(authorize -> authorize.anyRequest().permitAll());
    } else {
      addAccessRules(http);
      addAnonymousAccessRules(http);
    }

    if (serviceConfigurationProperties.getRedirector().isEnabled()) {
      http.headers(headers -> headers.httpStrictTransportSecurity(HstsConfig::disable));
    }

    http.oauth2ResourceServer(
        oauth2 -> oauth2.jwt(jwt -> jwt.jwtAuthenticationConverter(authConverter)));

    http.authorizeHttpRequests(
        authorize ->
            authorize
                .requestMatchers(
                    "/",
                    "/robots.txt",
                    PathConstants.ASSETS_PATH + "/**",
                    PathConstants.AUTHN_INFO_PATH,
                    PathConstants.ACTUATOR_PATH + "/**",
                    PathConstants.ERRORS_PATH + "/*",
                    "/status/metrics",
                    PathConstants.OAUTH_TOKEN_PATH,
                    "/.well-known/oauth-authorization-server",
                    "/.well-known/openid-configuration",
                    "/.well-known/wlcg-tape-rest-api")
                .permitAll());

    configureOidcAuthn(http);

    if (!serviceConfigurationProperties.getAuthz().isDisabled()) {
      http.authorizeHttpRequests(
          authorize -> authorize.anyRequest().access(fineGrainedAuthorizationManager(null)));
    }

    AccessDeniedHandlerImpl handler = new AccessDeniedHandlerImpl();
    handler.setErrorPage(PathConstants.ERRORS_PATH + "/403");

    http.logout(
        logout ->
            logout
                .logoutUrl(PathConstants.LOGOUT_PATH)
                .clearAuthentication(true)
                .invalidateHttpSession(true)
                .logoutSuccessUrl("/"));

    if (!oauthProperties.isEnableOidc()) {
      http.exceptionHandling(
          exception -> exception.authenticationEntryPoint(new ErrorPageAuthenticationEntryPoint()));
    }

    http.csrf(csrf -> csrf.disable());
    http.cors(cors -> cors.disable());

    return http.build();
  }

  @Bean
  static ErrorPageRegistrar securityErrorPageRegistrar() {
    return r -> {
      r.addErrorPages(
          new ErrorPage(RequestRejectedException.class, PathConstants.ERRORS_PATH + "/400"));
      r.addErrorPages(
          new ErrorPage(
              InsufficientAuthenticationException.class, PathConstants.ERRORS_PATH + "/401"));
      r.addErrorPages(new ErrorPage(HttpStatus.BAD_REQUEST, PathConstants.ERRORS_PATH + "/400"));
      r.addErrorPages(new ErrorPage(HttpStatus.UNAUTHORIZED, PathConstants.ERRORS_PATH + "/401"));
      r.addErrorPages(new ErrorPage(HttpStatus.FORBIDDEN, PathConstants.ERRORS_PATH + "/403"));
      r.addErrorPages(new ErrorPage(HttpStatus.NOT_FOUND, PathConstants.ERRORS_PATH + "/404"));
      r.addErrorPages(
          new ErrorPage(HttpStatus.METHOD_NOT_ALLOWED, PathConstants.ERRORS_PATH + "/405"));
      r.addErrorPages(
          new ErrorPage(HttpStatus.UNSUPPORTED_MEDIA_TYPE, PathConstants.ERRORS_PATH + "/415"));
    };
  }

  @Bean
  RequestRejectedHandler requestRejectedHandler() {
    return new HttpMethodRequestRejectedHandler(ALLOWED_METHODS);
  }

  protected void addAnonymousAccessRules(HttpSecurity http) {
    final List<GrantedAuthority> anonymousAccessPermissions = new ArrayList<>();

    for (StorageAreaInfo sa : saConfiguration.getStorageAreaInfo()) {
      if (sa.anonymousReadEnabled()) {
        anonymousAccessPermissions.add(SAPermission.canRead(sa.name()));
      }
    }

    if (!anonymousAccessPermissions.isEmpty()) {
      http.anonymous(anonymous -> anonymous.authorities(anonymousAccessPermissions));
    }
  }

  protected void configureOidcAuthn(HttpSecurity http) {
    if (oauthProperties.isEnableOidc()) {
      http.authorizeHttpRequests(
          authorize -> authorize.requestMatchers(PathConstants.OIDC_LOGIN_PATH).permitAll());
      http.oauth2Login(oauth2Login -> oauth2Login.loginPage(PathConstants.OIDC_LOGIN_PATH));
    }
  }

  protected void addAccessRules(HttpSecurity http) {

    Map<String, String> accessPoints = new TreeMap<>(Comparator.reverseOrder());
    saConfiguration
        .getStorageAreaInfo()
        .forEach(sa -> sa.accessPoints().forEach(ap -> accessPoints.put(ap, sa.name())));
    for (Entry<String, String> e : accessPoints.entrySet()) {
      String ap = e.getKey();
      String sa = e.getValue();
      LOG.debug("Evaluating access rules for access-point '{}' and storage area '{}'", ap, sa);
      String writeAccessRule =
          String.format(
              "hasAuthority('%s') and hasAuthority('%s')",
              SAPermission.canRead(sa).getAuthority(), SAPermission.canWrite(sa).getAuthority());
      LOG.debug("Write access rule: {}", writeAccessRule);
      String readAccessRule =
          String.format("hasAuthority('%s')", SAPermission.canRead(sa).getAuthority());
      LOG.debug("Read access rule: {}", readAccessRule);
      http.authorizeHttpRequests(
          authorize ->
              authorize
                  .requestMatchers(new ReadonlyHttpMethodMatcher(ap + "/**"))
                  .access(
                      fineGrainedAuthorizationManager(
                          new WebExpressionAuthorizationManager(readAccessRule))));

      http.authorizeHttpRequests(
          authorize ->
              authorize
                  .requestMatchers(ap + "/**")
                  .access(
                      fineGrainedAuthorizationManager(
                          new WebExpressionAuthorizationManager(writeAccessRule))));
    }
  }

  protected AuthorizationManager<RequestAuthorizationContext> fineGrainedAuthorizationManager(
      WebExpressionAuthorizationManager webExpressionAuthorizationManager) {
    List<AuthorizationManager<RequestAuthorizationContext>> voters = new ArrayList<>();

    UnanimousDelegatedManager fineGrainedVoters =
        UnanimousDelegatedManager.forVoters(
            "FineGrainedAuthz",
            Arrays.asList(
                new FineGrainedAuthzManager(
                    serviceConfigurationProperties,
                    pathResolver,
                    fineGrainedAuthzPdp,
                    localURLService),
                new FineGrainedCopyMoveAuthzManager(
                    serviceConfigurationProperties,
                    pathResolver,
                    fineGrainedAuthzPdp,
                    localURLService)));

    WlcgStructuredPathAuthorizationPdp wlcgPdp =
        new WlcgStructuredPathAuthorizationPdp(
            serviceConfigurationProperties, pathResolver, localURLService);

    UnanimousDelegatedManager wlcgVoters =
        UnanimousDelegatedManager.forVoters(
            "WLCGScopeBasedAuthz",
            Arrays.asList(
                new WlcgScopeAuthzManager(
                    serviceConfigurationProperties, pathResolver, wlcgPdp, localURLService),
                new WlcgScopeAuthzCopyMoveManager(
                    serviceConfigurationProperties, pathResolver, wlcgPdp, localURLService)));

    if (serviceConfigurationProperties.getRedirector().isEnabled()) {
      voters.add(
          new LocalAuthzManager(
              serviceConfigurationProperties,
              pathResolver,
              new LocalAuthorizationPdp(serviceConfigurationProperties),
              localURLService));
    }
    if (serviceConfigurationProperties.getMacaroonFilter().isEnabled()) {
      voters.add(new MacaroonAuthzManager());
    }
    if (webExpressionAuthorizationManager != null) {
      voters.add(webExpressionAuthorizationManager);
    }
    voters.add(fineGrainedVoters);
    voters.add(wlcgVoters);
    return new ConsensusBasedManager("Consensus", voters);
  }
}
