// SPDX-FileCopyrightText: 2014 Istituto Nazionale di Fisica Nucleare
//
// SPDX-License-Identifier: Apache-2.0

package org.italiangrid.storm.webdav.config;

import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import org.italiangrid.storm.webdav.authz.PathAuthzPolicyParser;
import org.italiangrid.storm.webdav.authz.VOMSFQANAuthority;
import org.italiangrid.storm.webdav.authz.VOMSVOAuthority;
import org.italiangrid.storm.webdav.authz.VOMSVOMapAuthority;
import org.italiangrid.storm.webdav.authz.X509SubjectAuthority;
import org.italiangrid.storm.webdav.authz.pdp.PathAuthorizationPolicy;
import org.italiangrid.storm.webdav.authz.pdp.principal.AnonymousUser;
import org.italiangrid.storm.webdav.authz.pdp.principal.AnyAuthenticatedUser;
import org.italiangrid.storm.webdav.authz.pdp.principal.Anyone;
import org.italiangrid.storm.webdav.authz.pdp.principal.AuthorityHolder;
import org.italiangrid.storm.webdav.authz.pdp.principal.PrincipalMatcher;
import org.italiangrid.storm.webdav.authz.pdp.principal.PrincipalMatcherDebugWrapper;
import org.italiangrid.storm.webdav.authz.util.CustomHttpMethodMatcher;
import org.italiangrid.storm.webdav.authz.util.ReadonlyHttpMethodMatcher;
import org.italiangrid.storm.webdav.authz.util.WriteHttpMethodMatcher;
import org.italiangrid.storm.webdav.config.FineGrainedAuthzPolicyProperties.Action;
import org.italiangrid.storm.webdav.config.FineGrainedAuthzPolicyProperties.PrincipalProperties;
import org.italiangrid.storm.webdav.config.FineGrainedAuthzPolicyProperties.PrincipalProperties.PrincipalType;
import org.italiangrid.storm.webdav.oauth.authority.JwtClientAuthority;
import org.italiangrid.storm.webdav.oauth.authority.JwtGroupAuthority;
import org.italiangrid.storm.webdav.oauth.authority.JwtIssuerAuthority;
import org.italiangrid.storm.webdav.oauth.authority.JwtScopeAuthority;
import org.italiangrid.storm.webdav.oauth.authority.JwtSubjectAuthority;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.security.web.servlet.util.matcher.PathPatternRequestMatcher;
import org.springframework.security.web.util.matcher.AndRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.stereotype.Service;

@Service
public class FineGrainedAuthzPolicyParser implements PathAuthzPolicyParser {

  final ServiceConfigurationProperties properties;
  final StorageAreaConfiguration saConfig;

  @Autowired
  public FineGrainedAuthzPolicyParser(
      ServiceConfigurationProperties properties, StorageAreaConfiguration saConfig) {
    this.properties = properties;
    this.saConfig = saConfig;
  }

  Supplier<IllegalArgumentException> unknownStorageArea(String saName) {
    return () -> new IllegalArgumentException("Unknown storage area: " + saName);
  }

  StorageAreaInfo getStorageAreaInfo(String saName) {
    return saConfig.getStorageAreaInfo().stream()
        .filter(sa -> sa.name().equals(saName))
        .findAny()
        .orElseThrow(unknownStorageArea(saName));
  }

  String matcherPath(String accessPoint, String path) {
    String jointPath = String.format("%s/%s", accessPoint, path);
    return jointPath.replaceAll("\\/+", "/");
  }

  Supplier<RequestMatcher> matcherByActionSupplier(Action a, String pattern) {
    return () -> {
      if (Action.ALL.equals(a)) {
        return PathPatternRequestMatcher.withDefaults().matcher(pattern);
      } else if (Action.READ.equals(a)) {
        return new ReadonlyHttpMethodMatcher(pattern);
      } else if (Action.WRITE.equals(a)) {
        return new WriteHttpMethodMatcher(pattern);
      } else if (Action.DELETE.equals(a)) {
        return PathPatternRequestMatcher.withDefaults().matcher(HttpMethod.DELETE, pattern);
      } else if (Action.LIST.equals(a)) {
        return new AndRequestMatcher(
            new CustomHttpMethodMatcher(Set.of("PROPFIND")),
            PathPatternRequestMatcher.withDefaults().matcher(pattern));
      } else {
        throw new IllegalArgumentException("Unknown action: " + a);
      }
    };
  }

  void parseAction(
      Action a,
      FineGrainedAuthzPolicyProperties policyProperties,
      PathAuthorizationPolicy.Builder builder) {

    List<String> paths = policyProperties.getPaths();

    for (String ap : getStorageAreaInfo(policyProperties.getSa()).accessPoints()) {
      if (policyProperties.getPaths().isEmpty()) {
        builder.withRequestMatcher(matcherByActionSupplier(a, ap + "/**").get());
      } else {
        paths.forEach(
            p -> builder.withRequestMatcher(matcherByActionSupplier(a, matcherPath(ap, p)).get()));
      }
    }
  }

  PrincipalMatcher parsePrincipal(PrincipalProperties p) {
    PrincipalMatcher matcher = null;

    if (PrincipalType.ANONYMOUS.equals(p.getType())) {
      matcher = new AnonymousUser();
    } else if (PrincipalType.ANY_AUTHENTICATED_USER.equals(p.getType())) {
      matcher = new AnyAuthenticatedUser();
    } else if (PrincipalType.ANYONE.equals(p.getType())) {
      matcher = new Anyone();
    } else if (PrincipalType.FQAN.equals(p.getType())) {
      matcher = AuthorityHolder.fromAuthority(new VOMSFQANAuthority(p.getParams().get("fqan")));
    } else if (PrincipalType.JWT_GROUP.equals(p.getType())) {
      matcher =
          AuthorityHolder.fromAuthority(
              new JwtGroupAuthority(p.getParams().get("iss"), p.getParams().get("group")));
    } else if (PrincipalType.JWT_SCOPE.equals(p.getType())) {
      matcher =
          AuthorityHolder.fromAuthority(
              new JwtScopeAuthority(p.getParams().get("iss"), p.getParams().get("scope")));
    } else if (PrincipalType.JWT_ISSUER.equals(p.getType())) {
      matcher = AuthorityHolder.fromAuthority(new JwtIssuerAuthority(p.getParams().get("iss")));
    } else if (PrincipalType.JWT_SUBJECT.equals(p.getType())) {
      matcher =
          AuthorityHolder.fromAuthority(
              new JwtSubjectAuthority(p.getParams().get("iss"), p.getParams().get("sub")));
    } else if (PrincipalType.JWT_CLIENT.equals(p.getType())) {
      matcher =
          AuthorityHolder.fromAuthority(
              new JwtClientAuthority(p.getParams().get("iss"), p.getParams().get("id")));
    } else if (PrincipalType.VO.equals(p.getType())) {
      matcher = AuthorityHolder.fromAuthority(new VOMSVOAuthority(p.getParams().get("vo")));
    } else if (PrincipalType.VO_MAP.equals(p.getType())) {
      matcher = AuthorityHolder.fromAuthority(new VOMSVOMapAuthority(p.getParams().get("vo")));
    } else if (PrincipalType.X509_SUBJECT.equals(p.getType())) {
      matcher =
          AuthorityHolder.fromAuthority(new X509SubjectAuthority(p.getParams().get("subject")));
    }
    return new PrincipalMatcherDebugWrapper(matcher);
  }

  PathAuthorizationPolicy parsePolicy(FineGrainedAuthzPolicyProperties policy) {

    PathAuthorizationPolicy.Builder builder = PathAuthorizationPolicy.builder();

    builder
        .withId(UUID.randomUUID().toString())
        .withSa(policy.getSa())
        .withEffect(policy.getEffect())
        .withDescription(policy.getDescription());

    policy.getActions().forEach(a -> parseAction(a, policy, builder));

    policy.getPrincipals().stream()
        .map(this::parsePrincipal)
        .forEach(builder::withPrincipalMatcher);

    return builder.build();
  }

  @Override
  public List<PathAuthorizationPolicy> parsePolicies() {

    return properties.getAuthz().getPolicies().stream().map(this::parsePolicy).toList();
  }
}
