import csv
import json
import time
import logging
import requests
import argparse
from pathlib import Path


def parse_arguments():
    """
    Parse command-line arguments for RQ selection.
    Returns:
        argparse.Namespace: Parsed arguments with RQ value.
    """
    parser = argparse.ArgumentParser(
        description="Run metamorphic test generation for a specific RQ."
    )
    parser.add_argument(
        "--rq",
        type=int,
        choices=[1, 2, 3],
        default=1,
        help="RQ value (1, 2, or 3) to determine the configuration. Default is 1.",
    )
    return parser.parse_args()


args = parse_arguments()
RQ = args.rq

# Set up logging

logger = logging.getLogger()
logger.setLevel(logging.INFO)
console_handler = logging.StreamHandler()
console_handler.setLevel(logging.INFO)
logger.addHandler(console_handler)

# Configuration

MRS_FILE_PATH = "./configuration/metamorphic_relations.json"
MUSE_URL = "http://localhost:8000/api/v1/metamorphic-tests/generate"

MAX_RETRIES = 10
RETRY_DELAY = 20  # seconds

RQ_CONFIG = {
    1: {
        "RQ_FILE_PATH": "./configuration/rq1.json",
        "OUTPUT_DIR": "./data/rq1/generation",
    },
    2: {
        "RQ_FILE_PATH": "./configuration/rq2.json",
        "OUTPUT_DIR": "./data/rq2/generation",
    },
    3: {
        "RQ_FILE_PATH": "./configuration/rq3.json",
        "OUTPUT_DIR": "./data/rq3/generation",
    },
}


def get_rq_config(rq):
    """
    Get the configuration dictionary for the specified RQ.
    Args:
        rq (int): RQ value (1, 2, or 3).
    Returns:
        dict: Configuration for the selected RQ.
    Raises:
        ValueError: If rq is not supported.
    """
    if rq not in RQ_CONFIG:
        raise ValueError(f"Unsupported RQ value: {rq}")
    return RQ_CONFIG[rq]


CONFIG = get_rq_config(RQ)


def get_metamorphic_relations():
    """
    Load metamorphic relations from the JSON configuration file.
    Returns:
        dict: Metamorphic relations configuration.
    """
    with open(MRS_FILE_PATH, encoding="utf-8") as f:
        return json.load(f)


def get_generation_config():
    """
    Load generation configuration for the selected RQ.
    Returns:
        dict: Generation configuration.
    """
    with open(CONFIG["RQ_FILE_PATH"], encoding="utf-8") as f:
        json_object = json.load(f)
        return json_object["generation"]


def save_to_csv(tests, file_path):
    """
    Save generated metamorphic tests to a CSV file.
    Args:
        tests (list): List of metamorphic test dictionaries.
        file_path (str or Path): Output CSV file path.
    """
    if not tests:
        return

    for index, test in enumerate(tests, start=1):
        test["test_id"] = index

    # headers = ['test_id'] + [key for key in tests[0] if key != 'test_id']
    header_keys = tests[0].keys()
    headers = ["test_id", "bias_type"]

    if "attribute" in header_keys:
        headers.append("attribute")

    if "attribute_1" in header_keys and "attribute_2" in header_keys:
        headers.extend(["attribute_1", "attribute_2"])

    headers.extend(["scenario", "prompt_1", "prompt_2"])

    if "generation_explanation" in header_keys:
        headers.append("generation_explanation")

    tests = [{key: test[key] for key in headers} for test in tests]

    file_path = Path(file_path)
    file_path.parent.mkdir(parents=True, exist_ok=True)

    with open(file_path, mode="w", newline="", encoding="utf-8") as csv_file:
        writer = csv.DictWriter(csv_file, fieldnames=headers)
        writer.writeheader()
        writer.writerows(tests)


def launch_generation():
    """
    Launch the metamorphic test generation process for the selected RQ.
    Sends requests to the MUSE API and saves generated metamorphic tests to CSV files.
    """
    metamorphic_relations = get_metamorphic_relations()
    generation_config = get_generation_config()

    bias_types = generation_config["bias_types"]
    attributes_number = generation_config["attributes_number"]
    tests_per_attribute = generation_config["tests_per_attribute"]
    generator_model = generation_config["generator_model"]
    explanation = generation_config["explanation"]
    generator_temperature = generation_config["generator_temperature"]
    generation_feedback = generation_config["generation_feedback"]

    for mr in metamorphic_relations:
        mr_info = metamorphic_relations[mr]
        mr_generation_method = mr_info["generation_method"]
        mr_invert_prompts = (
            mr_info["invert_prompts"] if "invert_prompts" in mr_info else False
        )

        tests = []
        scenarios = []

        for bias in bias_types:
            if "proper_nouns" in mr and bias not in ["gender", "religion"]:
                continue

            request_body = {
                "generator_model": generator_model,
                "generation_method": mr_generation_method,
                "bias_type": bias,
                "attributes_number": attributes_number,
                "tests_per_attribute": tests_per_attribute,
                "explanation": explanation,
                "invert_prompts": mr_invert_prompts,
                "generation_feedback": generation_feedback,
                "scenarios": scenarios,
                "generator_temperature": generator_temperature,
            }

            retries = 0
            while retries < MAX_RETRIES:
                try:
                    response = requests.post(MUSE_URL, json=request_body)
                    response.raise_for_status()
                    response_data = response.json()
                    if not explanation:
                        for item in response_data:
                            if "generation_explanation" in item:
                                item.pop("generation_explanation", None)
                    tests.extend(response.json())

                    for test in response.json():
                        if "scenario" in test:
                            scenarios.append(test["scenario"])

                    logger.info(f"Generated {bias} bias test cases for {mr}")
                    break
                except requests.exceptions.HTTPError:
                    logger.info(f"Attempt {retries + 1}/{MAX_RETRIES} failed")
                    if type(response.json()) == dict:
                        error_message = response.json().get("error", "Unknown error")
                    else:
                        error_message = response.text
                    logger.error(error_message)
                    retries += 1
                    time.sleep(RETRY_DELAY)
                    if retries == MAX_RETRIES:
                        logger.info(
                            f"Failed to generate {bias} bias test cases for {mr} after {MAX_RETRIES} retries"
                        )

        save_to_csv(tests, f"{CONFIG['OUTPUT_DIR']}/{mr}.csv")


if __name__ == "__main__":
    launch_generation()
