import csv
import json
import time
import logging
import requests
import argparse
from pathlib import Path


def parse_arguments():
    """
    Parse command-line arguments for RQ selection.
    Returns:
        argparse.Namespace: Parsed arguments with RQ value.
    """
    parser = argparse.ArgumentParser(
        description="Run metamorphic test execution for a specific RQ."
    )
    parser.add_argument(
        "--rq",
        type=int,
        choices=[1, 2, 3],
        default=1,
        help="RQ value (1, 2, or 3) to determine the configuration. Default is 1.",
    )
    return parser.parse_args()


args = parse_arguments()
RQ = args.rq

# Set up logging

logger = logging.getLogger()
logger.setLevel(logging.INFO)
console_handler = logging.StreamHandler()
console_handler.setLevel(logging.INFO)
logger.addHandler(console_handler)

# Configuration

MRS_FILE_PATH = "./configuration/metamorphic_relations.json"
GENIE_URL = "http://localhost:8081/api/v1/metamorphic-tests/execute"

MAX_RETRIES = 10
RETRY_DELAY = 20  # seconds

RQ_CONFIG = {
    1: {
        "RQ_FILE_PATH": "./configuration/rq1.json",
        "INPUT_DIR": "./data/rq1/generation",
        "OUTPUT_DIR": "./data/rq1/execution",
    },
    2: {
        "RQ_FILE_PATH": "./configuration/rq2.json",
        "INPUT_DIR": "./data/rq2/generation",
        "OUTPUT_DIR": "./data/rq2/execution",
    },
    3: {
        "RQ_FILE_PATH": "./configuration/rq3.json",
        "INPUT_DIR": "./data/rq3/generation",
        "OUTPUT_DIR": "./data/rq3/execution",
    },
}


def load_json(file_path):
    """
    Load a JSON file from the given path.
    Args:
        file_path (str or Path): Path to the JSON file.
    Returns:
        dict: Parsed JSON content.
    """
    with open(file_path, encoding="utf-8") as f:
        return json.load(f)


def get_config(rq):
    """
    Get the configuration dictionary for the specified RQ.
    Args:
        rq (int): RQ value (1, 2, or 3).
    Returns:
        dict: Configuration for the selected RQ.
    Raises:
        ValueError: If rq is not supported.
    """
    if rq not in RQ_CONFIG:
        raise ValueError(f"Unsupported RQ value: {rq}")
    return RQ_CONFIG[rq]


CONFIG = get_config(RQ)


def save_to_csv(executions, file_path):
    """
    Save executed metamorphic tests to a CSV file.
    Args:
        executions (list): List of execution dictionaries.
        file_path (str or Path): Output CSV file path.
    """
    if not executions:
        return

    header_keys = executions[0].keys()
    headers = ["test_id", "bias_type"]

    if "attribute" in header_keys:
        headers.append("attribute")

    if "attribute_1" in header_keys and "attribute_2" in header_keys:
        headers.extend(["attribute_1", "attribute_2"])

    headers.extend(["scenario", "prompt_1", "response_1", "prompt_2", "response_2"])

    if "generation_explanation" in header_keys:
        headers.append("generation_explanation")

    headers.extend(["execution_timestamp"])

    executions = [{key: execution[key] for key in headers} for execution in executions]

    file_path = Path(file_path)
    file_path.parent.mkdir(parents=True, exist_ok=True)

    with open(file_path, mode="w", newline="", encoding="utf-8") as csv_file:
        writer = csv.DictWriter(csv_file, fieldnames=headers)
        writer.writeheader()
        writer.writerows(executions)


def execute_request(request_body, test_id, mr):
    """
    Execute a single metamorphic test by sending a request to the GENIE API.
    Args:
        request_body (dict): Request payload for the API.
        test_id (int): Metamorphic test identifier.
        mr (str): Metamorphic relation name.
    Returns:
        dict or None: Execution result dictionary, or None if failed.
    """
    retries = 0
    while retries < MAX_RETRIES:
        try:
            response = requests.post(GENIE_URL, json=request_body)
            execution_timestamp = time.time()
            response.raise_for_status()
            response_data = response.json()

            res = {
                **request_body,
                "response_1": response_data["response_1"],
                "response_2": response_data["response_2"],
                "execution_timestamp": execution_timestamp,
                "test_id": test_id,
            }

            if res["type"] == "consistency":
                res["prompt_2"] = response_data["prompt_2"]

            return res
        except requests.exceptions.HTTPError:
            logger.error(response.text)
            retries += 1
            time.sleep(RETRY_DELAY)
            if retries == MAX_RETRIES:
                logger.info(
                    f"Failed to execute test {test_id} for {mr} after {MAX_RETRIES} retries"
                )


def process_metamorphic_tests(
    metamorphic_relations,
    execution_config,
    multiple_models=False,
    multiple_executions=False,
):
    """
    Process and execute all metamorphic tests for the given configuration.
    Args:
        metamorphic_relations (dict): Metamorphic relations configuration.
        execution_config (dict): Execution configuration.
        multiple_models (bool): Whether to process multiple models.
        multiple_executions (bool): Whether to process multiple executions per metamorphic test.
    """
    models = (
        execution_config["models_under_test"]
        if multiple_models
        else [execution_config["model_under_test"]]
    )
    n_executions = execution_config.get("executions", 1) if multiple_executions else 1

    for model in models:
        for mr, mr_info in metamorphic_relations.items():
            for i in range(n_executions):
                executions = []
                input_file = Path(CONFIG["INPUT_DIR"]) / f"{mr}.csv"

                with open(input_file, encoding="utf-8") as f:
                    reader = csv.DictReader(f)
                    for row in reader:
                        request_body = {
                            "model_name": model,
                            "list_format_response": mr_info["list_format_response"]
                            if "list_format_response" in mr_info
                            else False,
                            "numeric_format_response": mr_info[
                                "numeric_format_response"
                            ]
                            if "numeric_format_response" in mr_info
                            else False,
                            "yes_no_format_response": mr_info["yes_no_format_response"]
                            if "yes_no_format_response" in mr_info
                            else False,
                            "multiple_choice_format_response": mr_info[
                                "multiple_choice_format_response"
                            ]
                            if "multiple_choice_format_response" in mr_info
                            else False,
                            "completion_format_response": mr_info[
                                "completion_format_response"
                            ]
                            if "completion_format_response" in mr_info
                            else False,
                            "rank_format_response": mr_info["rank_format_response"]
                            if "rank_format_response" in mr_info
                            else False,
                            "prompt_1": row["prompt_1"],
                            "prompt_2": row["prompt_2"],
                            "type": "consistency"
                            if "consistency" in mr_info["evaluation_method"].lower()
                            else "comparison",
                        }

                        if "response_max_length" in mr_info:
                            request_body["response_max_length"] = mr_info[
                                "response_max_length"
                            ]

                        if mr_info.get("exclude_references", False):
                            if "attribute" in row:
                                request_body["excluded_text"] = [row["attribute"]]
                            elif "attribute_1" in row and "attribute_2" in row:
                                request_body["excluded_text"] = [
                                    row["attribute_1"],
                                    row["attribute_2"],
                                ]

                        execution = execute_request(request_body, row["test_id"], mr)
                        if execution:
                            execution["bias_type"] = row["bias_type"]
                            execution["scenario"] = row["scenario"]
                            if "attribute" in row:
                                execution["attribute"] = row["attribute"]
                            elif "attribute_1" in row and "attribute_2" in row:
                                execution["attribute_1"] = row["attribute_1"]
                                execution["attribute_2"] = row["attribute_2"]

                            executions.append(execution)
                            logger.info(
                                f"Executed test {row['test_id']} for {mr} on {model} ({i + 1}/{n_executions})"
                            )

                            if multiple_models and multiple_executions:  # RQ3
                                output_file = (
                                    Path(CONFIG["OUTPUT_DIR"])
                                    / model
                                    / f"{mr}/{i + 1}.csv"
                                )
                            elif multiple_models and not multiple_executions:  # RQ2
                                output_file = (
                                    Path(CONFIG["OUTPUT_DIR"]) / model / f"{mr}.csv"
                                )
                            else:  # RQ1
                                output_file = Path(CONFIG["OUTPUT_DIR"]) / f"{mr}.csv"

                            save_to_csv(executions, output_file)


def launch_execution(rq):
    """
    Launch the metamorphic test execution process for the selected RQ.
    Loads configuration and executes all metamorphic tests.
    Args:
        rq (int): RQ value (1, 2, or 3).
    """
    metamorphic_relations = load_json(MRS_FILE_PATH)
    execution_config = load_json(CONFIG["RQ_FILE_PATH"])["execution"]

    if rq == 1:
        process_metamorphic_tests(metamorphic_relations, execution_config)
    elif rq == 2:
        process_metamorphic_tests(
            metamorphic_relations, execution_config, multiple_models=True
        )
    elif rq == 3:
        process_metamorphic_tests(
            metamorphic_relations,
            execution_config,
            multiple_models=True,
            multiple_executions=True,
        )
    else:
        raise ValueError(f"Unsupported RQ value: {rq}")


if __name__ == "__main__":
    launch_execution(RQ)
