# Replication Guide

This document provides step-by-step instructions to replicate the experimental results reported in our paper.

## Prerequisites

### Software Requirements
- Java 21 or later (OpenJDK recommended)
- Apache Maven 3.9+
- Git 2.x
- Python 3.8+ (for data processing scripts)

### Hardware Requirements
- **Minimum:** 8GB RAM, 10GB free disk space
- **Recommended:** 16GB RAM, 50GB free disk space (for full repository cloning)
- **Tested On:** Apple Mac Mini M4, 16GB RAM, 256GB SSD

## Step 1: Extract Source Code

```bash
# Unzip the source code snapshot
unzip source_code/JavaMLBugDetective_Source_Snapshot.zip -d JavaMLBugDetective

cd JavaMLBugDetective
```

## Step 2: Build the Framework

```bash
# Compile and package
mvn clean package -DskipTests

# Verify build success
ls -la target/JavaMLBugDetective-*.jar
```

## Step 3: Configure Analysis

Edit `config.properties` to match your environment:

```properties
# Repository settings
git.repo.url=https://github.com/google/gson.git
git.repo.local.path=./repositories/gson
project.name=Gson

# SZZ settings
szz.ignore_blank_lines=true
szz.ignore_comments=true

# ML settings
ml.balance.classes=true
ml.cost.false_negative=10
ml.cost.false_positive=1
```

## Step 4: Run Analysis

```bash
# Full analysis pipeline
./clean_and_run.sh

# Or run manually
java -jar target/JavaMLBugDetective-1.0-SNAPSHOT.jar
```

## Step 5: Verify Results

After analysis completes, verify output files:

```bash
# Check generated dataset
ls -la *.arff *.db

# Expected output structure
# - gson-dataset.arff (feature vectors)
# - gson.db (SQLite database with all metrics)
```

## Reproducing Specific Experiments

### Ablation Study

To reproduce the ablation study comparing Static-Only, Process-Only, and Hybrid models:

1. Modify `ml.feature.set` in `config.properties`:
   - `static` for Static-Only model
   - `process` for Process-Only model
   - `hybrid` for Hybrid model (default)

2. Run analysis for each configuration
3. Record F1-scores from the generated reports

### Cross-Project Validation

To reproduce cross-project validation results:

1. Configure each project separately:
   ```properties
   # Apache Kafka
   git.repo.url=https://github.com/apache/kafka.git
   
   # Google Gson
   git.repo.url=https://github.com/google/gson.git
   
   # Apache Commons-IO
   git.repo.url=https://github.com/apache/commons-io.git
   ```

2. Run analysis for each project
3. Compare results against the cross-project validation results in the accompanying paper

## Expected Results

Your results should approximately match:

| Project | F1-Score | Precision | Recall |
|---------|----------|-----------|--------|
| Kafka | 0.742 ± 0.02 | 0.61 ± 0.03 | 0.95 ± 0.02 |
| Gson | 0.685 ± 0.02 | 0.52 ± 0.03 | 0.99 ± 0.01 |
| Commons-IO | 0.570 ± 0.03 | 0.40 ± 0.03 | 0.99 ± 0.01 |

**Note:** Minor variations (±2-3%) are expected due to:
- Random seed differences in SMOTE oversampling
- Random Forest bootstrap sampling
- Stratified split variations

## Troubleshooting

### Common Issues

1. **OutOfMemoryError**
   ```bash
   # Increase heap size
   export MAVEN_OPTS="-Xmx4g"
   java -Xmx4g -jar target/JavaMLBugDetective-1.0-SNAPSHOT.jar
   ```

2. **Git Clone Timeout**
   ```bash
   # Pre-clone repositories manually
   git clone --depth=1 https://github.com/google/gson.git repositories/gson
   ```

3. **PMD Analysis Failures**
   - Some historical commits may contain syntax errors
   - The framework applies sensible defaults for unparseable files
   - This is expected behavior and documented in the paper

4. **Missing Dependencies**
   ```bash
   # Ensure all Maven dependencies are downloaded
   mvn dependency:resolve
   ```

## Execution Time Estimates

| Project | Commits | Estimated Time |
|---------|---------|----------------|
| Gson | 2,156 | ~4 minutes |
| Commons-IO | 3,947 | ~8 minutes |
| Kafka | 19,377 | ~40 minutes |
| **Full Dataset** | **25,480** | **~51 minutes** |

*Measured on Apple Mac Mini M4, 16GB RAM*

## Output Files

After successful execution, you will find:

```
output/
├── {project}-dataset.arff      # Weka-compatible feature vectors
├── {project}-dataset.csv       # CSV format for pandas/R
├── {project}.db                # SQLite database with all metrics
├── {project}-report-*.md       # Analysis report
└── {project}-prediction-*.md   # Prediction results
```

## Validating Against Paper Results

To verify your results match the paper:

1. **Dataset Size:** Check `*.arff` file has correct instance count
2. **F1-Score:** Compare against the results in the accompanying paper (should be within ±0.02)
3. **Feature Importance:** Compare against the feature importance rankings in the accompanying paper

## Contact

For replication issues, please contact:
- **Email:** turgay@taymaz.org
- **GitHub Issues:** https://github.com/ttaymaz/JavaMLBugDetective/issues

---

*Guide Version: 1.0*  
*Last Updated: January 2026*
