#!/usr/bin/env python3
"""Gate-3 post-processing helpers (plots from existing CSV artifacts).

This script does NOT attempt to re-run the full SPARC fitting pipeline.
Instead it:
  1) unzips gate3g_outputs.zip to a folder
  2) looks for common CSV outputs (e.g. a0_by_galaxy.csv, residuals.csv)
  3) generates a few lightweight matplotlib plots

Use: python gate3_postprocess.py --gate3g_zip ... --outdir plots
"""

import argparse
import zipfile
from pathlib import Path

import pandas as pd
import matplotlib.pyplot as plt


def extract(zip_path: Path, outdir: Path) -> Path:
    outdir.mkdir(parents=True, exist_ok=True)
    with zipfile.ZipFile(zip_path, 'r') as z:
        z.extractall(outdir)
    return outdir


def maybe_plot_a0(df: pd.DataFrame, outdir: Path) -> None:
    # Expected columns: galaxy, a0_best (or a_star), maybe morphology
    cols = {c.lower(): c for c in df.columns}
    a0_col = None
    for cand in ['a0', 'a0_best', 'a_star', 'astar']:
        if cand in cols:
            a0_col = cols[cand]
            break
    if a0_col is None:
        return
    plt.figure()
    plt.plot(df[a0_col].values, marker='o', linestyle='')
    plt.ylabel(a0_col)
    plt.xlabel('galaxy index')
    plt.tight_layout()
    plt.savefig(outdir / 'gate3g_a0_scatter.png', dpi=200)
    plt.close()


def main() -> None:
    ap = argparse.ArgumentParser()
    ap.add_argument('--gate3g_zip', type=str, required=True)
    ap.add_argument('--outdir', type=str, default='gate3_plots')
    args = ap.parse_args()

    zip_path = Path(args.gate3g_zip)
    outdir = Path(args.outdir)
    work = outdir / 'extracted_gate3g'
    extract(zip_path, work)

    outdir.mkdir(parents=True, exist_ok=True)

    # Search for candidate CSVs
    csvs = list(work.rglob('*.csv'))
    for csv in csvs:
        try:
            df = pd.read_csv(csv)
        except Exception:
            continue
        lower = csv.name.lower()
        if 'a0' in lower or 'astar' in lower:
            maybe_plot_a0(df, outdir)
            break

    (outdir / 'gate3_postprocess_done.txt').write_text(
        f"Extracted {zip_path.name} to {work}
Plotted {len(list(outdir.glob('*.png')))} figures
",
        encoding='utf-8'
    )


if __name__ == '__main__':
    main()
