#!/usr/bin/env python3
"""Gate-2D+ (geometry extension) — reference re-run.

Writes:
  - gate2d_plus_angles.csv (phi_deg,x,y,Ax,Ay,tau_z)
  - gate2d_plus_run_summary.txt

This is a transparency script. It produces Gate-2D+ style artifacts.
"""

import argparse
import math
from pathlib import Path

import numpy as np


def fft_helmholtz_solve(src: np.ndarray, m: float, L: float) -> np.ndarray:
    N = src.shape[0]
    k = 2.0 * math.pi * np.fft.fftfreq(N, d=L / N)
    kx, ky = np.meshgrid(k, k, indexing="ij")
    denom = (kx * kx + ky * ky + m * m)
    src_hat = np.fft.fft2(src)
    u_hat = src_hat / denom
    u_hat[0, 0] = 0.0
    return np.real(np.fft.ifft2(u_hat))


def build_capped_tube_current(N: int, L: float, R: float, cap: float, J0: float, w: float):
    """Current concentrated near a shape: cylinder-like midsection + end caps.

    In 2D we approximate this as:
      - ring around radius R for |x| <= cap
      - gaussian 'caps' centered at x=±cap
    """
    x = np.linspace(-L/2, L/2, N, endpoint=False)
    X, Y = np.meshgrid(x, x, indexing="ij")

    r = np.sqrt(Y*Y + (np.maximum(0.0, np.abs(X)-cap))**2)
    W = np.exp(-0.5*((r - R)/w)**2)

    # Tangent direction about local centerline
    # Use angle around (x_clamped, 0)
    Xc = np.clip(X, -cap, cap)
    dx = X - Xc
    dy = Y
    norm = np.sqrt(dx*dx + dy*dy) + 1e-30
    tx = -dy/norm
    ty = dx/norm

    Jx = J0 * W * tx
    Jy = J0 * W * ty
    return Jx, Jy


def sample_bilinear(field: np.ndarray, x: float, y: float, L: float) -> float:
    N = field.shape[0]
    dx = L / N
    fx = (x + L/2)/dx
    fy = (y + L/2)/dx
    i0 = int(math.floor(fx)) % N
    j0 = int(math.floor(fy)) % N
    i1 = (i0 + 1) % N
    j1 = (j0 + 1) % N
    tx = fx - math.floor(fx)
    ty = fy - math.floor(fy)
    v00 = field[i0, j0]
    v10 = field[i1, j0]
    v01 = field[i0, j1]
    v11 = field[i1, j1]
    return (1-tx)*(1-ty)*v00 + tx*(1-ty)*v10 + (1-tx)*ty*v01 + tx*ty*v11


def run(N: int, L: float, R: float, cap: float, m: float, J0: float, w: float, n_angles: int, d_ref: float, outdir: Path):
    outdir.mkdir(parents=True, exist_ok=True)
    Jx, Jy = build_capped_tube_current(N=N, L=L, R=R, cap=cap, J0=J0, w=w)
    Ax = fft_helmholtz_solve(Jx, m=m, L=L)
    Ay = fft_helmholtz_solve(Jy, m=m, L=L)

    rows = []
    for k in range(n_angles):
        phi = 2*math.pi*k/n_angles
        # sample around a ring at radius d_ref*R
        x = (cap + d_ref*R*math.cos(phi))
        y = (d_ref*R*math.sin(phi))
        ax = sample_bilinear(Ax, x, y, L)
        ay = sample_bilinear(Ay, x, y, L)
        tau_z = x*ay - y*ax
        rows.append((phi*180/math.pi, x, y, ax, ay, tau_z))

    csv_path = outdir / "gate2d_plus_angles.csv"
    np.savetxt(csv_path, np.array(rows), delimiter=",", header="phi_deg,x,y,Ax,Ay,tau_z", comments="")

    tau = np.array([r[-1] for r in rows])
    summary = (
        "Gate-2D+ re-run summary
"
        f"N={N}, L={L}, R={R}, cap={cap}, m={m}, J0={J0}, w={w}, n_angles={n_angles}, d_ref={d_ref}
"
        f"tau_z: min={tau.min():.6e}, median={np.median(tau):.6e}, max={tau.max():.6e}, RMS={math.sqrt(np.mean(tau*tau)):.6e}
"
        f"Wrote: {csv_path.name}
"
    )
    (outdir / "gate2d_plus_run_summary.txt").write_text(summary, encoding="utf-8")


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--N", type=int, default=512)
    ap.add_argument("--L", type=float, default=80.0)
    ap.add_argument("--R", type=float, default=8.0)
    ap.add_argument("--cap", type=float, default=6.0)
    ap.add_argument("--m", type=float, default=0.35)
    ap.add_argument("--J0", type=float, default=1.0)
    ap.add_argument("--w", type=float, default=0.9)
    ap.add_argument("--n_angles", type=int, default=36)
    ap.add_argument("--d_ref", type=float, default=10.0)
    ap.add_argument("--outdir", type=str, default="out_gate2d_plus")
    args = ap.parse_args()

    run(N=args.N, L=args.L, R=args.R, cap=args.cap, m=args.m, J0=args.J0, w=args.w,
        n_angles=args.n_angles, d_ref=args.d_ref, outdir=Path(args.outdir))


if __name__ == "__main__":
    main()
