
import numpy as np
import matplotlib.pyplot as plt

# ============================================================
# Utilidades
# ============================================================

def order_param(x):
    return float(np.abs(np.mean(np.exp(1j * x))))

def ring_adj(N, k):
    I = np.arange(N)
    D = np.abs(I[:, None] - I[None, :])
    D = np.minimum(D, N - D)
    A = ((D > 0) & (D <= k)).astype(float)
    np.fill_diagonal(A, 0.0)
    return A

def local_order(theta, A):
    z = np.exp(1j * theta)
    deg = A.sum(1)
    W = np.where(deg[:, None] > 0, A / np.maximum(deg[:, None], 1e-12), 0.0)
    return np.abs(W @ z)

def circ_mean_angle(x):
    z = np.exp(1j * x)
    return float(np.angle(np.mean(z)))

def intervals(ts, w):
    m = w > 0.5
    out = []
    i = 0
    while i < len(m):
        if m[i]:
            j = i
            while j < len(m) and m[j]:
                j += 1
            out.append((ts[i], ts[j - 1]))
            i = j
        else:
            i += 1
    return out

# ============================================================
# Patron-PC "triángulo invertido"
# - psi(t): campo global débil (semilla)
# - módulos locales deciden si abren ventana por resonancia+relajación
# - si abren: amplifican (refuerzan acoplamiento local + pequeño sesgo de fase)
# ============================================================

def patron_pc_invertido(
    N=240, M=12, T=50.0, dt=0.01,
    k=10,
    # dinámica base
    K_base=1.2,               # acoplamiento basal (siempre)
    omega_std=0.6, noise=0.01,
    # campo global débil
    psi_mode="fixed",         # "fixed" o "adaptive"
    psi_fixed=0.0,            # fase fija si psi_mode="fixed"
    psi_tau=5.0,              # tiempo de adaptación si "adaptive"
    psi_strength=0.15,        # fuerza del susurro (muy débil)
    # decisión local (resonancia)
    th_res=0.65,              # umbral de resonancia (0..1)
    th_local=0.30,            # umbral de coherencia local para "estar en forma"
    hold_local=1.0,           # duración mínima de ventana local (seg)
    # apertura por relajación
    relax_alpha=0.03,         # filtro tensión global
    # amplificación local cuando hay ventana
    K_boost=2.5,              # refuerzo de acoplamiento dentro del módulo
    bias_strength=0.25,       # sesgo de fase hacia psi (suave, no impone)
    # “memoria” estructural sencilla
    eta=0.0015, forget=0.0006, g_clip=(0.0, 3.0),
    seed=1
):
    rng = np.random.default_rng(seed)

    # estado inferior
    theta = rng.uniform(-np.pi, np.pi, N)
    omega = rng.normal(0.0, omega_std, N)

    A0 = ring_adj(N, k)
    G = A0.copy().astype(float)

    # módulos
    m_size = N // M
    mods = [np.arange(i*m_size, (i+1)*m_size) for i in range(M-1)]
    mods.append(np.arange((M-1)*m_size, N))

    # campo global
    psi = float(psi_fixed)

    # ventanas locales (una por módulo)
    until = np.full(M, -1.0, dtype=float)

    # tensión global suavizada: F = 1 - rG
    F_s = 1.0

    steps = int(T / dt)

    ts = np.zeros(steps)
    rG = np.zeros(steps)
    rL = np.zeros(steps)
    fever = np.zeros(steps)
    psi_log = np.zeros(steps)
    # porcentaje de módulos abiertos
    open_frac = np.zeros(steps)
    # ventana “global emergente” (si muchos módulos abren)
    w_emerg = np.zeros(steps)

    # para dibujar una ventana por módulo “resumida”: suma
    w_sum = np.zeros(steps)

    for s in range(steps):
        t = s * dt
        ts[s] = t

        # ----- lecturas globales
        rg_now = order_param(theta)
        rG[s] = rg_now

        # tensión y relajación
        F = 1.0 - rg_now
        F_prev = F_s
        F_s = (1.0 - relax_alpha) * F_s + relax_alpha * F
        dF = F_s - F_prev
        relajando = (dF < 0)

        # coherencia local “general”
        rloc = local_order(theta, A0)
        pre_local = float(0.75 * rloc.mean() + 0.25 * np.quantile(rloc, 0.80))
        rL[s] = pre_local

        # ----- decisión local por módulo (resonancia con psi)
        # Cada módulo calcula su fase media y su coherencia interna
        z = np.exp(1j * theta)
        m_phase = np.zeros(M)
        m_coh = np.zeros(M)
        m_res = np.zeros(M)

        for m, idx in enumerate(mods):
            zmean = np.mean(z[idx])
            m_phase[m] = np.angle(zmean)
            m_coh[m] = np.abs(zmean)  # coherencia interna del módulo
            # resonancia = cos(distancia angular) * coherencia interna
            m_res[m] = m_coh[m] * max(0.0, np.cos(m_phase[m] - psi))

        # abre ventana local si:
        # - está “en forma” localmente (m_coh > th_local)
        # - resuena con psi (m_res > th_res)
        # - y estamos relajando (barato, poco ruido)
        for m in range(M):
            if (t >= until[m]) and relajando and (m_coh[m] > th_local) and (m_res[m] > th_res):
                until[m] = t + hold_local

        w_m = (t < until).astype(float)  # vector de ventanas locales 0/1

        open_frac[s] = float(np.mean(w_m))
        w_sum[s] = float(np.sum(w_m))

        # ventana emergente: si una fracción suficiente está abierta
        w_emerg[s] = 1.0 if open_frac[s] > 0.35 else 0.0

        # fiebre: solo cuando hay “actividad de inducción” (módulos abiertos)
        fever[s] = (1.0 - rg_now) * (1.0 if open_frac[s] > 0 else 0.0)

        # ----- dinámica inferior: acoplamiento basal + amplificación local + sesgo suave
        # acoplamiento basal (siempre)
        diff = theta[None, :] - theta[:, None]
        S = (A0 * G)

        # amplificación local: dentro de cada módulo, si su ventana está abierta,
        # incrementamos el peso de los enlaces internos.
        # (esto NO impone fases; solo fortalece coherencia del subpatrón compatible)
        Boost = np.zeros_like(S)
        for m, idx in enumerate(mods):
            if w_m[m] > 0.5:
                Boost[np.ix_(idx, idx)] = 1.0

        # suma de acoplamientos
        S_eff = S * (K_base + K_boost * Boost)

        denom = max(S_eff.sum(1).mean(), 1e-12)
        coup = (S_eff * np.sin(diff)).sum(1) / denom

        # sesgo suave hacia psi solo en nodos de módulos abiertos
        bias = np.zeros(N)
        for m, idx in enumerate(mods):
            if w_m[m] > 0.5:
                bias[idx] = np.sin(psi - theta[idx])  # “tira” suave

        # update theta
        dtheta_det = omega + coup + bias_strength * bias
        theta += dt * dtheta_det
        theta += np.sqrt(dt) * noise * rng.normal(size=N)
        theta = (theta + np.pi) % (2*np.pi) - np.pi

        # ----- memoria estructural simple (solo si hay ventana emergente)
        # (cuando muchos módulos aceptan la inducción a la vez)
        if w_emerg[s] > 0.5 and relajando:
            C = np.cos(diff)
            G += eta * (C - 0.2) * A0
            G -= forget * (G - A0)
            G = np.clip(G, g_clip[0], g_clip[1])

        # ----- dinámica del campo psi
        if psi_mode == "fixed":
            psi = float(psi_fixed)
        else:
            # psi se adapta lentamente hacia la media de los módulos que están abiertos
            # (campo global “aprende” lo que el sistema aceptó)
            open_idx = np.where(w_m > 0.5)[0]
            if len(open_idx) > 0:
                target = np.angle(np.mean(np.exp(1j * m_phase[open_idx])))
                psi += (dt / psi_tau) * np.sin(target - psi)
            # si nadie abre, psi apenas cambia (se mantiene)
            psi = (psi + np.pi) % (2*np.pi) - np.pi

        psi_log[s] = psi

    return ts, rG, rL, fever, psi_log, open_frac, w_emerg, w_sum

# ============================================================
# MAIN + plots (una sola ventana)
# ============================================================

if __name__ == "__main__":
    ts, rG, rL, fev, psi, open_frac, w_emerg, w_sum = patron_pc_invertido(
        psi_mode="fixed",       # cambia a "adaptive" si quieres que el campo aprenda
        psi_fixed=0.0,
        psi_strength=0.15,
        seed=2
    )

    fig, ax = plt.subplots(4, 1, figsize=(12, 9), sharex=True)
    fig.suptitle("Patron-PC triángulo invertido: campo global débil + decisión local (amplifica/ignora)")

    ax[0].plot(ts, rG, label="rG (global)")
    ax[0].plot(ts, rL, label="rL* (local)", alpha=0.8)
    ax[0].set_ylim(0, 1.02)
    ax[0].set_ylabel("coherencia")
    ax[0].legend(loc="upper right")

    ax[1].plot(ts, fev, label="fiebre (solo si hay módulos abiertos)")
    ax[1].set_ylim(0, 1.02)
    ax[1].set_ylabel("fiebre")
    ax[1].legend(loc="upper right")

    ax[2].plot(ts, open_frac, label="fracción módulos con ventana abierta")
    ax[2].plot(ts, w_emerg, label="ventana emergente (muchos abiertos)", alpha=0.8)
    ax[2].set_ylim(-0.05, 1.05)
    ax[2].set_ylabel("ventanas")
    ax[2].legend(loc="upper right")

    ax[3].plot(ts, psi, label="psi(t) (campo global débil)")
    ax[3].set_ylabel("psi")
    ax[3].set_xlabel("tiempo")
    ax[3].legend(loc="upper right")

    plt.tight_layout()
    plt.show()
