# ============================================================
# Pattern-Based Computing (PBC)
# A/B experiment: Classical threshold vs Relaxation-based coupling
#
# This script is PAPER-SAFE:
#  - No plt.show() (never blocks)
#  - Uses non-interactive matplotlib backend ("Agg")
#  - Saves THREE separate PNG panels for side-by-side inclusion in LaTeX
#
# It is robust to two input styles:
#  (1) Summary-only dicts (no time series)  -> produces bar-panel figures
#  (2) Time-series dicts with keys like "t" -> optionally produces time-series panels
#
# By default, it runs the summary A/B numbers you already validated.
# ============================================================

import os
import numpy as np
import matplotlib

matplotlib.use("Agg")
import matplotlib.pyplot as plt


# -------------------------
# A/B data (default)
# -------------------------

def run_ab_experiment_summary():
    """
    Returns the A/B summary dicts (no time-series).
    Replace this function with your real simulation if desired.
    """
    A = {
        "E_total": 790.421890,
        "fever_area": 27.341703,
        "t_down": 39.890000,
        "t_mem": 39.710000,
        "d_rG": 0.119319,
        "rG_mean": 0.313789,
        "rG_max": 0.466247,
        "eff_E_per_dRG": 6624.421159,
    }

    B = {
        "E_total": 4974.791975,
        "fever_area": 9.884073,
        "t_down": 21.050000,
        "t_mem": 19.970000,
        "d_rG": 0.364895,
        "rG_mean": 0.395910,
        "rG_max": 0.829982,
        "eff_E_per_dRG": 13633.502653,
    }
    return A, B


# -------------------------
# Console output (English)
# -------------------------

def _print_block(title, d):
    print(f"\n{title}")
    for k in [
        "E_total",
        "fever_area",
        "t_down",
        "t_mem",
        "d_rG",
        "rG_mean",
        "rG_max",
        "eff_E_per_dRG",
    ]:
        if k in d:
            print(f"  {k:>12s}: {d[k]:.6f}")


def print_summary(A, B):
    print("\n==================== A/B SUMMARY ====================")
    print("A = Classical threshold (no relaxation)")
    print("B = Relaxation-based coupling (dF < 0)")

    _print_block("A", A)
    _print_block("B", B)

    print("\n-------------------- COMPARISONS --------------------")
    def ratio(key):
        a = A.get(key, np.nan)
        b = B.get(key, np.nan)
        return b / a if (a not in (0, None) and np.isfinite(a)) else np.nan

    print(f"  Total energy (B/A):     {ratio('E_total'):.6f}")
    print(f"  Fever area (B/A):       {ratio('fever_area'):.6f}")
    print(f"  w_down time (B/A):      {ratio('t_down'):.6f}")
    print(f"  ΔrG (B/A):              {ratio('d_rG'):.6f}")
    print(f"  Energy per ΔrG (B/A):   {ratio('eff_E_per_dRG'):.6f}")


# -------------------------
# Plot helpers
# -------------------------

def _ensure_dir(path):
    os.makedirs(path, exist_ok=True)

def _save(fig, outpath):
    _ensure_dir(os.path.dirname(outpath) or ".")
    fig.tight_layout()
    fig.savefig(outpath, dpi=300, bbox_inches="tight")
    plt.close(fig)
    print(f"Saved: {outpath}")


# -------------------------
# Panel figures (3 PNGs)
# -------------------------

def save_three_panels(A, B, outdir="paper/figures"):
    """
    Produces three separate PNGs suitable for side-by-side placement in LaTeX:
      1) Total Energy
      2) Fever (area) and downtime (two small bars in same figure)
      3) Structural gain ΔrG (plus optional rG_max as marker)
    """

    _ensure_dir(outdir)

    # ---- Panel 1: Total Energy ----
    fig, ax = plt.subplots(figsize=(3.4, 3.0))
    ax.bar(["A", "B"], [A["E_total"], B["E_total"]])
    ax.set_title("Energy Cost")
    ax.set_ylabel("Total energy (arb. units)")
    ax.grid(True, axis="y", alpha=0.3)
    _save(fig, os.path.join(outdir, "fig_pbc_ab_panel1_energy.png"))

    # ---- Panel 2: Fever / Down time ----
    fig, ax = plt.subplots(figsize=(3.4, 3.0))
    x = np.arange(2)
    width = 0.36
    ax.bar(x - width/2, [A["fever_area"], A["t_down"]], width, label="A")
    ax.bar(x + width/2, [B["fever_area"], B["t_down"]], width, label="B")
    ax.set_xticks(x)
    ax.set_xticklabels(["Fever area", "Down time"])
    ax.set_title("Incoherence / Latency")
    ax.set_ylabel("Value (arb. units)")
    ax.legend(frameon=False, fontsize=8)
    ax.grid(True, axis="y", alpha=0.3)
    _save(fig, os.path.join(outdir, "fig_pbc_ab_panel2_fever_down.png"))

    # ---- Panel 3: Structural gain ----
    fig, ax = plt.subplots(figsize=(3.4, 3.0))
    ax.bar(["A", "B"], [A["d_rG"], B["d_rG"]])
    # Optional: show rG_max as a point marker (same axis, comparable scale in [0,1])
    if "rG_max" in A and "rG_max" in B:
        ax.plot(["A", "B"], [A["rG_max"], B["rG_max"]], marker="o", linestyle="--", label="rG_max")
        ax.legend(frameon=False, fontsize=8)
    ax.set_title("Structural Gain")
    ax.set_ylabel("ΔrG (arb. units)")
    ax.grid(True, axis="y", alpha=0.3)
    _save(fig, os.path.join(outdir, "fig_pbc_ab_panel3_delta_rG.png"))


# -------------------------
# Optional: time-series panels (only if present)
# -------------------------

def save_timeseries_panels(A, B, outdir="paper/figures"):
    """
    If A and B contain time series (e.g., A['t'], A['fever'], ...),
    you can plot dynamics here. This function is only called if 't' exists.
    """
    _ensure_dir(outdir)
    tA = np.asarray(A["t"])
    tB = np.asarray(B["t"])

    # Example: fever(t)
    if "fever" in A and "fever" in B:
        fig, ax = plt.subplots(figsize=(6.0, 3.0))
        ax.plot(tA, A["fever"], label="A")
        ax.plot(tB, B["fever"], label="B")
        ax.set_title("Fever state over time")
        ax.set_xlabel("t")
        ax.set_ylabel("fever (arb.)")
        ax.grid(True, alpha=0.3)
        ax.legend(frameon=False)
        _save(fig, os.path.join(outdir, "fig_pbc_ab_timeseries_fever.png"))


# -------------------------
# Main
# -------------------------

def main():
    A, B = run_ab_experiment_summary()
    print_summary(A, B)

    # Always produce the 3 paper panels from summary numbers
    save_three_panels(A, B, outdir="paper/figures")

    # If you later extend A/B dicts to include time series, enable extra plots:
    if "t" in A and "t" in B:
        save_timeseries_panels(A, B, outdir="paper/figures")


if __name__ == "__main__":
    main()
