# src/

This directory contains **all executable code and experimental scripts** used in the
*Pattern-Based Computing (PBC)* project.  
Every script is designed to be **paper-safe, reproducible, and non-interactive**.

Nothing in this folder is a library or API: scripts are meant to be executed directly,
either individually or via the root-level pipeline.

---

## Design Philosophy

The code in `src/` follows these principles:

- **Reproducibility first**  
  All parameters are explicit, seeded, and defined inside each script.
- **No hidden state**  
  Scripts do not rely on global configuration files.
- **Paper-aligned outputs**  
  Figures and CSVs are generated exactly as referenced in the paper.
- **No interactive execution**  
  No `input()` calls, no blocking `plt.show()` in pipeline scripts.

---

## Main Entry Points

### Core illustrative example

- `synthetic_freeway_ctm_receptive.py`  
  Main illustrative system used in the paper (traffic-based example).
  Demonstrates Pattern-Based Computing vs ALINEA vs open-loop control.

### Robustness and rotation suite

- `pbc_ctm_rotations_all_in_one.py`  
  Large-scale robustness evaluation using:
  - temporal rotations
  - spatial rotations
  - module re-partitioning  
  Produces aggregated CSV results and paper-ready figures.

---

## A/B Experiments (Threshold vs Relaxation)

These scripts compare classical threshold-based coupling with
relaxation-based coupling (PBC principle).

- `energia_ab_pbc.py`  
  Generates paper-ready A/B comparison figures (energy, fever, structural gain).

- `energia_ab_patronpc.py`  
  Alternative A/B implementation with explicit internal dynamics.

- `energia_ab_seeds.py`  
  Runs the A/B comparison across multiple random seeds and reports
  stability-normalized efficiency metrics.

---

## Memory and Stability Experiments

- `estabilidad_memorias_ab.py`  
  Evaluates memory persistence under perturbations.
  Includes:
  - learning phase
  - frozen-memory stability phase
  - shock injection and recovery analysis

---

## Conceptual / Exploratory Models

These scripts explore structural ideas related to PBC but are not all
directly referenced in the paper figures.

- `pb.py`  
  Minimal Pattern-Based / Patron-PC style model for conceptual testing.

- `patron_pc_invertido` (inside scripts)  
  Demonstrates the “inverted triangle” architecture:
  weak global field + local acceptance + emergent coordination.

---

## Execution

### Run everything (recommended)

From the repository root:

```bash
python run_full_pipeline.py
```

This sequentially executes the relevant scripts in `src/`
and generates all figures and data.

### Run individual scripts

Each script can be executed directly, for example:

```bash
python src/energia_ab_seeds.py
```

Outputs are written to:
- `paper/figures/`
- local CSV files (depending on the script)

---

## Notes

- Scripts may duplicate utility functions intentionally to keep each file
  self-contained and reviewable.
- Performance optimizations are secondary to clarity and reproducibility.
- The code expresses **computational behavior**, not production engineering.

---

## Status

This folder contains **research code**.
Refactoring into a library is intentionally avoided to preserve
one-to-one correspondence with the paper.
