#!/usr/bin/env python3
"""
Run full reproducible pipeline for the PBC paper.

This script executes all core pipeline components in the correct order
to reproduce the figures and results reported in the manuscript.

Assumptions:
- All core scripts live in ./src
- Each script is self-contained and writes its own outputs (figures / csv)
- No interactive input is required

Author: PBC
"""

import subprocess
import sys
from pathlib import Path

# ---------------------------------------------------------------------
# Configuration
# ---------------------------------------------------------------------

PROJECT_ROOT = Path(__file__).resolve().parent
SRC_DIR = PROJECT_ROOT / "src"

PIPELINE_SCRIPTS = [
    # 1. Main illustrative example (traffic)
    "synthetic_freeway_ctm_receptive.py",

    # 2. Robustness across rotations and seeds
    "pbc_ctm_rotations_all_in_one.py",

    # 3. A/B energy comparison (threshold vs relaxation)
    "energia_ab_pbc.py",
    "energia_ab_patronpc.py",

    # 4. Robustness across seeds
    "energia_ab_seeds.py",

    # 5. Memory stability under perturbations
    "estabilidad_memorias_ab.py",
]

# ---------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------

def run_script(script_path: Path):
    print(f"\n▶ Running: {script_path.name}")
    print("-" * 60)

    result = subprocess.run(
        [sys.executable, script_path],
        cwd=SRC_DIR,
        stdout=sys.stdout,
        stderr=sys.stderr,
    )

    if result.returncode != 0:
        print(f"\n✖ ERROR while running {script_path.name}")
        sys.exit(result.returncode)

    print(f"✔ Finished: {script_path.name}")


# ---------------------------------------------------------------------
# Main
# ---------------------------------------------------------------------

def main():
    print("\n==============================================")
    print(" PBC — FULL REPRODUCIBLE PIPELINE")
    print("==============================================")

    if not SRC_DIR.exists():
        print("✖ src/ directory not found.")
        sys.exit(1)

    for script_name in PIPELINE_SCRIPTS:
        script_path = SRC_DIR / script_name

        if not script_path.exists():
            print(f"✖ Missing script: {script_name}")
            sys.exit(1)

        run_script(script_path)

    print("\n==============================================")
    print(" ✔ FULL PIPELINE COMPLETED SUCCESSFULLY")
    print("==============================================")
    print("Figures and data should now be available in their respective directories.")


if __name__ == "__main__":
    main()
