# Pattern-Based Computing (PBC)

Pattern-Based Computing (PBC) is a computational paradigm for continuous, distributed, and complex systems in which computation is realized through **dynamic relaxation toward patterns**, rather than through sequential symbolic execution or explicit trajectory optimization.

This repository contains:
- The conceptual and mathematical formulation of PBC.
- A fully reproducible experimental pipeline.
- An illustrative large-scale traffic system used to demonstrate robustness, self-correction, and structural adaptation.

---

## Core Idea

In PBC, **patterns are the primary computational objects**.

- A pattern is not a representation to be interpreted, but an *active structure* that biases system dynamics.
- Computation emerges as the system relaxes toward pattern-compatible regimes.
- Classical computation is confined to a **programmatic role**: configuring and modifying patterns, never executing coordination or control directly.

This collapses the traditional separation between **program, process, and result**:
- The **program** is the active pattern.
- The **process** is relaxation under pattern influence.
- The **result** is the stabilized regime reached by the system.

---

## Architectural Overview

PBC is explicitly hybrid:

- **Pattern-based layer (continuous, fast)**  
  Performs computation via relaxation, coordination, and stability.
- **Classical layer (discrete, slow)**  
  Programs the *lower pattern* to inject data, intent, or constraints.

Patterns are organized geometrically as an **inverted triangle**:
- Global pattern (system-wide stability and intent)
- Intermediate patterns (sensor integration and local coherence)
- Lower pattern (programmable interface)

Coupling between layers is modulated by **system receptivity**, enabling:
- Local decoherence for error self-correction
- Protected global adaptation via **coupling windows**

---

## Error Handling and Adaptation

PBC replaces explicit error correction with structural mechanisms:

- **Local decoherence**  
  Perturbations temporarily reduce coupling instead of triggering global correction.
- **Upward seduction**  
  Persistent lower-level signals can reshape the global pattern, but only during receptive phases.
- **Fever states**  
  When no compatible pattern exists, failure is expressed dynamically as sustained instability—not as a false solution.

---

## Repository Structure

```
.
├── main.pdf                  # Paper: Pattern-Based Computing
├── run_full_pipeline.py      # One-command reproducible pipeline
├── src/                      # All simulation and analysis scripts
│   ├── synthetic_freeway_ctm_receptive.py
│   ├── pbc_ctm_rotations_all_in_one.py
│   ├── energia_ab_pbc.py
│   ├── energia_ab_patronpc.py
│   ├── energia_ab_seeds.py
│   └── estabilidad_memorias_ab.py
├── figures/                  # Generated figures (after running pipeline)
├── data/                     # Generated data (after running pipeline)
└── README.md
```

---

## Reproducibility

All results reported in the paper are fully reproducible.

### Requirements
- Python 3.9+
- NumPy
- SciPy
- Matplotlib
- Standard scientific Python stack

### Run the full pipeline

From the project root:

```bash
python run_full_pipeline.py
```

This will:
1. Run the illustrative traffic example
2. Execute robustness tests across rotations and seeds
3. Generate all figures and metrics reported in the paper

No manual intervention or parameter tuning is required.

---

## Evaluation Philosophy

PBC is **not evaluated by average-case optimality**.

Instead, experiments focus on:
- Robustness under perturbations
- Stability across structural variation
- Bounded amplification of intervention
- Graceful degradation instead of catastrophic failure

Performance metrics are interpreted as **computational stability indicators**, not domain-specific objectives.

---

## Scope and Limitations

PBC is suitable for:
- Continuous or quasi-continuous systems
- Distributed, multi-agent dynamics
- Domains where stability and coordination dominate over exact optimality

PBC is **not suitable** for:
- Discrete symbolic computation
- Exact arithmetic or formal logic
- Combinatorial optimization with strict correctness constraints

This is a defining characteristic, not a limitation.

---

## Citation

If you use this work, please cite:

> *Pattern-Based Computing: A Hybrid Computational Paradigm Based on Patterns, Receptivity, and Dynamic Relaxation*, 2026.

---

## License

This project is released for research and academic use.  
See the LICENSE file for details.

---

## Status

This repository represents a **research prototype** and conceptual framework.  
APIs and internal structure may evolve as the paradigm is applied to new domains.
