#!/bin/bash
#SBATCH --time=1:00:00
#SBATCH --partition=main
#SBATCH --output=slurm_out.%A
#SBATCH --array=0-25%1
#SBATCH --mem=880GB
#SBATCH --open-mode=append
#SBATCH --nodes=1
#SBATCH --ntasks-per-node=128

declare -ar nbasis_list=(100 200 300 400 600 800 1000 1200 1500 2000 2500 3000 3500 4000 4500 5000 5500 6000 6500 7000 7500 8000 9000 10000)
nbasis=${nbasis_list[$SLURM_ARRAY_TASK_ID]}

if [ "$nbasis" == "" ]; then
    echo "Nothing to do for $SLURM_ARRAY_TASK_ID"
    exit 0
else
    echo "Running task $SLURM_ARRAY_TASK_ID"
fi


dpath=data/nbasis_$(printf '%06d' $nbasis)
mkdir -p $dpath

log() {
    echo $(date '+[%Y-%m-%d %H:%M:%S]') $* >> $dpath/benchmark.log
}

write_header() {
    printf '# %8s %8s %8s %16s %16s %8s\n' "Basis" "Cores" "Nodes" "Wall-time(s)" "Memory(MiB)" "Type" > $dpath/result.log
}

write_result() {
    printf '%10s %8s %8s %16s %16s %8s\n' $* >> $dpath/result.log
}

benchmark() {
    testname="$1"
    suffix="ncores$(printf '%04d' $SLURM_NTASKS)_${testname}"
    log "== Starting benchmark with $SLURM_NTASKS cores for ${testname} =="
    # Start memory poll in background
    ./poll_memory_usage > $dpath/memory_$suffix.log &
    pid=$!

    # Start benchmark
    SECONDS=0
    srun python run_benchmark_${testname}.py $dpath > $dpath/rhodent_$suffix.log 2> /dev/null
    code=$?
    elapsed=$SECONDS
    [ $code == 0 ] && status=Passed || status=Failed
    log "$status in $elapsed seconds"

    # Clean up
    sleep 10  # Wait so the memory poll has time to flush
    kill $pid
    startmem=$(cat $dpath/memory_$suffix.log | grep -v '#' | head -n 1 | awk '{print $3}')
    mem=$(awk -v max=0 'NR>1 {if($3>max){time=$1; max=$3}}END{print max}' $dpath/memory_$suffix.log)
    mem=$(echo $mem-$startmem | bc)
    memtime=$(awk -v max=0 'NR>1 {if($3>max){time=$1; max=$3}}END{print time}' $dpath/memory_$suffix.log)
    [ $code == 0 ] && write_result $nbasis $SLURM_NTASKS $SLURM_NNODES $elapsed $mem $testname
    log "Max memory $mem MiB at $memtime (idle memory is $startmem)"
    rm -rf $dpath/pulserho
}

log "Start job"
write_header

module load gpaw/frozen

export OMP_NUM_THREADS=1
export RHODENT_RESPONSE_MAX_MEM="100000"  # Attempt limiting response calculator to 100GB

# Generate the data
python generate_data.py $dpath $nbasis || exit 1

log "Generated test data"
log $(ls -lah $dpath/ksd.ulm)
log $(ls -lah $dpath/wfs.ulm)

benchmark full
benchmark single

# Clean up
rm -rf $dpath/ksd.ulm $dpath/wfs.ulm
