"""
Goldbach Phase 2A: Modular Obstruction Kernel

Goal: Characterize the necessary modular structure of a hypothetical counterexample.

NOT proving counterexamples are impossible.
NOT relying on averages or enhancement factors.

ONLY: Mapping the shape of what would need to survive.

Methodology: Counterexample Stress Simulator
"""

import math
from typing import List, Set, Dict, Tuple
from collections import defaultdict
from goldbach_kernel import GoldbachKernel


class ModularObstructionKernel:
    """
    Modular Obstruction Kernel for Goldbach Conjecture
    
    Philosophy: Make counterexamples uncomfortable by mapping their required structure.
    """
    
    def __init__(self, max_n: int = 10**6):
        self.kernel = GoldbachKernel(max_n=max_n)
        print(f"Modular Obstruction Kernel initialized")
    
    def compute_obstruction_set(self, p: int) -> Set[int]:
        """
        Compute obstruction set O_p for prime p
        
        O_p = {n mod p | no primes p1, p2 exist with p1 + p2 ≡ n (mod p)}
        
        For a number n in O_p, Goldbach would fail modulo p.
        """
        # All residues mod p
        all_residues = set(range(p))
        
        # Residues that CAN be represented as sum of two primes mod p
        representable = set()
        
        # Get all primes less than p (for mod p analysis)
        primes_mod_p = set()
        for prime in self.kernel.primes:
            if prime >= p:
                break
            primes_mod_p.add(prime % p)
        
        # Also include p itself if it's in our prime list
        if p in self.kernel.prime_set:
            primes_mod_p.add(0)  # p ≡ 0 (mod p)
        
        # For each pair of primes, mark their sum as representable
        for p1 in primes_mod_p:
            for p2 in primes_mod_p:
                sum_mod = (p1 + p2) % p
                representable.add(sum_mod)
        
        # Obstruction set = residues that are NOT representable
        obstruction_set = all_residues - representable
        
        return obstruction_set
    
    def compute_intersection(self, primes: List[int]) -> Dict:
        """
        Compute intersection of obstruction sets for multiple primes
        
        Returns: {
            'primes': list of primes used,
            'individual_obstructions': {p: O_p},
            'intersection': set of (n mod p1, n mod p2, ...) tuples,
            'intersection_size': size of intersection,
            'density': intersection_size / product(primes)
        }
        """
        if not primes:
            return {}
        
        # Compute individual obstruction sets
        obstructions = {}
        for p in primes:
            obstructions[p] = self.compute_obstruction_set(p)
        
        # Compute intersection using Chinese Remainder Theorem structure
        # For each combination of residues, check if it's in all obstruction sets
        
        # Start with first prime's obstruction set
        p0 = primes[0]
        candidates = [(r,) for r in obstructions[p0]]
        
        # Iteratively intersect with remaining primes
        for p in primes[1:]:
            new_candidates = []
            for residue_tuple in candidates:
                for r in obstructions[p]:
                    new_candidates.append(residue_tuple + (r,))
            candidates = new_candidates
        
        intersection = set(candidates)
        
        # Compute density
        total_space = 1
        for p in primes:
            total_space *= p
        
        density = len(intersection) / total_space if total_space > 0 else 0
        
        return {
            'primes': primes,
            'individual_obstructions': obstructions,
            'intersection': intersection,
            'intersection_size': len(intersection),
            'total_space': total_space,
            'density': density
        }
    
    def test_rigidity(self, max_primes: int = 10) -> List[Dict]:
        """
        Test how fast obstruction intersections shrink as we add more primes
        
        This measures the "rigidity" of hypothetical counterexamples.
        """
        print("\n" + "=" * 70)
        print("MODULAR RIGIDITY TEST")
        print("=" * 70)
        print("Measuring how fast obstruction intersections collapse")
        print()
        
        # Use first k primes
        test_primes = [p for p in self.kernel.primes if p <= 100][:max_primes]
        
        results = []
        
        print(f"Testing with first {len(test_primes)} primes: {test_primes}")
        print()
        
        # Incrementally add primes and measure intersection
        for k in range(1, len(test_primes) + 1):
            primes_subset = test_primes[:k]
            
            intersection_data = self.compute_intersection(primes_subset)
            
            results.append(intersection_data)
            
            print(f"Primes {primes_subset[:3]}{'...' if k > 3 else ''} "
                  f"({k} total) | "
                  f"Intersection size: {intersection_data['intersection_size']:8d} | "
                  f"Density: {intersection_data['density']:.6f}")
        
        print()
        print("Rigidity Analysis:")
        
        if results:
            initial_density = results[0]['density']
            final_density = results[-1]['density']
            shrinkage = (initial_density - final_density) / initial_density if initial_density > 0 else 0
            
            print(f"  Initial density (1 prime): {initial_density:.6f}")
            print(f"  Final density ({len(test_primes)} primes): {final_density:.6f}")
            print(f"  Shrinkage: {shrinkage * 100:.2f}%")
            
            if final_density < 0.001:
                print(f"  ✅ Intersection becomes negligible")
            elif shrinkage > 0.9:
                print(f"  ✅ Strong rigidity (>90% shrinkage)")
            else:
                print(f"  ⚠️  Moderate rigidity")
        
        return results
    
    def find_surviving_patterns(self, primes: List[int], max_patterns: int = 10) -> List[Tuple]:
        """
        Find specific residue patterns that survive the intersection
        
        These are the "shapes" of hypothetical counterexamples.
        """
        intersection_data = self.compute_intersection(primes)
        
        surviving = list(intersection_data['intersection'])[:max_patterns]
        
        print("\n" + "=" * 70)
        print("SURVIVING RESIDUE PATTERNS")
        print("=" * 70)
        print(f"Primes: {primes}")
        print(f"Total surviving patterns: {len(intersection_data['intersection'])}")
        print()
        
        if surviving:
            print(f"Sample patterns (first {len(surviving)}):")
            for i, pattern in enumerate(surviving, 1):
                # Convert pattern to n mod p for each prime
                residue_str = ", ".join(f"n≡{r} (mod {p})" for r, p in zip(pattern, primes))
                print(f"  {i}. {residue_str}")
        else:
            print("  ✅ No patterns survive (intersection is empty)")
        
        return surviving
    
    def stress_test_counterexample(self, n: int, primes: List[int]) -> Dict:
        """
        Stress test: Does n satisfy the necessary modular conditions for a counterexample?
        
        For n to be a counterexample, it must be in the obstruction set for ALL primes.
        """
        if n % 2 != 0 or n <= 2:
            return {'error': 'n must be even and > 2'}
        
        # Check if n has Goldbach representations (should be True)
        goldbach_result = self.kernel.verify_goldbach(n)
        
        # Check modular obstructions
        obstruction_status = {}
        
        for p in primes:
            n_mod_p = n % p
            obstruction_set = self.compute_obstruction_set(p)
            
            is_obstructed = n_mod_p in obstruction_set
            obstruction_status[p] = {
                'n_mod_p': n_mod_p,
                'in_obstruction_set': is_obstructed,
                'obstruction_set_size': len(obstruction_set)
            }
        
        # Count how many primes obstruct n
        obstructed_count = sum(1 for data in obstruction_status.values() if data['in_obstruction_set'])
        
        return {
            'n': n,
            'has_goldbach_representations': goldbach_result['satisfies_goldbach'],
            'representation_count': goldbach_result['representation_count'],
            'primes_tested': primes,
            'obstruction_status': obstruction_status,
            'obstructed_count': obstructed_count,
            'total_primes': len(primes),
            'is_fully_obstructed': obstructed_count == len(primes)
        }


def run_modular_obstruction_analysis():
    """
    Phase 2A: Modular Obstruction Analysis
    
    Characterize the shape of hypothetical counterexamples.
    """
    print("=" * 70)
    print("GOLDBACH PHASE 2A: MODULAR OBSTRUCTION ANALYSIS")
    print("=" * 70)
    print()
    print("Goal: Map the necessary modular structure of hypothetical counterexamples")
    print("NOT proving impossibility - only measuring rigidity")
    print()
    
    # Initialize kernel
    kernel = ModularObstructionKernel(max_n=100000)
    
    # Test 1: Individual obstruction sets
    print("[Test 1] Individual Obstruction Sets")
    print("-" * 70)
    
    test_primes = [2, 3, 5, 7, 11]
    for p in test_primes:
        obs_set = kernel.compute_obstruction_set(p)
        density = len(obs_set) / p
        print(f"p = {p:2d} | O_p = {sorted(obs_set)} | Size: {len(obs_set)} | Density: {density:.3f}")
    
    # Test 2: Rigidity test
    rigidity_results = kernel.test_rigidity(max_primes=8)
    
    # Test 3: Surviving patterns
    surviving = kernel.find_surviving_patterns(primes=[2, 3, 5], max_patterns=10)
    
    # Test 4: Stress test specific numbers
    print("\n" + "=" * 70)
    print("COUNTEREXAMPLE STRESS TEST")
    print("=" * 70)
    print("Testing if known numbers satisfy obstruction conditions")
    print()
    
    test_numbers = [4, 6, 8, 10, 100, 1000]
    test_primes_subset = [2, 3, 5, 7]
    
    for n in test_numbers:
        stress_result = kernel.stress_test_counterexample(n, test_primes_subset)
        
        print(f"n = {n:5d} | "
              f"Reps: {stress_result['representation_count']:3d} | "
              f"Obstructed by: {stress_result['obstructed_count']}/{stress_result['total_primes']} primes | "
              f"{'❌ IMPOSSIBLE' if stress_result['is_fully_obstructed'] and stress_result['has_goldbach_representations'] else '✅'}")
    
    print()
    print("=" * 70)
    print("PHASE 2A COMPLETE")
    print("=" * 70)
    
    return {
        'rigidity_results': rigidity_results,
        'surviving_patterns': surviving
    }


if __name__ == "__main__":
    run_modular_obstruction_analysis()
