"""
Goldbach Conjecture Verification Kernel - Phase 1

Conjecture: Every even integer > 2 is the sum of two primes.

Methodology: Adversarial testing + statistical analysis
Goal: Find counterexample OR prove structural impossibility
"""

import math
import json
from typing import List, Tuple, Dict, Set
from collections import defaultdict
import time


class GoldbachKernel:
    """
    Deterministic verification kernel for Goldbach Conjecture
    
    Philosophy: Kernel is authority, not speculation.
    """
    
    def __init__(self, max_n: int = 10**6):
        """
        Initialize kernel with prime sieve up to max_n
        
        Args:
            max_n: Maximum even number to test
        """
        self.max_n = max_n
        self.primes = self._sieve_of_eratosthenes(max_n)
        self.prime_set = set(self.primes)
        
        print(f"Goldbach Kernel initialized")
        print(f"Prime sieve computed up to {max_n:,}")
        print(f"Total primes found: {len(self.primes):,}")
    
    def _sieve_of_eratosthenes(self, limit: int) -> List[int]:
        """
        Generate all primes up to limit using Sieve of Eratosthenes
        
        Optimized for speed and memory efficiency.
        """
        if limit < 2:
            return []
        
        # Use bitarray for memory efficiency
        is_prime = [True] * (limit + 1)
        is_prime[0] = is_prime[1] = False
        
        for i in range(2, int(math.sqrt(limit)) + 1):
            if is_prime[i]:
                # Mark multiples as composite
                for j in range(i * i, limit + 1, i):
                    is_prime[j] = False
        
        # Extract primes
        primes = [i for i in range(2, limit + 1) if is_prime[i]]
        return primes
    
    def is_prime(self, n: int) -> bool:
        """Check if n is prime (using precomputed sieve)"""
        if n <= self.max_n:
            return n in self.prime_set
        
        # For numbers beyond sieve, use trial division
        if n < 2:
            return False
        if n == 2:
            return True
        if n % 2 == 0:
            return False
        
        for i in range(3, int(math.sqrt(n)) + 1, 2):
            if n % i == 0:
                return False
        return True
    
    def find_goldbach_pairs(self, n: int) -> List[Tuple[int, int]]:
        """
        Find all prime pairs (p, q) such that p + q = n
        
        Returns list of pairs (p, q) where p <= q
        """
        if n <= 2 or n % 2 != 0:
            return []
        
        pairs = []
        
        # Only check primes up to n/2 (avoid duplicates)
        for p in self.primes:
            if p > n // 2:
                break
            
            q = n - p
            
            if self.is_prime(q):
                pairs.append((p, q))
        
        return pairs
    
    def verify_goldbach(self, n: int) -> Dict:
        """
        Verify Goldbach conjecture for even number n
        
        Returns detailed analysis including all prime pairs.
        """
        result = {
            'n': n,
            'is_even': n % 2 == 0,
            'satisfies_goldbach': False,
            'prime_pairs': [],
            'representation_count': 0,
            'smallest_prime_used': None,
            'largest_prime_used': None
        }
        
        if n <= 2 or n % 2 != 0:
            return result
        
        pairs = self.find_goldbach_pairs(n)
        
        result['prime_pairs'] = pairs
        result['representation_count'] = len(pairs)
        result['satisfies_goldbach'] = len(pairs) > 0
        
        if pairs:
            all_primes = [p for pair in pairs for p in pair]
            result['smallest_prime_used'] = min(all_primes)
            result['largest_prime_used'] = max(all_primes)
        
        return result
    
    def adversarial_search(self, start: int = 4, end: int = None, 
                          report_interval: int = 10000) -> Dict:
        """
        Adversarial search for Goldbach counterexamples
        
        Tests every even number in range [start, end].
        Reports progress and any counterexamples found.
        """
        if end is None:
            end = self.max_n
        
        # Ensure start is even and >= 4
        if start < 4:
            start = 4
        if start % 2 != 0:
            start += 1
        
        print("=" * 70)
        print("GOLDBACH ADVERSARIAL SEARCH")
        print("=" * 70)
        print(f"Range: {start:,} to {end:,}")
        print(f"Testing {(end - start) // 2 + 1:,} even numbers")
        print()
        
        counterexamples = []
        tested = 0
        start_time = time.time()
        
        # Statistical tracking
        representation_counts = defaultdict(int)
        min_representations = float('inf')
        min_representation_n = None
        
        for n in range(start, end + 1, 2):
            tested += 1
            
            result = self.verify_goldbach(n)
            
            if not result['satisfies_goldbach']:
                # COUNTEREXAMPLE FOUND!
                counterexamples.append(n)
                print(f"\n❌ COUNTEREXAMPLE FOUND: {n}")
                print(f"   No prime pairs sum to {n}")
            
            # Track statistics
            count = result['representation_count']
            representation_counts[count] += 1
            
            if count < min_representations and count > 0:
                min_representations = count
                min_representation_n = n
            
            # Progress report
            if tested % report_interval == 0:
                elapsed = time.time() - start_time
                rate = tested / elapsed if elapsed > 0 else 0
                print(f"Progress: {tested:,} tested | "
                      f"Current n = {n:,} | "
                      f"Rate: {rate:.0f} tests/sec")
        
        elapsed = time.time() - start_time
        
        print()
        print("=" * 70)
        print("ADVERSARIAL SEARCH COMPLETE")
        print("=" * 70)
        print(f"Total tested: {tested:,}")
        print(f"Counterexamples found: {len(counterexamples)}")
        print(f"Time elapsed: {elapsed:.2f} seconds")
        print(f"Average rate: {tested / elapsed:.0f} tests/sec")
        print()
        
        if counterexamples:
            print(f"⚠️  COUNTEREXAMPLES: {counterexamples}")
        else:
            print(f"✅ NO COUNTEREXAMPLES FOUND")
            print(f"   Goldbach conjecture holds for all even n in [{start:,}, {end:,}]")
        
        print()
        print("Statistical Summary:")
        print(f"  Minimum representations: {min_representations} (at n = {min_representation_n:,})")
        print(f"  Representation distribution:")
        for count in sorted(representation_counts.keys())[:10]:
            freq = representation_counts[count]
            print(f"    {count} representations: {freq:,} numbers")
        
        return {
            'range_start': start,
            'range_end': end,
            'total_tested': tested,
            'counterexamples': counterexamples,
            'time_elapsed': elapsed,
            'min_representations': min_representations,
            'min_representation_n': min_representation_n,
            'representation_distribution': dict(representation_counts)
        }
    
    def analyze_prime_density(self, n: int) -> Dict:
        """
        Analyze prime density around n
        
        Uses Prime Number Theorem to estimate expected representations.
        """
        # Prime Number Theorem: π(n) ≈ n / ln(n)
        pi_n = n / math.log(n) if n > 1 else 0
        
        # Heuristic: Expected number of Goldbach representations
        # Based on Hardy-Littlewood conjecture
        expected_reps = pi_n / (2 * math.log(n)) if n > 2 else 0
        
        actual_result = self.verify_goldbach(n)
        actual_reps = actual_result['representation_count']
        
        return {
            'n': n,
            'pi_n_estimate': pi_n,
            'expected_representations': expected_reps,
            'actual_representations': actual_reps,
            'ratio': actual_reps / expected_reps if expected_reps > 0 else 0
        }


def run_phase_1_tests():
    """
    Phase 1: Empirical verification and counterexample search
    """
    print("=" * 70)
    print("GOLDBACH KERNEL - PHASE 1: EMPIRICAL VERIFICATION")
    print("=" * 70)
    print()
    
    # Test 1: Small range verification
    print("[Test 1] Small Range Verification (n ≤ 1000)")
    print("-" * 70)
    kernel_small = GoldbachKernel(max_n=1000)
    result_small = kernel_small.adversarial_search(start=4, end=1000, report_interval=200)
    print()
    
    # Test 2: Medium range verification
    print("[Test 2] Medium Range Verification (n ≤ 100,000)")
    print("-" * 70)
    kernel_medium = GoldbachKernel(max_n=100000)
    result_medium = kernel_medium.adversarial_search(start=4, end=100000, report_interval=10000)
    print()
    
    # Test 3: Prime density analysis
    print("[Test 3] Prime Density Analysis")
    print("-" * 70)
    test_values = [100, 1000, 10000, 100000]
    for n in test_values:
        if n % 2 != 0:
            n += 1
        analysis = kernel_medium.analyze_prime_density(n)
        print(f"n = {n:,}:")
        print(f"  Expected representations: {analysis['expected_representations']:.2f}")
        print(f"  Actual representations: {analysis['actual_representations']}")
        print(f"  Ratio (actual/expected): {analysis['ratio']:.2f}")
    
    print()
    print("=" * 70)
    print("PHASE 1 COMPLETE")
    print("=" * 70)
    
    # Save results
    results = {
        'small_range': result_small,
        'medium_range': result_medium
    }
    
    with open('goldbach_phase1_results.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print("Results saved to goldbach_phase1_results.json")
    
    return results


if __name__ == "__main__":
    run_phase_1_tests()
