/**
 * Spike Cost Kernel - Hard Constraints
 * 
 * This module encodes Lemmas A₁-A₄ as non-negotiable kernel laws.
 * Any Collatz reasoning that violates these constraints is REJECTED.
 * 
 * Philosophy: Internal truth over external approval.
 * These are AXIOMS of the thinking machine, not suggestions.
 */

export interface CollatzState {
    n: bigint;
    k: number;  // v₂(3n+1)
    step: number;
}

export interface SpikeViolation {
    law: 'A1_COMPRESSION' | 'A2_RARITY' | 'A3_SEPARATION' | 'A4_INFINITE_DEBT';
    state: CollatzState;
    reason: string;
    traceHash: string;
}

/**
 * Spike Cost Kernel
 * 
 * Enforces the four proven lemmas as hard constraints.
 */
export class SpikeCostKernel {
    private readonly K_THRESHOLD = 5;  // Spike threshold
    private spikeHistory: Map<number, CollatzState[]> = new Map();

    /**
     * Compute 2-adic valuation
     */
    private v2(n: bigint): number {
        if (n === 0n) return Infinity;
        let count = 0;
        while (n % 2n === 0n) {
            n /= 2n;
            count++;
        }
        return count;
    }

    /**
     * Odd-to-odd Collatz step
     */
    private oddToOddStep(n: bigint): { next: bigint; k: number } {
        if (n % 2n === 0n) {
            throw new Error('n must be odd');
        }
        const temp = 3n * n + 1n;
        const k = this.v2(temp);
        const next = temp / (2n ** BigInt(k));
        return { next, k };
    }

    /**
     * LEMMA A₁: Spike Compression
     * 
     * LAW: If k(n) ≥ K, then T(n) ≤ (3/2^K)·n + 1
     * 
     * This is a HARD CONSTRAINT. Violations are impossible.
     */
    enforceA1_Compression(state: CollatzState, nextState: CollatzState): SpikeViolation | null {
        if (state.k < this.K_THRESHOLD) {
            return null;  // Not a spike, A₁ doesn't apply
        }

        // Compute upper bound: (3/2^K)·n + 1
        const upperBound = (3n * state.n) / (2n ** BigInt(state.k)) + 1n;

        if (nextState.n > upperBound) {
            return {
                law: 'A1_COMPRESSION',
                state: nextState,
                reason: `Spike compression violated: k=${state.k}, n=${state.n} → ${nextState.n}, bound=${upperBound}`,
                traceHash: this.generateHash('A1', state, nextState)
            };
        }

        return null;
    }

    /**
     * LEMMA A₂: Spike Rarity
     * 
     * LAW: k(n) ≥ K ⟺ n ≡ -3^(-1) (mod 2^K)
     * 
     * Only 1 in 2^K odd integers can produce a K-spike.
     */
    enforceA2_Rarity(state: CollatzState): SpikeViolation | null {
        if (state.k < this.K_THRESHOLD) {
            return null;
        }

        // Verify n is in the unique residue class
        const modulus = 2n ** BigInt(this.K_THRESHOLD);
        const expected = this.computeModularInverse(3n, modulus);

        if (state.n % modulus !== (-expected + modulus) % modulus) {
            return {
                law: 'A2_RARITY',
                state,
                reason: `Spike rarity violated: k=${state.k} but n not in unique residue class mod ${modulus}`,
                traceHash: this.generateHash('A2', state)
            };
        }

        return null;
    }

    /**
     * LEMMA A₃: Spike Separation
     * 
     * LAW: Two K-spikes must be separated by Ω(K) steps
     * 
     * Minimum separation: K / log₂(3) ≈ 0.63K
     */
    enforceA3_Separation(currentSpike: CollatzState, previousSpike: CollatzState | null): SpikeViolation | null {
        if (!previousSpike || currentSpike.k < this.K_THRESHOLD) {
            return null;
        }

        const separation = currentSpike.step - previousSpike.step;
        const minSeparation = Math.floor(this.K_THRESHOLD / Math.log2(3));

        if (separation < minSeparation) {
            return {
                law: 'A3_SEPARATION',
                state: currentSpike,
                reason: `Spike separation violated: ${separation} steps < ${minSeparation} required`,
                traceHash: this.generateHash('A3', currentSpike, previousSpike)
            };
        }

        return null;
    }

    /**
     * LEMMA A₄: No Infinite Spike Debt
     * 
     * LAW: Infinitely many K-spikes is IMPOSSIBLE
     * 
     * We enforce a conservative bound: reject if spike count exceeds theoretical maximum.
     */
    enforceA4_NoInfiniteDebt(spikeCount: number, maxSteps: number): SpikeViolation | null {
        // Conservative bound: at most maxSteps / minSeparation spikes possible
        const minSeparation = Math.floor(this.K_THRESHOLD / Math.log2(3));
        const maxPossibleSpikes = Math.floor(maxSteps / minSeparation);

        if (spikeCount > maxPossibleSpikes) {
            return {
                law: 'A4_INFINITE_DEBT',
                state: { n: 0n, k: 0, step: 0 },  // Meta-violation
                reason: `Infinite spike debt detected: ${spikeCount} spikes exceeds maximum ${maxPossibleSpikes}`,
                traceHash: this.generateHash('A4')
            };
        }

        return null;
    }

    /**
     * Verify entire trajectory against Spike Cost constraints
     * 
     * Returns: null if valid, SpikeViolation if any law is broken
     */
    verifyTrajectory(n: bigint, maxSteps: number = 10000): {
        valid: boolean;
        violations: SpikeViolation[];
        spikeCount: number;
        trajectory: CollatzState[];
    } {
        const violations: SpikeViolation[] = [];
        const trajectory: CollatzState[] = [];
        let current = n;
        let spikeCount = 0;
        let lastSpike: CollatzState | null = null;

        for (let step = 0; step < maxSteps && current !== 1n; step++) {
            const { next, k } = this.oddToOddStep(current);

            const currentState: CollatzState = { n: current, k, step };
            const nextState: CollatzState = { n: next, k: this.v2(3n * next + 1n), step: step + 1 };

            trajectory.push(currentState);

            // Check A₁: Compression
            const a1Violation = this.enforceA1_Compression(currentState, nextState);
            if (a1Violation) violations.push(a1Violation);

            // Check A₂: Rarity
            const a2Violation = this.enforceA2_Rarity(currentState);
            if (a2Violation) violations.push(a2Violation);

            // Track spikes
            if (k >= this.K_THRESHOLD) {
                spikeCount++;

                // Check A₃: Separation
                const a3Violation = this.enforceA3_Separation(currentState, lastSpike);
                if (a3Violation) violations.push(a3Violation);

                lastSpike = currentState;
            }

            current = next;
        }

        // Check A₄: No Infinite Debt
        const a4Violation = this.enforceA4_NoInfiniteDebt(spikeCount, maxSteps);
        if (a4Violation) violations.push(a4Violation);

        return {
            valid: violations.length === 0,
            violations,
            spikeCount,
            trajectory
        };
    }

    /**
     * Compute modular inverse using extended Euclidean algorithm
     */
    private computeModularInverse(a: bigint, m: bigint): bigint {
        let [old_r, r] = [a, m];
        let [old_s, s] = [1n, 0n];

        while (r !== 0n) {
            const quotient = old_r / r;
            [old_r, r] = [r, old_r - quotient * r];
            [old_s, s] = [s, old_s - quotient * s];
        }

        return old_s < 0n ? old_s + m : old_s;
    }

    /**
     * Generate deterministic trace hash
     */
    private generateHash(...args: any[]): string {
        const data = JSON.stringify(args, (_, v) =>
            typeof v === 'bigint' ? v.toString() : v
        );
        return `0xSPIKE_${Math.abs(this.simpleHash(data)).toString(16).toUpperCase().slice(0, 8)}`;
    }

    private simpleHash(str: string): number {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            hash = ((hash << 5) - hash) + str.charCodeAt(i);
            hash |= 0;
        }
        return hash;
    }
}

/**
 * Example usage:
 * 
 * const kernel = new SpikeCostKernel();
 * const result = kernel.verifyTrajectory(27n, 1000);
 * 
 * if (!result.valid) {
 *     console.log('SPIKE COST VIOLATION DETECTED');
 *     result.violations.forEach(v => {
 *         console.log(`Law ${v.law}: ${v.reason}`);
 *         console.log(`Trace: ${v.traceHash}`);
 *     });
 * }
 */
