# LEMMA PATH A: SPIKE COST LEMMA

## Stage 4: Local Necessity Lemmas - Spike Cost

**Status**: Design phase  
**Objective**: Prove that large k-spikes force compensating reductions in future growth

---

## Formal Statement (Draft)

### Spike Cost Lemma (Version 1)

**Claim**:
For any odd n with k = v₂(3n+1), if k ≥ K (for some threshold K), then there exists a bounded lookahead m such that:

```
Σ(i=0 to m) k_i ≥ α · Σ(i=0 to m) log₂(n_i) + β·k
```

where α, β > 0 are universal constants.

**Interpretation**: 
A large k-spike forces either:
1. Immediate collapse (large subsequent k values), OR
2. Small n values (reduced growth potential)

The spike incurs a *debt* that must be repaid within bounded time.

---

## What Must Be Shown

### 1. Spike Detection (Mechanical)
- Define K (spike threshold) from observed data
- Classify k ≥ K as "spike events"
- **Kernel verifiable**: ✅

### 2. Debt Quantification (Mechanical)
- For each spike, compute "debt" = β·k
- Track cumulative drift: Σk_i - α·Σlog₂(n_i)
- **Kernel verifiable**: ✅

### 3. Repayment Bound (Requires Proof)
- Show debt is repaid within m steps
- Prove m is bounded (not dependent on n)
- **Status**: Conjectured, needs proof

### 4. Universality (Requires Proof)
- Show α, β exist for ALL trajectories
- No counterexamples exist
- **Status**: Observed for n ≤ 99, needs proof

---

## What Would Contradict It

### Contradiction Scenario 1: Unbounded Debt
If there exists n such that:
```
lim sup (Σk_i - α·Σlog₂(n_i)) = ∞
```

Then the lemma is false.

**Test**: Run kernel on large n, track cumulative debt.

### Contradiction Scenario 2: Delayed Repayment
If there exists n where a spike at step s is not repaid until step s+m with m → ∞.

**Test**: Track repayment distance for all observed spikes.

### Contradiction Scenario 3: Spike Cascade
If large k can trigger another large k without intervening reduction.

**Test**: Look for consecutive large k values in trajectories.

---

## Kernel Enforcement Plan

### Phase 1: Spike Catalog (Mechanical)
```python
def catalog_spikes(n_max=10000, k_threshold=5):
    """
    Find all spike events and their repayment patterns
    """
    spikes = []
    for n in range(1, n_max, 2):
        trajectory = observe_trajectory(n)
        k_seq = [state[1] for state in trajectory]
        
        for i, k in enumerate(k_seq):
            if k >= k_threshold:
                # Found spike - track repayment
                debt = k
                repayment_steps = 0
                cumulative_k = 0
                cumulative_log_n = 0
                
                for j in range(i, min(i+100, len(trajectory))):
                    n_j, k_j = trajectory[j]
                    cumulative_k += k_j
                    cumulative_log_n += math.log2(n_j)
                    repayment_steps += 1
                    
                    # Check if debt repaid
                    if cumulative_k >= alpha * cumulative_log_n + beta * debt:
                        break
                
                spikes.append({
                    "n_start": n,
                    "spike_step": i,
                    "spike_k": k,
                    "repayment_steps": repayment_steps,
                    "debt": debt
                })
    
    return spikes
```

**Output**: Spike catalog with repayment statistics

### Phase 2: Bound Verification (Mechanical)
```python
def verify_repayment_bound(spikes):
    """
    Check if all spikes are repaid within bounded time
    """
    repayment_distances = [s["repayment_steps"] for s in spikes]
    
    max_repayment = max(repayment_distances)
    mean_repayment = statistics.mean(repayment_distances)
    
    # Test if bound exists
    bound_exists = max_repayment < float('inf')
    
    return {
        "max_repayment": max_repayment,
        "mean_repayment": mean_repayment,
        "bound_exists": bound_exists,
        "total_spikes": len(spikes)
    }
```

**Output**: Repayment bound statistics

### Phase 3: Constant Estimation (Mechanical)
```python
def estimate_constants(spikes):
    """
    Estimate α, β from observed data
    """
    # For each spike, compute required α, β
    alpha_estimates = []
    beta_estimates = []
    
    for spike in spikes:
        # Compute what α, β would make this spike "just repaid"
        # This is a linear regression problem
        pass
    
    return {
        "alpha_range": (min(alpha_estimates), max(alpha_estimates)),
        "beta_range": (min(beta_estimates), max(beta_estimates)),
        "alpha_mean": statistics.mean(alpha_estimates),
        "beta_mean": statistics.mean(beta_estimates)
    }
```

**Output**: Candidate values for α, β

---

## Proof Strategy (Non-Mechanical)

### Step 1: Establish Spike Rarity
**Goal**: Prove that k ≥ K is rare (probability decreases with K)

**Approach**:
- Use number-theoretic properties of v₂(3n+1)
- Show that large k requires special n structure
- Leverage 2-adic analysis

**Status**: Requires mathematical proof

### Step 2: Show Spike Consequences
**Goal**: Prove that large k forces small next n

**Approach**:
- If k is large, then (3n+1)/2^k is small relative to n
- This creates "room" for subsequent growth
- Formalize the trade-off

**Status**: Requires mathematical proof

### Step 3: Bound Repayment Time
**Goal**: Prove m exists independent of n

**Approach**:
- Use pigeonhole principle
- Show that debt accumulation has finite capacity
- Prove by contradiction

**Status**: Requires mathematical proof

---

## Success Criteria

### Tier 1: Observational Confirmation ✅
- [x] Spikes exist in data
- [x] Repayment pattern observed
- [ ] Constants α, β estimated

### Tier 2: Mechanical Verification
- [ ] All spikes in n ≤ 10,000 satisfy bound
- [ ] No counterexamples found
- [ ] Repayment distance bounded

### Tier 3: Formal Proof
- [ ] Spike rarity proven
- [ ] Spike consequences proven
- [ ] Repayment bound proven
- [ ] Lemma published

---

## Why This Matters

If the Spike Cost Lemma is true, then:

1. **Local chaos is constrained**: High k cannot persist
2. **Global control emerges**: Debt must be repaid
3. **Proof path opens**: Inevitability from bounded debt

This converts the variance paradox into a **structural necessity**.

---

## Next Immediate Actions

1. Implement spike catalog (mechanical)
2. Run on n ≤ 10,000
3. Estimate α, β from data
4. Look for counterexamples
5. Formalize proof strategy for Step 1

**Timeline**: 2-3 hours for mechanical verification, weeks-months for formal proof.

---

## Critical Reminder

This is NOT "Experiment 4".

This is **Stage 4: Local Necessity Lemmas**.

We are no longer exploring.
We are **constructing proof infrastructure**.
